/** 
 *******************************************************************************
 * @file    bsp.c
 * @brief   Board Support Package. \n
 *          TMPM3HQA Evaluation Board.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include "bsp.h"

#if defined(__BSP_H)
/**
 *  @addtogroup BSP
 *  @{
 */

/** 
 *  @defgroup TMPM3HQFDA_EVAL TMPM3HQA Evaluation Board
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup TMPM3HQFDA_EVAL_Private_macro TMPM3HQA Evaluation Board Private Macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group TMPM3HQFDA_EVAL_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup TMPM3HQFDA_EVAL_Private_define TMPM3HQA Evaluation Board Private Define
 *  @{
 */

/** 
 *  @defgroup ELOSCConfiguration ELOSC Configuration
 *  @brief    ELOSC Configuration.
 *  @{
 */
#define ELOSC_CFG_WARM_UP_TIME  ((uint32_t)(5000))      /*!< Warm up time(us). */
#define ELOSC_CFG_CLOCK         ((uint32_t)(32768))     /*!< Clock(hz).        */

/** 
 *  @}
 */ /* End of group ELOSCConfiguration */

/** 
 *  @}
 */ /* End of group TMPM3HQFDA_EVAL_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup TMPM3HQFDA_EVAL_Private_define TMPM3HQA Evaluation Board Private Define
 *  @{
 */

#define BSP_NULL        ((void *)0) /*!< Null Pointer */

/** 
 *  @defgroup  CGWUPLCR  CGWUPLCR Register
 *  @brief     LOSC warming up Register.
 *  @details   Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 31:27 | -          |
 *  | 26:12 | WUPTL      |
 *  | 11:8  | WUPTL      |
 *  | 7:2   | -          |
 *  | 1     | WULEF      |
 *  | 0     | WULON      |
 *  @{
 */
/* WUPTL  */
#define CGWUPLCR_WUPTL_HIGH_MASK        ((uint32_t)0x07FFF000)    /*!< WUPTL   :High Bit Mask.  */
/* WULEF  */
#define CGWUPLCR_WULEF_MASK             ((uint32_t)0x00000002)    /*!< WULEF   :Mask.           */
#define CGWUPLCR_WULEF_R_DONE           ((uint32_t)0x00000000)    /*!< WULEF   :[R]  :Done      */
#define CGWUPLCR_WULEF_R_RUNNING        ((uint32_t)0x00000002)    /*!< WULEF   :[R]  :Running   */
/* WULON  */
#define CGWUPLCR_WULON_MASK             ((uint32_t)0x00000001)    /*!< WULON   :Mask.           */
#define CGWUPLCR_WULON_W_ENABLE         ((uint32_t)0x00000001)    /*!< WULON   :[W]  :Enable     */
/** 
 *  @}
 */ /* End of group CGWUPLCR */

/** 
 *  @defgroup  RLMLOSCCR  RLMLOSCCR Register
 *  @brief     LOSC Control Register.
 *  @details   Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 7:2   | -          |
 *  | 1     | -          |
 *  | 0     | XTEN       |
 *  @{
 */
/* XTEN */
#define RLMLOSCCR_XTEN_RW_DISABLE       ((uint8_t)0x00)    /*!< XTEN :[R/W]  :Disable  */
#define RLMLOSCCR_XTEN_RW_ENABLE        ((uint8_t)0x01)    /*!< XTEN :[R/W]  :Enable   */
/** 
 *  @}
 */ /* End of group RLMLOSCCR */

/** 
 *  @}
 */ /* End of group TMPM3HQFDA_EVAL_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup TMPM3HQFDA_EVAL_Private_define TMPM3HQA Evaluation Board Private Define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group TMPM3HQFDA_EVAL_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup TMPM3HQFDA_EVAL_Private_typedef TMPM3HQA Evaluation Board Private Typedef
 *  @{
 */

/*----------------------------------*/
/** 
 * @brief   Timer Information Table.
 * @details 
 * | [][0] | [][1] | [][2] |
 * | :---  | :---  | :---  |
 * | CH    | Type  | IRQ   |
*/
/*----------------------------------*/
typedef struct
{
    uint32_t  ch;       /*!< channel  */
    uint32_t  type;     /*!< type     */
    IRQn_Type irq;      /*!< irq Type */
} bsp_timer_info_t;

/*----------------------------------*/
/** 
 * @brief   Output Pulse Information Table.
 * @details 
 * | [][0] | [][1] | [][2] |
 * | :---  | :---  | :---  |
 * | CH    | Type  | IRQ   |
*/
/*----------------------------------*/
typedef struct
{
    uint32_t  ch;       /*!< channel  */
    uint32_t  type;     /*!< type     */
    IRQn_Type irq;      /*!< irq Type */
} bsp_output_pulse_info_t;

/*----------------------------------*/
/** 
 * @brief   RMC Information Table.
 * @details 
 * | [][0] | [][1] | [][2] |
 * | :---  | :---  | :---  |
 * | CH    | reg  | IRQ   |
*/
/*----------------------------------*/
typedef struct
{
    uint32_t  ch;       /*!< channel  */
    IRQn_Type irq;      /*!< irq Type */
} bsp_rmc_info_t;

/**
 *  @}
 */ /* End of group TMPM3HQFDA_EVAL_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup TMPM3HQFDA_EVAL_Private_variables TMPM3HQA Evaluation Board Private Variables
 *  @{
 */
static gpio_t port;     /*!< Port driver instance. */
static BSPExInt exi;

/*----------------------------------*/
/** 
 * @brief   Thermistors AD Value Table.
 * @details  V   = AVCC*(TH1/TH1+R60)
 * @        AD   = V/(AVCC/0xFFF)
 * @               AVCC :5(v)
 * @               R60  :10(k-ohm)
 * @               TH1  :Refer thermistors temperature characteristics.
 * @        Resolution  :12 bit(=0xFFF).

*/
/*----------------------------------*/
static const int32_t NCP15XV103J03RC[][4] = 
{
    /*----------------------------------------*/
    /* TEMP. | AD-low   | AD-center| AD-high  */
    /* deg.C | -        | -        | -        */
    /*----------------------------------------*/
    { -40, 0xF72, 0xF86, 0xF96 },
    { -39, 0xF69, 0xF7E, 0xF8F },
    { -38, 0xF60, 0xF75, 0xF88 },
    { -37, 0xF56, 0xF6C, 0xF80 },
    { -36, 0xF4B, 0xF63, 0xF77 },
    { -35, 0xF40, 0xF59, 0xF6E },
    { -34, 0xF35, 0xF4E, 0xF65 },
    { -33, 0xF28, 0xF43, 0xF5B },
    { -32, 0xF1C, 0xF38, 0xF50 },
    { -31, 0xF0E, 0xF2C, 0xF45 },
    { -30, 0xF00, 0xF1F, 0xF3A },
    { -29, 0xEF2, 0xF12, 0xF2D },
    { -28, 0xEE2, 0xF04, 0xF21 },
    { -27, 0xED2, 0xEF5, 0xF13 },
    { -26, 0xEC2, 0xEE5, 0xF05 },
    { -25, 0xEB0, 0xED5, 0xEF6 },
    { -24, 0xE9E, 0xEC4, 0xEE6 },
    { -23, 0xE8B, 0xEB3, 0xED6 },
    { -22, 0xE77, 0xEA1, 0xEC5 },
    { -21, 0xE63, 0xE8E, 0xEB3 },
    { -20, 0xE4E, 0xE7A, 0xEA1 },
    { -19, 0xE38, 0xE65, 0xE8D },
    { -18, 0xE21, 0xE4F, 0xE79 },
    { -17, 0xE09, 0xE39, 0xE64 },
    { -16, 0xDF0, 0xE22, 0xE4E },
    { -15, 0xDD7, 0xE0A, 0xE37 },
    { -14, 0xDBD, 0xDF1, 0xE1F },
    { -13, 0xDA2, 0xDD7, 0xE07 },
    { -12, 0xD86, 0xDBC, 0xDED },
    { -11, 0xD69, 0xDA1, 0xDD3 },
    { -10, 0xD4C, 0xD84, 0xDB8 },
    {  -9, 0xD2D, 0xD67, 0xD9C },
    {  -8, 0xD0E, 0xD48, 0xD7E },
    {  -7, 0xCED, 0xD29, 0xD60 },
    {  -6, 0xCCC, 0xD09, 0xD41 },
    {  -5, 0xCAA, 0xCE8, 0xD21 },
    {  -4, 0xC88, 0xCC6, 0xD00 },
    {  -3, 0xC64, 0xCA4, 0xCDE },
    {  -2, 0xC40, 0xC80, 0xCBC },
    {  -1, 0xC1B, 0xC5C, 0xC98 },
    {   0, 0xBF6, 0xC37, 0xC74 },
    {   1, 0xBCF, 0xC11, 0xC4F },
    {   2, 0xBA8, 0xBEB, 0xC28 },
    {   3, 0xB81, 0xBC3, 0xC02 },
    {   4, 0xB59, 0xB9C, 0xBDA },
    {   5, 0xB30, 0xB73, 0xBB2 },
    {   6, 0xB07, 0xB4A, 0xB89 },
    {   7, 0xADD, 0xB20, 0xB5F },
    {   8, 0xAB3, 0xAF6, 0xB35 },
    {   9, 0xA88, 0xACB, 0xB0A },
    {  10, 0xA5D, 0xAA0, 0xADE },
    {  11, 0xA32, 0xA74, 0xAB2 },
    {  12, 0xA07, 0xA48, 0xA86 },
    {  13, 0x9DB, 0xA1C, 0xA59 },
    {  14, 0x9AF, 0x9EF, 0xA2C },
    {  15, 0x983, 0x9C2, 0x9FE },
    {  16, 0x956, 0x995, 0x9D0 },
    {  17, 0x92A, 0x968, 0x9A3 },
    {  18, 0x8FE, 0x93B, 0x974 },
    {  19, 0x8D1, 0x90D, 0x946 },
    {  20, 0x8A5, 0x8E0, 0x918 },
    {  21, 0x879, 0x8B3, 0x8E9 },
    {  22, 0x84D, 0x885, 0x8BB },
    {  23, 0x821, 0x858, 0x88D },
    {  24, 0x7F6, 0x82C, 0x85F },
    {  25, 0x7CA, 0x7FF, 0x831 },
    {  26, 0x79D, 0x7D3, 0x806 },
    {  27, 0x770, 0x7A7, 0x7DB },
    {  28, 0x743, 0x77B, 0x7B1 },
    {  29, 0x716, 0x750, 0x786 },
    {  30, 0x6EA, 0x725, 0x75D },
    {  31, 0x6BF, 0x6FA, 0x733 },
    {  32, 0x694, 0x6D0, 0x70A },
    {  33, 0x66A, 0x6A7, 0x6E1 },
    {  34, 0x640, 0x67E, 0x6B9 },
    {  35, 0x617, 0x655, 0x692 },
    {  36, 0x5EE, 0x62D, 0x66A },
    {  37, 0x5C7, 0x606, 0x644 },
    {  38, 0x5A0, 0x5DF, 0x61E },
    {  39, 0x579, 0x5B9, 0x5F8 },
    {  40, 0x553, 0x594, 0x5D3 },
    {  41, 0x52E, 0x56F, 0x5AF },
    {  42, 0x50A, 0x54B, 0x58B },
    {  43, 0x4E7, 0x528, 0x568 },
    {  44, 0x4C4, 0x505, 0x545 },
    {  45, 0x4A2, 0x4E3, 0x524 },
    {  46, 0x481, 0x4C2, 0x502 },
    {  47, 0x460, 0x4A1, 0x4E2 },
    {  48, 0x440, 0x481, 0x4C2 },
    {  49, 0x421, 0x462, 0x4A2 },
    {  50, 0x403, 0x443, 0x484 },
    {  51, 0x3E6, 0x425, 0x466 },
    {  52, 0x3C9, 0x408, 0x448 },
    {  53, 0x3AD, 0x3EC, 0x42C },
    {  54, 0x392, 0x3D0, 0x410 },
    {  55, 0x377, 0x3B5, 0x3F4 },
    {  56, 0x35D, 0x39B, 0x3DA },
    {  57, 0x344, 0x381, 0x3BF },
    {  58, 0x32C, 0x368, 0x3A6 },
    {  59, 0x314, 0x350, 0x38D },
    {  60, 0x2FD, 0x338, 0x375 },
    {  61, 0x2E6, 0x321, 0x35D },
    {  62, 0x2D0, 0x30A, 0x346 },
    {  63, 0x2BB, 0x2F4, 0x330 },
    {  64, 0x2A6, 0x2DF, 0x31A },
    {  65, 0x292, 0x2CA, 0x304 },
    {  66, 0x27F, 0x2B6, 0x2F0 },
    {  67, 0x26C, 0x2A2, 0x2DB },
    {  68, 0x25A, 0x28F, 0x2C7 },
    {  69, 0x248, 0x27D, 0x2B4 },
    {  70, 0x236, 0x26B, 0x2A1 },
    {  71, 0x226, 0x259, 0x28F },
    {  72, 0x216, 0x248, 0x27E },
    {  73, 0x206, 0x238, 0x26C },
    {  74, 0x1F7, 0x228, 0x25C },
    {  75, 0x1E8, 0x218, 0x24C },
    {  76, 0x1DA, 0x209, 0x23C },
    {  77, 0x1CC, 0x1FB, 0x22C },
    {  78, 0x1BE, 0x1ED, 0x21E },
    {  79, 0x1B1, 0x1DF, 0x20F },
    {  80, 0x1A5, 0x1D1, 0x201 },
    {  81, 0x199, 0x1C4, 0x1F3 },
    {  82, 0x18D, 0x1B8, 0x1E6 },
    {  83, 0x181, 0x1AC, 0x1D9 },
    {  84, 0x176, 0x1A0, 0x1CC },
    {  85, 0x16B, 0x194, 0x1C0 },
    {  86, 0x161, 0x189, 0x1B4 },
    {  87, 0x157, 0x17E, 0x1A9 },
    {  88, 0x14D, 0x174, 0x19E },
    {  89, 0x143, 0x169, 0x193 },
    {  90, 0x13A, 0x160, 0x188 },
    {  91, 0x131, 0x156, 0x17E },
    {  92, 0x129, 0x14D, 0x174 },
    {  93, 0x120, 0x144, 0x16A },
    {  94, 0x118, 0x13B, 0x161 },
    {  95, 0x110, 0x132, 0x158 },
    {  96, 0x109, 0x12A, 0x14F },
    {  97, 0x101, 0x122, 0x146 },
    {  98,  0xFA, 0x11A, 0x13E },
    {  99,  0xF3, 0x113, 0x136 },
    { 100,  0xED, 0x10C, 0x12E },
    { 101,  0xE6, 0x104, 0x126 },
    { 102,  0xE0,  0xFE, 0x11E },
    { 103,  0xDA,  0xF7, 0x117 },
    { 104,  0xD4,  0xF0, 0x110 },
    { 105,  0xCE,  0xEA, 0x109 },
    { 106,  0xC8,  0xE4, 0x103 },
    { 107,  0xC3,  0xDE,  0xFC },
    { 108,  0xBE,  0xD8,  0xF6 },
    { 109,  0xB9,  0xD3,  0xF0 },
    { 110,  0xB4,  0xCE,  0xEA },
    { 111,  0xAF,  0xC8,  0xE4 },
    { 112,  0xAA,  0xC3,  0xDF },
    { 113,  0xA6,  0xBE,  0xD9 },
    { 114,  0xA2,  0xB9,  0xD4 },
    { 115,  0x9D,  0xB5,  0xCF },
    { 116,  0x99,  0xB0,  0xCA },
    { 117,  0x95,  0xAC,  0xC5 },
    { 118,  0x92,  0xA8,  0xC0 },
    { 119,  0x8E,  0xA3,  0xBC },
    { 120,  0x8A,  0x9F,  0xB7 },
    { 121,  0x87,  0x9B,  0xB3 },
    { 122,  0x83,  0x98,  0xAF },
    { 123,  0x80,  0x94,  0xAB },
    { 124,  0x7D,  0x90,  0xA7 },
    { 125,  0x7A,  0x8D,  0xA3 }
};

/*-----------------------------------------*/
/* Name    | Pin    | Port |  CN           */
/*-----------------------------------------*/
/* USB_TXD | 17     | PA1  |  5: 10        */
/*-----------------------------------------*/
/*----------------------------------*/
/** 
 * @brief   USB UART TX Information Table.
 * @details 
 * | [][0] | [][1] | [][2] | [][3]  | [][4]         | 
 * | :---  | :---  | :---  | :---   | :---          |
 * | Group | Num   | CH    | DMA CH | Baudrate(bps) |
*/
/*----------------------------------*/
static const uint32_t usbUartTxTbl[BSP_USB_UART_MAX][5] =
{
    {(uint32_t)GPIO_PORT_A, (uint32_t)GPIO_PORT_1, (uint32_t)(0), (uint32_t)(7), (uint32_t)(115200)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_1, (uint32_t)(1), (uint32_t)(9), (uint32_t)(115200)},
    {(uint32_t)GPIO_PORT_L, (uint32_t)GPIO_PORT_0, (uint32_t)(2), (uint32_t)(11), (uint32_t)(115200)},
    {(uint32_t)GPIO_PORT_A, (uint32_t)GPIO_PORT_6, (uint32_t)(3), (uint32_t)(13), (uint32_t)(115200)},
    {(uint32_t)GPIO_PORT_V, (uint32_t)GPIO_PORT_6, (uint32_t)(4), (uint32_t)(11), (uint32_t)(115200)}
};

/*-----------------------------------------*/
/* Name        | Pin    | Port |  CN       */
/*-----------------------------------------*/
/* USB_RXD     | 16     | PA2  |  5: 12    */
/*-----------------------------------------*/
/*----------------------------------*/
/** 
 * @brief   USB UART RX Information Table.
 * @details 
 * | [][0] | [][1] | [][2] | [][3]  | [][4]         | 
 * | :---  | :---  | :---  | :---   | :---          |
 * | Group | Num   | CH    | DMA CH | Baudrate(bps) |
*/
/*----------------------------------*/
static const uint32_t usbUartRxTbl[BSP_USB_UART_MAX][5] =
{
    {(uint32_t)GPIO_PORT_A, (uint32_t)GPIO_PORT_2, (uint32_t)(0), (uint32_t)(6), (uint32_t)(115200)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_2, (uint32_t)(1), (uint32_t)(8), (uint32_t)(115200)},
    {(uint32_t)GPIO_PORT_L, (uint32_t)GPIO_PORT_1, (uint32_t)(2), (uint32_t)(10), (uint32_t)(115200)},
    {(uint32_t)GPIO_PORT_A, (uint32_t)GPIO_PORT_7, (uint32_t)(3), (uint32_t)(12), (uint32_t)(115200)},
    {(uint32_t)GPIO_PORT_V, (uint32_t)GPIO_PORT_7, (uint32_t)(4), (uint32_t)(10), (uint32_t)(115200)}
};

/*----------------------------------*/
/** 
 * @brief   TSPI TX Information Table.
 * @details   Detail.
 * | [][0] | [][1] | [][2] | [][3]  | 
 * | :---  | :---  | :---  | :---   |
 * | Group | Num   | CH    | DMA CH |
*/
/*----------------------------------*/
static const uint32_t tspiTxTbl[BSP_SFLASH_MAX][4] =
{
    {(uint32_t)GPIO_PORT_A, (uint32_t)GPIO_PORT_1, (uint32_t)(0), (uint32_t)(1)},
    {(uint32_t)GPIO_PORT_B, (uint32_t)GPIO_PORT_3, (uint32_t)(1), (uint32_t)(3)},
    {(uint32_t)GPIO_PORT_T, (uint32_t)GPIO_PORT_3, (uint32_t)(2), (uint32_t)(1)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_4, (uint32_t)(3), (uint32_t)(3)},
    {(uint32_t)GPIO_PORT_H, (uint32_t)GPIO_PORT_5, (uint32_t)(4), (uint32_t)(5)}
};
/*----------------------------------*/
/** 
 * @brief   TSPI CLK Information Table.
 * @details   Detail.
 * | [][0] | [][1] | [][2] |
 * | :---  | :---  | :---  |
 * | Group | Num   | CH    |
*/
/*----------------------------------*/
static const uint32_t tspiCLKTbl[BSP_SFLASH_MAX][3] =
{
    {(uint32_t)GPIO_PORT_A, (uint32_t)GPIO_PORT_0, (uint32_t)(0)},
    {(uint32_t)GPIO_PORT_B, (uint32_t)GPIO_PORT_2, (uint32_t)(1)},
    {(uint32_t)GPIO_PORT_T, (uint32_t)GPIO_PORT_2, (uint32_t)(2)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_5, (uint32_t)(3)},
    {(uint32_t)GPIO_PORT_H, (uint32_t)GPIO_PORT_4, (uint32_t)(4)}
};

/*-----------------------------------------*/
/* Name        | Pin    | Port |  CN       */
/*-----------------------------------------*/
/* USB_RXD     | 16     | PK0  |  5: 12    */
/*-----------------------------------------*/
/*----------------------------------*/
/** 
 * @brief   TSPI RX Information Table.
 * @details   Detail.
 * | [][0] | [][1] | [][2] | [][3]  | 
 * | :---  | :---  | :---  | :---   |
 * | Group | Num   | CH    | DMA CH |
*/
/*----------------------------------*/
static const uint32_t tspiRxTbl[BSP_SFLASH_MAX][4] =
{
    {(uint32_t)GPIO_PORT_A, (uint32_t)GPIO_PORT_2, (uint32_t)(0), (uint32_t)(0)},
    {(uint32_t)GPIO_PORT_B, (uint32_t)GPIO_PORT_4, (uint32_t)(1), (uint32_t)(2)},
    {(uint32_t)GPIO_PORT_T, (uint32_t)GPIO_PORT_4, (uint32_t)(2), (uint32_t)(2)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_3, (uint32_t)(3), (uint32_t)(2)},
    {(uint32_t)GPIO_PORT_H, (uint32_t)GPIO_PORT_6, (uint32_t)(4), (uint32_t)(4)}
};
/*-----------------------------------------*/
/* Name        | Pin    | Port |  CN       */
/*-----------------------------------------*/
/* PORT_LED[0] | 30     | PB4  |  5: 28    */
/* PORT_LED[1] | 31     | PB5  |  5: 30    */
/* PORT_LED[2] | 32     | PB6  |  5: 32    */
/* PORT_LED[3] | 33     | PB7  |  5: 34    */
/*-----------------------------------------*/
/*----------------------------------*/
/** 
 * @brief   Port LED Information Table.
 * @details 
 * | [][0] | [][1] |
 * | :---  | :---  |
 * | Group | Num   |
*/
/*----------------------------------*/
static const uint32_t ledTbl[BSP_LED_MAX][2] =
{
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_4},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_5},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_6},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_7}
};

/*-----------------------------------------*/
/* Name        | Pin    | Port |  CN       */
/*-----------------------------------------*/
/* PORT_PSW[3] | 69     | PN1  |  9: 26    */
/* PORT_PSW[2] | 68     | PN2  |  9: 24    */
/* PORT_PSW[1] | 67     | PN3  |  9: 22    */
/* PORT_PSW[0] | 66     | PN4  |  9: 20    */
/*-----------------------------------------*/
/*----------------------------------*/
/** 
 * @brief   Push SW Information Table.
 * @details 
 * | [][0] | [][1] |
 * | :---  | :---  |
 * | Group | Num   |
*/
/*----------------------------------*/
static const uint32_t pswTbl[BSP_PSW_MAX][2] =
{
    {(uint32_t)GPIO_PORT_V, (uint32_t)GPIO_PORT_0},
    {(uint32_t)GPIO_PORT_V, (uint32_t)GPIO_PORT_1},
    {(uint32_t)GPIO_PORT_V, (uint32_t)GPIO_PORT_2},
    {(uint32_t)GPIO_PORT_V, (uint32_t)GPIO_PORT_3},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_2}
};

/*----------------------------------*/
/** 
 * @brief   Timer Information Table.
*/
/*----------------------------------*/
static const bsp_timer_info_t timerTbl[BSP_TIMER_MAX] =
{
    {(uint32_t)0, (uint32_t)T32A_TIMERA, INTT32A00A_IRQn},
    {(uint32_t)2, (uint32_t)T32A_TIMERA, INTT32A02A_IRQn}
};

/*----------------------------------*/
/** 
 * @brief   Output Pulse Information Table.
*/
/*----------------------------------*/
static const bsp_output_pulse_info_t pulseOutTbl[BSP_OUT_PULSE_SAMPLE_MAX] =
{
    {(uint32_t)3, (uint32_t)T32A_TIMERA, INTT32A03A_IRQn},
    {(uint32_t)3, (uint32_t)T32A_TIMERB, INTT32A03B_IRQn}
};

/*-----------------------------------------*/
/* Name        | Pin    | Port |  CN       */
/*-----------------------------------------*/
/* PORT_AIN[2] | 2      | PD2  |  9: 20    */
/*-----------------------------------------*/
/*----------------------------------*/
/** 
 * @brief   Thermistors Information Table.
 * @details 
 * | [][0] | [][1] | [][2]    | [][3]  | [][4]  |
 * | :---  | :---  | :---     | :---   | :---   |
 * | Group | Num   | ADIN CH  | ADC CH | DMA CH |
*/
/*----------------------------------*/
static const uint32_t thermistorsTbl[BSP_THERMISTORS_MAX][5] =
{
    {(uint32_t)GPIO_PORT_E, (uint32_t)GPIO_PORT_0, (uint32_t)(4), (uint32_t)(0), (uint32_t)(14) },
    {(uint32_t)GPIO_PORT_D, (uint32_t)GPIO_PORT_4, (uint32_t)(19), (uint32_t)(1), (uint32_t)(14) },
    {(uint32_t)GPIO_PORT_D, (uint32_t)GPIO_PORT_5, (uint32_t)(20), (uint32_t)(2), (uint32_t)(14) }
};

/*----------------------------------*/
/** 
 * @brief   DLCD Information Table.
 * @details 
 * | [][0] | [][1] | [][2] |
 * | :---  | :---  | :---  |
 * | Group | Num   | Func  |
*/
/*----------------------------------*/
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HQA) || defined(TMPM3HPA))
static const uint32_t dlcdTbl[BSP_DLCD_MAX][3] =
{
#if defined(TMPM3HQA)
    {(uint32_t)GPIO_PORT_V, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PV3_SEG00)},
    {(uint32_t)GPIO_PORT_V, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PV2_SEG01)},
    {(uint32_t)GPIO_PORT_V, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PV1_SEG02)},
    {(uint32_t)GPIO_PORT_V, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PV0_SEG03)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_7, (uint32_t)(GPIO_PP7_SEG04)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_6, (uint32_t)(GPIO_PP6_SEG05)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PP5_SEG06)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PP4_SEG07)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PP3_SEG08)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_7, (uint32_t)(GPIO_PK7_SEG09)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_6, (uint32_t)(GPIO_PK6_SEG10)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PK5_SEG11)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PK4_SEG12)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PK3_SEG13)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PK2_SEG14)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PK1_SEG15)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PK0_SEG16)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PJ5_SEG17)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PJ4_SEG18)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PJ3_SEG19)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PJ2_SEG20)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PJ1_SEG21)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PJ0_SEG22)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PN0_SEG23)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PN1_SEG24)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PN2_SEG25)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PN3_SEG26)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PN4_SEG27)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PN5_SEG28)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_7, (uint32_t)(GPIO_PR7_SEG29)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_6, (uint32_t)(GPIO_PR6_SEG30)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PR5_SEG31)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PR4_SEG32)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PR3_SEG33)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PR2_SEG34)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PR1_SEG35)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PR0_SEG36)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_6, (uint32_t)(GPIO_PC6_SEG37)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PC5_SEG38)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PC4_SEG39)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PC3_DCOM0)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PC2_DCOM1)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PC1_DCOM2)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PC0_DCOM3)}
#endif
#if defined(TMPM3HPA)
    {(uint32_t)GPIO_PORT_V, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PV3_SEG00)},
    {(uint32_t)GPIO_PORT_V, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PV2_SEG01)},
    {(uint32_t)GPIO_PORT_V, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PV1_SEG02)},
    {(uint32_t)GPIO_PORT_V, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PV0_SEG03)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_7, (uint32_t)(GPIO_PP7_SEG04)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_6, (uint32_t)(GPIO_PP6_SEG05)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PP5_SEG06)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PP4_SEG07)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PP3_SEG08)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_7, (uint32_t)(GPIO_PK7_SEG09)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_6, (uint32_t)(GPIO_PK6_SEG10)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PK5_SEG11)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PK4_SEG12)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PK3_SEG13)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PK2_SEG14)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PK1_SEG15)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PK0_SEG16)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PJ5_SEG17)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PJ4_SEG18)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PJ3_SEG19)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PJ2_SEG20)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PJ1_SEG21)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PJ0_SEG22)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PN0_SEG23)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PN1_SEG24)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PN2_SEG25)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PN3_SEG26)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PN4_SEG27)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PN5_SEG28)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_7, (uint32_t)(GPIO_PR7_SEG29)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_6, (uint32_t)(GPIO_PR6_SEG30)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PR5_SEG31)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PR4_SEG32)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PR3_SEG33)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PR2_SEG34)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PR1_SEG35)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PR0_SEG36)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_6, (uint32_t)(GPIO_PC6_SEG37)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PC5_SEG38)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PC4_SEG39)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PC3_DCOM0)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PC2_DCOM1)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PC1_DCOM2)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PC0_DCOM3)}
#endif
#if defined(TMPM3HNA)
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_7, (uint32_t)(GPIO_PP7_SEG04)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_6, (uint32_t)(GPIO_PP6_SEG05)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PP5_SEG06)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PP4_SEG07)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PP3_SEG08)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_7, (uint32_t)(GPIO_PK7_SEG09)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_6, (uint32_t)(GPIO_PK6_SEG10)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PK5_SEG11)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PK4_SEG12)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PK3_SEG13)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PK2_SEG14)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PK1_SEG15)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PK0_SEG16)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PJ5_SEG17)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PJ4_SEG18)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PJ3_SEG19)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PJ2_SEG20)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PJ1_SEG21)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PJ0_SEG22)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PN0_SEG23)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PN1_SEG24)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PN2_SEG25)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PN3_SEG26)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PN4_SEG27)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PN5_SEG28)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PR3_SEG33)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PR2_SEG34)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PR1_SEG35)},
    {(uint32_t)GPIO_PORT_R, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PR0_SEG36)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_6, (uint32_t)(GPIO_PC6_SEG37)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PC5_SEG38)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PC4_SEG39)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PC3_DCOM0)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PC2_DCOM1)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PC1_DCOM2)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PC0_DCOM3)}
#endif
#if defined(TMPM3HMA)
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_6, (uint32_t)(GPIO_PP6_SEG05)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PP5_SEG06)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PP4_SEG07)},
    {(uint32_t)GPIO_PORT_P, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PP3_SEG08)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_7, (uint32_t)(GPIO_PK7_SEG09)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_6, (uint32_t)(GPIO_PK6_SEG10)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PK5_SEG11)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PK4_SEG12)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PK3_SEG13)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PK2_SEG14)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PK1_SEG15)},
    {(uint32_t)GPIO_PORT_K, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PK0_SEG16)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PJ5_SEG17)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PJ4_SEG18)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PJ3_SEG19)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PJ2_SEG20)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PJ1_SEG21)},
    {(uint32_t)GPIO_PORT_J, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PJ0_SEG22)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PN0_SEG23)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PN1_SEG24)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PN2_SEG25)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PN3_SEG26)},
    {(uint32_t)GPIO_PORT_N, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PN4_SEG27)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_6, (uint32_t)(GPIO_PC6_SEG37)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_5, (uint32_t)(GPIO_PC5_SEG38)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_4, (uint32_t)(GPIO_PC4_SEG39)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_3, (uint32_t)(GPIO_PC3_DCOM0)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_2, (uint32_t)(GPIO_PC2_DCOM1)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_1, (uint32_t)(GPIO_PC1_DCOM2)},
    {(uint32_t)GPIO_PORT_C, (uint32_t)GPIO_PORT_0, (uint32_t)(GPIO_PC0_DCOM3)}
#endif
};
#endif
/*----------------------------------*/
/** 
 * @brief   DIO Information Table.
 * @details 
 * | [][0] | [][1] | [][2] |
 * | :---  | :---  | :---  |
 * | Group | Num   | Func  |
*/
/*----------------------------------*/
static const uint32_t dioTbl[BSP_DIAGNOSIS_DIGITALIO_MAX][3] =
{
    {(uint32_t)GPIO_PORT_G, (uint32_t)GPIO_PORT_0, (uint32_t)GPIO_PG0_INPUT}
};

/*----------------------------------*/
/** 
 * @brief   DNF Information Table.
 * @details   Detail.
 * | [][0] | 
 * | :---  | 
 * | Num | 
*/
/*-----------------------------------------*/
static const uint32_t dnfTbl[BSP_DNF_MAX][1] =
{
    {(uint32_t)18},
};
/*----------------------------------*/
/** 
 * @brief  RMC Information Table.
 * @details   Detail.
 * | [][0] | 
 * | :---  | 
 * | Num | 
*/
/*-----------------------------------------*/
static const bsp_rmc_info_t rmcTbl[BSP_RMC_MAX]=
{
    {(uint32_t)0,  INTRMC0_IRQn},
};
/**
 *  @}
 */ /* End of group TMPM3HQFDA_EVAL_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup TMPM3HQFDA_EVAL_Private_fuctions TMPM3HQA Evaluation Board Private Fuctions
 *  @{
 */
static void bsp_error(void);
static void fsys_all_enable(void);
static void fsys_all_disable(void);
static void dnf_clock_enable(void);
static void dnf_clock_disable(void);
static void adc_clock_enable(void);
static void adc_clock_disable(void);
static void trace_clock_enable(void);
static void trace_clock_disable(void);
static void port_initialize(void);
static void port_finalize(void);
static void external_losc_enable(void);
static void external_losc_disable(void);

/*--------------------------------------------------*/
/** 
  * @brief  Error Hook
  * @param  -
  * @return -
  * @retval -
  * @note   for debug
  */
/*--------------------------------------------------*/
static void bsp_error(void)
{
#ifdef BSP_DEBUG
    while(1)
    {
        __nop();
    }
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  CG Fsys All Enable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void fsys_all_enable(void)
{
    /*--- [CGFSYSENA] ---*/
    TSB_CG_FSYSENA_IPENA00 = 1;     /* Port A    */
    TSB_CG_FSYSENA_IPENA01 = 1;     /* Port B    */
    TSB_CG_FSYSENA_IPENA02 = 1;     /* Port C    */
    TSB_CG_FSYSENA_IPENA03 = 1;     /* Port D    */
    TSB_CG_FSYSENA_IPENA04 = 1;     /* Port E    */
    TSB_CG_FSYSENA_IPENA05 = 1;     /* Port F    */
    TSB_CG_FSYSENA_IPENA06 = 1;     /* Port G    */
    TSB_CG_FSYSENA_IPENA07 = 1;     /* Port H    */
    TSB_CG_FSYSENA_IPENA08 = 1;     /* Port J    */
    TSB_CG_FSYSENA_IPENA09 = 1;     /* Port K    */
    TSB_CG_FSYSENA_IPENA10 = 1;     /* Port L    */
    TSB_CG_FSYSENA_IPENA11 = 1;     /* Port M    */
    TSB_CG_FSYSENA_IPENA12 = 1;     /* Port N    */
    TSB_CG_FSYSENA_IPENA13 = 1;     /* Port P    */
#if (defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port R is not available. */
#else
    TSB_CG_FSYSENA_IPENA14 = 1;     /* Port R    */
#endif
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port T is not available. */
#else
    TSB_CG_FSYSENA_IPENA15 = 1;     /* Port T    */
#endif
#if (defined(TMPM3HPA) || defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port U is not available. */
#else
    TSB_CG_FSYSENA_IPENA16 = 1;     /* Port U    */
#endif
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port V is not available. */
#else
    TSB_CG_FSYSENA_IPENA17 = 1;     /* Port V    */
#endif
    TSB_CG_FSYSENA_IPENA18 = 1;     /* DMAC 0    */
    TSB_CG_FSYSENA_IPENA19 = 1;     /* DMAC 1    */
    TSB_CG_FSYSENA_IPENA20 = 1;     /* A-PMD     */
    TSB_CG_FSYSENA_IPENA21 = 1;     /* A-ENC     */
    TSB_CG_FSYSENA_IPENA22 = 1;     /* RMC       */
    TSB_CG_FSYSENA_IPENA23 = 1;     /* RTC       */
    TSB_CG_FSYSENA_IPENA24 = 1;     /* T32A 0    */
    TSB_CG_FSYSENA_IPENA25 = 1;     /* T32A 1    */
    TSB_CG_FSYSENA_IPENA26 = 1;     /* T32A 2    */
    TSB_CG_FSYSENA_IPENA27 = 1;     /* T32A 3    */
    TSB_CG_FSYSENA_IPENA28 = 1;     /* T32A 4    */
    TSB_CG_FSYSENA_IPENA29 = 1;     /* T32A 5    */
    TSB_CG_FSYSENA_IPENA30 = 1;     /* T32A 6    */
    TSB_CG_FSYSENA_IPENA31 = 1;     /* T32A 7    */
    /*--- [CGFSYSENB] ---*/
    TSB_CG_FSYSENB_IPENB00 = 1;     /* TSPI 0    */
#if defined(TMPM3HLA)
    /* TSPI 1 is not available. */
#else
    TSB_CG_FSYSENB_IPENB01 = 1;     /* TSPI 1    */
#endif
#if defined(TMPM3HLA)
    /* TSPI 2 is not available. */
#else
    TSB_CG_FSYSENB_IPENB02 = 1;     /* TSPI 2    */
#endif
#if defined(TMPM3HLA)
    /* TSPI 3 is not available. */
#else
    TSB_CG_FSYSENB_IPENB03 = 1;     /* TSPI 3    */
#endif
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* TSPI 4 is not available. */
#else
    TSB_CG_FSYSENB_IPENB04 = 1;     /* TSPI 4    */
#endif
    TSB_CG_FSYSENB_IPENB05 = 1;     /* UART 0    */
    TSB_CG_FSYSENB_IPENB06 = 1;     /* UART 1    */
    TSB_CG_FSYSENB_IPENB07 = 1;     /* UART 2    */
    TSB_CG_FSYSENB_IPENB08 = 1;     /* UART 3    */
    TSB_CG_FSYSENB_IPENB09 = 1;     /* UART 4    */
    TSB_CG_FSYSENB_IPENB10 = 1;     /* UART 5    */
    TSB_CG_FSYSENB_IPENB11 = 1;     /* I2C 0     */
#if defined(TMPM3HLA)
    /* I2C 1 is not available. */
#else
    TSB_CG_FSYSENB_IPENB12 = 1;     /* I2C 1     */
#endif
    TSB_CG_FSYSENB_IPENB13 = 1;     /* I2C 2     */
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* I2C 3 is not available. */
#else
    TSB_CG_FSYSENB_IPENB14 = 1;     /* I2C 3     */
#endif
    TSB_CG_FSYSENB_IPENB15 = 1;     /* ADC       */
    TSB_CG_FSYSENB_IPENB16 = 0;     /* Reserved  */
    TSB_CG_FSYSENB_IPENB17 = 1;     /* DAC 0     */
    TSB_CG_FSYSENB_IPENB18 = 1;     /* DAC 1     */
    TSB_CG_FSYSENB_IPENB19 = 1;     /* RAMPARITY */
    TSB_CG_FSYSENB_IPENB20 = 1;     /* CRC       */
    TSB_CG_FSYSENB_IPENB21 = 1;     /* OFD       */
    TSB_CG_FSYSENB_IPENB22 = 1;     /* TRM       */
    TSB_CG_FSYSENB_IPENB23 = 1;     /* TRGSEL    */
    TSB_CG_FSYSENB_IPENB24 = 1;     /* UART 6    */
#if (defined(TMPM3HNA) || defined(TMPM3HPA) || defined(TMPM3HQA))
    TSB_CG_FSYSENB_IPENB25 = 1;     /* UART 7    */
#endif
    TSB_CG_FSYSENB_IPENB27 = 1;     /* Port W    */
    TSB_CG_FSYSENB_IPENB31 = 1;     /* SIWDT     */
    TSB_CG_FSYSMENB_IPMENB11 = 1;   /* EI2C 0 */
#if defined(TMPM3HLA)
    /* EI2C 1 is not available. */
#else
    TSB_CG_FSYSMENB_IPMENB12 = 1;   /* EI2C 1 */
#endif
    TSB_CG_FSYSMENB_IPMENB13 = 1;   /* EI2C 2 */
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* EI2C 3 is not available. */
#else
    TSB_CG_FSYSMENB_IPMENB14 = 1;   /* EI2C 3 */
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  CG Fsys All Disable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void fsys_all_disable(void)
{
    /*--- [CGFSYSENA] ---*/
    TSB_CG_FSYSENA_IPENA00 = 0;     /* Port A    */
    TSB_CG_FSYSENA_IPENA01 = 0;     /* Port B    */
    TSB_CG_FSYSENA_IPENA02 = 0;     /* Port C    */
    TSB_CG_FSYSENA_IPENA03 = 0;     /* Port D    */
    TSB_CG_FSYSENA_IPENA04 = 0;     /* Port E    */
    TSB_CG_FSYSENA_IPENA05 = 0;     /* Port F    */
    TSB_CG_FSYSENA_IPENA06 = 0;     /* Port G    */
    TSB_CG_FSYSENA_IPENA07 = 0;     /* Port H    */
    TSB_CG_FSYSENA_IPENA08 = 0;     /* Port J    */
    TSB_CG_FSYSENA_IPENA09 = 0;     /* Port K    */
    TSB_CG_FSYSENA_IPENA10 = 0;     /* Port L    */
    TSB_CG_FSYSENA_IPENA11 = 0;     /* Port M    */
    TSB_CG_FSYSENA_IPENA12 = 0;     /* Port N    */
    TSB_CG_FSYSENA_IPENA13 = 0;     /* Port P    */
#if (defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port R is not available. */
#else
    TSB_CG_FSYSENA_IPENA14 = 0;     /* Port R    */
#endif
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port T is not available. */
#else
    TSB_CG_FSYSENA_IPENA15 = 0;     /* Port T    */
#endif
#if (defined(TMPM3HPA) || defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port U is not available. */
#else
    TSB_CG_FSYSENA_IPENA16 = 0;     /* Port U    */
#endif
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port V is not available. */
#else
    TSB_CG_FSYSENA_IPENA17 = 0;     /* Port V    */
#endif
    TSB_CG_FSYSENA_IPENA18 = 0;     /* DMAC 0    */
    TSB_CG_FSYSENA_IPENA19 = 0;     /* DMAC 1    */
    TSB_CG_FSYSENA_IPENA20 = 0;     /* A-PMD     */
    TSB_CG_FSYSENA_IPENA21 = 0;     /* A-ENC     */
    TSB_CG_FSYSENA_IPENA22 = 0;     /* RMC       */
    TSB_CG_FSYSENA_IPENA23 = 0;     /* RTC       */
    TSB_CG_FSYSENA_IPENA24 = 0;     /* T32A 0    */
    TSB_CG_FSYSENA_IPENA25 = 0;     /* T32A 1    */
    TSB_CG_FSYSENA_IPENA26 = 0;     /* T32A 2    */
    TSB_CG_FSYSENA_IPENA27 = 0;     /* T32A 3    */
    TSB_CG_FSYSENA_IPENA28 = 0;     /* T32A 4    */
    TSB_CG_FSYSENA_IPENA29 = 0;     /* T32A 5    */
    TSB_CG_FSYSENA_IPENA30 = 0;     /* T32A 6    */
    TSB_CG_FSYSENA_IPENA31 = 0;     /* T32A 7    */
    /*--- [CGFSYSENB] ---*/
    TSB_CG_FSYSENB_IPENB00 = 0;     /* TSPI 0    */
#if defined(TMPM3HLA)
    /* TSPI 1 is not available. */
#else
    TSB_CG_FSYSENB_IPENB01 = 0;     /* TSPI 1    */
#endif
#if defined(TMPM3HLA)
    /* TSPI 2 is not available. */
#else
    TSB_CG_FSYSENB_IPENB02 = 0;     /* TSPI 2    */
#endif
#if defined(TMPM3HLA)
    /* TSPI 3 is not available. */
#else
    TSB_CG_FSYSENB_IPENB03 = 0;     /* TSPI 3    */
#endif
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* TSPI 4 is not available. */
#else
    TSB_CG_FSYSENB_IPENB04 = 0;     /* TSPI 4    */
#endif
    TSB_CG_FSYSENB_IPENB05 = 0;     /* UART 0    */
    TSB_CG_FSYSENB_IPENB06 = 0;     /* UART 1    */
    TSB_CG_FSYSENB_IPENB07 = 0;     /* UART 2    */
    TSB_CG_FSYSENB_IPENB08 = 0;     /* UART 3    */
    TSB_CG_FSYSENB_IPENB09 = 0;     /* UART 4    */
    TSB_CG_FSYSENB_IPENB10 = 0;     /* UART 5    */
    TSB_CG_FSYSENB_IPENB11 = 0;     /* I2C 0     */
#if defined(TMPM3HLA)
    /* I2C 1 is not available. */
#else
    TSB_CG_FSYSENB_IPENB12 = 0;     /* I2C 1     */
#endif
    TSB_CG_FSYSENB_IPENB13 = 0;     /* I2C 2     */
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* I2C 3 is not available. */
#else
    TSB_CG_FSYSENB_IPENB14 = 0;     /* I2C 3     */
#endif
    TSB_CG_FSYSENB_IPENB15 = 0;     /* ADC       */
    TSB_CG_FSYSENB_IPENB16 = 0;     /* Reserved  */
    TSB_CG_FSYSENB_IPENB17 = 0;     /* DAC 0     */
    TSB_CG_FSYSENB_IPENB18 = 0;     /* DAC 1     */
    TSB_CG_FSYSENB_IPENB19 = 0;     /* RAMPARITY */
    TSB_CG_FSYSENB_IPENB20 = 0;     /* CRC       */
    TSB_CG_FSYSENB_IPENB21 = 0;     /* OFD       */
    TSB_CG_FSYSENB_IPENB22 = 0;     /* TRM       */
    TSB_CG_FSYSENB_IPENB23 = 0;     /* TRGSEL    */
    TSB_CG_FSYSENB_IPENB24 = 0;     /* UART 6    */
#if (defined(TMPM3HNA) || defined(TMPM3HPA) || defined(TMPM3HQA))
    TSB_CG_FSYSENB_IPENB25 = 0;     /* UART 7    */
#endif
    TSB_CG_FSYSENB_IPENB27 = 0;     /* Port W    */
    TSB_CG_FSYSENB_IPENB31 = 0;     /* SIWDT     */
    TSB_CG_FSYSMENB_IPMENB11 = 0;   /* EI2C 0 */
#if defined(TMPM3HLA)
    /* EI2C 1 is not available. */
#else
    TSB_CG_FSYSMENB_IPMENB12 = 0;   /* EI2C 1 */
#endif
    TSB_CG_FSYSMENB_IPMENB13 = 0;   /* EI2C 2 */
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* EI2C 3 is not available. */
#else
    TSB_CG_FSYSMENB_IPMENB14 = 0;   /* EI2C 3 */
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  DNF Clock Enable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void dnf_clock_enable(void)
{
    /*--- [CGFCEN:FCIPEN07] ---*/
    TSB_CG_FCEN_FCIPEN07 = 1;
}

/*--------------------------------------------------*/
/** DNF Clock Disable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void dnf_clock_disable(void)
{
    /*--- [CGFCEN:FCIPEN07] ---*/
    TSB_CG_FCEN_FCIPEN07 = 0;
}

/*--------------------------------------------------*/
/** 
  * @brief  ADC Clock Enable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void adc_clock_enable(void)
{
    /*--- [CGSPCLKEN:ADCKEN] ---*/
    TSB_CG_SPCLKEN_ADCKEN = 1;
}

/*--------------------------------------------------*/
/** 
  * @brief  ADC Clock Disable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void adc_clock_disable(void)
{
    /*--- [CGSPCLKEN:ADCKEN] ---*/
    TSB_CG_SPCLKEN_ADCKEN = 0;
}

/*--------------------------------------------------*/
/** 
  * @brief  ETM Trace Clock Enable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void trace_clock_enable(void)
{
#if defined(TMPM3HLA)
    /* ETM Trace is not available. */
#else
    /*--- [CGSPCLKEN:ADCKEN] ---*/
    TSB_CG_SPCLKEN_TRCKEN = 1;
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  ETM Trace Clock Disable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void trace_clock_disable(void)
{
#if defined(TMPM3HLA)
    /* ETM Trace is not available. */
#else
    /*--- [CGSPCLKEN:ADCKEN] ---*/
    TSB_CG_SPCLKEN_TRCKEN = 0;
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  Port Initialize.
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void port_initialize(void)
{
    /*===============================================================================*/
    /*  [Port A]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PADATA | PACR | PAFRn | PAOD | PAPUP | PADN | PAIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PA0  | -         | O   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PA1  | UT0TXDA   | O   | 0      | 1    | PAFR1 | 0    | 0     | 0    | 0     */
    /*  PA2  | UT0RXD    | I   | 0      | 0    | PAFR1 | 0    | 0     | 0    | 1     */
    /*  PA3  | -         | I   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PA4  | I2C1SCL   | I/O | 1      | 0    | PAFR1 | 0    | 0     | 0    | 1     */
    /*  PA5  | I2C1SDA   | I/O | 1      | 0    | PAFR1 | 0    | 0     | 0    | 1     */
    /*  PA6  | -         | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PA7  | -         | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pa_instance = TSB_PA;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_A) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PA0 -*/
    if (gpio_func(&port, GPIO_PORT_A, GPIO_PORT_0, (uint32_t)GPIO_PA0_T32A00OUTA, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
         bsp_error();
    }
    /*- PA1 -*/
    gpio_write_bit(&port,  GPIO_PORT_A, GPIO_PORT_1, GPIO_Mode_DATA, GPIO_PIN_SET);
    if (gpio_func(&port, GPIO_PORT_A, GPIO_PORT_1, (uint32_t)GPIO_PA1_UT0TXDA, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PA2 -*/
    if (gpio_func(&port, GPIO_PORT_A, GPIO_PORT_2, (uint32_t)GPIO_PA2_UT0RXD, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PA3 -*/
    /* Use reset value.  */
#if defined(TMPM3HLA)
    /* Port A4, A5 is not available. */
#else
#ifdef EI2C_USE
    /*- PA4 -*/
    if (gpio_func(&port, GPIO_PORT_A, GPIO_PORT_4, (uint32_t)GPIO_PA4_EI2C1SCL, GPIO_PIN_INOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PA5 -*/
    if (gpio_func(&port, GPIO_PORT_A, GPIO_PORT_5, (uint32_t)GPIO_PA5_EI2C1SDA, GPIO_PIN_INOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#else
    /*- PA4 -*/
    if (gpio_func(&port, GPIO_PORT_A, GPIO_PORT_4, (uint32_t)GPIO_PA4_I2C1SCL, GPIO_PIN_INOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PA5 -*/
    if (gpio_func(&port, GPIO_PORT_A, GPIO_PORT_5, (uint32_t)GPIO_PA5_I2C1SDA, GPIO_PIN_INOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#endif
#endif
    /*- PA6 -*/
    /* Use reset value.  */
    /*- PA7 -*/
    /* Use reset value.  */
    /*===============================================================================*/
    /*  [Port B]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PBDATA | PBCR | PBFRn | PBOD | PBPUP | PBDN | PBIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PB0  | BOOT_N    | -   | 0      | 0    | 0     | 0    | 0     | 0    | N/A   */
    /*  PB1  | RXIN      | I   | 0      | 0    | PBFR1 | 0    | 0     | 0    | 1     */
    /*  PB2  | TSPI1SCK  | O   | 0      | 1    | PBFR3 | 0    | 0     | 0    | 0     */
    /*  PB3  | TSPI1TXD  | O   | 0      | 1    | PBFR3 | 0    | 0     | 0    | 0     */
    /*  PB4  | TSPI1RXD  | I   | 0      | 0    | PBFR3 | 0    | 0     | 0    | 1     */
    /*  PB5  | TSPI1CS0  | O   | 0      | 1    | 0     | 0    | 0     | 0    | 0     */
    /*  PB6  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PB7  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pb_instance = TSB_PB;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_B) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PB0 -*/
    /* Use reset value.  */
    /*- PB1 -*/
    if (gpio_func(&port, GPIO_PORT_B, GPIO_PORT_1, (uint32_t)GPIO_PB1_RXIN0, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#if defined(TMPM3HMA)
#ifdef TSPI_SENSOR_SAMPLE
    /*- PB2 -*/
    if (gpio_func(&port, GPIO_PORT_B, GPIO_PORT_2, (uint32_t)GPIO_PB2_TSPI1SCK, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PB3 -*/
    if (gpio_func(&port, GPIO_PORT_B, GPIO_PORT_3, (uint32_t)GPIO_PB3_TSPI1TXD, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PB4 -*/
    if (gpio_func(&port, GPIO_PORT_B, GPIO_PORT_4, (uint32_t)GPIO_PB4_TSPI1RXD, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /* Port B5-7 is not available. */
#else
    /*- PB2 -*/
    if (gpio_func(&port, GPIO_PORT_B, GPIO_PORT_2, (uint32_t)GPIO_PB2_TSPI1SCK, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PB3 -*/
    if (gpio_func(&port, GPIO_PORT_B, GPIO_PORT_3, (uint32_t)GPIO_PB3_TSPI1TXD, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PB4 -*/
    if (gpio_func(&port, GPIO_PORT_B, GPIO_PORT_4, (uint32_t)GPIO_PB4_TSPI1RXD, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /* Port B5-7 is not available. */
#endif
#elif defined(TMPM3HLA)
    /*- PB2 -*/
    /* Use reset value.  */
    /*- PB3 -*/
    /* Use reset value.  */
    /* Port B4-7 is not available. */
#else
#ifdef TSPI_SENSOR_SAMPLE
    /*- PB2 -*/
    if (gpio_func(&port, GPIO_PORT_B, GPIO_PORT_2, (uint32_t)GPIO_PB2_TSPI1SCK, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PB3 -*/
    if (gpio_func(&port, GPIO_PORT_B, GPIO_PORT_3, (uint32_t)GPIO_PB3_TSPI1TXD, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PB4 -*/
    if (gpio_func(&port, GPIO_PORT_B, GPIO_PORT_4, (uint32_t)GPIO_PB4_TSPI1RXD, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PB5 -*/
    if (gpio_func(&port, GPIO_PORT_B, GPIO_PORT_5, (uint32_t)GPIO_PB5_OUTPUT, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#else
    /*- PB2 -*/
    if (gpio_func(&port, GPIO_PORT_B, GPIO_PORT_2, (uint32_t)GPIO_PB2_TSPI1SCK, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PB3 -*/
    if (gpio_func(&port, GPIO_PORT_B, GPIO_PORT_3, (uint32_t)GPIO_PB3_TSPI1TXD, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PB4 -*/
    if (gpio_func(&port, GPIO_PORT_B, GPIO_PORT_4, (uint32_t)GPIO_PB4_TSPI1RXD, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PB5 -*/
    if (gpio_func(&port, GPIO_PORT_B, GPIO_PORT_5, (uint32_t)GPIO_PB5_TSPI1CSIN, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#endif
    /*- PB6 -*/
    /* Use reset value.  */
    /*- PB7 -*/
    /* Use reset value.  */
#endif
    /*===============================================================================*/
    /*  [Port C]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PCDATA | PCCR | PCFRn | PCOD | PCPUP | PCDN | PCIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PC0  | I2C0SCL   | I/O | 0      | 1    | PCFR1 | 1    | 0     | 0    | 1     */
    /*  PC1  | ISC0SDA   | I/O | 0      | 1    | PCFR1 | 1    | 0     | 0    | 1     */
    /*  PC2  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PC3  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PC4  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PC5  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PC6  | -         | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pc_instance = TSB_PC;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_C) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
#ifdef EI2C_USE
    /*- PC0 -*/
    if (gpio_func(&port, GPIO_PORT_C, GPIO_PORT_0, (uint32_t)GPIO_PC0_EI2C0SCL, GPIO_PIN_INOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PC1 -*/
    if (gpio_func(&port, GPIO_PORT_C, GPIO_PORT_1, (uint32_t)GPIO_PC1_EI2C0SDA, GPIO_PIN_INOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#else
    /*- PC0 -*/
    if (gpio_func(&port, GPIO_PORT_C, GPIO_PORT_0, (uint32_t)GPIO_PC0_I2C0SCL, GPIO_PIN_INOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PC1 -*/
    if (gpio_func(&port, GPIO_PORT_C, GPIO_PORT_1, (uint32_t)GPIO_PC1_I2C0SDA, GPIO_PIN_INOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#endif
    /*- PC2 -*/
    if (gpio_func(&port, GPIO_PORT_C, GPIO_PORT_2, (uint32_t)GPIO_PC2_INT02, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PC3 -*/
    /* Use reset value.  */
    /*- PC4 -*/
    /* Use reset value.  */
    /*- PC5 -*/
    /* Use reset value.  */
    /*- PC6 -*/
    /* Use reset value.  */
    /*===============================================================================*/
    /*  [Port D]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PDDATA | PDCR | PDFRn | PDOD | PDPUP | PDDN | PDIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PD0  | AINA00    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PD1  | AINA01    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PD2  | AINA02    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PD3  | AINA03    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PD4  | AINA19    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PD5  | AINA20    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pd_instance = TSB_PD;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_D) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PD0 -*/
    /* Use reset value.  */
    /*- PD1 -*/
    /* Use reset value.  */
    /*- PD2 -*/
    /* Use reset value.  */
    /*- PD3 -*/
#if defined(TMPM3HQA)
    /*- PD4 -*/
    if (gpio_func(&port, GPIO_PORT_D, GPIO_PORT_4, (uint32_t)GPIO_PD4_AINA19, GPIO_PIN_NOTINOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PD5 -*/
    if (gpio_func(&port, GPIO_PORT_D, GPIO_PORT_5, (uint32_t)GPIO_PD5_AINA20, GPIO_PIN_NOTINOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#endif
    /*===============================================================================*/
    /*  [Port E]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PEDATA | PECR | PEFRn | PEOD | PEPUP | PEDN | PEIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PE0  | AINA04    | I   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PE1  | AINA05    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PE2  | AINA06    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PE3  | AINA07    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PE4  | AINA08    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PE5  | AINA09    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PE6  | AINA10    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pe_instance = TSB_PE;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_E) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PE0 -*/
    if (gpio_func(&port, GPIO_PORT_E, GPIO_PORT_0, (uint32_t)GPIO_PE0_AINA04, GPIO_PIN_NOTINOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PE1 -*/
    /* Use reset value.  */
    /*- PE2 -*/
    /* Use reset value.  */
    /*- PE3 -*/
    /* Use reset value.  */
    /*- PE4 -*/
    /* Use reset value.  */
    /*- PE5 -*/
    /* Use reset value.  */
    /*- PE6 -*/
    /* Use reset value.  */
    /*===============================================================================*/
    /*  [Port F]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PFDATA | PFCR | PFFRn | PFOD | PFPUP | PFDN | PFIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PF0  | AINA11    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PF1  | AINA12    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PF2  | AINA13    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PF3  | AINA14    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PF4  | AINA15    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PF5  | AINA16    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PF6  | AINA17    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PF7  | AINA18    | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pf_instance = TSB_PF;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_F) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PF0 -*/
    /* Use reset value.  */
    /*- PF1 -*/
    /* Use reset value.  */
    /*- PF2 -*/
    /* Use reset value.  */
    /*- PF3 -*/
    /* Use reset value.  */
    /*- PF4 -*/
    /* Use reset value.  */
    /*- PF5 -*/
    /* Use reset value.  */
    /*- PF6 -*/
    /* Use reset value.  */
    /*- PF7 -*/
    /* Use reset value.  */
    /*===============================================================================*/
    /*  [Port G]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PGDATA | PGCR | PGFRn | PGOD | PGPUP | PGDN | PGIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PG0  | DAC0      | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PG1  | DAC1      | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PG2  |  -        | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PG3  |  -        | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PG4  |  -        | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PG5  |  -        | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PG6  |  -        | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PG7  |  -        | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pg_instance = TSB_PG;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_G) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PG0 -*/
    /* Use reset value.  */
    /*- PG1 -*/
    /* Use reset value.  */
    /*- PG2 -*/
    /* Use reset value.  */
    /*- PG3 -*/
    /* Use reset value.  */
    /*- PG4 -*/
    /* Use reset value.  */
    /*- PG5 -*/
    /* Use reset value.  */
    /*- PG6 -*/
    /* Use reset value.  */
    /*- PG7 -*/
    /* Use reset value.  */
    /*===============================================================================*/
    /*  [Port H]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PHDATA | PHCR | PHFRn | PHOD | PHPUP | PHDN | PHIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PH0  | X1        | I   | 0      | N/A  | N/A   | N/A  | N/A   | 0    | 0     */
    /*  PH1  | X2        | O   | 0      | N/A  | N/A   | N/A  | N/A   | 0    | 0     */
    /*  PH2  | XT1       | I   | 0      | N/A  | N/A   | N/A  | N/A   | 0    | 0     */
    /*  PH3  | XT2       | O   | 0      | N/A  | N/A   | N/A  | N/A   | 0    | 0     */
    /*  PH4  |  -        | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PH5  |  -        | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PH6  |  -        | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PH7  |  -        | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_ph_instance = TSB_PH;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_H) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PH0 -*/
    if (gpio_func(&port, GPIO_PORT_H, GPIO_PORT_0, (uint32_t)GPIO_PH0_X1, GPIO_PIN_NOTINOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PH1 -*/
    if (gpio_func(&port, GPIO_PORT_H, GPIO_PORT_1, (uint32_t)GPIO_PH1_X2, GPIO_PIN_NOTINOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PH2 -*/
    if (gpio_func(&port, GPIO_PORT_H, GPIO_PORT_2, (uint32_t)GPIO_PH2_XT1, GPIO_PIN_NOTINOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PH3 -*/
    if (gpio_func(&port, GPIO_PORT_H, GPIO_PORT_3, (uint32_t)GPIO_PH3_XT2, GPIO_PIN_NOTINOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PH4 -*/
    /* Use reset value.  */
    /*- PH5 -*/
    /* Use reset value.  */
    /*- PH6 -*/
    /* Use reset value.  */
    /*- PH7 -*/
    /* Use reset value.  */
    /*===============================================================================*/
    /*  [Port J]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PJDATA | PJCR | PJFRn | PJOD | PJPUP | PJDN | PJIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PJ0  | T32A3OUTA | O   | 0      | 1    | PJFR3 | 0    | 0     | 0    | 0     */
    /*  PJ1  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PJ2  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PJ3  | T32A3OUTB | O   | 0      | 1    | PJFR3 | 0    | 0     | 0    | 0     */
    /*  PJ4  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PJ5  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pj_instance = TSB_PJ;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_J) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PJ0 -*/
    if (gpio_func(&port, GPIO_PORT_J, GPIO_PORT_0, (uint32_t)GPIO_PJ0_T32A03OUTA, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PJ1 -*/
    /* Use reset value.  */
    /*- PJ2 -*/
    /* Use reset value.  */
    /*- PJ3 -*/
    if (gpio_func(&port, GPIO_PORT_J, GPIO_PORT_3, (uint32_t)GPIO_PJ3_T32A03OUTB, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PJ4 -*/
    /* Use reset value.  */
    /*- PJ5 -*/
    /* Use reset value.  */
    /*===============================================================================*/
    /*  [Port K]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PKDATA | PKCR | PKFRn | PKOD | PKPUP | PKDN | PKIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PK0  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PK1  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PK2  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PK3  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PK4  |PORT_LED[0]| O   | 0      | 1    | 0     | 0    | 0     | 0    | 0     */
    /*  PK5  |PORT_LED[1]| O   | 0      | 1    | 0     | 0    | 0     | 0    | 0     */
    /*  PK6  |PORT_LED[2]| O   | 0      | 1    | 0     | 0    | 0     | 0    | 0     */
    /*  PK7  |PORT_LED[3]| O   | 0      | 1    | 0     | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pk_instance = TSB_PK;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_K) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PK0 -*/
    /* Use reset value.  */
    /*- PK1 -*/
    /* Use reset value.  */
    /*- PK2 -*/
    /* Use reset value.  */
    /*- PK3 -*/
    /* Use reset value.  */
    /*- PK4 -*/
    if (gpio_func(&port, GPIO_PORT_K, GPIO_PORT_4, (uint32_t)GPIO_PK4_OUTPUT, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PK5 -*/
    if (gpio_func(&port, GPIO_PORT_K, GPIO_PORT_5, (uint32_t)GPIO_PK5_OUTPUT, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PK6 -*/
    if (gpio_func(&port, GPIO_PORT_K, GPIO_PORT_6, (uint32_t)GPIO_PK6_OUTPUT, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#if defined(TMPM3HLA)
    /* Port K7 is not available. */
#else
    /*- PK7 -*/
    if (gpio_func(&port, GPIO_PORT_K, GPIO_PORT_7, (uint32_t)GPIO_PK7_OUTPUT, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#endif
    /*===============================================================================*/
    /*  [Port L]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PLDATA | PLCR | PLFRn | PLOD | PLPUP | PLDN | PLIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PL0  | TRST_N    | -   | 0      | 0    | PKFR5 | 0    | 1     | 0    | 1     */
    /*  PL1  | TDI       | -   | 0      | 0    | PKFR5 | 0    | 1     | 0    | 1     */
    /*  PL2  | TDO/SWV   | -   | 0      | 1    | PKFR5 | 0    | 0     | 0    | 0     */
    /*  PL3  | TCK/SWCLK | -   | 0      | 0    | PKFR5 | 0    | 0     | 1    | 1     */
    /*  PL4  | TMS/SWDIO | -   | 0      | 1    | PKFR5 | 0    | 1     | 0    | 1     */
    /*  PL5  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PL6  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PL7  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pl_instance = TSB_PL;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_L) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PL0 -*/
    if (gpio_func(&port, GPIO_PORT_L, GPIO_PORT_0, (uint32_t)GPIO_PL0_EI2C2SCL, GPIO_PIN_INOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PL1 -*/
    if (gpio_func(&port, GPIO_PORT_L, GPIO_PORT_1, (uint32_t)GPIO_PL1_EI2C2SDA, GPIO_PIN_INOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PL2 -*/
    /* Use reset value.  */
    /*- PL3 -*/
    /* Use reset value.  */
    /*- PL4 -*/
    /* Use reset value.  */
    /*- PL5 -*/
    /* Use reset value.  */
    /*- PL6 -*/
    /* Use reset value.  */
    /*- PL7 -*/
    /* Use reset value.  */
    /*===============================================================================*/
    /*  [Port M]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PMDATA | PMCR | PMFRn | PMOD | PMPUP | PMDN | PMIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PM0  | T32A00OUTA| -   | 0      | 0    | PMFR4 | 0    | 0     | 0    | 0     */
    /*  PM1  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PM2  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PM3  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PM4  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PM5  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PM6  | -         | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PM7  | -         | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pm_instance = TSB_PM;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_M) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PM0 -*/
    if (gpio_func(&port, GPIO_PORT_M, GPIO_PORT_0, (uint32_t)GPIO_PM0_T32A00OUTA, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PM1 -*/
    /* Use reset value.  */
    /*- PM2 -*/
    /* Use reset value.  */
    /*- PM3 -*/
    /* Use reset value.  */
    /*- PM4 -*/
    /* Use reset value.  */
    /*- PM5 -*/
    /* Use reset value.  */
    /*- PM6 -*/
    /* Use reset value.  */
    /*- PM7 -*/
    /* Use reset value.  */
    /*===============================================================================*/
    /*  [Port N]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PNDATA | PNCR | PNFRn | PNOD | PNPUP | PNDN | PNIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PN0  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PN1  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PN2  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PN3  | INT10     | I   | 0      | 0    | 0     | 0    | 0     | 0    | 1     */
    /*  PN4  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PN5  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pn_instance = TSB_PN;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_N) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PN0 -*/
    /* Use reset value.  */
    /*- PN1 -*/
    /* Use reset value.  */
    /*- PN2 -*/
    /* Use reset value.  */
    /*- PN3 -*/
    if (gpio_func(&port, GPIO_PORT_N, GPIO_PORT_3, (uint32_t)GPIO_PN3_INT10, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PN4 -*/
    /* Use reset value.  */
    /*- PN5 -*/
    /* Use reset value.  */
    /*===============================================================================*/
    /*  [Port P]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PPDATA | PPCR | PPFRn | PPOD | PPPUP | PPDN | PPIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PP0  | TSPI1CLK  | O   | 0      | 1    | PPFR1 | 0    | 0     | 0    | 0     */
    /*  PP1  | TSPI1TXD  | O   | 0      | 1    | PPFR1 | 0    | 0     | 0    | 0     */
    /*  PP2  | TSPI1RXD  | I   | 0      | 0    | PPFR1 | 0    | 0     | 0    | 1     */
    /*  PP3  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PP4  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PP5  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PP6  |PORT_DSW[0]| I   | 0      | 0    | 0     | 0    | 0     | 0    | 1     */
    /*  PP7  |PORT_DSW[1]| I   | 0      | 0    | 0     | 0    | 0     | 0    | 1     */
    /*===============================================================================*/
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pp_instance = TSB_PP;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_P) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
#if defined(TMPM3HLA)
    /* Port P0-2 is not available. */
#else
    /*- PP0 -*/
    if (gpio_func(&port, GPIO_PORT_P, GPIO_PORT_0, (uint32_t)GPIO_PP0_TSPI2SCK, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PP1 -*/
    if (gpio_func(&port, GPIO_PORT_P, GPIO_PORT_1, (uint32_t)GPIO_PP1_TSPI2TXD, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PP2 -*/
    if (gpio_func(&port, GPIO_PORT_P, GPIO_PORT_2, (uint32_t)GPIO_PP2_TSPI2RXD, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#endif
    /*- PP3 -*/
    /* Use reset value.  */
#if (defined(TMPM3HQA) || defined(TMPM3HPA) || defined(TMPM3HNA))
    /*- PP4 -*/
    /* Use reset value.  */
    /*- PP5 -*/
    /* Use reset value.  */
    /*- PP6 -*/
    if (gpio_func(&port, GPIO_PORT_P, GPIO_PORT_6, (uint32_t)GPIO_PP6_INPUT, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PP7 -*/
    if (gpio_func(&port, GPIO_PORT_P, GPIO_PORT_7, (uint32_t)GPIO_PP7_INPUT, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#endif
    /*===============================================================================*/
    /*  [Port R]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PRDATA | PRCR | PRFRn | PROD | PRPUP | PRDN | PRIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PR0  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PR1  |T32A02INA0 | I   | 0      | 0    | PRFR3 | 0    | 0     | 0    | 1     */
    /*  PR2  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PR3  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PR4  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PR5  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PR6  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PR7  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
#if (defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port R is not available. */
#else
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pr_instance = TSB_PR;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_R) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PR0 -*/
    /* Use reset value.  */
    /*- PR1 -*/
    if (gpio_func(&port, GPIO_PORT_R, GPIO_PORT_1, (uint32_t)GPIO_PR1_T32A02INA0, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PR2 -*/
    /* Use reset value.  */
    /*- PR3 -*/
    /* Use reset value.  */
    /*- PR4 -*/
    /* Use reset value.  */
    /*- PR5 -*/
    /* Use reset value.  */
    /*- PR6 -*/
    /* Use reset value.  */
    /*- PR7 -*/
    /* Use reset value.  */
#endif
    /*===============================================================================*/
    /*  [Port T]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PLDATA | PLCR | PLFRn | PLOD | PLPUP | PLDN | PLIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PT0  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PT1  | TSPI2CS0  | O   | 0      | 1    | PTFR1 | 0    | 0     | 0    | 0     */
    /*  PT2  | TSPI2SCK  | I   | 0      | 0    | PTFR1 | 0    | 0     | 0    | 1     */
    /*  PT3  | TSPI2TXD  | O   | 0      | 1    | PTFR1 | 0    | 0     | 0    | 0     */
    /*  PT4  | TSPI2RXD  | I   | 0      | 0    | PTFR1 | 0    | 0     | 0    | 1     */
    /*  PT5  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PT6  |  -        | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port T is not available. */
#else
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pt_instance = TSB_PT;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_T) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PT0 -*/
    if (gpio_func(&port, GPIO_PORT_T, GPIO_PORT_0, (uint32_t)GPIO_PT0_EI2C3SDA, GPIO_PIN_INOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#ifdef EI2C_USE
    /*- PT1 -*/
    if (gpio_func(&port, GPIO_PORT_T, GPIO_PORT_1, (uint32_t)GPIO_PT1_EI2C3SCL, GPIO_PIN_INOUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#else
    /*- PT1 -*/
    if (gpio_func(&port, GPIO_PORT_T, GPIO_PORT_1, (uint32_t)GPIO_PT1_TSPI2CS0, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#endif
    /*- PT2 -*/
    if (gpio_func(&port, GPIO_PORT_T, GPIO_PORT_2, (uint32_t)GPIO_PT2_TSPI2SCK, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PT3 -*/
    if (gpio_func(&port, GPIO_PORT_T, GPIO_PORT_3, (uint32_t)GPIO_PT3_TSPI2TXD, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#if defined(TMPM3HQA)
    /*- PT4 -*/
    if (gpio_func(&port, GPIO_PORT_T, GPIO_PORT_4, (uint32_t)GPIO_PT4_TSPI2RXD, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PT5 -*/
    /* Use reset value.  */
    /*- PT6 -*/
    /* Use reset value.  */
    /*- PT7 -*/
    /* Use reset value.  */
#endif
#endif
    /*===============================================================================*/
    /*  [Port U]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PUDATA | PUCR | PUFRn | PUOD | PUPUP | PUDN | PUIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PU0  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PU1  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PU2  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PU3  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PU4  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PU5  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
#if (defined(TMPM3HPA) || defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port U is not available. */
#else
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pu_instance = TSB_PU;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_U) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PU0 -*/
    /* Use reset value.  */
    /*- PU1 -*/
    /* Use reset value.  */
    /*- PU2 -*/
    /* Use reset value.  */
    /*- PU3 -*/
    /* Use reset value.  */
    /*- PU4 -*/
    /* Use reset value.  */
    /*- PU5 -*/
    /* Use reset value.  */
#endif
    /*===============================================================================*/
    /*  [Port V]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PLDATA | PLCR | PLFRn | PLOD | PLPUP | PLDN | PLIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PV0  |PORT_PSW[0]| I   | 0      | 0    | 0     | 0    | 0     | 0    | 1     */
    /*  PV1  |PORT_PSW[1]| I   | 0      | 0    | 0     | 0    | 0     | 0    | 1     */
    /*  PV2  |PORT_PSW[2]| I   | 0      | 0    | 0     | 0    | 0     | 0    | 1     */
    /*  PV3  |PORT_PSW[3]| I   | 0      | 0    | 0     | 0    | 0     | 0    | 1     */
    /*  PV4  | -         | -   | 0      | 0    | N/A   | 0    | 0     | 0    | 0     */
    /*  PV5  | -         | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*  PV6  |  -        | -   | 0      | 0    | 0     | 0    | 0     | 0    | 0     */
    /*===============================================================================*/
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port V is not available. */
#else
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pv_instance = TSB_PV;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_V) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PV0 -*/
    if (gpio_func(&port, GPIO_PORT_V, GPIO_PORT_0, (uint32_t)GPIO_PV0_INPUT, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PV1 -*/
    if (gpio_func(&port, GPIO_PORT_V, GPIO_PORT_1, (uint32_t)GPIO_PV1_INPUT, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PV2 -*/
    if (gpio_func(&port, GPIO_PORT_V, GPIO_PORT_2, (uint32_t)GPIO_PV2_INPUT, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PV3 -*/
    if (gpio_func(&port, GPIO_PORT_V, GPIO_PORT_3, (uint32_t)GPIO_PV3_INPUT, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#if defined(TMPM3HQA)
    /*- PV4 -*/
    /* Use reset value.  */
    /*- PV5 -*/
    /* Use reset value.  */
    /*- PV6 -*/
    if (gpio_func(&port, GPIO_PORT_V, GPIO_PORT_6, (uint32_t)GPIO_PV6_UT4TXDA, GPIO_PIN_OUTPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*- PV7 -*/
    if (gpio_func(&port, GPIO_PORT_V, GPIO_PORT_7, (uint32_t)GPIO_PV7_UT4RXD, GPIO_PIN_INPUT) != TXZ_SUCCESS)
    {
        bsp_error();
    }
#endif
#endif
    /*===============================================================================*/
    /*  [Port W]                                                                     */
    /*-------------------------------------------------------------------------------*/
    /*  Port | Function  | I/O | PLDATA | PLCR | PLFRn | PLOD | PLPUP | PLDN | PLIE  */
    /*-------------------------------------------------------------------------------*/
    /*  PW0  | -         | I   | 0      | 0    | N/A   | 0    | 0     | 0    | 1     */
    /*===============================================================================*/
    /*--- Construct  ---*/
    /* Register Allocation */
    port.p_pw_instance = TSB_PW;
    /*--- Initialize ---*/
    if (gpio_init(&port, GPIO_PORT_W) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Setting    ---*/
    /*- PW0 -*/
    /* Use reset value.  */

}

/*--------------------------------------------------*/
/** 
  * @brief  Port Finalize.
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void port_finalize(void)
{
    /*======================*/
    /*  [Port A]            */
    /*======================*/
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_A) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pa_instance = BSP_NULL;
    /*======================*/
    /*  [Port B]            */
    /*======================*/
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_B) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pb_instance = BSP_NULL;
    /*======================*/
    /*  [Port C]            */
    /*======================*/
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_C) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pc_instance = BSP_NULL;
    /*======================*/
    /*  [Port D]            */
    /*======================*/
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_D) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pd_instance = BSP_NULL;
    /*======================*/
    /*  [Port E]            */
    /*======================*/
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_E) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pe_instance = BSP_NULL;
    /*======================*/
    /*  [Port F]            */
    /*======================*/
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_F) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pf_instance = BSP_NULL;
    /*======================*/
    /*  [Port G]            */
    /*======================*/
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_G) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pg_instance = BSP_NULL;
    /*======================*/
    /*  [Port H]            */
    /*======================*/
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_H) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_ph_instance = BSP_NULL;
    /*======================*/
    /*  [Port J]            */
    /*======================*/
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_J) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pj_instance = BSP_NULL;
    /*======================*/
    /*  [Port K]            */
    /*======================*/
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_K) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pk_instance = BSP_NULL;
    /*======================*/
    /*  [Port L]            */
    /*======================*/
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_L) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pl_instance = BSP_NULL;
    /*======================*/
    /*  [Port M]            */
    /*======================*/
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_M) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pm_instance = BSP_NULL;
    /*======================*/
    /*  [Port N]            */
    /*======================*/
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_N) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pn_instance = BSP_NULL;
    /*======================*/
    /*  [Port P]            */
    /*======================*/
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_P) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pp_instance = BSP_NULL;
    /*======================*/
    /*  [Port R]            */
    /*======================*/
#if (defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port R is not available. */
#else
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_R) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pr_instance = BSP_NULL;
#endif
    /*======================*/
    /*  [Port T]            */
    /*======================*/
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port T is not available. */
#else
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_T) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pt_instance = BSP_NULL;
#endif
    /*======================*/
    /*  [Port U]            */
    /*======================*/
#if (defined(TMPM3HPA) || defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port U is not available. */
#else
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_U) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pu_instance = BSP_NULL;
#endif
    /*======================*/
    /*  [Port V]            */
    /*======================*/
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* Port V is not available. */
#else
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_V) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pv_instance = BSP_NULL;
#endif
    /*======================*/
    /*  [Port W]            */
    /*======================*/
    /*--- Finalize ---*/
    if (gpio_deinit(&port, GPIO_PORT_W) != TXZ_SUCCESS)
    {
        bsp_error();
    }
    /*--- Destruct ---*/
    /* Register Release */
    port.p_pw_instance = BSP_NULL;
}

/*--------------------------------------------------*/
/** 
  * @brief  External LOSC Enable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void external_losc_enable(void)
{
    uint32_t work;

    /* [CGWUPLCR]<WUPTL> :Warm up time        */
    /*-----------------------------------------------*/
    /* "1"counter  (s) = 1 / ELOSC                   */
    /* "1"counter (us) = (10^6) / ELOSC              */
    /* "x"counter (us) = time                        */
    /*-----------------------------------------------*/
    /* x : time = 1 : (10^6) / ELOSC                 */
    /*-----------------------------------------------*/
    {
        uint64_t x = (uint64_t)((uint64_t)(ELOSC_CFG_WARM_UP_TIME) * (uint64_t)(ELOSC_CFG_CLOCK));
        x = (uint64_t)((uint64_t)(x / (uint64_t)(1000000)) - 16);
        if (x > (uint64_t)(0x7FFFF))
        {
            /* invalid value */
        }
        work = (uint32_t)x;
    }
    work  &= (uint32_t)(0xFFFFFFF0);
    work <<= 8;
    TSB_CG->WUPLCR = work;
    /* [RLMLOSCCR]<XTEN> :LOSC Enable         */
    TSB_RLM->LOSCCR = RLMLOSCCR_XTEN_RW_ENABLE;
    /* [CGWUPLCR]<WULON> :Enable              */
    work = (uint32_t)(TSB_CG->WUPLCR & CGWUPLCR_WUPTL_HIGH_MASK);
    TSB_CG->WUPLCR = (uint32_t)(work | CGWUPLCR_WULON_W_ENABLE);
    /* [CGWUPLCR]<WULEF> :Read(wait for warm-up) */
    while ((TSB_CG->WUPLCR & CGWUPLCR_WULEF_MASK) != CGWUPLCR_WULEF_R_DONE)
    {
        /* no processing */
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  External LOSC Enable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void external_losc_disable(void)
{
    /* RLMLOSCCR:<XTEN=0>   */
    TSB_RLM->LOSCCR = RLMLOSCCR_XTEN_RW_DISABLE;
}

/**
 *  @}
 */ /* End of group TMPM3HQFDA_EVAL_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup TMPM3HQFDA_EVAL_Exported_functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Initialize.
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void bsp_initialize(void)
{
    /*--- Fsys Enable      ---*/
    fsys_all_enable();
    /*--- DNF Clock Enable ---*/
    dnf_clock_enable();
    /*--- ADC Clock Enable ---*/
    adc_clock_enable();
    /*--- EMT Trace Clock Enable ---*/
    trace_clock_enable();
    /*--- Port             ---*/
    port_initialize();
    /*--- ELOSC Enable     ---*/
    external_losc_enable();
}

/*--------------------------------------------------*/
/** 
  * @brief  Finalize.
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void bsp_finalize(void)
{
    /*--- ELOSC Disable     ---*/
    external_losc_disable();
    /*--- Port              ---*/
    port_finalize();
    /*--- DNF Clock Disable ---*/
    dnf_clock_disable();
    /*--- ADC Clock Disabl ---*/
    adc_clock_disable();
    /*--- EMT Trace Clock Disabl ---*/
    trace_clock_disable();
    /*--- Fsys Disable      ---*/
    fsys_all_disable();
}

/*--------------------------------------------------*/
/** 
  * @brief  Port Driver Instance Addess Get Function.
  * @param  -
  * @return Port Driver Instance Addess.
  * @retval != Null :Valid Value.
  * @note   -
  */
/*--------------------------------------------------*/
gpio_t *bsp_get_gpio_instance_address(void)
{
    return (&port);
}

/*--------------------------------------------------*/
/** 
  * @brief  Port LED GPIO Get Function.
  * @param  led     :Port LED.
  * @param  p_group :GPIO Group Destination Address. \n
  *                 :Value is @ref gpio_gr_t.
  * @param  p_num   :GPIO Number Destination Address.\n
  *                 :Value is @ref gpio_num_t.
  * @return -
  * @retval -
  * @note   You can read the Port LED GPIO Group and GPIO Num.
  */
/*--------------------------------------------------*/
void bsp_get_gpio_led(BSPLed led, uint32_t *p_group, uint32_t *p_num)
{
    *p_group = ledTbl[(uint32_t)led][0];
    *p_num   = ledTbl[(uint32_t)led][1];
}

/*--------------------------------------------------*/
/** 
  * @brief  Port LED GPIO Group Get Function.
  * @param  led     :Port LED.
  * @return Port LED GPIO Group. \n
  *         Value is @ref gpio_gr_t.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_gpio_group_led(BSPLed led)
{
    return (ledTbl[(uint32_t)led][0]);
}

/*--------------------------------------------------*/
/** 
  * @brief  Port LED GPIO Number Get Function.
  * @param  led     :Port LED.
  * @return Port LED GPIO Number. \n
  *         Value is @ref gpio_num_t.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_gpio_num_led(BSPLed led)
{
    return (ledTbl[(uint32_t)led][1]);
}

/*--------------------------------------------------*/
/** 
  * @brief  Push SW GPIO Get Function.
  * @param  psw     :Push SW.
  * @param  p_group :GPIO Group Destination Address. \n
  *                 :Value is @ref gpio_gr_t.
  * @param  p_num   :GPIO Number Destination Address. \n
  *                 :Value is @ref gpio_num_t.
  * @return -
  * @retval -
  * @note   You can read the Push SW GPIO Group and GPIO Num.
  */
/*--------------------------------------------------*/
void bsp_get_gpio_psw(BSPPsw psw, uint32_t *p_group, uint32_t *p_num)
{
    *p_group = pswTbl[(uint32_t)psw][0];
    *p_num   = pswTbl[(uint32_t)psw][1];
}

/*--------------------------------------------------*/
/** 
  * @brief  Push SW GPIO Group Get Function.
  * @param  psw :Push SW.
  * @return Push SW GPIO Group. \n
  *         Value is @ref gpio_gr_t.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_gpio_group_psw(BSPPsw psw)
{
    return (pswTbl[(uint32_t)psw][0]);
}

/*--------------------------------------------------*/
/** 
  * @brief  Push SW GPIO Number Get Function.
  * @param  psw :Push SW.
  * @return Push SW GPIO Number. \n
  *         Value is @ref gpio_num_t.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_gpio_num_psw(BSPPsw psw)
{
    return (pswTbl[(uint32_t)psw][1]);
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART Channel Get Function.
  * @param  uart :USB UART.
  * @return UART Channel.
  * @retval -
  * @note   for Transmission
  */
/*--------------------------------------------------*/
uint32_t bsp_get_usb_uart_tx_ch(BSPUsbUart uart)
{
    return (usbUartTxTbl[(uint32_t)uart][2]);
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART Channel Get Function.
  * @param  uart :USB UART
  * @return UART Channel.
  * @retval -
  * @note   for Receiving
  */
/*--------------------------------------------------*/
uint32_t bsp_get_usb_uart_rx_ch(BSPUsbUart uart)
{
    return (usbUartRxTbl[(uint32_t)uart][2]);
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART Interrupt Number Get Function.
  * @param  uart :USB UART.
  * @return Interrupt Number. \n
  *         Value is @ref IRQn_Type.
  * @retval -
  * @note   for Transmission
  */
/*--------------------------------------------------*/
IRQn_Type bsp_get_usb_uart_tx_nvic(BSPUsbUart uart)
{
    return (INTUART0TX_IRQn);
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART Interrupt Number Get Function.
  * @param  uart :USB UART.
  * @return Interrupt Number. \n
  *         Value is @ref IRQn_Type.
  * @retval -
  * @note   for Receiving
  */
/*--------------------------------------------------*/
IRQn_Type bsp_get_usb_uart_rx_nvic(BSPUsbUart uart)
{
    return (INTUART0RX_IRQn);
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART Interrupt Number Get Function.
  * @param  uart :USB UART.
  * @return Interrupt Number. \n
  *         Value is @ref IRQn_Type.
  * @retval -
  * @note   for Error
  */
/*--------------------------------------------------*/
IRQn_Type bsp_get_usb_uart_err_nvic(BSPUsbUart uart)
{
    return (INTUART0ERR_IRQn);
}


/*--------------------------------------------------*/
/**
 *   * @brief  USB UART Interrupt Number Get Function.
 *     * @param  uart :USB UART.
 *       * @return Interrupt Number. \n
 *         *         Value is @ref IRQn_Type.
 *           * @retval -
 *             * @note   for Transmission
 *               */
/*--------------------------------------------------*/
IRQn_Type bsp_get_uart4_tx_nvic(BSPUsbUart uart)
{
        return (INTUART4TX_IRQn);
}

/*--------------------------------------------------*/
/**
 *   * @brief  USB UART Interrupt Number Get Function.
 *     * @param  uart :USB UART.
 *       * @return Interrupt Number. \n
 *         *         Value is @ref IRQn_Type.
 *           * @retval -
 *             * @note   for Receiving
 *               */
/*--------------------------------------------------*/
IRQn_Type bsp_get_uart4_rx_nvic(BSPUsbUart uart)
{
        return (INTUART4RX_IRQn);
}

/*--------------------------------------------------*/
/**
 *   * @brief  USB UART Interrupt Number Get Function.
 *     * @param  uart :USB UART.
 *       * @return Interrupt Number. \n
 *         *         Value is @ref IRQn_Type.
 *           * @retval -
 *             * @note   for Error
 *               */
/*--------------------------------------------------*/
IRQn_Type bsp_get_uart4_err_nvic(BSPUsbUart uart)
{
        return (INTUART4ERR_IRQn);
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART DMA Channel Get Function.
  * @param  uart :USB UART.
  * @return DMA Channel.
  * @retval -
  * @note   for Transmission
  */
/*--------------------------------------------------*/
uint32_t bsp_get_usb_uart_tx_dma_ch(BSPUsbUart uart)
{
    return (usbUartTxTbl[(uint32_t)uart][3]);
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART DMA Channel Get Function.
  * @param  uart :USB UART.
  * @return DMA Channel.
  * @retval -
  * @note   for Receiving
  */
/*--------------------------------------------------*/
uint32_t bsp_get_usb_uart_rx_dma_ch(BSPUsbUart uart)
{
    return (usbUartRxTbl[(uint32_t)uart][3]);
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART Baudrate Get Function.
  * @param  uart :USB UART.
  * @return Baudrate(bps).
  * @retval -
  * @note   for Transmission
  */
/*--------------------------------------------------*/
uint32_t bsp_get_usb_uart_tx_baudrate(BSPUsbUart uart)
{
    return (usbUartTxTbl[(uint32_t)uart][4]);
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART Baudrate Get Function.
  * @param  uart :USB UART.
  * @return Baudrate(bps).
  * @retval -
  * @note   for Receiving
  */
/*--------------------------------------------------*/
uint32_t bsp_get_usb_uart_rx_baudrate(BSPUsbUart uart)
{
    return (usbUartRxTbl[(uint32_t)uart][4]);
}

/*--------------------------------------------------*/
/** 
  * @brief  TSPI Channel Get Function.
  * @param  tspi :TSPI.
  * @return TSPI Channel.
  * @retval -
  * @note   for Transmission
  */
/*--------------------------------------------------*/
uint32_t bsp_get_tspi_tx_ch(BSPSFlash tspi)
{
    return (tspiTxTbl[(uint32_t)tspi][2]);
}

/*--------------------------------------------------*/
/** 
  * @brief  TSPI Channel Get Function.
  * @param  tspi :TSPI.
  * @return TSPI Channel.
  * @retval -
  * @note   for Transmission
  */
/*--------------------------------------------------*/
uint32_t bsp_get_tspi_clk_ch(BSPSFlash tspi)
{
    return (tspiCLKTbl[(uint32_t)tspi][2]);
}

/*--------------------------------------------------*/
/** 
  * @brief  TSPI Channel Get Function.
  * @param  tspi :TSPI
  * @return TSPI Channel.
  * @retval -
  * @note   for Receiving
  */
/*--------------------------------------------------*/
uint32_t bsp_get_tspi_rx_ch(BSPSFlash tspi)
{
    return (tspiRxTbl[(uint32_t)tspi][2]);
}

/*--------------------------------------------------*/
/** 
  * @brief  TSPI Interrupt Number Get Function.
  * @param  tspi :TSPI.
  * @return Interrupt Number. \n
  *         Value is @ref IRQn_Type.
  * @retval -
  * @note   for Transmission
  */
/*--------------------------------------------------*/
IRQn_Type bsp_get_tspi_tx_nvic(BSPSFlash tspi)
{
#if defined(TMPM3HLA)
    /* TSPI1, TSPI2, TSPI3 is not available.  */
    return (INTT0TX_IRQn);
#else
    IRQn_Type type;
    switch(tspi){
        case BSP_SFLASH_1:
            type = INTT1TX_IRQn;
            break;
        case BSP_SFLASH_2:
            type = INTT2TX_IRQn;
            break;
        case BSP_SFLASH_3:
            type = INTT3TX_IRQn;
            break;
#if (defined(TMPM3HNA) || defined(TMPM3HMA))
        /* TSPI4 is not available.  */
#else
        case BSP_SFLASH_4:
            type = INTT4TX_IRQn;
            break;
#endif
        default:
            break;
    }
    return (type);
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  TSPI Interrupt Number Get Function.
  * @param  tspi :TSPI.
  * @return Interrupt Number. \n
  *         Value is @ref IRQn_Type.
  * @retval -
  * @note   for Receiving
  */
/*--------------------------------------------------*/
IRQn_Type bsp_get_tspi_rx_nvic(BSPSFlash tspi)
{
#if defined(TMPM3HLA)
    /* TSPI1, TSPI2, TSPI3 is not available.  */
    return (INTT0RX_IRQn);
#else
    IRQn_Type type;
    switch(tspi){
        case BSP_SFLASH_1:
            type = INTT1RX_IRQn;
            break;
        case BSP_SFLASH_2:
            type = INTT2RX_IRQn;
            break;
        case BSP_SFLASH_3:
            type = INTT3RX_IRQn;
            break;
#if (defined(TMPM3HNA) || defined(TMPM3HMA))
        /* TSPI4 is not available.  */
#else
        case BSP_SFLASH_4:
            type = INTT4RX_IRQn;
            break;
#endif
        default:
            break;
    }
    return (type);
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  TSPI Interrupt Number Get Function.
  * @param  tspi :TSPI.
  * @return Interrupt Number. \n
  *         Value is @ref IRQn_Type.
  * @retval -
  * @note   for Error
  */
/*--------------------------------------------------*/
IRQn_Type bsp_get_tspi_tx_err_nvic(BSPSFlash tspi)
{
#if defined(TMPM3HLA)
    /* TSPI1, TSPI2, TSPI3 is not available.  */
    return (INTT0ERR_IRQn);
#else
    IRQn_Type type;
    switch(tspi){
        case BSP_SFLASH_1:
            type = INTT1ERR_IRQn;
            break;
        case BSP_SFLASH_2:
            type = INTT2ERR_IRQn;
            break;
        case BSP_SFLASH_3:
            type = INTT3ERR_IRQn;
            break;
#if (defined(TMPM3HNA) || defined(TMPM3HMA))
        /* TSPI4 is not available.  */
#else
        case BSP_SFLASH_4:
            type = INTT4ERR_IRQn;
            break;
#endif
        default:
            break;
    }
    return (type);
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  TSPI Interrupt Number Get Function.
  * @param  tspi :TSPI.
  * @return Interrupt Number. \n
  *         Value is @ref IRQn_Type.
  * @retval -
  * @note   for Error
  */
/*--------------------------------------------------*/
IRQn_Type bsp_get_tspi_rx_err_nvic(BSPSFlash tspi)
{
#if defined(TMPM3HLA)
    /* TSPI1, TSPI2, TSPI3 is not available.  */
    return (INTT0ERR_IRQn);
#else
    IRQn_Type type;
    switch(tspi){
        case BSP_SFLASH_1:
            type = INTT1ERR_IRQn;
            break;
        case BSP_SFLASH_2:
            type = INTT2ERR_IRQn;
            break;
        case BSP_SFLASH_3:
            type = INTT3ERR_IRQn;
            break;
#if (defined(TMPM3HNA) || defined(TMPM3HMA))
        /* TSPI4 is not available.  */
#else
        case BSP_SFLASH_4:
            type = INTT4ERR_IRQn;
            break;
#endif
        default:
            break;
    }
    return (type);
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer Get Function.
  * @param  timer :Timer.
  * @param  p_ch  :Timer Channel Destination Address.
  * @param  p_type:Timer Type Destination Address. \n
  *               :Value is @ref t32_type_t.
  * @return -
  * @retval -
  * @note   You can read the 1ms Timer Channel and Type.
  */
/*--------------------------------------------------*/
void bsp_get_timer(BSPTimer timer, uint32_t *p_ch, uint32_t *p_type)
{
    *p_ch   = timerTbl[(uint32_t)timer].ch;
    *p_type = timerTbl[(uint32_t)timer].type;
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer Channel Get Function.
  * @param  timer :Timer.
  * @return Timer Channel.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_timer_ch(BSPTimer timer)
{
    return (timerTbl[(uint32_t)timer].ch);
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer Type Get Function.
  * @param  timer :Timer.
  * @return Timer Type. \n
  *         Value is @ref t32_type_t.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_timer_type(BSPTimer timer)
{
    return (timerTbl[(uint32_t)timer].type);
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer Interrupt Number Get Function.
  * @param  timer :Timer.
  * @return Interrupt Number. \n
  *         Value is @ref IRQn_Type.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
IRQn_Type bsp_get_timer_nvic(BSPTimer timer)
{
    return (timerTbl[(uint32_t)timer].irq);
}

/*--------------------------------------------------*/
/** 
  * @brief  Output Pulse Channel Get Function.
  * @param  pulse :Output Pulse.
  * @return Output Pulse Channel.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_output_pulse_ch(BSPOutputPulse pulse)
{
    return (pulseOutTbl[(uint32_t)pulse].ch);
}

/*--------------------------------------------------*/
/** 
  * @brief  Output Pulse Type Get Function.
  * @param  pulse :Output Pulse.
  * @return Output Pulse Type. \n
  *         Value is @ref t32_type_t.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_output_pulse_type(BSPOutputPulse pulse)
{
    return (pulseOutTbl[(uint32_t)pulse].type);
}

/*--------------------------------------------------*/
/** 
  * @brief  Output Pulse Interrupt Number Get Function.
  * @param  pulse :Output Pulse.
  * @return Interrupt Number. \n
  *         Value is @ref IRQn_Type.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
IRQn_Type bsp_get_output_pulse_nvic(BSPOutputPulse pulse)
{
    return (pulseOutTbl[(uint32_t)pulse].irq);
}

/*--------------------------------------------------*/
/** 
  * @brief  Thermistors ADIN Get Function
  * @param  thermistors :Thermistors.
  * @return Thermistors ADIN.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_thermistors_adin(BSPThermistors thermistors)
{
    return (thermistorsTbl[(uint32_t)thermistors][2]);
}

/*--------------------------------------------------*/
/** 
  * @brief  Thermistors ADC Channel Get Function
  * @param  thermistors :Thermistors.
  * @return Thermistors ADC Channel.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_thermistors_ch(BSPThermistors thermistors)
{
    return (thermistorsTbl[(uint32_t)thermistors][3]);
}
/*--------------------------------------------------*/
/** 
  * @brief  Thermistors ADC DMA Channel Get Function
  * @param  thermistors :Thermistors.
  * @return Thermistors ADC DMA Channel.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_thermistors_dma_ch(BSPThermistors thermistors)
{
    return (thermistorsTbl[(uint32_t)thermistors][4]);
}

/*--------------------------------------------------*/
/** 
  * @brief  Get Celsius.
  * @param  thermistors :Thermistors.
  * @param  cnt         :AD value.
  * @param  p_degC      :Celsius. Destination address.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Error
  */
/*--------------------------------------------------*/
TXZ_Result bsp_get_thermistors_degC(BSPThermistors thermistors, uint32_t cnt, int32_t *p_degC)
{
    TXZ_Result res = TXZ_ERROR;

    switch (thermistors)
    {
    case BSP_THERMISTORS_0:
    case BSP_THERMISTORS_1:
    case BSP_THERMISTORS_2:
      {
            uint32_t i;
            uint32_t num = (sizeof(NCP15XV103J03RC) / sizeof(NCP15XV103J03RC[0]));

            for (i=0; (i<num) && (res==TXZ_ERROR); i++)
            {
                /* Compare to AD-low & AD-hgh. */
                if (((uint32_t)NCP15XV103J03RC[i][1] <= cnt) && (cnt <= (uint32_t)NCP15XV103J03RC[i][3]))
                {
                    *p_degC = NCP15XV103J03RC[i][0];
                    res     = TXZ_SUCCESS;
                }
            }
        }
        break;
    default:
        /* no processing */
        break;
    }

    return (res);
}

#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HQA) || defined(TMPM3HPA))
/*--------------------------------------------------*/
/** 
  * @brief  DLCD GPIO Group Get Function.
  * @param  dlcd     :DLCD.
  * @return DLCD GPIO Group. \n
  *         Value is @ref gpio_gr_t.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_gpio_group_dlcd(BSPDlcd dlcd)
{
    return (dlcdTbl[(uint32_t)dlcd][0]);
}

/*--------------------------------------------------*/
/** 
  * @brief  DLCD GPIO Number Get Function.
  * @param  dlcd     :DLCD.
  * @return DLCD GPIO Number. \n
  *         Value is @ref gpio_num_t.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_gpio_num_dlcd(BSPDlcd dlcd)
{
    return (dlcdTbl[(uint32_t)dlcd][1]);
}

/*--------------------------------------------------*/
/** 
  * @brief  DLCD GPIO Func Get Function.
  * @param  dlcd     :DLCD.
  * @return DLCD GPIO Func. \n
  *         Value is @ref gpio_pxx_func_t.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_gpio_func_dlcd(BSPDlcd dlcd)
{
    return (dlcdTbl[(uint32_t)dlcd][2]);
}
#endif

/*--------------------------------------------------*/
/** 
  * @brief  DIO GPIO Group Get Function.
  * @param  dio     :DIO.
  * @return DIO GPIO Group. \n
  *         Value is @ref gpio_gr_t.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_gpio_group_dio(BSPDio dio)
{
    return (dioTbl[(uint32_t)dio][0]);
}

/*--------------------------------------------------*/
/** 
  * @brief  DIO GPIO Number Get Function.
  * @param  dio     :DIO.
  * @return DIO GPIO Number. \n
  *         Value is @ref gpio_num_t.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_gpio_num_dio(BSPDio dio)
{
    return (dioTbl[(uint32_t)dio][1]);
}

/*--------------------------------------------------*/
/** 
  * @brief  DIO GPIO Func Get Function.
  * @param  dio     :DIO.
  * @return DIO GPIO Func. \n
  *         Value is @ref gpio_pxx_func_t.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_gpio_func_dio(BSPDio dio)
{
    return (dioTbl[(uint32_t)dio][2]);
}

/*--------------------------------------------------*/
/** 
  * @brief  IRQ RTC Enable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void bsp_irq_rtc_enable(void)
{
    except_t except;
    except.p_instance1 = TSB_IA;
    except.p_instance2 = TSB_IB;

    except_set_int(&except, EXCEPT_RTC, TXZ_ENABLE, EXCEPT_INTMODE_FALLING);
}

/*--------------------------------------------------*/
/** 
  * @brief  IRQ RTC Disable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void bsp_irq_rtc_disable(void)
{
    except_t except;
    except.p_instance1 = TSB_IA;
    except.p_instance2 = TSB_IB;

    except_set_int(&except, EXCEPT_RTC, TXZ_DISABLE, EXCEPT_INTMODE_FALLING);
    except_clear_int(&except, EXCEPT_RTC);
}

/*--------------------------------------------------*/
/** 
  * @brief  IRQ RMC Enable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void bsp_irq_rmc_enable(void)
{
    except_t except;
    except.p_instance1 = TSB_IA;
    except.p_instance2 = TSB_IB;

    except_set_int(&except, EXCEPT_INTRMC0, TXZ_ENABLE, EXCEPT_INTMODE_LOW);
}

/*--------------------------------------------------*/
/** 
  * @brief  IRQ RMC Disable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void bsp_irq_rmc_disable(void)
{
    except_t except;
    except.p_instance1 = TSB_IA;
    except.p_instance2 = TSB_IB;

    except_set_int(&except, EXCEPT_INTRMC0, TXZ_DISABLE, EXCEPT_INTMODE_LOW);
    except_set_int(&except, EXCEPT_INTRMC0, TXZ_ENABLE, EXCEPT_INTMODE_LOW);
}
/*--------------------------------------------------*/
/** 
  * @brief  RMC Channel Get Function.
  * @param  rmc :RMC.
  * @return RMC Channel.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_rmc_ch(BSPRmc rmc)
{
    return (rmcTbl[(uint32_t)rmc].ch);
}

/*--------------------------------------------------*/
/** 
  * @brief  RMC Register Get Function.
  * @param  rmc :RMC.
  * @return RMC Register.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
TSB_RMC_TypeDef * bsp_get_rmc_reg(BSPRmc rmc)
{
    if(rmc == BSP_RMC_SAMPLE)
    {
        return (TSB_RMC0);
    }
    return (0);
}

/*--------------------------------------------------*/
/** 
  * @brief  RMC Interrupt Number Get Function.
  * @param  rmc :RMC.
  * @return Interrupt Number. \n
  *         Value is @ref IRQn_Type.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
IRQn_Type bsp_get_rmc_nvic(BSPRmc rmc)
{
    return (rmcTbl[(uint32_t)rmc].irq);
}
/*--------------------------------------------------*/
/** 
  * @brief  IRQ External Interrupt Enable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void bsp_irq_exint_enable(void)
{
#if defined(TMPM3HLA)
    /* INT02 is not available.  */
#else
    except_t except;
    except.p_instance1 = TSB_IA;
    except.p_instance2 = TSB_IB;

    except_set_int(&except, EXCEPT_INT02, TXZ_ENABLE, EXCEPT_INTMODE_FALLING);
#endif
}
/*--------------------------------------------------*/
/** 
  * @brief  IRQ External Interrupt Enable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void bsp_irq_exint_disable(void)
{
#if defined(TMPM3HLA)
    /* INT02 is not available.  */
#else
    except_t except;
    except.p_instance1 = TSB_IA;
    except.p_instance2 = TSB_IB;

    except_set_int(&except, EXCEPT_INT02, TXZ_DISABLE, EXCEPT_INTMODE_LOW);
    except_clear_int(&except, EXCEPT_INT02);
#endif
}
/*--------------------------------------------------*/
/** 
  * @brief  External Interrupt Number Get Function.
  * @param  dnf :DNF.
  * @return Interrupt Number. \n
  *         Value is @ref IRQn_Type.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
IRQn_Type bsp_get_exint_nvic(void)
{
#if defined(TMPM3HLA)
    /* INT02 is not available.  */
    return (INT01_IRQn);
#else
    return (INT02_IRQn);
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  IRQ External Interrupt (DNF) Enable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void bsp_irq_dnf_enable(void)
{
#if (defined(TMPM3HMA) || defined(TMPM3HLA))
    /* INT18 is not available. */
#else
    except_t except;
    except.p_instance1 = TSB_IA;
    except.p_instance2 = TSB_IB;

    except_set_int(&except, EXCEPT_INT18, TXZ_ENABLE, EXCEPT_INTMODE_FALLING);
#endif
}
/*--------------------------------------------------*/
/** 
  * @brief  IRQ External Interrupt (DNF) Disable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void bsp_irq_dnf_disable(void)
{
#if (defined(TMPM3HMA) || defined(TMPM3HLA))
    /* INT18 is not available. */
#else
    /*-----------------------------*/
    except_t except;
    except.p_instance1 = TSB_IA;
    except.p_instance2 = TSB_IB;
    except_set_int(&except, EXCEPT_INT18, TXZ_DISABLE, EXCEPT_INTMODE_LOW);
    except_set_int(&except, EXCEPT_INT18, TXZ_ENABLE, EXCEPT_INTMODE_LOW);
#endif
}
/*--------------------------------------------------*/
/** 
  * @brief  DNF Register Get Function.
  * @param  dnf :DNF.
  * @return DNF Register.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
TSB_DNF_TypeDef * bsp_get_dnf_reg(BSPDnf dnf)
{
    uint32_t int_num = dnfTbl[(uint32_t)dnf][0];
    if (int_num < 16)
    {
        return (TSB_DNFA);
    }else
    {    
        return (TSB_DNFB);
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  DNF InterruptNumber Get Function.
  * @param  dnf :DNF.
  * @return InterruptNumber Noise Filter Enable Value. \n
  *         Value is @ref t32_type_t.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t bsp_get_dnf_int_num(BSPDnf dnf)
{
    uint32_t num = 0x00000001;
    uint32_t int_num = dnfTbl[(uint32_t)dnf][0];
    if (int_num < 16)
    {
        num = (num << int_num);
    }else
    {
        int_num = int_num -16;
        num = (num << int_num);
    }        
    return (num);
}
/*--------------------------------------------------*/
/** 
  * @brief  DNF Interrupt Number Get Function.
  * @param  dnf :DNF.
  * @return Interrupt Number. \n
  *         Value is @ref IRQn_Type.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
IRQn_Type bsp_get_dnf_nvic(BSPDnf dnf)
{
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* INT18 is not available. */
    return (INT06_IRQn);
#else
    return (INT17_18_32_33_IRQn);
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  Systick IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   Please overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_systick(void)
{
    
}
/*--------------------------------------------------*/
/** 
  * @brief  Push SW State Change.
  * @param  psw :Push SW.
  * @return -
  * @retval -
  * @note   Please overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_psw(BSPPsw psw)
{
    
}
/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   Please overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_usb_uart_rx(BSPUsbUart uart)
{
    
}
/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   Please overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_usb_uart_tx(BSPUsbUart uart)
{
    
}
/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   Please overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_usb_uart_err(BSPUsbUart uart)
{
    
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART4 reception IRQ.
  * @param  uart :USB UART4.
  * @return -
  * @retval -
  * @note   Please overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_uart4_rx(BSPUsbUart uart)
{
    
}
/*--------------------------------------------------*/
/** 
  * @brief  USB UART4 transmit IRQ.
  * @param  uart :USB UART4.
  * @return -
  * @retval -
  * @note   Please overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_uart4_tx(BSPUsbUart uart)
{
    
}
/*--------------------------------------------------*/
/** 
  * @brief  USB UART4 error IRQ.
  * @param  uart :USB UART4.
  * @return -
  * @retval -
  * @note   Please overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_uart4_err(BSPUsbUart uart)
{
    
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer.
  * @param  timer :Timer.
  * @return -
  * @retval -
  * @note   Please overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_timer(BSPTimer timer)
{
    
}

/*--------------------------------------------------*/
/** 
  * @brief  RTC IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   Please overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_rtc(void)
{
    
}
/*--------------------------------------------------*/
/** 
  * @brief  RMC IRQ.
  * @param  status       :RMC status.
  * @return -
  * @retval -
  * @note   Please overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_rmc(uint32_t status)
{
    
}

/*--------------------------------------------------*/
/** 
  * @brief  ADC SGL IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   Please overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_adc_sgl(void)
{
    
}

/*--------------------------------------------------*/
/** 
  * @brief  ADC CNT IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   lease overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_adc_cnt(void)
{
    
}
/*--------------------------------------------------*/
/** 
  * @brief  External Interrupt  IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   lease overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_exint(BSPExInt exi)
{
    
}
/*--------------------------------------------------*/
/** 
  * @brief  TSPI reception IRQ
  * @param  -
  * @return -
  * @retval -
  * @note   lease overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_sflash_rx(BSPSFlash sflash)
{
  
}
/*--------------------------------------------------*/
/** 
  * @brief  TSPI transmission IRQ
  * @param  -
  * @return -
  * @retval -
  * @note   lease overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_sflash_tx(BSPSFlash sflash)
{
  
}
/*--------------------------------------------------*/
/** 
  * @brief  TSPI error IRQ
  * @param  -
  * @return -
  * @retval -
  * @note   lease overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_sflash_err(BSPSFlash sflash)
{
  
}
/*--------------------------------------------------*/
/** 
  * @brief  HDMA Done IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   lease overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_dmac_done(uint32_t ch)
{
  
}

/*--------------------------------------------------*/
/** 
  * @brief  HDMA error IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   lease overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_dmac_err(void)
{
  
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer Counterr IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   lease overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_counter(BSPCounter cnt)
{
  
}

/*--------------------------------------------------*/
/** 
  * @brief  I2C IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   lease overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_i2c(void)
{
  
}

/*--------------------------------------------------*/
/** 
  * @brief  I2C TBE IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   lease overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_i2c_tbe(void)
{
  
}

/*--------------------------------------------------*/
/** 
  * @brief  I2C RBF IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   lease overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_i2c_rbf(void)
{
  
}

/*--------------------------------------------------*/
/** 
  * @brief  Thermistors IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   lease overwrite in user application if you need this function.
  */
/*--------------------------------------------------*/
__weak void irq_thermistors_sgl(BSPThermistors thermistors)
{
  
}


/*--------------------------------------------------*/
/** 
  * @brief  SysTick Handler
  * @param  -
  * @return -
  * @retval -
  */
/*--------------------------------------------------*/
void SysTick_Handler(void)
{
    irq_systick();
}

/*--------------------------------------------------*/
/** 
  * @brief  INT00 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   0: Interrupt Pin0
  */
/*--------------------------------------------------*/
void INT00_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT01 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   1: Interrupt Pin1
  */
/*--------------------------------------------------*/
void INT01_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT02 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   2: Interrupt Pin2
  */
/*--------------------------------------------------*/
void INT02_IRQHandler(void)
{
#if defined(TMPM3HLA)
    /* INT02 is not available.  */
    if(exi==BSP_EXINT_1){ exi =BSP_EXINT_0; }
    else { exi =BSP_EXINT_1; }
#else
    except_t except;
    except.p_instance1 = TSB_IA;
    except.p_instance2 = TSB_IB;

    except_set_int(&except, EXCEPT_INT02, TXZ_ENABLE, EXCEPT_INTMODE_FALLING);
    if(exi==BSP_EXINT_1){ exi =BSP_EXINT_0; }
    else { exi =BSP_EXINT_1; }
    /*--- Handler Call     ---*/
    irq_exint(exi);
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  INT03 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   3: Interrupt Pin3
  */
/*--------------------------------------------------*/
void INT03_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT04 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   4: Interrupt Pin4
  */
/*--------------------------------------------------*/
void INT04_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT05 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   5: Interrupt Pin5
  */
/*--------------------------------------------------*/
void INT05_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT06 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   6: Interrupt Pin6
  */
/*--------------------------------------------------*/
void INT06_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT07 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   7: Interrupt Pin7
  */
/*--------------------------------------------------*/
void INT07_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT08 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   8: Interrupt Pin8
  */
/*--------------------------------------------------*/
void INT08_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT09 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   9: Interrupt Pin9
  */
/*--------------------------------------------------*/
void INT09_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT10 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   10: Interrupt Pin10
  */
/*--------------------------------------------------*/
void INT10_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT11 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   11: Interrupt Pin11
  */
/*--------------------------------------------------*/
void INT11_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT12 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   12: Interrupt Pin12
  */
/*--------------------------------------------------*/
void INT12_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT13 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   13: Interrupt Pin13
  */
/*--------------------------------------------------*/
void INT13_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT14 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   14: Interrupt Pin14
  */
/*--------------------------------------------------*/
void INT14_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT15 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   15: Interrupt Pin15
  */
/*--------------------------------------------------*/
void INT15_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT16 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   16: Interrupt Pin16
  */
/*--------------------------------------------------*/
void INT16_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT17_18 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   17: Interrupt Pin17_18
  */
/*--------------------------------------------------*/
void INT17_18_32_33_IRQHandler(void)
{
#if (defined(TMPM3HNA) || defined(TMPM3HMA) || defined(TMPM3HLA))
    /* INT18 is not available. */
    if(exi==BSP_EXINT_1){ exi =BSP_EXINT_0; }
    else { exi =BSP_EXINT_1; }
#else
    /*--- IBIMC080: INTxCLR ---*/
    /*-----------------------------*/
    /* Bit | Bit Symbol   | Value  */
    /*-----------------------------*/
    /* 7   | INTNCLR      | 1      */
    /* 6   | INTPCLR      | 1      */
    /* 5   | INTNFLG      | 0      */
    /* 4   | INTPFLG      | 0      */
    /* 3-1 | INTMODE[2:0] | 010    */
    /* 0   | INTEN        | 1      */
    /*-----------------------------*/
    except_t except;
    except.p_instance1 = TSB_IA;
    except.p_instance2 = TSB_IB;

    except_set_int(&except, EXCEPT_INT18, TXZ_ENABLE, EXCEPT_INTMODE_FALLING);
    if(exi==BSP_EXINT_1){ exi =BSP_EXINT_0; }
    else { exi =BSP_EXINT_1; }
    /*--- Handler Call     ---*/
    irq_exint(exi);
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  INT19_22 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   18: Interrupt Pin 19_22
  */
/*--------------------------------------------------*/
void INT19_22_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT23_26 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   19: Interrupt Pin 23_26
  */
/*--------------------------------------------------*/
void INT23_26_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT27_28 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   20: Interrupt Pin 27_28
  */
/*--------------------------------------------------*/
void INT27_28_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT29 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   21: Interrupt Pin 29
  */
/*--------------------------------------------------*/
void INT29_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INT30_31 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   22: Interrupt Pin 30_31
  */
/*--------------------------------------------------*/
void INT30_31_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTEMG0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   23: PMD0 EMG interrupt
  */
/*--------------------------------------------------*/
void INTEMG0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTOVV0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   24: PMD0 OVV interrupt
  */
/*--------------------------------------------------*/
void INTOVV0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTPMD0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   25: PMD0 interrupt
  */
/*--------------------------------------------------*/
void INTPMD0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTENC00 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   26: Encoder 0 interrupt 0
  */
/*--------------------------------------------------*/
void INTENC00_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTENC01 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   27: Encoder 0 interrupt 1
  */
/*--------------------------------------------------*/
void INTENC01_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTADAPDA IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   28: ADC conversion triggered by PMD is finished A
  */
/*--------------------------------------------------*/
void INTADAPDA_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTADAPDB IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   29: ADC conversion triggered by PMD is finished B
  */
/*--------------------------------------------------*/
void INTADAPDB_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTADACP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   30: ADC conversion monitoring function interrupt A
  */
/*--------------------------------------------------*/
void INTADACP0_IRQHandler(void)
{
#ifdef ADC_MONITOR_USE
    irq_adc_cp0();
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  INTADACP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   31: ADC conversion monitoring function interrupt B
  */
/*--------------------------------------------------*/
void INTADACP1_IRQHandler(void)
{
#ifdef ADC_MONITOR_USE
    irq_adc_cp1();
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  INTADATRG IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   32: ADC conversion triggered by General purpose is finished
  */
/*--------------------------------------------------*/
void INTADATRG_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTADASGL IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   33: ADC conversion triggered by Single program is finished
  */
/*--------------------------------------------------*/
void INTADASGL_IRQHandler(void)
{
    irq_adc_sgl();
}

/*--------------------------------------------------*/
/** 
  * @brief  INTADACNT IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   34: ADC conversion triggered by Continuity program is finished
  */
/*--------------------------------------------------*/
void INTADACNT_IRQHandler(void)
{
   irq_adc_cnt();
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT0RX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   35: TSPI/SIO reception (channel 0)
  */
/*--------------------------------------------------*/
void INTT0RX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT0TX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   36: TSPI/SIO transmit (channel 0)
  */
/*--------------------------------------------------*/
void INTT0TX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT0ERR IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   37: TSPI/SIO error (channel 0)
  */
/*--------------------------------------------------*/
void INTT0ERR_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT1RX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   38: TSPI/SIO reception (channel 1)
  */
/*--------------------------------------------------*/
void INTT1RX_IRQHandler(void)
{
    irq_sflash_rx(BSP_SFLASH_1);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT1TX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   39: TSPI/SIO transmit (channel 1)
  */
/*--------------------------------------------------*/
void INTT1TX_IRQHandler(void)
{
    irq_sflash_tx(BSP_SFLASH_1);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT1ERR IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   40: TSPI/SIO error (channel 1)
  */
/*--------------------------------------------------*/
void INTT1ERR_IRQHandler(void)
{
    irq_sflash_err(BSP_SFLASH_1);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT2RX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   41: TSPI/SIO reception (channel 2)
  */
/*--------------------------------------------------*/
void INTT2RX_IRQHandler(void)
{
    irq_sflash_rx(BSP_SFLASH_2);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT2TX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   42: TSPI/SIO transmit (channel 2)
  */
/*--------------------------------------------------*/
void INTT2TX_IRQHandler(void)
{
    irq_sflash_tx(BSP_SFLASH_2);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT2ERR IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   43: TSPI/SIO error (channel 2)
  */
/*--------------------------------------------------*/
void INTT2ERR_IRQHandler(void)
{
    irq_sflash_err(BSP_SFLASH_2);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT3RX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   44: TSPI/SIO reception (channel 3)
  */
/*--------------------------------------------------*/
void INTT3RX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT3TX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   45: TSPI/SIO transmit (channel 3)
  */
/*--------------------------------------------------*/
void INTT3TX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT3ERR IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   46: TSPI/SIO error (channel 3)
  */
/*--------------------------------------------------*/
void INTT3ERR_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT4RX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   47: TSPI/SIO reception (channel 4)
  */
/*--------------------------------------------------*/
void INTT4RX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT4TX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   48: TSPI/SIO transmit (channel 4)
  */
/*--------------------------------------------------*/
void INTT4TX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT4ERR IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   49: TSPI/SIO error (channel 4)
  */
/*--------------------------------------------------*/
void INTT4ERR_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2CWUP IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   50: Serial bus interface (WakeUp) interrupt (channel 0)
  */
/*--------------------------------------------------*/
void INTI2CWUP_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C0 I2C0ST IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   51: I2C0 transmission and reception interrupt / EI2C ch0 status interrupt 
  */
/*--------------------------------------------------*/
void INTI2C0NST_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C0AL I2C0TBE IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   52: I2C0 arbitration lost interrupt / EI2C ch0 transmit buffer empty interrupt
  */
/*--------------------------------------------------*/
void INTI2C0ATX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C0BF I2C0RBE IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   53: I2C0 bus free interrupt / EI2C ch0 receive buffer empty interrupt
  */
/*--------------------------------------------------*/
void INTI2C0BRX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C0NA IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   54: I2C0 no ack interrupt
  */
/*--------------------------------------------------*/
void INTI2C0NA_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C1 I2C1ST IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   55: I2C1 transmission and reception interrupt / EI2C ch1 status interrupt
  */
/*--------------------------------------------------*/
void INTI2C1NST_IRQHandler(void)
{
    irq_i2c();
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C1AL I2C1TBE IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   56: I2C1 arbitration lost interrupt / EI2C ch1 transmit buffer empty interrupt
  */
/*--------------------------------------------------*/
void INTI2C1ATX_IRQHandler(void)
{
    /* no process */
    irq_i2c_tbe();
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C1BF I2C1RBE IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   57: I2C1 bus free interrupt / EI2C ch1 receive buffer empty interrupt
  */
/*--------------------------------------------------*/
void INTI2C1BRX_IRQHandler(void)
{
    /* no process */
    irq_i2c_rbf();
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C1NA IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   58: I2C1 no ack interrupt
  */
/*--------------------------------------------------*/
void INTI2C1NA_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C2 I2C2ST IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   59: I2C2 transmission and reception interrupt / EI2C ch2 status interrupt
  */
/*--------------------------------------------------*/
void INTI2C2NST_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C2AL I2C2TBE IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   60: I2C2 arbitration lost interrupt / EI2C ch2 transmit buffer empty interrupt
  */
/*--------------------------------------------------*/
void INTI2C2ATX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C2BF I2C2RBE IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   61: I2C2 bus free interrupt / EI2C ch2 receive buffer empty interrupt
  */
/*--------------------------------------------------*/
void INTI2C2BRX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C2NA IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   62: I2C2 no ack interrupt
  */
/*--------------------------------------------------*/
void INTI2C2NA_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C3 I2C3ST IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   63: I2C3 transmission and reception interrupt / EI2C ch3 status interrupt
  */
/*--------------------------------------------------*/
void INTI2C3NST_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C3AL I2C3TBE IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   64: I2C3 arbitration lost interrupt / EI2C ch3 transmit buffer empty interrupt
  */
/*--------------------------------------------------*/
void INTI2C3ATX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C3BF I2C3RBE IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   65: I2C3 bus free interrupt / EI2C ch3 receive buffer empty interrupt
  */
/*--------------------------------------------------*/
void INTI2C3BRX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTI2C3NA IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   66: I2C3 no ack interrupt
  */
/*--------------------------------------------------*/
void INTI2C3NA_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART0RX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   67: UART reception (channel 0)
  */
/*--------------------------------------------------*/
void INTUART0RX_IRQHandler(void)
{
    irq_usb_uart_rx(BSP_USB_UART_0);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART0TX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   68: UART transmit (channel 0)
  */
/*--------------------------------------------------*/
void INTUART0TX_IRQHandler(void)
{
    irq_usb_uart_tx(BSP_USB_UART_0);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART0ERR IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   69: UART error (channel 0)
  */
/*--------------------------------------------------*/
void INTUART0ERR_IRQHandler(void)
{
    irq_usb_uart_err(BSP_USB_UART_0);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART1RX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   70: UART reception (channel 1)
  */
/*--------------------------------------------------*/
void INTUART1RX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART1TX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   71: UART transmit (channel 1)
  */
/*--------------------------------------------------*/
void INTUART1TX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART1ERR IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   72: UART error (channel 1)
  */
/*--------------------------------------------------*/
void INTUART1ERR_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART2RX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   73: UART reception (channel 2)
  */
/*--------------------------------------------------*/
void INTUART2RX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART2TX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   74: UART transmit (channel 2)
  */
/*--------------------------------------------------*/
void INTUART2TX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART2ERR IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   75: UART error (channel 2)
  */
/*--------------------------------------------------*/
void INTUART2ERR_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART3RX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   76: UART reception (channel 3)
  */
/*--------------------------------------------------*/
void INTUART3RX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART3TX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   77: UART transmit (channel 3)
  */
/*--------------------------------------------------*/
void INTUART3TX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART3ERR IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   78: UART error (channel 3)
  */
/*--------------------------------------------------*/
void INTUART3ERR_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART4RX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   79: UART reception (channel 4)
  */
/*--------------------------------------------------*/
void INTUART4RX_IRQHandler(void)
{
    irq_uart4_rx(BSP_USB_UART_4);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART4TX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   80: UART transmit (channel 4)
  */
/*--------------------------------------------------*/
void INTUART4TX_IRQHandler(void)
{
    irq_uart4_tx(BSP_USB_UART_4);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART4ERR IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   81: UART error (channel 4)
  */
/*--------------------------------------------------*/
void INTUART4ERR_IRQHandler(void)
{
    irq_uart4_err(BSP_USB_UART_4);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART5RX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   82: UART reception (channel 5)
  */
/*--------------------------------------------------*/
void INTUART5RX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART5TX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   83: UART transmit (channel 5)
  */
/*--------------------------------------------------*/
void INTUART5TX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART5ERR IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   84: UART error (channel 5)
  */
/*--------------------------------------------------*/
void INTUART5ERR_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT320A0A IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   85: 32bit T32A00A compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A00A_IRQHandler(void)
{
    irq_timer(BSP_TIMER_1MS);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A00ACAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   86: 32bit T32A00A input capture 0
  */
/*--------------------------------------------------*/
void INTT32A00ACAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A00ACAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   87: 32bit T32A00A input capture 1
  */
/*--------------------------------------------------*/
void INTT32A00ACAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A00B IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   88: 32bit T32A00B compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A00B_IRQHandler(void)
{
    irq_timer(BSP_TIMER_1MS);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A00BCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   89: 32bit T32A00B input capture 0
  */
/*--------------------------------------------------*/
void INTT32A00BCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A00BCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   90: 32bit T32A00B input capture 1
  */
/*--------------------------------------------------*/
void INTT32A00BCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A00C IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   91: 32bit T32A0C compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A00C_IRQHandler(void)
{
    irq_timer(BSP_TIMER_1MS);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A00CCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   92: 32bit T32A00C input capture 0
  */
/*--------------------------------------------------*/
void INTT32A00CCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A00CCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   93: 32bit T32A00C input capture 1
  */
/*--------------------------------------------------*/
void INTT32A00CCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A01A IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   94: 32bit T32A01A compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A01A_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A01ACAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   95: 32bit T32A01A input capture 0
  */
/*--------------------------------------------------*/
void INTT32A01ACAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A01ACAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   96: 32bit T32A01A input capture 1
  */
/*--------------------------------------------------*/
void INTT32A01ACAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A01B IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   97: 32bit T32A01B compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A01B_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A01BCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   98: 32bit T32A01B input capture 0
  */
/*--------------------------------------------------*/
void INTT32A01BCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A01BCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   99: 32bit T32A01B input capture 1
  */
/*--------------------------------------------------*/
void INTT32A01BCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A01C IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  100: 32bit T32A01C compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A01C_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A01CCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  101: 32bit T32A01C input capture 0
  */
/*--------------------------------------------------*/
void INTT32A01CCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A01CCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  102: 32bit T32A01C input capture 1
  */
/*--------------------------------------------------*/
void INTT32A01CCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A02A IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  103: 32bit T32A02A compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A02A_IRQHandler(void)
{
    irq_timer(BSP_TIMER_1S);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A02ACAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  104: 32bit T32A02A input capture 0
  */
/*--------------------------------------------------*/
void INTT32A02ACAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A02ACAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  105: 32bit T32A02A input capture 1
  */
/*--------------------------------------------------*/
void INTT32A02ACAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A02B IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  106: 32bit T32A02B compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A02B_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A02BCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  107: 32bit T32A02B input capture 0
  */
/*--------------------------------------------------*/
void INTT32A02BCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A02BCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  108: 32bit T32A02B input capture 1
  */
/*--------------------------------------------------*/
void INTT32A02BCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A02C IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  109: 32bit T32A02C compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A02C_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A02CCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  110: 32bit T32A02C input capture 0
  */
/*--------------------------------------------------*/
void INTT32A02CCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A02CCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  111: 32bit T32A02C input capture 1
  */
/*--------------------------------------------------*/
void INTT32A02CCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A03A IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  112: 32bit T32A03A compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A03A_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A03ACAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  113: 32bit T32A03A input capture 0
  */
/*--------------------------------------------------*/
void INTT32A03ACAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A03ACAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  114: 32bit T32A03A input capture 1
  */
/*--------------------------------------------------*/
void INTT32A03ACAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A03B IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  115: 32bit T32A03B compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A03B_IRQHandler(void)
{

}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A03BCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  116: 32bit T32A03B input capture 0
  */
/*--------------------------------------------------*/
void INTT32A03BCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A03BCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   117: 32bit T32A03B input capture 1
  */
/*--------------------------------------------------*/
void INTT32A03BCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A03C IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  118: 32bit T32A03C compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A03C_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A03CCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  119: 32bit T32A03C input capture 0
  */
/*--------------------------------------------------*/
void INTT32A03CCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A03CCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  120: 32bit T32A03C input capture 1
  */
/*--------------------------------------------------*/
void INTT32A03CCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A04A IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  121: 32bit T32A04A compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A04A_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A04ACAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  122: 32bit T32A04A input capture 0
  */
/*--------------------------------------------------*/
void INTT32A04ACAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A04ACAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  123: 32bit T32A04A input capture 1
  */
/*--------------------------------------------------*/
void INTT32A04ACAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A04B IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  124: 32bit T32A04B compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A04B_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A04BCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  125: 32bit T32A04B input capture 0
  */
/*--------------------------------------------------*/
void INTT32A04BCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A04BCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  126: 32bit T32A04B input capture 1
  */
/*--------------------------------------------------*/
void INTT32A04BCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A04C IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  127: 32bit T32A04C compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A04C_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A04CCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note  128: 32bit T32A04C input capture 0
  */
/*--------------------------------------------------*/
void INTT32A04CCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A04CCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   129: 32bit T32A04C input capture 1
  */
/*--------------------------------------------------*/
void INTT32A04CCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A05A IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   130: 32bit T32A05A compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A05A_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A05ACAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   131: 32bit T32A05A input capture 0
  */
/*--------------------------------------------------*/
void INTT32A05ACAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A05ACAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   132: 32bit T32A05A input capture 1
  */
/*--------------------------------------------------*/
void INTT32A05ACAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A05B IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   133: 32bit T32A05B compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A05B_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A05BCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   134: 32bit T32A05B input capture 0
  */
/*--------------------------------------------------*/
void INTT32A05BCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A05BCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   135: 32bit T32A05B input capture 1
  */
/*--------------------------------------------------*/
void INTT32A05BCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A05C IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   136: 32bit T32A05C compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A05C_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A05CCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   137: 32bit T32A05C input capture 0
  */
/*--------------------------------------------------*/
void INTT32A05CCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A05CCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   138: 32bit T32A05C input capture 1
  */
/*--------------------------------------------------*/
void INTT32A05CCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A06A IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   139: 32bit T32A06A compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A06A_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A06ACAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   140: 32bit T32A06A input capture 0
  */
/*--------------------------------------------------*/
void INTT32A06ACAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A06ACAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   141: 32bit T32A06A input capture 1
  */
/*--------------------------------------------------*/
void INTT32A06ACAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A06B IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   142: 32bit T32A06B compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A06B_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A06BCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   143: 32bit T32A06B input capture 0
  */
/*--------------------------------------------------*/
void INTT32A06BCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A06BCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   144: 32bit T32A06B input capture 1
  */
/*--------------------------------------------------*/
void INTT32A06BCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A06C IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   145: 32bit T32A06C compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A06C_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A06CCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   146: 32bit T32A06C input capture 0
  */
/*--------------------------------------------------*/
void INTT32A06CCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A06CCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   147: 32bit T32A06C input capture 1
  */
/*--------------------------------------------------*/
void INTT32A06CCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A07A IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   148: 32bit T32A07A compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A07A_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A07ACAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   149: 32bit T32A07A input capture 0
  */
/*--------------------------------------------------*/
void INTT32A07ACAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A07ACAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   150: 32bit T32A07A input capture 1
  */
/*--------------------------------------------------*/
void INTT32A07ACAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A07B IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   151: 32bit T32A07B compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A07B_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A07BCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   152: 32bit T32A07B input capture 0
  */
/*--------------------------------------------------*/
void INTT32A07BCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A07BCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   153: 32bit T32A07B input capture 1
  */
/*--------------------------------------------------*/
void INTT32A07BCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A07C IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   154: 32bit T32A07C compare match detection 0 / Over flow / under flow
  */
/*--------------------------------------------------*/
void INTT32A07C_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A07CCAP0 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   155: 32bit T32A07C input capture 0
  */
/*--------------------------------------------------*/
void INTT32A07CCAP0_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTT32A07CCAP1 IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   156: 32bit T32A07C input capture 1
  */
/*--------------------------------------------------*/
void INTT32A07CCAP1_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTPARI IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   157: RAM parity interrupt
  */
/*--------------------------------------------------*/
void INTPARI_IRQHandler(void)
{
    /* no process */
}


/*--------------------------------------------------*/
/** 
  * @brief  INTDMAATCINTDMACTC IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   158: DMAA end of transfer
  */
/*--------------------------------------------------*/
void INTDMAATC_IRQHandler(void)
{
    uint32_t i;
    uint32_t flag;
    uint32_t mask;
    except_t except;
    except.p_instance1 = TSB_IA;
    except.p_instance2 = TSB_IB;
    except.p_instance3 = TSB_IMN;

    for (i=0; i<DMA_CH_NUM; i++)
    {
        /* Read the interrupt flag(IMNFLG3).  */
        flag = except_get_int(&except, (i+EXCEPT_INTDMAATC00));

        /* Check the DMA channel.  */
        mask = (uint32_t)(1 << i);
        if ((flag & mask) == mask)
        {
            /* Clear the interrupt flag. */
            /* DMAC ch A(ch0 -ch31) [IBIMC000] - [IBIMC031] */
            except_clear_int(&except, (i+EXCEPT_INTDMAATC00));

            /* Call Handler. */
            irq_dmac_done(i);
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  INTDMAAERR IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   159: DMAA transfer error
  */
/*--------------------------------------------------*/
void INTDMAAERR_IRQHandler(void)
{
    irq_dmac_err();
}


/*--------------------------------------------------*/
/** 
  * @brief  INTDMABTCINTDMACTC IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   160: DMAB end of transfer
  */
/*--------------------------------------------------*/
void INTDMABTC_IRQHandler(void)
{
    uint32_t i;
    uint32_t flag4;
    uint32_t flag5;
    uint32_t mask;
    except_t except;
    except.p_instance1 = TSB_IA;
    except.p_instance2 = TSB_IB;
    except.p_instance3 = TSB_IMN;

    /* Read the interrupt flag(IMNFLG5).  */
    flag5 = except_get_int(&except, EXCEPT_INTDMABTC31);

    if((flag5 & EXCEPT_INT160FLG_DETECT) == EXCEPT_INT160FLG_DETECT)
    {
        except_clear_int(&except, EXCEPT_INTDMABTC31);
        /* Call Handler. */
        irq_dmac_done(31);
    }
    for (i=0; i<DMA_CH_NUM; i++)
    {
        /* Read the interrupt flag(IMNFLG4).  */
        flag4 = except_get_int(&except, (i+EXCEPT_INTDMABTC00));

        /* Check the DMA channel.  */
        mask = (uint32_t)(1 << (i + 1));
        if ((flag4 & mask) == mask)
        {
            /* Clear the interrupt flag. */
            /* DMAC ch B(ch0 -ch30) [IBIMC033] - [IBIMC063] */
            except_clear_int(&except, (i+EXCEPT_INTDMABTC00));
            /* Call Handler. */
            irq_dmac_done(i);
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  INTDMABERR IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   161: DMAB transfer error
  */
/*--------------------------------------------------*/
void INTDMABERR_IRQHandler(void)
{
    irq_dmac_err();
}
/*--------------------------------------------------*/
/** 
  * @brief  INTRTC IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   162: Real time clock interrupt 
  */
/*--------------------------------------------------*/
void INTRTC_IRQHandler(void)
{
    except_t except;
    except.p_instance1 = TSB_IA;
    except.p_instance2 = TSB_IB;

    except_set_int(&except, EXCEPT_RTC, TXZ_ENABLE, EXCEPT_INTMODE_FALLING);
    /*--- Handler Call     ---*/
    irq_rtc();
}

/*--------------------------------------------------*/
/** 
  * @brief  INTRMC IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   163: Remote control reception interrupt
  */
/*--------------------------------------------------*/
void INTRMC0_IRQHandler(void)
{
    uint32_t status = 0;
    except_t except;
    except.p_instance1 = TSB_IA;
    except.p_instance2 = TSB_IB;

    status = TSB_RMC0->RSTAT;
    except_set_int(&except, EXCEPT_INTRMC0, TXZ_ENABLE, EXCEPT_INTMODE_LOW);
    /*--- Handler Call     ---*/
    irq_rmc(status);
}

/*--------------------------------------------------*/
/** 
  * @brief  INTFLCRDY IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   164: Code FLASH Ready interrupt
  */
/*--------------------------------------------------*/
void INTFLCRDY_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTFLDRDY IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   165: Data FLASH Ready interrupt
  */
/*--------------------------------------------------*/
void INTFLDRDY_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTLCDBUSF IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   166: LCD register write end interrupt
  */
/*--------------------------------------------------*/
void INTLCDBUSF_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTLCDSTOP IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   167: LCD stop interrupt
  */
/*--------------------------------------------------*/
void INTLCDSTOP_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART6RX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   168: UART reception (channel 6)
  */
/*--------------------------------------------------*/
void INTUART6RX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART6TX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   169: UART transmit (channel 6)
  */
/*--------------------------------------------------*/
void INTUART6TX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART6ERR IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   170: UART error (channel 6)
  */
/*--------------------------------------------------*/
void INTUART6ERR_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART7RX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   171: UART reception (channel 7)
  */
/*--------------------------------------------------*/
void INTUART7RX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART7TX IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   172: UART transmit (channel 7)
  */
/*--------------------------------------------------*/
void INTUART7TX_IRQHandler(void)
{
    /* no process */
}

/*--------------------------------------------------*/
/** 
  * @brief  INTUART7ERR IRQ Handler
  * @param  -
  * @return -
  * @retval -
  * @note   173: UART error (channel 7)
  */
/*--------------------------------------------------*/
void INTUART7ERR_IRQHandler(void)
{
    /* no process */
}

/**
 *  @}
 */ /* End of group TMPM3HQFDA_EVAL_Exported_functions */

/**
 *  @}
 */ /* End of group TMPM3HQFDA_EVAL */

/**
 *  @} 
 */ /* End of group Example */

#endif /* defined(__BSP_H)  */
#ifdef __cplusplus
}
#endif /* __cplusplus */
