/** 
 *******************************************************************************
 * @file    bsp_config_adbun_m3hqa_dlcd.c
 * @brief   This file provides API functions for BSP DLCD driver.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "bsp_dlcd.h"

#if defined(__BSP_CONFIG_ADBUN_M3HQA_DLCD_H)

/**
 *  @addtogroup BSP BSP
 *  @{
 */

/** 
 *  @addtogroup BSP_DLCD
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_DLCD_Private_macro
 *  @{
 */

/** 
 *  @}
 */ /* End of group BSP_DLCD_Private_macro */

/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_DLCD_Private_define
 *  @{
 */

/** 
 *  @}
 */ /* End of group BSP_DLCD_Private_define */

/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_DLCD_Private_define
 *  @{
 */
#define WU_TIME_FS  ((uint32_t)(5000))      /*!< Warm up time(us). */
/** 
 *  @}
 */ /* End of group BSP_DLCD_Private_define */

/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_DLCD_Private_define
 *  @{
 */
/** 
 *  @}
 */ /* End of group BSP_DLCD_Private_define */

/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_DLCD_Private_typedef
 *  @{
 */
/** 
 *  @}
 */ /* End of group BSP_DLCD_Private_typedef */

/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_DLCD_Private_variables
 *  @{
 */

/**
 *  @}
 */ /* End of group BSP_DLCD_Private_variables */

/*------------------------------------------------------------------------------*/
/*  Const Table                                                                 */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_DLCD_Private_const
 *  @{
 */

/*----------------------------------*/
/** 
 * @brief   DLCD "A" display Table.
 * @details 
 * | [][0] | [][1] |
 * | :---  | :---  |
 * | Buf   | Com   |
*/
/*----------------------------------*/
static const uint32_t DlcdMbedSwATbl[BSP_DLCD_MbedSw_A_MAX][2] =
{
    {(uint32_t)DLCD_BUF30, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 1A, 1B, 1C */
    {(uint32_t)DLCD_BUF36, (uint8_t)(DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},                    /* 1E, 1F */
    {(uint32_t)DLCD_BUF37, (uint8_t)DLCD_SEGL_COM1},                                       /* 1J */
    {(uint32_t)DLCD_BUF31, (uint8_t)DLCD_SEGL_COM2}                                        /* 1N */
};

/*----------------------------------*/
/** 
 * @brief   DLCD "d" display Table.
 * @details 
 * | [][0] | [][1] |
 * | :---  | :---  |
 * | Buf   | Com   |
*/
/*----------------------------------*/
static const uint32_t DlcdMbedSwdTbl[BSP_DLCD_MbedSw_d_MAX][2] =
{
    {(uint32_t)DLCD_BUF28, (uint8_t)(DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 2B, 2C */
    {(uint32_t)DLCD_BUF38, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1)},   /* 2D, 2E */
    {(uint32_t)DLCD_BUF39, (uint8_t)DLCD_SEGL_COM1},                      /* 2J */
    {(uint32_t)DLCD_BUF29, (uint8_t)DLCD_SEGL_COM2}                       /* 2N */
};

/*----------------------------------*/
/** 
 * @brief   DLCD "c." display Table.
 * @details 
 * | [][0] | [][1] |
 * | :---  | :---  |
 * | Buf   | Com   |
*/
/*----------------------------------*/
static const uint32_t DlcdMbedSwcTbl[BSP_DLCD_MbedSw_c_MAX][2] =
{
    {(uint32_t)DLCD_BUF04, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1)},   /* 3D, 3E */
    {(uint32_t)DLCD_BUF05, (uint8_t)DLCD_SEGL_COM1},                      /* 3J */
    {(uint32_t)DLCD_BUF27, (uint8_t)DLCD_SEGL_COM2},                      /* 3N */
    {(uint32_t)DLCD_BUF26, (uint8_t)DLCD_SEGL_COM0}                       /* 3P */
};

/*----------------------------------*/
/** 
 * @brief   DLCD ":" display Table.
 * @details 
 * | [][0] | [][1] |
 * | :---  | :---  |
 * | Buf   | Com   |
*/
/*----------------------------------*/
static const uint32_t DlcdMbedSwSemicoronTbl[BSP_DLCD_MbedSw_Semicoron_MAX][2] =
{
    {(uint32_t)DLCD_BUF07, (uint8_t)DLCD_SEGL_COM3},   /* 4H */
    {(uint32_t)DLCD_BUF25, (uint8_t)DLCD_SEGL_COM0}    /* 4L */
};

/*----------------------------------*/
/** 
 * @brief   DLCD Left Number display Table.
 * @details 
 * | [][0] | [][1] |
 * | :---  | :---  |
 * | Buf   | Com   |
*/
/*----------------------------------*/
static const uint32_t DlcdMbedSwLeft0Tbl[BSP_DLCD_MbedSw_Left0_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 5A, 5B, 5C */
    {(uint32_t)DLCD_BUF08, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)}    /* 5D, 5E, 5F */
};

static const uint32_t DlcdMbedSwLeft1Tbl[BSP_DLCD_MbedSw_Left1_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)(DLCD_SEGL_COM2 | DLCD_SEGL_COM1)}    /* 5B, 5C */
};

static const uint32_t DlcdMbedSwLeft2Tbl[BSP_DLCD_MbedSw_Left2_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2)},   /* 5A, 5B */
    {(uint32_t)DLCD_BUF08, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1)},   /* 5D, 5E */
    {(uint32_t)DLCD_BUF09, (uint8_t)DLCD_SEGL_COM1},                      /* 5J */
    {(uint32_t)DLCD_BUF23, (uint8_t)DLCD_SEGL_COM2}                       /* 5N */
};

static const uint32_t DlcdMbedSwLeft3Tbl[BSP_DLCD_MbedSw_Left3_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 5A, 5B, 5C */
    {(uint32_t)DLCD_BUF08, (uint8_t)DLCD_SEGL_COM0},                                       /* 5D */
    {(uint32_t)DLCD_BUF09, (uint8_t)DLCD_SEGL_COM1},                                       /* 5J */
    {(uint32_t)DLCD_BUF23, (uint8_t)DLCD_SEGL_COM2}                                        /* 5N */
};

static const uint32_t DlcdMbedSwLeft4Tbl[BSP_DLCD_MbedSw_Left4_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)(DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 5B, 5C */
    {(uint32_t)DLCD_BUF08, (uint8_t)DLCD_SEGL_COM2},                      /* 5F */
    {(uint32_t)DLCD_BUF09, (uint8_t)DLCD_SEGL_COM1},                      /* 5J */
    {(uint32_t)DLCD_BUF23, (uint8_t)DLCD_SEGL_COM2}                       /* 5N */
};

static const uint32_t DlcdMbedSwLeft5Tbl[BSP_DLCD_MbedSw_Left5_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM1)},   /* 5A, 5C */
    {(uint32_t)DLCD_BUF08, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM2)},   /* 5D, 5F */
    {(uint32_t)DLCD_BUF09, (uint8_t)DLCD_SEGL_COM1},                      /* 5J */
    {(uint32_t)DLCD_BUF23, (uint8_t)DLCD_SEGL_COM2}                       /* 5N */
};

static const uint32_t DlcdMbedSwLeft6Tbl[BSP_DLCD_MbedSw_Left6_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM1)},                    /* 5A, 5C */
    {(uint32_t)DLCD_BUF08, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},   /* 5D, 5E, 5F */
    {(uint32_t)DLCD_BUF09, (uint8_t)DLCD_SEGL_COM1},                                       /* 5J */
    {(uint32_t)DLCD_BUF23, (uint8_t)DLCD_SEGL_COM2}                                        /* 5N */
};

static const uint32_t DlcdMbedSwLeft7Tbl[BSP_DLCD_MbedSw_Left7_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 5A, 5B, 5C */
    {(uint32_t)DLCD_BUF08, (uint8_t)DLCD_SEGL_COM2}                                        /* 5F */
};

static const uint32_t DlcdMbedSwLeft8Tbl[BSP_DLCD_MbedSw_Left8_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 5A, 5B, 5C */
    {(uint32_t)DLCD_BUF08, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},   /* 5D, 5E, 5F */
    {(uint32_t)DLCD_BUF09, (uint8_t)DLCD_SEGL_COM1},                                       /* 5J */
    {(uint32_t)DLCD_BUF23, (uint8_t)DLCD_SEGL_COM2}                                        /* 5N */
};

static const uint32_t DlcdMbedSwLeft9Tbl[BSP_DLCD_MbedSw_Left9_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 5A, 5B, 5C */
    {(uint32_t)DLCD_BUF08, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM2)},                    /* 5D, 5F */
    {(uint32_t)DLCD_BUF09, (uint8_t)DLCD_SEGL_COM1},                                       /* 5J */
    {(uint32_t)DLCD_BUF23, (uint8_t)DLCD_SEGL_COM2}                                        /* 5N */
};

static const uint32_t DlcdMbedSwLeftATbl[BSP_DLCD_MbedSw_LeftA_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 5A, 5B, 5C */
    {(uint32_t)DLCD_BUF08, (uint8_t)(DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},                    /* 5E, 5F */
    {(uint32_t)DLCD_BUF09, (uint8_t)DLCD_SEGL_COM1},                                       /* 5J */
    {(uint32_t)DLCD_BUF23, (uint8_t)DLCD_SEGL_COM2}                                        /* 5N */
};

static const uint32_t DlcdMbedSwLeftBTbl[BSP_DLCD_MbedSw_LeftB_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)DLCD_SEGL_COM1},                                       /* 5C */
    {(uint32_t)DLCD_BUF08, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},   /* 5D, 5E, 5F */
    {(uint32_t)DLCD_BUF09, (uint8_t)DLCD_SEGL_COM1},                                       /* 5J */
    {(uint32_t)DLCD_BUF23, (uint8_t)DLCD_SEGL_COM2}                                        /* 5N */
};

static const uint32_t DlcdMbedSwLeftCTbl[BSP_DLCD_MbedSw_LeftC_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)DLCD_SEGL_COM3},                                       /* 5A */
    {(uint32_t)DLCD_BUF08, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)}    /* 5D, 5E, 5F */
};

static const uint32_t DlcdMbedSwLeftDTbl[BSP_DLCD_MbedSw_LeftD_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)(DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 5B, 5C */
    {(uint32_t)DLCD_BUF08, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1)},   /* 5D, 5E */
    {(uint32_t)DLCD_BUF09, (uint8_t)DLCD_SEGL_COM1},                      /* 5J */
    {(uint32_t)DLCD_BUF23, (uint8_t)DLCD_SEGL_COM2}                       /* 5N */
};

static const uint32_t DlcdMbedSwLeftETbl[BSP_DLCD_MbedSw_LeftE_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)DLCD_SEGL_COM3},                                       /* 5A */
    {(uint32_t)DLCD_BUF08, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},   /* 5D, 5E, 5F */
    {(uint32_t)DLCD_BUF09, (uint8_t)DLCD_SEGL_COM1},                                       /* 5J */
    {(uint32_t)DLCD_BUF23, (uint8_t)DLCD_SEGL_COM2}                                        /* 5N */
};

static const uint32_t DlcdMbedSwLeftFTbl[BSP_DLCD_MbedSw_LeftF_MAX][2] =
{
    {(uint32_t)DLCD_BUF22, (uint8_t)DLCD_SEGL_COM3},                      /* 5A */
    {(uint32_t)DLCD_BUF08, (uint8_t)(DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},   /* 5E, 5F */
    {(uint32_t)DLCD_BUF09, (uint8_t)DLCD_SEGL_COM1},                      /* 5J */
    {(uint32_t)DLCD_BUF23, (uint8_t)DLCD_SEGL_COM2}                       /* 5N */
};

/*----------------------------------*/
/** 
 * @brief   DLCD Middle Number display Table.
 * @details 
 * | [][0] | [][1] |
 * | :---  | :---  |
 * | Buf   | Com   |
*/
/*----------------------------------*/
static const uint32_t DlcdMbedSwMiddle0Tbl[BSP_DLCD_MbedSw_Middle0_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 6A, 6B, 6C */
    {(uint32_t)DLCD_BUF10, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)}    /* 6D, 6E, 6F */
};

static const uint32_t DlcdMbedSwMiddle1Tbl[BSP_DLCD_MbedSw_Middle1_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)(DLCD_SEGL_COM2 | DLCD_SEGL_COM1)}    /* 6B, 6C */
};

static const uint32_t DlcdMbedSwMiddle2Tbl[BSP_DLCD_MbedSw_Middle2_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2)},   /* 6A, 6B */
    {(uint32_t)DLCD_BUF10, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1)},   /* 6D, 6E */
    {(uint32_t)DLCD_BUF11, (uint8_t)DLCD_SEGL_COM1},                      /* 6J */
    {(uint32_t)DLCD_BUF35, (uint8_t)DLCD_SEGL_COM2}                       /* 6N */
};

static const uint32_t DlcdMbedSwMiddle3Tbl[BSP_DLCD_MbedSw_Middle3_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 6A, 6B, 6C */
    {(uint32_t)DLCD_BUF10, (uint8_t)DLCD_SEGL_COM0},                                       /* 6D */
    {(uint32_t)DLCD_BUF11, (uint8_t)DLCD_SEGL_COM1},                                       /* 6J */
    {(uint32_t)DLCD_BUF35, (uint8_t)DLCD_SEGL_COM2}                                        /* 6N */
};

static const uint32_t DlcdMbedSwMiddle4Tbl[BSP_DLCD_MbedSw_Middle4_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)(DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 6B, 6C */
    {(uint32_t)DLCD_BUF10, (uint8_t)DLCD_SEGL_COM2},                      /* 6F */
    {(uint32_t)DLCD_BUF11, (uint8_t)DLCD_SEGL_COM1},                      /* 6J */
    {(uint32_t)DLCD_BUF35, (uint8_t)DLCD_SEGL_COM2}                       /* 6N */
};

static const uint32_t DlcdMbedSwMiddle5Tbl[BSP_DLCD_MbedSw_Middle5_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM1)},   /* 6A, 6C */
    {(uint32_t)DLCD_BUF10, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM2)},   /* 6D, 6F */
    {(uint32_t)DLCD_BUF11, (uint8_t)DLCD_SEGL_COM1},                      /* 6J */
    {(uint32_t)DLCD_BUF35, (uint8_t)DLCD_SEGL_COM2}                       /* 6N */
};

static const uint32_t DlcdMbedSwMiddle6Tbl[BSP_DLCD_MbedSw_Middle6_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM1)},                    /* 6A, 6C */
    {(uint32_t)DLCD_BUF10, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},   /* 6D, 6E, 6F */
    {(uint32_t)DLCD_BUF11, (uint8_t)DLCD_SEGL_COM1},                                       /* 6J */
    {(uint32_t)DLCD_BUF35, (uint8_t)DLCD_SEGL_COM2}                                        /* 6N */
};

static const uint32_t DlcdMbedSwMiddle7Tbl[BSP_DLCD_MbedSw_Middle7_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 6A, 6B, 6C */
    {(uint32_t)DLCD_BUF10, (uint8_t)DLCD_SEGL_COM2}                                        /* 6F */
};

static const uint32_t DlcdMbedSwMiddle8Tbl[BSP_DLCD_MbedSw_Middle8_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 6A, 6B, 6C */
    {(uint32_t)DLCD_BUF10, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},   /* 6D, 6E, 6F */
    {(uint32_t)DLCD_BUF11, (uint8_t)DLCD_SEGL_COM1},                                       /* 6J */
    {(uint32_t)DLCD_BUF35, (uint8_t)DLCD_SEGL_COM2}                                        /* 6N */
};

static const uint32_t DlcdMbedSwMiddle9Tbl[BSP_DLCD_MbedSw_Middle9_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 6A, 6B, 6C */
    {(uint32_t)DLCD_BUF10, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM2)},                    /* 6D, 6F */
    {(uint32_t)DLCD_BUF11, (uint8_t)DLCD_SEGL_COM1},                                       /* 6J */
    {(uint32_t)DLCD_BUF35, (uint8_t)DLCD_SEGL_COM2}                                        /* 6N */
};

static const uint32_t DlcdMbedSwMiddleATbl[BSP_DLCD_MbedSw_MiddleA_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 6A, 6B, 6C */
    {(uint32_t)DLCD_BUF10, (uint8_t)(DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},                    /* 6E, 6F */
    {(uint32_t)DLCD_BUF11, (uint8_t)DLCD_SEGL_COM1},                                       /* 6J */
    {(uint32_t)DLCD_BUF35, (uint8_t)DLCD_SEGL_COM2}                                        /* 6N */
};

static const uint32_t DlcdMbedSwMiddleBTbl[BSP_DLCD_MbedSw_MiddleB_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)DLCD_SEGL_COM1},                                       /* 6C */
    {(uint32_t)DLCD_BUF10, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},   /* 6D, 6E, 6F */
    {(uint32_t)DLCD_BUF11, (uint8_t)DLCD_SEGL_COM1},                                       /* 6J */
    {(uint32_t)DLCD_BUF35, (uint8_t)DLCD_SEGL_COM2}                                        /* 6N */
};

static const uint32_t DlcdMbedSwMiddleCTbl[BSP_DLCD_MbedSw_MiddleC_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)DLCD_SEGL_COM3},                                       /* 6A */
    {(uint32_t)DLCD_BUF10, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)}    /* 6D, 6E, 6F */
};

static const uint32_t DlcdMbedSwMiddleDTbl[BSP_DLCD_MbedSw_MiddleD_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)(DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 6B, 6C */
    {(uint32_t)DLCD_BUF10, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1)},   /* 6D, 6E */
    {(uint32_t)DLCD_BUF11, (uint8_t)DLCD_SEGL_COM1},                      /* 6J */
    {(uint32_t)DLCD_BUF35, (uint8_t)DLCD_SEGL_COM2}                       /* 6N */
};

static const uint32_t DlcdMbedSwMiddleETbl[BSP_DLCD_MbedSw_MiddleE_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)DLCD_SEGL_COM3},                                       /* 6A */
    {(uint32_t)DLCD_BUF10, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},   /* 6D, 6E, 6F */
    {(uint32_t)DLCD_BUF11, (uint8_t)DLCD_SEGL_COM1},                                       /* 6J */
    {(uint32_t)DLCD_BUF35, (uint8_t)DLCD_SEGL_COM2}                                        /* 6N */
};

static const uint32_t DlcdMbedSwMiddleFTbl[BSP_DLCD_MbedSw_MiddleF_MAX][2] =
{
    {(uint32_t)DLCD_BUF34, (uint8_t)DLCD_SEGL_COM3},                      /* 6A */
    {(uint32_t)DLCD_BUF10, (uint8_t)(DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},   /* 6E, 6F */
    {(uint32_t)DLCD_BUF11, (uint8_t)DLCD_SEGL_COM1},                      /* 6J */
    {(uint32_t)DLCD_BUF35, (uint8_t)DLCD_SEGL_COM2}                       /* 6N */
};

/*----------------------------------*/
/** 
 * @brief   DLCD Right Number display Table.
 * @details 
 * | [][0] | [][1] |
 * | :---  | :---  |
 * | Buf   | Com   |
*/
/*----------------------------------*/
static const uint32_t DlcdMbedSwRight0Tbl[BSP_DLCD_MbedSw_Right0_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 7A, 7B, 7C */
    {(uint32_t)DLCD_BUF12, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)}    /* 7D, 7E, 7F */
};

static const uint32_t DlcdMbedSwRight1Tbl[BSP_DLCD_MbedSw_Right1_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)(DLCD_SEGL_COM2 | DLCD_SEGL_COM1)}    /* 7B, 7C */
};

static const uint32_t DlcdMbedSwRight2Tbl[BSP_DLCD_MbedSw_Right2_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2)},   /* 7A, 7B */
    {(uint32_t)DLCD_BUF12, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1)},   /* 7D, 7E */
    {(uint32_t)DLCD_BUF32, (uint8_t)DLCD_SEGL_COM1},                      /* 7J */
    {(uint32_t)DLCD_BUF19, (uint8_t)DLCD_SEGL_COM2}                       /* 7N */
};

static const uint32_t DlcdMbedSwRight3Tbl[BSP_DLCD_MbedSw_Right3_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 7A, 7B, 7C */
    {(uint32_t)DLCD_BUF12, (uint8_t)DLCD_SEGL_COM0},                                       /* 7D */
    {(uint32_t)DLCD_BUF32, (uint8_t)DLCD_SEGL_COM1},                                       /* 7J */
    {(uint32_t)DLCD_BUF19, (uint8_t)DLCD_SEGL_COM2}                                        /* 7N */
};

static const uint32_t DlcdMbedSwRight4Tbl[BSP_DLCD_MbedSw_Right4_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)(DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 7B, 7C */
    {(uint32_t)DLCD_BUF12, (uint8_t)DLCD_SEGL_COM2},                      /* 7F */
    {(uint32_t)DLCD_BUF32, (uint8_t)DLCD_SEGL_COM1},                      /* 7J */
    {(uint32_t)DLCD_BUF19, (uint8_t)DLCD_SEGL_COM2}                       /* 7N */
};

static const uint32_t DlcdMbedSwRight5Tbl[BSP_DLCD_MbedSw_Right5_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM1)},   /* 7A, 7C */
    {(uint32_t)DLCD_BUF12, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM2)},   /* 7D, 7F */
    {(uint32_t)DLCD_BUF32, (uint8_t)DLCD_SEGL_COM1},                      /* 7J */
    {(uint32_t)DLCD_BUF19, (uint8_t)DLCD_SEGL_COM2}                       /* 7N */
};

static const uint32_t DlcdMbedSwRight6Tbl[BSP_DLCD_MbedSw_Right6_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM1)},                    /* 7A, 7C */
    {(uint32_t)DLCD_BUF12, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},   /* 7D, 7E, 7F */
    {(uint32_t)DLCD_BUF32, (uint8_t)DLCD_SEGL_COM1},                                       /* 7J */
    {(uint32_t)DLCD_BUF19, (uint8_t)DLCD_SEGL_COM2}                                        /* 7N */
};

static const uint32_t DlcdMbedSwRight7Tbl[BSP_DLCD_MbedSw_Right7_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 7A, 7B, 7C */
    {(uint32_t)DLCD_BUF12, (uint8_t)DLCD_SEGL_COM2}                                        /* 7F */
};

static const uint32_t DlcdMbedSwRight8Tbl[BSP_DLCD_MbedSw_Right8_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 7A, 7B, 7C */
    {(uint32_t)DLCD_BUF12, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},   /* 7D, 7E, 7F */
    {(uint32_t)DLCD_BUF32, (uint8_t)DLCD_SEGL_COM1},                                       /* 7J */
    {(uint32_t)DLCD_BUF19, (uint8_t)DLCD_SEGL_COM2}                                        /* 7N */
};

static const uint32_t DlcdMbedSwRight9Tbl[BSP_DLCD_MbedSw_Right9_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 7A, 7B, 7C */
    {(uint32_t)DLCD_BUF12, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM2)},                    /* 7D, 7F */
    {(uint32_t)DLCD_BUF32, (uint8_t)DLCD_SEGL_COM1},                                       /* 7J */
    {(uint32_t)DLCD_BUF19, (uint8_t)DLCD_SEGL_COM2}                                        /* 7N */
};

static const uint32_t DlcdMbedSwRightATbl[BSP_DLCD_MbedSw_RightA_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)(DLCD_SEGL_COM3 | DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 7A, 7B, 7C */
    {(uint32_t)DLCD_BUF12, (uint8_t)(DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},                    /* 7E, 7F */
    {(uint32_t)DLCD_BUF32, (uint8_t)DLCD_SEGL_COM1},                                       /* 7J */
    {(uint32_t)DLCD_BUF19, (uint8_t)DLCD_SEGL_COM2}                                        /* 7N */
};

static const uint32_t DlcdMbedSwRightBTbl[BSP_DLCD_MbedSw_RightB_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)DLCD_SEGL_COM1},                                       /* 7C */
    {(uint32_t)DLCD_BUF12, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},   /* 7D, 7E, 7F */
    {(uint32_t)DLCD_BUF32, (uint8_t)DLCD_SEGL_COM1},                                       /* 7J */
    {(uint32_t)DLCD_BUF19, (uint8_t)DLCD_SEGL_COM2}                                        /* 7N */
};

static const uint32_t DlcdMbedSwRightCTbl[BSP_DLCD_MbedSw_RightC_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)DLCD_SEGL_COM3},                                       /* 7A */
    {(uint32_t)DLCD_BUF12, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)}    /* 7D, 7E, 7F */
};

static const uint32_t DlcdMbedSwRightDTbl[BSP_DLCD_MbedSw_RightD_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)(DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 7B, 7C */
    {(uint32_t)DLCD_BUF12, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1)},   /* 7D, 7E */
    {(uint32_t)DLCD_BUF32, (uint8_t)DLCD_SEGL_COM1},                      /* 7J */
    {(uint32_t)DLCD_BUF19, (uint8_t)DLCD_SEGL_COM2}                       /* 7N */
};

static const uint32_t DlcdMbedSwRightETbl[BSP_DLCD_MbedSw_RightE_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)DLCD_SEGL_COM3},                                       /* 7A */
    {(uint32_t)DLCD_BUF12, (uint8_t)(DLCD_SEGL_COM0 | DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},   /* 7D, 7E, 7F */
    {(uint32_t)DLCD_BUF32, (uint8_t)DLCD_SEGL_COM1},                                       /* 7J */
    {(uint32_t)DLCD_BUF19, (uint8_t)DLCD_SEGL_COM2}                                        /* 7N */
};

static const uint32_t DlcdMbedSwRightFTbl[BSP_DLCD_MbedSw_RightF_MAX][2] =
{
    {(uint32_t)DLCD_BUF18, (uint8_t)DLCD_SEGL_COM3},                                       /* 7A */
    {(uint32_t)DLCD_BUF12, (uint8_t)(DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},                    /* 7E, 7F */
    {(uint32_t)DLCD_BUF32, (uint8_t)DLCD_SEGL_COM1},                                       /* 7J */
    {(uint32_t)DLCD_BUF19, (uint8_t)DLCD_SEGL_COM2}                                        /* 7N */
};

/*----------------------------------*/
/** 
 * @brief   DLCD "H" display Table.
 * @details 
 * | [][0] | [][1] |
 * | :---  | :---  |
 * | Buf   | Com   |
*/
/*----------------------------------*/
static const uint32_t DlcdMbedSwHTbl[BSP_DLCD_MbedSw_H_MAX][2] =
{
    {(uint32_t)DLCD_BUF16, (uint8_t)(DLCD_SEGL_COM2 | DLCD_SEGL_COM1)},   /* 8B, 8C */
    {(uint32_t)DLCD_BUF14, (uint8_t)(DLCD_SEGL_COM1 | DLCD_SEGL_COM2)},   /* 8E, 8F */
    {(uint32_t)DLCD_BUF33, (uint8_t)DLCD_SEGL_COM1},                      /* 8J */
    {(uint32_t)DLCD_BUF17, (uint8_t)DLCD_SEGL_COM2}                       /* 8N */
};

/**
 *  @}
 */ /* End of group BSP_DLCD_Private_const */

/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_DLCD_Private_functions
 *  @{
 */
static void dlcd_number_left_set(display_t *p_instance, uint32_t value);
static void dlcd_number_middle_set(display_t *p_instance, uint32_t value);
static void dlcd_number_right_set(display_t *p_instance, uint32_t value);
static void dlcd_port_initialize(display_t *p_instance);
static void dlcd_error(void);

/*--------------------------------------------------*/
/** 
  * @brief  Display Left Number set.
  * @param  p_instance :Instance Source Address.
  * @param  value :ADC Data.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
static void dlcd_number_left_set(display_t *p_instance, uint32_t value)
{
    uint32_t i;
    uint8_t regval;
    BSPDlcdNumber number;

    dlcd_t *p_dlcd = p_instance->init.p_dlcd;
    value = (uint32_t)(value & 0x00000F00);
    number = (BSPDlcdNumber)(value >> 8);

    if(number == BSP_DLCD_NUMBER_0){
        /* "0" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Left0_MAX; i++)
        {
            regval = DlcdMbedSwLeft0Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeft0Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_1){
        /* "1" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Left1_MAX; i++)
        {
            regval = DlcdMbedSwLeft1Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeft1Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_2){
        /* "2" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Left2_MAX; i++)
        {
            regval = DlcdMbedSwLeft2Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeft2Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_3){
        /* "3" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Left3_MAX; i++)
        {
            regval = DlcdMbedSwLeft3Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeft3Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_4){
        /* "4" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Left4_MAX; i++)
        {
            regval = DlcdMbedSwLeft4Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeft4Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_5){
        /* "5" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Left5_MAX; i++)
        {
            regval = DlcdMbedSwLeft5Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeft5Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_6){
        /* "6" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Left6_MAX; i++)
        {
            regval = DlcdMbedSwLeft6Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeft6Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_7){
        /* "7" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Left7_MAX; i++)
        {
            regval = DlcdMbedSwLeft7Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeft7Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_8){
        /* "8" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Left8_MAX; i++)
        {
            regval = DlcdMbedSwLeft8Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeft8Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_9){
        /* "9" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Left9_MAX; i++)
        {
            regval = DlcdMbedSwLeft9Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeft9Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_A){
        /* "A" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_LeftA_MAX; i++)
        {
            regval = DlcdMbedSwLeftATbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeftATbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_B){
        /* "B" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_LeftB_MAX; i++)
        {
            regval = DlcdMbedSwLeftBTbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeftBTbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_C){
        /* "C" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_LeftC_MAX; i++)
        {
            regval = DlcdMbedSwLeftCTbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeftCTbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_D){
        /* "D" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_LeftD_MAX; i++)
        {
            regval = DlcdMbedSwLeftDTbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeftDTbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_E){
        /* "E" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_LeftE_MAX; i++)
        {
            regval = DlcdMbedSwLeftETbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeftETbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_F){
        /* "F" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_LeftF_MAX; i++)
        {
            regval = DlcdMbedSwLeftFTbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwLeftFTbl[i][0], regval);
        }
    }
    else{
        /* Do nothing */
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Display Middle Number set.
  * @param  p_instance :Instance Source Address.
  * @param  value :ADC Data.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
static void dlcd_number_middle_set(display_t *p_instance, uint32_t value)
{
    uint32_t i;
    uint8_t regval;
    BSPDlcdNumber number;

    dlcd_t *p_dlcd = p_instance->init.p_dlcd;
    value = (uint32_t)(value & 0x000000F0);
    number = (BSPDlcdNumber)(value >> 4);

    if(number == BSP_DLCD_NUMBER_0){
        /* "0" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Middle0_MAX; i++)
        {
            regval = DlcdMbedSwMiddle0Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddle0Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_1){
        /* "1" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Middle1_MAX; i++)
        {
            regval = DlcdMbedSwMiddle1Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddle1Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_2){
        /* "2" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Middle2_MAX; i++)
        {
            regval = DlcdMbedSwMiddle2Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddle2Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_3){
        /* "3" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Middle3_MAX; i++)
        {
            regval = DlcdMbedSwMiddle3Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddle3Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_4){
        /* "4" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Middle4_MAX; i++)
        {
            regval = DlcdMbedSwMiddle4Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddle4Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_5){
        /* "5" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Middle5_MAX; i++)
        {
            regval = DlcdMbedSwMiddle5Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddle5Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_6){
        /* "6" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Middle6_MAX; i++)
        {
            regval = DlcdMbedSwMiddle6Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddle6Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_7){
        /* "7" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Middle7_MAX; i++)
        {
            regval = DlcdMbedSwMiddle7Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddle7Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_8){
        /* "8" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Middle8_MAX; i++)
        {
            regval = DlcdMbedSwMiddle8Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddle8Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_9){
        /* "9" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Middle9_MAX; i++)
        {
            regval = DlcdMbedSwMiddle9Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddle9Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_A){
        /* "A" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_MiddleA_MAX; i++)
        {
            regval = DlcdMbedSwMiddleATbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddleATbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_B){
        /* "B" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_MiddleB_MAX; i++)
        {
            regval = DlcdMbedSwMiddleBTbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddleBTbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_C){
        /* "C" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_MiddleC_MAX; i++)
        {
            regval = DlcdMbedSwMiddleCTbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddleCTbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_D){
        /* "D" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_MiddleD_MAX; i++)
        {
            regval = DlcdMbedSwMiddleDTbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddleDTbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_E){
        /* "E" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_MiddleE_MAX; i++)
        {
            regval = DlcdMbedSwMiddleETbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddleETbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_F){
        /* "F" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_MiddleF_MAX; i++)
        {
            regval = DlcdMbedSwMiddleFTbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwMiddleFTbl[i][0], regval);
        }
    }
    else{
        /* Do nothing */
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Display Right Number set.
  * @param  p_instance :Instance Source Address.
  * @param  value :ADC Data.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
static void dlcd_number_right_set(display_t *p_instance, uint32_t value)
{
    uint32_t i;
    uint8_t regval;
    BSPDlcdNumber number;

    dlcd_t *p_dlcd = p_instance->init.p_dlcd;
    value = (uint32_t)(value & 0x0000000F);
    number = (BSPDlcdNumber)(value);

    if(number == BSP_DLCD_NUMBER_0){
        /* "0" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Right0_MAX; i++)
        {
            regval = DlcdMbedSwRight0Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRight0Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_1){
        /* "1" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Right1_MAX; i++)
        {
            regval = DlcdMbedSwRight1Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRight1Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_2){
        /* "2" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Right2_MAX; i++)
        {
            regval = DlcdMbedSwRight2Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRight2Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_3){
        /* "3" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Right3_MAX; i++)
        {
            regval = DlcdMbedSwRight3Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRight3Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_4){
        /* "4" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Right4_MAX; i++)
        {
            regval = DlcdMbedSwRight4Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRight4Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_5){
        /* "5" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Right5_MAX; i++)
        {
            regval = DlcdMbedSwRight5Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRight5Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_6){
        /* "6" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Right6_MAX; i++)
        {
            regval = DlcdMbedSwRight6Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRight6Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_7){
        /* "7" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Right7_MAX; i++)
        {
            regval = DlcdMbedSwRight7Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRight7Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_8){
        /* "8" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Right8_MAX; i++)
        {
            regval = DlcdMbedSwRight8Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRight8Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_9){
        /* "9" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Right9_MAX; i++)
        {
            regval = DlcdMbedSwRight9Tbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRight9Tbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_A){
        /* "A" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_RightA_MAX; i++)
        {
            regval = DlcdMbedSwRightATbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRightATbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_B){
        /* "B" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_RightB_MAX; i++)
        {
            regval = DlcdMbedSwRightBTbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRightBTbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_C){
        /* "C" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_RightC_MAX; i++)
        {
            regval = DlcdMbedSwRightCTbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRightCTbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_D){
        /* "D" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_RightD_MAX; i++)
        {
            regval = DlcdMbedSwRightDTbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRightDTbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_E){
        /* "E" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_RightE_MAX; i++)
        {
            regval = DlcdMbedSwRightETbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRightETbl[i][0], regval);
        }
    }
    else if(number == BSP_DLCD_NUMBER_F){
        /* "F" display */
        for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_RightF_MAX; i++)
        {
            regval = DlcdMbedSwRightFTbl[i][1];
            DLCD_reg_set(p_dlcd, DlcdMbedSwRightFTbl[i][0], regval);
        }
    }
    else{
        /* Do nothing */
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Port initialize
  * @param  p_instance :Instance Source Address.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
static void dlcd_port_initialize(display_t *p_instance)
{
    /*--- DLCD Port Set ---*/
    if (gpio_func(p_instance->init.p_gpio, 
                  (gpio_gr_t)p_instance->init.port.group, 
                  (gpio_num_t)p_instance->init.port.num, 
                  p_instance->init.port.func, 
                  GPIO_PIN_OUTPUT)
                   != TXZ_SUCCESS)
    {
        dlcd_error();
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Error Hook
  * @param  -
  * @retval -
  * @note   for debug
  */
/*--------------------------------------------------*/
static void dlcd_error(void)
{
#ifdef DLCD_CFG_DEBUG
    while(1)
    {
        __nop();
    }
#endif
}

/**
 *  @}
 */ /* End of group BSP_DLCD_Private_functions */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_DLCD_Exported_functions
 *  @{
 */

/*--------------------------------------------------*/
/** 
  * @brief  Initialize
  * @param  p_instance :Instance Source Address.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void dlcd_initialize(display_t *p_instance)
{
    uint32_t i;
    dlcd_t *p_dlcd = p_instance->init.p_dlcd;
    cg_t *p_cg = p_instance->init.p_cg;
    rlm_t *p_rlm = p_instance->init.p_rlm;

    /*--- DLCD Terminal Set ---*/
    DLCD_Output_Change(p_dlcd, p_instance->init.dlcd_data);
    DLCD_Terminal_Change(p_dlcd, p_instance->init.dlcd_data);
    for (i=0; i<(uint32_t)BSP_DLCD_MAX; i++)
    {
         p_instance->init.p_gpio                   = bsp_get_gpio_instance_address();
         p_instance->init.port.group               = bsp_get_gpio_group_dlcd((BSPDlcd)(i));
         p_instance->init.port.num                 = bsp_get_gpio_num_dlcd((BSPDlcd)(i));
         p_instance->init.port.func                = bsp_get_gpio_func_dlcd((BSPDlcd)(i));
         dlcd_port_initialize(p_instance);
    }
    /*--- DLCD Clock Set    ---*/
    if (p_instance->init.clock == DLCD_CLOCK_IHOSC2)
    {
        cg_ihosc2_enable();
        while(cg_get_ihosc2_state() == TXZ_BUSY){
            ;
        }
    } else {
        /* [CGWUPLCR]<WUPTL> :Warm up time        */
        cg_set_low_warm_up_time(p_cg, WU_TIME_FS);
        /* [RLMLOSCCR]<XTEN> :LOSC Enable         */
        rlm_fs_enable(p_rlm);
        /* [CGWUPLCR]<WULON> :Enable              */
        cg_start_low_warmup();
        /* [CGWUPLCR]<WULEF> :Read(wait for warm-up) */
        while (cg_get_low_warm_up_state() == TXZ_BUSY)
        {
            /* no processing */
        }
    }
    /*--- DLCD Enable       ---*/
    DLCD_Display_Start(p_dlcd, p_instance->init.dlcd_data);

}

/*--------------------------------------------------*/
/** 
  * @brief  Finalize
  * @param  p_instance :Instance Source Address.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void dlcd_finalize(display_t *p_instance)
{
    dlcd_t *p_dlcd = p_instance->init.p_dlcd;

    /*--- DLCD Disable     ---*/
    DLCD_Stop(p_dlcd);
}

/*--------------------------------------------------*/
/** 
  * @brief  Display set.
  * @param  p_instance :Instance Source Address.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void dlcd_display_set(display_t *p_instance, uint32_t data)
{
    uint32_t i;
    uint8_t regval;
    dlcd_t *p_dlcd = p_instance->init.p_dlcd;

    DLCD_Buffer_ALL_Clear(p_dlcd);
    /* "A" display */
    for (i=0; i<BSP_DLCD_MbedSw_A_MAX; i++)
    {
        regval = DlcdMbedSwATbl[i][1];
        DLCD_reg_set(p_dlcd, DlcdMbedSwATbl[i][0], regval);
    }
    /* "d" display */
    for (i=0; i<BSP_DLCD_MbedSw_d_MAX; i++)
    {
        regval = DlcdMbedSwdTbl[i][1];
        DLCD_reg_set(p_dlcd, DlcdMbedSwdTbl[i][0], regval);
    }
    /* "c." display */
    for (i=0; i<BSP_DLCD_MbedSw_c_MAX; i++)
    {
        regval = DlcdMbedSwcTbl[i][1];
        DLCD_reg_set(p_dlcd, DlcdMbedSwcTbl[i][0], regval);
    }
    /* ":" display */
    for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_Semicoron_MAX; i++)
    {
        regval = DlcdMbedSwSemicoronTbl[i][1];
        DLCD_reg_set(p_dlcd, DlcdMbedSwSemicoronTbl[i][0], regval);
    }
    /* number left display */
    dlcd_number_left_set(p_instance, data);
    /* number middle display */
    dlcd_number_middle_set(p_instance, data);
    /* number right display */
    dlcd_number_right_set(p_instance, data);
    /* "H" display */
    for (i=0; i<(uint32_t)BSP_DLCD_MbedSw_H_MAX; i++)
    {
        regval = DlcdMbedSwHTbl[i][1];
        DLCD_reg_set(p_dlcd, DlcdMbedSwHTbl[i][0], regval);
    }
}
/**
 *  @}
 */ /* End of group BSP_DLCD_Exported_functions */

/**
 *  @} 
 */ /* End of group BSP_DLCD */

/**
 *  @} 
 */ /* End of group BSP */

#endif /* defined(__BSP_CONFIG_ADBUN_M3HQA_DLCD_H)  */

#ifdef __cplusplus
}
#endif /* __cplusplus */
