/** 
 *******************************************************************************
 * @file    bsp_config_adbun_m3hqa_ei2c.c
 * @brief   This file provides API functions for BSP EI2C driver.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "bsp_ei2c.h"
#include "sys_timer.h"

#if defined(__BSP_EI2C_H)

/**
 *  @addtogroup BSP BSP
 *  @{
 */

/** 
 *  @addtogroup BSP_EI2C
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_EI2C_Private_macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group BSP_EI2C_Private_macro */

/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_EI2C_Private_define
 *  @{
 */

/** 
 *  @name  Parameter Result
 *  @brief Whether the parameter is specified or not.
 *  @{
 */
#define EI2C_PARAM_OK              ((int32_t)1)              /*!< Parameter is valid(specified).         */
#define EI2C_PARAM_NG              ((int32_t)0)              /*!< Parameter is invalid(not specified).   */
/**
 *  @}
 */ /* End of name Parameter Result */

/** 
 *  @name  timeout 
 *  @brief This timeouts are not based on accurate values, this just guarantee that 
           the application will not remain stuck if the EI2C communication is corrupted.
 *  @{
 */
#define EI2C_TIMEOUT     (100000)    /*>! fail safe. */

/**
 *  @}
 */ /* End of name timeout */

#define EI2CxSR_AL               ((uint32_t)0x00000008)     /*!< AL                      */
/** 
 *  @}
 */ /* End of group BSP_EI2C_Private_define */

/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_EI2C_Private_define
 *  @{
 */
#define EI2C_CH0                     (0)         /*!< EI2C Channel 0.            */
#define EI2C_CH1                     (1)         /*!< EI2C Channel 1.            */
#define EI2C_CH2                     (2)         /*!< EI2C Channel 2.            */
#define EI2C_CH3                     (3)         /*!< EI2C Channel 3.            */
#define EI2C_CH_NUM                  (5)         /*!< Number of EI2C Channel.    */

/** 
 *  @}
 */ /* End of group BSP_EI2C_Private_define */

/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_EI2C_Private_define
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief  Transfer State.
*/
/*----------------------------------*/
enum {
    EI2C_TRANSFER_STATE_IDLE = 0U,   /*!< Idle.                      */
    EI2C_TRANSFER_STATE_BUSY         /*!< Busy.                      */
} TransferState;

/** 
 *  @}
 */ /* End of group BSP_EI2C_Private_define */

/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_EI2C_Private_typedef
 *  @{
 */

/*----------------------------------*/
/** 
 * @brief  For IRQn_Type number definition.
*/
/*----------------------------------*/
typedef struct
{
    IRQn_Type ei2c;
    IRQn_Type tbe;
    IRQn_Type rbf;
} ei2c_irq_t;

/** 
 *  @}
 */ /* End of group BSP_EI2C_Private_typedef */

/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_EI2C_Private_variables
 *  @{
 */

/* no define */

/**
 *  @}
 */ /* End of group BSP_EI2C_Private_variables */

/*------------------------------------------------------------------------------*/
/*  Const Table                                                                 */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_EI2C_Private_const
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief  Channel 0 IRQn_Type number table.
*/
/*----------------------------------*/
static const ei2c_irq_t EI2C_CH0_IRQN_TBL[1] = 
{
    { INTI2C0NST_IRQn, INTI2C0ATX_IRQn, INTI2C0BRX_IRQn }
};

#if !defined(TMPM3HLA)
/*----------------------------------*/
/** 
 * @brief  Channel 1 IRQn_Type number table.
*/
/*----------------------------------*/
static const ei2c_irq_t EI2C_CH1_IRQN_TBL[1] = 
{
    { INTI2C1NST_IRQn, INTI2C1ATX_IRQn, INTI2C1BRX_IRQn }
};
#endif

/*----------------------------------*/
/** 
 * @brief  Channel 2 IRQn_Type number table.
*/
/*----------------------------------*/
static const ei2c_irq_t EI2C_CH2_IRQN_TBL[1] = 
{
    { INTI2C2NST_IRQn, INTI2C2ATX_IRQn, INTI2C2BRX_IRQn }
};

#if !defined(TMPM3HLA) && !defined(TMPM3HMA) && !defined(TMPM3HNA)
/*----------------------------------*/
/** 
 * @brief  Channel 3 IRQn_Type number table.
*/
/*----------------------------------*/
static const ei2c_irq_t EI2C_CH3_IRQN_TBL[1] = 
{
    { INTI2C3NST_IRQn, INTI2C3ATX_IRQn, INTI2C3BRX_IRQn }
};
#endif

/**
 *  @}
 */ /* End of group BSP_EI2C_Private_const */

/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_EI2C_Private_functions
 *  @{
 */
#ifdef __DEBUG__
__STATIC_INLINE int32_t check_param_irqn(uint32_t irqn);
__STATIC_INLINE int32_t check_param_address(int32_t address);
#endif
__STATIC_INLINE void enable_irq(uint32_t irqn);
__STATIC_INLINE void disable_irq(uint32_t irqn);
__STATIC_INLINE void clear_irq(uint32_t irqn);
__STATIC_INLINE void set_port_c(ei2c_port_t sda, ei2c_port_t scl);
#if !defined(TMPM3HLA)
__STATIC_INLINE void set_port_a(ei2c_port_t sda, ei2c_port_t scl);
#endif
__STATIC_INLINE void set_port_l(ei2c_port_t sda, ei2c_port_t scl);
#if !defined(TMPM3HLA) && !defined(TMPM3HMA) && !defined(TMPM3HNA)
__STATIC_INLINE void set_port_t(ei2c_port_t sda, ei2c_port_t scl);
#endif
__STATIC_INLINE uint32_t set_ei2c(uint8_t ch, uint32_t *p_irqn);
__STATIC_INLINE void reset_asynch(bsp_ei2c_t *p_obj);
static void ei2c_irq_handler(bsp_ei2c_t *p_obj);
static void ei2c_slave_irq_handler(bsp_ei2c_t *p_obj);

#ifdef __DEBUG__
/*--------------------------------------------------*/
/** 
  * @brief  Compare the IRQn's parameter.
  * @param  irqn         :EI2C IRQn List.
  * @retval EI2C_PARAM_OK :Available.
  * @retval EI2C_PARAM_NG :Not Available.
  * @note   -.
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_irqn(uint32_t irqn)
{
    int32_t result = EI2C_PARAM_NG;

    if (irqn == (uint32_t)&EI2C_CH0_IRQN_TBL)
    {
        result = EI2C_PARAM_OK;
    }
    if (irqn == (uint32_t)&EI2C_CH1_IRQN_TBL)
    {
        result = EI2C_PARAM_OK;
    }
    if (irqn == (uint32_t)&EI2C_CH2_IRQN_TBL)
    {
        result = EI2C_PARAM_OK;
    }
    if (irqn == (uint32_t)&EI2C_CH3_IRQN_TBL)
    {
        result = EI2C_PARAM_OK;
    }
    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Compare the Slave address's parameter.
  * @param  address      :Address.
  * @retval EI2C_PARAM_OK :Available.
  * @retval EI2C_PARAM_NG :Not Available.
  * @note   Here, 10bit address has not supported.
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_address(int32_t address)
{
    int32_t result = EI2C_PARAM_NG;

    if ((address >= 0) && (address <= 255))
    {
        result = EI2C_PARAM_OK;
    }
    return (result);
}
#endif

/*--------------------------------------------------*/
/**
  * @brief     Enable EI2C IRQ
  * @param     irqn     :EI2C IRQn List.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void enable_irq(uint32_t irqn)
{
    ei2c_irq_t *p_irqn = (ei2c_irq_t *)irqn;
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(check_param_irqn(irqn));
#endif /* #ifdef __DEBUG__ */
    NVIC_EnableIRQ(p_irqn->ei2c);
    NVIC_EnableIRQ(p_irqn->rbf);
    NVIC_EnableIRQ(p_irqn->tbe);
}

/*--------------------------------------------------*/
/**
  * @brief     Disable EI2C IRQ
  * @param     irqn     :EI2C IRQn List.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void disable_irq(uint32_t irqn)
{
    ei2c_irq_t *p_irqn = (ei2c_irq_t *)irqn;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(check_param_irqn(irqn));
#endif /* #ifdef __DEBUG__ */
    NVIC_DisableIRQ(p_irqn->ei2c);
    NVIC_DisableIRQ(p_irqn->tbe);
    NVIC_DisableIRQ(p_irqn->rbf);
}

/*--------------------------------------------------*/
/**
  * @brief     ClearPending EI2C IRQ
  * @param     irqn     :EI2C IRQn List.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void clear_irq(uint32_t irqn)
{
    ei2c_irq_t *p_irqn = (ei2c_irq_t *)irqn;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(check_param_irqn(irqn));
#endif /* #ifdef __DEBUG__ */
    NVIC_ClearPendingIRQ(p_irqn->ei2c);
    NVIC_ClearPendingIRQ(p_irqn->tbe);
    NVIC_ClearPendingIRQ(p_irqn->rbf);
}

/*--------------------------------------------------*/
/**
  * @brief     EI2C Port Setting (PC0, PC1)
  * @param     sda         :SDA port.
  * @param     scl         :SCL port.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void set_port_c(ei2c_port_t sda, ei2c_port_t scl)
{   
    if ((sda == EI2C_PORT_PC1) && (scl == EI2C_PORT_PC0))
    {
        /* Port C  */
        TSB_CG_FSYSENA_IPENA02 = 1;     /* PORT  C   */

        /* SCL */
        TSB_PC_IE_PC0IE   = 0;    /* Input       :Disable */
        TSB_PC_CR_PC0C    = 0;    /* Output      :Disable */
        TSB_PC_OD_PC0OD   = 1;    /* OD Control  :Open Drain */
        TSB_PC_PUP_PC0UP  = 0;    /* Pull-up     :Disable */
        TSB_PC_PDN_PC0DN  = 0;    /* Pull-down   :Disable */
        TSB_PC_DATA_PC0   = 0;    /* Data        :0       */
        TSB_PC_FR2_PC0F2  = 1;    /* Function    :EI2C0SCL */
        TSB_PC_IE_PC0IE   = 1;    /* Input       :Enable */
        TSB_PC_CR_PC0C    = 1;    /* Output      :Enable  */

        /* SDA */
        TSB_PC_IE_PC1IE   = 0;    /* Input      :Disable  */
        TSB_PC_CR_PC1C    = 0;    /* Output     :Disable  */
        TSB_PC_OD_PC1OD   = 1;    /* OD Control :Open Drain */
        TSB_PC_PUP_PC1UP  = 0;    /* Pull-up    :Disable  */
        TSB_PC_PDN_PC1DN  = 0;    /* Pull-down  :Disable  */
        TSB_PC_DATA_PC1   = 0;    /* Data       :0        */
        TSB_PC_FR2_PC1F2  = 1;    /* Function   :EI2C0SDA  */
        TSB_PC_IE_PC1IE   = 1;    /* Input      :Enable   */
        TSB_PC_CR_PC1C    = 1;    /* Output     :Enable   */

    }
}

#if !defined(TMPM3HLA)
/*--------------------------------------------------*/
/**
  * @brief     EI2C Port Setting (PA4, PA5)
  * @param     sda         :SDA port.
  * @param     scl         :SCL port.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void set_port_a(ei2c_port_t sda, ei2c_port_t scl)
{
    if ((sda == EI2C_PORT_PA5) && (scl == EI2C_PORT_PA4))
    {
        /* Port A  */
        TSB_CG_FSYSENA_IPENA00 = 1;     /* PORT  A   */
        
        /* SCL */
        TSB_PA_IE_PA4IE   = 0;    /* Input       :Disable */
        TSB_PA_CR_PA4C    = 0;    /* Output      :Disable */
        TSB_PA_OD_PA4OD   = 1;    /* OD Control  :Open Drain */
        TSB_PA_PUP_PA4UP  = 0;    /* Pull-up     :Disable */
        TSB_PA_PDN_PA4DN  = 0;    /* Pull-down   :Disable */
        TSB_PA_DATA_PA4   = 0;    /* Data        :0       */
        TSB_PA_FR2_PA4F2  = 1;    /* Function    :EI2C0SCL */
        TSB_PA_IE_PA4IE   = 1;    /* Input       :Enable  */
        TSB_PA_CR_PA4C    = 1;    /* Output      :Enable  */

        /* SDA */
        TSB_PA_IE_PA5IE   = 0;    /* Input      :Disable  */
        TSB_PA_CR_PA5C    = 0;    /* Output     :Disable  */
        TSB_PA_OD_PA5OD   = 1;    /* OD Control :Open Drain */
        TSB_PA_PUP_PA5UP  = 0;    /* Pull-up    :Disable  */
        TSB_PA_PDN_PA5DN  = 0;    /* Pull-down  :Disable  */
        TSB_PA_DATA_PA5   = 0;    /* Data       :0        */
        TSB_PA_FR2_PA5F2  = 1;    /* Function   :EI2C0SDA  */
        TSB_PA_IE_PA5IE   = 1;    /* Input      :Enable   */
        TSB_PA_CR_PA5C    = 1;    /* Output     :Enable   */
    }
}
#endif

/*--------------------------------------------------*/
/**
  * @brief     EI2C Port Setting (PL0, PL1)
  * @param     sda         :SDA port.
  * @param     scl         :SCL port.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void set_port_l(ei2c_port_t sda, ei2c_port_t scl)
{
    if ((sda == EI2C_PORT_PL1) && (scl == EI2C_PORT_PL0))
    {
        /* Port L  */
        TSB_CG_FSYSENA_IPENA10 = 1;     /* PORT  L   */
        
        /* SCL */
        TSB_PL_FR5_PL0F5  = 0;    /* Function    :Disable */
        TSB_PL_IE_PL0IE   = 0;    /* Input       :Disable */
        TSB_PL_CR_PL0C    = 0;    /* Output      :Disable */
        TSB_PL_OD_PL0OD   = 1;    /* OD Control  :Open Drain */
        TSB_PL_PUP_PL0UP  = 0;    /* Pull-up     :Disable */
        TSB_PL_PDN_PL0DN  = 0;    /* Pull-down   :Disable */
        TSB_PL_DATA_PL0   = 0;    /* Data        :0       */
        TSB_PL_FR4_PL0F4  = 1;    /* Function    :EI2C2SCL */
        TSB_PL_IE_PL0IE   = 1;    /* Input       :Enable  */
        TSB_PL_CR_PL0C    = 1;    /* Output      :Enable  */

        /* SDA */
        TSB_PL_FR5_PL1F5  = 0;    /* Function   :Disable */
        TSB_PL_IE_PL1IE   = 0;    /* Input      :Disable  */
        TSB_PL_CR_PL1C    = 0;    /* Output     :Disable  */
        TSB_PL_OD_PL1OD   = 1;    /* OD Control :Open Drain */
        TSB_PL_PUP_PL1UP  = 0;    /* Pull-up    :Disable  */
        TSB_PL_PDN_PL1DN  = 0;    /* Pull-down  :Disable  */
        TSB_PL_DATA_PL1   = 0;    /* Data       :0        */
        TSB_PL_FR4_PL1F4  = 1;    /* Function   :EI2C2SDA  */
        TSB_PL_IE_PL1IE   = 1;    /* Input      :Enable   */
        TSB_PL_CR_PL1C    = 1;    /* Output     :Enablee  */

    }
}

#if !defined(TMPM3HLA) && !defined(TMPM3HMA) && !defined(TMPM3HNA)
/*--------------------------------------------------*/
/**
  * @brief     EI2C Port Setting (PT1, PT0)
  * @param     sda         :SDA port.
  * @param     scl         :SCL port.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void set_port_t(ei2c_port_t sda, ei2c_port_t scl)
{
    if ((sda == EI2C_PORT_PT0) && (scl == EI2C_PORT_PT1))
    {
        /* Port T  */
        TSB_CG_FSYSENA_IPENA15 = 1;     /* PORT  T   */
        
        /* SCL */
        TSB_PT_IE_PT1IE   = 0;    /* Input       :Disable */
        TSB_PT_CR_PT1C    = 0;    /* Output      :Disable */
        TSB_PT_OD_PT1OD   = 1;    /* OD Control  :Open Drain */
        TSB_PT_PUP_PT1UP  = 0;    /* Pull-up     :Disable */
        TSB_PT_PDN_PT1DN  = 0;    /* Pull-down   :Disable */
        TSB_PT_DATA_PT1   = 0;    /* Data        :0       */
        TSB_PT_FR4_PT1F4  = 1;    /* Function    :EI2C3SCL */
        TSB_PT_IE_PT1IE   = 1;    /* Input       :Enable  */
        TSB_PT_CR_PT1C    = 1;    /* Output      :Enable  */

        /* SDA */
        TSB_PT_IE_PT0IE   = 0;    /* Input      :Disable  */
        TSB_PT_CR_PT0C    = 0;    /* Output     :Disable  */
        TSB_PT_OD_PT0OD   = 1;    /* OD Control :Open Drain */
        TSB_PT_PUP_PT0UP  = 0;    /* Pull-up    :Disable  */
        TSB_PT_PDN_PT0DN  = 0;    /* Pull-down  :Disable  */
        TSB_PT_DATA_PT0   = 0;    /* Data       :0        */
        TSB_PT_FR4_PT0F4  = 1;    /* Function   :EI2C3SDA  */
        TSB_PT_IE_PT0IE   = 1;    /* Input      :Enable   */
        TSB_PT_CR_PT0C    = 1;    /* Output     :Enable   */

    }
}
#endif


/*--------------------------------------------------*/
/**
  * @brief     EI2C Setting
  * @param     ch       :EI2C Channel.
  * @param     p_irqn   :Destination Address of a EI2C IRQn List.
  * @retval    non-zero :Instance Address.
  * @retval    zero     :Channel not supported.
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE uint32_t set_ei2c(uint8_t ch, uint32_t *p_irqn)
{   
    uint32_t instance = 0;

    switch (ch)
    {
        case EI2C_CH0:
            TSB_CG_FSYSMENB_IPMENB11 = 1;     /* EI2C   0   */
            instance = (uint32_t)TSB_EI2C0;
            *p_irqn = (uint32_t)&EI2C_CH0_IRQN_TBL;
            break;

#if !defined(TMPM3HLA)
        case EI2C_CH1:
            TSB_CG_FSYSMENB_IPMENB12 = 1;     /* EI2C   1   */
            instance = (uint32_t)TSB_EI2C1;
            *p_irqn = (uint32_t)&EI2C_CH1_IRQN_TBL;
            break;
#endif

        case EI2C_CH2:
            TSB_CG_FSYSMENB_IPMENB13 = 1;     /* EI2C   2   */
            instance = (uint32_t)TSB_EI2C2;
            *p_irqn = (uint32_t)&EI2C_CH2_IRQN_TBL;
            break;

#if !defined(TMPM3HLA) && !defined(TMPM3HMA) && !defined(TMPM3HNA)
        case EI2C_CH3:
            TSB_CG_FSYSMENB_IPMENB14 = 1;     /* EI2C   3   */
            instance = (uint32_t)TSB_EI2C3;
            *p_irqn = (uint32_t)&EI2C_CH3_IRQN_TBL;
            break;
#endif

        
        default:
            break;
    }
    return (instance);
}

/*--------------------------------------------------*/
/**
  * @brief     Reset Asynch Transfer
  * @param     p_obj    :ei2c object
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void reset_asynch(bsp_ei2c_t *p_obj)
{
    disable_irq(p_obj->info.irqn);
    EI2C_disable_interrupt(&p_obj->ei2c);
}

/*--------------------------------------------------*/
/**
  * @brief     EI2C Transfer handler
  * @param     p_obj     :ei2c object.
  * @retval    -
  * @note      Called by ei2c_irq_handler_asynch.
  */
/*--------------------------------------------------*/
static void ei2c_irq_handler(bsp_ei2c_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    if(p_obj->info.asynch.state != EI2C_TRANSFER_STATE_BUSY)
    {
        p_obj->info.asynch.event = EI2C_EVENT_ERROR;
        p_obj->info.asynch.state = EI2C_TRANSFER_STATE_IDLE;
    }
    else
    { 
        /* check ACK */
       if (!EI2C_get_ack(&p_obj->ei2c))
       {
          if (EI2C_transmitter(&p_obj->ei2c))
          {
              /* transmit data */
              if (p_obj->tx_buff.pos < p_obj->tx_buff.length)
              {
                 if(EI2C_status_tbe(&p_obj->ei2c))
                 {
                    EI2C_write_data(&p_obj->ei2c, (uint32_t)p_obj->tx_buff.p_buffer[p_obj->tx_buff.pos++]);
                 }
              }
           }
           else
           {
               /*  receive data */
               if (p_obj->rx_buff.pos < p_obj->rx_buff.length)
               {
                  /* read data */
                  p_obj->rx_buff.p_buffer[p_obj->rx_buff.pos++] = (uint8_t)EI2C_read_data(&p_obj->ei2c);
               
                  /* when next transmit over set ACKWAIT */
                  if (p_obj->rx_buff.pos == (p_obj->rx_buff.length-1))
                  {
                      EI2C_set_ack(&p_obj->ei2c,1);
                  }
                  /* when transmit over set ACKSEL_NACK */
                  else if (p_obj->rx_buff.pos == p_obj->rx_buff.length)
                  {
                     p_obj->ei2c.p_instance->ACR1 |= EI2CxCR1_ACKSEL_NACK;
                     EI2C_stop_condition(&p_obj->ei2c);                      
                  }
               }
            }
         }
         else
         {
             /* when status NACK,stop condition */
              EI2C_stop_condition(&p_obj->ei2c);
              p_obj->ei2c.p_instance->ASR1 |= EI2CxSR1_NACK;
         }
     }
   
    if (p_obj->info.asynch.state == EI2C_TRANSFER_STATE_IDLE)
    {
       reset_asynch(p_obj);
    }
}

/*--------------------------------------------------*/
/**
  * @brief     EI2C Slave handler
  * @param     p_obj     :ei2c object.
  * @retval    -
  * @note      Called by ei2c_slave_irq_handler_asynch.
  */
/*--------------------------------------------------*/
static void ei2c_slave_irq_handler(bsp_ei2c_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    if ((EI2C_master(&p_obj->ei2c)) || (p_obj->info.asynch.state != EI2C_TRANSFER_STATE_BUSY))
    {
        p_obj->info.asynch.event = EI2C_EVENT_ERROR;
        p_obj->info.asynch.state = EI2C_TRANSFER_STATE_IDLE;
    }
    else
    {
        if (EI2C_transmitter(&p_obj->ei2c))
        {
            if (!EI2C_get_ack(&p_obj->ei2c))
            {
                /* slave trnasmit(write data)  */
                if (p_obj->tx_buff.pos < p_obj->tx_buff.length)
                {
                    if(EI2C_status_tbe(&p_obj->ei2c))
                    {
                       EI2C_write_data(&p_obj->ei2c, (uint32_t)p_obj->tx_buff.p_buffer[p_obj->tx_buff.pos++]);
                    }
                }
            }
            else
            {
                /* error event not be set */
                p_obj->info.asynch.event = EI2C_EVENT_TRANSFER_COMPLETE;
                p_obj->info.asynch.state = EI2C_TRANSFER_STATE_IDLE;
            }
        }
        else
        {
            /*  receive data */
            if (p_obj->rx_buff.pos < p_obj->rx_buff.length)
            {
                /* read data */
                p_obj->rx_buff.p_buffer[p_obj->rx_buff.pos++] = (uint8_t)EI2C_read_data(&p_obj->ei2c);
               
                /* when next transmit over set ACKWAIT */
                if (p_obj->rx_buff.pos == (p_obj->rx_buff.length-1))
                {
                    p_obj->ei2c.p_instance->ACR1 |= EI2CxCR1_ACKWAIT;
                }
                /* when transmit over set ACKSEL_NACK */
                else if (p_obj->rx_buff.pos == p_obj->rx_buff.length)
                {
                    p_obj->ei2c.p_instance->ACR1 |= EI2CxCR1_ACKSEL_NACK;                   
                }
            }
        }
    }
 
    if (p_obj->info.asynch.state == EI2C_TRANSFER_STATE_IDLE)
    {
        reset_asynch(p_obj);
        EI2C_slave_init(&p_obj->ei2c);
    }
}

/*--------------------------------------------------*/
/**
  * @brief     Enable EI2C IRQ
  * @param     p_obj     :ei2c object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
void ei2c_enable_irq(bsp_ei2c_t *p_obj)
{
    enable_irq(p_obj->info.irqn);
}

/*--------------------------------------------------*/
/**
  * @brief     Disable EI2C IRQ
  * @param     p_obj     :ei2c object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
void ei2c_disable_irq(bsp_ei2c_t *p_obj)
{
    disable_irq(p_obj->info.irqn);
}

/*--------------------------------------------------*/
/**
  * @brief     Return the EI2c clock setting
  * @param     p_obj       :EI2C object.
  * @param     frequency   :Maximum frequency.
  * @param     fsys        :SystemCoreClock.
  * @param     p_setting   :Clock data pointer.
  * @retval    Non-zero    :Scl frequency.
  * @retval    0           :Error.
  * @note      -
  */
/*--------------------------------------------------*/
uint8_t ei2c_get_dnf_num(uint32_t dnf)
{
    uint8_t dnf_num = 0;

    switch(dnf)
    {
        case EI2CxCR0_DNF_1PRSCK:
        case EI2CxCR0_DNF_NA7:
        case EI2CxCR0_DNF_NA8:
             dnf_num =0;
             break;

        case EI2CxCR0_DNF_2PRSCK:
             dnf_num =1;
             break;

        case EI2CxCR0_DNF_3PRSCK:
             dnf_num =2;
             break;

        case EI2CxCR0_DNF_4PRSCK:
             dnf_num =3;
             break;

        case EI2CxCR0_DNF_5PRSCK:
             dnf_num =4;
             break;

        case EI2CxCR0_DNF_6PRSCK:
             dnf_num =5;
             break;

        default:
            dnf_num =0;
            break;  
    }
    return dnf_num;
}

/**
 *  @}
 */ /* End of group BSP_EI2C_Private_functions */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_EI2C_Exported_functions
 *  @{
 */

/*--------------------------------------------------*/
/**
  * @brief     Initialize the EI2C Driver
  * @param     p_obj       :ei2c object.
  * @param     sda         :SDA port.
  * @param     scl         :SCL port.
  * @retval    TXZ_SUCCESS :Success.
  * @retval    TXZ_ERROR   :Failure.
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
TXZ_Result ei2c_init(bsp_ei2c_t *p_obj, ei2c_port_t sda, ei2c_port_t scl)
{
    TXZ_Result result = TXZ_ERROR;
    uint32_t instance = 0;
    uint32_t irqn = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    /* ch0 */
    if ((sda == EI2C_PORT_PC1) && (scl == EI2C_PORT_PC0))
    {
        set_port_c(sda, scl);
        instance = set_ei2c(EI2C_CH0, &irqn);
    }
#if !defined(TMPM3HLA)
    /* ch1 */
    if ((sda == EI2C_PORT_PA5) && (scl == EI2C_PORT_PA4))
    {
        set_port_a(sda, scl);
        instance = set_ei2c(EI2C_CH1, &irqn);
    }
#endif
    /* ch2 */
    if ((sda == EI2C_PORT_PL1) && (scl == EI2C_PORT_PL0))
    {
        set_port_l(sda, scl);
        instance = set_ei2c(EI2C_CH2, &irqn);
    }
#if !defined(TMPM3HLA) && !defined(TMPM3HMA) && !defined(TMPM3HNA)
    /* ch3 */
    if ((sda == EI2C_PORT_PT0) && (scl == EI2C_PORT_PT1))
    {
        set_port_t(sda, scl);
        instance = set_ei2c(EI2C_CH3, &irqn);
    }
#endif

    if ((instance != 0) && (irqn != 0))
    {
        disable_irq(irqn);
        clear_irq(irqn);

        /* Set irqn table */
        p_obj->info.irqn = irqn;

        /* Set instance */
        p_obj->ei2c.p_instance = (TSB_EI2C_TypeDef *)instance;

        /* EI2C Reset */
        ei2c_reset(p_obj);

        /* set prsclk scl information */
        result = ei2c_set_prs_scl(&p_obj->ei2c);

        if( result != TXZ_ERROR )
        {
          /* control0 Register setting */
          p_obj->ei2c.init.cr0.toper = EI2CxCR0_TOPER_4PRSCK;
          p_obj->ei2c.init.cr0.toe   = EI2CxCR0_TOE_DISABLE;
          p_obj->ei2c.init.cr0.espe  = EI2CxCR0_ESPE_ENABLE;
          p_obj->ei2c.init.cr0.este  = EI2CxCR0_ESTE_ENABLE;
          p_obj->ei2c.init.cr0.nacke = EI2CxCR0_NACKE_DISABLE;
          p_obj->ei2c.init.cr0.gce   = EI2CxCR0_GCE_ENABLE;
          p_obj->ei2c.init.cr0.ale   = EI2CxCR0_ALE_ENABLE;

          /* EI2C driver Register init */
          EI2C_init(&p_obj->ei2c);
        }
    }
    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief     Return the EI2c clock setting
  * @param     p_obj       :EI2C object.
  * @param     frequency   :Maximum frequency.
  * @param     fsys        :SystemCoreClock.
  * @param     p_setting   :Clock data pointer.
  * @retval    Non-zero    :Scl frequency.
  * @retval    0           :Error.
  * @note      -
  */
/*--------------------------------------------------*/
TXZ_Result ei2c_set_prs_scl_dig(ei2c_t *p_obj)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    uint32_t frequency,prs,prs_tmp;
    uint32_t prsck;
    uint32_t scl_tf,scl_tr;
    uint32_t sclh,sclh_tmp,scll;
    int32_t scll_tmp;
    uint32_t t_total,t_high,t_low;
    uint32_t low_width_min;
    uint32_t high_width_min;
    cg_t     paramCG ;
    uint32_t fsys;
    uint8_t dnf;

    frequency = p_obj->init.clock.sck;
    paramCG.p_instance = TSB_CG; 
    cg_get_system_core_clock(&paramCG);
    fsys = SystemCoreClock ;
    dnf=ei2c_get_dnf_num(p_obj->init.cr0.dnf);

    /* unuse value init */
    prs  = EI2C_UNUSE_VALUE;
    sclh = EI2C_UNUSE_VALUE;
    scll = EI2C_UNUSE_VALUE;

    if(frequency <= EI2CSCL_FRQ_FMP)
    {
        /* STD */
        if(frequency <= EI2CSCL_FRQ_STD)
        {
            high_width_min = EI2CSCL_SCLH_STDMIN;
            low_width_min  = EI2CSCL_SCLL_STDMIN;
            scl_tf = EI2CSCL_TF_STD;
            scl_tr = EI2CSCL_TR_STD;
        }
        /* FM */
        else if(frequency <= EI2CSCL_FRQ_FM)
        {
            high_width_min = EI2CSCL_SCLH_FMMIN;
            low_width_min  = EI2CSCL_SCLL_FMMIN;
            scl_tf = EI2CSCL_TF_FM;
            scl_tr = EI2CSCL_TR_FM;
        }
        /* FMP */
        else
        {
            high_width_min = EI2CSCL_SCLH_FMPMIN;
            low_width_min  = EI2CSCL_SCLL_FMPMIN;
            scl_tf = EI2CSCL_TF_FMP;
            scl_tr = EI2CSCL_TR_FMP;
        }

        /*---------------------------------------------------------*/
        /* freq(hz)=(1/(tHIGH(ns)+tLOW(ns)+tr(ns)+tf(ns)))*(10^9)  */
        /* tHIGH(ns)+tLOW(ns)=(10^9/freq(hz))-tr(ns)-tf(ns)        */
        /*---------------------------------------------------------*/
        t_total = (EI2C_UNIT_1US*1000/frequency) - scl_tf - scl_tr;

        /* if small than min return */
        if(t_total < (high_width_min + low_width_min))
        {
           result = TXZ_ERROR;
           return result;
        }

       for (prs_tmp = 0; prs_tmp < EI2CxPRS_MAX; prs_tmp++)
       {
           /*-----------------------------------------------*/
           /* clock width (s) = 1 / PCLK                    */
           /* clock width (s) = 1 / (fsys / PRS)           */
           /* clock width(ns) = (10^9) / (fsys / PRS)      */
           /*-----------------------------------------------*/
           /* NFSEL=0: (50ns/(<DNF>+1))<=clock width(ns)    */
           /*            <=(tHIGH(Min)/(<DNF>+2)            */
           /* ((10^9)*(<DNF>+2))/(tHIGH(Min) <= (fsys/PRS) */
           /*            <=((10^9)*(<DNF>+1))/(50ns)        */
           /*-----------------------------------------------*/
           /* PRS: 1~64   DNF:1~6                           */
           /* (tHIGH(Min): Use @ref EI2CSCL_MAX             */
           /*-----------------------------------------------*/
           prsck = ((uint32_t)fsys /((prs_tmp+1)*1000));

           if((prsck < ((EI2C_UNIT_1US/high_width_min)*(dnf+2)))
           || (prsck > ((EI2C_UNIT_1US/50)*(dnf+1))))
           {
              continue;
           }

           
           if(t_total == (high_width_min + low_width_min))
           {
                t_high = high_width_min;
                t_low = low_width_min;
                sclh_tmp = ((t_high*prsck)/EI2C_UNIT_1US) - 3 - dnf;
                scll_tmp = ((t_low*prsck)/EI2C_UNIT_1US) - 3 - dnf;
               
                /* paramater check */
                if(( sclh_tmp <= EI2CxSCL_MAX )
                && ( scll_tmp <= EI2CxSCL_MAX ))
                {
                     prs  = prs_tmp;
                     sclh = sclh_tmp;
                     scll = scll_tmp;
                     break;
                }
           }
           else
           {
               /*-----------------------------------------------*/
               /* t_high(ns)=((SCLH+3+DNF)/prsck(Hz))*(10^9)    */
               /* t_low(ns) =((SCLL+3+DNF)/prsck(Hz))*(10^9)    */
               /* t_high(ns) + t_low(ns) = t_total              */
               /* t_high(ns)>= high_width_min                   */
               /* t_low(ns) >= low_width_min                    */
               /*-----------------------------------------------*/
               for(sclh_tmp = 0; sclh_tmp <= EI2CxSCL_MAX; sclh_tmp++)
               {
                   t_high = ((sclh_tmp + dnf + 3)*EI2C_UNIT_1US)/prsck;
                   t_low  = t_total - t_high;

                   /*  paramater check */
                   if((t_high < high_width_min)
                   || (t_low  < low_width_min))
                   {
                      continue;
                   }
                   
                   scll_tmp = ((t_low*prsck)/EI2C_UNIT_1US) - 3 - dnf;
                   
                   /* paramater check */
                   if(( scll_tmp >= 0 )
                   && ( scll_tmp <= EI2CxSCL_MAX ))
                   {
                       prs  = prs_tmp;
                       sclh = sclh_tmp;
                       scll = scll_tmp;
                       break;
                   }
               }
            }

           /*  when set over break the loop  */
           if((prs != EI2C_UNUSE_VALUE)
           &&(sclh != EI2C_UNUSE_VALUE)
           &&(scll != EI2C_UNUSE_VALUE))
           {
              break;
           }
       }

       /* set the paramater value */
       p_obj->init.clock.prs = prs;
       p_obj->init.clock.sclh = sclh;
       p_obj->init.clock.scll = scll;
   }
   else
   {
      result = TXZ_ERROR;
   }

return (result);
}
/*--------------------------------------------------*/
/**
  * @brief     Return the EI2c clock setting
  * @param     p_obj       :EI2C object.
  * @param     frequency   :Maximum frequency.
  * @param     fsys        :SystemCoreClock.
  * @param     p_setting   :Clock data pointer.
  * @retval    Non-zero    :Scl frequency.
  * @retval    0           :Error.
  * @note      -
  */
/*--------------------------------------------------*/
TXZ_Result ei2c_set_prs_scl_ana(ei2c_t *p_obj)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    uint32_t frequency,prs,prs_tmp;
    uint32_t prsck;
    uint32_t scl_tf,scl_tr;
    uint32_t sclh,sclh_tmp,scll;
    int32_t scll_tmp;
    uint32_t t_total,t_high,t_low;
    uint32_t low_width_min;
    uint32_t high_width_min;
    cg_t     paramCG ;
    uint32_t fsys;

    frequency = p_obj->init.clock.sck;
    paramCG.p_instance = TSB_CG; 
    cg_get_system_core_clock(&paramCG);
    fsys = SystemCoreClock ;



    /* unuse value init */
    prs  = EI2C_UNUSE_VALUE;
    sclh = EI2C_UNUSE_VALUE;
    scll = EI2C_UNUSE_VALUE;

    if(frequency <= EI2CSCL_FRQ_FMP)
    {
        /* STD */
         if(frequency <= EI2CSCL_FRQ_STD)
         {
            high_width_min = EI2CSCL_SCLH_STDMIN;
            low_width_min  = EI2CSCL_SCLL_STDMIN;
            scl_tf = EI2CSCL_TF_STD;
            scl_tr = EI2CSCL_TR_STD;
         }
         /* FM */
         else if(frequency <= EI2CSCL_FRQ_FM)
         {
             high_width_min = EI2CSCL_SCLH_FMMIN;
             low_width_min  = EI2CSCL_SCLL_FMMIN;
             scl_tf = EI2CSCL_TF_FM;
             scl_tr = EI2CSCL_TR_FM;
         }
         /* FMP */
         else
         {
             high_width_min = EI2CSCL_SCLH_FMPMIN;
             low_width_min  = EI2CSCL_SCLL_FMPMIN;
             scl_tf = EI2CSCL_TF_FMP;
             scl_tr = EI2CSCL_TR_FMP;
         }

        /*---------------------------------------------------------*/
        /* freq(hz)=(1/(tHIGH(ns)+tLOW(ns)+tr(ns)+tf(ns)))*(10^9)  */
        /* tHIGH(ns)+tLOW(ns)=(10^9/freq(hz))-tr(ns)-tf(ns)        */
        /*---------------------------------------------------------*/
        t_total = (EI2C_UNIT_1US*1000/frequency) - scl_tf - scl_tr;
        
        /* if small than min return */
        if(t_total < (high_width_min + low_width_min))
        {
            result = TXZ_ERROR;
            return result;
        }
        
        for (prs_tmp = 0; prs_tmp < EI2CxPRS_MAX; prs_tmp++)
        {
            /*-----------------------------------------------*/
            /* clock width (s) = 1 / PRSCK                   */
            /* clock width (s) = 1 / (fsys /(PRS+1))        */
            /* clock width(ns) = (10^9) / (fsys/(PRS+1))    */
            /*-----------------------------------------------*/
            /* NFSEL=1: clock width(ns)<= (tHIGH(Min)/2)     */
            /* ((10^9)*2)/(tHIGH(Min)) <= (fsys/(PRS+1))    */
            /*-----------------------------------------------*/
            /* PRS: 0~63                                     */
            /* (tHIGH(Min): Use @ref EI2CSCL_MAX             */
            /*-----------------------------------------------*/
            prsck = ((uint32_t)fsys /((prs_tmp+1)*1000));

            if(prsck < ((EI2C_UNIT_1US*2)/high_width_min))
            {
               continue;
            }

            /*-----------------------------------------------*/
            /* t_high(ns)=((SCLH+ 2)/prsck(Hz))*(10^9)       */
            /* t_low(ns) =((SCLL+ 2)/prsck(Hz))*(10^9)       */
            /* t_high(ns) + t_low(ns) = t_total              */
            /* t_high(ns)>= high_width_min                   */
            /* t_low(ns) >= low_width_min                    */
            /*-----------------------------------------------*/
            for(sclh_tmp = 0; sclh_tmp <= EI2CxSCL_MAX; sclh_tmp++)
            {
               t_high = ((sclh_tmp + 2)*EI2C_UNIT_1US)/prsck;
               t_low  = t_total - t_high;
               
               /*  paramater chec */
               if((t_high < high_width_min)
               || (t_low  < low_width_min))
               {
                  continue;
               }
               
               scll_tmp = (t_low*prsck)/EI2C_UNIT_1US - 2;
               
               /*  paramater chec */
               if(( scll_tmp >= 0 )
               && ( scll_tmp <= EI2CxSCL_MAX ))
               {
                  prs  = prs_tmp;
                  sclh = sclh_tmp;
                  scll = scll_tmp;
                  break;
               }
            }

            /* when set over break the loop */
            if((prs != EI2C_UNUSE_VALUE)
            &&(sclh != EI2C_UNUSE_VALUE)
            &&(scll != EI2C_UNUSE_VALUE))
            {
              break;
            }
       }

       /* set the paramater value */
       p_obj->init.clock.prs = prs;
       p_obj->init.clock.sclh = sclh;
       p_obj->init.clock.scll = scll;
    }
    else
    {
        result = TXZ_ERROR;
    }

    return (result);
}
/*--------------------------------------------------*/
/**
  * @brief     Return the EI2c clock setting
  * @param     p_obj       :EI2C object.
  * @param     frequency   :Maximum frequency.
  * @param     fsys        :SystemCoreClock.
  * @param     p_setting   :Clock data pointer.
  * @retval    Non-zero    :Scl frequency.
  * @retval    0           :Error.
  * @note      -
  */
/*--------------------------------------------------*/
TXZ_Result ei2c_set_prs_scl(ei2c_t *p_obj)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    /* Digtal noise filter */
    if(p_obj->init.cr0.nfsel == EI2CxCR0_NFSEL_DIG)
    {
       result = ei2c_set_prs_scl_dig(p_obj);
    }
    /* Analog noise filter */
    else if(p_obj->init.cr0.nfsel == EI2CxCR0_NFSEL_ANA)
    {
       result = ei2c_set_prs_scl_ana(p_obj);
    }
    /* other */
    else
    {
       result = TXZ_ERROR;
    }

    return (result);
}
/*--------------------------------------------------*/
/**
  * @brief     Reset EI2C peripheral
  * @param     p_obj       :ei2c object.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void ei2c_reset(bsp_ei2c_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    /* Software reset */
    EI2C_reset(&p_obj->ei2c);
}

/*--------------------------------------------------*/
/**
  * @brief     Reset EI2C peripheral
  * @param     p_obj       :ei2c object.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void ei2c_acr1_clear(bsp_ei2c_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    /* Software reset */
    EI2C_acr1_clear(&p_obj->ei2c);
}

/*--------------------------------------------------*/
/**
  * @brief     Attempts to determine if the EI2C bus is already in use
  * @param     p_obj     :ei2c object.
  * @retval    0         :Non-active.
  * @retval    1         :Active.
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
uint8_t ei2c_active(bsp_ei2c_t *p_obj)
{
    uint8_t result;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    if (EI2C_status_busy(&p_obj->ei2c))
    {
        result = 1;     
    }
    else
    {
        result = 0;
    }
    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief     Start EI2C asynchronous transfer
  * @param     p_obj     :ei2c object.
  * @param     p_tx      :Buffer of write data.
  * @param     tx_length :Length of write data.
  * @param     p_rx      :Buffer of read data.
  * @param     rx_length :Length of read data.
  * @param     address   :Slave address(7-bit) and last bit is 0.
  * @param     stop      :Stop to be generated after the transfer is done.
  * @retval    TXZ_SUCCESS :Success.
  * @retval    TXZ_ERROR   :Failure.
  * @note      Master and non-blocking function. 
  * @note      Events of this function will be notified on ei2c_irq_handler_asynch.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
TXZ_Result ei2c_transfer_asynch(bsp_ei2c_t *p_obj, uint8_t *p_tx, int32_t tx_length, uint8_t *p_rx, int32_t rx_length, int32_t address, int32_t stop)
{
    TXZ_Result result = TXZ_ERROR;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(check_param_address(address));
#endif /* #ifdef __DEBUG__ */

    if (p_obj->info.asynch.state == EI2C_TRANSFER_STATE_IDLE)
    {
        if (((p_tx != EI2C_NULL) && (tx_length > 0)) || ((p_rx != EI2C_NULL) && (rx_length > 0)))
        {
            reset_asynch(p_obj);
            EI2C_clear_int_status(&p_obj->ei2c);
            clear_irq(p_obj->info.irqn);
            p_obj->info.asynch.address = (uint32_t)address;
            p_obj->info.asynch.event = 0;
            p_obj->info.asynch.stop = (uint32_t)stop;
            p_obj->tx_buff.p_buffer = p_tx;
            p_obj->tx_buff.length = (uint32_t)tx_length;
            p_obj->tx_buff.pos = 0;
            p_obj->rx_buff.p_buffer = p_rx;
            p_obj->rx_buff.length = (uint32_t)rx_length;
            p_obj->rx_buff.pos = 0;
            p_obj->info.asynch.state = EI2C_TRANSFER_STATE_BUSY;
            EI2C_enable_interrupt(&p_obj->ei2c);
            enable_irq(p_obj->info.irqn);
            EI2C_startcondition(&p_obj->ei2c);
                 
            p_obj->info.bus_free = 0;
            p_obj->info.start = 0;
            result = TXZ_SUCCESS;
        }
    }
    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief     ei2c_master_st_irq_handler
  * @param     p_obj     :ei2c object.
  * @retval    -
  * @note      ei2c status interrupt.
  */
/*--------------------------------------------------*/
uint32_t ei2c_master_st_irq_handler(bsp_ei2c_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    if(EI2C_status_arbitration(&p_obj->ei2c))
    {
       EI2C_clear_int_status(&p_obj->ei2c);
       p_obj->info.asynch.event = EI2C_EVENT_BUS_ARBITRATION;
       p_obj->info.asynch.state = EI2C_TRANSFER_STATE_IDLE;  
    }
    else if(EI2C_status_toerr(&p_obj->ei2c))
    {
       EI2C_clear_int_status(&p_obj->ei2c);
       p_obj->info.asynch.event = EI2C_EVENT_ERROR;
       p_obj->info.asynch.state = EI2C_TRANSFER_STATE_IDLE;  
    }
    else if(EI2C_status_stcf(&p_obj->ei2c))
    {

        /* clear start condition&tbe status */
        EI2C_clear_int_stcf(&p_obj->ei2c);
        /* transmit over change to receive */
        if((p_obj->tx_buff.length == 0)
        && (p_obj->rx_buff.length != 0))
        {
          if (p_obj->rx_buff.length == 1) 
          {
             EI2C_set_ack(&p_obj->ei2c,1);          
          }
          /* clear restart condition&tbe status */
          EI2C_write_data(&p_obj->ei2c, (uint32_t)(p_obj->info.asynch.address|1U));   
        }
        else if(p_obj->tx_buff.length != 0)
        {
          EI2C_write_data(&p_obj->ei2c, (uint32_t)p_obj->info.asynch.address);    
        }


        EI2C_clear_int_tbe(&p_obj->ei2c);  
    }
    /* restart condition */
    else if(EI2C_status_rscf(&p_obj->ei2c))
    {
        /* clear restart condition&tbe status */
         EI2C_clear_int_rscf(&p_obj->ei2c);
        /* transmit over change to receive */
        if((p_obj->tx_buff.pos == p_obj->tx_buff.length)
        && (p_obj->rx_buff.length != 0))
        {
          if (p_obj->rx_buff.length == 1) 
          {
             EI2C_set_ack(&p_obj->ei2c,1);          
          }
          EI2C_write_data(&p_obj->ei2c, (uint32_t)(p_obj->info.asynch.address|1U));   
        }
              
        EI2C_clear_int_tbe(&p_obj->ei2c);  
    }
    else if(EI2C_status_spcf(&p_obj->ei2c))
    {
         EI2C_clear_int_status(&p_obj->ei2c);
         p_obj->info.asynch.event = EI2C_EVENT_TRANSFER_COMPLETE;
         p_obj->info.asynch.state = EI2C_TRANSFER_STATE_IDLE;
    }
    else if(EI2C_status_tend(&p_obj->ei2c))
    {
         if(p_obj->tx_buff.pos < p_obj->tx_buff.length)
         {
              EI2C_clear_int_tend(&p_obj->ei2c);
         }
         /* check the receive buff */
         else if(p_obj->rx_buff.length != 0)
         {
             /* restart condition*/
             EI2C_clear_int_tend(&p_obj->ei2c);
             EI2C_restartcondition(&p_obj->ei2c);
         }
         else
         {
             /* transmit over clear the tend and stop condition */
             EI2C_stop_condition(&p_obj->ei2c);
             EI2C_clear_int_tend(&p_obj->ei2c); 
         }
    }
 
    if (p_obj->info.asynch.state == EI2C_TRANSFER_STATE_IDLE)
    {
       reset_asynch(p_obj);
    }
    
    return (p_obj->info.asynch.event & EI2C_EVENT_ALL);
}

/*--------------------------------------------------*/
/**
  * @brief     ei2c_slave_st_irq_handler
  * @param     p_obj     :ei2c object.
  * @retval    -
  * @note      ei2c status interrupt.
  */
/*--------------------------------------------------*/
uint32_t ei2c_slave_st_irq_handler(bsp_ei2c_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    if((p_obj->ei2c.p_instance->ASR1 & EI2CxSR1_AAS1) == EI2CxSR1_AAS1)
    {
        /* clear AAS1 status */        
         p_obj->ei2c.p_instance->ASR1 |= EI2CxSR1_AAS1; 
    }
    else if((p_obj->ei2c.p_instance->ASR1 & EI2CxSR1_AAS2) == EI2CxSR1_AAS2)
    {
        /* clear AAS1 status */        
         p_obj->ei2c.p_instance->ASR1 |= EI2CxSR1_AAS2; 
    }
    else if(EI2C_status_spcf(&p_obj->ei2c))
    {
         EI2C_clear_int_status(&p_obj->ei2c);
         p_obj->info.asynch.event = EI2C_EVENT_TRANSFER_COMPLETE;
         p_obj->info.asynch.state = EI2C_TRANSFER_STATE_IDLE;
    }
    else if(EI2C_status_tend(&p_obj->ei2c))
    {
         EI2C_clear_int_tend(&p_obj->ei2c);
    }

    if (p_obj->info.asynch.state == EI2C_TRANSFER_STATE_IDLE)
    {
        reset_asynch(p_obj);
        EI2C_slave_init(&p_obj->ei2c);
    }

    return (p_obj->info.asynch.event & EI2C_EVENT_ALL);
}
/*--------------------------------------------------*/
/**
  * @brief     The asynchronous IRQ handler
  * @param     p_obj     :ei2c object.
  * @retval    zero      :Transfer in progress.
  * @retval    non-zero  :Event information.
  * @note      Macro definition of return values is @ref EI2C_Events.
  * @attention This function should be implement as INTI2Cx_IRQHandler.
  */
/*--------------------------------------------------*/
uint32_t ei2c_irq_handler_asynch(bsp_ei2c_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    ei2c_irq_handler(p_obj);

    return (p_obj->info.asynch.event & EI2C_EVENT_ALL);
}

/*--------------------------------------------------*/
/**
  * @brief     Abort asynchronous transfer
  * @param     p_obj     :ei2c object.
  * @retval    -
  * @note      After error event occurred on ei2c_irq_handler_asynch, 
  * @note      call this function and clear error status. 
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void ei2c_abort_asynch(bsp_ei2c_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    reset_asynch(p_obj);
    p_obj->info.asynch.state = EI2C_TRANSFER_STATE_IDLE;
    ei2c_reset(p_obj);
    EI2C_init(&p_obj->ei2c);
    clear_irq(p_obj->info.irqn);
}

/*--------------------------------------------------*/
/**
  * @brief     Configure EI2C as slave or master.
  * @param     p_obj        :ei2c object.
  * @param     enable_slave :Enable slave mode.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void ei2c_slave_mode(bsp_ei2c_t *p_obj, int32_t enable_slave)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    disable_irq(p_obj->info.irqn);

    if (enable_slave)
    {
        EI2C_slave_init(&p_obj->ei2c);
    }
    else
    {
        /* Slave Disable Settings. */
        ei2c_reset(p_obj);
        EI2C_init(&p_obj->ei2c);
    }
    p_obj->info.bus_free = 0;
    p_obj->info.start = 0;
    EI2C_clear_int_status(&p_obj->ei2c);
}

/*--------------------------------------------------*/
/**
  * @brief     Configure EI2C slave address.
  * @param     p_obj     :ei2c object.
  * @param     address   :Address to be set.
  * @param     bit7      :7bit Address or 10bit address.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void ei2c_slave_address1(bsp_ei2c_t *p_obj, uint32_t address,int32_t bit7)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(check_param_address((int32_t)address));
#endif /* #ifdef __DEBUG__ */

    EI2C_set_address1(&p_obj->ei2c, address,bit7);
}


/*--------------------------------------------------*/
/**
  * @brief     Start EI2C asynchronous transfer
  * @param     p_obj     :ei2c object.
  * @param     p_tx      :Buffer of write data.
  * @param     tx_length :Length of write data.
  * @param     p_rx      :Buffer of read data.
  * @param     rx_length :Length of read data.
  * @retval    TXZ_SUCCESS :Success.
  * @retval    TXZ_ERROR   :Failure.
  * @note      Slave and non-blocking function.
  * @note      Events of this function will be notified on ei2c_slave_irq_handler_asynch.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
TXZ_Result ei2c_slave_transfer_asynch(bsp_ei2c_t *p_obj, uint8_t *p_tx, int32_t tx_length, uint8_t *p_rx, int32_t rx_length)
{
    TXZ_Result result = TXZ_ERROR;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    if (p_obj->info.asynch.state == EI2C_TRANSFER_STATE_IDLE)
    {
        if (((p_tx != EI2C_NULL) && (tx_length > 0)) || ((p_rx != EI2C_NULL) && (rx_length > 0)))
        {
            reset_asynch(p_obj);
            EI2C_clear_int_status(&p_obj->ei2c);
            clear_irq(p_obj->info.irqn);
            p_obj->info.asynch.address = 0;
            p_obj->info.asynch.event = 0;
            p_obj->info.asynch.stop = 0;
            p_obj->tx_buff.p_buffer = p_tx;
            p_obj->tx_buff.length = (uint32_t)tx_length;
            p_obj->tx_buff.pos = 0;
            p_obj->rx_buff.p_buffer = p_rx;
            p_obj->rx_buff.length = (uint32_t)rx_length;
            p_obj->rx_buff.pos = 0;
            p_obj->info.asynch.state = EI2C_TRANSFER_STATE_BUSY;
            EI2C_enable_interrupt(&p_obj->ei2c);
            enable_irq(p_obj->info.irqn);
            result = TXZ_SUCCESS;
        }
    }
    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief     The asynchronous IRQ handler
  * @param     p_obj     :ei2c object.
  * @retval    zero      :Transfer in progress.
  * @retval    non-zero  :Event information.
  * @note      Macro definition of return values is @ref EI2C_Events.
  * @attention This function should be implement as INTI2Cx_IRQHandler.
  */
/*--------------------------------------------------*/
uint32_t ei2c_slave_irq_handler_asynch(bsp_ei2c_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    ei2c_slave_irq_handler(p_obj);

    return (p_obj->info.asynch.event & EI2C_EVENT_ALL);
}

/*--------------------------------------------------*/
/**
  * @brief     Abort asynchronous transfer
  * @param     p_obj     :ei2c object.
  * @retval    -
  * @note      For a non-blocking function.
  * @note      After error event occurred on ei2c_slave_irq_handler_asynch, 
  * @note      call this function and clear error status. 
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void ei2c_slave_abort_asynch(bsp_ei2c_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
#endif /* #ifdef __DEBUG__ */

    reset_asynch(p_obj);
    p_obj->info.asynch.state = EI2C_TRANSFER_STATE_IDLE;
    EI2C_slave_init(&p_obj->ei2c);
    EI2C_clear_int_status(&p_obj->ei2c);
    clear_irq(p_obj->info.irqn);
}

/**
 *  @}
 */ /* End of group BSP_EI2C_Exported_functions */

/**
 *  @} 
 */ /* End of group BSP_EI2C */

/**
 *  @} 
 */ /* End of group BSP */

#endif /* defined(__BSP_EI2C_H)  */

#ifdef __cplusplus
}
#endif /* __cplusplus */
