/** 
 *******************************************************************************
 * @file    bsp_config_adbun_m3hqa_ei2c.h
 * @brief   This file provides all the functions prototypes for EI2C Driver.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
/*------------------------------------------------------------------------------*/
/*  Define to prevent recursive inclusion                                       */
/*------------------------------------------------------------------------------*/
#ifndef __BSP_EI2C_H
#define __BSP_EI2C_H

#ifdef __cplusplus
extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "ei2c.h"
#include "cg.h"

/**
 *  @addtogroup BSP BSP
 *  @{
 */

/** 
 *  @addtogroup BSP_EI2C
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_EI2C_Exported_macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group BSP_EI2C_Exported_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_EI2C_Exported_define
 *  @{
 */
/** 
 *  @defgroup EI2C_NullPointer  Null Pointer
 *  @brief    EI2C NULL Pointer.
 *  @{
 */
#define EI2C_NULL        ((void *)0)
/**
 *  @}
 */ /* End of group EI2C_NullPointer */

/** 
 *  @}
 */ /* End of group BSP_EI2C_Exported_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_EI2C_Exported_define
 *  @{
 */

/**
 * @defgroup EI2C_ACK EI2C ACK Macros
 * @brief    EI2C Type of Acknowledge.
 * @{
 */
#define EI2C_NACK                    (0)         /*!< NACK Received.             */
#define EI2C_ACK                     (1)         /*!< ACK Received.              */
/**
 *  @}
 */ /* End of group EI2C_ACK */


/**
 * @defgroup EI2C_ERROR I2C ERROR Macros
 * @brief    EI2C Error definitions.
 * @{
 */
#define EI2C_ERROR_NO_ERROR          (0)         /*!< No Error.                          */
#define EI2C_ERROR_NO_SLAVE          (-1)        /*!< No Slave Error.                    */
#define EI2C_ERROR_BUS_BUSY          (-2)        /*!< Bus Busy Error.(now, not support)  */
#define EI2C_ERROR_PARAM             (-3)        /*!< Parameter Error.                   */
#define EI2C_ERROR_OTHERS            (-4)        /*!< Others Error.                      */
#define EI2C_ERROR_ARBITRATION       (-5)        /*!< Arbitration Error.                 */
/**
 *  @}
 */ /* End of group EI2C_ERROR */

/** 
 *  @name  EI2C Macro Definition.
 *  @brief EI2C frequency Macro Definition.
 *  @{
 */
#define EI2C_UNUSE_VALUE      ((uint32_t)0xFFFFFFFF)  /*!< UNUSE Value             */
#define EI2C_UNIT_1US         ((uint32_t)1000000)     /*!< 1S=1000000us            */
/**
 *  @}
 */ /* End of name EI2C Definition */
 
/** 
 *  @name  EI2CSCL_MAX Macro Definition.
 *  @brief EI2CSCL_MAX frequency Macro Definition.
 *  @{
 */
#define EI2CSCL_FRQ_STD       ((uint32_t)100000)    /*!< STD  100kHz             */
#define EI2CSCL_FRQ_FM        ((uint32_t)400000)    /*!< Fm   400kHz             */
#define EI2CSCL_FRQ_FMP       ((uint32_t)1000000)   /*!< Fm+ 1000kHz             */
#define EI2CSCL_SCLH_STDMIN   ((uint32_t)4000)      /*!< SCLH  STD MIN           */
#define EI2CSCL_SCLH_FMMIN    ((uint32_t)600)       /*!< SCLH  FM  MIN           */
#define EI2CSCL_SCLH_FMPMIN   ((uint32_t)260)       /*!< SCLH  FMP MIN           */
#define EI2CSCL_SCLL_STDMIN   ((uint32_t)4700)      /*!< SCLL  STD MIN           */
#define EI2CSCL_SCLL_FMMIN    ((uint32_t)1300)      /*!< SCLL  FM  MIN           */
#define EI2CSCL_SCLL_FMPMIN   ((uint32_t)500)       /*!< SCLL  FMP MIN           */
#define EI2CSCL_TR_STD        ((uint32_t)1000)      /*!< TR  STD MAX(100000)     */
#define EI2CSCL_TF_STD        ((uint32_t)300)       /*!< TF  STD MAX(100000)     */
#define EI2CSCL_TR_FM         ((uint32_t)300)       /*!< TR  FM  MAX(400000)     */
#define EI2CSCL_TF_FM         ((uint32_t)300)       /*!< TF  FM  MAX(400000)     */
#define EI2CSCL_TR_FMP        ((uint32_t)120)       /*!< TR  FMP MAX(1000000)    */
#define EI2CSCL_TF_FMP        ((uint32_t)120)       /*!< TF  FMP MAX(1000000)    */
/**
 *  @}
 */ /* End of name EI2CSCL_MAX Macro Definition */

/**
 * @defgroup EI2C_Events EI2C Events Macros
 * @brief    EI2C Asynch Events.
 * @{
 */
#define EI2C_EVENT_ERROR               (1 << 1)  /*!< Error.                     */
#define EI2C_EVENT_ERROR_NO_SLAVE      (1 << 2)  /*!< No Slave.                  */
#define EI2C_EVENT_TRANSFER_COMPLETE   (1 << 3)  /*!< Transfer Complete.         */
#define EI2C_EVENT_TRANSFER_EARLY_NACK (1 << 4)  /*!< End of Transfer.           */
#define EI2C_EVENT_BUS_BUSY            (1 << 5)  /*!< bus busy.                  */
#define EI2C_EVENT_BUS_ARBITRATION     (1 << 6)  /*!< arbitration.               */
#define EI2C_EVENT_ALL                 (EI2C_EVENT_ERROR |  EI2C_EVENT_TRANSFER_COMPLETE | EI2C_EVENT_ERROR_NO_SLAVE | EI2C_EVENT_TRANSFER_EARLY_NACK | EI2C_EVENT_BUS_BUSY | EI2C_EVENT_BUS_ARBITRATION)
/**
 *  @}
 */ /* End of group EI2C_Events */

/**
 * @defgroup EI2C_SlaveReceive EI2C Slave Receive Return Macros
 * @brief    EI2C Received Contents of Slave.
 * @{
 */
#define EI2C_NO_DATA             (0)     /*!< the slave has not been addressed. */
#define EI2C_READ_ADDRESSED      (1)     /*!< the master has requested a read from this slave. */
#define EI2C_WRITE_GENERAL       (2)     /*!< the master is writing to all slave.(now, not support) */
#define EI2C_WRITE_ADDRESSED     (3)     /*!< the master is writing to this slave. */
/**
 *  @}
 */ /* End of group EI2C_SlaveReceive */

/** 
 *  @}
 */ /* End of group BSP_EI2C_Exported_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_EI2C_Exported_define
 *  @{
 */

/*----------------------------------*/
/** 
 * @brief  ei2c Port Enumerated Type Definition.
*/
/*----------------------------------*/
typedef enum {
  EI2C_PORT_PC0 = 0,     /*!< 0: PC0 EI2C0(SCL) */  
  EI2C_PORT_PC1,         /*!< 1: PC1 EI2C0(SDA) */  
  EI2C_PORT_PA4,         /*!< 2: PA4 EI2C1(SCL) */  
  EI2C_PORT_PA5,         /*!< 3: PA5 EI2C1(SDA) */  
  EI2C_PORT_PL0,         /*!< 4: PL0 EI2C2(SCL) */  
  EI2C_PORT_PL1,         /*!< 5: PL1 EI2C2(SDA) */  
  EI2C_PORT_PT0,         /*!< 6: PT0 EI2C3(SDA) */  
  EI2C_PORT_PT1,         /*!< 7: PT1 EI2C3(SCL) */  
}ei2c_port_t;

/*----------------------*/
/*  EI2C Setting         */
/*----------------------*/
/* #define EI2C_CHANNEL0 */
   #define EI2C_CHANNEL1 
/* #define EI2C_CHANNEL2 */
/* #define EI2C_CHANNEL3 */
/* #define EI2C_CHANNEL4 */
/* #define EI2C_CHANNEL2P */
/* #define EI2C_CHANNEL3P */
/* #define EI2C_CHANNEL4P */

#if defined(EI2C_CHANNEL0)
 #define EI2C_CFG_PORT_SCL     (EI2C_PORT_PC0)        /*!<  SCL Port.                  */
 #define EI2C_CFG_PORT_SDA     (EI2C_PORT_PC1)        /*!<  SDA Port.                  */
#elif defined(EI2C_CHANNEL1)
 #define EI2C_CFG_PORT_SCL     (EI2C_PORT_PA4)        /*!<  SCL Port.                  */
 #define EI2C_CFG_PORT_SDA     (EI2C_PORT_PA5)        /*!<  SDA Port.                  */
#elif defined(EI2C_CHANNEL2)
 #define EI2C_CFG_PORT_SCL     (EI2C_PORT_PL0)        /*!<  SCL Port.                  */
 #define EI2C_CFG_PORT_SDA     (EI2C_PORT_PL1)        /*!<  SDA Port.                  */
#elif defined(EI2C_CHANNEL3)
 #define EI2C_CFG_PORT_SCL     (EI2C_PORT_PT1)        /*!<  SCL Port.                  */
 #define EI2C_CFG_PORT_SDA     (EI2C_PORT_PT0)        /*!<  SDA Port.                  */
#else
 #error "target channel is non-select."
#endif

/** 
 *  @}
 */ /* End of group BSP_EI2C_Exported_define */

/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_EI2C_Exported_typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief  EI2C internal information structure definenition.
*/
/*----------------------------------*/
typedef struct
{
    uint8_t         bus_free;       /*!< Bus free information.          */
    uint8_t         start;          /*!< Start condition information.   */
    uint32_t        irqn;           /*!< IRQ number table pointer.      */
    struct
    {
        uint32_t    address;        /*!< Slave address.                 */
        uint32_t    stop;           /*!< Stop control                   */
        uint32_t    event;          /*!< EI2C Event information.        */
        uint32_t    state;          /*!< Transfer State.                */
    } asynch;
} ei2c_internal_info_t;

/*----------------------------------*/
/** 
 * @brief  EI2C buffer structure definenition.
*/
/*----------------------------------*/
typedef struct
{
    uint8_t     *p_buffer;          /*!< Buffer address.      */
    uint32_t    length;             /*!< Buffer length.       */
    uint32_t    pos;                /*!< Buffer pointer.      */
} ei2c_buffer_t;

/*----------------------------------*/
/** 
 * @brief  EI2C handle structure definenition.
*/
/*----------------------------------*/
typedef struct
{
    ei2c_t                 ei2c;     /*!< EI2C class structure.   */
    ei2c_internal_info_t   info;     /*!< Internal Information.  */
    ei2c_buffer_t          tx_buff;  /*!< Tx buffer structure.   */
    ei2c_buffer_t          rx_buff;  /*!< Rx buffer structure.   */
} bsp_ei2c_t;

/**
 *  @}
 */ /* End of group BSP_EI2C_Exported_typedef */


/*------------------------------------------------------------------------------*/
/*  Functions                                                                   */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_EI2C_Exported_functions
 *  @{
 */
TXZ_Result ei2c_init(bsp_ei2c_t *p_obj, ei2c_port_t sda, ei2c_port_t scl);
TXZ_Result ei2c_set_prs_scl_dig(ei2c_t *p_obj);
TXZ_Result ei2c_set_prs_scl_ana(ei2c_t *p_obj);
TXZ_Result ei2c_set_prs_scl(ei2c_t *p_obj);
void ei2c_reset(bsp_ei2c_t *p_obj);
void ei2c_acr1_clear(bsp_ei2c_t *p_obj);
uint8_t ei2c_active(bsp_ei2c_t *p_obj);
TXZ_Result ei2c_transfer_asynch(bsp_ei2c_t *p_obj, uint8_t *p_tx, int32_t tx_length, uint8_t *p_rx, int32_t rx_length, int32_t address, int32_t stop);
uint32_t ei2c_master_st_irq_handler(bsp_ei2c_t *p_obj);
uint32_t ei2c_slave_st_irq_handler(bsp_ei2c_t *p_obj);
uint32_t ei2c_irq_handler_asynch(bsp_ei2c_t *p_obj);
void ei2c_abort_asynch(bsp_ei2c_t *p_obj);

/* For slave */
void ei2c_slave_mode(bsp_ei2c_t *p_obj, int32_t enable_slave);
void ei2c_slave_address1(bsp_ei2c_t *p_obj, uint32_t address,int32_t bit7);
TXZ_Result ei2c_slave_transfer_asynch(bsp_ei2c_t *p_obj, uint8_t *p_tx, int32_t tx_length, uint8_t *p_rx, int32_t rx_length);
uint32_t ei2c_slave_irq_handler_asynch(bsp_ei2c_t *p_obj);
void ei2c_slave_abort_asynch(bsp_ei2c_t *p_obj);

/** 
 *  @}
 */ /* End of group BSP_EI2C_Exported_functions */

/**
 *  @}
 */ /* End of group BSP_EI2C */

/**
 *  @} 
 */ /* End of group BSP */

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* __BSP_EI2C_H */


