/** 
 *******************************************************************************
 * @file    bsp_config_adbun_m3hqa_i2c.h
 * @brief   This file provides all the functions prototypes for I2C Driver.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
/*------------------------------------------------------------------------------*/
/*  Define to prevent recursive inclusion                                       */
/*------------------------------------------------------------------------------*/
#ifndef __BSP_CONFIG_ADBUN_M3HQA_I2C_H
#define __BSP_CONFIG_ADBUN_M3HQA_I2C_H

#ifdef __cplusplus
extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "i2c.h"

/**
 *  @addtogroup BSP BSP
 *  @{
 */

/** 
 *  @addtogroup BSP_I2C
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_I2C_Exported_macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group BSP_I2C_Exported_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_I2C_Exported_define
 *  @{
 */
/** 
 *  @defgroup I2C_NullPointer  Null Pointer
 *  @brief    I2C NULL Pointer.
 *  @{
 */
#define I2C_NULL        ((void *)0)
/**
 *  @}
 */ /* End of group I2C_NullPointer */

/** 
 *  @}
 */ /* End of group BSP_I2C_Exported_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_I2C_Exported_define
 *  @{
 */

/**
 * @defgroup I2C_ACK I2C ACK Macros
 * @brief    I2C Type of Acknowledge.
 * @{
 */
#define I2C_NACK                    (0)         /*!< NACK Received.             */
#define I2C_ACK                     (1)         /*!< ACK Received.              */
/**
 *  @}
 */ /* End of group I2C_ACK */


/**
 * @defgroup I2C_ERROR I2C ERROR Macros
 * @brief    I2C Error definitions.
 * @{
 */
#define I2C_ERROR_NO_ERROR          (0)         /*!< No Error.                          */
#define I2C_ERROR_NO_SLAVE          (-1)        /*!< No Slave Error.                    */
#define I2C_ERROR_BUS_BUSY          (-2)        /*!< Bus Busy Error.(now, not support)  */
#define I2C_ERROR_PARAM             (-3)        /*!< Parameter Error.                   */
#define I2C_ERROR_OTHERS            (-4)        /*!< Others Error.                      */
#define I2C_ERROR_ARBITRATION       (-5)        /*!< Arbitration Error.                 */
/**
 *  @}
 */ /* End of group I2C_ERROR */


/**
 * @defgroup I2C_Events I2C Events Macros
 * @brief    I2C Asynch Events.
 * @{
 */
#define I2C_EVENT_ERROR               (1 << 1)  /*!< Error.                     */
#define I2C_EVENT_ERROR_NO_SLAVE      (1 << 2)  /*!< No Slave.                  */
#define I2C_EVENT_TRANSFER_COMPLETE   (1 << 3)  /*!< Transfer Complete.         */
#define I2C_EVENT_TRANSFER_EARLY_NACK (1 << 4)  /*!< End of Transfer.           */
#define I2C_EVENT_ALL                 (I2C_EVENT_ERROR |  I2C_EVENT_TRANSFER_COMPLETE | I2C_EVENT_ERROR_NO_SLAVE | I2C_EVENT_TRANSFER_EARLY_NACK)
/**
 *  @}
 */ /* End of group I2C_Events */

/**
 * @defgroup I2C_SlaveReceive I2C Slave Receive Return Macros
 * @brief    I2C Received Contents of Slave.
 * @{
 */
#define I2C_NO_DATA             (0)     /*!< the slave has not been addressed. */
#define I2C_READ_ADDRESSED      (1)     /*!< the master has requested a read from this slave. */
#define I2C_WRITE_GENERAL       (2)     /*!< the master is writing to all slave.(now, not support) */
#define I2C_WRITE_ADDRESSED     (3)     /*!< the master is writing to this slave. */
/**
 *  @}
 */ /* End of group I2C_SlaveReceive */

/** 
 *  @}
 */ /* End of group BSP_I2C_Exported_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_I2C_Exported_define
 *  @{
 */

/*----------------------------------*/
/** 
 * @brief  i2c Port Enumerated Type Definition.
*/
/*----------------------------------*/
typedef enum {
  I2C_PORT_PC0 = 0,     /*!< 0: PC0 I2C0 */
  I2C_PORT_PC1,         /*!< 1: PC1 I2C0 */
  I2C_PORT_PA4,         /*!< 2: PA4 I2C1 */
  I2C_PORT_PA5,         /*!< 3: PA5 I2C1 */
  I2C_PORT_PL0,         /*!< 4: PL0 I2C2 */
  I2C_PORT_PL1,         /*!< 5: PL1 I2C2 */
  I2C_PORT_PT0,         /*!< 6: PT0 I2C3 */
  I2C_PORT_PT1,         /*!< 7: PT1 I2C3 */
} i2c_port_t;

/*----------------------*/
/*  I2C Setting         */
/*----------------------*/
/* #define I2C_CHANNEL0 */
#define I2C_CHANNEL1
/* #define I2C_CHANNEL2 */
/* #define I2C_CHANNEL3 */
#if defined(I2C_CHANNEL0)
 #define I2C_CFG_PORT_SCL     (I2C_PORT_PC0)        /*!<  SCL Port.                  */
 #define I2C_CFG_PORT_SDA     (I2C_PORT_PC1)        /*!<  SDA Port.                  */
#elif defined(I2C_CHANNEL1)
 #define I2C_CFG_PORT_SCL     (I2C_PORT_PA4)        /*!<  SCL Port.                  */
 #define I2C_CFG_PORT_SDA     (I2C_PORT_PA5)        /*!<  SDA Port.                  */
#elif defined(I2C_CHANNEL2)
 #define I2C_CFG_PORT_SCL     (I2C_PORT_PL0)        /*!<  SCL Port.                  */
 #define I2C_CFG_PORT_SDA     (I2C_PORT_PL1)        /*!<  SDA Port.                  */
#elif defined(I2C_CHANNEL3)
 #define I2C_CFG_PORT_SCL     (I2C_PORT_PT1)        /*!<  SCL Port.                  */
 #define I2C_CFG_PORT_SDA     (I2C_PORT_PT0)        /*!<  SDA Port.                  */
#else
 #error "target channel is non-select."
#endif

/** 
 *  @}
 */ /* End of group BSP_I2C_Exported_define */

/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_I2C_Exported_typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief  I2C internal information structure definenition.
*/
/*----------------------------------*/
typedef struct
{
    uint8_t         bus_free;       /*!< Bus free information.          */
    uint8_t         start;          /*!< Start condition information.   */
    uint32_t        irqn;           /*!< IRQ number table pointer.      */
    struct
    {
        uint32_t    address;        /*!< Slave address.                 */
        uint32_t    stop;           /*!< Stop control                   */
        uint32_t    event;          /*!< I2C Event information.         */
        uint32_t    state;          /*!< Transfer State.                */
    } asynch;
} i2c_internal_info_t;

/*----------------------------------*/
/** 
 * @brief  I2C buffer structure definenition.
*/
/*----------------------------------*/
typedef struct
{
    uint8_t     *p_buffer;          /*!< Buffer address.      */
    uint32_t    length;             /*!< Buffer length.       */
    uint32_t    pos;                /*!< Buffer pointer.      */
} i2c_buffer_t;

/*----------------------------------*/
/** 
 * @brief  I2C handle structure definenition.
*/
/*----------------------------------*/
typedef struct
{
    I2C_t                 i2c;      /*!< I2C class structure.   */
    i2c_internal_info_t   info;     /*!< Internal Information.  */
    i2c_buffer_t          tx_buff;  /*!< Tx buffer structure.   */
    i2c_buffer_t          rx_buff;  /*!< Rx buffer structure.   */
} i2c_t;

/**
 *  @}
 */ /* End of group BSP_I2C_Exported_typedef */


/*------------------------------------------------------------------------------*/
/*  Functions                                                                   */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_I2C_Exported_functions
 *  @{
 */
TXZ_Result i2c_init(i2c_t *p_obj, i2c_port_t sda, i2c_port_t scl);
TXZ_Result i2c_frequency(i2c_t *p_obj, int32_t hz);
void i2c_reset(i2c_t *p_obj);
TXZ_Result i2c_check_bus_free(i2c_t *p_obj);
TXZ_Result i2c_start(i2c_t *p_obj);
TXZ_Result i2c_stop(i2c_t *p_obj);
int32_t i2c_read(i2c_t *p_obj, int32_t address, uint8_t *p_data, int32_t length, int32_t stop);
int32_t i2c_write(i2c_t *p_obj, int32_t address, uint8_t *p_data, int32_t length, int32_t stop);
int32_t i2c_byte_read(i2c_t *p_obj, int32_t last);
int32_t i2c_byte_write(i2c_t *p_obj, int32_t data);
uint8_t i2c_active(i2c_t *p_obj);
TXZ_Result i2c_transfer_asynch(i2c_t *p_obj, uint8_t *p_tx, int32_t tx_length, uint8_t *p_rx, int32_t rx_length, int32_t address, int32_t stop);
uint32_t i2c_irq_handler_asynch(i2c_t *p_obj);
void i2c_abort_asynch(i2c_t *p_obj);

/* For slave */
void i2c_slave_mode(i2c_t *p_obj, int32_t enable_slave);
int32_t i2c_slave_receive(i2c_t *p_obj);
int32_t i2c_slave_read(i2c_t *p_obj, uint8_t *p_data, int32_t length);
int32_t i2c_slave_write(i2c_t *p_obj, uint8_t *p_data, int32_t length);
void i2c_slave_address(i2c_t *p_obj, uint32_t address);
TXZ_Result i2c_slave_transfer_asynch(i2c_t *p_obj, uint8_t *p_tx, int32_t tx_length, uint8_t *p_rx, int32_t rx_length);
uint32_t i2c_slave_irq_handler_asynch(i2c_t *p_obj);
void i2c_slave_abort_asynch(i2c_t *p_obj);

/** 
 *  @}
 */ /* End of group BSP_I2C_Exported_functions */

/**
 *  @}
 */ /* End of group BSP_I2C */

/**
 *  @} 
 */ /* End of group BSP */

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* __BSP_CONFIG_ADBUN_M3HQA_I2C_H */


