/**
 *******************************************************************************
 * @file    bsp_config_adbun_m3hqa_rtc.c
 * @brief   This file provides API functions for RTC Class.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */

#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "bsp_rtc.h"

#if defined(__BSP_CONFIG_ADBUN_M3HQA_RTC_H)
/**
 *  @addtogroup BSP BSP
 *  @{
 */

/** 
 *  @addtogroup BSP_BSP_RTC
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_RTC_Private_macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group BSP_BSP_RTC_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_RTC_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group BSP_BSP_RTC_Private_define */

/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_RTC_Private_define
 *  @{
 */

/** 
 *  @name  RTC_NULL Pointer
 *  @brief NULL Pointer.
 *  @{
 */
#define RTC_NULL                  ((void *)0)
/**
 *  @}
 */ /* End of name RTC_NULL Pointer */

/** 
 *  @name  Parameter Result
 *  @brief Whether the parameter is specified or not.
 *  @{
 */
#define RTC_PARAM_OK              ((int32_t)1)              /*!< Parameter is valid(specified).         */
#define RTC_PARAM_NG              ((int32_t)0)              /*!< Parameter is invalid(not specified).   */
/**
 *  @}
 */ /* End of name Parameter Result */

/** 
 *  @name  RTCHOURR Macro Definition.
 *  @brief RTCHOURR Register Macro Definition.
 *  @details  Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 7:6   | -          |
 *  | 5:0   | HO[5:0]    |
 *  @{
 */
/* HO[5:0] */
#define RTCPAGER_HO_12_HOUR_MASK      ((uint8_t)0x1F)           /*!< HO :When 12-hour notaion, hour value mask.    */
#define RTCPAGER_HO_12_MERIDIEM_MASK  ((uint8_t)0x20)           /*!< HO :When 12-hour notaion, meridiem value mask.*/
#define RTCPAGER_HO_12_MERIDIEM_AM    ((uint8_t)0x00)           /*!< HO :When 12-hour notaion, A.M.                */
#define RTCPAGER_HO_12_MERIDIEM_PM    ((uint8_t)0x20)           /*!< HO :When 12-hour notaion, P.M.                */
/**
 *  @}
 */ /* End of name RTCHOURR Macro Definition */

/** 
 *  @name  RTCPAGER Macro Definition.
 *  @brief RTCPAGER Register Macro Definition.
 *  @details  Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 7     | INTENA     |
 *  | 6:5   | -          |
 *  | 4     | ADJUST     |
 *  | 3     | ENATMR     |
 *  | 2     | ENAALM     |
 *  | 1     | -          |
 *  | 0     | PAGE       |
 *  @{
 */
/* INTENA */
#define RTCPAGER_INTENA_MASK      ((uint8_t)0x80)           /*!< INTENA :Mask.                                 */
#define RTCPAGER_INTENA_DISABLE   ((uint8_t)0x00)           /*!< INTENA :Disable.                              */
#define RTCPAGER_INTENA_ENABLE    ((uint8_t)0x80)           /*!< INTENA :Enable.                               */
/* ADJUST */
#define RTCPAGER_ADJUST_W_REQ     ((uint8_t)0x10)           /*!< ADJUST :[write] Request.                      */
#define RTCPAGER_ADJUST_R_IDLE    ((uint8_t)0x00)           /*!< ADJUST :[read]  Idle(No request).             */
#define RTCPAGER_ADJUST_R_RUN     ((uint8_t)0x10)           /*!< ADJUST :[read]  Running(Processing Request).  */
/* ENATMR */
#define RTCPAGER_ENATMR_MASK      ((uint8_t)0x08)           /*!< ENATMR :Mask.                                 */
#define RTCPAGER_ENATMR_DISABLE   ((uint8_t)0x00)           /*!< ENATMR :Disable.                              */
#define RTCPAGER_ENATMR_ENABLE    ((uint8_t)0x08)           /*!< ENATMR :Enable.                               */
/* ENAALM */
#define RTCPAGER_ENAALM_MASK      ((uint8_t)0x04)           /*!< ENAALM :Mask.                                 */
#define RTCPAGER_ENAALM_DISABLE   ((uint8_t)0x00)           /*!< ENAALM :Disable.                              */
#define RTCPAGER_ENAALM_ENABLE    ((uint8_t)0x04)           /*!< ENAALM :Enable.                               */
/* PAGE   */
#define RTCPAGER_PAGE_MASK        ((uint8_t)0x01)           /*!< PAGE   :Mask.                                 */
#define RTCPAGER_PAGE_0           ((uint8_t)0x00)           /*!< PAGE   :Select Page0.                         */
#define RTCPAGER_PAGE_1           ((uint8_t)0x01)           /*!< PAGE   :Select Page1.                         */
/**
 *  @}
 */ /* End of name RTCPAGER Macro Definition */

/** 
 *  @name  RTCRESTR Macro Definition.
 *  @brief RTCRESTR Register Macro Definition.
 *  @details  Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 7     | DIS1HZ     |
 *  | 6     | DIS16HZ    |
 *  | 5     | RSTTMR     |
 *  | 4     | RSTALM     |
 *  | 3     | -          |
 *  | 2     | DIS2HZ     |
 *  | 1     | DIS4HZ     |
 *  | 0     | DIS8HZ     |
 *  @{
 */
/* RSTTMR */
#define RTCRESTR_RSTTMR_MASK      ((uint8_t)0x20)           /*!< RSTTMR :Mask.                                 */
#define RTCRESTR_RSTTMR_W_RESET   ((uint8_t)0x20)           /*!< RSTTMR :[read]  Reset request.                */
#define RTCRESTR_RSTTMR_R_IDLE    ((uint8_t)0x00)           /*!< RSTTMR :[read]  Idle(No request).             */
#define RTCRESTR_RSTTMR_R_RUN     ((uint8_t)0x20)           /*!< RSTTMR :[read]  Running(Processing Request).  */
/* RSTALM */
#define RTCRESTR_RSTALM_MASK      ((uint8_t)0x10)           /*!< RSTALM :Mask.                                 */
#define RTCRESTR_RSTALM_RESET     ((uint8_t)0x10)           /*!< RSTALM :Reset request.                        */
/**
 *  @}
 */ /* End of name RTCRESTR Macro Definition */

/** 
 *  @}
 */ /* End of group BSP_BSP_RTC_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_RTC_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group BSP_BSP_RTC_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_RTC_Private_typedef
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group BSP_BSP_RTC_Private_typedef */

/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_RTC_Private_variables
 *  @{
 */

/* no define */

/**
 *  @}
 */ /* End of group BSP_BSP_RTC_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_RTC_Private_fuctions
 *  @{
 */
__STATIC_INLINE uint8_t convert_time_to_reg(uint8_t src);
__STATIC_INLINE uint8_t convert_time_to_dec(uint8_t src);
__STATIC_INLINE void set_page(rtc_t *p_obj, uint8_t page);
#ifdef DEBUG
    __STATIC_INLINE int32_t check_param_int_source_signal(uint8_t param);
    __STATIC_INLINE int32_t check_param_notation(uint8_t param);
    __STATIC_INLINE int32_t check_param_sec(uint8_t param);
    __STATIC_INLINE int32_t check_param_min(uint8_t param);
    __STATIC_INLINE int32_t check_param_hour_24(uint8_t param);
    __STATIC_INLINE int32_t check_param_hour_12(uint8_t param);
    __STATIC_INLINE int32_t check_param_meridiem(uint8_t param);
    __STATIC_INLINE int32_t check_param_day(uint8_t param);
    __STATIC_INLINE int32_t check_param_date(uint8_t param);
    __STATIC_INLINE int32_t check_param_month(uint8_t param);
    __STATIC_INLINE int32_t check_param_year(uint8_t param);
    __STATIC_INLINE int32_t check_param_leap(uint8_t param);
    __STATIC_INLINE int32_t check_param_min_alarm(uint8_t param);
    __STATIC_INLINE int32_t check_param_hour_alarm(uint8_t param);
    __STATIC_INLINE int32_t check_param_day_alarm(uint8_t param);
    __STATIC_INLINE int32_t check_param_date_alarm(uint8_t param);
#endif /* #ifdef DEBUG */
#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Check the Interrupt Source Signal's parameter.
  * @param[in] param     :Interrupt Source Signal's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RTC_IntSource"RTC_INT_SRC_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_int_source_signal(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    switch (param) 
    {
    case RTC_INT_SRC_1HZ:
    case RTC_INT_SRC_2HZ:
    case RTC_INT_SRC_4HZ:
    case RTC_INT_SRC_8HZ:
    case RTC_INT_SRC_16HZ:
    case RTC_INT_SRC_ALARM:
        result = RTC_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Hour notation's parameter.
  * @param[in] param     :Hour notation's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref BSP_RTC_HourNotation"BSP_RTC_HOUR_NOTATION_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_notation(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    switch (param) 
    {
    case RTC_HOUR_NOTATION_12:
    case RTC_HOUR_NOTATION_24:
        result = RTC_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Second's parameter.
  * @param[in] param     :Second's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RTC_RangeSec"RTC_RANGE_SEC_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_sec(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    /*--- Now, RTC_RANGE_SEC_MIN is 0. ---*/
#if 0
    if ((RTC_RANGE_SEC_MIN <= param) && (param <= RTC_RANGE_SEC_MAX))
#else
    if (param <= RTC_RANGE_SEC_MAX)
#endif
    {
        result = RTC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Minute's parameter.
  * @param[in] param     :Minute's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RTC_RangeMin"RTC_RANGE_MIN_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_min(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    /*--- Now, RTC_RANGE_MIN_MIN is 0. ---*/
#if 0
    if ((RTC_RANGE_MIN_MIN <= param) && (param <= RTC_RANGE_MIN_MAX))
#else
    if (param <= RTC_RANGE_MIN_MAX)
#endif
    {
        result = RTC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Hour's parameter.
  * @param[in] param     :Hour's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RTC_Range24Hour"RTC_RANGE_24_HOUR_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_hour_24(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    /*--- Now, RTC_RANGE_24_HOUR_MIN is 0. ---*/
#if 0
    if ((RTC_RANGE_24_HOUR_MIN <= param) && (param <= RTC_RANGE_24_HOUR_MAX))
#else
    if (param <= RTC_RANGE_24_HOUR_MAX)
#endif
    {
        result = RTC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Hour's parameter.
  * @param[in] param     :Hour's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RTC_Range12Hour"RTC_RANGE_12_HOUR_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_hour_12(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    /*--- Now, RTC_RANGE_12_HOUR_MIN is 0. ---*/
#if 0
    if ((RTC_RANGE_12_HOUR_MIN <= param) && (param <= RTC_RANGE_12_HOUR_MAX))
#else
    if (param <= RTC_RANGE_12_HOUR_MAX)
#endif
    {
        result = RTC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Meridiem's parameter.
  * @param[in] param     :Meridiem's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RTC_Meridiem"RTC_MERIDIEM_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_meridiem(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    switch (param) 
    {
    case RTC_MERIDIEM_AM:
    case RTC_MERIDIEM_PM:
        result = RTC_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Day's parameter.
  * @param[in] param     :Day's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RTC_Day"RTC_DAY_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_day(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    switch (param) 
    {
    case RTC_DAY_SUNDAY:
    case RTC_DAY_MONDAY:
    case RTC_DAY_TUESDAY:
    case RTC_DAY_WEDNESDAY:
    case RTC_DAY_THURSDAY:
    case RTC_DAY_FRIDAY:
    case RTC_DAY_SATURDAY:
        result = RTC_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Date's parameter.
  * @param[in] param     :Date's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RTC_RangeDate"RTC_RANGE_DATE_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_date(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    /*--- Now, RTC_RANGE_DATE_MIN is 0. ---*/
#if 0
    if ((RTC_RANGE_DATE_MIN <= param) && (param <= RTC_RANGE_DATE_MAX))
#else
    if (param <= RTC_RANGE_DATE_MAX)
#endif
    {
        result = RTC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Month's parameter.
  * @param[in] param     :Month's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RTC_Month"RTC_MONTH_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_month(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    switch (param) 
    {
    case RTC_MONTH_JAN:
    case RTC_MONTH_FEB:
    case RTC_MONTH_MAR:
    case RTC_MONTH_APR:
    case RTC_MONTH_MAY:
    case RTC_MONTH_JUN:
    case RTC_MONTH_JUL:
    case RTC_MONTH_AUG:
    case RTC_MONTH_SEP:
    case RTC_MONTH_OCT:
    case RTC_MONTH_NOV:
    case RTC_MONTH_DEC:
        result = RTC_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Year's parameter.
  * @param[in] param     :Year's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RTC_RangeYear"RTC_RANGE_YEAR_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_year(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    /*--- Now, RTC_RANGE_YEAR_MIN is 0. ---*/
#if 0
    if ((RTC_RANGE_YEAR_MIN <= param) && (param <= RTC_RANGE_YEAR_MAX))
#else
    if (param <= RTC_RANGE_YEAR_MAX)
#endif
    {
        result = RTC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Leap's parameter.
  * @param[in] param     :Leap's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RTC_LeapYear"RTC_LEAP_YEAR_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_leap(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    switch (param) 
    {
    case RTC_LEAP_YEAR_0:
    case RTC_LEAP_YEAR_1:
    case RTC_LEAP_YEAR_2:
    case RTC_LEAP_YEAR_3:
        result = RTC_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Alarm minute's parameter.
  * @param[in] param     :Alarm minute's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RTC_AlarmMin"RTC_ALARM_MIN_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_min_alarm(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    switch (param) 
    {
    case RTC_ALARM_MIN_OFF:
    case RTC_ALARM_MIN_ON:
        result = RTC_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Alarm hour's parameter.
  * @param[in] param     :Alarm hour's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RTC_AlarmHour"RTC_ALARM_HOUR_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_hour_alarm(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    switch (param) 
    {
    case RTC_ALARM_HOUR_OFF:
    case RTC_ALARM_HOUR_ON:
        result = RTC_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Alarm day's parameter.
  * @param[in] param     :Alarm day's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RTC_AlarmDay"RTC_ALARM_DAY_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_day_alarm(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    switch (param) 
    {
    case RTC_ALARM_DAY_OFF:
    case RTC_ALARM_DAY_ON:
        result = RTC_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Alarm date's parameter.
  * @param[in] param     :Alarm date's parameter
  * @retval RTC_PARAM_OK :Valid
  * @retval RTC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RTC_AlarmDate"RTC_ALARM_DATE_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_date_alarm(uint8_t param)
{
    int32_t result = RTC_PARAM_NG;

    switch (param) 
    {
    case RTC_ALARM_DATE_OFF:
    case RTC_ALARM_DATE_ON:
        result = RTC_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}
#endif /* #ifdef DEBUG */

/*--------------------------------------------------*/
/** 
  * @brief  Convet time value to register setting value.
  * @param[in] param     :Decimal.
  * @retval Register setting value.
  * @note   -
  */
/*--------------------------------------------------*/
__STATIC_INLINE uint8_t convert_time_to_reg(uint8_t src)
{
    uint8_t dst = 0;

    dst   = (uint8_t)(src / 10);
    dst <<= 4;
    dst  |= (uint8_t)(src % 10);

    return (dst);
}

/*--------------------------------------------------*/
/** 
  * @brief  Convet time value to decimal.
  * @param[in] param     :Register setting value.
  * @retval Decimal.
  * @note   -
  */
/*--------------------------------------------------*/
__STATIC_INLINE uint8_t convert_time_to_dec(uint8_t src)
{
    uint8_t dst = 0;

    dst   = (uint8_t)(src & 0xF0);
    dst >>= 4;
    dst  *= 10;
    dst  += (uint8_t)(src & 0x0F);

    return (dst);
}

/*--------------------------------------------------*/
/** 
  * @brief  Set PAGE.
  * @param[in] p_obj :Object address.
  * @param[in] page  :Page.
  * @retval -
  * @note   Access to RTCPAGER.
  */
/*--------------------------------------------------*/
__STATIC_INLINE void set_page(rtc_t *p_obj, uint8_t page)
{
    /*--- RTCPAGER ---*/
    /* PAGE */
    uint8_t work = p_obj->p_instance->PAGER;
    work &= (uint8_t)(~RTCPAGER_PAGE_MASK);
    work |= page;
    p_obj->p_instance->PAGER = work;
}
/**
 *  @}
 */ /* End of group BSP_BSP_RTC_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_RTC_Exported_functions
 *  @{
 */
/*--------------------------------------------------*/
/**
  * @brief      Construct object.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       -
  */
/*--------------------------------------------------*/
TXZ_Result rtc_init(rtc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Disable INTRTC              */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* INTENA */
    if (rtc_disable_int(p_obj) != TXZ_SUCCESS)
    {
        res = TXZ_ERROR;
    }
    /*------------------------------*/
    /*  Disable Clock/Alarm         */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* ENATMR */
    if (rtc_clock_disable(p_obj) != TXZ_SUCCESS)
    {
        res = TXZ_ERROR;
    }
    /* ENAALM */
    if (rtc_alarm_disable(p_obj) != TXZ_SUCCESS)
    {
        res = TXZ_ERROR;
    }
    /*------------------------------*/
    /*  Reset Clock Sec Counter     */
    /*------------------------------*/
    /*--- RTCRESTR ---*/
    /* RSTTMR */
    if (rtc_clock_reset_counter(p_obj) != TXZ_SUCCESS)
    {
        res = TXZ_ERROR;
    }
    /*------------------------------*/
    /*  Reset Alarm                 */
    /*------------------------------*/
    /*--- RTCRESTR ---*/
    /* RSTALM */
    if (rtc_alarm_reset(p_obj) != TXZ_SUCCESS)
    {
        res = TXZ_ERROR;
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Destruct object.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       -
  */
/*--------------------------------------------------*/
TXZ_Result rtc_deinit(rtc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Disable INTRTC              */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* INTENA */
    if (rtc_disable_int(p_obj) != TXZ_SUCCESS)
    {
        res = TXZ_ERROR;
    }
    /*------------------------------*/
    /*  Disable Clock/Alarm         */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* ENATMR */
    if (rtc_clock_disable(p_obj) != TXZ_SUCCESS)
    {
        res = TXZ_ERROR;
    }
    /* ENAALM */
    if (rtc_alarm_disable(p_obj) != TXZ_SUCCESS)
    {
        res = TXZ_ERROR;
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Enable INTRTC.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_enable_int(rtc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Enable INTRTC               */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* INTENA */
    {
        uint8_t work = p_obj->p_instance->PAGER;
        work &= (uint8_t)(~RTCPAGER_INTENA_MASK);
        work |= RTCPAGER_INTENA_ENABLE;
        p_obj->p_instance->PAGER = work;
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Disable INTRTC.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_disable_int(rtc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Disable INTRTC              */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* INTENA */
    {
        uint8_t work = p_obj->p_instance->PAGER;
        work &= (uint8_t)(~RTCPAGER_INTENA_MASK);
        work |= RTCPAGER_INTENA_DISABLE;
        p_obj->p_instance->PAGER = work;
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set interrupt source signal.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  src     :Interrupt source signal. Use @ref RTC_IntSource
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCRESTR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_set_int_source(rtc_t *p_obj, uint8_t src)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_int_source_signal(src));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set DISxHZ                  */
    /*------------------------------*/
    /*--- RTCRESTR ---*/
    /* DISxHZ */
    {
        uint8_t work = p_obj->p_instance->RESTR;
        work &= (RTCRESTR_RSTTMR_MASK | RTCRESTR_RSTALM_MASK);
        work |= src;
        p_obj->p_instance->RESTR = work;
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set hour notation.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  notation:Hour notation. Use @ref BSP_RTC_HourNotation.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCMONTHR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_set_hour_notation(rtc_t *p_obj, uint8_t notation)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_notation(notation));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_1);
    /*------------------------------*/
    /*  Set Hour notation           */
    /*------------------------------*/
    /*--- RTCMONTHR ---*/
    /* MO0 */
    p_obj->p_instance->MONTHR = notation;

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get hour notation.
  * @details    -
  * @param[in]  p_obj     :Object address.
  * @param[out] p_notation:Hour notation @ref BSP_RTC_HourNotation.
  * @retval     TXZ_SUCCESS   :Success
  * @retval     TXZ_ERROR     :Failure
  * @note       Access to RTCPAGER, RTCMONTHR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_get_hour_notation(rtc_t *p_obj, uint8_t *p_notation)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_notation));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_1);
    /*------------------------------*/
    /*  Get Hour notation           */
    /*------------------------------*/
    /*--- RTCMONTHR ---*/
    /* MO0 */
    *p_notation = p_obj->p_instance->MONTHR;

    return (res);
}

/*--------------------------------------------------*/
/*  Clock                                           */
/*--------------------------------------------------*/
/*--------------------------------------------------*/
/**
  * @brief      Enable clock(timer).
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_enable(rtc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Enable ENATMR               */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* ENATMR */
    {
        uint8_t work = p_obj->p_instance->PAGER;
        work &= (uint8_t)(~RTCPAGER_ENATMR_MASK);
        work |= RTCPAGER_ENATMR_ENABLE;
        p_obj->p_instance->PAGER = work;
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Disable clock(timer).
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_disable(rtc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Disable ENATMR              */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* ENATMR */
    {
        uint8_t work = p_obj->p_instance->PAGER;
        work &= (uint8_t)(~RTCPAGER_ENATMR_MASK);
        work |= RTCPAGER_ENATMR_DISABLE;
        p_obj->p_instance->PAGER = work;
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Reset the second counter.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCRESTR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_reset_counter(rtc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Reset RSTTMR                */
    /*------------------------------*/
    /*--- RTCRESTR ---*/
    /* Wait to "RSTTMR = RTCRESTR_RSTTMR_R_IDLE". */
    while (((p_obj->p_instance->RESTR) & RTCRESTR_RSTTMR_MASK) == RTCRESTR_RSTTMR_R_RUN) 
    {
        /* no process */
    }
    /* RSTTMR */
    {
        uint8_t work = p_obj->p_instance->RESTR;
        work &= (uint8_t)(~RTCRESTR_RSTTMR_MASK);
        work |= RTCRESTR_RSTTMR_W_RESET;
        p_obj->p_instance->RESTR = work;
    }
    /* Wait to "RSTTMR = RTCRESTR_RSTTMR_R_IDLE". */
    while (((p_obj->p_instance->RESTR) & RTCRESTR_RSTTMR_MASK) == RTCRESTR_RSTTMR_R_RUN) 
    {
        /* no process */
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set clock second.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  sec     :Second. Range is @ref RTC_RangeSec.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCSECR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_set_sec(rtc_t *p_obj, uint8_t sec)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_sec(sec));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Set Second                  */
    /*------------------------------*/
    /*--- RTCSECR ---*/
    /* SE[6:0] */
    p_obj->p_instance->SECR = convert_time_to_reg(sec);

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get clock second.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[out] *p_sec  :Second. Destination address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCSECR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_get_sec(rtc_t *p_obj, uint8_t *p_sec)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_sec));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Get Second                  */
    /*------------------------------*/
    /*--- RTCSECR ---*/
    /* SE[6:0] */
    {
        uint8_t work = p_obj->p_instance->SECR;
        *p_sec = convert_time_to_dec(work);
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set clock minute.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  min     :Minute. Range is @ref RTC_RangeMin.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCMINR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_set_min(rtc_t *p_obj, uint8_t min)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_min(min));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Set Minute                  */
    /*------------------------------*/
    /*--- RTCMINR ---*/
    /* MI[6:0] */
    p_obj->p_instance->MINR = convert_time_to_reg(min);

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get clock minute.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[out] p_min   :Minute. Destination address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCMINR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_get_min(rtc_t *p_obj, uint8_t *p_min)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_min));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Get Minute                  */
    /*------------------------------*/
    /*--- RTCMINR ---*/
    /* MI[6:0] */
    {
        __IO uint8_t work = p_obj->p_instance->MINR;
        *p_min = convert_time_to_dec(work);
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set clock hour.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  hour    :Hour. Range is @ref RTC_Range24Hour.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Only use 24-hour notation.
  * @note       Access to RTCPAGER, RTCHOURR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_set_hour_24(rtc_t *p_obj, uint8_t hour)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_hour_24(hour));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Set Hour                    */
    /*------------------------------*/
    /*--- RTCHOURR ---*/
    /* HO[5:0] */
    p_obj->p_instance->HOURR = convert_time_to_reg(hour);

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get clock hour.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[out] p_hour  :Hour. Destination address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Only use 24-hour notation.
  * @note       Access to RTCPAGER, RTCHOURR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_get_hour_24(rtc_t *p_obj, uint8_t *p_hour)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_hour));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Get Hour                    */
    /*------------------------------*/
    /*--- RTCHOURR ---*/
    /* HO[5:0] */
    {
        __IO uint8_t work = p_obj->p_instance->HOURR;
        *p_hour = convert_time_to_dec(work);
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set clock hour.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  meridiem:Meridiem. Use @ref RTC_Meridiem.
  * @param[in]  hour    :Hour. Range is @ref RTC_Range12Hour.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Only use 12-hour notation.
  * @note       Access to RTCPAGER, RTCHOURR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_set_hour_12(rtc_t *p_obj, uint8_t meridiem, uint8_t hour)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_meridiem(meridiem));
    assert_param(check_param_hour_12(hour));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Set Hour                    */
    /*------------------------------*/
    /*--- RTCHOURR ---*/
    /* HO[5:0] */
    {
        __IO uint8_t work = convert_time_to_reg(hour);
        work |= meridiem;
        p_obj->p_instance->HOURR = work;
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get clock hour.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[out] meridiem:Meridiem @ref RTC_Meridiem. Destination address.
  * @param[out] p_hour  :Hour. Destination address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Only use 12-hour notation.
  * @note       Access to RTCPAGER, RTCHOURR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_get_hour_12(rtc_t *p_obj, uint8_t *p_meridiem, uint8_t *p_hour)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_meridiem));
    assert_param(IS_POINTER_NOT_NULL(p_hour));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Get Hour                    */
    /*------------------------------*/
    /*--- RTCHOURR ---*/
    /* HO[5:0] */
    {
        __IO uint8_t work = p_obj->p_instance->HOURR;
        /* Meridiem */
        if ((work & RTCPAGER_HO_12_MERIDIEM_MASK) == RTCPAGER_HO_12_MERIDIEM_AM)
        {
            *p_meridiem = RTC_MERIDIEM_AM;
        }
        else
        {
            *p_meridiem = RTC_MERIDIEM_PM;
        }
        /* hour */
        *p_hour = convert_time_to_dec((work & RTCPAGER_HO_12_HOUR_MASK));
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set clock day.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  day     :Day. Use @ref RTC_Day.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCDAYR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_set_day(rtc_t *p_obj, uint8_t day)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_day(day));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Set Day                     */
    /*------------------------------*/
    /*--- RTCDAYR ---*/
    /* WE[2:0] */
    p_obj->p_instance->DAYR = day;

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get clock day.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[out] p_day   :Day @ref RTC_Day. Destination address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCDAYR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_get_day(rtc_t *p_obj, uint8_t *p_day)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_day));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Get Day                     */
    /*------------------------------*/
    /*--- RTCDAYR ---*/
    /* WE[2:0] */
    *p_day = p_obj->p_instance->DAYR;

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set clock date.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  date    :Date. Range is @ref RTC_RangeDate.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCDATER.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_set_date(rtc_t *p_obj, uint8_t date)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_date(date));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Set Date                    */
    /*------------------------------*/
    /*--- RTCDATER ---*/
    /* DA[5:0] */
    p_obj->p_instance->DATER = convert_time_to_reg(date);

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get clock date.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[out] *p_date :Date. Destination address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCDATER.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_get_date(rtc_t *p_obj, uint8_t *p_date)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_date));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Get Date                    */
    /*------------------------------*/
    /*--- RTCDATER ---*/
    /* DA[5:0] */
    {
        __IO uint8_t work = p_obj->p_instance->DATER;
        *p_date = convert_time_to_dec(work);
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set clock month.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  month   :Month. Use @ref RTC_Month.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCMONTHR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_set_month(rtc_t *p_obj, uint8_t month)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_month(month));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Set Month                   */
    /*------------------------------*/
    /*--- RTCMONTHR ---*/
    /* MO[4:0] */
    p_obj->p_instance->MONTHR = month;

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get clock month.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[out] *p_month:Month @ref RTC_Month. Destination address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCMONTHR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_get_month(rtc_t *p_obj, uint8_t *p_month)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_month));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Get Month                   */
    /*------------------------------*/
    /*--- RTCMONTHR ---*/
    /* MO[4:0] */
    *p_month = p_obj->p_instance->MONTHR;

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set clock year.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  year    :Year. Range is @ref RTC_RangeYear.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCYEARR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_set_year(rtc_t *p_obj, uint8_t year)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_year(year));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Set Year                    */
    /*------------------------------*/
    /*--- RTCYEARR ---*/
    /* YE[7:0] */
    p_obj->p_instance->YEARR = convert_time_to_reg(year);

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get clock year.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[out] p_year  :Year. Destination address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCYEARR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_get_year(rtc_t *p_obj, uint8_t *p_year)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_year));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_0);
    /*------------------------------*/
    /*  Get Year                    */
    /*------------------------------*/
    /*--- RTCYEARR ---*/
    /* YE[7:0] */
    {
        __IO uint8_t work = p_obj->p_instance->YEARR;
        *p_year = convert_time_to_dec(work);
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set clock leap.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  leap    :Leap. Use @ref RTC_LeapYear.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCYEARR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_set_leap(rtc_t *p_obj, uint8_t leap)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_leap(leap));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_1);
    /*------------------------------*/
    /*  Set Leap                    */
    /*------------------------------*/
    /*--- RTCMONTHR ---*/
    /* LEAP[1:0] */
    p_obj->p_instance->YEARR = leap;

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get clock leap.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[out] *p_leap :Leap @ref RTC_LeapYear. Destination address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCYEARR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_clock_get_leap(rtc_t *p_obj, uint8_t *p_leap)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_leap));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_1);
    /*------------------------------*/
    /*  Get Leap                    */
    /*------------------------------*/
    /*--- RTCMONTHR ---*/
    /* LEAP[1:0] */
    *p_leap = p_obj->p_instance->YEARR;

    return (res);
}

/*--------------------------------------------------*/
/*  Alarm                                           */
/*--------------------------------------------------*/
/*--------------------------------------------------*/
/**
  * @brief      Enable alarm.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_alarm_enable(rtc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Enable ENATMR               */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* ENAALM */
    {
        uint8_t work = p_obj->p_instance->PAGER;
        work &= (uint8_t)(~RTCPAGER_ENAALM_MASK);
        work |= RTCPAGER_ENAALM_ENABLE;
        p_obj->p_instance->PAGER = work;
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Disable alarm.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_alarm_disable(rtc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Disable ENATMR              */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* ENAALM */
    {
        uint8_t work = p_obj->p_instance->PAGER;
        work &= (uint8_t)(~RTCPAGER_ENAALM_MASK);
        work |= RTCPAGER_ENAALM_DISABLE;
        p_obj->p_instance->PAGER = work;
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Reset alarm.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCRESTR.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_alarm_reset(rtc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Reset RSTTMR                */
    /*------------------------------*/
    /*--- RTCRESTR ---*/
    /* RSTALM */
    {
        uint8_t work = p_obj->p_instance->RESTR;
        work &= (uint8_t)(~RTCRESTR_RSTALM_MASK);
        work |= RTCRESTR_RSTALM_RESET;
        p_obj->p_instance->RESTR = work;
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set alarm minute.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  compare :Compare to clock's minute. Use @ref RTC_AlarmMin.
  * @param[in]  min     :Minute. Range is @ref RTC_RangeMin.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCMINR.
  * @note       When the "compare" is RTC_ALARM_MIN_OFF, the "min" is invalid.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_alarm_set_min(rtc_t *p_obj, uint8_t compare, uint8_t min)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_min_alarm(compare));
    assert_param(check_param_min(min));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_1);
    /*------------------------------*/
    /*  Set Minute                  */
    /*------------------------------*/
    /*--- RTCMINR ---*/
    /* MI[6:0] */
    if (compare == RTC_ALARM_MIN_ON)
    {
        p_obj->p_instance->MINR = convert_time_to_reg(min);
    }
    else
    {
        p_obj->p_instance->MINR = compare;
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get alarm minute.
  * @details    -
  * @param[in]  p_obj     :Object address.
  * @param[out] p_compare :Compare to clock's minute(@ref RTC_AlarmMin). Destination address.
  * @param[out] p_min     :Minute. Destination address.
  * @retval     TXZ_SUCCESS   :Success
  * @retval     TXZ_ERROR     :Failure
  * @note       Access to RTCPAGER, RTCMINR.
  * @note       When the "*p_compare" is RTC_ALARM_MIN_OFF, the "*p_min" is invalid.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_alarm_get_min(rtc_t *p_obj, uint8_t *p_compare, uint8_t *p_min)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_compare));
    assert_param(IS_POINTER_NOT_NULL(p_min));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_1);
    /*------------------------------*/
    /*  Get Minute                  */
    /*------------------------------*/
    /*--- RTCMINR ---*/
    /* MI[6:0] */
    {
        __IO uint8_t work;

        work = p_obj->p_instance->MINR;
        if (work == RTC_ALARM_MIN_OFF)
        {
            *p_compare = RTC_ALARM_MIN_OFF;
            *p_min = 0;
        }
        else
        {
            *p_compare = RTC_ALARM_MIN_ON;
            *p_min = convert_time_to_dec(work);
        }
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set alarm hour.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  compare :Compare to clock's hour. Use @ref RTC_AlarmHour.
  * @param[in]  hour    :Hour. Range is @ref RTC_Range24Hour.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Only use 24-hour notation.
  * @note       Access to RTCPAGER, RTCHOURR.
  * @note       When the "compare" is RTC_ALARM_HOUR_OFF, the "hour" is invalid.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_alarm_set_hour_24(rtc_t *p_obj, uint8_t compare, uint8_t hour)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_hour_alarm(compare));
    assert_param(check_param_hour_24(hour));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_1);
    /*------------------------------*/
    /*  Set Hour                    */
    /*------------------------------*/
    /*--- RTCHOURR ---*/
    /* HO[5:0] */
    if (compare == RTC_ALARM_HOUR_ON)
    {
        p_obj->p_instance->HOURR = convert_time_to_reg(hour);
    }
    else
    {
        p_obj->p_instance->HOURR = compare;
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get alarm hour.
  * @details    -
  * @param[in]  p_obj     :Object address.
  * @param[out] p_compare :Compare to clock's hour (@ref RTC_AlarmHour). Destination address.
  * @param[out] p_hour    :Hour. Destination address.
  * @retval     TXZ_SUCCESS   :Success
  * @retval     TXZ_ERROR     :Failure
  * @note       Only use 24-hour notation.
  * @note       Access to RTCPAGER, RTCHOURR.
  * @note       When the "*p_compare" is RTC_ALARM_HOUR_OFF, the "*p_hour" is invalid.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_alarm_get_hour_24(rtc_t *p_obj, uint8_t *p_compare, uint8_t *p_hour)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_compare));
    assert_param(IS_POINTER_NOT_NULL(p_hour));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_1);
    /*------------------------------*/
    /*  Get Hour                    */
    /*------------------------------*/
    /*--- RTCHOURR ---*/
    /* HO[5:0] */
    {
        __IO uint8_t work;

        work = p_obj->p_instance->HOURR;
        if (work == RTC_ALARM_HOUR_OFF)
        {
            *p_compare = RTC_ALARM_HOUR_OFF;
            *p_hour = 0;
        }
        else
        {
            *p_compare = RTC_ALARM_HOUR_ON;
            *p_hour = convert_time_to_dec(work);
        }
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set alarm hour.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  compare :Compare to clock's hour. Use @ref RTC_AlarmHour.
  * @param[in]  meridiem:Meridiem. Use @ref RTC_Meridiem.
  * @param[in]  hour    :Hour. Range is @ref RTC_Range12Hour.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Only use 12-hour notation.
  * @note       Access to RTCPAGER, RTCHOURR.
  * @note       When the "compare" is RTC_ALARM_HOUR_OFF, the "meridiem" and the "hour" are invalid.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_alarm_set_hour_12(rtc_t *p_obj, uint8_t compare, uint8_t meridiem, uint8_t hour)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_hour_alarm(compare));
    assert_param(check_param_meridiem(meridiem));
    assert_param(check_param_hour_12(hour));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_1);
    /*------------------------------*/
    /*  Set Hour                    */
    /*------------------------------*/
    /*--- RTCHOURR ---*/
    /* HO[5:0] */
    if (compare == RTC_ALARM_HOUR_ON)
    {
        __IO uint8_t work = convert_time_to_reg(hour);
        work |= meridiem;
        p_obj->p_instance->HOURR = work;
    }
    else
    {
        p_obj->p_instance->HOURR = compare;
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get alarm hour.
  * @details    -
  * @param[in]  p_obj     :Object address.
  * @param[out] p_compare :Compare to clock's hour (@ref RTC_AlarmHour). Destination address.
  * @param[out] p_meridiem:Meridiem @ref RTC_Meridiem. Destination address.
  * @param[out] p_hour    :Hour. Destination address.
  * @retval     TXZ_SUCCESS   :Success
  * @retval     TXZ_ERROR     :Failure
  * @note       Only use 12-hour notation.
  * @note       Access to RTCPAGER, RTCHOURR.
  * @note       When the "*p_compare" is RTC_ALARM_HOUR_OFF, the "*p_meridiem" and the "*p_hour" is invalid.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_alarm_get_hour_12(rtc_t *p_obj, uint8_t *p_compare, uint8_t *p_meridiem, uint8_t *p_hour)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_compare));
    assert_param(IS_POINTER_NOT_NULL(p_meridiem));
    assert_param(IS_POINTER_NOT_NULL(p_hour));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_1);
    /*------------------------------*/
    /*  Get Hour                    */
    /*------------------------------*/
    /*--- RTCHOURR ---*/
    /* HO[5:0] */
    {
        __IO uint8_t work;

        work = p_obj->p_instance->HOURR;
        if (work == RTC_ALARM_HOUR_OFF)
        {
            *p_compare = RTC_ALARM_HOUR_OFF;
            *p_meridiem = 0;
            *p_hour = 0;
        }
        else
        {
            *p_compare = RTC_ALARM_HOUR_ON;
            /* Meridiem */
            if ((work & RTCPAGER_HO_12_MERIDIEM_MASK) == RTCPAGER_HO_12_MERIDIEM_AM)
            {
                *p_meridiem = RTC_MERIDIEM_AM;
            }
            else
            {
                *p_meridiem = RTC_MERIDIEM_PM;
            }
            /* hour */
            *p_hour = convert_time_to_dec((work & RTCPAGER_HO_12_HOUR_MASK));
        }
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set alarm day.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  compare :Compare to clock's day. Use @ref RTC_AlarmDay.
  * @param[in]  day     :Day. Use @ref RTC_Day.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCDAYR.
  * @note       When the "compare" is RTC_ALARM_DAY_OFF, the "day" is invalid.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_alarm_set_day(rtc_t *p_obj, uint8_t compare, uint8_t day)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_day_alarm(compare));
    assert_param(check_param_day(day));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_1);
    /*------------------------------*/
    /*  Set Day                     */
    /*------------------------------*/
    /*--- RTCDAYR ---*/
    /* WE[2:0] */
    if (compare == RTC_ALARM_DAY_ON)
    {
        p_obj->p_instance->DAYR = day;
    }
    else
    {
        p_obj->p_instance->DAYR = compare;
    }


    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get alarm day.
  * @details    -
  * @param[in]  p_obj     :Object address.
  * @param[out] p_compare :Compare to clock's day (@ref RTC_AlarmDay). Destination address.
  * @param[out] p_day     :Day @ref RTC_Day. Destination address.
  * @retval     TXZ_SUCCESS   :Success
  * @retval     TXZ_ERROR     :Failure
  * @note       Access to RTCPAGER, RTCDAYR.
  * @note       When the "*p_compare" is RTC_ALARM_DAY_OFF, the "*p_day" is invalid.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_alarm_get_day(rtc_t *p_obj, uint8_t *p_compare, uint8_t *p_day)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_compare));
    assert_param(IS_POINTER_NOT_NULL(p_day));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_1);
    /*------------------------------*/
    /*  Get Day                     */
    /*------------------------------*/
    /*--- RTCDAYR ---*/
    /* WE[2:0] */
    {
        __IO uint8_t work;

        work = p_obj->p_instance->DAYR;
        if (work == RTC_ALARM_DAY_OFF)
        {
            *p_compare = RTC_ALARM_DAY_OFF;
            *p_day = 0;
        }
        else
        {
            *p_compare = RTC_ALARM_DAY_ON;
            *p_day = work;
        }
    }

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Set alarm date.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @param[in]  compare :Compare to clock's date. Use @ref RTC_AlarmDate.
  * @param[in]  date    :Date. Range is @ref RTC_RangeDate.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Access to RTCPAGER, RTCDATER.
  * @note       When the "compare" is RTC_ALARM_DATE_OFF, the "date" is invalid.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_alarm_set_date(rtc_t *p_obj, uint8_t compare, uint8_t date)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter. */
    assert_param(check_param_date_alarm(compare));
    assert_param(check_param_date(date));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_1);
    /*------------------------------*/
    /*  Set Date                    */
    /*------------------------------*/
    /*--- RTCDATER ---*/
    /* DA[5:0] */
    if (compare == RTC_ALARM_DATE_ON)
    {
        p_obj->p_instance->DATER = convert_time_to_reg(date);
    }
    else
    {
        p_obj->p_instance->DATER = compare;
    }


    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get alarm date.
  * @details    -
  * @param[in]  p_obj     :Object address.
  * @param[out] p_compare :Compare to clock's date (@ref RTC_AlarmDate). Destination address.
  * @param[out] *p_date   :Date. Destination address.
  * @retval     TXZ_SUCCESS   :Success
  * @retval     TXZ_ERROR     :Failure
  * @note       Access to RTCPAGER, RTCDATER.
  * @note       When the "*p_compare" is RTC_ALARM_DATE_OFF, the "*p_date" is invalid.
  */
/*--------------------------------------------------*/
TXZ_Result rtc_alarm_get_date(rtc_t *p_obj, uint8_t *p_compare, uint8_t *p_date)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RTC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_compare));
    assert_param(IS_POINTER_NOT_NULL(p_date));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Set Page                    */
    /*------------------------------*/
    /*--- RTCPAGER ---*/
    /* PAGE */
    set_page(p_obj, RTCPAGER_PAGE_1);
    /*------------------------------*/
    /*  Get Date                    */
    /*------------------------------*/
    /*--- RTCDATER ---*/
    /* DA[5:0] */
    {
        __IO uint8_t work;

        work = p_obj->p_instance->DATER;
        if (work == RTC_ALARM_DATE_OFF)
        {
            *p_compare = RTC_ALARM_DATE_OFF;
            *p_date = 0;
        }
        else
        {
            *p_compare = RTC_ALARM_DATE_ON;
            *p_date = convert_time_to_dec(work);
        }
    }

    return (res);
}

/**
 *  @}
 */ /* End of group BSP_BSP_RTC_Exported_functions */


/**
 *  @}
 */ /* End of group BSP_BSP_RTC */

/**
 *  @} 
 */ /* End of group BSP */

#endif /* defined(__BSP_CONFIG_ADBUN_M3HQA_RTC_H)  */
#ifdef __cplusplus
}
#endif /* __cplusplus */


