/** 
 *******************************************************************************
 * @file    bsp_config_adbun_m3hqa_uart_io.c
 * @brief   UART input/output
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif
/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdarg.h>
#include "bsp_uart_io.h"

#if defined(__BSP_CONFIG_ADBUN_M3HQA_UART_IO_H)
/**
 *  @addtogroup BSP BSP
 *  @{
 */

/** 
 *  @addtogroup BSP_BSP_UART_IO
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_Private_macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group BSP_BSP_UART_IO_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group BSP_BSP_UART_IO_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_Private_define
 *  @{
 */

#define UART_IO_NULL           ((void *)0)  /*!< Null Pointer */

/*----------------------*/
/*  for Debug           */
/*----------------------*/
#ifdef DEBUG
    #if 1
        #define UART_IO_DEBUG
    #endif
#endif
/** 
 *  @}
 */ /* End of group BSP_BSP_UART_IO_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_Private_define
 *  @{
 */

/** 
 *  @}
 */ /* End of group BSP_BSP_UART_IO_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_Private_typedef
 *  @{
 */

/**
 *  @}
 */ /* End of group BSP_BSP_UART_IO_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_Private_variables
 *  @{
 */
static uart_io_t instance;  /*!< USB UART Instance.  */
/**
 *  @}
 */ /* End of group BSP_BSP_UART_IO_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_Private_fuctions
 *  @{
 */
static void uart_io_error(void);
static int put_char(int c);
static int get_char(void);
static void transmit_handler(uint32_t id, TXZ_Result result);
static void receive_handler(uint32_t id, TXZ_Result result, uart_receive_t *p_info);
/*--------------------------------------------------*/
/** 
  * @brief  Error Hook
  * @param  -
  * @return -
  * @retval -
  * @note   for debug
  */
/*--------------------------------------------------*/
static void uart_io_error(void)
{
#ifdef UART_IO_DEBUG
    while(1)
    {
        __NOP();
    }
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  The character is shown to an output stream.
  * @param  c :Character data.
  * @return Output Character data.
  * @retval (value < 0) :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static int put_char(int c)
{
    int ret = -1;
    uart_io_private_tx_t *p_tx = &instance.info.tx;

    if (p_tx->state == UART_IO_SEND_IDLE)
    {
        uart_transmit_t param;
        volatile TXZ_WorkState loopBreak = TXZ_BUSY;
        uint32_t status;

        /*------------------------------*/
        /*  Transmit                    */
        /*------------------------------*/
        p_tx->buff[0] = (uint8_t)c;
        p_tx->wp      = 1;
        p_tx->rp      = 0;
        p_tx->state   = UART_IO_SEND_RUN;
        param.tx8.p_data    = &p_tx->buff[0];
        param.tx8.num       = p_tx->wp;
        if (uart_transmitIt(&instance.info.uart, &param) != TXZ_SUCCESS)
        {
            uart_io_error();
        }
        /*------------------------------*/
        /*  Wait to transmit            */
        /*------------------------------*/
        loopBreak = TXZ_BUSY;
        while(loopBreak == TXZ_BUSY)
        {
            switch (p_tx->state)
            {
            case UART_IO_SEND_RUN:
                /* no process */
                break;
            case UART_IO_SEND_SUCCESS:
                if (uart_get_status(&instance.info.uart, &status) == TXZ_SUCCESS)
                {
                    if ((status & UART_TX_STATE_RUN) != UART_TX_STATE_RUN)
                    {
                        loopBreak = TXZ_DONE;
                    }
                }
                break;
            default:
                loopBreak = TXZ_DONE;
                break;
            }
        }
        /*------------------------------*/
        /*  Check Error                 */
        /*------------------------------*/
        if (p_tx->state == UART_IO_SEND_FAILURE)
        {
            uart_io_error();
        }
        /* Clear status. */
        p_tx->state = UART_IO_SEND_IDLE;
        /* Set result. */
        ret = c;
    }

    return (ret);
}

/*--------------------------------------------------*/
/** 
  * @brief  The character is acquired from an input stream.
  * @param  -
  * @return Input Character data.
  * @retval (value < 0) :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static int get_char(void)
{
    volatile TXZ_WorkState loopBreak = TXZ_BUSY;
    volatile int c = -1;
    uart_io_private_rx_t *p_rx = &instance.info.rx;

    while (loopBreak == TXZ_BUSY)
    {
        if (p_rx->state == UART_IO_RECV_RUN)
        {
            /*>>> Critical information. */
            volatile uint32_t length = p_rx->length;
            /*<<< Critical information. */

            /*------------------------------*/
            /*  Check Receive Length        */
            /*------------------------------*/
            if (length > 0)
            {
                c = (int)p_rx->buff[p_rx->rp];
                p_rx->rp++;
                if (p_rx->rp >= UART_IO_RECEIVE_MAX)
                {
                    p_rx->rp = 0;
                }
                /*>>> Critical information. */
                NVIC_DisableIRQ(instance.init.irq.rx);
                p_rx->length--;
                NVIC_EnableIRQ(instance.init.irq.rx);
                /*<<< Critical information. */
                loopBreak = TXZ_DONE;
            }
        }
        else
        {
            loopBreak = TXZ_DONE;
        }
    }

    return (c);
}

/*--------------------------------------------------*/
/** 
  * @brief  Transmit handler.
  * @param  id      :User ID.
  * @param  result  :Result.
  * @return -
  * @retval -
  * @note   Called by uart driver.
  */
/*--------------------------------------------------*/
static void transmit_handler(uint32_t id, TXZ_Result result)
{
    uart_io_t *p_instance = (uart_io_t *)id;

    if (p_instance != UART_IO_NULL)
    {
        uart_io_private_tx_t *p_tx = &p_instance->info.tx;

        if (p_tx->state != UART_IO_SEND_IDLE)
        {
            /*------------------------------*/
            /*  Result Check                */
            /*------------------------------*/
            if (result == TXZ_SUCCESS)
            {
                p_tx->state = UART_IO_SEND_SUCCESS;
            }
            else
            {
                p_tx->state = UART_IO_SEND_FAILURE;
            }
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Receive handler.
  * @param  id      :User ID.
  * @param  result  :Result.
  * @param  p_info  :Received information Source Address.
  * @return -
  * @retval -
  * @note   Called by uart driver.
  */
/*--------------------------------------------------*/
static void receive_handler(uint32_t id, TXZ_Result result, uart_receive_t *p_info)
{
    uart_io_t *p_instance = (uart_io_t *)id;

    if ((p_instance != UART_IO_NULL) && (p_info != UART_IO_NULL))
    {
        uart_io_private_rx_t *p_rx = &p_instance->info.rx;

        if (p_rx->state == UART_IO_RECV_RUN)
        {
            if (result == TXZ_SUCCESS)
            {
                uint32_t i;
                uint32_t num    = p_info->rx8.num;
                /*>>> Critical information. */
                volatile uint32_t length = p_rx->length;
                /*<<< Critical information. */

                if (length < UART_IO_RECEIVE_MAX) 
                {
                    if ((length + num) > UART_IO_RECEIVE_MAX)
                    {
                        num = (uint32_t)(UART_IO_RECEIVE_MAX - length);
                    }
                    for (i=0; i<num; i++)
                    {
                        p_rx->buff[p_rx->wp + i] = *(p_info->rx8.p_data + i);
                        p_rx->wp++;
                        if (p_rx->wp >= UART_IO_RECEIVE_MAX)
                        {
                            p_rx->wp = 0;
                        }
                    }
                    /*>>> Critical information. */
                    p_rx->length += num;
                    /*<<< Critical information. */
                }
                else
                {
                    p_rx->state = UART_IO_RECV_FAILURE;
                }
            }
            else
            {
                p_rx->state = UART_IO_RECV_FAILURE;
            }
        }
    }
}
/**
 *  @}
 */ /* End of group BSP_BSP_UART_IO_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_Exported_functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Initialize
  * @param  p_param :Initial Setting Information Source Address.
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   Parameter check isn't performed.
  * @attention   Singleton.
  */
/*--------------------------------------------------*/
TXZ_Result uart_io_initialize(uart_io_initial_setting_t *p_param)
{
    /*------------------------------*/
    /*  Initialize Variable         */
    /*------------------------------*/
    instance.init.id            = p_param->id;
    instance.init.p_reg         = p_param->p_reg;
    instance.init.boudrate      = p_param->boudrate;
    instance.init.irq.tx        = p_param->irq.tx;
    instance.init.irq.rx        = p_param->irq.rx;
    instance.init.irq.err       = p_param->irq.err;
    instance.info.tx.state      = UART_IO_SEND_IDLE;
    instance.info.tx.wp         = 0;
    instance.info.tx.rp         = 0;
    instance.info.rx.state      = UART_IO_RECV_RUN;     /* Auto run. */
    instance.info.rx.wp         = 0;
    instance.info.rx.rp         = 0;
    instance.info.rx.length     = 0;
    /*------------------------------*/
    /*  Uart Setting                */
    /*------------------------------*/
    /*--- Construct (Register Allocation) ---*/
    instance.info.uart.p_instance = instance.init.p_reg;
    /*--- Driver Initialize ---*/
    instance.info.uart.init.id            = (uint32_t)&instance;  /* User ID: Any Value (BSP...UART driver instance address). */
    instance.info.uart.init.clock.prsel   = UART_PLESCALER_1;
    {
        cg_t paramCG;

        paramCG.p_instance = TSB_CG;
        if (uart_get_boudrate_setting(
                cg_get_phyt0(&paramCG), 
                &instance.info.uart.init.clock, 
                instance.init.boudrate, 
                &instance.info.uart.init.boudrate
                ) != TXZ_SUCCESS)
        {
            uart_io_error();
            return(TXZ_ERROR);
        }
    }
    /* Uart     */
    instance.info.uart.init.inttx         = UART_TX_INT_ENABLE;
    instance.info.uart.init.intrx         = UART_RX_INT_ENABLE;
    instance.info.uart.init.interr        = UART_ERR_INT_ENABLE;
    instance.info.uart.init.txfifo.inttx  = UART_TX_FIFO_INT_DISABLE;
    instance.info.uart.init.txfifo.level  = 0;
    instance.info.uart.init.rxfifo.intrx  = UART_RX_FIFO_INT_DISABLE;
    instance.info.uart.init.rxfifo.level  = 8;
    instance.info.uart.init.nf            = UART_NOISE_FILTER_NON;
    instance.info.uart.init.ctse          = UART_CTS_DISABLE;
    instance.info.uart.init.rtse          = UART_RTS_DISABLE;
    instance.info.uart.init.iv            = UART_DATA_COMPLEMENTION_DISABLE;
    instance.info.uart.init.dir           = UART_DATA_DIRECTION_LSB;
    instance.info.uart.init.sblen         = UART_STOP_BIT_1;
    instance.info.uart.init.even          = UART_PARITY_BIT_ODD;
    instance.info.uart.init.pe            = UART_PARITY_DISABLE;
    instance.info.uart.init.sm            = UART_DATA_LENGTH_8;
    instance.info.uart.transmit.handler   = transmit_handler;
    instance.info.uart.receive.handler    = receive_handler;
    if (uart_init(&instance.info.uart) != TXZ_SUCCESS)
    {
        uart_io_error();
        return(TXZ_ERROR);
    }
    /*--- Driver Start :Receive ---*/
    {
        uart_receive_t rx;

        rx.rx8.p_data = &instance.info.work[0];
        rx.rx8.num    = UART_RX_FIFO_MAX;
        if (uart_receiveIt(&instance.info.uart, &rx) != TXZ_SUCCESS)
        {
            uart_io_error();
            return(TXZ_ERROR);
        }
    }
    /*------------------------------*/
    /*  Enable Interrupt            */
    /*------------------------------*/
    NVIC_EnableIRQ(instance.init.irq.tx);
    NVIC_EnableIRQ(instance.init.irq.rx);
    NVIC_EnableIRQ(instance.init.irq.err);

    return(TXZ_SUCCESS);
}

/*--------------------------------------------------*/
/** 
  * @brief  Finalize
  * @param  p_instance :Instance Address.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention   Singleton.
  */
/*--------------------------------------------------*/
void uart_io_finalize(void)
{
    if (instance.info.uart.p_instance != UART_IO_NULL)
    {    
        /*------------------------------*/
        /*  Disable Enable              */
        /*------------------------------*/
        NVIC_DisableIRQ(instance.init.irq.tx);
        NVIC_DisableIRQ(instance.init.irq.rx);
        NVIC_DisableIRQ(instance.init.irq.err);
        /*------------------------------*/
        /*  Uart Finalize               */
        /*------------------------------*/
        /*--- Driver Finalize ---*/
        if (uart_deinit(&instance.info.uart) != TXZ_SUCCESS)
        {
            uart_io_error();
        }
        /*--- Destruct (Register release) ---*/
        instance.info.uart.p_instance = UART_IO_NULL;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  TX IRQ Handler.
  * @param  p_instance :Instance Address.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention   Singleton.
  */
/*--------------------------------------------------*/
void uart_io_tx_irq_handler(void)
{
    uart_transmit_irq_handler(&instance.info.uart);
}

/*--------------------------------------------------*/
/** 
  * @brief  RX IRQ Handler.
  * @param  p_instance :Instance Address.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention   Singleton.
  */
/*--------------------------------------------------*/
void uart_io_rx_irq_handler(void)
{
    uart_receive_irq_handler(&instance.info.uart);
}

/*--------------------------------------------------*/
/** 
  * @brief  ERR IRQ Handler.
  * @param  p_instance :Instance Address.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention   Singleton.
  */
/*--------------------------------------------------*/
void uart_io_err_irq_handler(void)
{
    
    uart_error_irq_handler(&instance.info.uart);
}
/**
 *  @}
 */ /* End of group BSP_BSP_UART_IO_Exported_functions */


/*------------------------------------------------------------------------------*/
/*  Semihosting Retarget                                                        */
/*------------------------------------------------------------------------------*/
#if defined ( __CC_ARM   )      /* RealView Compiler */
struct __FILE { int handle; /* Add whatever you need here */ };
FILE __stdout;
FILE __stdin;
int fputc(int ch, FILE *f) {
#elif defined ( __ICCARM__ )    /*IAR Compiler */
int putchar(int ch) {
#endif
    return (put_char(ch));
}

#if defined ( __CC_ARM   )      /* RealView Compiler */
int getc(FILE * f)
#elif defined ( __ICCARM__ )    /*IAR Compiler */
int getchar(void)
#endif
{
    return (get_char());
}

#if defined ( __CC_ARM   )      /* RealView Compiler */
int ferror(FILE *f)
#elif defined ( __ICCARM__ )    /*IAR Compiler */
int ferror(void)
#endif
{
    return (0);
}

/**
 *  @}
 */ /* End of group BSP_BSP_UART_IO */

/**
 *  @} 
 */ /* End of group BSP */

#endif /* defined(__BSP_CONFIG_ADBUN_M3HQA_UART_IO_H)  */
#ifdef __cplusplus
}
#endif /* __cplusplus */
