/** 
 *******************************************************************************
 * @file    bsp_config_adbun_m3hqa_uart_io_dma.c
 * @brief   UART input/output
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif
/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdarg.h>
#include "bsp_uart_io_dma.h"

#if defined(__BSP_CONFIG_ADBUN_M3HQA_UART_IO_DMA_H)
/**
 *  @addtogroup BSP BSP
 *  @{
 */

/** 
 *  @addtogroup BSP_BSP_UART_IO_DMA
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_DMA_Private_macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group BSP_BSP_UART_IO_DMA_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_DMA_Private_define
 *  @{
 */
/*----------------------*/
/*  for Debug           */
/*----------------------*/
#ifdef DEBUG
    #if 1
        #define UART_IO_CFG_DEBUG
    #endif
#endif
/** 
 *  @}
 */ /* End of group BSP_BSP_UART_IO_DMA_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_DMA_Private_define
 *  @{
 */

#define UART_IO_NULL           ((void *)0)  /*!< Null Pointer */

/** 
 *  @}
 */ /* End of group BSP_BSP_UART_IO_DMA_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_DMA_Private_define
 *  @{
 */
/** 
 *  @}
 */ /* End of group BSP_BSP_UART_IO_DMA_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_DMA_Private_typedef
 *  @{
 */
/**
 *  @}
 */ /* End of group BSP_BSP_UART_IO_DMA_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_DMA_Private_variables
 *  @{
 */
static uart_io_dma_t    instance;   /*!< USB UART Instance. */
/**
 *  @}
 */ /* End of group BSP_BSP_UART_IO_DMA_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_DMA_Private_fuctions
 *  @{
 */
static void uart_io_error(void);
static int put_char(int c);
static int get_char(void);
static void transmit_handler(uint32_t id, TXZ_Result result);
static void receive_handler(uint32_t id, TXZ_Result result, uart_receive_t *p_info);
/*--------------------------------------------------*/
/** 
  * @brief  Error Hook
  * @param  -
  * @return -
  * @retval -
  * @note   for debug
  */
/*--------------------------------------------------*/
static void uart_io_error(void)
{
#ifdef UART_IO_CFG_DEBUG
    while(1)
    {
        __nop();
    }
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  The character is shown to an output stream.
  * @param  c :Character data.
  * @return Output Character data.
  * @retval (value < 0) :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static int put_char(int c)
{
    int ret = -1;
    uart_io_dma_private_tx_t *p_tx = &instance.info.tx;

    if (p_tx->state == UART_IO_DMA_SEND_IDLE)
    {
        uart_transmit_t param;
        volatile TXZ_WorkState loopBreak = TXZ_BUSY;
        uint32_t status;

        /*------------------------------*/
        /*  Transmit                    */
        /*------------------------------*/
        p_tx->buff[0] = (uint8_t)c;
        p_tx->state   = UART_IO_DMA_SEND_RUN;
        param.tx8.p_data    = &p_tx->buff[0];
        param.tx8.num       = 1;
        if (uart_dma_transmitIt(&instance.info.uart, &param) != TXZ_SUCCESS)
        {
            uart_io_error();
        }
        /*------------------------------*/
        /*  Wait to transmit            */
        /*------------------------------*/
        loopBreak = TXZ_BUSY;
        while(loopBreak == TXZ_BUSY)
        {
            switch (p_tx->state)
            {
            case UART_IO_DMA_SEND_RUN:
                /* no process */
                break;
            case UART_IO_DMA_SEND_SUCCESS:
                if (uart_get_status(&instance.info.uart.super, &status) == TXZ_SUCCESS)
                {
                    if (((status & UART_TX_STATE_MASK) == UART_TX_STATE_SLEEP) && 
                        ((status & UART_TX_FIFO_LEVEL_MASK) == 0)) 
                    {
                        loopBreak = TXZ_DONE;
                    }
                }
                break;
            default:
                loopBreak = TXZ_DONE;
                break;
            }
        }
        /*------------------------------*/
        /*  Check Error                 */
        /*------------------------------*/
        if (p_tx->state == UART_IO_DMA_SEND_FAILURE)
        {
            uart_io_error();
        }
        /* Clear status. */
        p_tx->state = UART_IO_DMA_SEND_IDLE;
        /* Set result. */
        ret = c;
    }

    return (ret);
}

/*--------------------------------------------------*/
/** 
  * @brief  The character is acquired from an input stream.
  * @param  -
  * @return Input Character data.
  * @retval (value < 0) :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static int get_char(void)
{
    volatile int c = -1;
    uart_io_dma_private_rx_t *p_rx = &instance.info.rx;

    if (p_rx->state == UART_IO_DMA_RECV_IDLE)
    {
        uart_receive_t param;

        /*------------------------------*/
        /*  Receive                     */
        /*------------------------------*/
        p_rx->buff[0] = 0;
        p_rx->state   = UART_IO_DMA_RECV_RUN;
        param.rx8.p_data    = &p_rx->buff[0];
        param.rx8.num       = 1;
        if (uart_dma_receiveIt(&instance.info.uart, &param) != TXZ_SUCCESS)
        {
            uart_io_error();
        }
        /*------------------------------*/
        /*  Wait to receive             */
        /*------------------------------*/
        while(p_rx->state == UART_IO_DMA_RECV_RUN)
        {
            /* no processing */
        }
        /*------------------------------*/
        /*  Check Error                 */
        /*------------------------------*/
        if (p_rx->state == UART_IO_DMA_RECV_FAILURE)
        {
            uart_io_error();
        }
        /* Clear status. */
        p_rx->state = UART_IO_DMA_RECV_IDLE;
        /* Set result. */
        c = p_rx->buff[0];
    }

    return (c);
}

/*--------------------------------------------------*/
/** 
  * @brief  Transmit handler.
  * @param  id      :User ID.
  * @param  result  :Result.
  * @return -
  * @retval -
  * @note   Called by uart driver.
  */
/*--------------------------------------------------*/
static void transmit_handler(uint32_t id, TXZ_Result result)
{
    uart_io_dma_t *p_instance = (uart_io_dma_t *)id;

    if (p_instance != UART_IO_NULL)
    {
        uart_io_dma_private_tx_t *p_tx = &p_instance->info.tx;

        if (p_tx->state != UART_IO_DMA_SEND_IDLE)
        {
            /*------------------------------*/
            /*  Result Check                */
            /*------------------------------*/
            if (result == TXZ_SUCCESS)
            {
                p_tx->state = UART_IO_DMA_SEND_SUCCESS;
            }
            else
            {
                p_tx->state = UART_IO_DMA_SEND_FAILURE;
            }
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Receive handler.
  * @param  id      :User ID.
  * @param  result  :Result.
  * @param  p_info  :Received information Source Address.
  * @return -
  * @retval -
  * @note   Called by uart driver.
  */
/*--------------------------------------------------*/
static void receive_handler(uint32_t id, TXZ_Result result, uart_receive_t *p_info)
{
    uart_io_dma_t *p_instance = (uart_io_dma_t *)id;

    if ((p_instance != UART_IO_NULL) && (p_info != UART_IO_NULL))
    {
        uart_io_dma_private_rx_t *p_rx = &p_instance->info.rx;


        if (p_rx->state == UART_IO_DMA_RECV_RUN)
        {
            /*------------------------------*/
            /*  Result Check                */
            /*------------------------------*/
            if (result == TXZ_SUCCESS)
            {
                p_rx->state = UART_IO_DMA_RECV_SUCCESS;
            }
            else
            {
                p_rx->state = UART_IO_DMA_RECV_FAILURE;
            }
        }
    }
}
/**
 *  @}
 */ /* End of group BSP_BSP_UART_IO_DMA_Private_fuctions */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup BSP_BSP_UART_IO_DMA_Exported_functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Initialize
  * @param  p_param :Initial Setting Information Source Address.
  * @return Instance Address.
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention   Singleton.
  */
/*--------------------------------------------------*/
uart_io_dma_t *uart_io_dma_initialize(uart_io_dma_initial_setting_t *p_param)
{
    /*------------------------------*/
    /*  Initialize Variable         */
    /*------------------------------*/
    instance.init.id           = p_param->id;
    instance.init.p_reg        = p_param->p_reg;
    instance.init.boudrate     = p_param->boudrate;
    instance.init.dma.p_dma    = p_param->dma.p_dma;
    instance.init.dma.tx       = p_param->dma.tx;
    instance.init.dma.rx       = p_param->dma.rx;
    instance.info.tx.state     = UART_IO_DMA_SEND_IDLE;
    instance.info.rx.state     = UART_IO_DMA_RECV_IDLE;
    /*------------------------------*/
    /*  Uart Setting                */
    /*------------------------------*/
    /*--- Construct (Register Allocation) ---*/
    instance.info.uart.super.p_instance = instance.init.p_reg;
    /*--- Driver Initialize ---*/
    instance.info.uart.super.init.id            = (uint32_t)&instance;  /* User ID: Any Value (BSP...UART driver instance address). */
    instance.info.uart.super.init.clock.prsel   = UART_PLESCALER_1;
    {
        cg_t paramCG;

        paramCG.p_instance = TSB_CG;
        if (uart_get_boudrate_setting(
                cg_get_phyt0(&paramCG), 
                &instance.info.uart.super.init.clock, 
                instance.init.boudrate, 
                &instance.info.uart.super.init.boudrate
                ) != TXZ_SUCCESS)
        {
            uart_io_error();
        }
    }
    /* Uart     */
    instance.info.uart.super.init.inttx         = UART_TX_INT_DISABLE;
    instance.info.uart.super.init.intrx         = UART_RX_INT_DISABLE;
    instance.info.uart.super.init.interr        = UART_ERR_INT_DISABLE;
    instance.info.uart.super.init.txfifo.inttx  = UART_TX_FIFO_INT_DISABLE;
    instance.info.uart.super.init.txfifo.level  = 0;
    instance.info.uart.super.init.rxfifo.intrx  = UART_RX_FIFO_INT_DISABLE;
    instance.info.uart.super.init.rxfifo.level  = 8;
    instance.info.uart.super.init.nf            = UART_NOISE_FILTER_NON;
    instance.info.uart.super.init.ctse          = UART_CTS_DISABLE;
    instance.info.uart.super.init.rtse          = UART_RTS_DISABLE;
    instance.info.uart.super.init.iv            = UART_DATA_COMPLEMENTION_DISABLE;
    instance.info.uart.super.init.dir           = UART_DATA_DIRECTION_LSB;
    instance.info.uart.super.init.sblen         = UART_STOP_BIT_1;
    instance.info.uart.super.init.even          = UART_PARITY_BIT_ODD;
    instance.info.uart.super.init.pe            = UART_PARITY_DISABLE;
    instance.info.uart.super.init.sm            = UART_DATA_LENGTH_8;
    instance.info.uart.super.transmit.handler   = transmit_handler;
    instance.info.uart.super.receive.handler    = receive_handler;
    /* Uart DMA */
    instance.info.uart.init.p_obj    = instance.init.dma.p_dma;
    instance.info.uart.init.tx.type  = UART_DMA_TYPE_MEM_TO_PERI;
    instance.info.uart.init.tx.ch    = instance.init.dma.tx;
    instance.info.uart.init.rx.type  = UART_DMA_TYPE_PERI_TO_MEM;
    instance.info.uart.init.rx.ch    = instance.init.dma.rx;
    if (uart_dma_init(&instance.info.uart) != TXZ_SUCCESS)
    {
        uart_io_error();
    }

    return(&instance);
}

/*--------------------------------------------------*/
/** 
  * @brief  Finalize
  * @param  p_instance :Instance Address.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention   Singleton.
  */
/*--------------------------------------------------*/
void uart_io_dma_finalize(uart_io_dma_t *p_instance)
{
    /*------------------------------*/
    /*  Uart Finalize               */
    /*------------------------------*/
    /*--- Driver Finalize ---*/
    if (uart_dma_deinit(&p_instance->info.uart) != TXZ_SUCCESS)
    {
        uart_io_error();
    }
    /*--- Destruct (Register release) ---*/
    p_instance->info.uart.super.p_instance = UART_IO_NULL;
}
/**
 *  @}
 */ /* End of group BSP_BSP_UART_IO_DMA_Exported_functions */


/*------------------------------------------------------------------------------*/
/*  Semihosting Retarget                                                        */
/*------------------------------------------------------------------------------*/
#if defined ( __CC_ARM   )      /* RealView Compiler */
struct __FILE { int handle; /* Add whatever you need here */ };
FILE __stdout;
FILE __stdin;
int fputc(int ch, FILE *f) {
#elif defined ( __ICCARM__ )    /*IAR Compiler */
int putchar(int ch) {
#endif
    return (put_char(ch));
}

#if defined ( __CC_ARM   )      /* RealView Compiler */
int getc(FILE * f)
#elif defined ( __ICCARM__ )    /*IAR Compiler */
int getchar(void)
#endif
{
    return (get_char());
}

#if defined ( __CC_ARM   )      /* RealView Compiler */
int ferror(FILE *f)
#elif defined ( __ICCARM__ )    /*IAR Compiler */
int ferror(void)
#endif
{
    return (0);
}

/**
 *  @}
 */ /* End of group BSP_BSP_UART_IO_DMA */

/**
 *  @} 
 */ /* End of group BSP */

#endif /* defined(__BSP_CONFIG_ADBUN_M3HQA_UART_IO_DMA_H)  */
#ifdef __cplusplus
}
#endif /* __cplusplus */
