/** 
 *******************************************************************************
 * @file    main.c
 * @brief   ADC Monitor Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_uart_io.h"
#include "bsp_timer.h"
#include "bsp_led.h"
#include "adc_dma.h"

/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup ADC_Monitor ADC_Monitor Sample Appli
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup ADC_Monitor_Private_macro ADC_Monitor Private Macro
 *  @{
 */

#define MAIN_NULL       ((void *)0)     /*!< Null Pointer  */
#define USER_ID_UART    ((uint32_t)1)   /*!< User ID :USB UART (Any value) */
/* no define */

/** 
 *  @}
 */ /* End of group ADC_Monitor_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup ADC_Monitor_Private_define ADC_Monitor Private Define
 *  @{
 */

/*----------------------*/
/*  ADC Setting         */
/*----------------------*/
/** 
 *  @defgroup ADC_Setting ADC Setting
 *  @{
 */
#define CFG_ADC_WAIT_TIME             ((uint32_t)1000)    /*!< Conversion Wait Time(tick) */

#define CFG_ADC_CMP_25_SMALL_THAN (0x00007F60UL);   /*! If temp >= 25 c, Small than this value */
#define CFG_ADC_CMP_35_BIG_THAN   (0x00006160UL);   /*! If temp < 36 c,  Big than this value */
   

/** 
 *  @}
 */ /* End of group ADC_Setting */
 
/** 
 *  @defgroup ADC_ADxMOD1_Setting ADC ADxMOD1 Setting
 *  @{
 */
#define CFG_ADC_MOD1_AVDD5_5V         ((uint32_t)0x00004000)    /*!< AVDD5 = 4.5 - 5.5[V] */
#define CFG_ADC_MOD1_AVDD5_3V         ((uint32_t)0x0000B001)    /*!< AVDD5 = 2.7 - 4.5[V] */
/** 
 *  @}
 */ /* End of group ADC_ADxMOD1_Setting */

/*----------------------*/
/*  Output Setting      */
/*----------------------*/
/** 
 *  @defgroup Output_Setting Output Setting
 *  @{
 */
#define CFG_OUTPUT_INTERVAL           ((uint32_t)5000)   /*!< deg.C output period(ms) */
/** 
 *  @}
 */ /* End of group Output_Setting */


/** 
 *  @}
 */ /* End of group ADC_Monitor_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup ADC_Monitor_Private_define ADC_Monitor Private Define
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_IDLE,          /*!< Idle.             */
    PROC_START,         /*!< Start.            */
    PROC_STOP,          /*!< Stop.             */
    PROC_INTERVAL_WAIT, /*!< Interval Wait.    */
    PROC_CONVERSION,    /*!< Conversion.       */
    PROC_OUTPUT,        /*!< Output.           */
    PROC_ERROR          /*!< Error.            */
} Processing;
/** 
 *  @}
 */ /* End of group ADC_Monitor_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup ADC_Monitor_Private_typedef ADC_Monitor Private Typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
    t32a_t      t32a;              /*!< Driver      :T32A.     */
    adc_t       adc;               /*!< Driver      :ADC.      */
    uart_io_t   *p_uart;           /*!< Application :USB UART. */
    timer_t     timer;             /*!< Application :Timer (1ms timer). */
    led_t       led[(uint32_t)BSP_LED_MAX]; /*!< Application :LED                */
} instance_t;

/*----------------------------------*/
/** 
 * @brief Information.
*/
/*----------------------------------*/
typedef struct
{
   TXZ_FunctionalState update;     /*!< Update flag.       */
   uint32_t            count;      /*!< Update flag.       */
} information_t;
/**
 *  @}
 */ /* End of group ADC_Monitor_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup ADC_Monitor_Private_variables ADC_Monitor Private Variables
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief Variables.
*/
/*----------------------------------*/
static instance_t instance;     /*!< Instance.                */
static Processing proc;         /*!< Processing mode.         */
static information_t info;      /*!< Clock information.       */

static uint32_t value ;         /*!< Value.                   */
static uint32_t value_shift ;   /*!< Value Shift.             */
static int32_t degC ;           /*!< degC.                    */
static uint32_t cmp0 , cmp1 ;   /*!< cmp0, cmp1               */
static led_t *p_led[4] = {MAIN_NULL, MAIN_NULL, MAIN_NULL, MAIN_NULL };   /*!< LED instance.          */
static char led_value[2] ;      /*!< LED status.          */

/**
 *  @}
 */ /* End of group ADC_Monitor_Private_variables */



/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup ADC_Monitor_Private_fuctions ADC_Monitor Private Fuctions
 *  @{
 */
static void variable_initialize(void);
static TXZ_Result driver_initialize(void);
static TXZ_Result driver_finalize(void);
static TXZ_Result application_initialize(void);
static TXZ_Result application_finalize(void);
static void timer_interval_handler(uint32_t id);
extern void irq_adc_cp0 ( void ) ;
extern void irq_adc_cp1 ( void ) ;
/*--------------------------------------------------*/
/** 
  * @brief  Variable Initialize
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void variable_initialize(void)
{
    /* Instance. */
    instance.t32a.p_instance    = MAIN_NULL;
    instance.adc.p_instance     = MAIN_NULL;
    instance.p_uart             = MAIN_NULL;
    instance.timer.init.p_t32a  = MAIN_NULL;
    /* Processing mode. */
    proc = PROC_START;
    /* Information. */
    info.update = TXZ_ENABLE;  /* Init Output. */
    info.count  = 0;
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize
  * @param  -
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (p_t32a->p_instance != MAIN_NULL)
        {
            if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver initialize will process within USB UART. */
    /*----------------------*/
    /* ADC                  */
    /*----------------------*/
    {
        adc_t *p_adc = &instance.adc;

        /*--- Construct  ---*/
        /* Register Allocation */
        p_adc->p_instance       = TSB_ADA;
        /*--- Initialize ---*/
        p_adc->init.id          = (uint32_t)p_adc;
        p_adc->init.clk.exaz0    = ADC_SAMPLING_PERIOD0_X2N;
        p_adc->init.clk.exaz1    = ADC_SAMPLING_PERIOD1_X2N;
        p_adc->init.clk.vadcld  = ADC_SCLK_3;   /* less than 40MHz  120MHz / 3 = 40MHz */
        p_adc->init.mod1        = ADC_MOD1_AVDD5_3V_SCLK_15;
        p_adc->init.mod2        = ADC_MOD2_TMPM3Hz;
        p_adc->init.trm         = ADC_TRM_TMPM3Hz;

        p_adc->init.CMPxEN[0].CMPEN      = ADCMP0EN_DISABLE;
        p_adc->init.CMPxEN[1].CMPEN      = ADCMP1EN_DISABLE;        
        
        p_adc->p_instance->CMPCR0 = ADCMPCNT_1;
        p_adc->p_instance->CMPCR0 |= ADCMPCond_ACC;
        p_adc->p_instance->CMPCR0 |= ADCMPBigSml_Sml;
        p_adc->p_instance->CMPCR0 |= ADCMPStr_Reg0;
        p_adc->p_instance->CMP0   = CFG_ADC_CMP_25_SMALL_THAN   /*! If temp >= 25 c, Small than this value */

        p_adc->p_instance->CMPCR1 = ADCMPCNT_1;
        p_adc->p_instance->CMPCR1 |= ADCMPCond_ACC;
        p_adc->p_instance->CMPCR1 |= ADCMPBigSml_Big;
        p_adc->p_instance->CMPCR1 |= ADCMPStr_Reg0;
        p_adc->p_instance->CMP1   = CFG_ADC_CMP_35_BIG_THAN;    /*! If temp < 36 c,  Big than this value */
        p_adc->p_instance->CMPEN  = (ADCMP0EN_ENABLE | ADCMP1EN_ENABLE);      
        
        p_adc->handler.single          = MAIN_NULL;
        if (adc_init(p_adc) != TXZ_SUCCESS)
        {
            result = TXZ_ERROR;
        }
        NVIC_EnableIRQ(INTADACP0_IRQn);
        NVIC_EnableIRQ(INTADACP1_IRQn);   
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Finalize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* ADC                  */
    /*----------------------*/
    {
        adc_t *p_adc = &instance.adc;

        /*--- Finalize ---*/
        if (adc_deinit(p_adc) != TXZ_SUCCESS)
        {
            result = TXZ_ERROR;
        }
        /*--- Destruct ---*/
        /* Register Release */
        p_adc->p_instance = MAIN_NULL;
    }
    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver finalize has done within USB UART.    */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Finalize ---*/
        /* no processing */
        /*--- Destruct ---*/
        /* Register Release */
        p_t32a->p_instance = MAIN_NULL;
    }
    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver finalize will process within BSP.     */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize
  * @param  -
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* LED                  */
    /*----------------------*/
    {
        uint32_t i;
        led_t *p_led = MAIN_NULL;

        for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
        {
            p_led = &instance.led[i];

            p_led->init.id                  = (uint32_t)p_led;
            p_led->init.p_gpio              = bsp_get_gpio_instance_address();
            p_led->init.port.group          = bsp_get_gpio_group_led((BSPLed)(i));
            p_led->init.port.num            = bsp_get_gpio_num_led((BSPLed)(i));
            p_led->init.blink.func          = TXZ_DISABLE;
            p_led->init.blink.interval.on   = 0;
            p_led->init.blink.interval.off  = 0;
            p_led->init.state           = LED_STATE_OFF;
            led_initialize(p_led);
        }
    }
    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    {
        timer_t *p_timer = &instance.timer;

        p_timer->init.id       = (uint32_t)&instance.timer;
        p_timer->init.p_t32a   = &instance.t32a;
        p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
        p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
        p_timer->init.interval = 1000;
        p_timer->init.handler  = timer_interval_handler;
        timer_initialize(p_timer);
    }
    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = USER_ID_UART;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = bsp_get_usb_uart_tx_baudrate(BSP_USB_UART_0);
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result  = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Thermistors          */
    /*----------------------*/
    
    {
        adc_t *p_adc = &instance.adc;
        adc_channel_setting_t param;
        param.interrupt = ADC_INT_ENABLE;
        param.type = ADC_CONVERSION_SGL;
        param.ain = bsp_get_thermistors_adin(BSP_THERMISTORS_0);
        if (adc_channel_setting(p_adc, bsp_get_thermistors_ch(BSP_THERMISTORS_0), &param) != TXZ_SUCCESS)
        {
            result = TXZ_ERROR;
        }
    }
    
    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Finalize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Thermistors          */
    /*----------------------*/
    if (instance.adc.p_instance != MAIN_NULL)
    {
        if (adc_channel_clear(&instance.adc, bsp_get_thermistors_ch(BSP_THERMISTORS_0)) != TXZ_SUCCESS)
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    if (instance.p_uart != MAIN_NULL)
    {
        uart_io_finalize();
    }
    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    if (instance.timer.init.p_t32a != MAIN_NULL)
    {
        timer_finalize(&instance.timer);
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    if (id == (uint32_t)&instance.timer)
    {
        /* Increment Tick. Tick is used to driver's Timeout. */
        hal_inc_tick();
        /* Interval Time Count. */
        info.count++;
        if (info.count >= CFG_OUTPUT_INTERVAL)
        {
            /*>>> Critical information. */
            info.update = TXZ_ENABLE;
            /*<<< Critical information. */
            info.count  = 0;
        }
    }
}
/**
 *  @}
 */ /* End of group ADC_Monitor_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup ADC_Monitor_Exported_functions ADC_Monitor Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{

    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /*--- variable    ---*/
    variable_initialize();
    /*--- driver      ---*/
    if (driver_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    
    adc_t *p_adc = &instance.adc;
    p_adc->p_instance       = TSB_ADA;

    /*--- application ---*/
    if (application_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    
    p_adc->p_instance       = TSB_ADA;    
       
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(proc != PROC_ERROR)
    {
        switch(proc)
        {
        /*--- Start ---*/
        case PROC_START:
            {
                timer_start(&instance.timer);
                proc = PROC_INTERVAL_WAIT;
            }

            cmp0 = p_adc->p_instance->CMP0 ;
            cmp1 = p_adc->p_instance->CMP1 ;
            
            p_led[0] = &instance.led[0];
            p_led[1] = &instance.led[1];

            led_turn_off(p_led[0]);
            led_turn_off(p_led[1]);
            
            led_value[0] = 0 ;
            led_value[1] = 0 ;

            break;
        /*--- Stop ---*/
        case PROC_STOP:
            {
                if (adc_stop(&instance.adc) != TXZ_SUCCESS)
                {
                    /* no processing */
                }
                timer_stop(&instance.timer);
                proc = PROC_ERROR;
            }
            break;
        /*--- Interval Wait ---*/
        case PROC_INTERVAL_WAIT:
            {
                if (info.update == TXZ_ENABLE)
                {
                    /*>>> Critical information. */
                    info.update = TXZ_DISABLE;
                    /*<<< Critical information. */
                    /* Blocking Conversion Start */
                    if (adc_start(&instance.adc) == TXZ_SUCCESS)
                    {
                        proc = PROC_CONVERSION;
                    }
                    else
                    {
                        proc = PROC_STOP;
                    }
                }
            }
            break;
        /*--- Conversion ---*/
        case PROC_CONVERSION:
            {
                /* Conversion Wait */
                if (adc_poll_conversion(&instance.adc, CFG_ADC_WAIT_TIME) == TXZ_DONE)
                {
                    proc = PROC_OUTPUT;
                }
            }
            break;
        /*--- Output ---*/
        case PROC_OUTPUT:
            {

                /* Get conversion value. */
                proc = PROC_INTERVAL_WAIT;
                /* Get conversion value. */
                if (adc_channel_get_value(&instance.adc, bsp_get_thermistors_ch(BSP_THERMISTORS_0), &value) == TXZ_SUCCESS)
                {

                    /* Convert to celsius. */
                    if (bsp_get_thermistors_degC(BSP_THERMISTORS_0, value, &degC) == TXZ_SUCCESS)
                    {
                        /* Output information. */
                        if (printf("Convert Result Value[THERMISTOR]:0x%03x(%d degrees)\n", value, degC) < 0)
                        {
                            proc = PROC_STOP;
                        }

                        if (led_value[0]==1) {
                          led_turn_on(p_led[0]);      
                        } else {
                          led_turn_off(p_led[0]);      
                        }
                        if (led_value[1]==1) {
                          led_turn_on(p_led[1]);      
                        } else {
                          led_turn_off(p_led[1]);      
                        }

                    }
                    else
                    {
                        /* no processing */
                    }
                }
                else
                {
                    proc = PROC_STOP;
                }
            }
            break;
            
        /*--- Error ---*/
        case PROC_ERROR:
            /* no processing */
            break;
        /*--- Default ---*/
        default:
            proc = PROC_ERROR;
            break;
        }
    }
    /*>>> System Error. Please Debug. */
    /*----------------------*/
    /* Finalize             */
    /*----------------------*/
    /*--- application ---*/
    if (application_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*--- driver      ---*/
    if (driver_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*----------------------*/
    /* BSP finalize         */
    /*----------------------*/
    bsp_finalize();
#ifdef DEBUG
    assert_failed(MAIN_NULL, 0);
#endif
}
/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer.
  * @param  timer :Timer.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    default:
        /* no processing */
        break;
    }
}


/*--------------------------------------------------*/
/** 
  * @brief  ADC conversion monitoring0 IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_adc_cp0(void)
{
    
    value_shift = value << 4 ;
    if( value_shift <= cmp1 ) { 
      led_value[0] = 1 ;
    }
    led_value[1] = 1 ;
}

/*--------------------------------------------------*/
/** 
  * @brief  ADC conversion monitoring1 IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_adc_cp1(void)
{

    value_shift = value << 4 ;
    led_value[0] = 0 ;
    if(value_shift >= cmp0 ){ 
      led_value[1] = 0 ;
    }
  
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __nop();
    }
}
#endif
/**
 *  @}
 */ /* End of group ADC_Monitor_Exported_functions */

/**
 *  @}
 */ /* End of group ADC_Monitor */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
