/** 
 *******************************************************************************
 * @file    main.c
 * @brief   CG_EXINT Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_sw.h"
#include "bsp_led.h"
#include "bsp_timer.h"
#include "siwdt.h"
/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup CG_EXINT CG_EXINT Sample Appli
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup CG_EXINT_Private_macro CG_EXINT Private Macro
 *  @{
 */



/** 
 *  @}
 */ /* End of group CG_EXINT_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup CG_EXINT_Private_define CG_EXINT Private Define
 *  @{
 */
/** 
 *  @defgroup LEDBlinkConfiguration LED Blink Time Configuration
 *  @brief    LED Blink Interval Time(ms).
 *  @{
 */
#define CFG_LED_BLINK_TIME      ((uint32_t)(1000))
/** 
 *  @}
 */ /* End of group LEDBlinkConfiguration */

/** 
 *  @defgroup SWPollingConfiguration SW Polling Time Configuration
 *  @brief    SW Polling Time(ms).
 *  @{
 */
#define CFG_SW_POLLING_TIME     ((uint32_t)(100))
/** 
 *  @}
 */ /* End of group SWPollingConfiguration */

/** 
 *  @defgroup IHOSCConfiguration IHOSC Configuration
 *  @brief    IHOSC Configuration.
 *  @{
 */
#define CFG_IHOSC_WARM_UP_TIME  ((uint32_t)(5000))      /*!< Warm up time(us).  Minimam warm-up time is 100(us). */
#define CFG_IHOSC_CLOCK         ((uint32_t)(10000000))  /*!< Clock(hz).        */
/** 
 *  @}
 */ /* End of group IHOSCConfiguration */

/** 
 *  @}
 */ /* End of group CG_EXINT_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup CG_EXINT_Private_define CG_EXINT Private Define
 *  @{
 */

#define MAIN_NULL               ((void *)0) /*!< Null Pointer */

/** 
 *  @defgroup  SIWDxEN  SIWDxEN Register
 *  @brief     WD Enable Register.
 *  @details   Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 31:2  | -          |
 *  | 1     | WDTF       |
 *  | 0     | WDTE       |
 *  @{
 */
/* WDTE */
#define SIWDxEN_WDTE_RW_DISABLE         ((uint32_t)0x00000000)    /*!< WDTE  :[R/W] :Disable    */
#define SIWDxEN_WDTE_RW_ENABLE          ((uint32_t)0x00000001)    /*!< WDTE  :[R/W] :Enable     */
/** 
 *  @}
 */ /* End of group SIWDxEN */

/** 
 *  @defgroup  SIWDxCR  SIWDxCR Register
 *  @brief     WD Control Register.
 *  @details   Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 31:8  | -          |
 *  | 7:0   | WDCR       |
 *  @{
 */
/* WDCR */
#define SIWDxCR_WDCR_W_DISABLE          ((uint32_t)0x000000B1)    /*!< WDCR  :[W] :Disable      */
#define SIWDxCR_WDCR_W_CLEAR            ((uint32_t)0x0000004E)    /*!< WDCR  :[W] :Clear        */
/** 
 *  @}
 */ /* End of group SIWDxCR */

/** 
 *  @defgroup  CGOSCCR  CGOSCCR Register
 *  @brief     CG Control Register.
 *  @details   Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 31:20 | -          |
 *  | 19    | IHOSC2F    |
 *  | 18:17 | -          |
 *  | 16    | IHOSC1F    |
 *  | 15:10 | -          |
 *  | 9     | OSCF       |
 *  | 8     | OSCSEL     |
 *  | 7:4   | -          |
 *  | 3     | IHOSC2EN   |
 *  | 2:1   | EOSCEN     |
 *  | 0     | IHOSC1EN   |
 *  @{
 */
/* IHOSC2F  */
#define CGOSCCR_IHOSC2F_MASK            ((uint32_t)0x00080000)    /*!< IHOSC2F   :Mask                  */
#define CGOSCCR_IHOSC2F_R_STOP          ((uint32_t)0x00000000)    /*!< IHOSC2F   :[R]   :Stop           */
#define CGOSCCR_IHOSC2F_R_RUNNING       ((uint32_t)0x00080000)    /*!< IHOSC2F   :[R]   :Running        */
/* OSCF     */
#define CGOSCCR_OSCF_MASK               ((uint32_t)0x00000200)    /*!< OSCF      :Mask                  */
#define CGOSCCR_OSCF_R_IHOSC            ((uint32_t)0x00000000)    /*!< OSCF      :[R]   :Internal HOSC  */
#define CGOSCCR_OSCF_R_EHOSC            ((uint32_t)0x00000200)    /*!< OSCF      :[R]   :External HOSC  */
/* OSCSEL   */
#define CGOSCCR_OSCSEL_MASK             ((uint32_t)0x00000100)    /*!< OSCSEL    :Mask                  */
#define CGOSCCR_OSCSEL_RW_IHOSC         ((uint32_t)0x00000000)    /*!< OSCSEL    :[R/W] :Internal HOSC  */
#define CGOSCCR_OSCSEL_RW_EHOSC         ((uint32_t)0x00000100)    /*!< OSCSEL    :[R/W] :External HOSC  */
/* IHOSC2EN */
#define CGOSCCR_IHOSC2EN_MASK           ((uint32_t)0x00000008)    /*!< IHOSC2EN  :Mask                  */
#define CGOSCCR_IHOSC2EN_RW_DISABLE     ((uint32_t)0x00000000)    /*!< IHOSC2EN  :[R/W] :Disable        */
#define CGOSCCR_IHOSC2EN_RW_ENABLE      ((uint32_t)0x00000008)    /*!< IHOSC2EN  :[R/W] :Enable         */
/* EOSCEN   */
#define CGOSCCR_EOSCEN_MASK             ((uint32_t)0x00000006)    /*!< EOSCEN    :Mask                  */
#define CGOSCCR_EOSCEN_RW_UNUSE         ((uint32_t)0x00000000)    /*!< EOSCEN    :[R/W] :Unuse          */
#define CGOSCCR_EOSCEN_RW_EHOSC         ((uint32_t)0x00000002)    /*!< EOSCEN    :[R/W] :External HOSC  */
#define CGOSCCR_EOSCEN_RW_ECLK          ((uint32_t)0x00000004)    /*!< EOSCEN    :[R/W] :External Clock */
/* IHOSC1EN */
#define CGOSCCR_IHOSC1EN_MASK           ((uint32_t)0x00000001)    /*!< IHOSC1EN  :Mask                  */
#define CGOSCCR_IHOSC1EN_RW_DISABLE     ((uint32_t)0x00000000)    /*!< IHOSC1EN  :[R/W] :Disable        */
#define CGOSCCR_IHOSC1EN_RW_ENABLE      ((uint32_t)0x00000001)    /*!< IHOSC1EN  :[R/W] :Enable         */
/** 
 *  @}
 */ /* End of group CGOSCCR */

/** 
 *  @defgroup  CGSTBYCR  CGSTBYCR Register
 *  @brief     Standby Control Register.
 *  @details   Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 31:2  | -          |
 *  | 1:0   | STBY       |
 *  @{
 */
/* STBY */
#define CGSTBYCR_STBY_MASK              ((uint32_t)0x00000003)    /*!< STBY :[R/W] :Mask   */
#define CGSTBYCR_STBY_RW_IDLE           ((uint32_t)0x00000000)    /*!< STBY :[R/W] :IDLE   */
#define CGSTBYCR_STBY_RW_STOP1          ((uint32_t)0x00000001)    /*!< STBY :[R/W] :STOP1  */
#define CGSTBYCR_STBY_RW_STOP2          ((uint32_t)0x00000002)    /*!< STBY :[R/W] :STOP2  */
/** 
 *  @}
 */ /* End of group CGSTBYCR */

/** 
 *  @defgroup  CGPLL0SEL  CGPLL0SEL Register
 *  @brief     fsys PLL Select Register.
 *  @details   Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 31:8  | PLL0SET    |
 *  | 7:3   | -          |
 *  | 2     | PLL0ST     |
 *  | 1     | PLL0SEL    |
 *  | 0     | PLL0ON     |
 *  @{
 */
/* PLL0ST  */
#define CGPLL0SEL_PLL0ST_MASK           ((uint32_t)0x00000004)    /*!< PLL0ST  :Mask.           */
#define CGPLL0SEL_PLL0ST_R_FOSC         ((uint32_t)0x00000000)    /*!< PLL0ST  :[R]     :fosc   */
#define CGPLL0SEL_PLL0ST_R_FPLL         ((uint32_t)0x00000004)    /*!< PLL0ST  :[R]     :fpll   */
/* PLL0SEL */
#define CGPLL0SEL_PLL0SEL_MASK          ((uint32_t)0x00000002)    /*!< PLL0SEL :Mask.           */
#define CGPLL0SEL_PLL0SEL_RW_FOSC       ((uint32_t)0x00000000)    /*!< PLL0SEL :[R/W]   :fosc   */
#define CGPLL0SEL_PLL0SEL_RW_FPLL       ((uint32_t)0x00000002)    /*!< PLL0SEL :[R/W]   :fpll   */
/* PLL0ON  */
#define CGPLL0SEL_PLL0ON_MASK           ((uint32_t)0x00000001)    /*!< PLL0ON  :Mask.           */
#define CGPLL0SEL_PLL0ON_RW_DISABLE     ((uint32_t)0x00000000)    /*!< PLL0ON  :[R/W]   :Disable*/
#define CGPLL0SEL_PLL0ON_RW_ENABLE      ((uint32_t)0x00000001)    /*!< PLL0ON  :[R/W]   :Enable */
/** 
 *  @}
 */ /* End of group CGPLL0SEL */

/** 
 *  @defgroup  CGWUPHCR  CGWUPHCR Register
 *  @brief     HOSC warming up Register.
 *  @details   Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 31:20 | WUPT       |
 *  | 19:16 | WUPT       |
 *  | 15:9  | -          |
 *  | 8     | WUCLK      |
 *  | 7:2   | -          |
 *  | 1     | WUEF       |
 *  | 0     | WUON       |
 *  @{
 */
/* WUPT  */
#define CGWUPHCR_WUPT_HIGH_MASK         ((uint32_t)0xFFF00000)    /*!< WUPT    :High Bit Mask.  */
/* WUCLK */
#define CGWUPHCR_WUCLK_MASK             ((uint32_t)0x00000100)    /*!< WUCLK   :Mask.           */
#define CGWUPHCR_WUCLK_RW_IHOSC         ((uint32_t)0x00000000)    /*!< WUCLK   :[R/W]  :IHOSC   */
#define CGWUPHCR_WUCLK_RW_EHOSC         ((uint32_t)0x00000100)    /*!< WUCLK   :[R/W]  :EHOSC   */
/* WUEF  */
#define CGWUPHCR_WUEF_MASK              ((uint32_t)0x00000002)    /*!< WUEF    :Mask.           */
#define CGWUPHCR_WUEF_R_DONE            ((uint32_t)0x00000000)    /*!< WUEF    :[R]    :Done    */
#define CGWUPHCR_WUEF_R_RUNNING         ((uint32_t)0x00000002)    /*!< WUEF    :[R]    :Running */
/** 
 *  @}
 */ /* End of group CGWUPHCR */

/** 
 *  @defgroup  CGWUPLCR  CGWUPLCR Register
 *  @brief     LOSC warming up Register.
 *  @details   Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 31:27 | -          |
 *  | 26:12 | WUPTL      |
 *  | 11:8  | WUPTL      |
 *  | 7:2   | -          |
 *  | 1     | WULEF      |
 *  | 0     | WULON      |
 *  @{
 */
/* WUPTL  */
#define CGWUPLCR_WUPTL_HIGH_MASK        ((uint32_t)0x07FFF000)    /*!< WUPTL   :High Bit Mask.  */
/* WULEF  */
#define CGWUPLCR_WULEF_MASK             ((uint32_t)0x00000002)    /*!< WULEF   :Mask.           */
#define CGWUPLCR_WULEF_R_DONE           ((uint32_t)0x00000000)    /*!< WULEF   :[R]  :Done      */
#define CGWUPLCR_WULEF_R_RUNNING        ((uint32_t)0x00000002)    /*!< WULEF   :[R]  :Running   */
/* WULON  */
#define CGWUPLCR_WULON_MASK             ((uint32_t)0x00000001)    /*!< WULON   :Mask.           */
#define CGWUPLCR_WULON_W_ENABLE         ((uint32_t)0x00000001)    /*!< WULON   :[W]  :Enable     */
/** 
 *  @}
 */ /* End of group CGWUPLCR */

/** 
 *  @defgroup  RLMRSTFLG0  RLMRSTFLG0 Register
 *  @brief     Reset Flag Register 0.
 *  @details   Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 7:6   | -          |
 *  | 5     | LVDRSTF    |
 *  | 4     | STOP2RSTF  |
 *  | 3     | PINRSTF    |
 *  | 2:1   | -          |
 *  | 0     | PORSTF     |
 *  @{
 */
/* ALL */
#define RLMRSTFLG0_W_CLEAR              ((uint8_t)0x00)           /*!< ALL BITS  :[W]  :Clear  */
/* STOP2RSTF */
#define RLMRSTFLG0_STOP2RSTF_MASK       ((uint8_t)0x10)           /*!< STOP2RSTF :Mask.        */
#define RLMRSTFLG0_STOP2RSTF_R_RESET    ((uint8_t)0x10)           /*!< STOP2RSTF :[R]  :Reset  */
/* PINRSTF */
#define RLMRSTFLG0_PINRSTF_MASK         ((uint8_t)0x08)           /*!< PINRSTF :Mask.        */
/** 
 *  @}
 */ /* End of group RLMRSTFLG0 */

/** 
 *  @defgroup  RLMRSTFLG1  RLMRSTFLG1 Register
 *  @brief     Reset Flag Register 1.
 *  @details   Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 7:4   | -          |
 *  | 3     | OFDRSTF    |
 *  | 2     | WDTRSTF    |
 *  | 1     | LOCKRSTF   |
 *  | 0     | SYSRSTF    |
 *  @{
 */
/* ALL */
#define RLMRSTFLG1_W_CLEAR              ((uint8_t)0x00)           /*!< ALL BITS  :[W]  :Clear  */
/** 
 *  @}
 */ /* End of group RLMRSTFLG1 */

/** 
 *  @defgroup  RLMSHTDNOP  RLMSHTDNOP Register
 *  @brief     Shut Down Control Register.
 *  @details   Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 31:1  | -          |
 *  | 0     | PTKEEP     |
 *  @{
 */
/* PTKEEP */
#define RLMSHTDNOP_PTKEEP_RW_RELEASE    ((uint8_t)0x00)    /*!< PTKEEP :[R/W] :Release  */
#define RLMSHTDNOP_PTKEEP_RW_KEEP       ((uint8_t)0x01)    /*!< PTKEEP :[R/W] :Keep     */
/** 
 *  @}
 */ /* End of group RLMSHTDNOP */

/** 
 *  @defgroup  FCSR0  FCSR0 Register
 *  @brief     Flash Status Register 0.
 *  @details   Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 31:11 | -          |
 *  | 10    | RDYBSY2    |
 *  | 9     | -          |
 *  | 8     | RDYBSY0    |
 *  | 7:1   | -          |
 *  | 0     | RDYBSY     |
 *  @{
 */
/* RDYBSY */
#define FCSR0_RDYBSY_MASK               ((uint32_t)0x00000001)    /*!< RDYBSY :Mask            */
#define FCSR0_RDYBSY_R_RUNNING          ((uint32_t)0x00000000)    /*!< RDYBSY :[R] :Running    */
#define FCSR0_RDYBSY_R_DONE             ((uint32_t)0x00000001)    /*!< RDYBSY :[R] :Done       */
/** 
 *  @}
 */ /* End of group FCSR0 */


/** 
 *  @}
 */ /* End of group CG_EXINT_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup CG_EXINT_Private_define CG_EXINT Private Define
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_IDLE,              /*!< Idle.                      */
    PROC_NORMAL_START,      /*!< Normal Mode    :START.     */
    PROC_NORMAL,            /*!< Normal Mode    :RUNNING.   */
    PROC_NORMAL_END,        /*!< Normal Mode    :END.       */
    PROC_LOW_POWER_START,   /*!< Low Power Mode :START.     */
    PROC_LOW_POWER_END,     /*!< Low Power Mode :END.       */
    PROC_ERROR              /*!< Error.                     */
} Processing;

/*----------------------------------*/
/** 
 * @brief  Low Power Mode.
*/
/*----------------------------------*/
typedef enum
{
    LOW_POWER_IDLE,         /*!< Idle Mode.     */
    LOW_POWER_STOP_1,       /*!< Stop1 Mode.    */
    LOW_POWER_STOP_2,       /*!< Stop2 Mode.    */
} LowPowerMode;
/** 
 *  @}
 */ /* End of group CG_EXINT_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup CG_EXINT_Private_typedef CG_EXINT Private Typedef
 *  @{
 */

/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
   t32a_t      t32a;                       /*!< Driver      :T32A               */
   timer_t     timer;                      /*!< Application :Timer (1ms timer). */
   sw_t        sw[(uint32_t)BSP_PSW_MAX];  /*!< Application :Push SW            */
   led_t       led[(uint32_t)BSP_LED_MAX]; /*!< Application :LED                */
   rlm_t       rlm;                        /*!< Driver      :RLM.      */
   cg_t        cg;                         /*!< Driver      :CG.       */
} instance_t;

/**
 *  @}
 */ /* End of group CG_EXINT_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup CG_EXINT_Private_variables CG_EXINT Private Variables
 *  @{
 */
static instance_t   instance;  /*!< Instance.                    */
static Processing   proc;      /*!< Processing mode.             */
static LowPowerMode low;       /*!< Select Low Power Mode.       */
static rlm_t        *p_rlm ;   /*!< RLM */
static cg_t         *p_cg ;    /*!< CG */
/**
 *  @}
 */ /* End of group CG_EXINT_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup CG_EXINT_Private_fuctions CG_EXINT Private Fuctions
 *  @{
 */

static void pll_disable(void);
static void internal_hosc_enable(void);
static void variable_initialize(void);
static void timer_interval_handler(uint32_t id);
static void sw_state_change_handler(uint32_t id, gpio_pinstate_t state);
static TXZ_Result driver_initialize_normal(void);
static TXZ_Result driver_finalize_normal(void);
static TXZ_Result application_initialize_normal(void);
static TXZ_Result application_finalize_normal(void);
static TXZ_Result normal_start(void);
static TXZ_Result normal_end(void);
static TXZ_Result driver_initialize_idle(void);
static TXZ_Result driver_finalize_idle(void);
static TXZ_Result application_initialize_idle(void);
static TXZ_Result application_finalize_idle(void);
static TXZ_Result low_power_start_idle(void);
static TXZ_Result low_power_end_idle(void);
static TXZ_Result driver_initialize_stop1(void);
static TXZ_Result driver_finalize_stop1(void);
static TXZ_Result application_initialize_stop1(void);
static TXZ_Result application_finalize_stop1(void);
static TXZ_Result low_power_start_stop1(void);
static TXZ_Result low_power_end_stop1(void);
static TXZ_Result driver_initialize_stop2(void);
static TXZ_Result application_initialize_stop2(void);
static TXZ_Result low_power_start_stop2(void);















/*--------------------------------------------------*/
/** 
  * @brief  PLL Disable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void pll_disable(void)
{
    uint32_t work;

    /* [CGPLL0SEL]<PLL0SEL>: fosc           */
    work = (uint32_t)(TSB_CG->PLL0SEL & ~CGPLL0SEL_PLL0SEL_MASK);
    TSB_CG->PLL0SEL = (uint32_t)(work | CGPLL0SEL_PLL0SEL_RW_FOSC);
    /* [CGPLL0SEL]<PLL0ST> : wait for fosc  */
    while ((TSB_CG->PLL0SEL & CGPLL0SEL_PLL0ST_MASK) != CGPLL0SEL_PLL0ST_R_FOSC) 
    {
        /* no processing */
    }
    /* [CGPLL0SEL]<PLL0ON> : PLL Disable    */
    work = (uint32_t)(TSB_CG->PLL0SEL & ~CGPLL0SEL_PLL0ON_MASK);
    TSB_CG->PLL0SEL = (uint32_t)(work | CGPLL0SEL_PLL0ON_RW_DISABLE);
}

/*--------------------------------------------------*/
/** 
  * @brief  Internal HOSC Enable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void internal_hosc_enable(void)
{
    uint32_t work;

    /* [CGOSCCR]<IHOSC1EN> : HOSC Enable    */
    work = (uint32_t)(TSB_CG->OSCCR & ~CGOSCCR_IHOSC1EN_MASK);
    TSB_CG->OSCCR = (uint32_t)(work | CGOSCCR_IHOSC1EN_RW_ENABLE);
    /* [CGWUPHCR]<WUON> : Warm up Start    */
    TSB_CG_WUPHCR_WUON = 1U;
    /* [CGWUPHCR]<WUEF>  : Read(wait for done) */
    while( TSB_CG_WUPHCR_WUEF != CGWUPHCR_WUEF_R_DONE )
    {
        /* no processing */
    }
    /* [CGOSCCR]<OSCSEL>   : IHOSC          */
    work = (uint32_t)(TSB_CG->OSCCR & ~CGOSCCR_OSCSEL_MASK);
    TSB_CG->OSCCR = (uint32_t)(work | CGOSCCR_OSCSEL_RW_IHOSC);
    /* [CGOSCCR]<OSCF>     : Read(wait for IHOSC) */
    while ((TSB_CG->OSCCR & CGOSCCR_OSCF_MASK) != CGOSCCR_OSCF_R_IHOSC) 
    {
        /* no processing */
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Variable Initialize
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void variable_initialize(void)
{
    /* Instance. */
    instance.t32a.p_instance    = MAIN_NULL;
    instance.timer.init.p_t32a  = MAIN_NULL;
    {
        uint32_t i;

        for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
        {
            instance.sw[i].init.p_gpio  = MAIN_NULL;
        }
        for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
        {
            instance.led[i].init.p_gpio = MAIN_NULL;
        }
    }
    /* Processing mode. */
    proc = PROC_NORMAL_START;
}

/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    if (id == (uint32_t)&instance.timer)
    {
        uint32_t i;

        /*----------------------*/
        /* Tick                 */
        /*----------------------*/
        /* Increment Tick. Tick is used to driver's Timeout. */
        hal_inc_tick();
        /*----------------------*/
        /* SW                   */
        /*----------------------*/
        for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
        {
            sw_1ms_timer_handler(&instance.sw[i]);
        }
        /*----------------------*/
        /* LED                  */
        /*----------------------*/
        for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
        {
            led_1ms_timer_handler(&instance.led[i]);
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  SW State Change Handler.
  * @param  id    :User ID
  * @param  state :Push SW Port State
  * @return -
  * @retval -
  * @note   Notify that, "State is changed." \n
  *         Call by SW Task \n
  *         If SW is pushed, change the processing mode(@ref Processing).
  */
/*--------------------------------------------------*/
static void sw_state_change_handler(uint32_t id, gpio_pinstate_t state)
{
    /* Check SW Push(= GPIO_PIN_RESET) */
    if (state == GPIO_PIN_RESET)
    {
        uint32_t i;

        for (i=0; (i < (uint32_t)BSP_PSW_MAX) && (proc != PROC_NORMAL_END); i++)
        {
            if (id == (uint32_t)&instance.sw[i])
            {
                /* Select Low Power Mode */
                switch ((BSPPsw)i)
                {
                case BSP_PSW_1:
                    low  = LOW_POWER_IDLE;
                    proc = PROC_NORMAL_END;
                    break;
                case BSP_PSW_2:
                    low = LOW_POWER_STOP_1;
                    proc = PROC_NORMAL_END;
                    break;
                case BSP_PSW_3:
                    low = LOW_POWER_STOP_2;
                    proc = PROC_NORMAL_END;
                    break;
                default:
                    /* no processing */
                    break;
                }
            }
        }
    }
}

/*==========================*/
/* Normal Mode              */
/*==========================*/
/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for Normal Mode
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize_normal(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
        {
            result = TXZ_ERROR;
        }
    }

    {
        p_rlm = &instance.rlm;
        p_cg  = &instance.cg;

        p_rlm->p_instance = TSB_RLM;
        p_cg->p_instance = TSB_CG;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Finalize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for Normal Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result driver_finalize_normal(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Finalize ---*/
        /* no processing */
        /*--- Destruct ---*/
        /* Register Release */
        p_t32a->p_instance = MAIN_NULL;
    }
    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver finalize will process within BSP.     */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for Normal Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize_normal(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    {
        timer_t *p_timer = &instance.timer;

        p_timer->init.id       = (uint32_t)&instance.timer;
        p_timer->init.p_t32a   = &instance.t32a;
        p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
        p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
        p_timer->init.interval = 1000;
        p_timer->init.handler  = timer_interval_handler;
        timer_initialize(p_timer);
    }
    /*----------------------*/
    /* LED                  */
    /*----------------------*/
    {
        uint32_t i;
        led_t *p_led = MAIN_NULL;

        for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
        {
            p_led = &instance.led[i];

            p_led->init.id          = (uint32_t)p_led;
            p_led->init.p_gpio      = bsp_get_gpio_instance_address();
            p_led->init.port.group  = bsp_get_gpio_group_led((BSPLed)(i));
            p_led->init.port.num    = bsp_get_gpio_num_led((BSPLed)(i));
            /* LED Normal :blinking */
            if (i == (uint32_t)BSP_LED_1)
            {
                p_led->init.state               = LED_STATE_ON;
                p_led->init.blink.func          = TXZ_ENABLE;
                p_led->init.blink.interval.on   = CFG_LED_BLINK_TIME;
                p_led->init.blink.interval.off  = CFG_LED_BLINK_TIME;
            }
            /* default    :off      */
            else
            {
                p_led->init.state               = LED_STATE_OFF;
                p_led->init.blink.func          = TXZ_DISABLE;
                p_led->init.blink.interval.on   = 0;
                p_led->init.blink.interval.off  = 0;
            }
            led_initialize(p_led);
        }
    }
    /*----------------------*/
    /* PSW                  */
    /*----------------------*/
    {
        uint32_t i;
        sw_t *p_sw = MAIN_NULL;

        for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
        {
            p_sw = &instance.sw[i];

            p_sw->init.id                   = (uint32_t)p_sw;
            p_sw->init.p_gpio               = bsp_get_gpio_instance_address();
            p_sw->init.port.group           = bsp_get_gpio_group_psw((BSPPsw)(i));
            p_sw->init.port.num             = bsp_get_gpio_num_psw((BSPPsw)(i));
            p_sw->init.polling.func         = TXZ_ENABLE;
            p_sw->init.polling.interval     = CFG_SW_POLLING_TIME;
            p_sw->init.chattering.func      = TXZ_DISABLE;
            p_sw->init.chattering.count     = 0;
            p_sw->init.handler              = sw_state_change_handler;
            sw_initialize(p_sw);
        }
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Finalize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for Normal Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result application_finalize_normal(void)
{
    TXZ_Result result = TXZ_SUCCESS;
    uint32_t i;

    /*----------------------*/
    /* PSW                  */
    /*----------------------*/
    for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
    {
        sw_finalize(&instance.sw[i]);
    }
    /*----------------------*/
    /* LED                  */
    /*----------------------*/
    for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
    {
        led_finalize(&instance.led[i]);
    }
    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    timer_finalize(&instance.timer);

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Start Processing.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for Normal Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result normal_start(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Driver               */
    /*----------------------*/
    if (driver_initialize_normal() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*----------------------*/
    /* Application          */
    /*----------------------*/
    if (application_initialize_normal() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*----------------------*/
    /* 1ms Timer Start      */
    /*----------------------*/
    timer_start(&instance.timer);

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  End Processing.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for Normal Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result normal_end(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* 1ms Timer Stop       */
    /*----------------------*/
    timer_stop(&instance.timer);
    /*----------------------*/
    /* Application          */
    /*----------------------*/
    if (application_finalize_normal() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*----------------------*/
    /* Driver               */
    /*----------------------*/
    if (driver_finalize_normal() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }

    return (result);
}

/*==========================*/
/* IDLE Mode                */
/*==========================*/
/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for IDLE Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize_idle(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Finalize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for IDLE Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result driver_finalize_idle(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver finalize will process within BSP.     */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for IDLE Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize_idle(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* LED                  */
    /*----------------------*/
    {
        uint32_t i;
        led_t *p_led = MAIN_NULL;

        for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
        {
            p_led = &instance.led[i];

            p_led->init.id                  = (uint32_t)p_led;
            p_led->init.p_gpio              = bsp_get_gpio_instance_address();
            p_led->init.port.group          = bsp_get_gpio_group_led((BSPLed)(i));
            p_led->init.port.num            = bsp_get_gpio_num_led((BSPLed)(i));
            p_led->init.blink.func          = TXZ_DISABLE;
            p_led->init.blink.interval.on   = 0;
            p_led->init.blink.interval.off  = 0;
            /* LED IDLE :on  */
            if (i == (uint32_t)BSP_LED_2)
            {
                p_led->init.state           = LED_STATE_ON;
            }
            /* default  :off */
            else
            {
                p_led->init.state           = LED_STATE_OFF;
            }
            led_initialize(p_led);
        }
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Finalize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for IDLE Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result application_finalize_idle(void)
{
    TXZ_Result result = TXZ_SUCCESS;
    uint32_t i;

    /*----------------------*/
    /* LED                  */
    /*----------------------*/
    for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
    {
        led_finalize(&instance.led[i]);
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Start Processing.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for IDLE Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result low_power_start_idle(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* WDT Disable          */
    /*----------------------*/
    wdt_disable(TSB_SIWD0);
    /*----------------------*/
    /* Flash Wait           */
    /*----------------------*/
    /* [FCSR0]<RDYBSY>   : Read(wait for done) */
    while((TSB_FC->SR0 & FCSR0_RDYBSY_MASK) != FCSR0_RDYBSY_R_DONE)
    {
        /* no processing */
    }
    /*----------------------*/
    /* Set STBY             */
    /*----------------------*/
    /* [CGSTBYCR]<STBY>    :IDLE            */
    TSB_CG->STBYCR = CGSTBYCR_STBY_RW_IDLE;
    /*----------------------*/
    /* STBY Wait            */
    /*----------------------*/
    /* [CGSTBYCR]<STBY>   : Read(wait for done) */
    while ((TSB_CG->STBYCR & CGSTBYCR_STBY_MASK) != CGSTBYCR_STBY_RW_IDLE)
    {
        /* no processing */
    }
    /*----------------------*/
    /* Driver               */
    /*----------------------*/
    if (driver_initialize_idle() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*----------------------*/
    /* Application          */
    /*----------------------*/
    if (application_initialize_idle() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*---------------------------*/
    /* External Interrupt Enable */
    /*---------------------------*/
    /* Exint Enable        */
    bsp_irq_exint_enable();
    /* NVIC Enable */
    NVIC_EnableIRQ(bsp_get_exint_nvic());

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  End Processing.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for IDLE Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result low_power_end_idle(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------------*/
    /* External Interrupt Disable */
    /*----------------------------*/
    /* NVIC Disable         */
    NVIC_DisableIRQ(bsp_get_exint_nvic());
    /* IRQ Disable          */
    bsp_irq_exint_disable();
    /*----------------------*/
    /* Application          */
    /*----------------------*/
    if (application_finalize_idle() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*----------------------*/
    /* Driver               */
    /*----------------------*/
    if (driver_finalize_idle() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }

    return (result);
}

/*==========================*/
/* STOP1 Mode               */
/*==========================*/
/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for STOP1 Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize_stop1(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Finalize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for STOP1 Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result driver_finalize_stop1(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver finalize will process within BSP.     */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for STOP1 Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize_stop1(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* LED                  */
    /*----------------------*/
    {
        uint32_t i;
        led_t *p_led = MAIN_NULL;

        for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
        {
            p_led = &instance.led[i];

            p_led->init.id                  = (uint32_t)p_led;
            p_led->init.p_gpio              = bsp_get_gpio_instance_address();
            p_led->init.port.group          = bsp_get_gpio_group_led((BSPLed)(i));
            p_led->init.port.num            = bsp_get_gpio_num_led((BSPLed)(i));
            p_led->init.blink.func          = TXZ_DISABLE;
            p_led->init.blink.interval.on   = 0;
            p_led->init.blink.interval.off  = 0;
            /* LED STOP1 :on  */
            if (i == (uint32_t)BSP_LED_3)
            {
                p_led->init.state           = LED_STATE_ON;
            }
            /* default   :off */
            else
            {
                p_led->init.state           = LED_STATE_OFF;
            }
            led_initialize(p_led);
        }
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Finalize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for STOP1 Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result application_finalize_stop1(void)
{
    TXZ_Result result = TXZ_SUCCESS;
    uint32_t i;

    /*----------------------*/
    /* LED                  */
    /*----------------------*/
    for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
    {
        led_finalize(&instance.led[i]);
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Low Power Mode Start Processing.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   STOP1 Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result low_power_start_stop1(void)
{
    TXZ_Result result = TXZ_SUCCESS;
    uint32_t work;

    /*----------------------*/
    /* WDT Disable          */
    /*----------------------*/
    wdt_disable(TSB_SIWD0);
    /*----------------------*/
    /* Flash Wait           */
    /*----------------------*/
    /* [FCSR0]<RDYBSY>   : Read(wait for done) */
    while ((TSB_FC->SR0 & FCSR0_RDYBSY_MASK) != FCSR0_RDYBSY_R_DONE)
    {
        /* no processing */
    }
    /*----------------------*/
    /* HOSC Warm-up Wait    */
    /*----------------------*/
    /* [CGWUPHCR]<WUEF>  : Read(wait for done) */
    while ((TSB_CG->WUPHCR & CGWUPHCR_WUEF_MASK) != CGWUPHCR_WUEF_R_DONE)
    {
        /* no processing */
    }
    /*----------------------*/
    /* Internal HOSC Select */
    /*----------------------*/
    /* [CGWUPHCR]<WUCLK> : IHOSC */
    work = (uint32_t)(TSB_CG->WUPHCR & ~CGWUPHCR_WUCLK_MASK);
    TSB_CG->WUPHCR = (uint32_t)(work | CGWUPHCR_WUCLK_RW_IHOSC);
    /*----------------------*/
    /* Internal HOSC Warm-up Time Set*/
    /*----------------------*/
    cg_set_warm_up_time(p_cg, CG_WARM_UP_SRC_IHOSC1, WU_TIME_INT);
    /*----------------------*/
    /* Set STBY             */
    /*----------------------*/
    /* [CGSTBYCR]<STBY>    :STOP1           */
    TSB_CG->STBYCR = CGSTBYCR_STBY_RW_STOP1;
    /*----------------------*/
    /* PLL Disable          */
    /*----------------------*/
    pll_disable();
    /*----------------------*/
    /* Internal HOSC Enable */
    /*----------------------*/
    internal_hosc_enable();
    /*----------------------*/
    /* External HOSC Disable*/
    /*----------------------*/
    /* [CGOSCCR]<EOSCEN>   : EHOSC Disable  */
    work = (uint32_t)(TSB_CG->OSCCR & ~CGOSCCR_EOSCEN_MASK);
    TSB_CG->OSCCR = (uint32_t)(work | CGOSCCR_EOSCEN_RW_UNUSE);
    /*----------------------*/
    /* OFD Internal HOSC Disable*/
    /*----------------------*/
    /* [CGOSCCR]<IHOSC2EN> : OFD IHOSC Disable  */
    work = (uint32_t)(TSB_CG->OSCCR & ~CGOSCCR_IHOSC2EN_MASK);
    TSB_CG->OSCCR = (uint32_t)(work | CGOSCCR_IHOSC2EN_RW_DISABLE);
    /*----------------------*/
    /* External HOSC Disable Check*/
    /*----------------------*/
    /* [CGOSCCR]<EOSCEN>   : Read  */
    while ((TSB_CG->OSCCR & CGOSCCR_EOSCEN_MASK) != CGOSCCR_EOSCEN_RW_UNUSE) 
    {
        /* no processing */
    }
    /*----------------------*/
    /* OFD Internal HOSC Disable Wait*/
    /*----------------------*/
    /* [CGOSCCR]<IHOSC2F>  : Read(wait for OFD IHOSC Disable)  */
    while ((TSB_CG->OSCCR & CGOSCCR_IHOSC2F_MASK) != CGOSCCR_IHOSC2F_R_STOP) 
    {
        /* no processing */
    }
    /*----------------------*/
    /* Driver               */
    /*----------------------*/
    if (driver_initialize_stop1() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*----------------------*/
    /* Application          */
    /*----------------------*/
    if (application_initialize_stop1() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*---------------------------*/
    /* External Interrupt Enable */
    /*---------------------------*/
    /* Exint Enable        */
    bsp_irq_exint_enable();
    /* NVIC Enable */
    NVIC_EnableIRQ(bsp_get_exint_nvic());

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Low Power Mode End Processing.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   STOP1 Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result low_power_end_stop1(void)
{
    TXZ_Result result = TXZ_SUCCESS;
    
    /*----------------------------*/
    /* External Interrupt Disable */
    /*----------------------------*/
    /* NVIC Disable         */
    NVIC_DisableIRQ(bsp_get_exint_nvic());
    /* IRQ Disable          */
    bsp_irq_exint_disable();
    /*----------------------*/
    /* Application          */
    /*----------------------*/
    if (application_finalize_stop1() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*----------------------*/
    /* Driver               */
    /*----------------------*/
    if (driver_finalize_stop1() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }

    return (result);
}

/*==========================*/
/* STOP2 Mode               */
/*==========================*/
/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for STOP2 Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize_stop2(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for STOP2 Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize_stop2(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* LED                  */
    /*----------------------*/
    /* LED initialize has done within BSP. */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Start Processing.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for STOP2 Mode.
  */
/*--------------------------------------------------*/
static TXZ_Result low_power_start_stop2(void)
{
    TXZ_Result result = TXZ_SUCCESS;
    uint32_t work;
    uint8_t mask;

    /*----------------------*/
    /* Application          */
    /*----------------------*/
    /*--- LED ---*/
    /* Set LED Port before IO Keep. */
    {
        uint32_t i;
        led_t *p_led = MAIN_NULL;

        for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
        {
            p_led = &instance.led[i];

            p_led->init.id                  = (uint32_t)p_led;
            p_led->init.p_gpio              = bsp_get_gpio_instance_address();
            p_led->init.port.group          = bsp_get_gpio_group_led((BSPLed)(i));
            p_led->init.port.num            = bsp_get_gpio_num_led((BSPLed)(i));
            p_led->init.blink.func          = TXZ_DISABLE;
            p_led->init.blink.interval.on   = 0;
            p_led->init.blink.interval.off  = 0;
            /* LED STOP2 :on  */
            if (i == (uint32_t)BSP_LED_4)
            {
                p_led->init.state           = LED_STATE_ON;
            }
            /* default   :off */
            else
            {
                p_led->init.state           = LED_STATE_OFF;
            }
            led_initialize(p_led);
        }
    }
    /*----------------------*/
    /* WDT Disable          */
    /*----------------------*/
    wdt_disable(TSB_SIWD0);
    /*----------------------*/
    /* Flash Wait           */
    /*----------------------*/
    /* [FCSR0]<RDYBSY>   : Read(wait for done) */
    while ((TSB_FC->SR0 & FCSR0_RDYBSY_MASK) != FCSR0_RDYBSY_R_DONE)
    {
        /* no processing */
    }
    /*----------------------*/
    /* IO Keep              */
    /*----------------------*/
    /* RLMSHTDNOP:<PTKEEP=1>   */
    rlm_port_keep_enable(p_rlm) ;
    /*----------------------*/
    /* Set STBY             */
    /*----------------------*/
    /* [CGSTBYCR]<STBY>    :STOP2           */
    cg_set_stby_mode(p_cg, CGSTBYCR_STBY_RW_STOP2) ;
    /*----------------------*/
    /* PLL Disable          */
    /*----------------------*/
    pll_disable();
    /*----------------------*/
    /* Internal HOSC Select */
    /*----------------------*/
    /* [CGWUPHCR]<WUCLK> : IHOSC */
    work = (uint32_t)(TSB_CG->WUPHCR & ~CGWUPHCR_WUCLK_MASK);
    TSB_CG->WUPHCR = (uint32_t)(work | CGWUPHCR_WUCLK_RW_IHOSC);
    /*----------------------*/
    /* Internal HOSC Warm-up Time Set*/
    /*----------------------*/
    cg_set_warm_up_time(p_cg, CG_WARM_UP_SRC_IHOSC1, WU_TIME_INT);
    /*----------------------*/
    /* Internal HOSC Enable */
    /*----------------------*/
    internal_hosc_enable();
    /*----------------------*/
    /* External HOSC Disable*/
    /*----------------------*/
    /* [CGOSCCR]<EOSCEN>   : EHOSC Disable  */
    work = (uint32_t)(TSB_CG->OSCCR & ~CGOSCCR_EOSCEN_MASK);
    TSB_CG->OSCCR = (uint32_t)(work | CGOSCCR_EOSCEN_RW_UNUSE);
    /*----------------------*/
    /* OFD Internal HOSC Disable*/
    /*----------------------*/
    /* [CGOSCCR]<IHOSC2EN> : OFD IHOSC Disable  */
    work = (uint32_t)(TSB_CG->OSCCR & ~CGOSCCR_IHOSC2EN_MASK);
    TSB_CG->OSCCR = (uint32_t)(work | CGOSCCR_IHOSC2EN_RW_DISABLE);
    /*----------------------*/
    /* External HOSC Disable Check*/
    /*----------------------*/
    /* [CGOSCCR]<EOSCEN>   : Read  */
    while ((TSB_CG->OSCCR & CGOSCCR_EOSCEN_MASK) != CGOSCCR_EOSCEN_RW_UNUSE) 
    {
        /* no processing */
    }
    /*----------------------*/
    /* OFD Internal HOSC Disable Wait*/
    /*----------------------*/
    /* [CGOSCCR]<IHOSC2F>  : Read(wait for OFD IHOSC Disable)  */
    while ((TSB_CG->OSCCR & CGOSCCR_IHOSC2F_MASK) != CGOSCCR_IHOSC2F_R_STOP) 
    {
        /* no processing */
    }
    /*----------------------*/
    /* STOP2 Reset Flag RESET Pin Flag Clear*/
    /*----------------------*/
    /* [RLMRSTFLG0]<STOP2RSTF>,<PINRSTF>  : Clear */
    mask = (uint8_t)(TSB_RLM->RSTFLG0 &~ (RLMRSTFLG0_STOP2RSTF_MASK | RLMRSTFLG0_PINRSTF_MASK));
    TSB_RLM->RSTFLG0 = (uint8_t)(mask | RLMRSTFLG0_W_CLEAR);
    /*----------------------*/
    /* Driver               */
    /*----------------------*/
    if (driver_initialize_stop2() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*----------------------*/
    /* Application          */
    /*----------------------*/
    if (application_initialize_stop2() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /* Exint Enable        */
    bsp_irq_exint_enable();
    /* NVIC Enable */
    NVIC_EnableIRQ(bsp_get_exint_nvic());

    return (result);
}

/**
 *  @}
 */ /* End of group CG_EXINT_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup CG_EXINT_Exported_functions CG_EXINT Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    /*----------------------*/
    /* Reset Flag Check     */
    /*----------------------*/
    if ((TSB_RLM->RSTFLG0 & RLMRSTFLG0_STOP2RSTF_MASK) == RLMRSTFLG0_STOP2RSTF_R_RESET)
    {
        /* It's  executed, when STOP2 mode is released. */
        /*----------------------*/
        /* BSP initialize       */
        /*----------------------*/
        bsp_initialize();
        /*----------------------*/
        /* Release "IO Keep"    */
        /*----------------------*/
        TSB_RLM->SHTDNOP = RLMSHTDNOP_PTKEEP_RW_RELEASE;
        /* Clear Flag */
        TSB_RLM->RSTFLG0 = RLMRSTFLG0_W_CLEAR;
        TSB_RLM->RSTFLG1 = RLMRSTFLG1_W_CLEAR;
    }else
    {
        /*----------------------*/
        /* BSP initialize       */
        /*----------------------*/
        bsp_initialize();
    }

    /*--- Jump to the Interrupt routine, STOP2 mode is released. ---*/
    /*  STOP2 REQ Interrupt Enable */
    bsp_irq_exint_enable();
    /* NVIC Enable */
    NVIC_EnableIRQ(bsp_get_exint_nvic());
    
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    variable_initialize();
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(proc != PROC_ERROR)
    {
        switch (proc)
        {
        /*--- Normal Mode     :START   ---*/
        case PROC_NORMAL_START:
            {
                if (normal_start() == TXZ_SUCCESS)
                {
                    proc = PROC_NORMAL;
                }
                else
                {
                    proc = PROC_ERROR;
                }
            }
            break;
        /*--- Normal Mode    :RUNNING ---*/
        case PROC_NORMAL:
            {
                uint32_t i;
                /*----------------------*/
                /* SW                   */
                /*----------------------*/
                for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
                {
                    sw_task(&instance.sw[i]);
                }
                /*----------------------*/
                /* LED                  */
                /*----------------------*/
                for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
                {
                    led_task(&instance.led[i]);
                }
            }
            break;
        /*--- Normal Mode    :END     ---*/
        case PROC_NORMAL_END:
            {
                /*----------------------*/
                /* End Processing       */
                /*----------------------*/
                if (normal_end() == TXZ_SUCCESS)
                {
                    /*----------------------*/
                    /* BSP finalize         */
                    /*----------------------*/
                    /* if need, BSP finalize for normal mode. */
                    bsp_finalize();
                    proc = PROC_LOW_POWER_START;
                }
                else
                {
                    proc = PROC_ERROR;
                }
            }
            break;
        /*--- Low Power Mode :START   ---*/
        case PROC_LOW_POWER_START:
            {
                switch (low)
                {
                case LOW_POWER_IDLE:
                    {
                        /*----------------------*/
                        /* BSP initialize       */
                        /*----------------------*/
                        /* If need, BSP initialize for IDLE Mode. (Now, use Norma Mode) */
                        bsp_initialize();
                        /*----------------------*/
                        /* Start Processing     */
                        /*----------------------*/
                        if (low_power_start_idle() == TXZ_SUCCESS)
                        {
                            /*>>>Wait For Interrupt */
                            __WFI();
                            /*<<<Wait For Interrupt */
                            proc = PROC_LOW_POWER_END;
                        }
                        else
                        {
                            proc = PROC_ERROR;
                        }
                    }
                    break;
                case LOW_POWER_STOP_1:
                    {
                        /*----------------------*/
                        /* BSP initialize       */
                        /*----------------------*/
                        /* If need, BSP initialize for Stop1 Mode. (Now, use Norma Mode) */
                        bsp_initialize();
                        /*----------------------*/
                        /* Start Processing     */
                        /*----------------------*/
                        if (low_power_start_stop1() == TXZ_SUCCESS)
                        {
                            /*>>>Wait For Interrupt */
                            __WFI();
                            /*<<<Wait For Interrupt */
                            proc = PROC_LOW_POWER_END;
                        }
                        else
                        {
                            proc = PROC_ERROR;
                        }
                    }
                    break;
                case LOW_POWER_STOP_2:
                    {
                        /*----------------------*/
                        /* BSP initialize       */
                        /*----------------------*/
                        /* If need, BSP initialize for Stop2 Mode. (Now, use Norma Mode) */
                        bsp_initialize();
                        /*----------------------*/
                        /* Start Processing     */
                        /*----------------------*/
                        /* When release the "Low Power Mode", reset sequence. */
                        if (low_power_start_stop2() == TXZ_SUCCESS)
                        {
                            /*>>>Wait For Interrupt */
                            while(1)
                            {
                                __WFI();
                            }
                            /*<<<Wait For Interrupt */
                            /*--- Reset ---*/
                        }
                        else
                        {
                            proc = PROC_ERROR;
                        }
                    }
                    break;
                default:
                    proc = PROC_ERROR;
                    break;
                }
            }
            break;
        /*--- Low Power Mode :END     ---*/
        case PROC_LOW_POWER_END:
            {
                switch (low)
                {
                case LOW_POWER_IDLE:
                    {
                        /*----------------------*/
                        /* End Processing       */
                        /*----------------------*/
                        if (low_power_end_idle() == TXZ_SUCCESS)
                        {
                            /*----------------------*/
                            /* BSP finalize         */
                            /*----------------------*/
                            /* If need, BSP finalize for IDLE Mode. (Now, use Norma Mode) */
                            bsp_finalize();
                            /*----------------------*/
                            /* BSP initialize       */
                            /*----------------------*/
                            bsp_initialize();
                            /*----------------------*/
                            /* Initialize           */
                            /*----------------------*/
                            variable_initialize();
                            /*<<< If need, Re-Setting for Normal Mode. */
                        }
                        else
                        {
                            proc = PROC_ERROR;
                        }
                    }
                    break;
                case LOW_POWER_STOP_1:
                    {
                        /*----------------------*/
                        /* End Processing       */
                        /*----------------------*/
                        if (low_power_end_stop1() == TXZ_SUCCESS)
                        {
                            /*----------------------*/
                            /* BSP finalize         */
                            /*----------------------*/
                            /* If need, BSP finalize for IDLE Mode. (Now, use Norma Mode) */
                            bsp_finalize();

                            /*>>> If need, Re-Setting for Normal Mode. */
                            /*----------------------*/
                            /* System Initialize    */
                            /*----------------------*/
                            SystemInit();
                            /*----------------------*/
                            /* BSP initialize       */
                            /*----------------------*/
                            bsp_initialize();
                            /*----------------------*/
                            /* Initialize           */
                            /*----------------------*/
                            variable_initialize();
                            /*<<< If need, Re-Setting for Normal Mode. */
                        }
                        else
                        {
                            proc = PROC_ERROR;
                        }
                    }
                    break;
                case LOW_POWER_STOP_2:
                default:
                    /* Stop2 :Reset Sequence. */
                    proc = PROC_ERROR;
                    break;
                }
            }
            break;
        /*--- Error                   ---*/
        case PROC_ERROR:
            /* no processing */
            break;
        /*--- default                 ---*/
        default:
            proc = PROC_ERROR;
            break;
        }
    }
    /*>>> System Error. Please Debug. */
    /*----------------------*/
    /* BSP finalize         */
    /*----------------------*/
    bsp_finalize();
#ifdef DEBUG
    assert_failed(MAIN_NULL, 0);
#endif

    return (0);
}
/*--------------------------------------------------*/
/** 
  * @brief  External Interrupt IRQ.
  * @param  exi : BSPExInt.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_exint(BSPExInt exi)
{
    /*----------------------------*/
    /* External Interrupt Disable */
    /*----------------------------*/
    /* NVIC Disable         */
    NVIC_DisableIRQ(bsp_get_exint_nvic());
    /* IRQ Disable          */
    bsp_irq_exint_disable();
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer.
  * @param  timer :Timer.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    default:
        /* no processing */
        break;
    }
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __nop();
    }
}
#endif
/**
 *  @}
 */ /* End of group CG_EXINT_Exported_functions */

/**
 *  @}
 */ /* End of group CG_EXINT */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
