/** 
 *******************************************************************************
 * @file    main.c
 * @brief   DAC Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif
/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup DAC_Sample_Appli DAC Sample Appli
 *  @{
 */

/** 
 *  @defgroup MAIN_PRIVATE  MAIN Private
 *  @brief    MAIN Private Group.
 *  @{
 */

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "bsp.h"
#include "bsp_com.h"
#include "bsp_uart_io.h"
#include "bsp_timer.h"
#include "driver_com.h"
#include "dac.h"


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/* no define */

/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/*----------------------*/
/*  UART Setting        */
/*----------------------*/
/** 
 *  @defgroup UART_Setting UART Setting
 *  @{
 */
#define CFG_UART_BAUDRATE               ((uint32_t)115200)  /*!< Boudrate(bps) */
/** 
 *  @}
 */ /* End of group UART_Setting */

/** 
 *  @}
 */ /* End of group Timer_EVENT_COUNT_Private_define */
#define MAIN_NULL       ((void *)0)     /*!< NULL. */
#define USER_ID_UART    ((uint32_t)1)   /*!< User ID :USB UART (Any value) */
/** 
 *  @defgroup UART_IO_SendMax  UART IO Send Max.
 *  @brief    Size of send max. (byte)
 *  @{
 */
#define UART_IO_SEND_MAX       ((uint32_t)32)
/**
 *  @}
 */ /* End of group UART_IO_SendMax */

/** 
 *  @defgroup UART_IO_ReceiveMax  UART IO Receive Max.
 *  @brief    Size of receive max. (byte)
 *  @{
 */
   #define UART_IO_RECEIVE_MAX_LOCAL    ((uint32_t)5)
/**
 *  @}
 */ /* End of group UART_IO_ReceiveMax */
/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/*----------------------------------*/
/** 
 * @enum   Processing
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_IDLE,              /*!< Idle.                        */
    PROC_OUTPUT_MSG,        /*!< Output first message.        */
    PROC_INPUT,             /*!< Wait to input.               */
    PROC_DAC_EXEC,          /*!< Output Echo back data.       */
    PROC_OUTPUT_ERR,        /*!< Output input error message.  */
    PROC_ERROR,             /*!< Error.                       */
    PROC_PARITY_ERR         /*!< Parity Error.                */
} Processing;

/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/*----------------------------------*/
/** 
 * @struct input_t
 * @brief  Input data information.
*/
/*----------------------------------*/
typedef struct
{
    char    data[(UART_IO_RECEIVE_MAX_LOCAL)];  /*!< Input data(+ for NULL 1byte). */
    uint8_t wp;                             /*!< Write pointer.                */
} input_t;
/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
static Processing proc;     /*!< Processing mode.        */
static input_t    input;        /*!< Input data information. */

;

/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
static void clear_data(void);
static TXZ_Result application_initialize(void);
/*--------------------------------------------------*/
/** 
  * @brief  Clear input data.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void clear_data(void)
{
    uint8_t i;

    for (i=0; i<(UART_IO_RECEIVE_MAX_LOCAL); i++)
    {
        input.data[i] = 0;
    }
    input.wp = 0;
}
/**
 *  @}
 */ /* End of group UART_IO_PRIVATE */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
   TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = USER_ID_UART;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = (uint32_t)CFG_UART_BAUDRATE;
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result  = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    return (result);
}
/**
 *  @}
 */ /* End of group MAIN_PRIVATE */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    /*----------------------*/
    /* Set Processing Mode  */
    /*----------------------*/
    proc = PROC_OUTPUT_MSG;
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /* BSP initialize       */
    bsp_initialize();
    /* Application initialization */
    if(application_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(1)
    {
        switch(proc)
        {
        /*--- Output "First Message" ---*/
        case PROC_OUTPUT_MSG:
            {
                /* Clear input data.     */
                clear_data(); 
                /* Output first message. */
                if (printf("Input > ") != -1 )
                {
                    proc = PROC_INPUT;
                }
                else
                {
                    proc = PROC_ERROR;
                }
            }
            break;
        /*--- Wait to Input ---*/
        case PROC_INPUT:
            {   
                int data;
                
                data = getchar();
                if ((data != -1) && (input.wp < UART_IO_RECEIVE_MAX_LOCAL))
                {
                    input.data[input.wp] = (char)data;
                    input.wp++;
                    switch ((char)data)
                    {
                    /*--- LF: line feed ---*/
                    case '\n':
                        proc = PROC_DAC_EXEC;
                        input.data[input.wp - 1] = '\0';
                        break;
                    /*--- default ---*/
                    default:
                        if(input.wp == UART_IO_RECEIVE_MAX_LOCAL)
                        {
                            proc = PROC_OUTPUT_ERR;
                            if (printf("\n") == -1 )
                            {
                                proc = PROC_ERROR;
                            }
                      }
                        break;
                    }
                }
                else
                {
                    proc = PROC_OUTPUT_ERR;
                }
            }
            break;
        /*--- Output "DAC Result data" ---*/
            case PROC_DAC_EXEC:
                {
                    TXZ_Result result;
                    /* Caractor -> HEX number */
                    char *endptr;
                    uint32_t res = (uint32_t)strtol(input.data, &endptr, 0);
                    if(*endptr != '\0')
                    {
                        proc = PROC_OUTPUT_ERR;
                    }
                    else {
                        if(res <= 0xFF)
                        {
                            proc = PROC_OUTPUT_MSG;
                            /* DAC Enable */
                            result = REG_DAC_enable(TSB_DA0);
                            if(result != TXZ_SUCCESS){
                                proc = PROC_ERROR;
                            }
                            /* DAC REG Set */
                            result = REG_DAC_DAxREG_set(TSB_DA0, res);
                            if(result != TXZ_SUCCESS){
                                proc = PROC_ERROR;
                            }
                        }else{
                            proc = PROC_OUTPUT_ERR;
                        }
                    }
                }
                break;
      /*--- Output "Input Error Message" ---*/
        case PROC_OUTPUT_ERR:
            {
                if (printf("Parameter Error!!\n") != -1 )
                {
                    TXZ_Result result;
                    proc = PROC_OUTPUT_MSG;
                    result = REG_DAC_DAxREG_set(TSB_DA0, (uint32_t)0);
                    if(result != TXZ_SUCCESS){
                        proc = PROC_ERROR;
                    }
                }
                else
                {
                    proc = PROC_ERROR;
                }
            }
            break;
        /*--- Error ---*/
        case PROC_ERROR:
            /* System Error. Please Debug. */
#ifdef DEBUG
            assert_failed(MAIN_NULL, 0);
#endif
            break;
        /*--- Default ---*/
        default:
            proc = PROC_ERROR;
            break;
        }
    }
}


/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_rx_irq_handler();
        break;
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_tx_irq_handler();
        break;
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_err_irq_handler();
        break;
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __nop();
    }
}
#endif

/**
 *  @} 
 */ /* End of group DAC_Sample_Appli */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
