/** 
 *******************************************************************************
 * @file    main.c
 * @brief   DLCD ADC Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif
/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_timer.h"
#include "bsp_dlcd.h"


/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup DLCD_ADC DLCD_ADC Sample Appli
 *  @{
 */

/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DLCD_ADC Sample Appli Private Macro
 *  @{
 */

/** 
 *  @}
 */ /* End of group DLCD_ADC_Private_macro */

/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DLCD_ADC_Private_define
 *  @{
 */

/*----------------------*/
/*  Output Setting      */
/*----------------------*/
/** 
 *  @defgroup Output_Setting Output Setting
 *  @{
 */

/** 
 *  @}
 */ /* End of group Output_Setting */

/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DLCD_ADC_Private_macro DLCD_ADC Private Macro
 *  @{
 */
#define MAIN_NULL       ((void *)0)     /*!< Null Pointer  */

#define CFG_OUTPUT_INTERVAL             ((uint32_t)100)     /*!< output period(ms) */
/** 
 *  @}
 */ /* End of group DLCD_ADC_Private_macro */

/** 
 *  @}
 */ /* End of group DLCD_ADC_Private_define */

/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DLCD_ADC_Private_define
 *  @{
 */

/*----------------------------------*/
/** 
 * @enum   Processing
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_IDLE,          /*!< Idle.             */
    PROC_START,         /*!< Start.            */
    PROC_STOP,          /*!< Stop.             */
    PROC_INTERVAL_WAIT, /*!< Interval Wait.    */
    PROC_OUTPUT,        /*!< Output.           */
    PROC_ERROR          /*!< Error.            */
} Processing;

/*----------------------------------*/
/** 
 * @enum   CoversionState
 * @brief  Sonversion state
*/
/*----------------------------------*/
typedef enum {
    CONVERSION_IDLE = 0U,     /*!< Idle         */
    CONVERSION_RUN,           /*!< Conversion Running.    */
    CONVERSION_SUCCESS,       /*!< Conversion Success.    */
    CONVERSION_FAILURE        /*!< Conversion Failure.    */
} CoversionState;
/** 
 *  @}
 */ /* End of group DLCD_ADC_Private_define */

/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DLCD_ADC_Private_typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
    t32a_t      t32a;              /*!< Driver      :T32A.     */
    adc_t       adc;               /*!< Driver      :ADC.      */
    dlcd_t      dlcd;              /*!< Driver      :DLCD.     */
    cg_t        cg;                /*!< Driver      :CG.       */
    rlm_t       rlm;               /*!< Driver      :RLM.      */
    timer_t     timer;             /*!< Application :Timer (1ms timer). */
    display_t   display;           /*!< Display instance.       */

} instance_t;

/*----------------------------------*/
/** 
 * @brief Information.
*/
/*----------------------------------*/
typedef struct
{
   TXZ_FunctionalState update;     /*!< Update flag.       */
   uint32_t            count;      /*!< Update flag.       */
} information_t;
/** 
 *  @}
 */ /* End of group DLCD_ADC_Private_typedef */

/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DLCD_ADC_Private_variables
 *  @{
 */
static instance_t instance;     /*!< Instance.                */
static Processing proc;         /*!< Processing mode.         */
static CoversionState status;   /*!< ADC Processing mode.     */
static information_t info;      /*!< Clock information.       */

/**
 *  @}
 */ /* End of group DLCD_ADC_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DLCD_ADC_Private_functions
 *  @{
 */
static void variable_initialize(void);
static TXZ_Result driver_initialize(void);
static TXZ_Result driver_finalize(void);
static TXZ_Result application_initialize(void);
static TXZ_Result application_finalize(void);
static void timer_interval_handler(uint32_t id);
static void done_handler(uint32_t id, TXZ_Result result);

/*--------------------------------------------------*/
/** 
  * @brief  Variable Initialize
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void variable_initialize(void)
{
    /* Instance. */
    instance.t32a.p_instance     = MAIN_NULL;
    instance.adc.p_instance      = MAIN_NULL;
    instance.timer.init.p_t32a   = MAIN_NULL;
    instance.display.init.p_gpio = MAIN_NULL;
    instance.display.init.p_dlcd = MAIN_NULL;
    instance.display.init.p_cg   = MAIN_NULL;
    instance.display.init.p_rlm  = MAIN_NULL;
    /* Processing mode. */
    proc = PROC_START;
}
/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize
  * @param  -
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (p_t32a->p_instance != MAIN_NULL)
        {
            if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* ADC                  */
    /*----------------------*/
    {
        adc_t *p_adc = &instance.adc;

        /*--- Construct  ---*/
        /* Register Allocation */
        p_adc->p_instance       = TSB_ADA;
        /*--- Initialize ---*/
        p_adc->init.id          = (uint32_t)p_adc;
        p_adc->init.clk.exaz0    = ADC_SAMPLING_PERIOD0_X2N;
        p_adc->init.clk.exaz1    = ADC_SAMPLING_PERIOD1_X2N;
        p_adc->init.clk.vadcld  = ADC_SCLK_3;   /* less than 40MHz  120MHz / 3 = 40MHz */
        p_adc->init.clk.sampling_select = 0;
        p_adc->init.mod1        = ADC_MOD1_AVDD5_3V_SCLK_15;
        p_adc->init.mod2        = ADC_MOD2_TMPM3Hz;
        p_adc->init.trm         = ADC_TRM_TMPM3Hz;
        p_adc->handler.continuity   = done_handler;

        if (adc_init(p_adc) != TXZ_SUCCESS)
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* DLCD                 */
    /*----------------------*/
    {
        dlcd_t *p_dlcd = &instance.dlcd;
        cg_t *p_cg = &instance.cg;
        rlm_t *p_rlm = &instance.rlm;

        /*--- Construct  ---*/
        /* Register Allocation */
        p_dlcd->p_instance1 = TSB_DLCD;
        p_dlcd->p_instance2 = TSB_AGP;
        p_cg->p_instance = TSB_CG;
        p_rlm->p_instance = TSB_RLM;
    }
    return (result);
}
/*--------------------------------------------------*/
/** 
  * @brief  Driver Finalize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* ADC                  */
    /*----------------------*/
    {
        adc_t *p_adc = &instance.adc;

        /*--- Finalize ---*/
        if (adc_deinit(p_adc) != TXZ_SUCCESS)
        {
            result = TXZ_ERROR;
        }
        /*--- Destruct ---*/
        /* Register Release */
        p_adc->p_instance = MAIN_NULL;
    }
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Finalize ---*/
        /* no processing */
        /*--- Destruct ---*/
        /* Register Release */
        p_t32a->p_instance = MAIN_NULL;
    }
    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver finalize will process within BSP.     */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize
  * @param  -
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    {
        timer_t *p_timer = &instance.timer;

        p_timer->init.id       = (uint32_t)&instance.timer;
        p_timer->init.p_t32a   = &instance.t32a;
        p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
        p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
        p_timer->init.interval = 1000;
        p_timer->init.handler  = timer_interval_handler;
        timer_initialize(p_timer);
    }
    /*----------------------*/
    /* Thermistors          */
    /*----------------------*/
    {
        uint32_t ch;
        adc_channel_setting_t param;
        ch = bsp_get_thermistors_ch(BSP_THERMISTORS_1);
        param.interrupt = ADC_INT_ENABLE;
        param.type      = ADC_CONVERSION_CNT;
        param.ain       = bsp_get_thermistors_adin(BSP_THERMISTORS_1);
        if (adc_channel_setting(&instance.adc, ch, &param) != TXZ_SUCCESS)
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Display              */
    /*----------------------*/
    {
        display_t *p_display = &instance.display;
        p_display->init.p_dlcd = &instance.dlcd;
        p_display->init.p_cg = &instance.cg;
        p_display->init.p_rlm = &instance.rlm;

        p_display->init.id                          = (uint32_t)p_display;
        p_display->init.dlcd_data.DLCDCR1_data      = DLCDCR1_EDSP_ENABLE_11;
        p_display->init.dlcd_data.DLCDCR2_data      = DLCDCR2_CONTSEL;
        p_display->init.dlcd_data.DLCDCR2_data     |= DLCDCR2_CLKSEL;
        p_display->init.dlcd_data.DLCDCR2_data     |= DLCDCR2_DUTY;
        p_display->init.dlcd_data.DLCDCR3_data      = DLCDCR3_INVALID;
        p_display->init.dlcd_data.DLCDCR3_data      = DLCDCR3_BSEL_LOW;
        p_display->init.dlcd_data.DLCDCR4_data      = DLCDCR4_RVDPIN_000;
        p_display->init.dlcd_data.DLSTCLKCYC_data   = DLCD_DLSTCLKCYC_DATA;
        p_display->init.dlcd_data.DLCONTCYC_data    = DLCD_DLCONTCYC_DATA;
        p_display->init.dlcd_data.AGPREG0_data      = DLCD_AGP_HIZ;
        p_display->init.dlcd_data.AGPREG1_data      = DLCD_AGP_REG_ALL;
        p_display->init.clock                       = DLCD_CLOCK;
        dlcd_initialize(p_display);
    }
    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Finalize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Thermistors          */
    /*----------------------*/
    if (instance.adc.p_instance != MAIN_NULL)
    {
        if (adc_channel_clear(&instance.adc, bsp_get_thermistors_ch(BSP_THERMISTORS_1)) != TXZ_SUCCESS)
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    if (instance.timer.init.p_t32a != MAIN_NULL)
    {
        timer_finalize(&instance.timer);
    }
    /*----------------------*/
    /* Display              */
    /*----------------------*/
    if ((instance.display.init.p_gpio != MAIN_NULL) && 
        (instance.display.init.p_dlcd != MAIN_NULL) && 
        (instance.display.init.p_cg != MAIN_NULL)   && 
        (instance.display.init.p_rlm != MAIN_NULL))
    {
        dlcd_finalize(&instance.display);
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    if (id == (uint32_t)&instance.timer)
    {
        /* Increment Tick. Tick is used to driver's Timeout. */
        hal_inc_tick();
        /* Interval Time Count. */
        info.count++;
        if (info.count >= CFG_OUTPUT_INTERVAL)
        {
            /*>>> Critical information. */
            info.update = TXZ_ENABLE;
            /*<<< Critical information. */
            info.count  = 0;
        }
    }
}
/*--------------------------------------------------*/
/** 
  * @brief      Conversion end handler
  * @param[in]  id      :User id.
  * @param[in]  result  :Result.
  * @retval     -
  * @note       Called by adc driver
  */
/*--------------------------------------------------*/
static void done_handler(uint32_t id, TXZ_Result result)
{
    if (result == TXZ_SUCCESS)
    {
        status = CONVERSION_SUCCESS;
    }
    else
    {
        status = CONVERSION_FAILURE;
    }
}

/**
 *  @}
 */ /* End of group DLCD_ADC_Private_functions */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DLCD_ADC_Exported_functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /*--- variable    ---*/
    variable_initialize();

    /*--- driver      ---*/
    if (driver_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    /*--- application ---*/
    if (application_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(proc != PROC_ERROR)
    {
        switch(proc)
        {
        /*--- Start ---*/
        case PROC_START:
            {
                timer_start(&instance.timer);
                /*<<< Critical information. */
                status = CONVERSION_RUN;
                NVIC_EnableIRQ(INTADACNT_IRQn);
                /* Blocking Conversion Start */
                if (adc_start(&instance.adc) == TXZ_SUCCESS)
                {
                    proc = PROC_INTERVAL_WAIT;
                }
                else
                {
                    proc = PROC_STOP;
                }
            }
            break;
        /*--- Stop ---*/
        case PROC_STOP:
            {
                if (adc_stop(&instance.adc) != TXZ_SUCCESS)
                {
                    /* no processing */
                }
                timer_stop(&instance.timer);
                NVIC_DisableIRQ(INTADACNT_IRQn);
                status = CONVERSION_IDLE;
                proc = PROC_ERROR;
            }
            break;
        /*--- Interval Wait ---*/
        case PROC_INTERVAL_WAIT:
            {
                if ((info.update == TXZ_ENABLE) && (status == CONVERSION_SUCCESS))
                {
                    /*>>> Critical information. */
                    info.update = TXZ_DISABLE;
                    proc = PROC_OUTPUT;
                }
            }
            break;
        /*--- Output ---*/
        case PROC_OUTPUT:
            {
                uint32_t value;
                display_t *p_display = MAIN_NULL;
                p_display = &instance.display;

                /* Get conversion value1. */
                proc = PROC_INTERVAL_WAIT;
                /* Get conversion value. */
                if (adc_channel_get_value(&instance.adc, bsp_get_thermistors_ch(BSP_THERMISTORS_1), &value) == TXZ_SUCCESS)
                {
                    /* Output information. */
                    p_display->init.data = value;
                    dlcd_display_set(p_display, p_display->init.data);
                }
                else
                {
                    proc = PROC_STOP;
                }
            }
            break;
        case PROC_ERROR:
            /* System Error. Please Debug. */
            break;

        default:
            proc = PROC_ERROR;
            break;
        }
    }
    /*>>> System Error. Please Debug. */
    /*----------------------*/
    /* Finalize             */
    /*----------------------*/
    /*--- application ---*/
    if (application_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*--- driver      ---*/
    if (driver_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*----------------------*/
    /* BSP finalize         */
    /*----------------------*/
    bsp_finalize();
#ifdef DEBUG
    assert_failed(MAIN_NULL, 0);
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  ADC CNT IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_adc_cnt(void)
{
    done_handler(instance.adc.init.id,TXZ_SUCCESS);
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer.
  * @param  timer :Timer.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    default:
        /* no processing */
        break;
    }
}


#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __NOP();
    }
}
#endif

/**
 *  *  @}
 *   */ /* End of group DLCD_ADC_Exported_functions */

/**
 *  @} 
 */ /* End of group DLCD_ADC */

/**
 *  *  @}
 *   */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
