/** 
 *******************************************************************************
 * @file    main.c
 * @brief   DMAC_UART Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_uart_io_dma.h"
#include "bsp_timer.h"
/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup DMAC_UART DMAC_UART Sample Appli
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DMAC_UART_Private_macro DMAC UART Private Macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group DMAC_UART_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DMAC_UART_Private_define DMAC UART Private Define
 *  @{
 */

/*----------------------*/
/*  DMA Setting         */
/*----------------------*/
/* Work Area = ((The total number of the used channel) * (4 * DWORD) * 2)  */
#define CFG_DMA_WORK_SIZE     ((uint32_t)(DMA_CH_NUM * 4 * sizeof(uint32_t) * 2))     /*!< DMA work area. */

/** 
 *  @}
 */ /* End of group DMAC_UART_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DMAC_UART_Private_define DMAC UART Private Define
 *  @{
 */

#define MAIN_NULL               ((void *)0) /*!< Null Pointer */

#define CFG_DMA_ch_dstinc_1byte               0x00000000 ; /*!< dstinc 1 byte */
#define CFG_DMA_ch_dstinc_2byte               0x00000001 ; /*!< dstinc 2 byte */
#define CFG_DMA_ch_dstinc_4byte               0x00000010 ; /*!< dstinc 4 byte */
#define CFG_DMA_ch_dstinc_none                0x00000011 ; /*!< dstinc none */
#define CFG_DMA_ch_dstsize_1byte              0x00000000 ; /*!< dstsize 1 byte */
#define CFG_DMA_ch_dstsize_2byte              0x00000001 ; /*!< dstsize 2 byte */
#define CFG_DMA_ch_dstsize_4byte              0x00000010 ; /*!< dstsize 4 byte */
#define CFG_DMA_ch_srcinc_1byte               0x00000000 ; /*!< srcinc 1 byte */
#define CFG_DMA_ch_srcinc_2byte               0x00000001 ; /*!< srcinc 2 byte */
#define CFG_DMA_ch_srcinc_4byte               0x00000010 ; /*!< srcinc 4 byte */
#define CFG_DMA_ch_srcinc_none                0x00000011 ; /*!< srcinc none */
#define CFG_DMA_ch_srcsize_1byte              0x00000000 ; /*!< srcsize 1 byte */
#define CFG_DMA_ch_srcsize_2byte              0x00000001 ; /*!< srcsize 2 byte */
#define CFG_DMA_ch_srcsize_4byte              0x00000010 ; /*!< srcsize 4 byte */
#define CFG_DMA_ch_rpow_001                   0x00000000 ; /*!< R power 1 */
#define CFG_DMA_ch_rpow_002                   0x00000001 ; /*!< R power 2 */
#define CFG_DMA_ch_rpow_004                   0x00000010 ; /*!< R power 4 */
#define CFG_DMA_ch_rpow_008                   0x00000011 ; /*!< R power 8 */
#define CFG_DMA_ch_rpow_016                   0x00000100 ; /*!< R power 16 */
#define CFG_DMA_ch_rpow_032                   0x00000101 ; /*!< R power 32 */
#define CFG_DMA_ch_rpow_064                   0x00000110 ; /*!< R power 64 */
#define CFG_DMA_ch_rpow_128                   0x00000111 ; /*!< R power 128 */
#define CFG_DMA_ch_rpow_256                   0x00001000 ; /*!< R power 256 */
#define CFG_DMA_ch_rpow_512                   0x00001001 ; /*!< R power 512 */
#define CFG_DMA_ch_rpow_none                  0x00001111 ; /*!< R power none */
#define CFG_DMA_ch_next_usesingle             0x00000000 ; /*!< next use single */
#define CFG_DMA_ch_next_useburst              0x00000001 ; /*!< next use burst  */

/** 
 *  @}
 */ /* End of group DMAC_UART_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DMAC_UART_Private_define DMAC UART Private Define
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_IDLE,              /*!< Idle.                        */
    PROC_START,             /*!< Start.                       */
    PROC_STOP,              /*!< Stop.                        */
    PROC_OUTPUT_MSG,        /*!< Output first message.        */
    PROC_INPUT,             /*!< Wait to input.               */
    PROC_OUTPUT_ECHO,       /*!< Output Echo back data.       */
    PROC_OUTPUT_ERR,        /*!< Output input error message.  */
    PROC_ERROR              /*!< Error.                       */
} Processing;
/** 
 *  @}
 */ /* End of group DMAC_UART_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DMAC_UART_Private_typedef DMAC UART Private Typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
   t32a_t        t32a;      /*!< Driver      :T32A.              */
   dma_t         dma;       /*!< Driver      :DMA.               */
   timer_t       timer;     /*!< Application :Timer (1ms timer). */
   uart_io_dma_t *p_uart;   /*!< Application :USB UART.          */
} instance_t;

/*----------------------------------*/
/** 
 * @brief  Input data information.
*/
/*----------------------------------*/
typedef struct
{
    char    data[(UART_IO_DMA_RECEIVE_MAX+2)];  /*!< Input data(+ for LF + NULL 1byte). */
    uint8_t wp;                                 /*!< Write pointer.                */
} input_t;
/**
 *  @}
 */ /* End of group DMAC_UART_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DMAC_UART_Private_variables DMAC UART Private Variables
 *  @{
 */
static instance_t instance; /*!< Instance.               */
static Processing proc;     /*!< Processing mode.        */
static input_t    input;    /*!< Input data information. */

#if defined ( __CC_ARM   )      /* RealView Compiler */
    static __align(0x400) uint8_t primary[CFG_DMA_WORK_SIZE]; /*!< DMA work area. */
#elif defined ( __ICCARM__ )    /*IAR Compiler */
    #pragma location="dma_work"
    #pragma data_alignment=1024
    static uint8_t primary[CFG_DMA_WORK_SIZE]; /*!< DMA work area. */
#endif

/**
 *  @}
 */ /* End of group DMAC_UART_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DMAC_UART_Private_fuctions DMAC UART Private Fuctions
 *  @{
 */
static void variable_initialize(void);
static TXZ_Result driver_dma_initialize(void);
static TXZ_Result driver_dma_finalize(void);
static TXZ_Result driver_initialize(void);
static TXZ_Result driver_finalize(void);
static TXZ_Result application_initialize(void);
static TXZ_Result application_finalize(void);
static void timer_interval_handler(uint32_t id);
static void clear_input_data(void);
/*--------------------------------------------------*/
/** 
  * @brief  Variable Initialize
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void variable_initialize(void)
{
    uint32_t i;

    /* Instance. */
    instance.p_uart             = MAIN_NULL;
    instance.t32a.p_instance    = MAIN_NULL;
    instance.timer.init.p_t32a  = MAIN_NULL;
    instance.dma.p_instance     = MAIN_NULL;
    /* Processing mode. */
    proc = PROC_START;
    /* Input data information. */
    clear_input_data();
    /* DMA work area. */
    for (i=0; i<CFG_DMA_WORK_SIZE; i++)
    {
        primary[i] = 0;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  DMA Driver Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_dma_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*--- Construct (Register Allocation) ---*/
    instance.dma.p_instance = TSB_DMAA;

    /*--- Initialize ---*/
    instance.dma.init.baseAdd = (uint32_t)&primary[0];
    if (dma_init(&instance.dma) != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }



    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  DMA Driver Finalize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_dma_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*--- Finalize ---*/
    if (dma_deinit(&instance.dma) != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*--- Destruct (Register Release) ---*/
    /* Register Allocation */
    instance.dma.p_instance = MAIN_NULL;

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (p_t32a->p_instance != MAIN_NULL)
        {
            if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
    }
    /*----------------------*/
    /* DMA                  */
    /*----------------------*/
    if (driver_dma_initialize() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver initialize will process within USB UART. */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Finalize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver finalize has done within USB UART.    */
    /*----------------------*/
    /* DMA                  */
    /*----------------------*/
    if (driver_dma_finalize() != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Finalize ---*/
        /* no processing */
        /*--- Destruct ---*/
        /* Register Release */
        p_t32a->p_instance = MAIN_NULL;
    }
    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver finalize will process within BSP.     */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    {
        timer_t *p_timer = &instance.timer;

        p_timer->init.id       = (uint32_t)&instance.timer;
        p_timer->init.p_t32a   = &instance.t32a;
        p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
        p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
        p_timer->init.interval = 1000;
        p_timer->init.handler  = timer_interval_handler;
        timer_initialize(p_timer);
    }
    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx are same channel */
    {
        uart_io_dma_initial_setting_t param;

        param.id         = 0;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = bsp_get_usb_uart_tx_baudrate(BSP_USB_UART_0);
        param.dma.p_dma  = &instance.dma;
        param.dma.tx     = bsp_get_usb_uart_tx_dma_ch(BSP_USB_UART_0);
        param.dma.rx     = bsp_get_usb_uart_rx_dma_ch(BSP_USB_UART_0);

        param.dma.p_dma->ch[param.dma.rx].cfg.dstInc    = CFG_DMA_ch_dstinc_1byte ;
        param.dma.p_dma->ch[param.dma.rx].cfg.dstSize   = CFG_DMA_ch_dstsize_1byte ;
        param.dma.p_dma->ch[param.dma.rx].cfg.srcInc    = CFG_DMA_ch_srcinc_none ;
        param.dma.p_dma->ch[param.dma.rx].cfg.srcSize   = CFG_DMA_ch_srcsize_1byte ;
        param.dma.p_dma->ch[param.dma.rx].cfg.arbitNum  = CFG_DMA_ch_rpow_001 ;
        param.dma.p_dma->ch[param.dma.rx].cfg.burst     = CFG_DMA_ch_next_useburst ;
    
        param.dma.p_dma->ch[param.dma.tx].cfg.dstInc    = CFG_DMA_ch_dstinc_none ;
        param.dma.p_dma->ch[param.dma.tx].cfg.dstSize   = CFG_DMA_ch_dstsize_1byte ;
        param.dma.p_dma->ch[param.dma.tx].cfg.srcInc    = CFG_DMA_ch_srcinc_1byte ;
        param.dma.p_dma->ch[param.dma.tx].cfg.srcSize   = CFG_DMA_ch_srcsize_1byte ;
        param.dma.p_dma->ch[param.dma.tx].cfg.arbitNum  = CFG_DMA_ch_rpow_001 ;
        param.dma.p_dma->ch[param.dma.tx].cfg.burst     = CFG_DMA_ch_next_useburst ;


        if (param.p_reg != MAIN_NULL)
        {
            instance.p_uart  = uart_io_dma_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Finalize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* USB UART             */
    /*----------------------*/
    if (instance.p_uart != MAIN_NULL)
    {
        uart_io_dma_finalize(instance.p_uart);
    }
    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    timer_finalize(&instance.timer);

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    if (id == (uint32_t)&instance.timer)
    {
        /* Increment Tick. Tick is used to driver's Timeout. */
        hal_inc_tick();
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Clear input data.
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void clear_input_data(void)
{
    uint8_t i;

    for (i=0; i<(UART_IO_DMA_RECEIVE_MAX+2); i++)
    {
        input.data[i] = 0;
    }
    input.wp = 0;
}
/**
 *  @}
 */ /* End of group DMAC_UART_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DMAC_UART_Exported_functions DMAC UART Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /*--- variable    ---*/
    variable_initialize();
    /*--- driver      ---*/
    if (driver_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    /*--- application ---*/
    if (application_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(proc != PROC_ERROR)
    {
        switch(proc)
        {
        /*--- Start ---*/
        case PROC_START:
            {
                /* DMA  */
                NVIC_EnableIRQ(INTDMAATC_IRQn);
                NVIC_EnableIRQ(INTDMAAERR_IRQn);
                /* Timer */
                timer_start(&instance.timer);
                proc = PROC_OUTPUT_MSG;
            }
            break;
        /*--- Stop ---*/
        case PROC_STOP:
            {
                /* DMA  */
                NVIC_DisableIRQ(INTDMAATC_IRQn);
                NVIC_DisableIRQ(INTDMAAERR_IRQn);
                /* Timer */
                timer_stop(&instance.timer);
                proc = PROC_ERROR;
            }
            break;
        /*--- Output "First Message" ---*/
        case PROC_OUTPUT_MSG:
            {
                /* Clear input data.     */
                clear_input_data();
                /* Output first message. */
                if (printf("Input = ") != -1 )
                {
                    proc = PROC_INPUT;
                }
                else
                {
                    proc = PROC_STOP;
                }
            }
            break;
        /*--- Wait to Input ---*/
        case PROC_INPUT:
            {
                int data;

                data = getchar();
                if ((data != -1) && (input.wp < UART_IO_DMA_RECEIVE_MAX+1))
                {
                    input.data[input.wp] = (char)data;
                    input.wp++;
                    switch ((char)data)
                    {
                    /*--- LF: line feed ---*/
                    case '\n':
                        proc = PROC_OUTPUT_ECHO;
                        break;
                    /*--- default ---*/
                    default:
                        /* no process */
                        break;
                    }
                }
                else
                {
                    proc = PROC_OUTPUT_ERR;
                }
            }
            break;
        /*--- Output "Echo Back Data" ---*/
        case PROC_OUTPUT_ECHO:
            {
                if (printf(" Echo = %s", (const char *)&input.data[0]) != -1 )
                {
                    proc = PROC_OUTPUT_MSG;
                }
                else
                {
                    proc = PROC_STOP;
                }
            }
            break;
        /*--- Output "Input Error Message" ---*/
        case PROC_OUTPUT_ERR:
            {
                if (printf("\n Input Error !!\n") != -1 )
                {
                    proc = PROC_OUTPUT_MSG;
                }
                else
                {
                    proc = PROC_STOP;
                }
            }
            break;
        /*--- Error ---*/
        case PROC_ERROR:
            /* no processing */
            break;
        /*--- Default ---*/
        default:
            proc = PROC_ERROR;
            break;
        }
    }
    /*>>> System Error. Please Debug. */
    /*----------------------*/
    /* Finalize             */
    /*----------------------*/
    /*--- application ---*/
    if (application_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*--- driver      ---*/
    if (driver_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*----------------------*/
    /* BSP finalize         */
    /*----------------------*/
    bsp_finalize();
#ifdef DEBUG
    assert_failed(MAIN_NULL, 0);
#endif

    return (0);
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer.
  * @param  timer
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    /*--- 1ms timer 0 ---*/
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  DMA transfer done IRQ.
  * @param  ch :DMA Channel
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_dmac_done(uint32_t ch)
{
    /* Call the DMA driver's handler. */
    dma_irq_handler(&instance.dma, ch);
}

/*--------------------------------------------------*/
/** 
  * @brief  DMA transfer error IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_dmac_err(void)
{
    /* Call the DMA driver's handler. */
    dma_error_irq_handler(&instance.dma);
}


#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __nop();
    }
}
#endif
/**
 *  @}
 */ /* End of group DMAC_UART_Exported_functions */

/**
 *  @}
 */ /* End of group DMAC_UART */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
