/** 
 *******************************************************************************
 * @file    main.c
 * @brief   DNF Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_sw.h"
#include "bsp_led.h"
#include "bsp_timer.h"
#include "dnf.h"
/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup DNF_Sample_Appli DNF Sample Appli
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DNF_Private_macro DNF Private Macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group DNF_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DNF_Private_define DNF Private Define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group DNF_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DNF_Private_define DNF Private Define
 *  @{
 */
/** 
 *  @defgroup SWPollingConfiguration SW Polling Time Configuration
 *  @brief    SW Polling Time(ms).
 *  @{
 */
#define CFG_SW_POLLING_TIME     ((uint32_t)(100))
/** 
 *  @}
 */ /* End of group SWPollingConfiguration */

#define MAIN_NULL       ((void *)0)     /*!< NULL. */

/** 
 *  @}
 */ /* End of group DNF_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DNF_Private_define DNF Private Define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group DNF_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DNF_Private_typedef DNF Private Typedef
 *  @{
 */

/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
    t32a_t      t32a_timer;                /*!< Application      :T32A.     */
    sw_t        sw[(uint32_t)BSP_PSW_MAX];  /*!< Application :Push SW            */
    led_t       led[(uint32_t)BSP_LED_MAX]; /*!< Application :LED                */
    timer_t     timer;             /*!< Application :Timer (ims timer). */
    gpio_t      gpio_obj;            /*!< Application :GPIO. */
} instance_t;
/**
 *  @}
 */ /* End of group DNF_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DNF_Private_variables DNF Private Variables
 *  @{
 */

static instance_t instance; /*!< Instance.                    */
BSPExInt status=BSP_EXINT_0;

/**
 *  @}
 */ /* End of group DNF_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DNF_Private_fuctions DNF Private Fuctions
 *  @{
 */
static TXZ_Result application_initialize(void);
static void timer_interval_handler(uint32_t id);
static TXZ_Result timer_driver_initialize(void);

/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    int32_t i;
   /*----------------------*/
   /* SW                   */
   /*----------------------*/
   for (i=0; i<(int32_t)BSP_PSW_MAX; i++)
   {
       sw_1ms_timer_handler(&instance.sw[i]);
   }
}
/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
   TXZ_Result result = TXZ_SUCCESS;
    /*----------------------*/
    /* LED                  */
    /*----------------------*/
    {
        uint32_t i;
        led_t *p_led = MAIN_NULL;

        for (i=0; i<(int32_t)BSP_LED_MAX; i++)
        {
            p_led = &instance.led[i];

            p_led->init.id                  = (uint32_t)p_led;
            p_led->init.p_gpio              = bsp_get_gpio_instance_address();
            p_led->init.port.group          = bsp_get_gpio_group_led((BSPLed)(i));
            p_led->init.port.num            = bsp_get_gpio_num_led((BSPLed)(i));
            p_led->init.blink.func          = TXZ_DISABLE;
            p_led->init.blink.interval.on   = 0;
            p_led->init.blink.interval.off  = 0;
            p_led->init.state           = LED_STATE_OFF;
            led_initialize(p_led);
        }
    }
    /*----------------------*/
    /* PSW                  */
    /*----------------------*/
    {
        uint32_t i;
        sw_t *p_sw = MAIN_NULL;

        for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
        {
            p_sw = &instance.sw[i];

            p_sw->init.id                   = (uint32_t)p_sw;
            p_sw->init.p_gpio               = bsp_get_gpio_instance_address();
            p_sw->init.port.group           = bsp_get_gpio_group_psw((BSPPsw)(i));
            p_sw->init.port.num             = bsp_get_gpio_num_psw((BSPPsw)(i));
            p_sw->init.polling.func         = TXZ_DISABLE;
            p_sw->init.polling.interval     = CFG_SW_POLLING_TIME;
            p_sw->init.chattering.func      = TXZ_DISABLE;
            p_sw->init.chattering.count     = 0;
            p_sw->init.handler              = MAIN_NULL;
            sw_initialize(p_sw);
        }
    }
    return (result);
}
/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result timer_driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a_timer;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (p_t32a->p_instance != MAIN_NULL)
        {
            if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    {
        timer_t *p_timer = &instance.timer;

        p_timer->init.id       = (uint32_t)&instance.timer;
        p_timer->init.p_t32a   = &instance.t32a_timer;
        p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
        p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
        p_timer->init.interval = 1000;
        p_timer->init.handler  = timer_interval_handler;
        timer_initialize(p_timer);
    }
    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver initialize will process within USB UART. */

    return (result);
}

/**
 *  @}
 */ /* End of group DNF_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup DNF_Exported_functions DNF Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /* Timer driver initialization */
    if (timer_driver_initialize() != TXZ_SUCCESS)
    {
    }
    /*--- application ---*/
    if (application_initialize() != TXZ_SUCCESS)
    {
    }
    timer_start(&instance.timer);
    /*----------------------*/
    /* DNF Register Set             */
    /*----------------------*/
    {
        /* Clock Setting (Noize Filter Time Setting) */
        REG_DNFCKCR_set((TSB_DNF_TypeDef *)bsp_get_dnf_reg(BSP_DNF_0), REG_DNF_NFCKCR_NFCKS_CLOCK_2);
    }
    /*---------------------------*/
    /* External Interrupt Enable */
    /*---------------------------*/
    bsp_irq_dnf_enable();
    /*----------------------*/
    /* DNF Enable Register Set   */
    /*----------------------*/
    REG_DNFENCR_enable((TSB_DNF_TypeDef *)bsp_get_dnf_reg(BSP_DNF_0), bsp_get_dnf_int_num(BSP_DNF_0));
    /*------------------------*/
    /* External Interrupt Enable      */
    /*------------------------*/
    NVIC_EnableIRQ(bsp_get_dnf_nvic(BSP_DNF_0));
    while(1){
        __NOP();
    }
}
/*--------------------------------------------------*/
/** 
  * @brief  External Interrupt(INT09) IRQ.
  * @param  exi : DNF status.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_exint(BSPExInt exi)
{
    led_t *p_led = MAIN_NULL;
    if(exi == BSP_EXINT_1){
        p_led = &instance.led[0];
        led_turn_on(p_led);
    }else{
        p_led = &instance.led[0];
        led_turn_off(p_led);
    }
    bsp_irq_dnf_enable();
}

/*--------------------------------------------------*/
/** 
  * @brief  1ms Timer.
  * @param  timer :1ms Timer.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    default:
        /* no processing */
        break;
    }
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __nop();
    }
}
#endif
/**
 *  @}
 */ /* End of group DNF_Exported_functions */

/**
 *  @}
 */ /* End of group DNF_Sample_Appli */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
