/** 
 *******************************************************************************
 * @file    main.c
 * @brief   EI2C MASTER and SLAVE Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif
/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_uart_io.h"
#include "bsp_ei2c.h"
#include "ei2c.h"

/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup EI2C_MasterSlave EI2C_MasterSlave Sample Appli
 *  @{
 */

/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MasterSlave_Private_macro EI2C_MasterSlave Private Macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group EI2C_MasterSlave_Private_macro */

/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MasterSlave_Private_define EI2C_MasterSlave Private Define
 *  @{
 */

/** 
 *  @}
 */ /* End of group EI2C_MasterSlave_Private_define */

/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MasterSlave_Private_define EI2C_MasterSlave Private Define
 *  @{
 */

#define MAIN_NULL       ((void *)0)         /*!< Null Pointer.  */
#define SELF_ADDR       ((uint8_t)0x60)     /*!< Self address for test.        */
#define SLAVE_ADDR      ((uint8_t)0x60)     /*!< Slave address for test.       */
#define SUB_ADDR        ((uint16_t)0x0000)     /*!< Sub address for test.      */
#define PARAM_A_SIZE_MAX (0x04)             /*!< Parameter A Size Max          */
#define PARAM_B_SIZE_MAX (0x02)             /*!< Parameter B Size Max          */
#define PARAM_B_DATA_MAX (0x04)             /*!< Parameter B Size Max          */
#define INIT_RNUM       (2)                 /*!< Init RNUM                     */
#define DATA_LENGTH     (16)                /*!< Length of base data.          */
#define SLAVE_CONTINUE  (10)                /*!< Number of keeping slave mode. */
#define TEST_TIMEOUT    (16000000)          /*!< for verification.             */

#define CMD_MASTER_WRITE  (0x00)            /*!< Master writing                */
#define CMD_MASTER_READ   (0x01)            /*!< Master reading                */
/*----------------------*/
/*  UART Setting        */
/*----------------------*/
/** 
 *  @defgroup UART_Setting UART Setting
 *  @{
 */
#define USER_ID_REMOTE  ((uint32_t)0)   /*!< User ID :Remote Control (Any value) */
#define USER_ID_UART    ((uint32_t)1)   /*!< User ID :USB UART (Any value) */
#define CFG_UART_BAUDRATE               ((uint32_t)115200)  /*!< Boudrate(bps) */
/** 
 *  @}
 */ /* End of group UART_Setting */

/** 
 *  @}
 */ /* End of group EI2C_MasterSlave_Private_define */

/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MasterSlave_Private_define EI2C_MasterSlave Private Define
 *  @{
 */

/*----------------------------------*/
/** 
 * @brief  ei2c mode.
*/
/*----------------------------------*/
typedef enum
{
    MODE_INIT,
    MODE_MASTER,
    MODE_SLAVE
} Mode;

/*----------------------------------*/
/** 
 * @enum   Processing
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_IDLE,              /*!< Idle.                        */
    PROC_MASTER_START,      /*!< Master start                 */
    PROC_COMMAND_MSG,       /*!< Output first message.        */
    PROC_INPUT,             /*!< Wait to input.               */
    PROC_COMMAND_PARSE,     /*!< Output Echo back data.       */
    PROC_COMMAND_ERR,       /*!< Output command error message.  */
    PROC_PARAMETER_ERR,     /*!< Output parameter error message.  */
    PROC_COMMAND_END,       /*!< Command end.                 */
    PROC_MASTER_WRITE,      /*!< Master write.                */
    PROC_MASTER_WRITE_READ, /*!< Master write and read.       */
    PROC_MASTER_END,        /*!< Master end.                  */
    PROC_SLAVE_START,       /*!< Slave start.                 */
    PROC_SLAVE_RECEIVE,     /*!< Slave receive.               */
    PROC_SLAVE_END,         /*!< Slave end.                   */
    PROC_ERROR              /*!< Error.                       */
} Processing;

/** 
 *  @}
 */ /* End of group EI2C_MasterSlave_Private_define */

/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MasterSlave_Private_typedef EI2C_MasterSlave Private Typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @struct input_t
 * @brief  Input data information.
*/
/*----------------------------------*/
typedef struct
{
    char    data[(UART_IO_RECEIVE_MAX+1)];  /*!< Input data(+ for NULL 1byte). */
    uint8_t wp;                             /*!< Write pointer.                */
} input_t;
/** 
 *  @}
 */ /* End of group EI2C_MasterSlave_Private_typedef */

/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MasterSlave_Private_variables EI2C_MasterSlave Private Variables
 *  @{
 */

static Processing proc;                         /*!< Processing mode.         */
static input_t  input;                          /*!< Input data information.  */

static bsp_ei2c_t ei2c_obj;                     /*!< EI2C driver instance.    */
static uint32_t self_address;                   /*!< EI2C Self Address.       */
static uint32_t slave_address;                  /*!< EI2C Slave Address.      */
static volatile uint32_t transfer_event;        /*!< EI2C Transfer Event      */
static volatile Mode      mode;                 /*!< EI2C mode.               */


static int32_t data_form2;                      /*!< Length of 2nd Data form. */
static int32_t slave_continue;                  /*!< Counts of the slave.     */
static int32_t timeout;                         /*!< Timeout work.            */

static uint32_t master_rx_length;               /*!< Length of Master Rx Data. */
static uint8_t  master_rx_data[DATA_LENGTH];    /*!< Rx Data of Master.        */
static uint32_t master_tx_length;               /*!< Length of Master Tx Data. */
static uint8_t  master_tx_data[DATA_LENGTH];    /*!< Tx Data of Master.        */

static uint32_t slave_rx_length;                /*!< Length of Slave Rx Data.  */
static uint8_t  slave_rx_data[DATA_LENGTH];     /*!< Rx Data of Slave.         */
static uint32_t slave_tx_length;                /*!< Length of Slave Tx Data.  */
static uint8_t  slave_tx_data[DATA_LENGTH];     /*!< Tx Data of Slave.         */

static uint8_t  MEM[DATA_LENGTH] ;              /*!< Memory */

/**
 *  @}
 */ /* End of group EI2C_MasterSlave_Private_variables */

/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MasterSlave_Private_functions EI2C_MasterSlave Private Functions
 *  @{
 */
static TXZ_Result application_initialize(void);
static void configuration(void);
static void clear_input_data(void);
static void clear_master_rx_data(void);
static void clear_master_tx_data(void);
static void clear_slave_rx_data(void);
static void clear_slave_tx_data(void);
static uint8_t get_param_hex_byte(char *p_src);
static void get_param_hex_array(uint8_t *p_dst, char *p_src, int len);
static void ts_ei2c_initialize(void);
static void ts_ei2c_set_master_mode(void);
static void ts_ei2c_set_slave_mode(void);
static void ts_ei2c_master_non_blocking_write_read(void);
static void ts_ei2c_master_non_blocking_write(void);
static void ts_ei2c_slave_non_blocking_receive(void);
static void ts_ei2c_slave_abort_check(void);

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
   TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = USER_ID_UART;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }

        param.boudrate   = (uint32_t) CFG_UART_BAUDRATE ;
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result  = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    return (result);
}
/*--------------------------------------------------*/
/** 
  * @brief  Configuration
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void configuration(void)
{
    mode            = MODE_INIT;

    self_address    = (uint32_t)SELF_ADDR;
    slave_address   = (uint32_t)SLAVE_ADDR;


    data_form2      = 2;
    slave_continue  = SLAVE_CONTINUE;
}

/*--------------------------------------------------*/
/** 
  * @brief  Clear input data
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void clear_input_data(void)
{
    uint8_t i;

    for (i = 0; i < (UART_IO_RECEIVE_MAX+1); i++)
    {
        input.data[i] = 0;
    }
    input.wp = 0;
}

/*--------------------------------------------------*/
/** 
  * @brief  clear_master_rx_data
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void clear_master_rx_data(void)
{
    int i;

    for (i = 0; i < DATA_LENGTH; i++)
    {
        master_rx_data[i] = 0;
    }
    master_rx_length = 0;
}

/*--------------------------------------------------*/
/** 
  * @brief  clear_master_tx_data
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void clear_master_tx_data(void)
{
    int i;

    for (i = 0; i < DATA_LENGTH; i++)
    {
        master_tx_data[i] = i;
    }
    master_tx_length = 0;
}

/*--------------------------------------------------*/
/** 
  * @brief  clear_slave_rx_data
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void clear_slave_rx_data(void)
{
    int i;

    for (i = 0; i < DATA_LENGTH; i++)
    {
        slave_rx_data[i] = 0;
    }
    slave_rx_length = 0;
}

/*--------------------------------------------------*/
/** 
  * @brief  clear_slave_tx_data
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void clear_slave_tx_data(void)
{
    int i;

    for (i = 0; i < DATA_LENGTH; i++)
    {
        slave_tx_data[i] = (i | 0x80);
        MEM[i] = slave_tx_data[i] ;
    }
    slave_tx_length = 0;
}

/*--------------------------------------------------*/
/** 
  * @brief  Getting parameter from input data
  * @param  p_src       :Input source address.
  * @retval 1byte data
  * @note   -
  */
/*--------------------------------------------------*/
static uint8_t get_param_hex_byte(char *p_src)
{
    uint32_t data = 0;
    char str[2+1] = {'\0'};

    str[0]= p_src[0];
    str[1]= p_src[1];
    data = strtol(str, 0, 16);
    data &= 0x000000FF;
    return ((uint8_t)data);
}

/*--------------------------------------------------*/
/** 
  * @brief  Getting parameter from input data
  * @param  p_dst       :Destination address.
  * @param  p_src       :Source address.
  * @param  len         :Source length.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void get_param_hex_array(uint8_t *p_dst, char *p_src, int len)
{
    int i;

    for (i = 0; (((i * 2) < (len - 1)) && (i < DATA_LENGTH)); i++)
    {
        p_dst[i] = get_param_hex_byte(p_src + (i * 2));
    }
}

/*--------------------------------------------------*/
/**
 *   * @brief  Parameter check
 *     * @param  p_src       :Source address.
 *       * @param  len         :Source length.
 *         * @retval -
 *           * @note   -
 *             */
/*--------------------------------------------------*/
static TXZ_Result check_param_hex_array(char *p_src, int len)
{
    int i;
    char c;
    TXZ_Result res = TXZ_SUCCESS ;

    for (i = 0; ((i < (len - 1)) && (i < DATA_LENGTH)); i++)
    {
        c = p_src[i] ;
        if(((c>=0x30)&&(c<=0x39))||((c>=0x41)&&(c<=0x46))||((c>=0x61)&&(c<=0x66)))   /* 0-9,A-F,a-f */
        {
        }
        else
        {
            res = TXZ_ERROR ;
        }
    }
    return res ;
}

/*--------------------------------------------------*/
/**
  * @brief     Initializing of I2C Driver
  * @param     -
  * @retval    -
  */
/*--------------------------------------------------*/
static void ts_ei2c_initialize(void)
{
    mode = MODE_MASTER;

    /* ei2c frequency setting */

    ei2c_obj.ei2c.init.clock.sck = EI2CSCL_FRQ_FM;
    ei2c_obj.ei2c.init.cr0.nfsel = EI2CxCR0_NFSEL_DIG;
    ei2c_obj.ei2c.init.cr0.dnf = EI2CxCR0_DNF_2PRSCK;

    /* Initializing of EI2C */
    if (ei2c_init(&ei2c_obj, EI2C_CFG_PORT_SDA, EI2C_CFG_PORT_SCL) == TXZ_ERROR)
    {
        /* fatal error */
        printf("ei2c init error!!\n");
    }
}

/*--------------------------------------------------*/
/**
  * @brief     Changing to Master Mode
  * @param     -
  * @retval    -
  */
/*--------------------------------------------------*/
static void ts_ei2c_set_master_mode(void)
{
    ei2c_slave_mode(&ei2c_obj, 0);

    mode = MODE_MASTER;
}

/*--------------------------------------------------*/
/**
  * @brief     Changing to Slave Mode
  * @param     -
  * @retval    -
  */
/*--------------------------------------------------*/
static void ts_ei2c_set_slave_mode(void)
{
    ei2c_slave_address1(&ei2c_obj, self_address,1);
    ei2c_slave_mode(&ei2c_obj, 1);

    mode = MODE_SLAVE;
}

/*--------------------------------------------------*/
/**
  * @brief     Writing and Reading by Master Non-Blocking
  * @param     -
  * @retval    -
  */
/*--------------------------------------------------*/
static void ts_ei2c_master_non_blocking_write_read(void)
{
    uint32_t i;

    master_tx_length = 0;
    master_rx_length = 0;

    transfer_event = 0;
    if (ei2c_transfer_asynch(&ei2c_obj, 
                            master_tx_data, 
                            2, 
                            EI2C_NULL, 
                            0, 
                            (int32_t)slave_address, 
                            1
                            ) != TXZ_SUCCESS)
    {
        printf("Reading Error!!\n");
    }
    else
    {
        timeout = TEST_TIMEOUT;
        while (1)
        {
            volatile uint32_t event = transfer_event;   /* asynch information */

            if ((event & EI2C_EVENT_ALL) != 0)
            {
                /* event occured */
                if (event & EI2C_EVENT_TRANSFER_COMPLETE)
                {
                    /* complete */
                    ei2c_acr1_clear(&ei2c_obj);
                    /* ei2c_abort_asynch(&ei2c_obj); */

                    master_tx_length = ei2c_obj.tx_buff.pos;
                    master_rx_length = ei2c_obj.rx_buff.pos;
                }
                else if (event & EI2C_EVENT_TRANSFER_EARLY_NACK)
                {
                    /* early complete */
                    ei2c_abort_asynch(&ei2c_obj);
                    master_tx_length = ei2c_obj.tx_buff.pos;
                    master_rx_length = ei2c_obj.rx_buff.pos;
                    printf("EARLY NACK!!\n");
                }
                else if (event & EI2C_EVENT_ERROR_NO_SLAVE)
                {
                    /* no slave */
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("NO SLAVE!!\n");
                }
                else  /* if (event & I2C_EVENT_ERROR) */
                {
                    /* other error */
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("ERROR!!\n");
                }
                break;
            }
            else
            {
                /* no event */
                if ((timeout--) == 0)
                {
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("time error!!\n");
                    break;
                }
            }
        }
        while (ei2c_active(&ei2c_obj))
        {
            /* checking bus release */
        }
    }

    for(i=0; i<1000; i++) { } ;

    transfer_event = 0;
    if (ei2c_transfer_asynch(&ei2c_obj, 
                            EI2C_NULL, 
                            0, 
                            master_rx_data, 
                            data_form2, 
                            (int32_t)slave_address, 
                            1
                            ) != TXZ_SUCCESS)
    {
        printf("Reading Error!!\n");
    }
    else
    {
        timeout = TEST_TIMEOUT;
        while (1)
        {
            volatile uint32_t event = transfer_event;   /* asynch information */

            if ((event & EI2C_EVENT_ALL) != 0)
            {
                /* event occured */
                if (event & EI2C_EVENT_TRANSFER_COMPLETE)
                {
                    /* complete */
                    ei2c_acr1_clear(&ei2c_obj);


                    master_tx_length = ei2c_obj.tx_buff.pos;
                    master_rx_length = ei2c_obj.rx_buff.pos;
                }
                else if (event & EI2C_EVENT_TRANSFER_EARLY_NACK)
                {
                    /* early complete */
                    ei2c_abort_asynch(&ei2c_obj);
                    master_tx_length = ei2c_obj.tx_buff.pos;
                    master_rx_length = ei2c_obj.rx_buff.pos;
                    printf("EARLY NACK!!\n");
                }
                else if (event & EI2C_EVENT_ERROR_NO_SLAVE)
                {
                    /* no slave */
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("NO SLAVE!!\n");
                }
                else  /* if (event & I2C_EVENT_ERROR) */
                {
                    /* other error */
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("ERROR!!\n");
                }
                break;
            }
            else
            {
                /* no event */
                if ((timeout--) == 0)
                {
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("time error!!\n");
                    break;
                }
            }
        }
        while (ei2c_active(&ei2c_obj))
        {
            /* checking bus release */
        }
    }
}

/*--------------------------------------------------*/
/**
  * @brief     Writing by Master Non-Blocking
  * @param     -
  * @retval    -
  */
/*--------------------------------------------------*/
static void ts_ei2c_master_non_blocking_write(void)
{
    uint32_t i;

    master_tx_length = 0;
    master_rx_length = 0;
    transfer_event = 0;
    /* start condition */
    if (ei2c_transfer_asynch(&ei2c_obj, 
              master_tx_data, 
             2, 
             EI2C_NULL, 
             0, 
            (int32_t)slave_address, 
             1
            ) != TXZ_SUCCESS)
    {
          master_tx_length = 0;
          printf("Writing Error!!\n");
    }
    else
    {
        timeout = TEST_TIMEOUT;
        while (1)
        {
            volatile uint32_t event = transfer_event;    /* asynch information. */
            
            /* event occured */
            if ((event & EI2C_EVENT_ALL) != 0)
            {
                if (event & EI2C_EVENT_TRANSFER_COMPLETE)
                {
                    /* complete */
                    master_tx_length = ei2c_obj.tx_buff.pos;
                    master_rx_length = ei2c_obj.rx_buff.pos;
                }
                else if (event & EI2C_EVENT_TRANSFER_EARLY_NACK)
                {
                    /* early complete */
                    ei2c_abort_asynch(&ei2c_obj);
                    master_tx_length = ei2c_obj.tx_buff.pos;
                    master_rx_length = ei2c_obj.rx_buff.pos;
                    printf("EARLY NACK!!\n");
                }
                else if (event & EI2C_EVENT_ERROR_NO_SLAVE)
                {
                    /* no slave */
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("NO SLAVE!!\n");
                }
                else  /* if (event & I2C_EVENT_ERROR) */
                {
                    /* other error */
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("ERROR!!\n");
                }
                break;
            }
            else
            {
                /* no event */
                if ((timeout--) == 0)
                {
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("time error!!\n");
                    break;
                }
            }
        }
        while (ei2c_active(&ei2c_obj))
        {
            /* checking bus release */
        }
    }

    for(i=0; i<1000; i++) { } ;

    transfer_event = 0;
    if (ei2c_transfer_asynch(&ei2c_obj, 
              &master_tx_data[2], 
             master_tx_data[1], 
             EI2C_NULL, 
             0, 
            (int32_t)slave_address, 
             1
            ) != TXZ_SUCCESS)
    {
          master_tx_length = 0;
          printf("Writing Error!!\n");
    }
    else
    {
        timeout = TEST_TIMEOUT;
        while (1)
        {
            volatile uint32_t event = transfer_event;    /* asynch information. */
            
            /* event occured */
            if ((event & EI2C_EVENT_ALL) != 0)
            {
                if (event & EI2C_EVENT_TRANSFER_COMPLETE)
                {
                    /* complete */
                    master_tx_length = ei2c_obj.tx_buff.pos;
                    master_rx_length = ei2c_obj.rx_buff.pos;
                }
                else if (event & EI2C_EVENT_TRANSFER_EARLY_NACK)
                {
                    /* early complete */
                    ei2c_abort_asynch(&ei2c_obj);
                    master_tx_length = ei2c_obj.tx_buff.pos;
                    master_rx_length = ei2c_obj.rx_buff.pos;
                    printf("EARLY NACK!!\n");
                }
                else if (event & EI2C_EVENT_ERROR_NO_SLAVE)
                {
                    /* no slave */
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("NO SLAVE!!\n");
                }
                else  /* if (event & I2C_EVENT_ERROR) */
                {
                    /* other error */
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("ERROR!!\n");
                }
                break;
            }
            else
            {
                /* no event */
                if ((timeout--) == 0)
                {
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("time error!!\n");
                    break;
                }
            }
        }
        while (ei2c_active(&ei2c_obj))
        {
            /* checking bus release */
        }
    }
}

/*--------------------------------------------------*/
/**
  * @brief     Receiving by Slave non-blocking
  * @param     -
  * @retval    -
  */
/*--------------------------------------------------*/
static void ts_ei2c_slave_non_blocking_receive(void)
{
    TXZ_Result result_mem ;
    uint8_t write_read_mode ;
    uint8_t write_read_size ;
    uint32_t i ;

    slave_rx_length = 0;
    slave_tx_length = 0;

    transfer_event = 0;
    if (ei2c_slave_transfer_asynch(&ei2c_obj, 
                            EI2C_NULL, 
                            0, 
                            slave_rx_data, 
                            2
                            ) != TXZ_SUCCESS)
    {
        printf("slave error!!\n");
    }
    else
    {
        while (1)
        {
            volatile uint32_t event = transfer_event;   /* asynch information */

            if ((event & EI2C_EVENT_ALL) != 0)
            {
                /* event occured */
                if (event & EI2C_EVENT_TRANSFER_COMPLETE)
                {
                    /* complete */
                    ei2c_acr1_clear(&ei2c_obj);
                    /* ei2c_abort_asynch(&ei2c_obj); */

                    slave_tx_length = ei2c_obj.tx_buff.pos;
                    slave_rx_length = ei2c_obj.rx_buff.pos;
                }
                else if (event & EI2C_EVENT_TRANSFER_EARLY_NACK)
                {
                    /* early complete */
                    ei2c_slave_abort_asynch(&ei2c_obj);
                    slave_tx_length = ei2c_obj.tx_buff.pos;
                    slave_rx_length = ei2c_obj.rx_buff.pos;
                    printf("EARLY NACK!!\n");
                }
                else  /* if (event & I2C_EVENT_ERROR) */
                {
                    /* other error */
                    ei2c_slave_abort_asynch(&ei2c_obj);
                    printf("ERROR!!\n");
                }
                break;
            }
            else
            {
                /* no event */
                /* wait forever */
            }
        }   
    }

    ts_ei2c_slave_abort_check() ;
    write_read_mode = slave_rx_data[0] ;
    write_read_size = slave_rx_data[1] ;
    transfer_event = 0;

    result_mem = TXZ_ERROR ;
    if ( write_read_mode == CMD_MASTER_WRITE )           /* Master is writing */
    {
         result_mem = ei2c_slave_transfer_asynch(&ei2c_obj, 
                            EI2C_NULL, 
                            0, 
                            slave_rx_data, 
                            write_read_size
                            ) ;
    }
    else if ( write_read_mode == CMD_MASTER_READ )       /* Master is reading */
    {
         for(i=0;i<write_read_size;i++)
         {
             slave_tx_data[i] = MEM[i] ;
         }
         result_mem = ei2c_slave_transfer_asynch(&ei2c_obj, 
                            slave_tx_data, 
                            write_read_size, 
                            EI2C_NULL, 
                            0
                            ) ;
    }
    else
    {
        result_mem = TXZ_ERROR ;
    }

    if ( result_mem != TXZ_SUCCESS )
    {
        printf("slave error!!\n");
    }
    else
    {
        while (1)
        {
            volatile uint32_t event = transfer_event;   /* asynch information */

            if ((event & EI2C_EVENT_ALL) != 0)
            {
                /* event occured */
                if (event & EI2C_EVENT_TRANSFER_COMPLETE)
                {
                    /* complete */
                    ei2c_acr1_clear(&ei2c_obj);
                    /* ei2c_abort_asynch(&ei2c_obj); */

                    slave_tx_length = ei2c_obj.tx_buff.pos;
                    slave_rx_length = ei2c_obj.rx_buff.pos;

                    if ( write_read_mode == CMD_MASTER_WRITE )           /* Master is writing */
                    {
                        for(i=0;i<write_read_size;i++)
                        {
                            MEM[i] = slave_rx_data[i] ;
                        }
                    }
                }
                else if (event & EI2C_EVENT_TRANSFER_EARLY_NACK)
                {
                    /* early complete */
                    ei2c_slave_abort_asynch(&ei2c_obj);
                    slave_tx_length = ei2c_obj.tx_buff.pos;
                    slave_rx_length = ei2c_obj.rx_buff.pos;
                    printf("EARLY NACK!!\n");
                }
                else  /* if (event & I2C_EVENT_ERROR) */
                {
                    /* other error */
                    ei2c_slave_abort_asynch(&ei2c_obj);
                    printf("ERROR!!\n");
                }
                break;
            }
            else
            {
                /* no event */
                /* wait forever */
            }
        }   
    }
}

/*--------------------------------------------------*/
/**
  * @brief     Checking abort of slave
  * @param     -
  * @retval    -
  */
/*--------------------------------------------------*/
static void ts_ei2c_slave_abort_check(void)
{
    timeout = TEST_TIMEOUT;
    ei2c_acr1_clear(&ei2c_obj);
    while (ei2c_active(&ei2c_obj))
    {
        /* checking bus release */
        if ((timeout--) == 0)
        {
            printf("TimeOut Error\n");
            printf("slave reset!!\n");
            printf("\n");
            ei2c_reset(&ei2c_obj);
            ei2c_slave_address1(&ei2c_obj, self_address,1);
            ei2c_slave_mode(&ei2c_obj, 1);
            break;
        }           
    }
}

/**
 *  @}
 */ /* End of group EI2C_MasterSlave_Private_functions */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MasterSlave_Exported_functions EI2C_MasterSlave Exported Functions
 *  @{
 */

/*--------------------------------------------------*/
/**
  * @brief      Implementation of INTI2Cx_IRQHandler
  * @param      -
  * @retval     -
  */
/*--------------------------------------------------*/
void ts_ei2cst_irq_handler(void)
{
    uint32_t event = 0;

    if (mode == MODE_MASTER)
    {
        event = ei2c_master_st_irq_handler(&ei2c_obj);
    }
    else
    {
        event = ei2c_slave_st_irq_handler(&ei2c_obj);
    }
    if (event != 0)
    {
        transfer_event = event;
    }
}

/*--------------------------------------------------*/
/**
  * @brief      Implementation of INTI2Cx_IRQHandler
  * @param      -
  * @retval     -
  */
/*--------------------------------------------------*/
void ts_ei2ctberbf_irq_handler(void)
{
    uint32_t event = 0;

    if (mode == MODE_MASTER)
    {
        event = ei2c_irq_handler_asynch(&ei2c_obj);
    }
    else if (mode == MODE_SLAVE)
    {
        event = ei2c_slave_irq_handler_asynch(&ei2c_obj);
    }
    else
    {
    }
    if (event != 0)
    {
        transfer_event = event;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    int len = 0;
    int i ;
    int base ;
    
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /* Application initialization */
    application_initialize();  
    configuration();
    clear_slave_rx_data();
    clear_slave_tx_data();
    clear_master_rx_data();
    clear_master_tx_data();

    proc = PROC_MASTER_START;
    ts_ei2c_initialize();

    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(1)
    {
        switch(proc)
        {
        case PROC_MASTER_START:
            printf("\n");
            printf( "%s\n", " ----------------- ");
            printf( "%s\n", "|EI2C master mode |");
            printf( "%s\n", " ----------------- ");
            proc = PROC_COMMAND_MSG;
            break;

        case PROC_COMMAND_MSG:
            {
                /* Clear input data.     */
                clear_input_data();
                /* Output first message. */
                if (printf("command > ") != -1)
                {
                    proc = PROC_INPUT;
                }
                else
                {
                    proc = PROC_ERROR;
                }
            }
            break;

        case PROC_INPUT:
            {
                int data;

                data = getchar();
                if ((data != -1) && (input.wp < UART_IO_RECEIVE_MAX))
                {
                    input.data[input.wp] = (char)data;
                    input.wp++;
                    switch ((char)data)
                    {
                        /*--- LF: line feed ---*/
                        case '\n':
                              len = input.wp;
                              proc = PROC_COMMAND_PARSE;
                              break;
                        /*--- default ---*/
                        default:
                              /* no process */
                              break;
                    }
                }
                else
                {
                    proc = PROC_COMMAND_ERR;
                }
            }
            break;

        case PROC_COMMAND_PARSE:
            if ((strncmp((const char *)&input.data[0], "write", 5)) == 0)
            {
                /* Write Command */
                clear_master_rx_data();
                len -= 1;
                if ((len==5)||(len==6))
                {
                    if((len==6)&&(input.data[5]!=0x20))
                    {
                        proc = PROC_PARAMETER_ERR ;
                        break ;
                    }
                    else
                    {
                        slave_address = 0x60 ;
                        master_tx_data[0] = 0x00 ; /* cmd : 0=write / 1=read */
                        master_tx_data[1] = 0x04 ; /* size */
                        master_tx_data[2] = 0x11 ;
                        master_tx_data[3] = 0x22 ;
                        master_tx_data[4] = 0x33 ;
                        master_tx_data[5] = 0x44 ;

                        data_form2 = 0 ;
                    }
                }
                if (((len>=7)&&((len%2)==1))||(len>(8+PARAM_A_SIZE_MAX*2)))  /* 7,9,11,13,15-- */
                {
                    proc = PROC_PARAMETER_ERR ;
                    break ;
                }
                if((check_param_hex_array(&input.data[6],len-6))==TXZ_ERROR)
                {
                    proc = PROC_PARAMETER_ERR ;
                    break ;
                }
                if (len >= 8)
                {
                    slave_address = get_param_hex_byte(&input.data[6]);
                    if (slave_address >= 128)
                    {
                        proc = PROC_PARAMETER_ERR ;
                        break ;
                     }
                }
                if (len == 8)
                {
                    master_tx_data[0] = 0x00 ; /* cmd : 0=write / 1=read */
                    master_tx_data[1] = 0x04 ; /* size */
                    master_tx_data[2] = 0x11 ;
                    master_tx_data[3] = 0x22 ;
                    master_tx_data[4] = 0x33 ;
                    master_tx_data[5] = 0x44 ;

                    data_form2 = 0 ;
                }
                if ((len>=10)&&((len%2)==0)&&(len<=(8+PARAM_A_SIZE_MAX*2)))          /* 10,12,14,16 */
                {
                    master_tx_data[0] = 0x00 ;      /* cmd : 0=write / 1=read */
                    master_tx_data[1] = (len-8)/2 ; /* size */
                    get_param_hex_array(&master_tx_data[2], &input.data[8], (len - 8));

                    data_form2 = 0 ;
                }
                proc = PROC_MASTER_WRITE;
            }
            else if ((strncmp((const char *)&input.data[0], "read", 4)) == 0)
            {
                /* Read Command */
                clear_master_rx_data();
                len -= 1;
                if ((len==4)||(len==5))
                {
                    if((len==5)&&(input.data[4]!=0x20))
                    {
                        proc = PROC_PARAMETER_ERR ;
                        break ;
                     }
                     else
                     {
                        slave_address = 0x60 ;
                        /* slave_address = 0x61 ; */
                        master_tx_data[0] = 0x01 ; /* cmd : 0=write / 1=read */
                        master_tx_data[1] = INIT_RNUM ; /* size */

                        data_form2 = INIT_RNUM ;
                     }
                }
                if (((len>=6)&&((len%2)==0))||(len>(7+PARAM_B_SIZE_MAX*2)))  /* 6,8,10,12-- */
                {
                    proc = PROC_PARAMETER_ERR ;
                    break ;
                }
                if((check_param_hex_array(&input.data[5],len-5))==TXZ_ERROR)
                {
                    proc = PROC_PARAMETER_ERR ;
                    break ;
                }
                if (len >= 7)
                {
                    slave_address = get_param_hex_byte(&input.data[5]);
                    if (slave_address >= 128)
                    {
                        proc = PROC_PARAMETER_ERR ;
                        break ;
                    }
                }
                if (len == 7)
                {
                    master_tx_data[0] = 0x01 ; /* cmd : 0=write / 1=read */
                    master_tx_data[1] = INIT_RNUM ; /* size */

                    data_form2 = INIT_RNUM ;
                }
                if ((len>=9)&&((len%2)==1)&&(len<=(7+PARAM_B_SIZE_MAX*2)))          /* 9,11, */
                {
                    for(i=0;i<DATA_LENGTH;i++)
                    {
                        master_tx_data[i] = 0 ;
                    }
                    master_tx_data[0] = 0x01 ;      /* cmd : 0=write / 1=read */
                    get_param_hex_array(&master_tx_data[1], &input.data[7], (len-7) ); /* size */


                    data_form2 = 0 ;
                    base = 1 ;
                    for(i=(len-7)/2-1;i>=0;i--)   /* tx[2] + tx[1]*16 */
                    {
                        data_form2 += master_tx_data[i+1] * base ;
                        base *= 16 ;
                    }
                    master_tx_data[1] = data_form2 ;      /* size */
                    master_tx_data[2] = 0x00 ;            /* NULL */

                    if((data_form2==0)||(data_form2>PARAM_B_DATA_MAX)) {
                        proc = PROC_PARAMETER_ERR ;
                        break ;
                    }
                }

                /* Read Command */
                proc = PROC_MASTER_WRITE_READ;
            }
            else if ((strncmp((const char *)&input.data[0], "slave", 5)) == 0)
            {
                /* Slave Mode Command */
                len -= 1;
                if (len >= 8)
                {
                    self_address = get_param_hex_byte(&input.data[6]);
                }
                printf("\n");
                proc = PROC_SLAVE_START;
            }
            else
            {
                proc = PROC_COMMAND_ERR;
            }
            break;

        case PROC_COMMAND_END:
            printf("\n");
            proc = PROC_COMMAND_MSG;
            break;

        case PROC_MASTER_WRITE:
            ts_ei2c_master_non_blocking_write();
            proc = PROC_MASTER_END;
            break;

        case PROC_MASTER_WRITE_READ:



            ts_ei2c_master_non_blocking_write_read();
            proc = PROC_MASTER_END;
            break;

        case PROC_MASTER_END:
            {
                int i;
                printf("master\n");
                printf("sa    %02X\n", slave_address);
                for (i = 0; i < master_tx_length; i++)
                {

                    printf("tx[%d] %02X\n", i, master_tx_data[i+2]);
                }
                for (i = 0; i < master_rx_length; i++)
                {
                    printf("rx[%d] %02X\n", i, master_rx_data[i]);
                }
                proc = PROC_COMMAND_END;
            }
            break;

        case PROC_COMMAND_ERR:
            printf("Command Error!!\n");
            proc = PROC_COMMAND_MSG;
            break;

        case PROC_PARAMETER_ERR:
            printf("Parameter Error!!\n");
            proc = PROC_COMMAND_MSG;
            break;

        case PROC_SLAVE_START:
            printf("\n");
            printf( "%s\n", " ---------------- ");
            printf( "%s\n", "|EI2C slave mode |");
            printf( "%s\n", " ---------------- ");
            ts_ei2c_set_slave_mode();
            proc = PROC_SLAVE_RECEIVE;
            break;

        case PROC_SLAVE_RECEIVE:
            ts_ei2c_slave_abort_check();
            printf("slave\n");
            printf("sa    %02X\n", self_address);
            clear_slave_rx_data();
            ts_ei2c_slave_non_blocking_receive();
            proc = PROC_SLAVE_END;
            break;

        case PROC_SLAVE_END:
            {
                int i;
                for (i = 0; i < slave_rx_length; i++)
                {
                    printf("rx[%d] %02X\n", i, slave_rx_data[i]);
                }
                for (i = 0; i < slave_tx_length; i++)
                {
                    printf("tx[%d] %02X\n", i, slave_tx_data[i]);
                }
            }
            printf("\n");
            
            if (slave_continue != 0) slave_continue--;
            if (slave_continue == 0)
            {
                slave_continue = SLAVE_CONTINUE;
                ts_ei2c_set_master_mode();
                proc = PROC_MASTER_START;
            }
            else
            {

                ts_ei2c_set_slave_mode();

                proc = PROC_SLAVE_RECEIVE;
            }
            break;

        case PROC_ERROR:
            /* System Error. Please Debug. */
#ifdef __DEBUG__
            assert_failed(MAIN_NULL, 0);
#endif
            break;

        default:
            proc = PROC_ERROR;
            break;
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  EI2C IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_i2c(void)
{
   ts_ei2cst_irq_handler();
}

/*--------------------------------------------------*/
/** 
  * @brief  EI2C TBE IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_i2c_tbe(void)
{
    ts_ei2ctberbf_irq_handler();
}

/*--------------------------------------------------*/
/** 
  * @brief  EI2C RBF IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_i2c_rbf(void)
{
    ts_ei2ctberbf_irq_handler();
}
#ifdef __DEBUG__
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
      /*  __NOP(); */
    }
}
#endif

/**
 *  @}
 */ /* End of group EI2C_MasterSlave_Exported_functions */

/**
 *  @} 
 */ /* End of group EI2C_MasterSlave */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
