/** 
 *******************************************************************************
 * @file    main.c
 * @brief   EI2C MultiMaster Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif
/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_uart_io.h"
#include "bsp_sw.h"
#include "bsp_timer.h"
#include "bsp_ei2c.h"

/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup EI2C_MultiMaster EI2C_MultiMaster Sample Appli
 *  @{
 */

/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MultiMaster_Private_macro EI2C_MultiMaster Private Macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group EI2C_MultiMaster_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MultiMaster_Private_define EI2C_MultiMaster Private Define
 *  @{
 */
/* no define */

/** 
 *  @}
 */ /* End of group EI2C_MultiMaster_Private_define */

/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MultiMaster_Private_define EI2C_MultiMaster Private Define
 *  @{
 */

/* #define BOARD_B  */   /*! Board B case. */

#define MAIN_NULL             ((void *)0)             /*!< Null Pointer.            */
#define SAMPLE_ADDRESS        (0x0000)                /*!< Access location address. */
#define EI2C_EEPROM_ADDRESS    ((uint8_t)0xA0)             /*!< Slave Address.           */
#define TEST_TIMEOUT    (16000000)          /*!< for verification.             */
/*----------------------*/
/*  UART Setting        */
/*----------------------*/
/** 
 *  @defgroup UART_Setting UART Setting
 *  @{
 */
#define CFG_UART_BAUDRATE               ((uint32_t)115200)  /*!< Boudrate(bps) */
/** 
 *  @}
 */ /* End of group UART_Setting */

#define USER_ID_REMOTE  ((uint32_t)0)   /*!< User ID :Remote Control (Any value) */
#define USER_ID_UART    ((uint32_t)1)   /*!< User ID :USB UART (Any value) */


/** 
 *  @}
 */ /* End of group EI2C_MultiMaster_Private_define */

/** 
 *  @defgroup SWPollingConfiguration SW Polling Time Configuration
 *  @brief    SW Polling Time(ms).
 *  @{
 */
#define CFG_SW_POLLING_TIME     ((uint32_t)(1))
/** 
 *  @}
 */ /* End of group SWPollingConfiguration */

/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MultiMaster_Private_define EI2C_MultiMaster Private Define
 *  @{
 */

/*----------------------------------*/
/** 
 * @enum   Processing
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_IDLE,              /*!< Idle.                        */
    PROC_OUTPUT_MSG,        /*!< Output first message.        */
    PROC_INPUT,             /*!< Wait to input.               */
    PROC_SW_INPUT,          /*!< SW Input.                    */
    PROC_SW_REPEAT,         /*!< SW Input Repeat              */
    PROC_COMMAND_PARSE,     /*!< Output Echo back data.       */
    PROC_OUTPUT_ERR,        /*!< Output input error message.  */
    PROC_ERROR              /*!< Error.                       */
} Processing;

/** 
 *  @}
 */ /* End of group EI2C_MultiMaster_Private_define */

/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MultiMaster_Private_typedef EI2C_MultiMaster Private Typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @struct input_t
 * @brief  Input data information.
*/
/*----------------------------------*/
typedef struct
{
    char    data[(UART_IO_RECEIVE_MAX+1)];  /*!< Input data(+ for NULL 1byte). */
    uint8_t wp;                             /*!< Write pointer.                */
} input_t;

/** 
 *  @}
 */ /* End of group EI2C_MultiMaster_Private_typedef */

/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
    t32a_t      t32a;                       /*!< Application      :T32A.     */
    sw_t        sw[(uint32_t)BSP_PSW_MAX];  /*!< Application :Push SW            */
    timer_t     timer;                      /*!< Application :Timer (1sec timer). */
} instance_t;

/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MultiMaster_Private_variables EI2C_MultiMaster Private Variables
 *  @{
 */
static instance_t instance;     /*!< Instance.                */
static Processing proc;         /*!< Processing mode.        */
static input_t input;           /*!< Input data information. */
static uint8_t write_data[100];  /*!< Write data information. */
static uint8_t read_data[100];   /*!< Read data information. */
static uint8_t data_write_done = 0;     /*!< Write data Done. */
static bsp_ei2c_t   obj;         /*!< EI2C driver instance.        */

static volatile uint32_t transfer_event;        /*!< EI2C Transfer Event       */
static int32_t data_form1;                      /*!< Length of 1st Data form. */
static int32_t data_form2;                      /*!< Length of 2nd Data form. */
static int32_t timeout;                         /*!< Timeout work.            */

static uint32_t master_rx_length;               /*!< Length of Master Rx Data. */
static uint32_t master_tx_length;               /*!< Length of Master Tx Data. */

/**
 *  @}
 */ /* End of group EI2C_MultiMaster_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MultiMaster_Private_functions EI2C_MultiMaster Private Fuctions
 *  @{
 */
static void clear_data(void);
static TXZ_Result application_initialize(void);
TXZ_Result EEPROM_ei2c_init(void);

/*--------------------------------------------------*/
/**
  * @brief     Initializing of EI2C Driver
  * @param     -
  * @retval    TXZ_SUCCESS :Success.
  * @retval    TXZ_ERROR   :Failure.
  */
/*--------------------------------------------------*/
TXZ_Result EEPROM_ei2c_init(void)
{
    TXZ_Result result = TXZ_ERROR;

    obj.ei2c.init.clock.sck = EI2CSCL_FRQ_FM;
    obj.ei2c.init.cr0.nfsel = EI2CxCR0_NFSEL_DIG;
    obj.ei2c.init.cr0.dnf = EI2CxCR0_DNF_2PRSCK;
    
    /*------------------------------*/
    /*  EI2C Driver Initialize       */
    /*------------------------------*/
    if (ei2c_init(&obj, EI2C_CFG_PORT_SDA, EI2C_CFG_PORT_SCL) == TXZ_ERROR)
    {
       /* fatal error */
       printf("ei2c init error!!\n");  
    }
    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief     Writing and Reading by Master Non-Blocking
  * @param     -
  * @retval    -
  */
/*--------------------------------------------------*/
static void ts_ei2c_master_non_blocking_write_read(void)
{
    transfer_event = 0;
    master_rx_length = 0;
    
    if (ei2c_transfer_asynch(&obj, 
                            write_data, 
                            2, 
                            read_data, 
                            data_form2, 
                            (int32_t)EI2C_EEPROM_ADDRESS, 
                            1
                            ) != TXZ_SUCCESS)
    {
        printf("read error!!\r\n");
    }
    else
    {
        timeout = TEST_TIMEOUT;
        while (1)
        {
            volatile uint32_t event = transfer_event;   /* asynch information */

            if ((event & EI2C_EVENT_ALL) != 0)
            {
                /* event occured */
                if (event & EI2C_EVENT_TRANSFER_COMPLETE)
                {
                    if(obj.rx_buff.pos == 0)
                    {
                       printf("\nReading Error!!\n");
                    }
                    /* complete */
                    master_rx_length = obj.rx_buff.pos;
                    ei2c_abort_asynch(&obj);
                }
                else if (event & EI2C_EVENT_BUS_BUSY)
                {
                    /* bus busy */
                    ei2c_abort_asynch(&obj);
                    printf("\nbus busy error!!\n");
                }
                else if (event & EI2C_EVENT_BUS_ARBITRATION)
                {
                    /* bus arbitration */
                    ei2c_abort_asynch(&obj);
                    printf("\narbitration error!!\n");    
                }
                else
                {
                    /* other error */
                    ei2c_abort_asynch(&obj);
                    printf("\nReading Error!!\n");
                }
                break;
            }
            else
            {
                /* no event */
                if ((timeout--) == 0)
                {
                    ei2c_abort_asynch(&obj);
                    printf("timeout error!!\r\n");
                    break;
                }
            }
        }
        while (ei2c_active(&obj))
        {
            /* checking bus release */
        }
    }   
}

/*--------------------------------------------------*/
/**
  * @brief     Writing by Master Non-Blocking
  * @param     -
  * @retval    -
  */
/*--------------------------------------------------*/
static void ts_ei2c_master_non_blocking_write(void)
{
    transfer_event = 0;
    master_tx_length = 0;
    
    /* start condition */
    if (ei2c_transfer_asynch(&obj, 
             write_data, 
             data_form1, 
             EI2C_NULL, 
             0, 
            (int32_t)EI2C_EEPROM_ADDRESS, 
             1
            ) != TXZ_SUCCESS)
    {
          master_tx_length = 0;
          printf("\nWriting Error!!\n");
    }
    else
    {
        timeout = TEST_TIMEOUT;
        while (1)
        {
            volatile uint32_t event = transfer_event;    /* asynch information. */
            
            /* event occured */
            if ((event & EI2C_EVENT_ALL) != 0)
            {
                if (event & EI2C_EVENT_TRANSFER_COMPLETE)
                {
                    if(obj.tx_buff.pos == 0)
                    {
                       printf("\nWriting Error!!\n");
                    }
                    /* complete */
                    master_tx_length = obj.tx_buff.pos;
                    ei2c_abort_asynch(&obj);
                }
                else if (event & EI2C_EVENT_BUS_BUSY)
                {
                    /* bus busy */
                    ei2c_abort_asynch(&obj);
                    printf("\nbus busy error!!\n");
                }
                else if (event & EI2C_EVENT_BUS_ARBITRATION)
                {
                    /* bus arbitration */
                    ei2c_abort_asynch(&obj);
                    printf("\narbitration error!!\n");    
                }
                else  /* if (event & EI2C_EVENT_ERROR) */
                {
                    /* other error */
                    ei2c_abort_asynch(&obj);
                    printf("\nWriting Error!!\n");
                }
                break;
            }
            else
            {
                /* no event */
                if ((timeout--) == 0)
                {
                    ei2c_abort_asynch(&obj);
                    printf("timeout error!!\r\n");
                    break;
                }
            }
        }
        while (ei2c_active(&obj))
        {
            /* checking bus release */
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize
  * @param  -
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (p_t32a->p_instance != MAIN_NULL)
        {
            if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver initialize will process within USB UART. */

    return (result);
}
/*--------------------------------------------------*/
/** 
  * @brief  Clear input data.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void clear_data(void)
{
    uint8_t i;

    for (i = 0; i < (UART_IO_RECEIVE_MAX+1); i++)
    {
        input.data[i] = 0;
    }
    input.wp = 0;
    
    for (i = 0; i < 100; i++)
    {
        read_data[i] = 0;
        write_data[i] = 0;
    }
}
/*--------------------------------------------------*/
/** 
  * @brief  SW State Change Handler.
  * @param  id    :User ID
  * @param  state :Push SW Port State
  * @return -
  * @retval -
  * @note   Notify that, "State is changed." \n
  *         Call by SW Task \n
  *         If SW is pushed, change the processing mode(@ref Processing).
  */
/*--------------------------------------------------*/
static void sw_state_change_handler(uint32_t id, gpio_pinstate_t state)
{
    /* Check SW Push(= GPIO_PIN_RESET) */
    if (state == GPIO_PIN_RESET)
    {
        uint32_t i;

        for (i=0; (i < (uint32_t)BSP_PSW_MAX) && (proc != PROC_ERROR); i++)
        {
            if (id == (uint32_t)&instance.sw[i])
            {
                /* Select Low Power Mode */
                switch ((BSPPsw)i)
                {
                case BSP_PSW_1:
                    if(data_write_done == 0){
                        proc = PROC_SW_INPUT;
                    }
                    break;
                default:
                    /* no processing */
                    break;
                }
            }
        }
    }
}
/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    int32_t i;
    /*----------------------*/
    /* SW                   */
    /*----------------------*/
    for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
    {
        sw_1ms_timer_handler(&instance.sw[i]);
    }
     
}
/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
   TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    {
        timer_t *p_timer = &instance.timer;

        p_timer->init.id       = (uint32_t)&instance.timer;
        p_timer->init.p_t32a   = &instance.t32a;
        p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
        p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
        p_timer->init.interval = 1000;
        p_timer->init.handler  = timer_interval_handler;
        timer_initialize(p_timer);
    }
    /*----------------------*/
    /* PSW                  */
    /*----------------------*/
    {
        uint32_t i;
        sw_t *p_sw = MAIN_NULL;

        for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
        {
            p_sw = &instance.sw[i];

            p_sw->init.id                   = (uint32_t)p_sw;
            p_sw->init.p_gpio               = bsp_get_gpio_instance_address();
            p_sw->init.port.group           = bsp_get_gpio_group_psw((BSPPsw)(i));
            p_sw->init.port.num             = bsp_get_gpio_num_psw((BSPPsw)(i));
            p_sw->init.polling.func         = TXZ_ENABLE;
            p_sw->init.polling.interval     = CFG_SW_POLLING_TIME;
            p_sw->init.chattering.func      = TXZ_DISABLE;
            p_sw->init.chattering.count     = 0;
            p_sw->init.handler              = sw_state_change_handler;
            sw_initialize(p_sw);
        }
    }
    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = USER_ID_UART;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = (uint32_t)CFG_UART_BAUDRATE;
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result  = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief      Implementation of INTI2Cx_IRQHandler
  * @param      -
  * @retval     -
  */
/*--------------------------------------------------*/
void ts_ei2cst_irq_handler(void)
{
    uint32_t event = 0;

    event = ei2c_master_st_irq_handler(&obj);

    if (event != 0)
    {
        transfer_event = event;
    }
}

/*--------------------------------------------------*/
/**
  * @brief      Implementation of INTI2Cx_IRQHandler
  * @param      -
  * @retval     -
  */
/*--------------------------------------------------*/
void ts_ei2ctberbf_irq_handler(void)
{
    uint32_t event = 0;

    event = ei2c_irq_handler_asynch(&obj);

    if (event != 0)
    {
        transfer_event = event;
    }
}
/**
 *  @}
 */ /* End of group EI2C_MultiMaster_Private_functions */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_MultiMaster_Exported_functions EI2C_MultiMaster Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    int num;
    int len = 0;

    /*----------------------*/
    /* Set Processing Mode  */
    /*----------------------*/
    proc = PROC_OUTPUT_MSG;
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    if (driver_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    bsp_initialize();
    /* Application initialization */
    application_initialize();  
    EEPROM_ei2c_init();
    timer_start(&instance.timer);
    
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(1)
    {
        switch(proc)
        {
        /*--- Output "First Message" ---*/
        case PROC_OUTPUT_MSG:
            {
                /* Clear input data.     */
                clear_data();
                /* Output first message. */
                if (printf("command > ") != -1)
                {
                    proc = PROC_INPUT;
                }
                else
                {
                    proc = PROC_ERROR;
                }
            }
            break;
        /*--- Wait to Input ---*/
        case PROC_INPUT:
            {   
                int i;
                /*----------------------*/
                /* SW                   */
                /*----------------------*/
                for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
                {
                    sw_task(&instance.sw[i]);
                }
            }
            break;
        /*--- Wait to Input ---*/
        case PROC_SW_INPUT:
            {   
                int i;
                char command_data[] = "write toshibaABCDEFGHIJKLMNOPQRST";
                
                /*----------------------*/
                /* SW                   */
                /*----------------------*/
                for(i=0;i<sizeof(command_data);++i){
                   input.data[i] = command_data[i];
                }
                len = sizeof(command_data) + 1;
                proc = PROC_COMMAND_PARSE;
            }
            break;
        /*--- Output "Echo Back Data" ---*/
        /*--- Wait to Input ---*/
        case PROC_SW_REPEAT:
            {   
                int i;
                char command_data[] = "read";
                
                for(i=0;i<sizeof(command_data);++i){
                   input.data[i] = command_data[i];
                }
                len = sizeof(command_data) + 1;
                /*----------------------*/
                /* SW                   */
                /*----------------------*/
                {
                    sw_t *p_instance = &instance.sw[0];
                    gpio_pinstate_t crt = GPIO_PIN_RESET;
                    
                    
                    while(crt == GPIO_PIN_RESET){
                        if(gpio_read_bit(
                                            p_instance->init.p_gpio, 
                                            p_instance->init.port.group, 
                                            p_instance->init.port.num, 
                                            GPIO_Mode_DATA, 
                                            &crt
                        ) == TXZ_SUCCESS)
                        {
                        }
                    }
                    while(crt == GPIO_PIN_SET){
                        if(gpio_read_bit(
                                            p_instance->init.p_gpio, 
                                            p_instance->init.port.group, 
                                            p_instance->init.port.num, 
                                            GPIO_Mode_DATA, 
                                            &crt
                        ) == TXZ_SUCCESS)
                        {
                        }
                    }
                }
                
                
                /* READ Command */
              #ifndef BOARD_B
                /* for the time being */
                write_data[0] = (uint8_t)(SAMPLE_ADDRESS >> 8);
                write_data[1] = (uint8_t)(SAMPLE_ADDRESS);
                data_form2 = data_form1;
              #else
                /* for the time being */
                write_data[0] = (uint8_t)((SAMPLE_ADDRESS+4) >> 8);
                write_data[1] = (uint8_t)(SAMPLE_ADDRESS+4);
                data_form2 = data_form1-4;
              #endif
                ts_ei2c_master_non_blocking_write_read();
                if (master_rx_length == data_form2)
                {
                    for (i = master_rx_length; i < 10; i++)
                    {
                         read_data[i] = 0;
                    }
                    printf("read data > ");
                    printf("%s\n", (const char*)&read_data[0]);
                }
                proc = PROC_SW_REPEAT;
                printf("\n");
                break;
            }
        /*--- Output "Echo Back Data" ---*/
        case PROC_COMMAND_PARSE:
            {
                int i;

                if ((strncmp((const char *)&input.data[0], "write", 5)) == 0)
                {
                    /* WRITE Command */
                    if (len >= 9)
                    {
                        printf("write data > ");
                        num = (len - 6);
                        write_data[0] = (uint8_t)(SAMPLE_ADDRESS >> 8);
                        write_data[1] = (uint8_t)SAMPLE_ADDRESS;
                        for (i = 2; (i < (num+2) && (i < 64)); i++)
                        {
                            write_data[i] = (uint8_t)(input.data[6 + i - 2]);
                        }
                        data_form1 = i;
                        ts_ei2c_master_non_blocking_write();
                        if ( master_tx_length == data_form1)
                        {
                            printf("%s\n", (const char *)&write_data[2]);
                            data_write_done = 1;
                        }
                        proc = PROC_SW_REPEAT;
                        printf("\n");
                    }
                    else
                    {
                        proc = PROC_OUTPUT_ERR;
                    }
                    break;
                }
            }
            break;
                
        /*--- Output "Input Error Message" ---*/
        case PROC_OUTPUT_ERR:
            printf("? \n");
            proc = PROC_OUTPUT_MSG;
            break;
        /*--- Error ---*/
        case PROC_ERROR:
            /* System Error. Please Debug. */
#ifdef DEBUG
            assert_failed(MAIN_NULL, 0);
#endif
            {
                int i;
                
                for(i = 0; i < 10000000; ++i){
                    __NOP();
                }
                proc = PROC_SW_REPEAT;
            }
            break;
        /*--- Default ---*/
        default:
            proc = PROC_ERROR;
            break;
        }
    }
}


/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_rx_irq_handler();
        break;
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_tx_irq_handler();
        break;
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_err_irq_handler();
        break;
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  1ms Timer.
  * @param  timer :1ms Timer.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  EI2C IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_i2c(void)
{
   ts_ei2cst_irq_handler();
}

/*--------------------------------------------------*/
/** 
  * @brief  EI2C TBE IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_i2c_tbe(void)
{
    ts_ei2ctberbf_irq_handler();
}

/*--------------------------------------------------*/
/** 
  * @brief  EI2C RBF IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_i2c_rbf(void)
{
    ts_ei2ctberbf_irq_handler();
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
     /*   __NOP(); */
    }
}
#endif

/**
 *  @}
 */ /* End of group EI2C_MultiMaster_Exported_functions */


/**
 *  @} 
 */ /* End of group EI2C_MultiMaster */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
