/** 
 *******************************************************************************
 * @file    main.c
 * @brief   EI2C SingleMaster Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif
/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_uart_io.h"
#include "bsp_ei2c.h"

/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup EI2C_SingleMaster EI2C_SingleMaster Sample Appli
 *  @{
 */

/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_SingleMaster_Private_macro EI2C_SingleMaster Private Macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group EI2C_SingleMaster_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_SingleMaster_Private_define EI2C_SingleMaster Private Define
 *  @{
 */
/* no define */

/** 
 *  @}
 */ /* End of group EI2C_SingleMaster_Private_define */

/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_SingleMaster_Private_define EI2C_SingleMaster Private Define
 *  @{
 */

#define MAIN_NULL       ((void *)0)         /*!< Null Pointer.            */
#define SAMPLE_ADDRESS  (0x0000)            /*!< Access location address. */
#define EI2C_EEPROM_ADDRESS    ((uint8_t)0xA0)           /*!< Slave Address.    */
#define TEST_TIMEOUT    (16000000)          /*!< for verification.             */
/*----------------------*/
/*  UART Setting        */
/*----------------------*/
/** 
 *  @defgroup UART_Setting UART Setting
 *  @{
 */
#define CFG_UART_BAUDRATE               ((uint32_t)115200)  /*!< Boudrate(bps) */
/** 
 *  @}
 */ /* End of group UART_Setting */

#define USER_ID_REMOTE  ((uint32_t)0)   /*!< User ID :Remote Control (Any value) */
#define USER_ID_UART    ((uint32_t)1)   /*!< User ID :USB UART (Any value) */


/** 
 *  @}
 */ /* End of group EI2C_SingleMaster_Private_define */

/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_SingleMaster_Private_define EI2C_SingleMaster Private Define
 *  @{
 */

/*----------------------------------*/
/** 
 * @enum   Processing
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_IDLE,              /*!< Idle.                        */
    PROC_OUTPUT_MSG,        /*!< Output first message.        */
    PROC_INPUT,             /*!< Wait to input.               */
    PROC_COMMAND_PARSE,     /*!< Output Echo back data.       */
    PROC_COMMAND_ERR,       /*!< Output command message.  */
    PROC_PARAMETER_ERR,     /*!< Output parameter error message.  */
    PROC_WRITING_ERR,       /*!< Output writing error message.  */
    PROC_READING_ERR,       /*!< Output reading error message.  */
    PROC_ERROR              /*!< Error.                       */
} Processing;

/** 
 *  @}
 */ /* End of group EI2C_SingleMaster_Private_define */

/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_SingleMaster_Private_typedef EI2C_SingleMaster Private Typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @struct input_t
 * @brief  Input data information.
*/
/*----------------------------------*/
typedef struct
{
    char    data[(UART_IO_RECEIVE_MAX+1)];  /*!< Input data(+ for NULL 1byte). */
    uint8_t wp;                             /*!< Write pointer.                */
} input_t;

/** 
 *  @}
 */ /* End of group EI2C_SingleMaster_Private_typedef */

/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_SingleMaster_Private_variables EI2C_SingleMaster Private Variables
 *  @{
 */
static Processing proc;         /*!< Processing mode.        */
static input_t input;           /*!< Input data information. */
static uint8_t write_data[10];  /*!< Write data information. */
static uint8_t wbuff[64+2];      /*!< The buffer for write data.  */
static uint8_t rbuff[64+2];      /*!< The buffer for read data.   */
static bsp_ei2c_t   ei2c_obj;  

static volatile uint32_t transfer_event;        /*!< I2C Transfer Event       */
static int32_t timeout;                         /*!< Timeout work.            */

static uint32_t master_rx_length;               /*!< Length of Master Rx Data. */
static uint32_t master_tx_length;               /*!< Length of Master Tx Data. */
/**
 *  @}
 */ /* End of group EI2C_SingleMaster_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_SingleMaster_Private_functions EI2C_SingleMaster Private Fuctions
 *  @{
 */
static void clear_data(void);
static TXZ_Result application_initialize(void);
TXZ_Result EEPROM_ei2c_init(void);

/*--------------------------------------------------*/
/**
  * @brief     Initializing of EI2C Driver
  * @param     -
  * @retval    TXZ_SUCCESS :Success.
  * @retval    TXZ_ERROR   :Failure.
  */
/*--------------------------------------------------*/
TXZ_Result EEPROM_ei2c_init(void)
{
    TXZ_Result result = TXZ_ERROR;

    ei2c_obj.ei2c.init.clock.sck = EI2CSCL_FRQ_FM;
    ei2c_obj.ei2c.init.cr0.nfsel = EI2CxCR0_NFSEL_DIG;
    ei2c_obj.ei2c.init.cr0.dnf = EI2CxCR0_DNF_2PRSCK;
    
    /*------------------------------*/
    /*  EI2C Driver Initialize       */
    /*------------------------------*/
    if (ei2c_init(&ei2c_obj, EI2C_CFG_PORT_SDA, EI2C_CFG_PORT_SCL) == TXZ_ERROR)
    {
       /* fatal error */
       printf("ei2c init error!!\n");    
    }
    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief     Writing and Reading by Master Non-Blocking
  * @param     -
  * @retval    -
  */
/*--------------------------------------------------*/
static void ts_ei2c_master_non_blocking_write_read(void)
{
    master_rx_length = 0;

    transfer_event = 0;
    if (ei2c_transfer_asynch(&ei2c_obj, 
                            wbuff, 
                            2, 
                            rbuff, 
                            1, 
                            (int32_t)EI2C_EEPROM_ADDRESS, 
                            1
                            ) != TXZ_SUCCESS)
    {
        printf("read error!!\n");
    }
    else
    {
        timeout = TEST_TIMEOUT;
        while (1)
        {
            volatile uint32_t event = transfer_event;   /* asynch information */

            if ((event & EI2C_EVENT_ALL) != 0)
            {
                /* event occured */
                if (event & EI2C_EVENT_TRANSFER_COMPLETE)
                {
                    /* complete */
                    master_rx_length = ei2c_obj.rx_buff.pos;
                }
                else if (event & EI2C_EVENT_TRANSFER_EARLY_NACK)
                {
                    /* early complete */
                    ei2c_abort_asynch(&ei2c_obj);
                    master_rx_length = ei2c_obj.rx_buff.pos;
                    printf("EARLY_NACK!!\n");
                }
                else if (event & EI2C_EVENT_ERROR_NO_SLAVE)
                {
                    /* no slave */
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("NO_SLAVE!!\n");
                }
                else  /* if (event & EI2C_EVENT_ERROR) */
                {
                    /* other error */
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("ERROR!!\n");
                }
                break;
            }
            else
            {
                /* no event */
                if ((timeout--) == 0)
                {
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("timeout error!!\n");
                    break;
                }
            }
        }
        while (ei2c_active(&ei2c_obj))
        {
            /* checking bus release */
        }
    }  
}

/*--------------------------------------------------*/
/**
  * @brief     Writing by Master Non-Blocking
  * @param     -
  * @retval    -
  */
/*--------------------------------------------------*/
static void ts_ei2c_master_non_blocking_write(void)
{
    master_tx_length = 0;
    transfer_event = 0;
    /* start condition */
    if (ei2c_transfer_asynch(&ei2c_obj, 
              wbuff, 
             3, 
             EI2C_NULL, 
             0, 
            (int32_t)EI2C_EEPROM_ADDRESS, 
             1
            ) != TXZ_SUCCESS)
    {
          master_tx_length = 0;
          printf("write error!!\n");
    }
    else
    {
        timeout = TEST_TIMEOUT;
        while (1)
        {
            volatile uint32_t event = transfer_event;    /* asynch information. */
            
            /* event occured */
            if ((event & EI2C_EVENT_ALL) != 0)
            {
                if (event & EI2C_EVENT_TRANSFER_COMPLETE)
                {
                    /* complete */
                    master_tx_length = ei2c_obj.tx_buff.pos;
                }
                else if (event & EI2C_EVENT_TRANSFER_EARLY_NACK)
                {
                    /* early complete */
                    ei2c_abort_asynch(&ei2c_obj);
                    master_tx_length = ei2c_obj.tx_buff.pos;
                    printf("EARLY_NACK!!\n");
                }
                else if (event & EI2C_EVENT_ERROR_NO_SLAVE)
                {
                    /* no slave */
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("NO_SLAVE!!\n");
                }
                else  /* if (event & EI2C_EVENT_ERROR) */
                {
                    /* other error */
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("ERROR!!\n");
                }
                break;
            }
            else
            {
                /* no event */
                if ((timeout--) == 0)
                {
                    ei2c_abort_asynch(&ei2c_obj);
                    printf("timeout error!!\n");
                    break;
                }
            }
        }
        while (ei2c_active(&ei2c_obj))
        {
            /* checking bus release */
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Clear input data.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void clear_data(void)
{
    uint8_t i;

    for (i = 0; i < (UART_IO_RECEIVE_MAX+1); i++)
    {
        input.data[i] = 0;
    }
    input.wp = 0;
    
    for (i = 0; i < 10; i++)
    {
        write_data[i] = 0;
    }
}
/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
   TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = USER_ID_UART;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = (uint32_t)CFG_UART_BAUDRATE;
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result  = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief      Implementation of INTI2Cx_IRQHandler
  * @param      -
  * @retval     -
  */
/*--------------------------------------------------*/
void ts_ei2cst_irq_handler(void)
{
    uint32_t event = 0;

    event = ei2c_master_st_irq_handler(&ei2c_obj);

    if (event != 0)
    {
        transfer_event = event;
    }
}

/*--------------------------------------------------*/
/**
  * @brief      Implementation of INTI2Cx_IRQHandler
  * @param      -
  * @retval     -
  */
/*--------------------------------------------------*/
void ts_ei2ctberbf_irq_handler(void)
{
    uint32_t event = 0;

    event = ei2c_irq_handler_asynch(&ei2c_obj);

    if (event != 0)
    {
        transfer_event = event;
    }
}

/**
 *  @}
 */ /* End of group EI2C_SingleMaster_Private_functions */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EI2C_SingleMaster_Exported_functions EI2C_SingleMaster Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    int num;
    int len = 0;
    int cnum = 0;
    /*----------------------*/
    /* Set Processing Mode  */
    /*----------------------*/
    proc = PROC_OUTPUT_MSG;
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /* Application initialization */
    application_initialize();  
    EEPROM_ei2c_init();
    
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(1)
    {
        switch(proc)
        {
        /*--- Output "First Message" ---*/
        case PROC_OUTPUT_MSG:
            {
                /* Clear input data.     */
                clear_data();
                /* Output first message. */
                if (printf("command > ") != -1)
                {
                    proc = PROC_INPUT;
                }
                else
                {
                    proc = PROC_ERROR;
                }
            }
            break;
        /*--- Wait to Input ---*/
        case PROC_INPUT:
            {   
                int data;
                
                data = getchar();
                if ((data != -1) && (input.wp < UART_IO_RECEIVE_MAX))
                {
                    input.data[input.wp] = (char)data;
                    input.wp++;
                    switch ((char)data)
                    {
                    /*--- LF: line feed ---*/
                    case '\n':
                        len = input.wp;
                        proc = PROC_COMMAND_PARSE;
                        break;
                    /*--- default ---*/
                    default:
                        /* no process */
                        break;
                    }
                }
                else
                {
                    proc = PROC_COMMAND_ERR;
                }
            }
            break;
        /*--- Output "Echo Back Data" ---*/
        case PROC_COMMAND_PARSE:
            {
                int i;

                uint8_t ascii_tmp = 0 ;
                int num_tmp = 0 ;
                int hex_num = 0 ;
                int base = 1 ;
                uint8_t bytedata = 0 ;

                if ((strncmp((const char *)&input.data[0], "write", 5)) == 0)
                {
                    /* WRITE Command */
                    if (len >= 8)
                    {
                        num = (len - (6 + 1));
                        for (i = 0; (i < num) && (i < 8); i++)
                        {
                            write_data[i] = (uint8_t)(input.data[6 + i]);
                        }
                        cnum = i;

                        for(i=cnum; i>0; i--) {
                            ascii_tmp = write_data[i-1];
                            if(((input.data[5+i]>=0x30)&&(input.data[5+i]<=0x39)) ||  /* 0-9 */
                               ((input.data[5+i]>=0x41)&&(input.data[5+i]<=0x46)) ||  /* A-F */
                               ((input.data[5+i]>=0x61)&&(input.data[5+i]<=0x66)) )   /* a-f */
                            {
                                if((ascii_tmp>=0x30)&&(ascii_tmp<=0x39)) num_tmp = ascii_tmp - 0x30 ; /* 0-9 */
                                if((ascii_tmp>=0x41)&&(ascii_tmp<=0x46)) num_tmp = ascii_tmp - 55 ;   /* A-F */
                                if((ascii_tmp>=0x61)&&(ascii_tmp<=0x66)) num_tmp = ascii_tmp - 87 ;   /* a-f */
                                hex_num += base * num_tmp ;
                                base *= 16 ;
                            }
                            else
                            {
                                 proc = PROC_PARAMETER_ERR;
                            }
                            if(hex_num > 255) {
                                proc = PROC_PARAMETER_ERR;
                            } else {
                                bytedata = (uint8_t) hex_num ;
                            }
                        }

                        if ( proc != PROC_COMMAND_ERR && proc != PROC_PARAMETER_ERR )
                        {
                            /* Start Condition */
                            wbuff[0] = 0x00;
                            wbuff[1] = 0x00;
                            wbuff[2] = bytedata;
                            ts_ei2c_master_non_blocking_write();
                            if (master_tx_length == 3)
                            {
                                printf("write data > ");
                                printf("%2x\n", bytedata);
                                proc = PROC_OUTPUT_MSG;
                            }
                            else
                            {
                                printf("\n write error !! \n");
                                proc = PROC_WRITING_ERR ;
                            }
                            printf("\n");
                        }
                    }
                    else
                    {
                        proc = PROC_PARAMETER_ERR;
                    }
                    break;
                }
                else if ((strncmp((const char *)&input.data[0], "read", 4)) == 0)
                {
                    /* READ Command */
                    printf("read data > ");
                    wbuff[0] = 0x00;
                    wbuff[1] = 0x00;
                    
                    ts_ei2c_master_non_blocking_write_read();
                    if (master_rx_length == 1)
                    {
                        printf("%2x\n", rbuff[0]);
                        proc = PROC_OUTPUT_MSG;
                    }
                    else
                    {
                        printf("\n read error !! \n");
                        proc = PROC_READING_ERR;
                    }
                    printf("\n");
                }
                else
                {
                    proc = PROC_COMMAND_ERR;
                }
            }
            break;
                
        /*--- Output "Command Error Message" ---*/
        case PROC_COMMAND_ERR:
            printf("Command Error!!\n");
            proc = PROC_OUTPUT_MSG;
            break;
        /*--- Output "Parameter Error Message" ---*/
        case PROC_PARAMETER_ERR:
            printf("Parameter Error!!\n");
            proc = PROC_OUTPUT_MSG;
            break;
        /*--- Output "Writing Error Message" ---*/
        case PROC_WRITING_ERR:
            printf("Writing Error!!\n");
            proc = PROC_OUTPUT_MSG;
            break;
        /*--- Output "Reading Error Message" ---*/
        case PROC_READING_ERR:
            printf("Reading Error!!\n");
            proc = PROC_OUTPUT_MSG;
            break;
        /*--- Error ---*/
        case PROC_ERROR:
            /* System Error. Please Debug. */
#ifdef DEBUG
            assert_failed(MAIN_NULL, 0);
#endif
            proc = PROC_OUTPUT_MSG;
            break;
        /*--- Default ---*/
        default:
            proc = PROC_ERROR;
            break;
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_rx_irq_handler();
        break;
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_tx_irq_handler();
        break;
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_err_irq_handler();
        break;
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  EI2C IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_i2c(void)
{
   ts_ei2cst_irq_handler();
}

/*--------------------------------------------------*/
/** 
  * @brief  EI2C TBE IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_i2c_tbe(void)
{
    ts_ei2ctberbf_irq_handler();
}

/*--------------------------------------------------*/
/** 
  * @brief  EI2C RBF IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_i2c_rbf(void)
{
    ts_ei2ctberbf_irq_handler();
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
     /*   __nop(); */
    }
}
#endif

/**
 *  @}
 */ /* End of group EI2C_SingleMaster_Exported_functions */


/**
 *  @} 
 */ /* End of group EI2C_SingleMaster */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
