/** 
 *******************************************************************************
 * @file    main.c
 * @brief   FLASH_UserInformation Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include <string.h>
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_sw.h"
#include "bsp_uart_io.h"
#include "bsp_timer.h"
#include "flash.h"

/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup FLASH_UserInformation FLASH_UserInformation Sample Appli
 *  @{
 */
extern TXZ_Result erase_user_information_area(void);
extern TXZ_Result write_user_information_area(void);

/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_UserInformation_Private_macro FLASH_UserInformation Private Macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group FLASH_UserInformation_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_UserInformation_Private_define FLASH_UserInformation Private Define
 *  @{
 */

/** 
 *  @defgroup SWPollingConfiguration SW Polling Time Configuration
 *  @brief    SW Polling Time(ms).
 *  @{
 */
#define CFG_SW_POLLING_TIME     ((uint32_t)(100))
/** 
 *  @}
 */ /* End of group SWPollingConfiguration */

/** 
 *  @}
 */ /* End of group FLASH_UserInformation_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_UserInformation_Private_define FLASH_UserInformation Private Define
 *  @{
 */

#define MAIN_NULL       ((void *)0)     /*!< Null Pointer  */
#define USER_ID_UART    ((uint32_t)1)   /*!< User ID :USB UART (Any value) */

#if defined ( __CC_ARM   )      /* RealView Compiler */
extern uint32_t Load$$FLASH_CODE_RAM$$Base;
extern uint32_t Image$$FLASH_CODE_RAM$$Base;
extern uint32_t Load$$FLASH_CODE_RAM$$Length;

#elif defined ( __ICCARM__ )    /* IAR Compiler */
#pragma section = "FLASH_CODE_RAM"
#pragma section = "FLASH_CODE_ROM"
#endif


/** 
 *  @defgroup FLASH_UserInformation_Private_define FLASH_UserInformation Private Define
 *  @{
 */
#if defined ( __CC_ARM   )      /* RealView Compiler */
#define FLASH_API_ROM           (uint32_t *)&Load$$FLASH_CODE_RAM$$Base
#define FLASH_API_RAM           (uint32_t *)&Image$$FLASH_CODE_RAM$$Base
#define SIZE_FLASH_API          (uint32_t)&Load$$FLASH_CODE_RAM$$Length



#elif defined ( __ICCARM__ )    /* IAR Compiler */
#define FLASH_API_ROM           ((uint32_t *)__section_begin("FLASH_CODE_ROM"))
#define FLASH_API_RAM           ((uint32_t *)__section_begin("FLASH_CODE_RAM"))
#define SIZE_FLASH_API          ((uint32_t)__section_size("FLASH_CODE_ROM"))

#endif

/* code start address */
#define DEMO_START_ADDR         ((uint32_t)DEMO_A_FLASH)
/** 
 *  @}
 */ /* End of group FLASH_UserInformation_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_UserInformation_Private_define FLASH_UserInformation Private Define
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_IDLE,          /*!< Idle.             */
    PROC_START,         /*!< Start.            */
    PROC_RAMTRANSFER,   /*!< RAM transferring. */
    PROC_ERASING,       /*!< Erasing.          */
    PROC_WRITING,       /*!< writing.          */
    PROC_FINISHED,      /*!< Finished.         */
    PROC_ERROR,         /*!< Error.            */
    PROC_WRITE_ERROR,   /*!< Write Error.      */
    PROC_ERASE_ERROR    /*!< Erase Error.      */
} Processing;
#define USERINFOSIZE      (128)        /*<! User Information data size */

/** 
 *  @}
 */ /* End of group FLASH_UserInformation_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_UserInformation_Private_typedef FLASH_UserInformation Private Typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
    t32a_t      t32a;                       /*!< Application      :T32A.     */
    flash_t     flash;                      /*!< Application :flash.     */
    uart_io_t   *p_uart;                    /*!< Application :USB UART. */
    sw_t        sw[(uint32_t)BSP_PSW_MAX];  /*!< Application :Push SW            */
    timer_t     timer;                      /*!< Application :Timer (1sec timer). */
    cg_t        cg;                         /*!< Driver      :CG.       */
} instance_t;

/**
 *  @}
 */ /* End of group FLASH_UserInformation_Private_typedeff */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_UserInformation_Private_variables FLASH_UserInformation Private Variables
 *  @{
 */
static instance_t instance;     /*!< Instance.                */
static Processing proc;         /*!< Processing mode.         */
#if defined ( __CC_ARM   )      /* RealView Compiler */
static __align(4) char UserInformationData[USERINFOSIZE] =
{
"Toshiba TMPM3HQFDA User Information Example"
};                             /*!< User Information Data  */
#elif defined ( __ICCARM__ )    /* IAR Compiler */
static char UserInformationData[USERINFOSIZE] =
{
"Toshiba TMPM3HQFDA User Information Example"
};   
#endif
static char GetUserInformationData[USERINFOSIZE];

/**
 *  @}
 */ /* End of group FLASH_UserInformation_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_UserInformation_Private_fuctions FLASH_UserInformation Private Fuctions
 *  @{
 */
static void variable_initialize(void);
static void sw_state_change_handler(uint32_t id, gpio_pinstate_t state);
static void timer_interval_handler(uint32_t id);
static TXZ_Result driver_initialize(void);
static TXZ_Result driver_finalize(void);
static TXZ_Result application_initialize(void);
static TXZ_Result application_finalize(void);
static void Copy_Routine(uint32_t * dest, uint32_t * source, uint32_t size);
void (* startup) (void);
uint32_t gLoadAddr;

/*--------------------------------------------------*/
/** 
  * @brief  copy 32-bit data from source to dest
  * @param  -
  * @retval None.
  * @note   -
  */
/*--------------------------------------------------*/
static void Copy_Routine(uint32_t * dest, uint32_t * source, uint32_t size)
{
    uint32_t *dest_addr, *source_addr, tmpsize;
    uint32_t i, tmps, tmpd, mask;

    dest_addr = dest;
    source_addr = source;

    tmpsize = size >> 2U;
    for (i = 0U; i < tmpsize; i++) {    /* 32bits copy */
        *dest_addr = *source_addr;
        dest_addr++;
        source_addr++;
    }
    if (size & 0x00000003U) {   /* if the last data size is not 0(maybe 1,2 or 3), copy the last data */
        mask = 0xFFFFFF00U;
        i = size & 0x00000003U;
        tmps = *source_addr;
        tmpd = *dest_addr;
        while (i - 1U) {
            mask = mask << 8U;
            i--;
        }
        tmps = tmps & (~mask);
        tmpd = tmpd & (mask);
        *dest_addr = tmps + tmpd;       /* 32bits copy, but only change the bytes need to be changed */
    } else {
        /* Do nothing */
    }
}
/**
 *  @}
 */ /* End of group FLASH_UserInformation_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_UserInformation_Exported_functions FLASH_UserInformation Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  execution user program.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void exec_user_program(void)
{
    if((gLoadAddr & 0x01) == 0) {            /* Thumb check */
        gLoadAddr |= 0x01;
    }
    startup = (void(*)(void))gLoadAddr;
    startup();              /* jump to user_program */
}
/*--------------------------------------------------*/
/** 
  * @brief  copy flash control functions.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void copy_fc_func(void)
{
    Copy_Routine(FLASH_API_RAM, FLASH_API_ROM, SIZE_FLASH_API);     /* copy flash API to RAM */
}
/*--------------------------------------------------*/
/** 
  * @brief  Variable Initialize
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void variable_initialize(void)
{
    /* Instance. */
    instance.p_uart             = MAIN_NULL;
    instance.t32a.p_instance    = MAIN_NULL;
    instance.flash.p_instance   = MAIN_NULL;
    instance.cg.p_instance      = MAIN_NULL;
    /* Processing mode. */
    proc = PROC_IDLE;
}
/*--------------------------------------------------*/
/** 
  * @brief  SW State Change Handler.
  * @param  id    :User ID
  * @param  state :Push SW Port State
  * @return -
  * @retval -
  * @note   Notify that, "State is changed." \n
  *         Call by SW Task \n
  *         If SW is pushed, change the processing mode(@ref Processing).
  */
/*--------------------------------------------------*/
static void sw_state_change_handler(uint32_t id, gpio_pinstate_t state)
{
    /* Check SW Push(= GPIO_PIN_RESET) */
    if (state == GPIO_PIN_RESET)
    {
        uint32_t i;

        for (i=0; (i < (uint32_t)BSP_PSW_MAX) && (proc != PROC_ERROR); i++)
        {
            if (id == (uint32_t)&instance.sw[i])
            {
                /* Select Low Power Mode */
                switch ((BSPPsw)i)
                {
                case BSP_PSW_1:
                    proc = PROC_RAMTRANSFER;
                    break;
                default:
                    /* no processing */
                    break;
                }
            }
        }
    }
}
/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    int32_t i;
    /*----------------------*/
    /* SW                   */
    /*----------------------*/
    for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
    {
        sw_1ms_timer_handler(&instance.sw[i]);
    }
}
/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize
  * @param  -
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (p_t32a->p_instance != MAIN_NULL)
        {
            if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver initialize will process within USB UART. */
    /*----------------------*/
    /* Flash                */
    /*----------------------*/
    {
        flash_t *p_flash = &instance.flash;

        /*--- Construct  ---*/
        /* Register Allocation */
        p_flash->p_instance = TSB_FC;
        /*--- Initialize ---*/
        p_flash->init.status = FC_SR0_RDYBSY;
        p_flash->init.src_address = (uint32_t *)UserInformationData;
        p_flash->init.dst_address = (uint32_t *)GetUserInformationData;
        p_flash->init.size = USERINFOSIZE;
    }
    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Finalize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;
        
        /*--- Finalize ---*/
        /* no processing */
        /*--- Destruct ---*/
        /* Register Release */
        p_t32a->p_instance = MAIN_NULL;
    }
    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver finalize has done within USB UART.    */
    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver finalize will process within BSP.     */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize
  * @param  -
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    {
        timer_t *p_timer = &instance.timer;

        p_timer->init.id       = (uint32_t)&instance.timer;
        p_timer->init.p_t32a   = &instance.t32a;
        p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
        p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
        p_timer->init.interval = 1000;
        p_timer->init.handler  = timer_interval_handler;
        timer_initialize(p_timer);
    }

    /*----------------------*/
    /* PSW                  */
    /*----------------------*/
    {
        uint32_t i;
        sw_t *p_sw = MAIN_NULL;

        for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
        {
            p_sw = &instance.sw[i];

            p_sw->init.id                   = (uint32_t)p_sw;
            p_sw->init.p_gpio               = bsp_get_gpio_instance_address();
            p_sw->init.port.group           = bsp_get_gpio_group_psw((BSPPsw)(i));
            p_sw->init.port.num             = bsp_get_gpio_num_psw((BSPPsw)(i));
            p_sw->init.polling.func         = TXZ_ENABLE;
            p_sw->init.polling.interval     = CFG_SW_POLLING_TIME;
            p_sw->init.chattering.func      = TXZ_DISABLE;
            p_sw->init.chattering.count     = 0;
            p_sw->init.handler              = sw_state_change_handler;
            sw_initialize(p_sw);
        }
    }

    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = USER_ID_UART;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = bsp_get_usb_uart_tx_baudrate(BSP_USB_UART_0);
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Finalize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;
    uint32_t i;

    /*----------------------*/
    /* PSW                  */
    /*----------------------*/
    for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
    {
        sw_finalize(&instance.sw[i]);
    }

    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    if (instance.p_uart != MAIN_NULL)
    {
        uart_io_finalize();
    }

    return (result);
}

/**
 *  @}
 */ /* End of group FLASH_UserInformation_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_UserInformation_Exported_functions FLASH_UserInformation Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/

int main(void)
{
    uint32_t i;
    cg_t *p_cg = &instance.cg;
    p_cg->p_instance = TSB_CG;

    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /*--- variable    ---*/
    variable_initialize();
    /*--- driver      ---*/
    if (driver_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    /*--- application ---*/
    if (application_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    cg_set_warm_up_time(p_cg, CG_WARM_UP_SRC_IHOSC1, WU_TIME_INT);
    cg_ihosc1_enable();            /* internal HOSC enable */
    cg_start_warmup();
    cg_wait_warmup();
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(proc != PROC_ERROR)
    {
        switch(proc)
        {
        case PROC_IDLE:
            printf("FLASH User Information Area Write\n");
            printf("Please press the S4 \n");
            timer_start(&instance.timer);
            proc = PROC_START;
            break;
        case PROC_START:
            /*----------------------*/
            /* SW                   */
            /*----------------------*/
            for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
            {
                sw_task(&instance.sw[i]);
            }
            break;
        case PROC_RAMTRANSFER:
            printf("RAM transferring. \n");
            timer_stop(&instance.timer);
            copy_fc_func();
            proc = PROC_ERASING;
            break;
        case PROC_ERASING:
            if(fc_erase_user_information_area(&instance.flash) == TXZ_ERROR){
                proc = PROC_ERASE_ERROR;
            }
            else {
                printf("Erasing \n");
                proc = PROC_WRITING;
            }
            break;
        case PROC_WRITING:
            if(fc_write_user_information_area(&instance.flash) == TXZ_ERROR){
                proc = PROC_WRITE_ERROR;
            }
            else {
                printf("writing. \n");
                proc = PROC_FINISHED;
            }
            break;
        case PROC_FINISHED:
            fc_read_user_information_area(&instance.flash);
            printf("Finished \n");
            printf("check a variable GetUserInformationData \n");
            proc = PROC_IDLE;
            break;
        case PROC_ERROR:
            printf("Writing Error!! \n");
            proc = PROC_IDLE;
            break;
        case PROC_WRITE_ERROR:
            printf("Writing Error!! \n");
            proc = PROC_IDLE;
            break;
        case PROC_ERASE_ERROR:
            printf("Erasing Error!! \n");
            proc = PROC_IDLE;
            break;
        default:
            proc = PROC_ERROR;
            break;
        }
    }
    /*>>> System Error. Please Debug. */
    /*----------------------*/
    /* Finalize             */
    /*----------------------*/
    /*--- application ---*/
    if (application_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*--- driver      ---*/
    if (driver_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*----------------------*/
    /* BSP finalize         */
    /*----------------------*/
    bsp_finalize();
#ifdef DEBUG
    assert_failed(MAIN_NULL, 0);
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer.
  * @param  timer :Timer.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    default:
        /* no processing */
        break;
    }
}
#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __nop();
    }
}
#endif
/**
 *  @}
 */ /* End of group FLASH_UserInformation_Exported_functions */

/**
 *  @}
 */ /* End of group FLASH_UserInformation */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
