/** 
 *******************************************************************************
 * @file    main.c
 * @brief   I2C SingleMaster Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif
/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_uart_io.h"
#include "bsp_i2c.h"

/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup I2C_SingleMaster I2C_SingleMaster Sample Appli
 *  @{
 */

/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup I2C_SingleMaster_Private_macro I2C_SingleMaster Private Macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group I2C_SingleMaster_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup I2C_SingleMaster_Private_define I2C_SingleMaster Private Define
 *  @{
 */
/* no define */

/** 
 *  @}
 */ /* End of group I2C_SingleMaster_Private_define */

/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup I2C_SingleMaster_Private_define I2C_SingleMaster Private Define
 *  @{
 */

#define MAIN_NULL       ((void *)0)         /*!< Null Pointer.            */
#define SAMPLE_ADDRESS  (0x0000)            /*!< Access location address. */
#define I2C_EEPROM_ADDRESS    ((uint8_t)0xA0)           /*!< Slave Address.    */
#define I2C_EEPROM_CLOCK      (400000)                  /*!< Frequency.        */
/*----------------------*/
/*  UART Setting        */
/*----------------------*/
/** 
 *  @defgroup UART_Setting UART Setting
 *  @{
 */
#define CFG_UART_BAUDRATE               ((uint32_t)115200)  /*!< Boudrate(bps) */
/** 
 *  @}
 */ /* End of group UART_Setting */

#define USER_ID_REMOTE  ((uint32_t)0)   /*!< User ID :Remote Control (Any value) */
#define USER_ID_UART    ((uint32_t)1)   /*!< User ID :USB UART (Any value) */


/** 
 *  @}
 */ /* End of group I2C_SingleMaster_Private_define */

/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup I2C_SingleMaster_Private_define I2C_SingleMaster Private Define
 *  @{
 */

/*----------------------------------*/
/** 
 * @enum   Processing
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_IDLE,              /*!< Idle.                        */
    PROC_OUTPUT_MSG,        /*!< Output first message.        */
    PROC_INPUT,             /*!< Wait to input.               */
    PROC_COMMAND_PARSE,     /*!< Output Echo back data.       */
    PROC_COMMAND_ERR,       /*!< Output command message.  */
    PROC_PARAMETER_ERR,     /*!< Output parameter error message.  */
    PROC_WRITING_ERR,       /*!< Output writing error message.  */
    PROC_READING_ERR,       /*!< Output reading error message.  */
    PROC_ERROR              /*!< Error.                       */
} Processing;

/** 
 *  @}
 */ /* End of group I2C_SingleMaster_Private_define */

/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup I2C_SingleMaster_Private_typedef I2C_SingleMaster Private Typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @struct input_t
 * @brief  Input data information.
*/
/*----------------------------------*/
typedef struct
{
    char    data[(UART_IO_RECEIVE_MAX+1)];  /*!< Input data(+ for NULL 1byte). */
    uint8_t wp;                             /*!< Write pointer.                */
} input_t;

/** 
 *  @}
 */ /* End of group I2C_SingleMaster_Private_typedef */

/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup I2C_SingleMaster_Private_variables I2C_SingleMaster Private Variables
 *  @{
 */
static Processing proc;         /*!< Processing mode.        */
static input_t input;           /*!< Input data information. */
static uint8_t write_data[10];  /*!< Write data information. */
static uint8_t wbuff[64+2];      /*!< The buffer for write data.  */
static uint8_t rbuff[64+2];      /*!< The buffer for read data.   */
static i2c_t   obj;  
/**
 *  @}
 */ /* End of group I2C_SingleMaster_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup I2C_SingleMaster_Private_functions I2C_SingleMaster Private Fuctions
 *  @{
 */
static void clear_data(void);
static TXZ_Result application_initialize(void);
TXZ_Result EEPROM_i2c_init(void);
TXZ_Result EEPROM_i2c_ReadByte(int32_t address, uint8_t *p_data);
TXZ_Result EEPROM_i2c_WriteByte(int32_t address, uint8_t data);
int32_t EEPROM_i2c_ReadData(int32_t address, uint8_t *p_data, int32_t length);
int32_t EEPROM_i2c_WritePage(int32_t address, uint8_t *p_data, int32_t length);
TXZ_Result EEPROM_i2c_WriteProtection(void);
int32_t EEPROM_i2c_IsDeviceReady(void);
/*--------------------------------------------------*/
/**
  * @brief     Initializing of I2C Driver
  * @param     -
  * @retval    TXZ_SUCCESS :Success.
  * @retval    TXZ_ERROR   :Failure.
  */
/*--------------------------------------------------*/
TXZ_Result EEPROM_i2c_init(void)
{
    TXZ_Result result = TXZ_ERROR;

    /*------------------------------*/
    /*  I2C Driver Initialize       */
    /*------------------------------*/
    if (i2c_init(&obj, I2C_CFG_PORT_SDA, I2C_CFG_PORT_SCL) == TXZ_SUCCESS)
    {
#ifdef I2C_EEPROM_CLOCK
        if (i2c_frequency(&obj, I2C_EEPROM_CLOCK) == TXZ_SUCCESS)
        {
            result = TXZ_SUCCESS;
        }
#endif      
    }
    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief     EEPROM BYTE READ
  * @param     address     :Access location Address.
  * @param     p_data      :Destination address of Read data.
  * @retval    TXZ_SUCCESS :Success.
  * @retval    TXZ_ERROR   :Failure.
  */
/*--------------------------------------------------*/
TXZ_Result EEPROM_i2c_ReadByte(int32_t address, uint8_t *p_data)
{
  TXZ_Result result = TXZ_SUCCESS;

    /* Start */
    if (i2c_start(&obj) != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /* Control Byte */
    if (i2c_byte_write(&obj, (int32_t)I2C_EEPROM_ADDRESS) != I2C_ACK)
    {
        result = TXZ_ERROR;
    }
    /* Adddress High Byte */
    if (i2c_byte_write(&obj, (address >> 8)) != I2C_ACK)
    {
        result = TXZ_ERROR;
    }
    /* Adddress Low Byte */
    if (i2c_byte_write(&obj, (address)) != I2C_ACK)
    {
        result = TXZ_ERROR;
    }
    /* Start */
    if (i2c_start(&obj) != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /* Control Byte */
    if (i2c_byte_write(&obj, (int32_t)(I2C_EEPROM_ADDRESS|1)) != I2C_ACK)
    {
        result = TXZ_ERROR;
    }
    /* Data Byte */
    if (result != TXZ_ERROR)
    {
       *p_data = (uint8_t)i2c_byte_read(&obj, 1);
    }
    /* Stop */
    if (i2c_stop(&obj) != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
  return (result);
}

/*--------------------------------------------------*/
/**
  * @brief     EEPROM BYTE WRITE
  * @param     address     :Access location Address.
  * @param     data        :Write data.
  * @retval    TXZ_SUCCESS :Success.
  * @retval    TXZ_ERROR   :Failure.
  */
/*--------------------------------------------------*/
TXZ_Result EEPROM_i2c_WriteByte(int32_t address, uint8_t data)
{
  TXZ_Result result = TXZ_SUCCESS;

    /* Start */
    if (i2c_start(&obj) != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /* Control Byte */
    if (i2c_byte_write(&obj, (int32_t)I2C_EEPROM_ADDRESS) != I2C_ACK)
    {
        result = TXZ_ERROR;     
    }
    /* Adddress High Byte */
    if (i2c_byte_write(&obj, (address >> 8)) != I2C_ACK)
    {
        result = TXZ_ERROR;
    }
    /* Adddress Low Byte */
    if (i2c_byte_write(&obj, (address)) != I2C_ACK)
    {
        result = TXZ_ERROR;
    }
    /* Data Byte */
    if (i2c_byte_write(&obj, (int32_t)data) != I2C_ACK)
    {
        result = TXZ_ERROR;
    }
    /* Stop */
    if (i2c_stop(&obj) != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
  return (result);
}

/*--------------------------------------------------*/
/**
  * @brief     EEPROM SEQUENCIAL READ
  * @param     address     :Access location Address.
  * @param     p_data      :Destination address of Read data.
  * @param     length      :Number of read bytes.
  * @retval    Number of read bytes.
  */
/*--------------------------------------------------*/
int32_t EEPROM_i2c_ReadData(int32_t address, uint8_t *p_data, int32_t length)
{
    int32_t result = -1;
    int32_t i;

    /* for the time being */
    wbuff[0] = (uint8_t)(address >> 8);
    wbuff[1] = (uint8_t)(address);

    /* EEPROM address transmit */
    if (i2c_write(&obj, (int32_t)I2C_EEPROM_ADDRESS, wbuff, 2, 0) == 2)
    {
        /* EEPROM data read */
        int32_t rx_length = 0;
            
        rx_length = i2c_read(&obj, (int32_t)I2C_EEPROM_ADDRESS, rbuff, length, 1);
        if(rx_length < length)
        {
            /* error */
        }
        else
        {
            uint8_t *p_src = &rbuff[0];
            uint8_t *p_dst = p_data;

            for (i = 0; ((i < 64) && (i < rx_length)); i++)
            {
                *p_dst++ = *p_src++;
            }
            result = i;
        }
    }
    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief     EEPROM PAGE WRITE
  * @param     address     :Access location Address.
  * @param     p_data   :Source Address of Write data.
  * @param     length   :Number of write bytes.
  * @retval    Number of write bytes.
  */
/*--------------------------------------------------*/
int32_t EEPROM_i2c_WritePage(int32_t address, uint8_t *p_data, int32_t length)
{
    int32_t result = -1;
    int32_t i;
    
    wbuff[0] = (uint8_t)(address >> 8);
    wbuff[1] = (uint8_t)(address);
    {
        uint8_t *p_src = p_data;
        uint8_t *p_dst = &wbuff[2];
        for (i = 0; ((i < 64) && (i < length)); i++)
        {
            *p_dst++ = *p_src++;
        }   
        if (i != 0)
        {
            int32_t tx_length = 0;
            tx_length = i2c_write(&obj, (int32_t)I2C_EEPROM_ADDRESS, wbuff, i+2, 1);
            if (tx_length > 2)
            {
                result = tx_length;
            }
        }
    }
    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief      EEPROM_i2c_WriteProtection
  * @param      -
  * @retval     SUCCESS :Success.
  * @retval     ERROR   :Failure.
  */
/*--------------------------------------------------*/
TXZ_Result EEPROM_i2c_WriteProtection(void)
{
    TXZ_Result result = TXZ_SUCCESS;
    /*  not support */
    return (result);    
}

/*--------------------------------------------------*/
/**
  * @brief      EEPROM_i2c_IsDeviceReady
  * @param      -
  * @retval     1     :Ready.
  * @retval     0     :Not Ready.
  */
/*--------------------------------------------------*/
int32_t EEPROM_i2c_IsDeviceReady(void)
{
    /*  not support */
    return (1);
}
/*--------------------------------------------------*/
/** 
  * @brief  Clear input data.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void clear_data(void)
{
    uint8_t i;

    for (i = 0; i < (UART_IO_RECEIVE_MAX+1); i++)
    {
        input.data[i] = 0;
    }
    input.wp = 0;
    
    for (i = 0; i < 10; i++)
    {
        write_data[i] = 0;
    }
}
/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
   TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = USER_ID_UART;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = (uint32_t)CFG_UART_BAUDRATE;
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result  = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    return (result);
}
/**
 *  @}
 */ /* End of group I2C_SingleMaster_Private_functions */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup I2C_SingleMaster_Exported_functions I2C_SingleMaster Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    int num;
    int len = 0;
    int cnum = 0;
    /*----------------------*/
    /* Set Processing Mode  */
    /*----------------------*/
    proc = PROC_OUTPUT_MSG;
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /* Application initialization */
    application_initialize();  
    EEPROM_i2c_init();
    
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(1)
    {
        switch(proc)
        {
        /*--- Output "First Message" ---*/
        case PROC_OUTPUT_MSG:
            {
                /* Clear input data.     */
                clear_data();
                /* Output first message. */
                if (printf("command > ") != -1)
                {
                    proc = PROC_INPUT;
                }
                else
                {
                    proc = PROC_ERROR;
                }
            }
            break;
        /*--- Wait to Input ---*/
        case PROC_INPUT:
            {   
                int data;
                
                data = getchar();
                if ((data != -1) && (input.wp < UART_IO_RECEIVE_MAX))
                {
                    input.data[input.wp] = (char)data;
                    input.wp++;
                    switch ((char)data)
                    {
                    /*--- LF: line feed ---*/
                    case '\n':
                        len = input.wp;
                        proc = PROC_COMMAND_PARSE;
                        break;
                    /*--- default ---*/
                    default:
                        /* no process */
                        break;
                    }
                }
                else
                {
                    proc = PROC_COMMAND_ERR;
                }
            }
            break;
        /*--- Output "Echo Back Data" ---*/
        case PROC_COMMAND_PARSE:
            {
                int i;
                int32_t res;

                uint8_t ascii_tmp = 0 ;
                int num_tmp = 0 ;
                int hex_num = 0 ;
                int base = 1 ;
                uint8_t bytedata = 0 ;

                if ((strncmp((const char *)&input.data[0], "write", 5)) == 0)
                {
                    /* WRITE Command */
                    if (len >= 8)
                    {
                        num = (len - (6 + 1));
                        for (i = 0; (i < num) && (i < 8); i++)
                        {
                            write_data[i] = (uint8_t)(input.data[6 + i]);
                        }
                        cnum = i;

                        for(i=cnum; i>0; i--) {
                            ascii_tmp = write_data[i-1];
                            if(((input.data[5+i]>=0x30)&&(input.data[5+i]<=0x39)) ||  /* 0-9 */
                               ((input.data[5+i]>=0x41)&&(input.data[5+i]<=0x46)) ||  /* A-F */
                               ((input.data[5+i]>=0x61)&&(input.data[5+i]<=0x66)) )   /* a-f */
                            {
                                if((ascii_tmp>=0x30)&&(ascii_tmp<=0x39)) num_tmp = ascii_tmp - 0x30 ; /* 0-9 */
                                if((ascii_tmp>=0x41)&&(ascii_tmp<=0x46)) num_tmp = ascii_tmp - 55 ;   /* A-F */
                                if((ascii_tmp>=0x61)&&(ascii_tmp<=0x66)) num_tmp = ascii_tmp - 87 ;   /* a-f */
                                hex_num += base * num_tmp ;
                                base *= 16 ;
                            }
                            else
                            {
                                 proc = PROC_PARAMETER_ERR;
                            }
                            if(hex_num > 255) {
                                proc = PROC_PARAMETER_ERR;
                            } else {
                                bytedata = (uint8_t) hex_num ;
                            }

                        }

                        if ( proc != PROC_COMMAND_ERR && proc != PROC_PARAMETER_ERR )
                        {
                            res = EEPROM_i2c_WriteByte((int32_t)SAMPLE_ADDRESS, bytedata );
                            if (res == TXZ_SUCCESS)
                            {
                                printf("write data > ");
                                printf("%2x\n", bytedata);
                                proc = PROC_OUTPUT_MSG;
                            }
                            else
                            {
                                proc = PROC_WRITING_ERR ;
                            }
                            printf("\n");
                        }
                    }
                    else
                    {
                        proc = PROC_PARAMETER_ERR;
                    }
                    break;
                }
                else if ((strncmp((const char *)&input.data[0], "read", 4)) == 0)
                {
                    /* READ Command */
                    printf("read data > ");
                    res = EEPROM_i2c_ReadByte((int32_t)SAMPLE_ADDRESS, &bytedata);
                    if (res == TXZ_SUCCESS)
                    {
                        printf("%2x\n", bytedata);
                        proc = PROC_OUTPUT_MSG;
                    }
                    else
                    {
                        proc = PROC_READING_ERR;
                    }
                    printf("\n");
                }
                else
                {
                    proc = PROC_COMMAND_ERR;
                }
            }
            break;
                
        /*--- Output "Command Error Message" ---*/
        case PROC_COMMAND_ERR:
            printf("Command Error!!\n");
            proc = PROC_OUTPUT_MSG;
            break;
        /*--- Output "Parameter Error Message" ---*/
        case PROC_PARAMETER_ERR:
            printf("Parameter Error!!\n");
            proc = PROC_OUTPUT_MSG;
            break;
        /*--- Output "Writing Error Message" ---*/
        case PROC_WRITING_ERR:
            printf("Writing Error!!\n");
            proc = PROC_OUTPUT_MSG;
            break;
        /*--- Output "Reading Error Message" ---*/
        case PROC_READING_ERR:
            printf("Reading Error!!\n");
            proc = PROC_OUTPUT_MSG;
            break;
        /*--- Error ---*/
        case PROC_ERROR:
            /* System Error. Please Debug. */
#ifdef DEBUG
            assert_failed(MAIN_NULL, 0);
#endif
            proc = PROC_OUTPUT_MSG;
            break;
        /*--- Default ---*/
        default:
            proc = PROC_ERROR;
            break;
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_rx_irq_handler();
        break;
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_tx_irq_handler();
        break;
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_err_irq_handler();
        break;
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
     /*   __nop(); */
    }
}
#endif

/**
 *  @}
 */ /* End of group I2C_SingleMaster_Exported_functions */


/**
 *  @} 
 */ /* End of group I2C_SingleMaster */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
