/** 
 *******************************************************************************
 * @file    main.c
 * @brief   LVD_Interrupt Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include "bsp_com.h"
#include "bsp.h"
#include "lvd.h"
#include "bsp_led.h"
#include "bsp_timer.h"
/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup LVD_Interrupt LVD_Interrupt Sample Appli
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup LVD_Interrupt_Private_macro LVD_Interrupt Private Macro
 *  @{
 */
/*----------------------*/
/*  UART Setting        */
/*----------------------*/
/** 
 *  @defgroup LEDBlinkConfiguration LED Blink Time Configuration
 *  @brief    LED Blink Interval Time(ms).
 *  @{
 */
#define CFG_LED_BLINK_TIME      ((uint32_t)(1000))
/** 
 *  @}
 */ /* End of group LEDBlinkConfiguration */


#define MAIN_NULL       ((void *)0)     /*!< NULL. */
#define USER_ID_REMOTE  ((uint32_t)0)   /*!< User ID :Remote Control (Any value) */

/** 
 *  @}
 */ /* End of group LVD_Interrupt_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup LVD_Interrupt_Private_define LVD_Interrupt Private Define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group LVD_Interrupt_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup LVD_Interrupt_Private_define LVD_Interrupt Private Define
 *  @{
 */
#define LVD_UPPER               ((uint32_t)0x00000000)  /*!< LVD UPPER */
#define LVD_LOWER               ((uint32_t)0x00000001)  /*!< LVD LOWER */

#define LVD_STATUS_CT       ((uint32_t)0x00000008)      /*!< LVD status count         */
#define LVD_ST_MASK         ((uint32_t)0x00000080)      /*!< LVD ST mask              */
#define LVD_SHIFT           ((uint32_t)0x00000007)      /*!< LVD shift                */
/** 
 *  @}
 */ /* End of group LVD_Interrupt_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup LVD_Interrupt_Private_define LVD_Interrupt Private Define
 *  @{
 */
/* No Define */

/** 
 *  @}
 */ /* End of group LVD_Interrupt_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup LVD_Interrupt_Private_typedef LVD_Interrupt Private Typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
    gpio_t      gpio_obj;            /*!< Application :GPIO. */
    led_t       led[(uint32_t)BSP_LED_MAX]; /*!< Application :LED                */
    t32a_t      t32a;                       /*!< Driver      :T32A               */
    timer_t     timer;                      /*!< Application :Timer (1ms timer). */
} instance_t;


/**
 *  @}
 */ /* End of group LVD_Interrupt_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup LVD_Private_variables
 *  @{
 */
static instance_t instance; /*!< Instance.                    */
uint32_t lvd_status_now;                /*!< LVD status now           */
uint32_t lvd_status_old;   /*!< LVD status old           */
uint32_t lvd_status_count = 0;          /*!< LVD status count         */
uint32_t lvd_status = LVD_UPPER;        /*!< LVD status               */
TXZ_Result retval = TXZ_ERROR;          /*!< LVD get status result    */
led_t *p_led = MAIN_NULL;
/**
 *  @}
 */ /* End of group LVD_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup LVD_Interrupt_Private_fuctions LVD_Interrupt Private Fuctions
 *  @{
 */
static void variable_initialize(void);
static void timer_interval_handler(uint32_t id);
static TXZ_Result driver_initialize(void);
static TXZ_Result application_initialize(void);
/*--------------------------------------------------*/
/** 
  * @brief  Variable Initialize
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void variable_initialize(void)
{
    /* Instance. */
    instance.t32a.p_instance    = MAIN_NULL;
    instance.timer.init.p_t32a  = MAIN_NULL;
    {
        uint32_t i;
        /* Instance. */
        for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
        {
            instance.led[i].init.p_gpio = MAIN_NULL;
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    if (id == (uint32_t)&instance.timer)
    {
        uint32_t i;

        /*----------------------*/
        /* Tick                 */
        /*----------------------*/
        /* Increment Tick. Tick is used to driver's Timeout. */
        hal_inc_tick();
        /*----------------------*/
        /* LED                  */
        /*----------------------*/
        for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
        {
            led_1ms_timer_handler(&instance.led[i]);
        }
    }
}
/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   for Normal Mode
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
        {
            result = TXZ_ERROR;
        }
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
   TXZ_Result result = TXZ_SUCCESS;
    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    {
        timer_t *p_timer = &instance.timer;

        p_timer->init.id       = (uint32_t)&instance.timer;
        p_timer->init.p_t32a   = &instance.t32a;
        p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
        p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
        p_timer->init.interval = 1000;
        p_timer->init.handler  = timer_interval_handler;
        timer_initialize(p_timer);
    }
    /*----------------------*/
    /* LED                  */
    /*----------------------*/
    {
        uint32_t i;
        
        for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
        {
            p_led = &instance.led[i];

            p_led->init.id          = (uint32_t)p_led;
            p_led->init.p_gpio      = bsp_get_gpio_instance_address();
            p_led->init.port.group  = bsp_get_gpio_group_led((BSPLed)(i));
            p_led->init.port.num    = bsp_get_gpio_num_led((BSPLed)(i));
            p_led->init.state               = LED_STATE_OFF;
            if(i==0) {
                p_led->init.blink.func          = TXZ_ENABLE;
                p_led->init.blink.interval.on   = 250;
                p_led->init.blink.interval.off  = 250;
            }
            else
           {
                p_led->init.blink.func          = TXZ_DISABLE;
                p_led->init.blink.interval.on   = 0;
                p_led->init.blink.interval.off  = 0;
            }
            led_initialize(p_led);
        }
    }
    return (result);
}
/**
 *  @}
 */ /* End of group LVD_Interrupt_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup LVD_Interrupt_Exported_functions LVD_Interrupt Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    uint32_t i;
    led_t *p_led = MAIN_NULL;
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /* Valiable initialization */
    variable_initialize();
    /* Driver initialization */
    if(driver_initialize() != TXZ_SUCCESS)
    {
    }
    /* Application initialization */
    if(application_initialize() != TXZ_SUCCESS)
    {
    }

    /*----------------------*/
    /* 1ms Timer Start      */
    /*----------------------*/
    timer_start(&instance.timer);   

    init_LVD();
    start_LVD();

    retval = TXZ_SUCCESS;

    for(;;){
        /*----------------------*/
        /* LED                                */
        /*----------------------*/
        for (i = 0; i < (uint32_t)BSP_LED_MAX; i++)
        {
            led_task(&instance.led[i]);
        }
        /*----------------------*/
        /* LVD Status Check            */
        /*----------------------*/
        lvd_status_now = (uint32_t)((TSB_LVD->CR & LVD_ST_MASK) >> LVD_SHIFT);

        if(lvd_status_now == lvd_status_old)
        {
            lvd_status_count++;
            if(lvd_status_count >= LVD_STATUS_CT)
            {
                if(lvd_status != lvd_status_now)
                {
                    lvd_status = lvd_status_now;
                    retval = TXZ_SUCCESS;
                }
                lvd_status_count = 0;
            }
        }
        else
        {
            lvd_status_old = lvd_status_now;
            lvd_status_count = 0;
        }
        if(retval == TXZ_SUCCESS)
        {
            lvd_status = lvd_status_now;

            if(lvd_status == LVD_UPPER)
            {
                /* LED1 Turn OFF */
                p_led = &instance.led[0];
                led_turn_off(p_led);
                led_finalize(p_led);

                /* LED2 Turn ON */
                p_led = &instance.led[1];
                led_turn_on(p_led);

                retval = TXZ_ERROR;
            }
            else
            {
                /* LED1 Blink */
                p_led = &instance.led[0];
                led_turn_on(p_led);

                /* LED2Turn OFF */
                p_led = &instance.led[1];
                led_turn_off(p_led);

                retval = TXZ_ERROR;
            }
        }
        else
        {
            /* do nothing */
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  1ms Timer.
  * @param  timer :1ms Timer.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    default:
        /* no processing */
        break;
    }
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __nop();
    }
}
#endif
/**
 *  @}
 */ /* End of group LVD_Interrupt_Exported_functions */

/**
 *  @}
 */ /* End of group LVD_Interrupt */

/**
 *  @} 
 */ /* End of group Example */


#ifdef __cplusplus
}
#endif /* __cplusplus */
