/** 
 *******************************************************************************
 * @file    main.c
 * @brief   OFD HW TEST Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_led.h"
#include "bsp_timer.h"
#include "ofd.h"
/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup OFD_HW_TEST OFD_HW_TEST Sample Appli
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup OFD_HW_TEST_Private_macro OFD_HW_TEST Private Macro
 *  @{
 */

#define RSTFLG1_OFD_CLEAR                       ((uint32_t)0xFFFFFFF7)
/** 
 *  @}
 */ /* End of group OFD_HW_TEST_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup OFD_HW_TEST_Private_define OFD_HW_TEST Private Define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group OFD_HW_TEST_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup OFD_HW_TEST_Private_define OFD_HW_TEST Private Define
 *  @{
 */
/** 
 *  @defgroup LEDBlinkConfiguration LED Blink Time Configuration
 *  @brief    LED Blink Interval Time(ms).
 *  @{
 */
#define CFG_LED_BLINK_TIME      ((uint32_t)(1000))
/** 
 *  @}
 */ /* End of group LEDBlinkConfiguration */

#define MAIN_NULL       ((void *)0)     /*!< NULL. */

/* fIHOSC2: Min 9MHz,     TYP 10MHz, MAX 11MHz
   fEHOSC : Min 9.9MHz,   TYP 10MHz, MAX 10.1MHz
   fc     : Min 79.2MHz,  TYP 80MHz, MAX 80.8MHz  */
#define OFD_LOWER_COUNT  ((uint32_t)0x1CDU)
#define OFD_HIGHER_COUNT ((uint32_t)0x23EU)

#define OFD_LOWER_COUNT_EXTERNAL  ((uint32_t)0x39U)
#define OFD_HIGHER_COUNT_EXTERNAL ((uint32_t)0x48U)

#define CG_IHOSC2EN_ENABLE ((uint32_t)0x00000008U)
#define RSTFLG1_OFD_CLEAR  ((uint32_t)0xFFFFFFF7)

/** 
 *  @}
 */ /* End of group OFD_HW_TEST_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup OFD_HW_TEST_Private_define OFD_HW_TEST Private Define
 *  @{
 */
/* No Define */
/** 
 *  @}
 */ /* End of group OFD_HW_TEST_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup OFD_HW_TEST_Private_typedef OFD_HW_TEST Private Typedef
 *  @{
 */

/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
    t32a_t      t32a;                /*!< Application      :T32A.     */
    led_t       led[(uint32_t)BSP_LED_MAX]; /*!< Application :LED                */
    timer_t     timer;             /*!< Application :Timer (1sec timer). */
    gpio_t      gpio_obj;                   /*!< Application :GPIO               */
} instance_t;
/**
 *  @}
 */ /* End of group OFD_HW_TEST_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup OFD_HW_TEST_Private_variables OFD_HW_TEST Private Variables
 *  @{
 */

static instance_t instance; /*!< Instance.                    */
static uint32_t reset_flag = 0U;
static uint32_t status = 0U;
static uint32_t err_flag = 0U;
/**
 *  @}
 */ /* End of group OFD_HW_TEST_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup OFD_HW_TEST_Private_fuctions OFD_HW_TEST Private Fuctions
 *  @{
 */
static TXZ_Result application_initialize(void);
static void timer_interval_handler(uint32_t id);
static TXZ_Result driver_initialize(void);

/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    led_t *p_led = MAIN_NULL;

    /* Err Flag Check */
    if(err_flag == 0){
            /* OFD Running & No Error */
            p_led = &instance.led[0];
            led_turn_on(p_led);
    }else{
            /* OFD Abnormal Detected */
            p_led = &instance.led[1];
            led_turn_on(p_led);
    }
}
/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
   TXZ_Result result = TXZ_SUCCESS;
    /*----------------------*/
    /* LED                  */
    /*----------------------*/
    {
        uint32_t i;
        led_t *p_led = MAIN_NULL;

        for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
        {
            p_led = &instance.led[i];

            p_led->init.id                  = (uint32_t)p_led;
            p_led->init.p_gpio              = bsp_get_gpio_instance_address();
            p_led->init.port.group          = bsp_get_gpio_group_led((BSPLed)(i));
            p_led->init.port.num            = bsp_get_gpio_num_led((BSPLed)(i));
            p_led->init.blink.func          = TXZ_ENABLE;
            p_led->init.blink.interval.on   = CFG_LED_BLINK_TIME;
            p_led->init.blink.interval.off  = CFG_LED_BLINK_TIME;
            p_led->init.state               = LED_STATE_OFF;
            led_initialize(p_led);
        }
    }        
    return (result);
}
/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (p_t32a->p_instance != MAIN_NULL)
        {
            if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    {
        timer_t *p_timer = &instance.timer;

        p_timer->init.id       = (uint32_t)&instance.timer;
        p_timer->init.p_t32a   = &instance.t32a;p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
        p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
        p_timer->init.interval = 1000;
        p_timer->init.handler  = timer_interval_handler;
        timer_initialize(p_timer);
    }
    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver initialize will process within USB UART. */

    return (result);
}

/**
 *  @}
 */ /* End of group OFD_HW_TEST_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup OFD_HW_TEST_Exported_functions OFD_HW_TEST Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    uint32_t tmp = 0;
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /* OFD Reset flag Check */
   if((TSB_RLM->RSTFLG1 & 0x08) != 0U){
       reset_flag = 1;
       /* OFD Reset flag Clear */
       tmp = TSB_RLM->RSTFLG1;
       tmp &= RSTFLG1_OFD_CLEAR;
       TSB_RLM->RSTFLG1= (uint8_t)tmp;
    }else{
       /* IHOSC2 Enable */
       TSB_CG->OSCCR |= 0x8U;
    }
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /* Timer driver initialization */
    if (driver_initialize() != TXZ_SUCCESS)
    {
    }
    /*--- application ---*/
    if (application_initialize() != TXZ_SUCCESS)
    {
    }
    
    /*----------------------*/
    /* OFD                  */
    /*----------------------*/
    if(reset_flag == 0U)
    {
        /* Register Write Enable */
        REG_OFDWEN_enable(TSB_OFD);
        /* Reset Output Disable */
        REG_OFDRST_disable(TSB_OFD);
        /* EHOSC Monitor Enable */
        REG_OFDMON_set(TSB_OFD, REG_OFD_OFDMON_OFDMON_EHOSC);
        /* Frequency Detection Set */
        REG_OFDMN0_set(TSB_OFD, OFD_LOWER_COUNT_EXTERNAL);
        REG_OFDMX0_set(TSB_OFD, OFD_HIGHER_COUNT_EXTERNAL);
        REG_OFDMN1_set(TSB_OFD, OFD_LOWER_COUNT);
        REG_OFDMX1_set(TSB_OFD, OFD_HIGHER_COUNT);
        /* Frequency Detection Enable */
        REG_OFDEN_enable(TSB_OFD);
        /* OFD Status Register Check */
        while((status & REG_OFD_OFDSTAT_OFDBUSY_MASK) != REG_OFD_OFDSTAT_OFDBUSY_RUNNING)
        {
            REG_OFDSTAT_get(TSB_OFD, &status);
        }
        wait(1500);
        REG_OFDSTAT_get(TSB_OFD, &status);
        if((status & REG_OFD_OFDSTAT_FRQERR_MASK) == REG_OFD_OFDSTAT_FRQERR_NOERROR)
        {
            err_flag = 0;
            /* Frequency Detection Disable */
            REG_OFDEN_disable(TSB_OFD);
            /* Reset Output Enable */
            REG_OFDRST_enable(TSB_OFD);
            /* Frequency Detection Enable */
            REG_OFDEN_enable(TSB_OFD);
            /* Register Write Disable */
            REG_OFDWEN_disable(TSB_OFD);
            /* OFD Status Register Check */
            REG_OFDSTAT_get(TSB_OFD, &status);               
         }else{
            err_flag = 1;
            /* Frequency Detection Disable */
            REG_OFDEN_disable(TSB_OFD);
            /* Reset Output Enable */
            REG_OFDRST_enable(TSB_OFD);
            /* Frequency Detection Enable */
            REG_OFDEN_enable(TSB_OFD);
             /* ----------------- */
             /* OFD Reset Occurre */
             /* ----------------- */
        }  
    }else{
        /* OFD Reset Occurred */
        err_flag = 1;
        reset_flag = 0U;
    }
    /* Timer Start*/
    timer_start(&instance.timer);

    while(1){
        __NOP();
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  1ms Timer.
  * @param  timer :1ms Timer.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    default:
        /* no processing */
        break;
    }
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __nop();
    }
}
#endif
/**
 *  @}
 */ /* End of group OFD_HW_TEST_Exported_functions */

/**
 *  @}
 */ /* End of group OFD_HW_TEST */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
