/** 
 *******************************************************************************
 * @file    main.c
 * @brief   PORT Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif
/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup PORT_Sample_Appli PORT Sample Appli
 *  @{
 */

/** 
 *  @defgroup MAIN_PRIVATE  MAIN Private
 *  @brief    MAIN Private Group.
 *  @{
 */

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_sw.h"
#include "bsp_led.h"
#include "bsp_timer.h"

/*----------------------*/
/*  Timer Setting       */
/*----------------------*/
/**
*  @defgroup Timer_Setting Output Setting
*  @{
*/
#define CFG_PSW_INTERVAL           ((uint32_t)10)   /*!< PSW detection period(ms) */
#define CFG_LED_INTERVAL           ((uint32_t)10)   /*!< LED refrash period(ms) */
#define PSW_MASK                   ((uint32_t)0x00000007)  /* Recent 3bit detect */
/**
 *  *  @}
 *   */ /* End of group Output_Setting */



/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------*/ 
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
    t32a_t      t32a;                       /*!< Driver      :T32A.     */
    sw_t        sw[(uint32_t)BSP_PSW_MAX];  /*!< Application :Push SW            */
    led_t       led[(uint32_t)BSP_LED_MAX]; /*!< Application :LED                */
    gpio_t      gpio_obj;                   /*!< Application :GPIO. */
    timer_t     timer;                      /*!< Application :Timer (1ms timer). */
} instance_t;


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
static instance_t instance; /*!< Instance.                    */
static uint32_t count = 0 ;   /*!< 1ms * 10 loop count */
static uint32_t sw_sft[(uint32_t)BSP_PSW_MAX];  /*!< Push SW Shift Register  */
static uint32_t sw_flag = 0 ; /*!< Push SW Priority flag */
static uint32_t my_bit ;      /*!< Push SW My Bit */
static uint32_t all_off = 1 ; /*!< ALL SW Off */
/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
#define MAIN_NULL       ((void *)0)     /*!< NULL. */
/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
static TXZ_Result application_initialize(void);
static void timer_interval_handler(uint32_t id);


/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize
  * @param  -
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (p_t32a->p_instance != MAIN_NULL)
        {
            if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
   TXZ_Result result = TXZ_SUCCESS;
    /*----------------------*/
    /* LED                  */
    /*----------------------*/
    {
        uint32_t i;
        led_t *p_led = MAIN_NULL;

        for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
        {
            p_led = &instance.led[i];

            p_led->init.id                  = (uint32_t)p_led;
            p_led->init.p_gpio              = bsp_get_gpio_instance_address();
            p_led->init.port.group          = bsp_get_gpio_group_led((BSPLed)(i));
            p_led->init.port.num            = bsp_get_gpio_num_led((BSPLed)(i));
            p_led->init.blink.func          = TXZ_DISABLE;
            p_led->init.blink.interval.on   = 0;
            p_led->init.blink.interval.off  = 0;
            p_led->init.state           = LED_STATE_OFF;
            led_initialize(p_led);
        }
    }
    /*----------------------*/
    /* PSW                  */
    /*----------------------*/
    {
        uint32_t i;
        sw_t *p_sw = MAIN_NULL;

        for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
        {
            sw_sft[i] = 0 ;

            p_sw = &instance.sw[i];

            p_sw->init.id                   = (uint32_t)p_sw;
            p_sw->init.p_gpio               = bsp_get_gpio_instance_address();
            p_sw->init.port.group           = bsp_get_gpio_group_psw((BSPPsw)(i));
            p_sw->init.port.num             = bsp_get_gpio_num_psw((BSPPsw)(i));
            p_sw->init.polling.func         = TXZ_DISABLE;
            p_sw->init.polling.interval     = 0;
            p_sw->init.chattering.func      = TXZ_DISABLE;
            p_sw->init.chattering.count     = 0;
            p_sw->init.handler              = MAIN_NULL;
            sw_initialize(p_sw);
        }
    }

    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    {
        timer_t *p_timer = &instance.timer;

        p_timer->init.id       = (uint32_t)&instance.timer;
        p_timer->init.p_t32a   = &instance.t32a;
        p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
        p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
        p_timer->init.interval = 1000;
        p_timer->init.handler  = timer_interval_handler;
        timer_initialize(p_timer);
    }


    return (result);
}

/**
 *  @}
 */ /* End of group MAIN_PRIVATE */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
void irq_systick(void);
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /*--- driver      ---*/
    if (driver_initialize() != TXZ_SUCCESS)
    {
    }

    /*--- application ---*/
    if (application_initialize() != TXZ_SUCCESS)
    {
    }
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    timer_start(&instance.timer);


    /* Period for 10msec SYSTICK */

    /* dummy loop */
    while (1){
        __NOP();
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    uint32_t i;
    uint32_t det ;
    uint32_t sw_check ;
    led_t *p_led = MAIN_NULL;
    gpio_pinstate_t val = GPIO_PIN_SET;


    if (id == (uint32_t)&instance.timer)
    {
        /* Increment Tick. Tick is used to driver's Timeout. */
        hal_inc_tick();
        /* Interval Time Count. */
        count++;
        if (count >= CFG_PSW_INTERVAL)
        {

            sw_check = 0 ;
            /* Check SW Push(= GPIO_PIN_RESET) */

            for (i=0; i < 4; i++)
            {
    
                if(gpio_read_bit(&instance.gpio_obj, bsp_get_gpio_group_psw((BSPPsw)i), bsp_get_gpio_num_psw((BSPPsw)i), GPIO_Mode_DATA, &val) != TXZ_ERROR)
                {
                }
                if ( val == 0 ) 
                {
                    sw_check = 1 ;
                }
                sw_sft[i] = (sw_sft[i] <<1) + val ;
                my_bit = 0x01 <<i ;
                if(((sw_sft[i] & PSW_MASK) == PSW_MASK) && (sw_flag == my_bit) ) {  /* My LED Off */
                    det = 1 ;
                    sw_flag = 0 ;
                } else if(((sw_sft[i] & PSW_MASK) == 0x00000000) && (((sw_flag == 0) && (all_off == 1)) || (sw_flag == my_bit)) ) {   /* Only My LED On */
                    det = 0 ;
                    sw_flag = my_bit ;
                    all_off = 0 ;
                } else {                     /* Other LED Off */
                    det = 1 ;
                }

                

                if(det == GPIO_PIN_RESET){
                    p_led = &instance.led[i];
                    led_turn_on(p_led);
                }else{
                    p_led = &instance.led[i];
                    led_turn_off(p_led);
                }
            }

            if (sw_check == 0 )
            {
                all_off = 1 ;
            }
            count  = 0;
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  1ms Timer.
  * @param  timer :1ms Timer.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    default:
        /* no processing */
        break;
    }
}


#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __nop();
    }
}
#endif
/**
 *  @} 
 */ /* End of group PORT_Sample_Appli */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
