/** 
 *******************************************************************************
 * @file    main.c
 * @brief   RAMParity Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif
/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @addtogroup RAMParity RAM_PARITY Sample Appli
 *  @{
 */

/** 
 *  @defgroup MAIN_PRIVATE  MAIN Private
 *  @brief    MAIN Private Group.
 *  @{
 */

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "bsp.h"
#include "ramp.h"
#include "bsp_uart_io.h"
#include "bsp_com.h"
/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/* no define */

/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/*----------------------*/
/*  UART Setting        */
/*----------------------*/
/** 
 *  @defgroup UART_Setting UART Setting
 *  @{
 */
#define CFG_UART_BAUDRATE               ((uint32_t)115200)  /*!< Boudrate(bps) */
/** 
 *  @}
 */ /* End of group UART_Setting */

/** 
 *  @}
 */ /* End of group Timer_EVENT_COUNT_Private_define */
#define MAIN_NULL       ((void *)0)     /*!< NULL. */
#define USER_ID_REMOTE  ((uint32_t)0)   /*!< User ID :Remote Control (Any value) */
#define USER_ID_UART    ((uint32_t)1)   /*!< User ID :USB UART (Any value) */
/** 
 *  @defgroup UART_IO_SendMax  UART IO Send Max.
 *  @brief    Size of send max. (byte)
 *  @{
 */
#define UART_IO_SEND_MAX       ((uint32_t)32)
/**
 *  @}
 */ /* End of group UART_IO_SendMax */

/** 
 *  @defgroup UART_IO_ReceiveMax  UART IO Receive Max.
 *  @brief    Size of receive max. (byte)
 *  @{
 */
#define UART_IO_RECEIVE_MAX    ((uint32_t)32)
/**
 *  @}
 */ /* End of group UART_IO_ReceiveMax */
/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/*----------------------------------*/
/** 
 * @enum   Processing
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_IDLE,              /*!< Idle.                        */
    PROC_OUTPUT_MSG,        /*!< Output first message.        */
    PROC_INPUT,             /*!< Wait to input.               */
    PROC_COMMAND_PARSE,     /*!< Output Echo back data.       */
    PROC_WRITE_EXEC,        /*!< Output Echo back data.       */
    PROC_READ_EXEC,         /*!< Output Echo back data.       */
    PROC_OUTPUT_ERR,        /*!< Output input error message.  */
    PROC_ERROR,             /*!< Error.                       */
    PROC_PARAM_ERROR,       /*!< Parameter Error.             */
    PROC_PARITY_ERR         /*!< Parity Error.                */
} Processing;

/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/*----------------------------------*/
/** 
 * @struct input_t
 * @brief  Input data information.
*/
/*----------------------------------*/
typedef struct
{
    char    data[(32+1)];  /*!< Input data(+ for NULL 1byte). */
    uint8_t wp;                             /*!< Write pointer.                */
} input_t;
/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
static Processing proc;     /*!< Processing mode.        */
static input_t    input;        /*!< Input data information. */
static uint8_t write_data[10]; /*!< Write data buffer. */
static uint8_t ram1[10] __attribute__((section(".ARM.__at_0x20001000")));

/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
static void ram_clear(void);
static void clear_data(void);
static void write_data_clear(void);
static TXZ_Result application_initialize(void);

/*--------------------------------------------------*/
/** 
  * @brief  Clear RAM1.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void ram_clear(void)
{
    uint8_t i;
    for (i=0; i<10; i++)
    {
        ram1[i] = 0;
    }
}
/*--------------------------------------------------*/
/** 
  * @brief  Clear input data.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void clear_data(void)
{
    uint8_t i;

    for (i=0; i<(UART_IO_RECEIVE_MAX+1); i++)
    {
        input.data[i] = 0;
    }
    input.wp = 0;
}
/*--------------------------------------------------*/
/** 
  * @brief  Clear data.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void write_data_clear(void)
{
    uint8_t i;
    for (i=0; i<10; i++)
    {
        write_data[i] = 0;
    }
}
/**
 *  @}
 */ /* End of group UART_IO_PRIVATE */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
   TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = USER_ID_UART;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = (uint32_t)CFG_UART_BAUDRATE;
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result  = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    return (result);
}
/**
 *  @}
 */ /* End of group MAIN_PRIVATE */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    int num;
    uint32_t status = 0;
    /*----------------------*/
    /* Set Processing Mode  */
    /*----------------------*/
    proc = PROC_OUTPUT_MSG;
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /* BSP initialize       */
    bsp_initialize();
    /* Application initialization */
    if (application_initialize() != TXZ_SUCCESS)
    {
    }
    /*----------------------*/
    /* RAM Clear            */
    /*----------------------*/
    ram_clear();
    /*----------------------*/
    /* RAM Parity Set       */
    /*----------------------*/
    REG_RPAREN_enable(TSB_RPAR);
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(1)
    {
        switch(proc)
        {
        /*--- Output "First Message" ---*/
        case PROC_OUTPUT_MSG:
            {
                /* Clear input data.     */
                clear_data(); 
                /* Output first message. */
                if (printf("command > ") != -1 )
                {
                    proc = PROC_INPUT;
                }
                else
                {
                    proc = PROC_ERROR;
                }
            }
            break;
        /*--- Wait to Input ---*/
        case PROC_INPUT:
            {   
                int data;
                
                data = getchar();
                if ((data != -1) && (input.wp < UART_IO_RECEIVE_MAX))
                {
                    input.data[input.wp] = (char)data;
                    input.wp++;
                    switch ((char)data)
                    {
                    /*--- LF: line feed ---*/
                    case '\n':
                        proc = PROC_COMMAND_PARSE;
                        break;
                    /*--- default ---*/
                    default:
                        /* no process */
                        break;
                    }
                }
                else
                {
                    proc = PROC_OUTPUT_ERR;
                }
            }
            break;
        /*--- Output "Echo Back Data" ---*/
        case PROC_COMMAND_PARSE:
            {
                int i;
                if ((strncmp((const char *)&input.data[0], "write", 5)) == 0)
                {
                    num = input.wp - 6; /* len - (write (5charactors) + space(1 caractors)) */
                    if (num>11){
                        proc = PROC_PARAM_ERROR ;
                        break;
                    }
                    for(i = 0; i < num ; i++){
                        write_data[i] = (uint8_t)input.data[6 + i];
                    }
                    if( printf("\n") == -1){
                        proc = PROC_ERROR;
                    }
                    if(printf("write data > %s\n", (const char *)&write_data[0])  == -1){
                            proc = PROC_ERROR;
                    }
                    proc = PROC_WRITE_EXEC;
                    break;
                }
                else if ((strncmp((const char *)&input.data[0], "read", 4)) == 0)
                {
                    /* RAM AREA Reading */
                    if( printf("\n") == -1 ){
                        proc = PROC_ERROR;
                    }
                    if( printf("read data > ") == -1 ){
                        proc = PROC_ERROR;
                    }
                    if( printf("%s", (const char*)&ram1[0]) == -1 ){
                        proc = PROC_ERROR;
                    }
                    if( printf("\n") == -1 ){
                        proc = PROC_ERROR;
                    }
                    if ( proc != PROC_ERROR ) {
                        status = REG_RPARFGx_get(TSB_RPAR);
                        if(status != 0){
                            proc = PROC_PARITY_ERR;
                        }
                        else
                        {
                            proc = PROC_OUTPUT_MSG;
                        }
                    }
                }   
                else
                {
                    proc = PROC_OUTPUT_ERR;
                }
            }
            break;
                
        case PROC_WRITE_EXEC:
            {
                /* RAM AREA writing */
                if(strncpy((char *)ram1, (const char *)write_data,10) == NULL)
                {
                    proc = PROC_OUTPUT_ERR;
                }
                status = REG_RPARFGx_get(TSB_RPAR);
                if(status != 0){
                    proc = PROC_PARITY_ERR;
                }
                else
                {
                    /* write_data CLEAR */
                    write_data_clear();
                    proc = PROC_OUTPUT_MSG;
                }
            }
            break;
        /*--- Output "Input Error Message" ---*/
        case PROC_OUTPUT_ERR:
            {
                if (printf("\n Command Error !! \n") != -1 )
                {
                    proc = PROC_OUTPUT_MSG;
                }
                else
                {
                    proc = PROC_ERROR;
                }
            }
            break;
        /*--- Output "Parameter Error Message" ---*/
        case PROC_PARAM_ERROR:
            {
                if (printf("\n Parameter Error !! \n") != -1 )
                {
                    proc = PROC_OUTPUT_MSG;
                }
                else
                {
                    proc = PROC_ERROR;
                }
            }
            break;
        /*--- Output "Parity Error Message" ---*/
        case PROC_PARITY_ERR:
            {
                uint32_t area_num ;
        area_num = status & 0x000000ff;
                if (printf("\n RAM AREA %d Parity Error !! \n", area_num) != -1 )
                {
                    proc = PROC_OUTPUT_MSG;
                }
                else
                {
                    proc = PROC_ERROR;
                }
            }
            break;
        /*--- Error ---*/
        case PROC_ERROR:
            /* System Error. Please Debug. */
#ifdef DEBUG
            assert_failed(MAIN_NULL, 0);
#endif
            break;
        /*--- Default ---*/
        default:
            proc = PROC_ERROR;
            break;
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}


/*--------------------------------------------------*/
/** 
  * @brief  RAM Parity IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_parity(void)
{
    uint32_t address;
    address = REG_RPAREADx_get(TSB_RPAR, 0);
    if(address != 0){
        if (printf("RAM AREA 0 Parity Error\n") == -1){
            proc = PROC_PARITY_ERR;
        }
        REG_RPARCLR_clear(TSB_RPAR, 0);
    }
    address = REG_RPAREADx_get(TSB_RPAR, 1);
    if(address != 0){
        if (printf("RAM AREA 1 Parity Error\n") == -1){
            proc = PROC_PARITY_ERR;
        }
        REG_RPARCLR_clear(TSB_RPAR, 1);
    }
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __NOP();
    }
}
#endif

/**
 *  @} 
 */ /* End of group RAMParity */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
