/** 
 *******************************************************************************
 * @file    main.c
 * @brief   RMC UART Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_uart_io.h"
#include "bsp_timer.h"
#include "rmc.h"

/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup RMC_Sample_Appli RMC Sample Appli
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/*--- for Debug ---*/
#ifdef DEBUG
    #if 1
        #define REMOTE_CFG_DEBUG
    #endif
#endif
/** 
 *  @defgroup RMC_Private_macro RMC Private Macro
 *  @{
 */
 /** 
 *  @defgroup NULL_Pointer Null Pointer
 *  @{
 */
#define REMOTE_NULL                ((void *)0)
/**
 *  @}
 */ /* End of defgroup NULL_Pointer */

/** 
 *  @defgroup FormatSampleDefinition Format Sample Definition
 *  @brief Format Type = Sample.
 *  @{
 */
#define FMT_SAMPLE_CUSTOM_BIT_NUM   (16UL)      /*!< Num of bits :Custom Code        */
#define FMT_SAMPLE_DATA_BIT_NUM     (8UL)       /*!< Num of bits :Data Code          */
#define FMT_SAMPLE_R_DATA_BIT_NUM   (8UL)       /*!< Num of bits :Reversed Data Code */
#define FMT_SAMPLE_TOTAL_BIT_NUM    ((uint32_t)(FMT_SAMPLE_CUSTOM_BIT_NUM + FMT_SAMPLE_DATA_BIT_NUM + FMT_SAMPLE_R_DATA_BIT_NUM))  /*!< Num of bits :Total */

#define REMOTE_RX_LIST_NUM_MAX          (8UL)              /*!< Max Num of Rx List.      */

#define TYPE_NEC_LCMAX                  (0x00000070UL)       /* Period Max    :13.7(msec)  */
/* #define TYPE_NEC_LCMIN                  (0x0000006CUL)  */     /* Period Min    :13.3(msec)  13.3 / 108 = 0.123, N * 4 / 32768 */
#define TYPE_NEC_LCMIN                  (0x0000005AUL)       /* Period Min    :11.0(msec)  */
#define TYPE_NEC_LLMAX                  (0x0000004BUL)       /* Low Width Max : 9.2(msec)  */
#define TYPE_NEC_LLMIN                  (0x00000048UL)       /* Low Width Min : 8.8(msec)  */
#define TYPE_NEC_RECEIVEDEND            (0x000000FEUL)       /* Received End  : 7.8(msec)  */
#define TYPE_NEC_THRESHOLD              (0x00000037UL)       /* Threshold     : 1.7(msec)  */

#define TYPE_TOSHIBA_LCMAX              (0x0000004BUL)       /* Period Max    : 9.2(msec)  */
/* #define TYPE_TOSHIBA_LCMIN              (0x00000048UL)  */   /* Period Min    : 8.8(msec)  */
#define TYPE_TOSHIBA_LCMIN              (0x00000035UL)       /* Period Min    : 6.5(msec)  */
#define TYPE_TOSHIBA_LLMAX              (0x00000026UL)       /* Low Width Max : 4.6(msec)  */
#define TYPE_TOSHIBA_LLMIN              (0x00000023UL)       /* Low Width Min : 4.3(msec)  */
#define TYPE_TOSHIBA_RECEIVEDEND        (0x000000FEUL)       /* Received End  : 7.8(msec)  */
#define TYPE_TOSHIBA_THRESHOLD          (0x00000037UL)       /* Threshold     : 1.7(msec)  */

/* #define TYPE_AEHA_LCMAX                 (0x0000002CUL)  */     /* Period Max    : 5.3(msec)  */
#define TYPE_AEHA_LCMAX                 (0x0000003AUL)       /* Period Max    : 7.0(msec)  */
#define TYPE_AEHA_LCMIN                 (0x00000028UL)       /* Period Min    : 4.9(msec)  */
#define TYPE_AEHA_LLMAX                 (0x0000001EUL)       /* Low Width Max : 3.7(msec)  */
#define TYPE_AEHA_LLMIN                 (0x0000001AUL)       /* Low Width Min : 3.2(msec)  */
#define TYPE_AEHA_RECEIVEDEND           (0x000000FEUL)       /* Received End  : 7.8(msec)  */
#define TYPE_AEHA_THRESHOLD             (0x00000028UL)       /* Threshold     : 1.7(msec)  */

#define RMC_CFG_NEC_FORMAT       (0)                         /* NEC Format     */
#define RMC_CFG_TOSHIBA_FORMAT   (1)                         /* TOSHIBA Format */
#define RMC_CFG_AEHA_FORMAT      (2)                         /* AEHA Format    */
/* #define RMC_CFG_FORMAT       (RMC_CFG_AEHA_FORMAT) */
#define RMC_CFG_FORMAT       (RMC_CFG_NEC_FORMAT) 
/* #define RMC_CFG_FORMAT       (RMC_CFG_TOSHIBA_FORMAT) */

#define RMC_LD_INT_FLAG_OFF         ((uint32_t)0x00000000)    /*!< It doesn't occur. */
#define RMC_LD_INT_FLAG_ON          ((uint32_t)0x00000001)    /*!< It occurs.        */
#define RMC_REPEAT_FLAG_OFF         ((uint32_t)0x00000000)    /*!< It doesn't occur. */
#define RMC_REPEAT_FLAG_ON          ((uint32_t)0x00000001)    /*!< It occurs.        */


#define MAIN_NULL       ((void *)0)     /*!< NULL. */
#define USER_ID_REMOTE  ((uint32_t)0)   /*!< User ID :Remote Control (Any value) */
#define USER_ID_UART    ((uint32_t)1)   /*!< User ID :USB UART (Any value) */

/** 
 *  @}
 */ /* End of group RMC_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup RMC_Private_define RMC Private Define
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief  Format Type.
*/
/*----------------------------------*/
typedef enum {
    REMOTE_FMT_TYPE_NEC     = 0U,     /*!< NEC Format      */
    REMOTE_FMT_TYPE_TOSHIBA = 1U,     /*!< TOSHIBA Format  */
    REMOTE_FMT_TYPE_AEHA    = 2U      /*!< AEHA Format     */
} RemoteFormatType;

/*----------------------------------*/
/** 
 * @enum   Processing
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_IDLE,          /*!< Idle.      */
    PROC_START,         /*!< Start.     */
    PROC_STOP,          /*!< Stop.      */
    PROC_RUN,           /*!< Running.   */
    PROC_ERROR          /*!< Error.     */
} Processing;
/** 
 *  @}
 */ /* End of group RMC_Exported_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Private_typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief  Receive Data Information.
*/
/*----------------------------------*/
typedef struct
{
    uint32_t    data[RMC_DATA_BUFF_NUM]; /*!< Rx Data.        */
    uint8_t     bits;                    /*!< Num of Rx Bits. */
} remote_private_rx_info_t;

/*----------------------------------*/
/** 
 * @brief  Receive Data List.
*/
/*----------------------------------*/
typedef struct
{
    uint32_t            num;    /*!< Num of RMC RX Data. */
    uint32_t            wp;     /*!< List Write Pointer. */
    uint32_t            rp;     /*!< List Read Pointer.  */
    remote_private_rx_info_t info[REMOTE_RX_LIST_NUM_MAX]; /*!< Receive Data Information. */
} remote_private_rx_list_t;

/*----------------------------------*/
/** 
 * @brief  Private Information.
*/
/*----------------------------------*/
typedef struct
{
    rmc_t  rmc;                      /*!< RMC Instance.      */
    remote_private_rx_list_t list;   /*!< Receive Data List.   */
} remote_private_info_t;

/**
 *  @}
 */ /* End of group RMC_Private_typedef */


/** 
 *  @addtogroup RMC_Exported_typedef
 *  @{
 */

/*----------------------------------*/
/** 
 * @brief  Remote data. \n
 *         for Format Type = Smaple.
*/
/*----------------------------------*/
typedef struct
{
    uint32_t customer;              /*!< Customer Code.  */
    uint8_t  data;                  /*!< Data Code.      */
    uint8_t  rdata;                 /*!< Data Code (Reverse). */
} remote_data_fmt_sample_t;

/*----------------------------------*/
/** 
 * @brief  Remote data. \n
 *         for Format Type = Smaple.
*/
/*----------------------------------*/
typedef struct
{
    uint8_t bit;                    /*!< Num of Data bits */
    uint8_t  customer[2];              /*!< Data Code.      */
    uint8_t  data[10];              /*!< Data Code.      */
} remote_data_fmt_aeha_t;

/*----------------------------------*/
/** 
 * @brief  Remote data.
*/
/*----------------------------------*/
typedef union
{
    remote_data_fmt_sample_t sample;    /*!< Format Type :Smaple */
    remote_data_fmt_aeha_t   aeha;      /*!< Format Type :AEHA   */
} remote_data_t;

/*----------------------------------*/
/** 
 * @brief  Initial setting. \n
 *         User must be set.
*/
/*----------------------------------*/
typedef struct
{
    uint32_t              id;           /*!< User ID.            */
    RemoteFormatType      format;       /*!< Format Type.        */
    void (*receive_data)(uint32_t id);  /*!< Event handler. Notify of receive data. */
} remote_initial_setting_t;

/*----------------------------------*/
/** 
 * @brief  Remote Control Class.
*/
/*----------------------------------*/
typedef struct
{
    remote_initial_setting_t init;   /*!< Initial setting.     */
    remote_private_info_t    info;   /*!< Private information. */
} remote_t;

/** 
 *  @}
 */ /* End of group RMC_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup RMC_Private_typedef RMC Private Typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
    t32a_t      t32a;              /*!< Driver      :T32A.     */
    timer_t     timer;             /*!< Application :Timer (1ms timer). */
    remote_t    *p_remote;         /*!< Application :Remote Control.  */
} instance_t;
/**
 *  @}
 */ /* End of group RMC_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup RMC_Private_variables RMC Private Variables
 *  @{
 */
static instance_t instance; /*!< Instance.                    */
static Processing proc;     /*!< Processing mode.             */

static remote_t rem_instance;    /*!< remote Instance. */
static uint32_t ld_int_flg = RMC_LD_INT_FLAG_OFF;
static uint32_t repeat_flg = RMC_REPEAT_FLAG_OFF;
/**
 *  @}
 */ /* End of group RMC_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup RMC_Private_fuctions RMC Private Fuctions
 *  @{
 */
static void variable_initialize(void);
static TXZ_Result driver_initialize(void);
static TXZ_Result driver_finalize(void);
static TXZ_Result application_initialize(void);
static TXZ_Result application_finalize(void);
static void timer_interval_handler(uint32_t id);
static void receive_data_handler(uint32_t id);
static TXZ_Result print_code(uint8_t src);
static TXZ_Result print_aeha_code(uint8_t src);
remote_t *remote_initialize(remote_initial_setting_t *p_param);
void remote_finalize(remote_t *p_instance);
void remote_start(remote_t *p_instance);
void remote_stop(remote_t *p_instance);
TXZ_Result remote_get_list_data(remote_t *p_instance, remote_data_t *p_data);
void remote_irq_handler(remote_t *p_instance, uint32_t status);

static void clear_rx_list(remote_private_rx_list_t *p_list);
static void debug_error(void);

/*--------------------------------------------------*/
/** 
  * @brief  Error Hook
  * @param  -
  * @retval -
  * @note   for debug
  */
/*--------------------------------------------------*/
static void debug_error(void)
{
#ifdef REMOTE_CFG_DEBUG
    while(1)
    {
        __nop();
    }
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  Clear Receive Data List.
  * @param  p_list :List Source Address.
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void clear_rx_list(remote_private_rx_list_t *p_list)
{
    p_list->num = 0;
    p_list->wp  = 0;
    p_list->rp  = 0;
}

/**
 *  @}
 */ /* End of group RMC_Private_fuctions */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Exported_functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Initialize.
  * @param  p_param :Initial Setting Information Source Address.
  * @return Instance Address.
  * @retval (value != NULL) :Valid value.
  * @note   Parameter check isn't performed.
  * @attention Singleton.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
remote_t *remote_initialize(remote_initial_setting_t *p_param)
{
    rmc_t *p_rmc = &rem_instance.info.rmc;

    /*------------------------------*/
    /*  Initial Parameter Copy      */
    /*------------------------------*/
    rem_instance.init.id            = p_param->id;
    rem_instance.init.format        = p_param->format;
    rem_instance.init.receive_data  = p_param->receive_data;
    /*------------------------------*/
    /*  Clear Private Information   */
    /*------------------------------*/
    clear_rx_list(&rem_instance.info.list);
    /*------------------------------*/
    /*  Construct                   */
    /*------------------------------*/
    /* Register Allocation */
    p_rmc->p_instance = bsp_get_rmc_reg(BSP_RMC_SAMPLE);
    /*------------------------------*/
    /*  Initialize                  */
    /*------------------------------*/
    /*--- Format Type = Smaple ---*/
    if (rem_instance.init.format == REMOTE_FMT_TYPE_NEC)
    {
        /* Select source clock. "fs" */
        p_rmc->init.fssel.clk  = RMC_CLK_LOW_SPEED;         /* low speed     :32.768(kHz) */
        /* Receive Control Setting "1". */
        p_rmc->init.cnt1.lcMax = TYPE_NEC_LCMAX;            /* Period Max    :13.7(msec)  */
        p_rmc->init.cnt1.lcMin = TYPE_NEC_LCMIN;            /* Period Min    :13.3(msec)  */
        p_rmc->init.cnt1.llMax = TYPE_NEC_LLMAX;            /* Low Width Max : 9.2(msec)  */
        p_rmc->init.cnt1.llMin = TYPE_NEC_LLMIN;            /* Low Width Min : 8.8(msec)  */
        /* Receive Control Setting "2". */
        p_rmc->init.cnt2.lien  = RMC_LI_ENABLE;             /* Enable                     */
        p_rmc->init.cnt2.edien = RMC_EDI_DISABLE;           /* Disable                    */
        p_rmc->init.cnt2.cld   = RMC_LD_DISABLE;            /* Disable                    */
        p_rmc->init.cnt2.phim  = RMC_PHM_DISABLE;           /* Disable                    */
        p_rmc->init.cnt2.ll    = RMC_LL_MAX;                /* Disable                    */
        p_rmc->init.cnt2.dmax  = TYPE_NEC_RECEIVEDEND;      /* Received End  : 7.8(msec)  */
        /* Receive Control Setting "3". */
        p_rmc->init.cnt3.dath  = RMC_THRESH_HIGH_MIN;       /* Disable                    */
        p_rmc->init.cnt3.datl  = TYPE_NEC_THRESHOLD;            /* Threshold     : 1.7(msec)  */
        /* Receive Control Setting "4". */
        p_rmc->init.cnt4.po    = RMC_POLARITY_POSITIVE;     /* Positive                  */
        p_rmc->init.cnt4.nc    = RMC_NOISE_REDUCTION_MIN;   /* Disable                   */
        /* Receive End Bit1 setting. */
        p_rmc->init.end1.end1  = RMC_RX_DATA_BITS_MIN;      /* Disable                   */
        /* Receive End Bit2 setting. */
        p_rmc->init.end2.end2  = RMC_RX_DATA_BITS_MIN;      /* Disable                   */
        /* Receive End Bit3 setting. */
        p_rmc->init.end3.end3  = RMC_RX_DATA_BITS_MIN;      /* Disable                   */
    }
    else if (rem_instance.init.format == REMOTE_FMT_TYPE_TOSHIBA)
    {
        p_rmc->init.fssel.clk  = RMC_CLK_LOW_SPEED;         /* low speed     :32.768(kHz) */
        /* Receive Control Setting "1". */
        p_rmc->init.cnt1.lcMax = TYPE_TOSHIBA_LCMAX;        /* Period Max    : 9.2(msec)  */
        p_rmc->init.cnt1.lcMin = TYPE_TOSHIBA_LCMIN;        /* Period Min    : 8.8(msec)  */
        p_rmc->init.cnt1.llMax = TYPE_TOSHIBA_LLMAX;        /* Low Width Max : 4.7(msec)  */
        p_rmc->init.cnt1.llMin = TYPE_TOSHIBA_LLMIN;        /* Low Width Min : 4.3(msec)  */
        /* Receive Control Setting "2". */
        p_rmc->init.cnt2.lien  = RMC_LI_ENABLE;             /* Enable                     */
        p_rmc->init.cnt2.edien = RMC_EDI_DISABLE;           /* Disable                    */
        p_rmc->init.cnt2.cld   = RMC_LD_DISABLE;            /* Disable                    */
        p_rmc->init.cnt2.phim  = RMC_PHM_DISABLE;           /* Disable                    */
        p_rmc->init.cnt2.ll    = RMC_LL_MAX;                /* Disable                    */
        p_rmc->init.cnt2.dmax  = TYPE_TOSHIBA_RECEIVEDEND;  /* Received End  : 7.8(msec)  */
        /* Receive Control Setting "3". */
        p_rmc->init.cnt3.dath  = RMC_THRESH_HIGH_MIN;       /* Disable                    */
        p_rmc->init.cnt3.datl  = TYPE_TOSHIBA_THRESHOLD;    /* Threshold     : 1.7(msec)  */
        /* Receive Control Setting "4". */
        p_rmc->init.cnt4.po    = RMC_POLARITY_POSITIVE;     /* Positive                  */
        p_rmc->init.cnt4.nc    = RMC_NOISE_REDUCTION_MIN;   /* Disable                   */
        /* Receive End Bit1 setting. */
        p_rmc->init.end1.end1  = RMC_RX_DATA_BITS_MIN;      /* Disable                   */
        /* Receive End Bit2 setting. */
        p_rmc->init.end2.end2  = RMC_RX_DATA_BITS_MIN;      /* Disable                   */
        /* Receive End Bit3 setting. */
        p_rmc->init.end3.end3  = RMC_RX_DATA_BITS_MIN;      /* Disable                   */
    }
    else if (rem_instance.init.format == REMOTE_FMT_TYPE_AEHA)
    {
        /* Select source clock. "fs" */
        p_rmc->init.fssel.clk  = RMC_CLK_LOW_SPEED;         /* low speed     :32.768(kHz) */
        /* Receive Control Setting "1". */
        p_rmc->init.cnt1.lcMax = TYPE_AEHA_LCMAX;           /* Period Max    : 5.4(msec)  */
        p_rmc->init.cnt1.lcMin = TYPE_AEHA_LCMIN;           /* Period Min    : 4.9(msec)  */
        p_rmc->init.cnt1.llMax = TYPE_AEHA_LLMAX;           /* Low Width Max : 3.7(msec)  */
        p_rmc->init.cnt1.llMin = TYPE_AEHA_LLMIN;           /* Low Width Min : 3.2(msec)  */
        /* Receive Control Setting "2". */
        p_rmc->init.cnt2.lien  = RMC_LI_ENABLE;             /* Enable                     */
        p_rmc->init.cnt2.edien = RMC_EDI_DISABLE;           /* Disable                    */
        p_rmc->init.cnt2.cld   = RMC_LD_DISABLE;            /* Disable                    */
        p_rmc->init.cnt2.phim  = RMC_PHM_DISABLE;           /* Disable                    */
        p_rmc->init.cnt2.ll    = RMC_LL_MAX;                /* Disable                    */
        p_rmc->init.cnt2.dmax  = TYPE_AEHA_RECEIVEDEND;     /* Received End  : 7.8(msec)  */
        /* Receive Control Setting "3". */
        p_rmc->init.cnt3.dath  = RMC_THRESH_HIGH_MIN;       /* Disable                    */
        p_rmc->init.cnt3.datl  = TYPE_AEHA_THRESHOLD;       /* Threshold     : 1.2(msec)  */
        /* Receive Control Setting "4". */
        p_rmc->init.cnt4.po    = RMC_POLARITY_POSITIVE;     /* Positive                  */
        p_rmc->init.cnt4.nc    = RMC_NOISE_REDUCTION_MIN;   /* Disable                   */
        /* Receive End Bit1 setting. */
        p_rmc->init.end1.end1  = RMC_RX_DATA_BITS_MIN;      /* Disable                   */
        /* Receive End Bit2 setting. */
        p_rmc->init.end2.end2  = RMC_RX_DATA_BITS_MIN;      /* Disable                   */
        /* Receive End Bit3 setting. */
        p_rmc->init.end3.end3  = RMC_RX_DATA_BITS_MIN;      /* Disable                   */
    }
    /*--- Other Type :Not Support ---*/
    else
    {
        debug_error();
    }
    /* initialize */
    if (rmc_init(p_rmc) != TXZ_SUCCESS)
    {
        debug_error();
    }

    return(&rem_instance);
}

/*--------------------------------------------------*/
/** 
  * @brief  Finalize.
  * @param  p_instance :Instance Source Address.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void remote_finalize(remote_t *p_instance)
{
    rmc_t *p_rmc = &p_instance->info.rmc;

    /*------------------------------*/
    /*  Stop                        */
    /*------------------------------*/
    remote_stop(p_instance);
    /*------------------------------*/
    /*  Finalize                    */
    /*------------------------------*/
    if (rmc_deinit(p_rmc) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*------------------------------*/
    /*  Destruct                    */
    /*------------------------------*/
    /* Register Release */
    p_rmc->p_instance = REMOTE_NULL;
}

/*--------------------------------------------------*/
/** 
  * @brief  Start to receive.
  * @param  p_instance :Instance Source Address.
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void remote_start(remote_t *p_instance)
{
    rmc_t *p_rmc = &p_instance->info.rmc;

    /*--- RMC    ---*/
    if (rmc_rx_enable(p_rmc) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- INTRMC ---*/
    bsp_irq_rmc_enable();
    /*--- NVIC   ---*/
    NVIC_EnableIRQ(bsp_get_rmc_nvic(BSP_RMC_SAMPLE));
}

/*--------------------------------------------------*/
/** 
  * @brief  Stop to receive.
  * @param  p_instance :Instance Source Address.
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void remote_stop(remote_t *p_instance)
{
    rmc_t *p_rmc = &p_instance->info.rmc;

    /*--- NVIC   ---*/
    NVIC_DisableIRQ(bsp_get_rmc_nvic(BSP_RMC_SAMPLE));
    /*--- RMC    ---*/
    if (rmc_rx_disable(p_rmc) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- INTRMC ---*/
    bsp_irq_rmc_disable();
}

/*--------------------------------------------------*/
/** 
  * @brief  Get Receive Data.
  * @param  p_instance :Instance Source Address.
  * @param  p_data     :Receive Data Destination Address.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :List Empty. or \n
                        :Not Support Format Type.
  * @note   Parameter check isn't performed.
  * @attention This function is not available in interrupt.
  * @attention In progress, RMC Interrupt Disable.
  */
/*--------------------------------------------------*/
TXZ_Result remote_get_list_data(remote_t *p_instance, remote_data_t *p_data)
{
    TXZ_Result res = TXZ_ERROR;
    if (p_instance->info.list.num != 0)
    {
        remote_private_rx_info_t *p_info = &p_instance->info.list.info[p_instance->info.list.rp];

        /*------------------------------*/
        /*  Get Data                    */
        /*------------------------------*/
        /*--- Format Type = Smaple ---*/
        if ((rem_instance.init.format == REMOTE_FMT_TYPE_NEC) || (rem_instance.init.format == REMOTE_FMT_TYPE_TOSHIBA))
        {
            if (p_info->bits == FMT_SAMPLE_TOTAL_BIT_NUM)
            {
                p_data->sample.customer = (uint16_t)((p_info->data[0] & 0xFFFF0000) >> 16);
                p_data->sample.data     = (uint8_t)((p_info->data[0] & 0x0000FF00) >> 8);
                p_data->sample.rdata    = (uint8_t)(p_info->data[0] & 0x000000FF);
                /* Success !! */
                res = TXZ_SUCCESS;
            }
        }
        else if (rem_instance.init.format == REMOTE_FMT_TYPE_AEHA)
        {
                uint32_t length = 0;
                uint32_t num = 0;
                uint32_t sf = 0;
                uint32_t i = 0;
                uint32_t mask;
                
                p_data->aeha.bit = p_info->bits;
                
                length = p_data->aeha.bit;
                if(length > 71)
                {
                    length = 71;
                }
                i = 0;
                while((length > 0) && (i<2))
                {
                    num = (length-1) / 32;
                    
                    mask = 0x000000FF;
                    sf = length % 32;
                    if(sf ==  0) sf = 32;
                    if(sf < 8)
                    {
                        uint8_t temp;
                        
                        mask >>= 8 - sf;
                        temp = (uint8_t)((p_info->data[num] & mask) << (8-sf));
                        if((length > 32) && (num > 0))
                        {
                            mask = 0xFF000000;
                            mask <<= sf;
                            temp |= (uint8_t)((p_info->data[num-1] & mask) >> (24+sf));
                        }
                        p_data->aeha.customer[i] = temp;
                    }
                    else{
                        sf -= 8;
                        mask <<= sf;
                        p_data->aeha.customer[i] = (uint8_t)((p_info->data[num] & mask) >> sf);
                    }
                    if(length < 8){
                        length = 0;
                    }
                    else
                    {
                        length -= 8;
                    }
                    ++i;
                }
                i = 0;
                while((length > 0) && (i<10))
                {
                    num = (length-1) / 32;
                    
                    mask = 0x000000FF;
                    sf = length % 32;
                    if(sf ==  0) sf = 32;
                    if(sf < 8)
                    {
                        uint8_t temp;
                        
                        mask >>= 8 - sf;
                        temp = (uint8_t)((p_info->data[num] & mask) << (8-sf));
                        if((length > 32) && (num > 0))
                        {
                            mask = 0xFF000000;
                            mask <<= sf;
                            temp |= (uint8_t)((p_info->data[num-1] & mask) >> (24+sf));
                        }
                        p_data->aeha.data[i] = temp;
                    }
                    else{
                        sf -= 8;
                        mask <<= sf;
                        p_data->aeha.data[i] = (uint8_t)((p_info->data[num] & mask) >> sf);
                    }
                    if(length < 8){
                        length = 0;
                    }
                    else
                    {
                        length -= 8;
                    }
                    ++i;
                }
                /* Success !! */
                res = TXZ_SUCCESS;
        }
        /*------------------------------*/
        /*  Update List                 */
        /*------------------------------*/
        /*--- Read Pointer ---*/
        p_instance->info.list.rp++;
        if (p_instance->info.list.rp >= REMOTE_RX_LIST_NUM_MAX)
        {
            p_instance->info.list.rp = 0;
        }
        /*--- List Num ---*/
        /*>>> Critical information. */
        NVIC_DisableIRQ(bsp_get_rmc_nvic(BSP_RMC_SAMPLE));
        p_instance->info.list.num -= 1;
        NVIC_EnableIRQ(bsp_get_rmc_nvic(BSP_RMC_SAMPLE));
        /*<<< Critical information. */
    }

    return (res);
}

/*--------------------------------------------------*/
/** 
  * @brief  IRQ Handler.
  * @param  p_instance :Instance Source Address.
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void remote_irq_handler(remote_t *p_instance, uint32_t status)
{ 
    rmc_t *p_rmc = &p_instance->info.rmc;
    /*------------------------------*/
    /*  Check Status                */
    /*------------------------------*/
        
    /* Check to the leader detection interrupt flag. */
    if ((status & RMC_LD_INT_OCCUR) == RMC_LD_INT_OCCUR)
    {
        /* leader detection interrupt ON */
        ld_int_flg =  RMC_LD_INT_FLAG_ON;
    }
    /* Check to the data bit cycle max interrupt flag. */
    if (((status & RMC_CYCLE_MAX_INT_OCCUR) == RMC_CYCLE_MAX_INT_OCCUR) && 
        (ld_int_flg ==  RMC_LD_INT_FLAG_ON))
    {
     
        remote_private_rx_info_t *p_info = &p_instance->info.list.info[p_instance->info.list.wp];
        /* leader detection interrupt CLEAR */
        ld_int_flg =  RMC_LD_INT_FLAG_OFF;
        /*------------------------------*/
        /*  Update List                 */
        /*------------------------------*/
        /*--- Bits ---*/
        p_info->bits = (uint8_t)(status & RMC_RX_BIT_NUM_MASK);
    if(p_info->bits == 0)
    {
            repeat_flg = RMC_REPEAT_FLAG_ON ;
    }
        /*--- Data ---*/
        if (rmc_get_data(p_rmc, &p_info->data[0]) != TXZ_SUCCESS)
        {
            debug_error();
        }
        /*--- Write Pointer ---*/
        p_instance->info.list.wp++;
        if (p_instance->info.list.wp >= REMOTE_RX_LIST_NUM_MAX)
        {
             p_instance->info.list.wp = 0;
        }
        /*--- List Num ---*/
        /*>>> Critical information. */
        p_instance->info.list.num += 1;
        /*<<< Critical information. */
        /*------------------------------*/
        /*  Call Handler                */
        /*------------------------------*/
        if (p_instance->init.receive_data != REMOTE_NULL)
        {
            p_instance->init.receive_data(p_instance->init.id);
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Variable Initialize
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void variable_initialize(void)
{
    /* Instance. */
    instance.t32a.p_instance    = MAIN_NULL;
    instance.timer.init.p_t32a  = MAIN_NULL;
    instance.p_remote           = MAIN_NULL;
    /* Proccesing. */
    proc = PROC_START;
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (p_t32a->p_instance != MAIN_NULL)
        {
            if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver initialize will process within USB UART. */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Finalize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver finalize has done within USB UART.    */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Finalize ---*/
        /* no processing */
        /*--- Destruct ---*/
        /* Register Release */
        p_t32a->p_instance = MAIN_NULL;
    }
    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver finalize will process within BSP.     */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    {
        timer_t *p_timer = &instance.timer;

        p_timer->init.id       = (uint32_t)&instance.timer;
        p_timer->init.p_t32a   = &instance.t32a;
        p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
        p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
        p_timer->init.interval = 1000;
        p_timer->init.handler  = timer_interval_handler;
        timer_initialize(p_timer);
    }
    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = USER_ID_UART;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = bsp_get_usb_uart_tx_baudrate(BSP_USB_UART_0);
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result  = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Remote Control       */
    /*----------------------*/
    {
        remote_initial_setting_t param;

        param.id            = USER_ID_REMOTE;
#if (RMC_CFG_FORMAT == RMC_CFG_NEC_FORMAT)
        param.format        = REMOTE_FMT_TYPE_NEC;
#elif (RMC_CFG_FORMAT == RMC_CFG_TOSHIBA_FORMAT)
        param.format        = REMOTE_FMT_TYPE_TOSHIBA;
#elif (RMC_CFG_FORMAT == RMC_CFG_AEHA_FORMAT)
        param.format        = REMOTE_FMT_TYPE_AEHA;
#endif
        param.receive_data  = receive_data_handler;
        instance.p_remote = remote_initialize(&param);
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Finalize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Remote Control       */
    /*----------------------*/
    if (instance.p_remote != MAIN_NULL)
    {
        remote_finalize(instance.p_remote);
    }
    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    {
        uart_io_finalize();
    }
    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    if (instance.timer.init.p_t32a != MAIN_NULL)
    {
        timer_finalize(&instance.timer);
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    if (id == (uint32_t)&instance.timer)
    {
        /* Increment Tick. Tick is used to driver's Timeout. */
        hal_inc_tick();
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Receive Data Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   Call by Remote Control.
  */
/*--------------------------------------------------*/
static void receive_data_handler(uint32_t id)
{
    if (id == USER_ID_REMOTE)
    {
        /* Sample Soft don't use. */
    }
}
/*--------------------------------------------------*/
/** 
  * @brief  Print Code.
  * @param  src   :Output Data.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result print_code(uint8_t src)
{
    TXZ_Result res = TXZ_SUCCESS;
    int32_t i = 0;

    if (printf("=>") < 0)
    {
        res = TXZ_ERROR;
    }
    for (i=7; i>=0; i--)
    {
        uint8_t dst = (uint8_t)((src & (uint8_t)(1 << i)) >> i);
        if (printf("%1d", dst) < 0)
        {
            res = TXZ_ERROR;
        }
    } 
    if (printf(":0x%02X  \n", src) < 0)
    {
        res = TXZ_ERROR;
    }

    return (res);
}
/*--------------------------------------------------*/
/** 
  * @brief  Print AEHA Code.
  * @param  src   :Output Data.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result print_aeha_code(uint8_t src)
{
    TXZ_Result res = TXZ_SUCCESS;
    int32_t i = 0;
    uint8_t tmp = 0;
    uint8_t tmp_l = 0;
    uint8_t tmp_h = 0;

    if (printf("=>") < 0)
    {
        res = TXZ_ERROR;
    }
    for (i=7; i>=0; i--)
    {
        uint8_t dst = (uint8_t)((src & (uint8_t)(1 << i)) >> i);
        tmp = tmp + (dst << (7- i));
        if (printf("%1d", dst) < 0)
        {
            res = TXZ_ERROR;
        }
    } 
    tmp_l = tmp & 0x0F;
    tmp_h = tmp & 0xF0;
    tmp = (tmp_l << 4) + (tmp_h >> 4);
    if (printf(":0x%02X  \n", tmp) < 0)
    {
        res = TXZ_ERROR;
    }

    return (res);
}
/**
 *  @}
 */ /* End of group RMC_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup RMC_Exported_functions RMC Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /*--- variable    ---*/
    variable_initialize();
    /*--- driver      ---*/
    if (driver_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    /*--- application ---*/
    if (application_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(proc != PROC_ERROR)
    {
        switch(proc)
        {
        /*--- Start ---*/
        case PROC_START:
            {
                timer_start(&instance.timer);
                remote_start(instance.p_remote);
                proc = PROC_RUN;
            }
            break;
        /*--- Stop ---*/
        case PROC_STOP:
            {
                remote_stop(instance.p_remote);
                timer_stop(&instance.timer);
                /* error */
                proc = PROC_ERROR;
            }
            break;
        /*--- Receive Data & Output Code. ---*/
        case PROC_RUN:
            {
                remote_data_t data;
                int32_t i =0;
                if(instance.p_remote->init.format == REMOTE_FMT_TYPE_AEHA)
                {
                    for(i=0; i<10; i++)
                    {
                        data.aeha.data[i]  = 0;
                    }
                }
                /* Get the Rx Data. */
                if (remote_get_list_data(instance.p_remote, &data) == TXZ_SUCCESS)
                {
                    if ((instance.p_remote->init.format == REMOTE_FMT_TYPE_NEC) || (instance.p_remote->init.format == REMOTE_FMT_TYPE_TOSHIBA))
                    {
                        /* Output :Customer Code */
                        if (printf("Customer Code \n") < 0)
                        {
                            proc = PROC_STOP;
                        }
                        if (print_code((uint8_t)((uint16_t)(data.sample.customer & 0xFF00) >> 8)) != TXZ_SUCCESS)
                        {
                            proc = PROC_STOP;
                        }
                        if (print_code((uint8_t)(data.sample.customer & 0x00FF)) != TXZ_SUCCESS)
                        {
                            proc = PROC_STOP;
                        }
                        /* Output :Data Code */
                        if (printf("Data Code \n") < 0)
                        {
                            proc = PROC_STOP;
                        }
                        if (print_code(data.sample.data) != TXZ_SUCCESS)
                        {
                            proc = PROC_STOP;
                        }
                        if (print_code(data.sample.rdata) != TXZ_SUCCESS)
                        {
                            proc = PROC_STOP;
                        }
                    }else if(instance.p_remote->init.format == REMOTE_FMT_TYPE_AEHA)
                    {
                        uint32_t    length = data.aeha.bit;
                        if(length > 72)
                        {
                            length = 72;
                        }
                        /* Output :Customer Code */
                        if (printf("Remocon Code \n") < 0)
                        {
                            proc = PROC_STOP;
                        }
                        if (print_aeha_code(data.aeha.customer[0]) != TXZ_SUCCESS)
                        {
                            proc = PROC_STOP;
                        }
                        if(length > 8)
                        {
                            length -= 8;
                            if (print_aeha_code(data.aeha.customer[1]) != TXZ_SUCCESS)
                            {
                                proc = PROC_STOP;
                            }
                        }
                        i =0;
                        while (length > 8)
                        {
                            if (print_aeha_code(data.aeha.data[i]) != TXZ_SUCCESS)
                            {
                                proc = PROC_STOP;
                            }
                            i++;
                            length -= 8;
                        }
                    }
                }
        if(repeat_flg == RMC_REPEAT_FLAG_ON)
              {
                    printf("Repeat Code\n") ;
            repeat_flg = RMC_REPEAT_FLAG_OFF ;
        }
            }
            break;
        /*--- Error ---*/
        case PROC_ERROR:
            /* no processing */
            break;
        /*--- Default ---*/
        default:
            proc = PROC_ERROR;
            break;
        }
    }
    /*>>> System Error. Please Debug. */
    /*----------------------*/
    /* Finalize             */
    /*----------------------*/
    /*--- application ---*/
    if (application_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*--- driver      ---*/
    if (driver_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*----------------------*/
    /* BSP finalize         */
    /*----------------------*/
    bsp_finalize();
#ifdef DEBUG
    assert_failed(MAIN_NULL, 0);
#endif

    return(0);
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer.
  * @param  timer :Timer.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  RMC IRQ.
  * @param  status       :RMC status.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_rmc(uint32_t status)
{
    remote_irq_handler(instance.p_remote, status);
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __nop();
    }
}
#endif

/**
 *  @}
 */ /* End of group RMC_Exported_functions */

/**
 *  @}
 */ /* End of group RMC_Sample_Appli */

/**
 *  @} 
 */ /* End of group Example */


#ifdef __cplusplus
}
#endif /* __cplusplus */
