/** 
 *******************************************************************************
 * @file    main.c
 * @brief   RTC Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include "bsp_com.h"
#include "bsp.h"
#include "app_horol.h"
#include "bsp_uart_io.h"
#include "bsp_rtc.h"
#include "bsp_timer.h"
#include "bsp_led.h"


/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup RTC RTC Sample Appli
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup RTC_Private_macro RTC Private Macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group RTC_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup RTC_Private_define RTC Private Define
 *  @{
 */

/*----------------------*/
/*  Horologe Setting    */
/*----------------------*/
/** 
 *  @defgroup Horologe_InitSetting  Horologe Init Setting
 *  @{
 */
/*---------------------------------*/
/* hour notation :24 Hour notation */
/* century       :21               */
/* year          :16               */
/* month         :January          */
/* date          :01               */
/* day           :Sunday           */
/* am/pm         :am               */
/* hour          :00               */
/* minute        :00               */
/* second        :00               */
/*---------------------------------*/
#define CFG_HOROL_HOUR_NOTATION       (RTC_HOUR_NOTATION_24)      /*!< Hour Notation :@ref RTC_HourNotation. */
#define CFG_HOROL_CENTURY             ((uint8_t)21)               /*!< Century       :(1 <= value)           */
#define CFG_HOROL_YEAR                ((uint8_t)17)               /*!< Year          :(0 <= value <= 99)     */
#define CFG_HOROL_MONTH               (RTC_MONTH_JAN)             /*!< Month         :@ref RTC_Month.        */
#define CFG_HOROL_DATE                ((uint8_t)1)                /*!< Date          :(0 <= value <= 31)     */
#define CFG_HOROL_DAY                 (RTC_DAY_SUNDAY)            /*!< Day           :@ref RTC_Day.          */
#define CFG_HOROL_MERIDIEM            (RTC_MERIDIEM_AM)           /*!< AM/PM         :@ref RTC_Meridiem.     */
#define CFG_HOROL_HOUR                ((uint8_t)21)               /*!< Hour          :(0 <= value <= 23)     */
#define CFG_HOROL_MIN                 ((uint8_t)17)               /*!< Minute        :(0 <= value <= 59)     */
#define CFG_HOROL_SEC                 ((uint8_t)30)               /*!< Second        *(0 <= value <= 59)     */
/** 
 *  @}
 */ /* End of group Horologe_InitSetting */

/** 
 *  @}
 */ /* End of group RTC_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup RTC_Private_define RTC Private Define
 *  @{
 */

#define MAIN_NULL       ((void *)0)     /*!< Null Pointer  */
#define USER_ID_HOROL   ((uint32_t)0)   /*!< User ID :Horologe (Any value) */
#define USER_ID_UART    ((uint32_t)1)   /*!< User ID :USB UART (Any value) */

/** 
 *  @}
 */ /* End of group RTC_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup RTC_Private_define RTC Private Define
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_IDLE,          /*!< Idle.             */
    PROC_START,         /*!< Start.            */
    PROC_STOP,          /*!< Stop.             */
    PROC_RUN,           /*!< Running.          */
    PROC_ERROR          /*!< Error.            */
} Processing;
/** 
 *  @}
 */ /* End of group RTC_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup RTC_Private_typedef RTC Private Typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
    t32a_t      t32a;              /*!< Driver      :T32A.     */
    led_t       led[(uint32_t)BSP_LED_MAX]; /*!< Application :LED                */
    timer_t     timer;             /*!< Application :Timer (1ms timer). */
    horol_t     *p_horol;          /*!< Application :Horologe. */
} instance_t;
/**
 *  @}
 */ /* End of group RTC_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup RTC_Private_variables RTC Private Variables
 *  @{
 */
static instance_t instance; /*!< Instance.                */
static Processing proc;     /*!< Processing mode.         */
static TXZ_FunctionalState update;   /*!< Update flag. TXZ_ENABLE :Update.  */
static uint32_t count = 0 ;
/**
 *  @}
 */ /* End of group RTC_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup RTC_Private_fuctions RTC Private Fuctions
 *  @{
 */
static void variable_initialize(void);
static TXZ_Result driver_initialize(void);
static TXZ_Result driver_finalize(void);
static TXZ_Result application_initialize(void);
static TXZ_Result application_finalize(void);
static void timer_interval_handler(uint32_t id);
static void clock_update(uint32_t id);
static uint8_t convert_month_to_appli(uint8_t month);
/*--------------------------------------------------*/
/** 
  * @brief  Variable Initialize
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void variable_initialize(void)
{
    /* Instance. */
    instance.t32a.p_instance    = MAIN_NULL;
    instance.timer.init.p_t32a  = MAIN_NULL;
    instance.p_horol            = MAIN_NULL;
    /* Processing mode. */
    proc = PROC_START;
    /* Update Flag. */
    update = TXZ_DISABLE;
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (p_t32a->p_instance != MAIN_NULL)
        {
            if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver initialize will process within USB UART. */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Finalize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver finalize has done within USB UART.    */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Finalize ---*/
        /* no processing */
        /*--- Destruct ---*/
        /* Register Release */
        p_t32a->p_instance = MAIN_NULL;
    }
    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver finalize will process within BSP.     */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* LED                  */
    /*----------------------*/
    {
        uint32_t i;
        led_t *p_led = MAIN_NULL;

        for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
        {
            p_led = &instance.led[i];

            p_led->init.id                  = (uint32_t)p_led;
            p_led->init.p_gpio              = bsp_get_gpio_instance_address();
            p_led->init.port.group          = bsp_get_gpio_group_led((BSPLed)(i));
            p_led->init.port.num            = bsp_get_gpio_num_led((BSPLed)(i));
            p_led->init.blink.func          = TXZ_DISABLE;
            p_led->init.blink.interval.on   = 0;
            p_led->init.blink.interval.off  = 0;
            p_led->init.state               = LED_STATE_OFF;
            led_initialize(p_led);
        }
    }        

    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    {
        timer_t *p_timer = &instance.timer;

        p_timer->init.id       = (uint32_t)&instance.timer;
        p_timer->init.p_t32a   = &instance.t32a;
        p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
        p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
        p_timer->init.interval = 1000;
        p_timer->init.handler  = timer_interval_handler;
        timer_initialize(p_timer);
    }
    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = USER_ID_UART;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = bsp_get_usb_uart_tx_baudrate(BSP_USB_UART_0);
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result  = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Horologe             */
    /*----------------------*/
    {
        horol_initial_setting_t param;

        param.id                = USER_ID_HOROL;
        param.setting.century   = CFG_HOROL_CENTURY;
        param.setting.year      = CFG_HOROL_YEAR;
        param.setting.notation  = CFG_HOROL_HOUR_NOTATION;
        param.setting.meridiem  = CFG_HOROL_MERIDIEM;
        param.setting.month     = CFG_HOROL_MONTH;
        param.setting.date      = CFG_HOROL_DATE;
        param.setting.day       = CFG_HOROL_DAY;
        param.setting.hour      = CFG_HOROL_HOUR;
        param.setting.min       = CFG_HOROL_MIN;
        param.setting.sec       = CFG_HOROL_SEC;
        param.clock_update      = clock_update;
        instance.p_horol = horol_initialize(&param);
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Finalize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Horologe             */
    /*----------------------*/
    if (instance.p_horol != MAIN_NULL)
    {
        horol_finalize(instance.p_horol);
    }
    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    {
        uart_io_finalize();
    }
    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    if (instance.timer.init.p_t32a != MAIN_NULL)
    {
        timer_finalize(&instance.timer);
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    led_t *p_led = MAIN_NULL ;
    p_led = &instance.led[0];
  
    if (id == (uint32_t)&instance.timer)
    {
        /* Increment Tick. Tick is used to driver's Timeout. */
        hal_inc_tick();
    }
    
    count ++ ;
    if ( count < 500 ) {
        led_turn_off(p_led);
    } else {
        led_turn_on(p_led);
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Horologe Clock Update Handler.
  * @param  id :User ID.
  * @return -
  * @retval -
  * @note   Called by Horologe.
  */
/*--------------------------------------------------*/
static void clock_update(uint32_t id)
{
    if (id == USER_ID_HOROL)
    {
        /*>>> Critical information. */
        update = TXZ_ENABLE;
        /*<<< Critical information. */
    }
}

/*--------------------------------------------------*/
/** 
  * @brief      Convert month to output format.
  * @param[in]  month: RTC format.
  * @return     Output format.
  * @retval     -
  * @note       -
  */
/*--------------------------------------------------*/
static uint8_t convert_month_to_appli(uint8_t month)
{
    uint8_t res;

    switch(month)
    {
    case RTC_MONTH_JAN:
        res = 1;
        break;
    case RTC_MONTH_FEB:
        res = 2;
        break;
    case RTC_MONTH_MAR:
        res = 3;
        break;
    case RTC_MONTH_APR:
        res = 4;
        break;
    case RTC_MONTH_MAY:
        res = 5;
        break;
    case RTC_MONTH_JUN:
        res = 6;
        break;
    case RTC_MONTH_JUL:
        res = 7;
        break;
    case RTC_MONTH_AUG:
        res = 8;
        break;
    case RTC_MONTH_SEP:
        res = 9;
        break;
    case RTC_MONTH_OCT:
        res = 10;
        break;
    case RTC_MONTH_NOV:
        res = 11;
        break;
    case RTC_MONTH_DEC:
    default:
        res = 12;
        break;
    }

    return (res);
}

/**
 *  @}
 */ /* End of group RTC_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup RTC_Exported_functions RTC Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    led_t *p_led = MAIN_NULL ;
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /*--- variable    ---*/
    variable_initialize();
    /*--- driver      ---*/
    if (driver_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    /*--- application ---*/
    if (application_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(proc != PROC_ERROR)
    {
        switch(proc)
        {
        /*--- Start ---*/
        case PROC_START:
            {
                timer_start(&instance.timer);
                horol_start(instance.p_horol);
                proc = PROC_RUN;
            }
            break;
        /*--- Stop ---*/
        case PROC_STOP:
            {
                horol_stop(instance.p_horol);
                timer_stop(&instance.timer);
                proc = PROC_ERROR;
            }
            break;
        /*--- Receive RTC & Output Code. ---*/
        case PROC_RUN:
            {
                if (update == TXZ_ENABLE)
                {
                    horol_clock_value_t value;

                    /*>>> Critical information. */
                    update = TXZ_DISABLE;
                    /*<<< Critical information. */
                    horol_get_current_clock(instance.p_horol, &value);
                    /* Output information. */
                    if (printf( "%4d-%02d-%02d ", 
                                horol_make_anno_domini(value.century, value.year), 
                                convert_month_to_appli(value.month), 
                                value.date
                                ) < 0 )
                    {
                        proc = PROC_STOP;
                    }
                    if (printf( "%02d:%02d:%02d \n", 
                                value.hour, 
                                value.min, 
                                value.sec
                                ) < 0)
                    {
                        proc = PROC_STOP;
                    }
                    p_led = &instance.led[0];
                    led_turn_off(p_led);
                    count = 0 ;
                }
            }
            break;
        /*--- Error ---*/
        case PROC_ERROR:
            /* no processing */
            break;
        /*--- Default ---*/
        default:
            proc = PROC_ERROR;
            break;
        }
    }
    /*>>> System Error. Please Debug. */
    /*----------------------*/
    /* Finalize             */
    /*----------------------*/
    /*--- application ---*/
    if (application_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*--- driver      ---*/
    if (driver_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*----------------------*/
    /* BSP finalize         */
    /*----------------------*/
    bsp_finalize();
#ifdef DEBUG
    assert_failed(MAIN_NULL, 0);
#endif

    return(0);
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer.
  * @param  timer :Timer.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  RTC IRQ.
  * @param  -
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_rtc(void)
{
    /* Call Horologe IRQ Handler. */
    horol_irq_handler(instance.p_horol);
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __nop();
    }
}
#endif
/**
 *  @}
 */ /* End of group RTC_Exported_functions */

/**
 *  @}
 */ /* End of group RTC */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
