/** 
 *******************************************************************************
 * @file    main.c
 * @brief   SIWDT Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "bsp.h"
#include "bsp_com.h"
#include "siwdt.h"

/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup SIWDT_Sample_Appli SIWDT Sample Appli
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup SIWDT_Private_macro SIWDT Private Macro
 *  @{
 */

#define DEMO_WDT_CLEAR  /* When you comment out this line,  NMI interrupt is occurred by WDT */ 

/** 
 *  @defgroup LEDBlinkConfiguration LED Blink Time Configuration
 *  @brief    LED Blink Interval Time(Interrupt Cycle).
 *  @{
 */
#define CFG_LED_BLINK_TIME             ((uint32_t)(20))
/** 
 *  @}
 */ /* End of group LEDBlinkConfiguration */

/** 
 *  @defgroup ProcModeConfiguration Processing Mode Configuration
 *  @brief    Processing Mode Normal Start.
 *  @{
 */
#define CFG_PROC_DEFAULT               ((uint32_t)(0))
#define CFG_PROC_NORMAL_START          ((uint32_t)(1))
/** 
 *  @}
 */ /* End of group ProcModeConfiguration */

 /** 
 *  @defgroup  IMNFLGNMI IMNFLGNMI Register
 *  @brief     NMI Monitor Register.
 *  @{
 */
#define IMNFLGNMI_INT016FLG             ((uint32_t)0x00010000)    /*!< INT016FLG :[R] :INTWDT0      */ 
#define IBNIC00_INTPFLG                 ((uint32_t)0x00000010)    /*!< INTPFLG   :[R] :INTWDT0      */ 
#define IMNFLGNMI_INT000FLG             ((uint32_t)0x00000001)    /*!< INT000FLG :[R] :INTLVD      */     
/** 
 *  @}
 */ /* End of group IMNFLGNMI */
 /** 
 *  @defgroup  FCSR0  FCSR0 Register
 *  @brief     Flash status Register 0.
 *  @{
 */
/*RDYBSY */
#define FCSR0_RDYBSY_R_BUSY             ((uint32_t)0x00000000)    /*!< WRDYBSY :[R] :Busy      */
#define FCSR0_RDYBSY_R_READY            ((uint32_t)0x00000001)    /*!< WRDYBSY :[R] :Ready     */
/** 
 *  @}
 */ /* End of group FCSR0 */
 /** 
 *  @defgroup  GPIO_EN GPIO_EN
 *  @brief     GPIO Enable/Disable
 *  @{
 */
#define GPIO_DISABLE                    ((uint32_t)0x00000000)    /*!< Disable                 */
#define GPIO_ENABLE                     ((uint32_t)0x00000001)    /*!< Enable                  */
/** 
 *  @}
 */ /* End of group GPIO_EN */
 /** 
 *  @defgroup  FSYS_EN FSYS_EN
 *  @brief     FSYS Enable/Disable
 *  @{
 */
#define FSYS_DISABLE                    ((uint32_t)0x00000000)    /*!< Disable                 */
#define FSYS_ENABLE                     ((uint32_t)0x00000001)    /*!< Enable                  */
/** 
 *  @}
 */ /* End of group FSYS_EN */

/** 
 *  @}
 */ /* End of group SIWDT_Private_macro */

/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup SIWDT_Private_define SIWDT Private Define
 *  @{
 */

/** 
 *  @}
 */ /* End of group SIWDT_Private_define */

/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup SIWDT_Private_typedef SIWDT Private Typedef
 *  @{
 */
 
typedef struct
{
    gpio_t      gpio_obj;            /*!< Application :GPIO. */
} instance_t;

/** 
 *  @}
 */ /* End of group SIWDT_Private_typedef */
/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup SIWDT_Private_variables SIWDT Private Variables
 *  @{
 */
static instance_t instance; /*!< Instance.                    */
static uint32_t counter  = CFG_LED_BLINK_TIME;    /* Blink Time Counter        */
/**
 *  @}
 */ /* End of group SIWDT_Private_variables */

/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup SIWDT_Private_functions SIWDT Private Functions
 *  @{
 */

/**
 *  @}
 */ /* End of group SIWDT_Private_functions */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup SIWDT_Exported_functions SIWDT Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
  uint32_t period;    /* Period for Systick */
  /*----------------------*/
  /* BSP initialize       */
  /*----------------------*/
  bsp_initialize();

  /*----------------------*/
  /* WDT Disable          */
  /*----------------------*/
  wdt_disable(TSB_SIWD0);
  wdt_config(TSB_SIWD0);
  /*------------------------*/
  /* Normal Mode Processing */
  /*------------------------*/
    /* Period for 10msec SYSTICK */
    SystemCoreClockUpdate();
    period = SystemCoreClock / 100;
    if (SysTick_Config(period ))    /* systick interrupt cycle setting */
    {
    }
   /*----------------------*/
  /* WDT Enable          */
  /*----------------------*/   
    wdt_enable(TSB_SIWD0);

    while (1) {
#ifdef DEMO_WDT_CLEAR
        wait(1000U);
        wdt_clear(TSB_SIWD0);
#endif
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  SysTick Handler
  * @param  -
  * @return -
  * @retval -
  */
/*--------------------------------------------------*/
void irq_systick (void)
{
    gpio_pinstate_t val = GPIO_PIN_SET ;
    if (counter-- == 0) 
    {
      /* The end of the counting down period, the data is
         inverted and stored in a register PK4- PK7           */
      if (gpio_read_bit(&instance.gpio_obj,  GPIO_PORT_K, GPIO_PORT_4, GPIO_Mode_DATA, &val) == TXZ_SUCCESS)
      {
          if(val==GPIO_PIN_SET) 
          {
              val = GPIO_PIN_RESET ;
          }
          else
          {
              val = GPIO_PIN_SET ;
          }
          gpio_write_bit(&instance.gpio_obj,  GPIO_PORT_K, GPIO_PORT_4, GPIO_Mode_DATA, val) ;
      }   /* led0 blink for 1sec period */
      if (gpio_read_bit(&instance.gpio_obj,  GPIO_PORT_K, GPIO_PORT_5, GPIO_Mode_DATA, &val) == TXZ_SUCCESS)
      {
          if(val==GPIO_PIN_SET) 
          {
              val = GPIO_PIN_RESET ;
          }
          else
          {
              val = GPIO_PIN_SET ;
          }
          gpio_write_bit(&instance.gpio_obj,  GPIO_PORT_K, GPIO_PORT_5, GPIO_Mode_DATA, val) ;
      }   /* led1 blink for 1sec period */
      if (gpio_read_bit(&instance.gpio_obj,  GPIO_PORT_K, GPIO_PORT_6, GPIO_Mode_DATA, &val) == TXZ_SUCCESS)
      {
          if(val==GPIO_PIN_SET) 
          {
              val = GPIO_PIN_RESET ;
          }
          else
          {
              val = GPIO_PIN_SET ;
          }
          gpio_write_bit(&instance.gpio_obj,  GPIO_PORT_K, GPIO_PORT_6, GPIO_Mode_DATA, val) ;
      }    /* led2 blink for 1sec period */
      if (gpio_read_bit(&instance.gpio_obj,  GPIO_PORT_K, GPIO_PORT_7, GPIO_Mode_DATA, &val) == TXZ_SUCCESS)
      {
          if(val==GPIO_PIN_SET) 
          {
              val = GPIO_PIN_RESET ;
          }
          else
          {
              val = GPIO_PIN_SET ;
          }
          gpio_write_bit(&instance.gpio_obj,  GPIO_PORT_K, GPIO_PORT_7, GPIO_Mode_DATA, val) ;
      }    /* led3 blink for 1sec period */
      counter = CFG_LED_BLINK_TIME;
    }else{
        /* no processing */
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  NMI Handler
  * @param  -
  * @return -
  * @retval -
  */
/*--------------------------------------------------*/
void NMI_Handler(void)
{
  uint32_t flag;
  flag = TSB_IMN->FLGNMI;
  if((flag & IMNFLGNMI_INT016FLG) == IMNFLGNMI_INT016FLG)
  {
    wdt_disable(TSB_SIWD0);
    /*--- IBNIC00: INTPCLR ---*/
    TSB_IB->NIC00 = 0x40;
    /* Dummy read is need. */
    {
        __IO uint8_t nic = TSB_IB->NIC00;
    }
    /* SysTick Disable */
    SysTick->CTRL = 0;
    /* LED turn off */
    gpio_write_bit(&instance.gpio_obj,  GPIO_PORT_K, GPIO_PORT_4, GPIO_Mode_DATA, GPIO_PIN_RESET);   /* led0 turn off */
    gpio_write_bit(&instance.gpio_obj,  GPIO_PORT_K, GPIO_PORT_5, GPIO_Mode_DATA, GPIO_PIN_RESET);   /* led1 turn off */
    gpio_write_bit(&instance.gpio_obj,  GPIO_PORT_K, GPIO_PORT_6, GPIO_Mode_DATA, GPIO_PIN_RESET);   /* led2 turn off */
    gpio_write_bit(&instance.gpio_obj,  GPIO_PORT_K, GPIO_PORT_7, GPIO_Mode_DATA, GPIO_PIN_RESET);   /* led3 turn off */
  }
}
/**
 *  @}
 */ /* End of group SIWDT_Exported_functions */

/**
 *  @}
 */ /* End of group SIWDT_Sample_Appli */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
