/** 
 *******************************************************************************
 * @file    main.c
 * @brief   T32A_IntervalTimer Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif
/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup T32A_IntervalTimer T32A_IntervalTimer Sample Appli
 *  @{
 */

/** 
 *  @defgroup MAIN_PRIVATE  MAIN Private
 *  @brief    MAIN Private Group.
 *  @{
 */

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdio.h>
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_sw.h"
#include "bsp_led.h"
#include "bsp_timer.h"
#include "bsp_uart_io.h"
/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/*----------------------------------*/
/** 
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
      PROC_NORMAL,            /*!< Normal Mode    :RUNNING.   */
      PROC_NORMAL_END,        /*!< Normal Mode    :END.       */
      PROC_NORMAL_START,      /*!< Normal Mode    :RESTART.     */
} Processing;

/*----------------------------------*/
/** 
 * @brief  LED mode.
*/
/*----------------------------------*/
typedef enum
{
      LED_RUN,            /*!< Normal Mode    :RUNNING.   */
      LED_STOP,        /*!< Normal Mode    :END.       */
} LED_STATUS;
/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
    t32a_t      t32a;                /*!< Application      :T32A.     */
    sw_t        sw[(uint32_t)BSP_PSW_MAX];  /*!< Application :Push SW            */
    led_t       led[(uint32_t)BSP_LED_MAX]; /*!< Application :LED                */
    timer_t     timer;             /*!< Application :Timer (1sec timer). */
    gpio_t      gpio_obj;            /*!< Application :GPIO. */
} instance_t;

/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/*----------------------*/
/*  UART Setting        */
/*----------------------*/
/** 
 *  @defgroup UART_Setting UART Setting
 *  @{
 */
#define CFG_UART_BAUDRATE               ((uint32_t)115200)  /*!< Boudrate(bps) */
/** 
 *  @}
 */ /* End of group UART_Setting */
/** 
 *  @defgroup LEDBlinkConfiguration LED Blink Time Configuration
 *  @brief    LED Blink Interval Time(ms).
 *  @{
 */
#define CFG_LED_BLINK_TIME      ((uint32_t)(1000))
/** 
 *  @}
 */ /* End of group LEDBlinkConfiguration */

/** 
 *  @defgroup SWPollingConfiguration SW Polling Time Configuration
 *  @brief    SW Polling Time(ms).
 *  @{
 */
#define CFG_SW_POLLING_TIME     ((uint32_t)(100))
/** 
 *  @}
 */ /* End of group SWPollingConfiguration */

#define MAIN_NULL       ((void *)0)     /*!< NULL. */
#define USER_ID_REMOTE  ((uint32_t)0)   /*!< User ID :Remote Control (Any value) */
#define USER_ID_UART    ((uint32_t)1)   /*!< User ID :USB UART (Any value) */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
static instance_t instance; /*!< Instance.                    */
static uint32_t count = 0U;
static uint32_t on = 0U;
static Processing   proc;      /*!< Processing mode.             */
static LED_STATUS led_st = LED_RUN;
/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
static void timer_interval_handler(uint32_t id);
static void variable_initialize(void);
static void sw_state_change_handler(uint32_t id, gpio_pinstate_t state);
static TXZ_Result application_initialize(void);
static TXZ_Result driver_initialize(void);
/*--------------------------------------------------*/
/** 
  * @brief  Variable Initialize
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void variable_initialize(void)
{
    /* Instance. */
    instance.t32a.p_instance    = MAIN_NULL;

}
/*--------------------------------------------------*/
/** 
  * @brief  SW State Change Handler.
  * @param  id    :User ID
  * @param  state :Push SW Port State
  * @return -
  * @retval -
  * @note   Notify that, "State is changed." \n
  *         Call by SW Task \n
  *         If SW is pushed, change the processing mode(@ref Processing).
  */
/*--------------------------------------------------*/
static void sw_state_change_handler(uint32_t id, gpio_pinstate_t state)
{
    /* Check SW Push(= GPIO_PIN_RESET) */
    if (state == GPIO_PIN_RESET)
    {
        uint32_t i;

        for (i=0; i < (uint32_t)BSP_PSW_MAX; i++)
        {
            if (id == (uint32_t)&instance.sw[i])
            {
                /* Select Low Power Mode */
                switch ((BSPPsw)i)
                {
                case BSP_PSW_1:
                    proc = PROC_NORMAL_END;
                    break;
                case BSP_PSW_2:
                    proc = PROC_NORMAL_START;
                    break;
                default:
                    /* no processing */
                    break;
                }
            }
        }
    }
}
/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
   TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = USER_ID_UART;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = (uint32_t)CFG_UART_BAUDRATE;
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result  = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* LED                  */
    /*----------------------*/
    {
        uint32_t i;
        led_t *p_led = MAIN_NULL;
        for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
        {
            p_led = &instance.led[i];

            p_led->init.id                  = (uint32_t)p_led;
            p_led->init.p_gpio              = bsp_get_gpio_instance_address();
            p_led->init.port.group          = bsp_get_gpio_group_led((BSPLed)(i));
            p_led->init.port.num            = bsp_get_gpio_num_led((BSPLed)(i));
            p_led->init.blink.func          = TXZ_ENABLE;
            p_led->init.blink.interval.on   = CFG_LED_BLINK_TIME;
            p_led->init.blink.interval.off  = CFG_LED_BLINK_TIME;
            p_led->init.state           = LED_STATE_OFF;
            led_initialize(p_led);
        }
    }
    /*----------------------*/
    /* PSW                  */
    /*----------------------*/
    {
        uint32_t i;
        sw_t *p_sw = MAIN_NULL;

        for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
        {
            p_sw = &instance.sw[i];

            p_sw->init.id                   = (uint32_t)p_sw;
            p_sw->init.p_gpio               = bsp_get_gpio_instance_address();
            p_sw->init.port.group           = bsp_get_gpio_group_psw((BSPPsw)(i));
            p_sw->init.port.num             = bsp_get_gpio_num_psw((BSPPsw)(i));
            p_sw->init.polling.func         = TXZ_ENABLE;
            p_sw->init.polling.interval     = CFG_SW_POLLING_TIME;
            p_sw->init.chattering.func      = TXZ_DISABLE;
            p_sw->init.chattering.count     = 0;
            p_sw->init.handler              = sw_state_change_handler;
            sw_initialize(p_sw);
        }
    }
    return (result);
}
/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    int32_t i;
    led_t *p_led = MAIN_NULL;
    /*----------------------*/
    /* SW                   */
    /*----------------------*/
    for (i=0; i<(int32_t)BSP_PSW_MAX; i++)
    {
        sw_1ms_timer_handler(&instance.sw[i]);
    }
     
    if(led_st != LED_STOP){
        if(count < 500) {   /* 1ms * 500 = 500msec LED on */
            count++;
            on = 1;
        }else if((count >= 500) && (count < 1000)){  /* 1ms * 500 = 500msec LED off */
            count++;
            on = 0;
        }else{
                count = 0;
        }
        if(on == 1){
           for (i=0; i<(int32_t)BSP_LED_MAX; i++)
           {
                p_led = &instance.led[i];
                led_turn_on(p_led);
           }
        }else{
            for (i=0; i<(int32_t)BSP_LED_MAX; i++)
            {
                p_led = &instance.led[i];
                led_turn_off(p_led);
            }
        }
    } else {                /* If LED_STOP then LED=OFF */
        count = 0 ;
        on = 0 ;
        for (i=0; i<(int32_t)BSP_LED_MAX; i++)
        {
            p_led = &instance.led[i];
            led_turn_off(p_led);
        }
    }
}
/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (p_t32a->p_instance != MAIN_NULL)
        {
            if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
        /*----------------------*/
        /* Timer                */
        /*----------------------*/
        /* 1ms timer */
        {
            timer_t *p_timer = &instance.timer;

            p_timer->init.id       = (uint32_t)&instance.timer;
            p_timer->init.p_t32a   = &instance.t32a;
            p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
            p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
            p_timer->init.interval = 1000;
            p_timer->init.handler  = timer_interval_handler;
            timer_initialize(p_timer);
        }
    }
    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver initialize will process within USB UART. */

    return (result);
}
/**
 *  @}
 */ /* End of group MAIN_PRIVATE */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    uint32_t i;
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /* Valiable initialization */
    variable_initialize();
    /* Timer driver initialization */
    if (driver_initialize() == TXZ_ERROR){
        
    }
    /* Application initialization */
    if (application_initialize() == TXZ_ERROR){
        
    }
    
    /* Timer Start*/

    proc = PROC_NORMAL_END;
        
    timer_start(&instance.timer);

    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(1){
        switch(proc)
        {
            case PROC_NORMAL:
            {
                /*----------------------*/
                /* SW                   */
                /*----------------------*/
                for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
                {
                    sw_task(&instance.sw[i]);
                }
                break;
            }
            case PROC_NORMAL_END:
            {
                led_st = LED_STOP;
                proc = PROC_NORMAL;
                break;
            }
            case PROC_NORMAL_START:
            {
                led_st = LED_RUN;
                proc = PROC_NORMAL;
                break;
            }
            default:
            {
                proc = PROC_NORMAL;
                break;
            }
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  1ms Timer.
  * @param  timer :1ms Timer.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    default:
        /* no processing */
        break;
    }
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __NOP();
    }
}
#endif
/**
 *  @} 
 */ /* End of group T32A_IntervalTimer */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
