/** 
 *******************************************************************************
 * @file    main.c
 * @brief   T32A_PPG Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif
/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup T32A_PPG T32A_PPG Sample Appli
 *  @{
 */

/** 
 *  @defgroup MAIN_PRIVATE  MAIN Private
 *  @brief    MAIN Private Group.
 *  @{
 */

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_sw.h"
#include "bsp_timer.h"
#include "bsp_uart_io.h"
#include "bsp_pulse.h"
/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/*----------------------------------*/
/** 
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_RUN,            /*!< Timer Start    :RUNNING.   */
    PROC_START,            /*!< Timer Start    :RUNNING.   */
    PROC_DUTY_CHANGE,      /*!< Duty Change    :CHANGE.       */
} Processing;
/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
    t32a_t      t32a_timer;                /*!< Application      :T32A.     */
    t32a_t      t32a_pulse;                /*!< Application      :T32A.     */
    sw_t        sw[(uint32_t)BSP_PSW_MAX];  /*!< Application :Push SW            */
    timer_t     timer;             /*!< Application :Timer (ims timer). */
    pulse_t     pulse;             /*!< Application :Timer (Output Pulse timer). */
    gpio_t      gpio_obj;            /*!< Application :GPIO. */
} instance_t;


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup Timer_EVENT_COUNT_Private_variables Timer_EVENT_COUNT Private Variables
 *  @{
 */
static instance_t instance; /*!< Instance.                    */
/**
 *  @}
 */ /* End of group Timer_EVENT_COUNT_Private_variables */
static uint8_t Rate = 0U;
static uint32_t tgtRisingTiming[5U] = { 10U, 25U, 50U, 75U, 90U }; /* rigingtiming: 10%, 25%, 50%, 75%, 90% */ 
static uint32_t RisingTimingus[5U] = {0U}; 
static Processing   proc;      /*!< Processing mode.             */
/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/*----------------------*/
/*  UART Setting        */
/*----------------------*/
/** 
 *  @defgroup UART_Setting UART Setting
 *  @{
 */
#define CFG_UART_BAUDRATE               ((uint32_t)115200)  /*!< Boudrate(bps) */
/** 
 *  @}
 */ /* End of group UART_Setting */

/** 
 *  @}
 */ /* End of group Timer_EVENT_COUNT_Private_define */
#define MAIN_NULL       ((void *)0)     /*!< NULL. */
#define USER_ID_REMOTE  ((uint32_t)0)   /*!< User ID :Remote Control (Any value) */
#define USER_ID_UART    ((uint32_t)1)   /*!< User ID :USB UART (Any value) */

#define LEADINGMAX         0x05U   /* LEADING RATE COUNTER MAX */
#define LEADINGINIT        0U      /* LEADING RATE COUNTER Initialize */
/** 
 *  @defgroup SWPollingConfiguration SW Polling Time Configuration
 *  @brief    SW Polling Time(ms).
 *  @{
 */
#define CFG_SW_POLLING_TIME     ((uint32_t)(100))
/** 
 *  @}
 */ /* End of group SWPollingConfiguration */
/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
static void risingtiming_display(void);
static void risingtiming_change_display(void);
static char *RisingTimingRateToString(uint8_t num);
static void variable_initialize(void);
static void sw_state_change_handler(uint32_t id, gpio_pinstate_t state);
static void timer_interval_handler(uint32_t id);
static TXZ_Result application_initialize(void);
static TXZ_Result pulse_driver_initialize(void);
static TXZ_Result timer_driver_initialize(void);
/*--------------------------------------------------*/
/** 
  * @brief  Variable Initialize
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void variable_initialize(void)
{
    /* Instance. */
    instance.t32a_timer.p_instance    = MAIN_NULL;
    instance.t32a_pulse.p_instance    = MAIN_NULL;
}
/*--------------------------------------------------*/
/** 
  * @brief  SW State Change Handler.
  * @param  id    :User ID
  * @param  state :Push SW Port State
  * @return -
  * @retval -
  * @note   Notify that, "State is changed." \n
  *         Call by SW Task \n
  *         If SW is pushed, change the processing mode(@ref Processing).
  */
/*--------------------------------------------------*/
static void sw_state_change_handler(uint32_t id, gpio_pinstate_t state)
{
    /* Check SW Push(= GPIO_PIN_RESET) */
    if (state == GPIO_PIN_RESET)
    {
        uint32_t i;

        for (i=0; i < (uint32_t)BSP_PSW_MAX; i++)
        {
            if (id == (uint32_t)&instance.sw[i])
            {
                /* Select Low Power Mode */
                switch ((BSPPsw)i)
                {
                case BSP_PSW_1:
                    proc = PROC_START;
                    break;
                case BSP_PSW_2:
                    proc = PROC_DUTY_CHANGE;
                    break;
                default:
                    /* no processing */
                    break;
                }
            }
        }
    }
}
/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    int32_t i;
   /*----------------------*/
   /* SW                   */
   /*----------------------*/
   for (i=0; i<(int32_t)BSP_PSW_MAX; i++)
   {
       sw_1ms_timer_handler(&instance.sw[i]);
   }
}
/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
   TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = USER_ID_UART;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = (uint32_t)CFG_UART_BAUDRATE;
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result  = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* PSW                  */
    /*----------------------*/
    {
        uint32_t i;
        sw_t *p_sw = MAIN_NULL;

        for (i=0; i<(int32_t)BSP_PSW_MAX; i++)
        {
            p_sw = &instance.sw[i];

            p_sw->init.id                   = (uint32_t)p_sw;
            p_sw->init.p_gpio               = bsp_get_gpio_instance_address();
            p_sw->init.port.group           = bsp_get_gpio_group_psw((BSPPsw)(i));
            p_sw->init.port.num             = bsp_get_gpio_num_psw((BSPPsw)(i));
            p_sw->init.polling.func         = TXZ_ENABLE;
            p_sw->init.polling.interval     = CFG_SW_POLLING_TIME;
            p_sw->init.chattering.func      = TXZ_DISABLE;
            p_sw->init.chattering.count     = 0;
            p_sw->init.handler              = sw_state_change_handler;
            sw_initialize(p_sw);
        }
    }
    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result pulse_driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
{
        t32a_t *p_t32a = &instance.t32a_pulse;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_output_pulse_ch(BSP_OUT_PULSE_SAMPLE_A))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (p_t32a->p_instance != MAIN_NULL)
        {
            if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* Output Pulse timer */
    {
        pulse_t *p_pulse = &instance.pulse;

        p_pulse->init.id       = (uint32_t)&instance.pulse;
        p_pulse->init.pulse_t32a   = &instance.t32a_pulse;
        p_pulse->init.type     = bsp_get_output_pulse_type(BSP_OUT_PULSE_SAMPLE_A);
        /* RisingTiming Initial Value 10%(50us) SET */
        p_pulse->init.pulse_rising = RisingTimingus[Rate];
        /* TrailingTiming 500us SET */
        p_pulse->init.pulse_trailing = 500U;
        p_pulse->init.irq = bsp_get_output_pulse_nvic(BSP_OUT_PULSE_SAMPLE_A);
        pulse_initialize(p_pulse);
    }
    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver initialize will process within USB UART. */

    return (result);
}
/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result timer_driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a_timer;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (p_t32a->p_instance != MAIN_NULL)
        {
            if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    {
        timer_t *p_timer = &instance.timer;

        p_timer->init.id       = (uint32_t)&instance.timer;
        p_timer->init.p_t32a   = &instance.t32a_timer;
        p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
        p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
        p_timer->init.interval = 1000;
        p_timer->init.handler  = timer_interval_handler;
        timer_initialize(p_timer);
    }
    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver initialize will process within USB UART. */

    return (result);
}

/**
  * @brief  Display the string of T32A_PPG demo
  * @param  -
  * @retval -
  */
static void risingtiming_display(void)
{
    if (printf(RisingTimingRateToString(Rate)) < 0) {
        /* no processing */
    }
    if (printf("%%\n") < 0) {
        /* no processing */
    }
}
static void risingtiming_change_display(void)
{
    if (printf("Change to ") < 0) {
        /* no processing */
    }
    if (printf(RisingTimingRateToString(Rate)) < 0) {
        /* no processing */
    }
    if (printf("%%\n") < 0) {
        /* no processing */
    }
}

/**
  * @brief  RisingTiming Rate to String
  * @param  num    :leadingtiming number
  * @retval RisingTiming rate string
  */
static char *RisingTimingRateToString(uint8_t num)
{
    static char leadingtiming[9] = "Duty:   ";

    switch (num) {
    case 0:
        leadingtiming[6] = '1';
        leadingtiming[7] = '0';
        break;                  /* RisingTiming: 10% */
    case 1:
        leadingtiming[6] = '2';
        leadingtiming[7] = '5';
        break;                  /* RisingTiming: 25% */
    case 2:
        leadingtiming[6] = '5';
        leadingtiming[7] = '0';
        break;                  /* RisingTiming: 50% */
    case 3:
        leadingtiming[6] = '7';
        leadingtiming[7] = '5';
        break;                  /* RisingTiming: 75% */
    case 4:
        leadingtiming[6] = '9';
        leadingtiming[7] = '0';
        break;                  /* RisingTiming: 90% */
    default:
        /* Do nothing */
        break;                  /* no other cases */
    }

    return leadingtiming;
}

/**
 *  @}
 */ /* End of group MAIN_PRIVATE */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    uint32_t i = 0U;  
    uint32_t out_toggle = 0U;
    /* RisingTimingus: 50us, 125us, 250us, 375us, 450us */
    for (i=0U;i<=4U;i++) {
        RisingTimingus[i] = tgtRisingTiming[i] * 5U;
    }
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /* Valiable initialization */
    variable_initialize();  
    /* Timer driver initialization */
    if(timer_driver_initialize() == TXZ_ERROR) {
        /* no processing */
    }
    if(pulse_driver_initialize() == TXZ_ERROR) {
        /* no processing */
    }
    /* Application initialization */
    if(application_initialize() == TXZ_ERROR) {
        /* no processing */
    }
    timer_start(&instance.timer);
    proc = PROC_RUN;
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(1){
        switch(proc)
        {
            case PROC_RUN:
            {
                /*----------------------*/
                /* SW                   */
                /*----------------------*/
                for (i=0; i<(uint32_t)BSP_PSW_MAX; i++)
                {
                    sw_task(&instance.sw[i]);
                }
                break;
            }
            case PROC_DUTY_CHANGE:
            {
                /* Pulse Timer Stop */
                pulse_timer_stop(&instance.pulse);
                /* Pulse Timer Finalize */
                pulse_timer_finalize(&instance.pulse);
                wait(0xFFFFU);         /* noise cancel */
                Rate++;
                if (Rate >= LEADINGMAX) {  /* change leadingtiming rate */
                    Rate = LEADINGINIT;
                } else {
                    /* Do nothing */
                }
                /* RisingTiming Value SET */
                if(pulse_driver_initialize() == TXZ_ERROR) {
                    /* no processing */
                }
                /* Risingtiming rate Display */
                risingtiming_change_display();
                /* Timer Start*/
                pulse_timer_start(&instance.pulse);
                proc = PROC_RUN;
                break;
            }
            case PROC_START:
            {
                out_toggle = 1 - out_toggle ;   /* toggle 0 <-> 1 */
                if(out_toggle == 1) {
                    if(printf("PPG Output\n") < 0) {
                        /* Do nothing */
                    }
                    risingtiming_display();
                    /* Timer Start*/
                    pulse_timer_start(&instance.pulse);
                    wait(0xFFFFU);             /* noise cancel */
                } else {
                    if(printf("PPG Stop\n") < 0) {
                        /* Do nothing */
                    }
                    /* Pulse Timer Stop */
                    pulse_timer_stop(&instance.pulse);
                    /* Pulse Timer Finalize */
                    pulse_timer_finalize(&instance.pulse);
                    wait(0xFFFFU);         /* noise cancel */
                }
                proc = PROC_RUN;
                break;
            }
            /*--- Default ---*/
            default:
            {
                proc = PROC_RUN;
                break;
            }
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer.
  * @param  timer :Timer.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    /*--- 1ms timer 0 ---*/
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __NOP();
    }
}
#endif

/**
 *  @} 
 */ /* End of group T32A_PPG */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
