/** 
 *******************************************************************************
 * @file    main.c
 * @brief   TRM Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */

#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include "bsp_com.h"
#include "bsp_uart_io.h"
#include "bsp.h"
#include "trm.h"
#include "uart_dma.h"
     
/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup TRMOSC TRMOSC Sample Appli
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup TRMOSC_Private_macro TRMOSC Private Macro
 *  @{
 */
#define TRMOSC_SAMPLE
/* #define TRMOSC_LOSC */

/** 
 *  @}
 */ /* End of group TRMOSC_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup TRMOSC_Private_define TRMOSC Private Define
 *  @{
 */
#ifdef TRMOSC_LOSC
#define CAPTUREVAL10MHZ        (9765)          /*!< Caputer value @10MHz */
#else
#define CAPTUREVAL10MHZ        (10416)         /*!< Caputer value @10MHz */
#endif
#define FIXEDPOINT             (100)           /*!< fixed point 2 */
#define TRIMSTEP               (7)             /*!< trimming step 0.07% * 100 * FIXEDPOINT */
#define MARGINADJUST           (20)            /*!< Adjustment margin 0.2% * 100 * FIEXDPOINT */
#define TRIMLIMIT              (0x0000007FUL)  /*!< Setting limit */
#define TRIM_MAX               (0x00000100UL)  /*!< max value */
/** 
 *  @defgroup ELOSCConfiguration ELOSC Configuration
 *  @brief    ELOSC Configuration.
 *  @{
 */
#define ELOSC_CFG_WARM_UP_TIME  ((uint32_t)(5000))      /*!< Warm up time(us). */
#define ELOSC_CFG_CLOCK         ((uint32_t)(32768))     /*!< Clock(hz).        */
/** 
 *  @}
 */ /* End of group ELOSCConfiguration */

/** 
 *  @defgroup  CGWUPLCR  CGWUPLCR Register
 *  @brief     LOSC warming up Register.
 *  @details   Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 31:27 | -          |
 *  | 26:12 | WUPTL      |
 *  | 11:8  | WUPTL      |
 *  | 7:2   | -          |
 *  | 1     | WULEF      |
 *  | 0     | WULON      |
 *  @{
 */
/* WUPTL  */
#define CGWUPLCR_WUPTL_HIGH_MASK        ((uint32_t)0x07FFF000)    /*!< WUPTL   :High Bit Mask.  */
/* WULEF  */
#define CGWUPLCR_WULEF_MASK             ((uint32_t)0x00000002)    /*!< WULEF   :Mask.           */
#define CGWUPLCR_WULEF_R_DONE           ((uint32_t)0x00000000)    /*!< WULEF   :[R]  :Done      */
#define CGWUPLCR_WULEF_R_RUNNING        ((uint32_t)0x00000002)    /*!< WULEF   :[R]  :Running   */
/* WULON  */
#define CGWUPLCR_WULON_MASK             ((uint32_t)0x00000001)    /*!< WULON   :Mask.           */
#define CGWUPLCR_WULON_W_ENABLE         ((uint32_t)0x00000001)    /*!< WULON   :[W]  :Enable     */
/** 
 *  @}
 */ /* End of group CGWUPLCR */

/** 
 *  @defgroup  RLMLOSCCR  RLMLOSCCR Register
 *  @brief     LOSC Control Register.
 *  @details   Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 31:1  | -          |
 *  | 0     | XTEN       |
 *  @{
 */
/* XTEN */
#define RLMLOSCCR_XTEN_RW_DISABLE       ((uint8_t)0x00)    /*!< XTEN :[R/W]  :Disable  */
#define RLMLOSCCR_XTEN_RW_ENABLE        ((uint8_t)0x01)    /*!< XTEN :[R/W]  :Enable   */
/** 
 *  @}
 */ /* End of group RLMLOSCCR */

/** 
 *  @}
 */ /* End of group TRMOSC_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup TRMOSC_Private_define TRMOSC Private Define
 *  @{
 */
#define COUNTMAX               (10)            /*!< retry counter max value1 */
/*----------------------*/
/*  UART Setting        */
/*----------------------*/
/** 
 *  @defgroup UART_Setting UART Setting
 *  @{
 */
#define CFG_UART_BAUDRATE               ((uint32_t)115200)  /*!< Boudrate(bps) */
/** 
 *  @}
 */ /* End of group UART_Setting */

#define MAIN_NULL       ((void *)0)     /*!< NULL. */
#define USER_ID_REMOTE  ((uint32_t)0)   /*!< User ID :Remote Control (Any value) */
#define USER_ID_UART    ((uint32_t)1)   /*!< User ID :USB UART (Any value) */


#define TRMOSC_RUN             (1)             /*!< run   */
#define TRMOSC_DONE            (0)             /*!< done  */
#define TRMOSC_ERROR           (-1)            /*!< error */

#ifdef TRMOSC_LOSC
    #define BSP_T32A_REFIN                 TSB_T32A6
    #define BSP_INTT32A_REFIN_BCAP1_IRQn   INTT32A06BCAP1_IRQn
    #define BSP_T32A_REFIN_RUNA_SFTSTAA    TSB_T32A6_RUNA_SFTSTAA
    #define BSP_T32A_REFIN_RUNB_SFTSTPB    TSB_T32A6_RUNB_SFTSTPB
#else
    #define BSP_T32A_REFIN                 TSB_T32A2
    #define BSP_INTT32A_REFIN_ACCAP0_IRQn  INTT32A02ACAP0_IRQn
    #define BSP_INTT32A_REFIN_ACCAP1_IRQn  INTT32A02ACAP1_IRQn
    #define BSP_T32A_REFIN_RUNA_SFTSTPA    TSB_T32A2_RUNA_SFTSTPA
#endif
    #define BSP_T32A_SOUT                  TSB_T32A0
    #define BSP_T32A_SOUT_RUNA_SFTSTAA     TSB_T32A0_RUNA_SFTSTAA

/** 
 *  @}
 */ /* End of group TRMOSC_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup TRMOSC_Private_define TRMOSC Private Define
 *  @{
 */
/* no define */

/** 
 *  @}
 */ /* End of group TRMOSC_Private_define */



/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup TRMOSC_Private_typedef TRMOSC Private Typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief Adjustment Information.
*/
/*----------------------------------*/
typedef struct
{
    uint32_t MeasureVal;
    int32_t  MeasureDiff;
    uint32_t Trim;
} IOSCADJ_TypeDef;
/**
 *  @}
 */ /* End of group TRMOSC_Private_typedef */

/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup TRMOSC_Private_variables TRMOSC Private Variables
 *  @{
 */
static uint32_t RetryCount=0;
static int32_t  AdjustResult;
static t32a_t   timer_sout;                /*!< Instance       :Timer Driver  */
static t32a_t   timer_refin;               /*!< Instance       :Timer Driver  */
static cg_t     cg;                        /*!< CG                            */
static IOSCADJ_TypeDef IoscAdjust;
static TRM_ControlTypeDef AdjustData;
static int32_t  difference;

/**
 *  @}
 */ /* End of group TRMOSC_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
static TXZ_Result application_initialize(void);
#ifdef TRMOSC_LOSC
static void external_losc_enable(void);
#endif
void InitT32A_Sout(void);
void InitT32A_RefIn(void);
void StartCapture(void);
uint32_t GetCaptureData(void);
void StopCapture(void);
int32_t IOSCAdjustment(void);
void InitTRM(void);
static void MeasureWave(IOSCADJ_TypeDef *Measure);
static int32_t CalcDiff(IOSCADJ_TypeDef *Diff);
static int32_t CalcAdjVal(IOSCADJ_TypeDef *Adjust);
static int32_t GetAdjustmentValue(void);
/*--------------------------------------------------*/
/** 
  * @brief  Measurement Waveform
  * @param  Measure  :TRMOSC adjustment Structure
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void MeasureWave(IOSCADJ_TypeDef *Measure)
{
    StartCapture();
    Measure->MeasureVal = GetCaptureData();
    StopCapture();
}

/*--------------------------------------------------*/
/** 
  * @brief  Calculation of difference
  * @param  Diff  :TRMOSC adjustment Structure
  * @retval TRMOSC_DONE, TRMOSC_ERROR
  * @note   -
  */
/*--------------------------------------------------*/
static int32_t CalcDiff(IOSCADJ_TypeDef *Diff)
{
    uint32_t absdiff;
    int32_t retval;

    Diff->MeasureDiff = (int32_t)((100*FIXEDPOINT)-((Diff->MeasureVal * (100*FIXEDPOINT)) / CAPTUREVAL10MHZ));
/**/
    absdiff = (uint32_t)abs(Diff->MeasureDiff);
    if (absdiff <= MARGINADJUST) 
    {
        retval = TRMOSC_DONE;
    }
    else 
    {
        retval = TRMOSC_ERROR;
    }

    return (retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Calculation of adjustment value
  * @param  Adjust  :TRMOSC adjustment Structure
  * @retval TRMOSC_RUN, TRMOSC_ERROR
  * @note   -
  */
/*--------------------------------------------------*/
static int32_t CalcAdjVal(IOSCADJ_TypeDef *Adjust)
{
    int32_t retval;
    uint32_t trim;
    uint32_t adjust_step;

    adjust_step = (uint32_t)(abs(Adjust->MeasureDiff)/TRIMSTEP);
    /* adjust step (at least 1)check */
    if(adjust_step == 0)
    {
       adjust_step += 1; 
    }

    /* plus diretion */
    if(Adjust->MeasureDiff > 0)
    {
        difference = (adjust_step*TRIMSTEP)+GetAdjustmentValue();
    }
    /* minus direction */
    else
    {
        difference = -(adjust_step*TRIMSTEP)+GetAdjustmentValue();
    }
     
    /* adujust value */    
    trim = (uint32_t)(abs(difference)/TRIMSTEP);
    
    if (difference < 0)
    {
        trim = TRIM_MAX-trim;
        if(trim > TRIMLIMIT)
        {
            Adjust->Trim = trim;
            retval = TRMOSC_RUN;
        }
        else 
        {
            retval = TRMOSC_ERROR;
        }
    }
    else 
    {
        if(trim <= TRIMLIMIT)
        {
            Adjust->Trim = trim;
            retval = TRMOSC_RUN;
        }
        else 
        {
            retval = TRMOSC_ERROR;
        }
    }

    return (retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Get Adjustment value
  * @param  -
  * @retval Adjustment value
  * @note   -
  */
/*--------------------------------------------------*/
static int32_t GetAdjustmentValue(void)
{
    uint32_t trimvalue;
    int32_t adjvalue;

    if (GetTrimmingControlValue() == TRMOSC_CONTROL_DISABLE) 
    {
        trimvalue = GetInitTrimValue();
    }
    else 
    {
        trimvalue = GetSetTrimValue();
    }
/**/
    if (trimvalue > TRIMLIMIT) 
    {
        trimvalue = TRIM_MAX - trimvalue;
        adjvalue = -(TRIMSTEP*trimvalue);
    }
    else 
    {
        adjvalue = TRIMSTEP*trimvalue;
    }

    return (adjvalue);
}

/*--------------------------------------------------*/
/** 
  * @brief  Adjustment of internal OSC
  * @param  -
  * @retval TRMOSC_DONE, TRMOSC_ERROR, TRMOSC_RUN
  * @note   -
  */
/*--------------------------------------------------*/
int32_t IOSCAdjustment(void)
{
    int32_t retval;

    cg.p_instance = TSB_CG;

    MeasureWave(&IoscAdjust);                                           /* Measurement Waveform */
    retval = CalcDiff(&IoscAdjust);                                     /* Calculation of difference */
    if (retval == TRMOSC_ERROR) 
    {
        retval = CalcAdjVal(&IoscAdjust);                               /* Calculation of adjustment value */
        if(retval == TRMOSC_RUN) 
        {
            AdjustData.TrimmingControl     = TRMOSC_CONTROL_ENABLE;
            AdjustData.TrimmingValue = IoscAdjust.Trim;
            SetTrmOsc(&AdjustData);                                     /* Set TRMOSC register */

            /*--------------------------------*/
            /* Internal HOSC Warm-up Time Set */
            /*--------------------------------*/
            cg_set_warm_up_time(&cg, CG_WARM_UP_SRC_IHOSC1, WU_TIME_INT);
            /*----------------------*/
            /* Warm-up Timer Start  */
            /*----------------------*/
            cg_start_warmup();
            /*----------------------*/
            /* Check Warm-up Timer status */
            /*----------------------*/
            cg_wait_warmup();
        }
        else 
        {                                                           /* TRMOSC_ERROR */
            /* do nothing */
        }
    }
    else 
    {                                                               /* TRMOSC_DONE  */
        /* do nothing */
    }
    return (retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Initialize TRM Function
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void InitTRM(void)
{
    TRM_ControlTypeDef InitData;

    cg.p_instance = TSB_CG;

    InitData.TrimmingControl     = TRMOSC_CONTROL_ENABLE;
    InitData.TrimmingValue = 0x00000080 ;
    SetTrmOsc(&InitData);
    /*--------------------------------*/
    /* Internal HOSC Warm-up Time Set */
    /*--------------------------------*/
    cg_set_warm_up_time(&cg, CG_WARM_UP_SRC_IHOSC1, WU_TIME_INT);
    /*----------------------*/
    /* Warm-up Timer Start  */
    /*----------------------*/
    cg_start_warmup();
    /*----------------------*/
    /* Check Warm-up Timer status */
    /*----------------------*/
    cg_wait_warmup();
}
/*--------------------------------------------------*/
/** 
  * @brief  Initialize T32A Signal Output Function
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void InitT32A_Sout(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Signal Output Timer         */
    /*------------------------------*/
    /*--- Construct (Register Allocation) ---*/
    /* Register Allocation */
    timer_sout.p_instance = BSP_T32A_SOUT;
    /*--- Initialize: Timer   ---*/
    /* Mode             */
    timer_sout.init_mode.mode.halt = T32A_DBG_HALT_RUN;
    timer_sout.init_mode.mode.mode = T32A_MODE_16;
    if (t32a_mode_init(&timer_sout) != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*--- Initialize: Timer A ---*/
    {
        t32a_initial_setting_t *p_init = &timer_sout.init[T32A_TIMERA];

        /* Run Control      */
        p_init->runx.sftstp        = T32A_COUNT_STOP;
        p_init->runx.sftsta        = T32A_COUNT_DONT_START;
        p_init->runx.run           = T32A_RUN_ENABLE;
        /* Counter Control  */
        p_init->crx.prscl          = T32A_PRSCLx_1;
        p_init->crx.clk            = T32A_CLKx_PRSCLx;
        p_init->crx.wbf            = T32A_WBF_DISABLE;
        p_init->crx.updn           = T32A_COUNT_UP;
        p_init->crx.reld           = T32A_RELOAD_TREGx;
        p_init->crx.stop           = T32A_STOP_NON;
        p_init->crx.start          = T32A_START_NON;
        /* Output Control 0 */
        p_init->outcrx0.ocr        = T32A_OCR_DISABLE;
        /* Output Control 1 */
        p_init->outcrx1.ocrcap1    = T32A_OCRCAPx1_DISABLE;
        p_init->outcrx1.ocrcap0    = T32A_OCRCAPx0_DISABLE;
        p_init->outcrx1.ocrcmp1    = T32A_OCRCMPx1_INVERSION;
        p_init->outcrx1.ocrcmp0    = T32A_OCRCMPx0_INVERSION;
        /* Timer Counter 1  */
        p_init->rgx0.rgx0          = 0;
        p_init->rgx1.rgx1          = 1;
        /* Reload  */
        p_init->reldx.reld         = 0;
        /* Counter Capture  */
        p_init->tmrx.tmrx          = 0;
        /* Counter Capture Control A0 */
        p_init->capx0.capx0        = 0;
        /* Counter Capture Control A1 */
        p_init->capx1.capx1        = 0;
        /* Interrupt mask register Set */
        p_init->imx.imuf           = T32A_IMUFx_MASK_REQ;
        p_init->imx.imof           = T32A_IMOFx_MASK_REQ;
        p_init->imx.imx1           = T32A_IMx1_MASK_REQ;
        p_init->imx.imx0           = T32A_IMx0_MASK_REQ;
        /* DMA Request      */
        p_init->dma_req.dmaenx2    = T32A_DMAENx2_DISABLE;
        p_init->dma_req.dmaenx1    = T32A_DMAENx1_DISABLE;
        p_init->dma_req.dmaenx0    = T32A_DMAENx0_DISABLE;
        /* User ID          */
        p_init->id                 = 0;
        /* Handler          */
        p_init->handler_T   = T32A_NULL;
        p_init->handler_TC0 = T32A_NULL;
        p_init->handler_TC1 = T32A_NULL;
        if (t32a_timer_init(&timer_sout, T32A_TIMERA) != TXZ_SUCCESS)
        {
            result = TXZ_ERROR;
        }
        BSP_T32A_SOUT_RUNA_SFTSTAA = 1;
    }
    if (result != TXZ_SUCCESS)
    {
        /* no processing */
    }
}
#ifdef TRMOSC_LOSC
/*--------------------------------------------------*/
/** 
  * @brief  Initialize T32A Reference Signal Input Function
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void InitT32A_RefIn(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*-------------------------------*/
    /*  Reference Signal Input Timer */
    /*-------------------------------*/
    /*--- Construct (Register Allocation) ---*/
    /* Register Allocation */
    timer_refin.p_instance = BSP_T32A_REFIN;
    /*--- Initialize: Timer   ---*/
    /* Mode             */
    timer_refin.init_mode.mode.halt = T32A_DBG_HALT_RUN;
    timer_refin.init_mode.mode.mode = T32A_MODE_16;
    if (t32a_mode_init(&timer_refin) != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*--- Initialize: Timer A ---*/
    {
        t32a_initial_setting_t *p_init = &timer_refin.init[T32A_TIMERA];

        /* Run Control      */
        p_init->runx.sftstp        = T32A_COUNT_STOP;
        p_init->runx.sftsta        = T32A_COUNT_DONT_START;
        p_init->runx.run           = T32A_RUN_ENABLE;
        /* Counter Control  */
        p_init->crx.prscl          = T32A_PRSCLx_1;
        p_init->crx.clk            = T32A_CLKx_INTRG;
        p_init->crx.wbf            = T32A_WBF_DISABLE;
        p_init->crx.updn           = T32A_COUNT_UP;
        p_init->crx.reld           = T32A_RELOAD_TREGx;
        p_init->crx.stop           = T32A_STOP_NON;
        p_init->crx.start          = T32A_START_NON;
        /* Output Control 0 */
        p_init->outcrx0.ocr        = T32A_OCR_DISABLE;
        /* Output Control 1 */
        p_init->outcrx1.ocrcap1    = T32A_OCRCAPx1_DISABLE;
        p_init->outcrx1.ocrcap0    = T32A_OCRCAPx0_DISABLE;
        p_init->outcrx1.ocrcmp1    = T32A_OCRCMPx1_DISABLE;
        p_init->outcrx1.ocrcmp0    = T32A_OCRCMPx0_DISABLE;
        /* Timer Counter 1  */
        p_init->rgx0.rgx0          = 0;
        p_init->rgx1.rgx1          = 64;
        /* Reload  */
        p_init->reldx.reld         = 0;
        /* Counter Capture  */
        p_init->tmrx.tmrx          = 0;
        /* Capture Control */
        p_init->capcrx.capmx1      = T32A_CAPMx1_DISABLE;
        p_init->capcrx.capmx0      = T32A_CAPMx1_DISABLE;
        /* Counter Capture Control A0 */
        p_init->capx0.capx0        = 0;
        /* Counter Capture Control A1 */
        p_init->capx1.capx1        = 0;
        /* Interrupt mask register Set */
        p_init->imx.imuf           = T32A_IMUFx_MASK_REQ;
        p_init->imx.imof           = T32A_IMOFx_MASK_REQ;
        p_init->imx.imx1           = T32A_IMx1_MASK_REQ;
        p_init->imx.imx0           = T32A_IMx0_MASK_REQ;
        /* DMA Request      */
        p_init->dma_req.dmaenx2    = T32A_DMAENx2_DISABLE;
        p_init->dma_req.dmaenx1    = T32A_DMAENx1_DISABLE;
        p_init->dma_req.dmaenx0    = T32A_DMAENx0_DISABLE;
        /* User ID          */
        p_init->id                 = 0;
        /* Handler          */
        p_init->handler_T   = T32A_NULL;
        p_init->handler_TC0 = T32A_NULL;
        p_init->handler_TC1 = T32A_NULL;

        if (t32a_timer_init(&timer_refin, T32A_TIMERA) != TXZ_SUCCESS)
        {
            result = TXZ_ERROR;
        }
        BSP_T32A_REFIN_RUNA_SFTSTAA = 1;
    }
    if (result != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*---------------------------*/
    /*--- Initialize: Timer B ---*/
    /*---------------------------*/
    {
        t32a_initial_setting_t *p_init = &timer_refin.init[T32A_TIMERB];

        /* Run Control      */
        p_init->runx.sftstp        = T32A_COUNT_DONT_STOP;
        p_init->runx.sftsta        = T32A_COUNT_START;
        p_init->runx.run           = T32A_RUN_ENABLE;
        /* Counter Control  */
        p_init->crx.prscl          = T32A_PRSCLx_2;
        p_init->crx.clk            = T32A_CLKx_PRSCLx;
        p_init->crx.wbf            = T32A_WBF_DISABLE;
        p_init->crx.updn           = T32A_COUNT_UP;
        p_init->crx.reld           = T32A_RELOAD_NON;
        p_init->crx.stop           = T32A_STOP_NON;
        p_init->crx.start          = T32A_START_NON;
        /* Output Control 0 */
        p_init->outcrx0.ocr        = T32A_OCR_DISABLE;
        /* Output Control 1 */
        p_init->outcrx1.ocrcap1    = T32A_OCRCAPx1_DISABLE;
        p_init->outcrx1.ocrcap0    = T32A_OCRCAPx0_DISABLE;
        p_init->outcrx1.ocrcmp1    = T32A_OCRCMPx1_DISABLE;
        p_init->outcrx1.ocrcmp0    = T32A_OCRCMPx0_DISABLE;
        /* Timer Counter 1  */
        p_init->rgx0.rgx0          = 0;
        p_init->rgx1.rgx1          = 0;
        /* Reload  */
        p_init->reldx.reld         = 0;
        /* Counter Capture  */
        p_init->tmrx.tmrx          = 0;
        /* Capture Control */
        p_init->capcrx.capmx1      = T32A_CAPMx1_INTRG;
        p_init->capcrx.capmx0      = T32A_CAPMx0_DISABLE;
        /* Counter Capture Control B0 */
        p_init->capx0.capx0        = 0;
        /* Counter Capture Control B1 */
        p_init->capx1.capx1        = 0;
        /* Interrupt mask register Set */
        p_init->imx.imuf           = T32A_IMUFx_MASK_REQ;
        p_init->imx.imof           = T32A_IMOFx_MASK_REQ;
        p_init->imx.imx1           = T32A_IMx1_MASK_REQ;
        p_init->imx.imx0           = T32A_IMx0_MASK_REQ;
        /* DMA Request      */
        p_init->dma_req.dmaenx2    = T32A_DMAENx2_DISABLE;
        p_init->dma_req.dmaenx1    = T32A_DMAENx1_DISABLE;
        p_init->dma_req.dmaenx0    = T32A_DMAENx0_DISABLE;
        /* User ID          */
        p_init->id                 = 0;
        /* Handler          */
        p_init->handler_T   = T32A_NULL;
        p_init->handler_TC0 = T32A_NULL;
        p_init->handler_TC1 = T32A_NULL;
    }
    if (result != TXZ_SUCCESS)
    {
        /* no processing */
    }
}
#else
/*--------------------------------------------------*/
/** 
  * @brief  Initialize T32A Reference Signal Input Function
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void InitT32A_RefIn(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*-------------------------------*/
    /*  Reference Signal Input Timer */
    /*-------------------------------*/
    /*--- Construct (Register Allocation) ---*/
    /* Register Allocation */
    timer_refin.p_instance = BSP_T32A_REFIN;
    /*--- Initialize: Timer   ---*/
    /* Mode             */
    timer_refin.init_mode.mode.halt = T32A_DBG_HALT_RUN;
    timer_refin.init_mode.mode.mode = T32A_MODE_16;
    if (t32a_mode_init(&timer_refin) != TXZ_SUCCESS)
    {
        result = TXZ_ERROR;
    }
    /*--- Initialize: Timer A ---*/
    {
        t32a_initial_setting_t *p_init = &timer_refin.init[T32A_TIMERA];

        /* Run Control      */
        p_init->runx.sftstp        = T32A_COUNT_STOP;
        p_init->runx.sftsta        = T32A_COUNT_DONT_START;
        p_init->runx.run           = T32A_RUN_ENABLE;
        /* Counter Control  */
        p_init->crx.prscl          = T32A_PRSCLx_2;
        p_init->crx.clk            = T32A_CLKx_PRSCLx;
        p_init->crx.wbf            = T32A_WBF_DISABLE;
        p_init->crx.updn           = T32A_COUNT_UP;
        p_init->crx.reld           = T32A_RELOAD_NON;
        p_init->crx.stop           = T32A_STOP_EXTTRG_RISING_EDGE;
        p_init->crx.start          = T32A_START_EXTTRG_TRAILING_EDGE;
        /* Output Control 0 */
        p_init->outcrx0.ocr        = T32A_OCR_DISABLE;
        /* Output Control 1 */
        p_init->outcrx1.ocrcap1    = T32A_OCRCAPx1_DISABLE;
        p_init->outcrx1.ocrcap0    = T32A_OCRCAPx0_DISABLE;
        p_init->outcrx1.ocrcmp1    = T32A_OCRCMPx1_DISABLE;
        p_init->outcrx1.ocrcmp0    = T32A_OCRCMPx0_DISABLE;
        /* Timer Counter 1  */
        p_init->rgx0.rgx0          = 0;
        p_init->rgx1.rgx1          = 0;
        /* Reload  */
        p_init->reldx.reld         = 0;
        /* Counter Capture  */
        p_init->tmrx.tmrx          = 0;
        /* Capture Control */
        p_init->capcrx.capmx1      = T32A_CAPMx1_INx0_RISING_EDGE;
        p_init->capcrx.capmx0      = T32A_CAPMx0_INx0_TRAILING_EDGE;
        /* Counter Capture Control A0 */
        p_init->capx0.capx0        = 0;
        /* Counter Capture Control A1 */
        p_init->capx1.capx1        = 0;
        /* Interrupt mask register Set */
        p_init->imx.imuf           = T32A_IMUFx_MASK_REQ;
        p_init->imx.imof           = T32A_IMOFx_MASK_REQ;
        p_init->imx.imx1           = T32A_IMx1_MASK_REQ;
        p_init->imx.imx0           = T32A_IMx0_MASK_REQ;
        /* DMA Request      */
        p_init->dma_req.dmaenx2    = T32A_DMAENx2_DISABLE;
        p_init->dma_req.dmaenx1    = T32A_DMAENx1_DISABLE;
        p_init->dma_req.dmaenx0    = T32A_DMAENx0_DISABLE;
        /* User ID          */
        p_init->id                 = 0;
        /* Handler          */
        p_init->handler_T   = T32A_NULL;
        p_init->handler_TC0 = T32A_NULL;
        p_init->handler_TC1 = T32A_NULL;
    }
    if (result != TXZ_SUCCESS)
    {
        /* no processing */
    }
}
#endif
/*--------------------------------------------------*/
/** 
  * @brief  Start Capture
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void StartCapture(void)
{
#ifdef TRMOSC_LOSC
    BSP_T32A_REFIN_RUNB_SFTSTPB = 1;
    if (t32a_deinit(&timer_refin, T32A_TIMERB) != TXZ_SUCCESS)
    {
        /* no processing */
    }
    NVIC_ClearPendingIRQ(BSP_INTT32A_REFIN_BCAP1_IRQn);
    if (t32a_timer_init(&timer_refin, T32A_TIMERB) != TXZ_SUCCESS)
    {
        /* no processing */
    }
#else
    BSP_T32A_REFIN_RUNA_SFTSTPA = 1;
    if (t32a_deinit(&timer_refin, T32A_TIMERA) != TXZ_SUCCESS)
    {
        /* no processing */
    }
    NVIC_ClearPendingIRQ(BSP_INTT32A_REFIN_ACCAP0_IRQn);
    NVIC_ClearPendingIRQ(BSP_INTT32A_REFIN_ACCAP1_IRQn);
    if (t32a_timer_init(&timer_refin, T32A_TIMERA) != TXZ_SUCCESS)
    {
        /* no processing */
    }
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  Get Capture Data
  * @param  -
  * @retval Capture value
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t GetCaptureData(void)
{
#ifdef TRMOSC_LOSC
    uint32_t retval = 0;
    uint32_t captop = 0;
    uint32_t capend = 0;

    for (;;)
    {
        NVIC_ClearPendingIRQ(BSP_INTT32A_REFIN_BCAP1_IRQn);
        while (NVIC_GetPendingIRQ(BSP_INTT32A_REFIN_BCAP1_IRQn) == 0)
        {
            /* waiting raising edge ... */          
        }
        captop = (timer_refin.p_instance->CAPB1 & T32A_CAPx1_MASK);

        NVIC_ClearPendingIRQ(BSP_INTT32A_REFIN_BCAP1_IRQn);
        while (NVIC_GetPendingIRQ(BSP_INTT32A_REFIN_BCAP1_IRQn) == 0)
        {
            /* waiting raising edge ... */          
        }
        capend = (timer_refin.p_instance->CAPB1 & T32A_CAPx1_MASK);

        if(captop < capend) {
            retval = capend - captop;
        }
        else {
            retval = capend + (0xffff - captop);
        }
        break;
    }
#else
    uint32_t retval = 0;

    for (;;)
    {
        while (NVIC_GetPendingIRQ(BSP_INTT32A_REFIN_ACCAP0_IRQn) == 0)
        {
            /* waiting falling edge ... */
        }
        NVIC_ClearPendingIRQ(BSP_INTT32A_REFIN_ACCAP1_IRQn);
        while (NVIC_GetPendingIRQ(BSP_INTT32A_REFIN_ACCAP1_IRQn) == 0)
        {
            /* waiting raising edge ... */          
        }
        retval = (timer_refin.p_instance->CAPA1 & T32A_CAPx1_MASK);
        break;
    }   
#endif
    return(retval); 
}

/*--------------------------------------------------*/
/** 
  * @brief  Stop Capture
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
void StopCapture(void)
{
#ifdef TRMOSC_LOSC
    BSP_T32A_REFIN_RUNB_SFTSTPB = 1;
    if (t32a_deinit(&timer_refin, T32A_TIMERB) != TXZ_SUCCESS)
    {
        /* no processing */
    }
    NVIC_ClearPendingIRQ(BSP_INTT32A_REFIN_BCAP1_IRQn);
#else
    BSP_T32A_REFIN_RUNA_SFTSTPA = 1;
    if (t32a_deinit(&timer_refin, T32A_TIMERA) != TXZ_SUCCESS)
    {
        /* no processing */
    }
    NVIC_ClearPendingIRQ(BSP_INTT32A_REFIN_ACCAP0_IRQn);
    NVIC_ClearPendingIRQ(BSP_INTT32A_REFIN_ACCAP1_IRQn);
#endif
}
/** 
 *  @defgroup TRMOSC_Private_fuctions TRMOSC Private Fuctions
 *  @{
 */

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
   TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = USER_ID_UART;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = (uint32_t)CFG_UART_BAUDRATE;
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result  = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    return (result);
}
/*--------------------------------------------------*/
/** 
  * @brief  External LOSC Enable
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
#ifdef TRMOSC_LOSC
static void external_losc_enable(void)
{
    uint32_t work;

    /* [CGWUPLCR]<WUPTL> :Warm up time        */
    /*-----------------------------------------------*/
    /* "1"counter  (s) = 1 / ELOSC                   */
    /* "1"counter (us) = (10^6) / ELOSC              */
    /* "x"counter (us) = time                        */
    /*-----------------------------------------------*/
    /* x : time = 1 : (10^6) / ELOSC                 */
    /*-----------------------------------------------*/
    {
        uint64_t x = (uint64_t)((uint64_t)(ELOSC_CFG_WARM_UP_TIME) * (uint64_t)(ELOSC_CFG_CLOCK));
        x = (uint64_t)((uint64_t)(x / (uint64_t)(1000000)) - 16);
        if (x > (uint64_t)(0x7FFFF))
        {
            /* invalid value */
        }
        work = (uint32_t)x;
    }
    work  &= (uint32_t)(0xFFFFFFF0);
    work <<= 8;
    TSB_CG->WUPLCR = work;
    /* [RLMLOSCCR]<XTEN> :LOSC Enable         */
    TSB_RLM->LOSCCR = RLMLOSCCR_XTEN_RW_ENABLE;
    /* [CGWUPLCR]<WULON> :Enable              */
    work = (uint32_t)(TSB_CG->WUPLCR & CGWUPLCR_WUPTL_HIGH_MASK);
    TSB_CG->WUPLCR = (uint32_t)(work | CGWUPLCR_WULON_W_ENABLE);
    /* [CGWUPLCR]<WULEF> :Read(wait for warm-up) */
    while ((TSB_CG->WUPLCR & CGWUPLCR_WULEF_MASK) != CGWUPLCR_WULEF_R_DONE)
    {
        /* no processing */
    }
}
#endif
/**
 *  @}
 */ /* End of group TRMOSC_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup TRMOSC_Exported_functions TRMOSC Exported Functions
 *  @{
 */

/*--------------------------------------------------*/
/** 
  * @brief  The main function of TRMOSC demo
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    uint32_t tmp, i;

    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
#ifdef TRMOSC_LOSC
    /* LOSC enable */
    external_losc_enable();
    /* LOSC monitor (37pin:SCOUT) */
    TSB_PB_FR6_PB0F6 = 1;
    TSB_PB_CR_PB0C   = 1;
    TSB_CG->SCOCR    = 0x00000005;

    /* TRIGGER SELECTOR initialize                  */
    /* T32A ch6 A input trigger : LOSC              */
    /* T32A ch6 B input trigger : timer register A1 */
    TSB_TSEL1->CR1 = 0x00001153;
    TSB_TSEL1->CR4 = 0x00010000;
#endif
    /* Application initialization */
    if(application_initialize() == TXZ_ERROR){
        /* Do Nothing */
    }
    InitT32A_Sout();
    InitT32A_RefIn();
    InitTRM();                   /* for debug */

    if(printf("%s\n", "please wait...") < 0){ /* Do Nothing */ };
    for(i=0; i<0x00400000; i++){};
    if(printf("%s\n", "---------------") < 0){ /* Do Nothing */ };
    if(printf("%s\n", "|    start    |") < 0){ /* Do Nothing */ };
    if(printf("%s\n", "---------------") < 0){ /* Do Nothing */ };

    for(RetryCount=0; RetryCount <= COUNTMAX; RetryCount++) 
    {
        AdjustResult = IOSCAdjustment();
        
        if (AdjustResult == TRMOSC_DONE)
        {
            if(printf("%s\n", "TRMOSC_DONE") < 0){ /* Do Nothing */ };
            if(GetTrimmingControlValue() == TRMOSC_CONTROL_ENABLE) 
            {
                tmp = GetSetTrimValue();
                if(printf("%s%X\n", "<TRIMSET>:", tmp) < 0){ /* Do Nothing */ };
            }
            else 
            {
            }
            break;
        }
        else if (AdjustResult == TRMOSC_RUN) 
        {
            if(printf("%s\n", "TRMOSC_RUN") < 0){ /* Do Nothing */ };
            if(GetTrimmingControlValue() == TRMOSC_CONTROL_ENABLE) 
            {
                tmp = GetSetTrimValue();
                if(printf("%s%X\n", "<TRIMSET>:", tmp) < 0){ /* Do Nothing */ };
            }
            else 
            {
            }
        }
        else if (AdjustResult == TRMOSC_ERROR) 
        {
            if(printf("%s\n", "TRMOSC_Error") < 0){ /* Do Nothing */ };
            if(GetTrimmingControlValue() == TRMOSC_CONTROL_ENABLE) 
            {
                tmp = GetSetTrimValue();
                if(printf("%s%X\n", "<TRIMSET>:", tmp) < 0){ /* Do Nothing */ };
            }
            else 
            {
            }
        }
        else 
        {
        }
    }

    while(1){};
}
/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}


#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __NOP();
    }
}
#endif
/**
 *  @}
 */ /* End of group TRMOSC_Exported_functions */

/**
 *  @}
 */ /* End of group TRMOSC */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
