/** 
 *******************************************************************************
 * @file    main.c
 * @brief   ACCELEROMETER Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "bsp_com.h"
#include "tspi.h"
#include "bsp.h"
#include "bsp_uart_io.h"
#include "bsp_timer.h"
#include "bsp_led.h"

/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup ACCELEROMETER ACCELEROMETER Sample Appli
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup ACCELEROMETER_Private_macro ACCELEROMETER Private Macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group ACCELEROMETER_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup ACCELEROMETER_Private_define ACCELEROMETER Private Define
 *  @{
 */
/*----------------------*/
/*  Output Setting      */
/*----------------------*/
/** 
 *  @defgroup Output_Setting Output Setting
 *  @{
 */
#define CFG_OUTPUT_INTERVAL           ((uint32_t)1000)   /*!< deg.C output period(ms) */
/** 
 *  @}
 */ /* End of group Output_Setting */

/** 
 *  @defgroup Int_Status Int Status
 *  @{
 */
#define DATA_RDY           (0x01UL)      /*!< INT STATUS DATA_RDY_INT */
/** 
 *  @}
 */ /* End of group Int_Status */

/** 
 *  @defgroup LEDBlinkConfiguration LED Blink Time Configuration
 *  @brief    LED Blink Interval Time(ms).
 *  @{
 */
#define LED_ON_RANGE_36        ((uint16_t)(36))
#define LED_ON_RANGE_72        ((uint16_t)(72))
#define LED_ON_RANGE_108       ((uint16_t)(108))
#define LED_ON_RANGE_144       ((uint16_t)(144))
#define LED_ON_RANGE_180       ((uint16_t)(180))
/** 
 *  @}
 */ /* End of group LEDBlinkConfiguration */

/** 
 *  @}
 */ /* End of group ACCELEROMETER_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup ACCELEROMETER_Private_define ACCELEROMETER Private Define
 *  @{
 */

#define MAIN_NULL       ((void *)0)           /*!< Null Pointer  */
#define USER_ID_UART    ((uint32_t)1)         /*!< User ID :USB UART (Any value) */
#define USER_ID_TSPI_TX ((uint32_t)2)         /*!< User ID :TSPI (Any value) */
#define USER_ID_TSPI_RX ((uint32_t)3)         /*!< User ID :TSPI (Any value) */

/** 
 *  @}
 */ /* End of group ACCELEROMETER_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup ACCELEROMETER_Private_define ACCELEROMETER Private Define
 *  @{
 */
/*----------------------------------*/
/** 
 * @enum   Processing
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_START,                 /*!< Start.             */
    PROC_SENSOR_ON,             /*!< Sensor on.         */
    PROC_ACCEL_XYZ_OUT,
    PROC_OUT_DATA_CONVERSION,
    PROC_OUTPUT_DATA,           /*!< x,y,z data output. */
    PROC_ERROR                  /*!< Error.             */
} Processing;

/*----------------------------------*/
/** 
 * @enum   SendState
 * @brief  Send state.
*/
/*----------------------------------*/
typedef enum {
    SEND_IDLE = 0U,       /*!< Idle                   */
    SEND_RUN,             /*!< Transmiting.           */
    SEND_SUCCESS,         /*!< Transmission complete. */
    SEND_FAILURE          /*!< Transmission error.    */
} SendState;

/*----------------------------------*/
/**
 * @struct send_t
 * @brief  Send information.
*/
/*----------------------------------*/
typedef struct
{
    volatile SendState  state;                          /*!< Status.                     */
} send_t;

/*----------------------------------*/
/** 
 * @struct receive_t
 * @brief  Receive information.
*/
/*----------------------------------*/
typedef struct
{
    int16_t     x_data;
    int16_t     y_data;
    int16_t     z_data;
    int16_t     ave_x_data;
    int16_t     ave_y_data;
    int16_t     ave_z_data;
    int16_t     x_data_temp[10];
    int16_t     y_data_temp[10];
    int16_t     z_data_temp[10];
    int16_t     angle_x_data;
    int16_t     angle_y_data;
    int16_t     angle_z_data;
} accel_t;

/** 
 *  @}
 */ /* End of group ACCELEROMETER_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup ACCELEROMETER_Private_typedef ACCELEROMETER Private Typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
    t32a_t      t32a;                       /*!< Driver      :T32A.     */
    uart_io_t   *p_uart;                    /*!< Application :USB UART. */
    timer_t     timer;                      /*!< Application :Timer (1ms timer). */
    led_t       led[(uint32_t)BSP_LED_MAX]; /*!< Application :LED                */
    tspi_t      tspi_tx_obj;                /*!< Application :TSPI. */
} instance_t;

/*----------------------------------*/
/** 
 * @brief Information.
*/
/*----------------------------------*/
typedef struct
{
   TXZ_FunctionalState update;     /*!< Update flag.       */
   uint32_t            count;      /*!< Update flag.       */
} information_t;
/**
 *  @}
 */ /* End of group ACCELEROMETER_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup ACCELEROMETER_Private_variables ACCELEROMETER Private Variables
 *  @{
 */
static instance_t instance;     /*!< Instance.                */
static Processing proc;         /*!< Processing mode.         */
static information_t info;      /*!< Clock information.       */
static tspi_transmit_t t_info;
static send_t  send;
static accel_t accel;
static uint8_t read_data;
static uint8_t data_rdy_int;
static uint8_t i;
static uint8_t counter;
static uint16_t led_range;

/**
 *  @}
 */ /* End of group ACCELEROMETER_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup ACCELEROMETER_Private_fuctions ACCELEROMETER Private Fuctions
 *  @{
 */
static void variable_initialize(void);
static TXZ_Result driver_initialize(void);
static TXZ_Result driver_finalize(void);
static TXZ_Result application_initialize(void);
static TXZ_Result application_finalize(void);
static void timer_interval_handler(uint32_t id);
static void tspi_writedata_trans(uint8_t write_data);
static int16_t angle_trans(int16_t data1, int16_t data2, int16_t data3);

/*--------------------------------------------------*/
/** 
  * @brief  Variable Initialize
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void variable_initialize(void)
{
    /* Instance. */
    instance.t32a.p_instance    = MAIN_NULL;
    instance.p_uart             = MAIN_NULL;
    instance.timer.init.p_t32a  = MAIN_NULL;
    /* Processing mode. */
    proc = PROC_START;
    /* Information. */
    info.update = TXZ_ENABLE;  /* Init Output. */
    info.count  = 0;
}

/*--------------------------------------------------*/
/** 
  * @brief      Transmit handler.
  * @param[in]  id      :Object address.
  * @param[in]  result  :Result.
  * @retval     -
  * @note       Called by tspi driver
  */
/*--------------------------------------------------*/
static void transmit_handler(uint32_t id, TXZ_Result result)
{
    if((instance.tspi_tx_obj.p_instance->SR & 0x0000000F) != 0) {
        read_data = (instance.tspi_tx_obj.p_instance->DR);
    }
    
    if (result == TXZ_SUCCESS)
    {
        /*------------------------------*/
        /*  Result Check                */
        /*------------------------------*/
        if (result == TXZ_SUCCESS)
        {
            send.state = SEND_SUCCESS;
        }
        else
        {
            send.state = SEND_FAILURE;
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize
  * @param  -
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (p_t32a->p_instance != MAIN_NULL)
        {
            if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver initialize will process within USB UART. */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Finalize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver finalize has done within USB UART.    */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Finalize ---*/
        /* no processing */
        /*--- Destruct ---*/
        /* Register Release */
        p_t32a->p_instance = MAIN_NULL;
    }
    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver finalize will process within BSP.     */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize
  * @param  -
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    {
        timer_t *p_timer = &instance.timer;

        p_timer->init.id       = (uint32_t)&instance.timer;
        p_timer->init.p_t32a   = &instance.t32a;
        p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
        p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
        p_timer->init.interval = 1000;
        p_timer->init.handler  = timer_interval_handler;
        timer_initialize(p_timer);
    }
    
    /*----------------------*/
    /* LED                  */
    /*----------------------*/
    {
        uint32_t i;
        led_t *p_led = MAIN_NULL;
        for (i=0; i<(uint32_t)BSP_LED_MAX; i++)
        {
            p_led = &instance.led[i];

            p_led->init.id                  = (uint32_t)p_led;
            p_led->init.p_gpio              = bsp_get_gpio_instance_address();
            p_led->init.port.group          = bsp_get_gpio_group_led((BSPLed)(i));
            p_led->init.port.num            = bsp_get_gpio_num_led((BSPLed)(i));
            p_led->init.state               = LED_STATE_OFF;
            led_initialize(p_led);
        }
    }
    
    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = USER_ID_UART;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = bsp_get_usb_uart_tx_baudrate(BSP_USB_UART_0);
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result  = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    
    /*----------------------*/
    /* TSPI             */
    /*----------------------*/
    /* Tx TSPI channel. */
    {
        instance.tspi_tx_obj.init.id  = USER_ID_TSPI_TX;
        switch (bsp_get_tspi_tx_ch(BSP_SFLASH_1))
        {
        case 0:
            instance.tspi_tx_obj.p_instance  = TSB_TSPI0;
            break;
        case 1:
            instance.tspi_tx_obj.p_instance  = TSB_TSPI1;
            break;
        case 2:
            instance.tspi_tx_obj.p_instance  = TSB_TSPI2;
            break;
        case 3:
            instance.tspi_tx_obj.p_instance  = TSB_TSPI3;
            break;
        default:
            instance.tspi_tx_obj.p_instance  = MAIN_NULL;
            break;
        }
        /* Set Initial parameter of TSPIxCR1 */
        instance.tspi_tx_obj.init.cnt1.trxe = TSPI_DISABLE;
        instance.tspi_tx_obj.init.cnt1.tspims = TSPI_SPI_MODE;
        instance.tspi_tx_obj.init.cnt1.mstr = TSPI_MASTER_OPERATION;
        instance.tspi_tx_obj.init.cnt1.tmmd = TSPI_TWO_WAY;
        instance.tspi_tx_obj.init.cnt1.cssel = TSPI_TSPIxCS0_ENABLE;
        instance.tspi_tx_obj.init.cnt1.fc = TSPI_TRANS_RANGE_CONTINUE;
        /* Set Initial parameter of TSPIxCR2 */
        instance.tspi_tx_obj.init.cnt2.tidle = TSPI_TIDLE_HI;
        instance.tspi_tx_obj.init.cnt2.txdemp = TSPI_TXDEMP_HI;
        instance.tspi_tx_obj.init.cnt2.rxdly = TSPI_RXDLY_FSYS_FSCK_12;
        instance.tspi_tx_obj.init.cnt2.til = TSPI_TX_FILL_LEVEL_1;
        instance.tspi_tx_obj.init.cnt2.ril = TSPI_RX_FILL_LEVEL_1;
        instance.tspi_tx_obj.init.cnt2.inttxwe = TSPI_TX_INT_ENABLE;
        instance.tspi_tx_obj.init.cnt2.intrxwe = TSPI_RX_INT_ENABLE;
        instance.tspi_tx_obj.init.cnt2.inttxfe = TSPI_TX_FIFO_INT_DISABLE;
        instance.tspi_tx_obj.init.cnt2.intrxfe = TSPI_RX_FIFO_INT_DISABLE;
        instance.tspi_tx_obj.init.cnt2.interr = TSPI_ERR_INT_DISABLE;
        instance.tspi_tx_obj.init.cnt2.dmate = TSPI_TX_DMA_INT_DISABLE;
        instance.tspi_tx_obj.init.cnt2.dmare = TSPI_RX_DMA_INT_DISABLE;
        /* Set Initial parameter of TSPIxBR */
        instance.tspi_tx_obj.init.brd.brck = TSPI_BR_CLOCK_1;
        instance.tspi_tx_obj.init.brd.brs = TSPI_BR_DIVIDER_6;
        /* Set Initial parameter of TSPIxFMTR0 */
        instance.tspi_tx_obj.init.fmr0.dir = TSPI_DATA_DIRECTION_MSB;
        instance.tspi_tx_obj.init.fmr0.fl = TSPI_DATA_LENGTH_8;
        instance.tspi_tx_obj.init.fmr0.fint = TSPI_INTERVAL_TIME_0;
        instance.tspi_tx_obj.init.fmr0.cs3pol = TSPI_TSPIxCS3_NEGATIVE;
        instance.tspi_tx_obj.init.fmr0.cs2pol = TSPI_TSPIxCS2_NEGATIVE;
        instance.tspi_tx_obj.init.fmr0.cs1pol = TSPI_TSPIxCS1_NEGATIVE;
        instance.tspi_tx_obj.init.fmr0.cs0pol = TSPI_TSPIxCS0_NEGATIVE;
        instance.tspi_tx_obj.init.fmr0.ckpha = TSPI_SERIAL_CK_2ND_EDGE;
        instance.tspi_tx_obj.init.fmr0.ckpol = TSPI_SERIAL_CK_IDLE_HI;
        instance.tspi_tx_obj.init.fmr0.csint = TSPI_MIN_IDLE_TIME_1;
        instance.tspi_tx_obj.init.fmr0.cssckdl = TSPI_SERIAL_CK_DELAY_1;
        instance.tspi_tx_obj.init.fmr0.sckcsdl = TSPI_NEGATE_1;
        /* Set Initial parameter of TSPIxFMTR1 */
        instance.tspi_tx_obj.init.fmr1.vpe = TSPI_PARITY_DISABLE;
        instance.tspi_tx_obj.init.fmr1.vpm = TSPI_PARITY_BIT_ODD;
        instance.tspi_tx_obj.transmit.handler = transmit_handler;
        if ( instance.tspi_tx_obj.p_instance != MAIN_NULL)
        {
            if (tspi_init(&instance.tspi_tx_obj) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Finalize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    if (instance.p_uart != MAIN_NULL)
    {
        uart_io_finalize();
    }
    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    if (instance.timer.init.p_t32a != MAIN_NULL)
    {
        timer_finalize(&instance.timer);
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    if (id == (uint32_t)&instance.timer)
    {
        /* Increment Tick. Tick is used to driver's Timeout. */
        hal_inc_tick();
        /* Interval Time Count. */
        info.count++;
        if (info.count >= CFG_OUTPUT_INTERVAL)
        {
            /*>>> Critical information. */
            info.update = TXZ_ENABLE;
            /*<<< Critical information. */
            info.count  = 0;
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void tspi_writedata_trans(uint8_t write_data)
{
    int ret;
    
    send.state = SEND_RUN;
    /* Receive TRXE Enable  */
    instance.tspi_tx_obj.p_instance->CR1 |= TSPI_TRXE_ENABLE;
    /* NVIC Enable (TX) */
    NVIC_EnableIRQ(bsp_get_tspi_tx_nvic(BSP_SFLASH_1));
    NVIC_ClearPendingIRQ(bsp_get_tspi_tx_nvic(BSP_SFLASH_1));
    /* Send Data Set */
    t_info.tx8.p_data = (uint8_t *)&write_data;
    t_info.tx8.num = 1;
    /* Transfer Start */
    ret = tspi_master_transfer(&instance.tspi_tx_obj, &t_info);
    
    while(send.state == SEND_RUN)
    {
        /* wait transmit */
    }
    
    if(send.state == SEND_FAILURE){
        send.state = SEND_IDLE;
        proc = PROC_ERROR;
    }
    else if(send.state == SEND_SUCCESS) 
    {
        /* NVIC Disable (TX) */
        NVIC_DisableIRQ(bsp_get_tspi_tx_nvic(BSP_SFLASH_1));
        if(ret == TXZ_ERROR)
        {
            proc = PROC_ERROR;
        }
        instance.tspi_tx_obj.p_instance->CR1 &= TSPI_TRXE_DISABLE_MASK;
        instance.tspi_tx_obj.p_instance->SR |=  TSPI_TX_DONE_CLR;
        send.state = SEND_IDLE;
    }
}


/*--------------------------------------------------*/
/** 
  * @brief  Angle Transformation
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static int16_t angle_trans(int16_t data1, int16_t data2, int16_t data3)
{
    int16_t temp_data;
    
    temp_data = (int16_t)(atan2(data1,sqrt(data2*data2 + data3*data3)) * (180 / (atan(1) * 4)));
    
    return(temp_data);

}
/**
 *  @}
 */ /* End of group ACCELEROMETER_Private_fuctions */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup ACCELEROMETER_Exported_functions ACCELEROMETER Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /*--- variable    ---*/
    variable_initialize();
    /*--- driver      ---*/
    if (driver_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    /*--- application ---*/
    if (application_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(proc != PROC_ERROR)
    {
        switch(proc)
        {
            case PROC_START:
                {
                    timer_start(&instance.timer);
                    proc = PROC_SENSOR_ON;
                    TSB_PB_DATA_PB5 = 1;
                }
                break;
            case PROC_SENSOR_ON:
                {
                    TSB_PB_DATA_PB5 = 0;
                    tspi_writedata_trans(0x6B);                                         /* Write addr 0x6B (PWR_MGMT_1) */
                    tspi_writedata_trans(0x20);                                         /* Write data 0x20 (CYCLE bit on) */
                    TSB_PB_DATA_PB5 = 1;
                    TSB_PB_DATA_PB5 = 0;
                    tspi_writedata_trans(0xF5);                                         /* Write addr 0x75 (WHO_AMI) */
                    tspi_writedata_trans(0xFF);                                         /* Read data */
                    TSB_PB_DATA_PB5 = 1;

                    TSB_PB_DATA_PB5 = 0;
                    tspi_writedata_trans(0x1D);                                         /* Write addr 0x1D (ACCELEROMETER CONFIGURATION 2) */
                    tspi_writedata_trans(0x37);                                         /* Write data 0x37 (DEC2_CFG=3, ACCEL_FCHOICE_B=0, A_DLPF_CFG= 7) */
                    TSB_PB_DATA_PB5 = 1;
                    proc = PROC_ACCEL_XYZ_OUT;
                }
                break;
            case PROC_ACCEL_XYZ_OUT:
                {
                    TSB_PB_DATA_PB5 = 0;
                    tspi_writedata_trans(0xBA);                                         /* Read addr 0x3A (INT_STATUS) */
                    tspi_writedata_trans(0xFF);                                         /* Dummy data */
                    TSB_PB_DATA_PB5 = 1;
                    data_rdy_int = (uint8_t)(read_data & DATA_RDY);
                    if(data_rdy_int == DATA_RDY)
                    {
                        /* Sensor value read */
                        /* ACCEL_XOUT_H read start */
                        TSB_PB_DATA_PB5 = 0;
                        tspi_writedata_trans(0xBB);                             /* Read addr 0x3B (ACCEL_XOUT_H) */
                        tspi_writedata_trans(0xFF);                             /* Dummy data */
                        TSB_PB_DATA_PB5 = 1;
                        accel.x_data = (int16_t)(read_data << 8);
                        /* ACCEL_XOUT_H read end */

                        /* ACCEL_XOUT_L read start */
                        TSB_PB_DATA_PB5 = 0;
                        tspi_writedata_trans(0xBC);                             /* Read addr 0x3C (ACCEL_XOUT_L) */
                        tspi_writedata_trans(0xFF);                             /* Dummy data */
                        TSB_PB_DATA_PB5 = 1;
                        accel.x_data |= (int16_t)(read_data);
                        /* ACCEL_XOUT_L read end */

                        /* ACCEL_YOUT_H read start */
                        TSB_PB_DATA_PB5 = 0;
                        tspi_writedata_trans(0xBD);                             /* Read addr 0x3D (ACCEL_YOUT_H) */
                        tspi_writedata_trans(0xFF);                             /* Dummy data */
                        TSB_PB_DATA_PB5 = 1;
                        accel.y_data = (int16_t)(read_data << 8);
                        /* ACCEL_YOUT_H read end */

                        /* ACCEL_YOUT_L read start */
                        TSB_PB_DATA_PB5 = 0;
                        tspi_writedata_trans(0xBE);                             /* Read addr 0x3E (ACCEL_YOUT_L) */
                        tspi_writedata_trans(0xFF);                             /* Dummy data */
                        TSB_PB_DATA_PB5 = 1;
                        accel.y_data |= (int16_t)(read_data);
                        /* ACCEL_YOUT_L read end */

                        /* ACCEL_ZOUT_H read start */
                        TSB_PB_DATA_PB5 = 0;
                        tspi_writedata_trans(0xBF);                             /* Read addr 0x3F (ACCEL_ZOUT_H) */
                        tspi_writedata_trans(0xFF);                             /* Dummy data */
                        TSB_PB_DATA_PB5 = 1;
                        accel.z_data = (int16_t)(read_data << 8);
                        /* ACCEL_ZOUT_H read end */

                        /* ACCEL_ZOUT_L read start */
                        TSB_PB_DATA_PB5 = 0;
                        tspi_writedata_trans(0xC0);                             /* Read addr 0x40 (ACCEL_ZOUT_L) */
                        tspi_writedata_trans(0xFF);                             /* Dummy data */
                        TSB_PB_DATA_PB5 = 1;
                        accel.z_data |= (int16_t)(read_data);
                        /* ACCEL_ZOUT_L read end */

                        proc = PROC_OUT_DATA_CONVERSION;
                    }
                }
                break;
            case PROC_OUT_DATA_CONVERSION:
                {
                    int16_t x_min,y_min,z_min;
                    int16_t x_max,y_max,z_max;
                    int32_t x_sum,y_sum,z_sum;
                    
                    accel.x_data_temp[counter] = accel.x_data;
                    accel.y_data_temp[counter] = accel.y_data;
                    accel.z_data_temp[counter] = accel.z_data;
                    
                    if(++counter >= 10){
                        counter = 0;
                    }
                    
                    x_min = 0xFFFF;
                    y_min = 0xFFFF;
                    z_min = 0xFFFF;
                    
                    x_max = 0x0000;
                    y_max = 0x0000;
                    z_max = 0x0000;
                    
                    x_sum = 0x00000000;
                    y_sum = 0x00000000;
                    z_sum = 0x00000000;
                    
                    for(i=0; i<10; i++){
                        
                        x_sum += accel.x_data_temp[i];
                        y_sum += accel.y_data_temp[i];
                        z_sum += accel.z_data_temp[i];
                        
                        /* x_data max and min */
                        if(accel.x_data_temp[i] < x_min){
                            x_min = accel.x_data_temp[i];
                        }
                        if(accel.x_data_temp[i] > x_max){
                            x_max = accel.x_data_temp[i];
                        }
                        
                        /* y_data max and min */
                        if(accel.y_data_temp[i] < y_min){
                            y_min = accel.y_data_temp[i];
                        }
                        if(accel.y_data_temp[i] > y_max){
                            y_max = accel.y_data_temp[i];
                        }
                        
                        /* z_data max and min */
                        if(accel.z_data_temp[i] < z_min){
                            z_min = accel.z_data_temp[i];
                        }
                        if(accel.z_data_temp[i] > z_max){
                            z_max = accel.z_data_temp[i];
                        }
                    }
                    
                    proc = PROC_ACCEL_XYZ_OUT;
                    
                    if(info.update == TXZ_ENABLE)
                    {
                        /*>>> Critical information. */
                        info.update = TXZ_DISABLE;
                        
                        /* Average calculation */
                        x_sum -= (x_max + x_min);
                        y_sum -= (y_max + y_min);
                        z_sum -= (z_max + z_min);
                        
                        accel.ave_x_data = (int16_t)(x_sum / 8);
                        accel.ave_y_data = (int16_t)(y_sum / 8);
                        accel.ave_z_data = (int16_t)(z_sum / 8);
                        
                        /* Angle conversion */
                        accel.angle_x_data = angle_trans(accel.ave_x_data, accel.ave_y_data, accel.ave_z_data);
                        accel.angle_y_data = angle_trans(accel.ave_y_data, accel.ave_x_data, accel.ave_z_data);
                        accel.angle_z_data = (int16_t)(atan2(sqrt(accel.ave_x_data*accel.ave_x_data + accel.ave_y_data*accel.ave_y_data),accel.ave_z_data) * (180 / (atan(1) * 4)));
                        
                        proc = PROC_OUTPUT_DATA;
                    }
                }
                break;
            case PROC_OUTPUT_DATA:
                {
                    led_t *p_led = MAIN_NULL;
                    
                    if(accel.ave_z_data >= 0){
                        printf("X:%d Y:%d Z:%d\n", accel.angle_x_data,accel.angle_y_data,accel.angle_z_data);
                    } else {
                        printf("Detection angle over!!\n");
                    }
                    
                    /* LED lightning by angle */
                    led_range = (abs(accel.angle_x_data) + abs(accel.angle_y_data) + abs(accel.angle_z_data));
                    
                    if(led_range < LED_ON_RANGE_36){
                        for (i=0; i<=3; i++)
                        {
                            p_led = &instance.led[i];
                            led_turn_off(p_led);
                        }
                    } else if((LED_ON_RANGE_36 <= led_range) && (led_range < LED_ON_RANGE_72)){
                        p_led = &instance.led[0];
                        led_turn_on(p_led);
                        for (i=1; i<=3; i++)
                        {
                            p_led = &instance.led[i];
                            led_turn_off(p_led);
                        }
                    } else if((LED_ON_RANGE_72 <= led_range) && (led_range < LED_ON_RANGE_108)){
                        for (i=0; i<=1; i++)
                        {
                            p_led = &instance.led[i];
                            led_turn_on(p_led);
                        }
                        for (i=2; i<=3; i++)
                        {
                            p_led = &instance.led[i];
                            led_turn_off(p_led);
                        }
                    } else if((LED_ON_RANGE_108 <= led_range) && (led_range < LED_ON_RANGE_144)){
                        for (i=0; i<=2; i++)
                        {
                            p_led = &instance.led[i];
                            led_turn_on(p_led);
                        }
                        p_led = &instance.led[3];
                        led_turn_off(p_led);
                    } else if((LED_ON_RANGE_144 <= led_range) && (led_range < LED_ON_RANGE_180)){
                        for (i=0; i<(int32_t)BSP_LED_MAX; i++)
                        {
                            p_led = &instance.led[i];
                            led_turn_on(p_led);
                        }
                    } else {
                        /* no processing */
                    }
                    proc = PROC_ACCEL_XYZ_OUT;
                }
                break;
            case PROC_ERROR:
                /* no processing */
                break;
            default:
                /* no processing */
                break;
        }
    }
    
    /*>>> System Error. Please Debug. */
    /*----------------------*/
    /* Finalize             */
    /*----------------------*/
    /*--- application ---*/
    if (application_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*--- driver      ---*/
    if (driver_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*----------------------*/
    /* BSP finalize         */
    /*----------------------*/
    bsp_finalize();
}
#ifdef DEBUG
    assert_failed(MAIN_NULL, 0);
#endif

/*--------------------------------------------------*/
/** 
  * @brief  SPI flash transmit IRQ.
  * @param  sflash :SPI Flash.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_sflash_tx(BSPSFlash sflash)
{
    if(sflash == 1)
    {
        tspi_irq_handler_transmit(&instance.tspi_tx_obj);
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer.
  * @param  timer :Timer.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    default:
        /* no processing */
        break;
    }
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __nop();
    }
}
#endif
/**
 *  @}
 */ /* End of group ACCELEROMETER_Exported_functions */

/**
 *  @}
 */ /* End of group ACCELEROMETER */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
