/** 
 *******************************************************************************
 * @file    main.c
 * @brief   TSPI SPI Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif
/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup TSPI_SPI TSPI_SPI Sample Appli
 *  @{
 */

/** 
 *  @defgroup MAIN_PRIVATE  MAIN Private
 *  @brief    MAIN Private Group.
 *  @{
 */

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "bsp.h"
#include "crc.h"
#include "bsp_uart_io.h"
/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/* no define */

/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/*----------------------*/
/*  UART Setting        */
/*----------------------*/
/** 
 *  @defgroup UART_Setting UART Setting
 *  @{
 */
#define CFG_UART_BAUDRATE               ((uint32_t)115200)  /*!< Boudrate(bps) */
/** 
 *  @}
 */ /* End of group UART_Setting */

/** 
 *  @}
 */ /* End of group Timer_EVENT_COUNT_Private_define */
#define MAIN_NULL       ((void *)0)     /*!< NULL. */
#define USER_ID_REMOTE  ((uint32_t)0)   /*!< User ID :Remote Control (Any value) */
#define USER_ID_UART    ((uint32_t)1)   /*!< User ID :USB UART (Any value) */
#define USER_ID_TSPI_TX    ((uint32_t)2)   /*!< User ID :TSPI (Any value) */
#define USER_ID_TSPI_RX    ((uint32_t)3)   /*!< User ID :TSPI (Any value) */
/** 
 *  @defgroup TSPI_IO_SendMax  TSPI IO Send Max.
 *  @brief    Size of send max. (byte)
 *  @{
 */
#define TSPI_IO_SEND_MAX       ((uint32_t)255)
/**
 *  @}
 */ /* End of group TSPI_IO_SendMax */

/** 
 *  @defgroup TSPI_IO_ReceiveMax  TSPI IO Receive Max.
 *  @brief    Size of receive max. (byte)
 *  @{
 */
#define TSPI_IO_RECEIVE_MAX    ((uint32_t)255)
/**
 *  @}
 */ /* End of group TSPI_IO_ReceiveMax */

#define CRC16
/* #define CRC32 */
/* #define CheckSum */

/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/*----------------------------------*/
/** 
 * @enum   Processing
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_IDLE,              /*!< Idle.                        */
    PROC_OUTPUT_MSG,        /*!< Output first message.        */
    PROC_INPUT,             /*!< Wait to input.               */
    PROC_COMMAND_PARSE,       /*!< Output Echo back data.       */
    PROC_WRITE_EXEC,       /*!< SEND CRC code      */
    PROC_WRITE_EXEC2,     /*!< SEND data      */
    PROC_READ_EXEC,       /*!< Output Echo back data.       */
    PROC_CRC_CHECK,       /*!< Output Echo back data.       */
    PROC_OUTPUT_ERR,        /*!< Output input error message.  */
    PROC_ERROR              /*!< Error.                       */
} Processing;

/*----------------------------------*/
/** 
 * @enum   SendState
 * @brief  Send state.
*/
/*----------------------------------*/
typedef enum {
    SEND_IDLE = 0U,     /*!< Idle                   */
    SEND_RUN,           /*!< Transmiting.           */
    SEND_SUCCESS,       /*!< Transmission complete. */
    SEND_FAILURE        /*!< Transmission error.    */
} SendState;
/*----------------------------------*/
/** 
 * @enum   ReceiveState
 * @brief  Receive state.
*/
/*----------------------------------*/
typedef enum {
    RECV_IDLE = 0U,     /*!< Idle                   */
    RECV_RUN,           /*!< Receiving              */
    RECV_SUCCESS,       /*!< Transmission complete. */
    RECV_FAILURE        /*!< Received error.        */
} ReceiveState;

/** 
 * @struct send_t
 * @brief  Send information.
*/
/*----------------------------------*/
typedef struct
{
    volatile SendState  state;                          /*!< Status.                     */
    uint32_t            wp;                             /*!< Write pointer.              */
    uint32_t            rp;                             /*!< Read pointer(Now, no use).  */
    uint8_t             buff[TSPI_IO_SEND_MAX];         /*!< Send data.                  */
} send_t;

/*----------------------------------*/
/** 
 * @struct receive_t
 * @brief  Receive information.
*/
/*----------------------------------*/
typedef struct
{
    volatile ReceiveState state;                        /*!< Status.                     */
    volatile uint32_t     wp;                           /*!< Write pointer.              */
    volatile uint32_t     rp;                           /*!< Read pointer(Now, no use).  */
    volatile uint32_t     length;                       /*!< Num of receive data.        */
    volatile uint8_t      buff[TSPI_IO_RECEIVE_MAX];    /*!< Receive data.               */
} receive_t;

/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/*----------------------------------*/
/** 
 * @struct input_t
 * @brief  Input data information.
*/
/*----------------------------------*/
typedef struct
{
    char    data[TSPI_IO_SEND_MAX];  /*!< Input data(+ for NULL 1byte). */
    uint16_t wp;                             /*!< Write pointer.                */
} input_t;
/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
    tspi_t      tspi_tx_obj;               /*!< Application :TSPI. */
    tspi_t      tspi_rx_obj;               /*!< Application :TSPI. */
    gpio_t      gpio_obj;            /*!< Application :GPIO. */
} instance_t;


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
static instance_t instance; /*!< Instance.                    */
static Processing proc;     /*!< Processing mode.        */
static input_t    input;        /*!< Input data information. */
static uint8_t write_data[TSPI_IO_SEND_MAX] = "TOSHIBA"; /*!< Write data buffer. */
static uint8_t read_data[TSPI_IO_RECEIVE_MAX];  /*!< read data buffer. */
static receive_t    receive;  /*!< Receive information.  */
static tspi_receive_t r_info;
static tspi_transmit_t t_info;
send_t       send;                               /*!< Send information.     */
static uint8_t t_data[TSPI_IO_SEND_MAX+1+4] ; /*!< Transmit data + length + code */
/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
static void variable_initialize(void);
static void clear_data(void);
void tspi_io_initialize(void);
static void tspi_io_error(void);
static void transmit_handler(uint32_t id, TXZ_Result result);
static void receive_handler(uint32_t id, TXZ_Result result, tspi_receive_t *p_info);
static TXZ_Result application_initialize(void);
#ifdef CheckSum
static uint8_t checksum_calc(uint8_t *data, uint16_t length) ;
#endif
/*--------------------------------------------------*/
/** 
  * @brief  Variable Initialize
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void variable_initialize(void)
{
    /* Instance. */
    instance.tspi_tx_obj.p_instance    = MAIN_NULL;
    instance.tspi_rx_obj.p_instance    = MAIN_NULL;

}

/*--------------------------------------------------*/
/** 
  * @brief  Clear input data.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void clear_data(void)
{
    uint16_t i;

    for (i=0; i<UART_IO_RECEIVE_MAX; i++)
    {
        input.data[i] = 0;
    t_data[i] = 0;
    }
    input.wp = 0;
    
    for (i=0; i<UART_IO_RECEIVE_MAX; i++)
    {
        read_data[i] = 0;
    }
}
/*--------------------------------------------------*/
/** 
  * @brief  Error Hook
  * @param  -
  * @retval -
  * @note   for debug
  */
/*--------------------------------------------------*/
static void tspi_io_error(void)
{
#ifdef TSPI_IO_CFG_DEBUG
    while(1)
    {
        __NOP();
    }
#endif
}
/*--------------------------------------------------*/
/** 
  * @brief      Transmit handler.
  * @param[in]  p_obj   :Object address.
  * @param[in]  result  :Result.
  * @retval     -
  * @note       Called by tspi driver
  */
/*--------------------------------------------------*/
static void transmit_handler(uint32_t id, TXZ_Result result)
{
    if (result == TXZ_SUCCESS)
    {
        /*------------------------------*/
        /*  Result Check                */
        /*------------------------------*/
        if (result == TXZ_SUCCESS)
        {
            send.state = SEND_SUCCESS;
        }
        else
        {
            send.state = SEND_FAILURE;
        }
    }
}
/*--------------------------------------------------*/
/** 
  * @brief      Receive handler.
  * @param[in]  p_obj   :Object address.
  * @param[in]  result  :Result.
  * @param[in]  p_info  :Received information.
  * @retval     -
  * @note       Called by tspi driver
  */
/*--------------------------------------------------*/
static void receive_handler(uint32_t id, TXZ_Result result, tspi_receive_t *p_info)
{
    if ((receive.state == RECV_RUN) && (p_info != MAIN_NULL))
    {
        if (result == TXZ_SUCCESS)
        {
            uint32_t i;
            uint32_t num    = p_info->rx8.num;
            /*>>> Critical information. */
            volatile uint32_t length = receive.length;
            /*<<< Critical information. */

            if (length < TSPI_IO_RECEIVE_MAX) 
            {
                if ((length + num) > TSPI_IO_RECEIVE_MAX)
                {
                    num = (uint32_t)(TSPI_IO_RECEIVE_MAX - length);
                }
                for (i=0; i<num; i++)
                {
                    receive.buff[receive.wp + i] = *(p_info->rx8.p_data + i);
                    receive.wp++;
                }
                /*>>> Critical information. */
                receive.length += num;
                receive.state = RECV_SUCCESS;
                /*<<< Critical information. */
            }
            else
            {
                receive.state = RECV_FAILURE;
            }
        }
        else
        {
            receive.state = RECV_FAILURE;
        }
    }else{
        tspi_io_error();
    }
}
/**
 *  @}
 */ /* End of group TSPI_IO_PRIVATE */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/*--------------------------------------------------*/
/** 
  * @brief  Initialize
  * @param  -
  * @retval -
  * @pre    -
  * @note   -
  */
/*--------------------------------------------------*/
void tspi_io_initialize(void)
{
    /*------------------------------*/
    /*  Clear RAM                   */
    /*------------------------------*/
    send.state      = SEND_IDLE;
    send.wp         = 0;
    send.rp         = 0;
    receive.state   = RECV_RUN;     /* Auto run. */
    receive.wp      = 0;
    receive.rp      = 0;
    receive.length  = 0;
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize.
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
   TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = USER_ID_UART;
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = (uint32_t)CFG_UART_BAUDRATE;
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result  = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* CRC                              */
    /*----------------------*/
    /* CRCTYP Register SET. */
    {
        crc_crctyp_t param;
        param.rrfl = REG_CRC_RRFL_INVERSION_OFF;
        param.drfl = REG_CRC_DRFL_INVERSION_OFF;
        param.binv = REG_CRC_BINV_INVERSION_OFF;
#ifdef CRC16
        param.cfmt = REG_CRC_CFMT_CRC16;
#endif
#ifdef CRC32
        param.cfmt = REG_CRC_CFMT_CRC32;
#endif
#ifdef CheckSum
        param.cfmt = REG_CRC_CFMT_CRC16;
#endif
        param.dbit = REG_CRC_DBIT_32bit;
        
        REG_CRCTYP_set(TSB_CRC, param);
#ifdef CRC16
        REG_CRCCLC_set(TSB_CRC, REG_CRCCLC_DEFAULT_VAL_16);
#endif
#ifdef CRC32
        REG_CRCCLC_set(TSB_CRC, REG_CRCCLC_DEFAULT_VAL_32);
#endif
#ifdef CheckSum
        REG_CRCCLC_set(TSB_CRC, REG_CRCCLC_DEFAULT_VAL_16);
#endif
    }
    /*----------------------*/
    /* TSPI             */
    /*----------------------*/
    /* Tx TSPI channel. */
    {
        instance.tspi_tx_obj.init.id  = USER_ID_TSPI_TX;
        switch (bsp_get_tspi_tx_ch(BSP_SFLASH_2))
        {
        case 0:
            instance.tspi_tx_obj.p_instance  = TSB_TSPI0;
            break;
        case 1:
            instance.tspi_tx_obj.p_instance  = TSB_TSPI1;
            break;
        case 2:
            instance.tspi_tx_obj.p_instance  = TSB_TSPI2;
            break;
        case 3:
            instance.tspi_tx_obj.p_instance  = TSB_TSPI3;
            break;
        default:
            instance.tspi_tx_obj.p_instance  = MAIN_NULL;
            break;
        }
        /* Set Initial parameter of TSPIxCR1 */
        instance.tspi_tx_obj.init.cnt1.inf = TSPI_INF_DISABLE;
        instance.tspi_tx_obj.init.cnt1.trgen = TSPI_TRGEN_DISABLE;
        instance.tspi_tx_obj.init.cnt1.trxe = TSPI_DISABLE;
        instance.tspi_tx_obj.init.cnt1.tspims = TSPI_SPI_MODE;
        instance.tspi_tx_obj.init.cnt1.mstr = TSPI_MASTER_OPERATION;
        instance.tspi_tx_obj.init.cnt1.tmmd = TSPI_TX_ONLY;
        instance.tspi_tx_obj.init.cnt1.cssel = TSPI_TSPIxCS0_ENABLE;
        instance.tspi_tx_obj.init.cnt1.fc = TSPI_TRANS_RANGE_CONTINUE;
        /* Set Initial parameter of TSPIxCR2 */
        instance.tspi_tx_obj.init.cnt2.tidle = TSPI_TIDLE_HI;
        instance.tspi_tx_obj.init.cnt2.txdemp = TSPI_TXDEMP_HI;
        instance.tspi_tx_obj.init.cnt2.rxdly = TSPI_RXDLY_FSYS_FSCK_16;  /* fsys(120MHz)/fsck(234375Hz)=512 > 16  */
        instance.tspi_tx_obj.init.cnt2.til = TSPI_TX_FILL_LEVEL_7;
        instance.tspi_tx_obj.init.cnt2.ril = TSPI_RX_FILL_LEVEL_7;
        instance.tspi_tx_obj.init.cnt2.inttxwe = TSPI_TX_INT_ENABLE;
        instance.tspi_tx_obj.init.cnt2.intrxwe = TSPI_RX_INT_DISABLE;
        instance.tspi_tx_obj.init.cnt2.inttxfe = TSPI_TX_FIFO_INT_DISABLE;
        instance.tspi_tx_obj.init.cnt2.intrxfe = TSPI_RX_FIFO_INT_DISABLE;
        instance.tspi_tx_obj.init.cnt2.interr = TSPI_ERR_INT_DISABLE;
        instance.tspi_tx_obj.init.cnt2.dmate = TSPI_TX_DMA_INT_DISABLE;
        instance.tspi_tx_obj.init.cnt2.dmare = TSPI_RX_DMA_INT_DISABLE;
        /* Set Initial parameter of TSPIxBR */
        instance.tspi_tx_obj.init.brd.brck = TSPI_BR_CLOCK_8;      /* T0(120MHz)/TSPI_BR_CLOCK_8(16) > Tx(7.5MHz)  */
        instance.tspi_tx_obj.init.brd.brs = TSPI_BR_DIVIDER_16;    /* Tx(7.5MHz)/TSPI_BR_DIVIDER_16(16) = 468750Hz , 468750Hz/2 = fsck(234375Hz)  */
        /* Set Initial parameter of TSPIxFMTR0 */
        instance.tspi_tx_obj.init.fmr0.dir = TSPI_DATA_DIRECTION_MSB;
        instance.tspi_tx_obj.init.fmr0.fl = TSPI_DATA_LENGTH_8;
        instance.tspi_tx_obj.init.fmr0.fint = TSPI_INTERVAL_TIME_0;
        instance.tspi_tx_obj.init.fmr0.cs3pol = TSPI_TSPIxCS3_NEGATIVE;
        instance.tspi_tx_obj.init.fmr0.cs2pol = TSPI_TSPIxCS2_NEGATIVE;
        instance.tspi_tx_obj.init.fmr0.cs1pol = TSPI_TSPIxCS1_NEGATIVE;
        instance.tspi_tx_obj.init.fmr0.cs0pol = TSPI_TSPIxCS0_NEGATIVE;
        instance.tspi_tx_obj.init.fmr0.ckpha = TSPI_SERIAL_CK_2ND_EDGE;
        instance.tspi_tx_obj.init.fmr0.ckpol = TSPI_SERIAL_CK_IDLE_HI;
        instance.tspi_tx_obj.init.fmr0.csint = TSPI_MIN_IDLE_TIME_1;
        instance.tspi_tx_obj.init.fmr0.cssckdl = TSPI_SERIAL_CK_DELAY_1;
        instance.tspi_tx_obj.init.fmr0.sckcsdl = TSPI_NEGATE_1;
        /* Set Initial parameter of TSPIxFMTR1 */
        instance.tspi_tx_obj.init.fmr1.vpe = TSPI_PARITY_DISABLE;
        instance.tspi_tx_obj.init.fmr1.vpm = TSPI_PARITY_BIT_ODD;
        /* Set Initial parameter of TSPISECTCR0 */
        instance.tspi_tx_obj.init.sectcr0.sect = TSPI_SECTCR0_SECT_FRAME_MODE;
        /* Set Initial parameter of TSPISECTCR1 */
        instance.tspi_tx_obj.init.sectcr1.sectl0 = TSPI_SECTCR1_SECTL0_1;
        instance.tspi_tx_obj.init.sectcr1.sectl1 = TSPI_SECTCR1_SECTL1_1;
        instance.tspi_tx_obj.init.sectcr1.sectl2 = TSPI_SECTCR1_SECTL2_0;
        instance.tspi_tx_obj.init.sectcr1.sectl3 = TSPI_SECTCR1_SECTL3_0;
        instance.tspi_tx_obj.transmit.handler = transmit_handler;
        if ( instance.tspi_tx_obj.p_instance != MAIN_NULL)
        {
            if (tspi_init(&instance.tspi_tx_obj) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /* Rx TSPI channel. */
    {
        instance.tspi_rx_obj.init.id  = USER_ID_TSPI_RX;
        switch (bsp_get_tspi_rx_ch(BSP_SFLASH_1))
        {
        case 0:
            instance.tspi_rx_obj.p_instance  = TSB_TSPI0;
            break;
        case 1:
            instance.tspi_rx_obj.p_instance  = TSB_TSPI1;
            break;
        case 2:
            instance.tspi_rx_obj.p_instance  = TSB_TSPI2;
            break;
        case 3:
            instance.tspi_rx_obj.p_instance  = TSB_TSPI3;
            break;
        default:
            instance.tspi_rx_obj.p_instance  = MAIN_NULL;
            break;
        }
        /* Set Initial parameter of TSPIxCR1 */
        instance.tspi_rx_obj.init.cnt1.inf = TSPI_INF_DISABLE;
        instance.tspi_tx_obj.init.cnt1.trgen = TSPI_TRGEN_DISABLE;
        instance.tspi_rx_obj.init.cnt1.trxe = TSPI_DISABLE;
        instance.tspi_rx_obj.init.cnt1.tspims = TSPI_SPI_MODE;
        instance.tspi_rx_obj.init.cnt1.mstr = TSPI_SLAVE_OPERATION;
        instance.tspi_rx_obj.init.cnt1.tmmd = TSPI_RX_ONLY;
        instance.tspi_rx_obj.init.cnt1.cssel = TSPI_TSPIxCS0_ENABLE;
        instance.tspi_rx_obj.init.cnt1.fc = TSPI_TRANS_RANGE_CONTINUE;
        /* Set Initial parameter of TSPIxCR2 */
        instance.tspi_rx_obj.init.cnt2.tidle = TSPI_TIDLE_HI;
        instance.tspi_rx_obj.init.cnt2.txdemp = TSPI_TXDEMP_HI;
        instance.tspi_rx_obj.init.cnt2.til = TSPI_TX_FILL_LEVEL_7;
        instance.tspi_rx_obj.init.cnt2.ril = TSPI_RX_FILL_LEVEL_7;
        instance.tspi_rx_obj.init.cnt2.inttxwe = TSPI_TX_INT_DISABLE;
        instance.tspi_rx_obj.init.cnt2.intrxwe = TSPI_RX_INT_DISABLE;
        instance.tspi_rx_obj.init.cnt2.inttxfe = TSPI_TX_FIFO_INT_DISABLE;
        instance.tspi_rx_obj.init.cnt2.intrxfe = TSPI_RX_FIFO_INT_DISABLE;
        instance.tspi_rx_obj.init.cnt2.interr = TSPI_ERR_INT_DISABLE;
        instance.tspi_rx_obj.init.cnt2.dmate = TSPI_TX_DMA_INT_DISABLE;
        instance.tspi_rx_obj.init.cnt2.dmare = TSPI_RX_DMA_INT_DISABLE;
        /* Set Initial parameter of TSPIxBR */
        instance.tspi_rx_obj.init.brd.brck = TSPI_BR_CLOCK_8;  /* T0(120MHz)/TSPI_BR_CLOCK_8(16) > Tx(7.5MHz) */
        instance.tspi_rx_obj.init.brd.brs = TSPI_BR_DIVIDER_16; /* Tx(7.5MHz)/TSPI_BR_DIVIDER_16(16) = 468750Hz , 468750Hz/2 = fsck(234375Hz) */
        /* Set Initial parameter of TSPIxFMTR0 */
        instance.tspi_rx_obj.init.fmr0.dir = TSPI_DATA_DIRECTION_MSB;
        instance.tspi_rx_obj.init.fmr0.fl = TSPI_DATA_LENGTH_8;
        instance.tspi_rx_obj.init.fmr0.fint = TSPI_INTERVAL_TIME_0;
        instance.tspi_rx_obj.init.fmr0.cs3pol = TSPI_TSPIxCS3_NEGATIVE;
        instance.tspi_rx_obj.init.fmr0.cs2pol = TSPI_TSPIxCS2_NEGATIVE;
        instance.tspi_rx_obj.init.fmr0.cs1pol = TSPI_TSPIxCS1_NEGATIVE;
        instance.tspi_rx_obj.init.fmr0.cs0pol = TSPI_TSPIxCS0_NEGATIVE;
        instance.tspi_rx_obj.init.fmr0.ckpha = TSPI_SERIAL_CK_2ND_EDGE;
        instance.tspi_rx_obj.init.fmr0.ckpol = TSPI_SERIAL_CK_IDLE_HI;
        instance.tspi_rx_obj.init.fmr0.csint = TSPI_MIN_IDLE_TIME_1;
        instance.tspi_rx_obj.init.fmr0.cssckdl = TSPI_SERIAL_CK_DELAY_1;
        instance.tspi_rx_obj.init.fmr0.sckcsdl = TSPI_NEGATE_1;
        /* Set Initial parameter of TSPIxFMTR1 */
        instance.tspi_rx_obj.init.fmr1.vpe = TSPI_PARITY_DISABLE;
        instance.tspi_rx_obj.init.fmr1.vpm = TSPI_PARITY_BIT_ODD;
        /* Set Initial parameter of TSPISECTCR0 */
        instance.tspi_rx_obj.init.sectcr0.sect = TSPI_SECTCR0_SECT_FRAME_MODE;
        /* Set Initial parameter of TSPISECTCR1 */
        instance.tspi_rx_obj.init.sectcr1.sectl0 = TSPI_SECTCR1_SECTL0_1;
        instance.tspi_rx_obj.init.sectcr1.sectl1 = TSPI_SECTCR1_SECTL1_1;
        instance.tspi_rx_obj.init.sectcr1.sectl2 = TSPI_SECTCR1_SECTL2_0;
        instance.tspi_rx_obj.init.sectcr1.sectl3 = TSPI_SECTCR1_SECTL3_0;
        instance.tspi_rx_obj.receive.handler = receive_handler;
        if ( instance.tspi_rx_obj.p_instance != MAIN_NULL)
        {
            if (tspi_init(&instance.tspi_rx_obj) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    return (result);
}
/**
 *  @}
 */ /* End of group MAIN_PRIVATE */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    int ret;
    uint16_t send_length ;
    /*----------------------*/
    /* Set Processing Mode  */
    /*----------------------*/
    proc = PROC_OUTPUT_MSG;
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /* BSP initialize       */
    bsp_initialize();
    /* Valiable initialization */
    variable_initialize();
    /* Application initialization */
    if (application_initialize() != TXZ_SUCCESS)
    {
    }
    tspi_io_initialize();
    
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(1)
    {
        switch(proc)
        {
        /*--- Output "First Message" ---*/
        case PROC_OUTPUT_MSG:
            {
                /* Clear input data.     */
                clear_data();
                /* Output first message. */
                if (printf("command > ") != -1 )
                {
                    proc = PROC_INPUT;
                }
                else
                {
                    proc = PROC_ERROR;
                }
            }
            break;
        /*--- Wait to Input ---*/
        case PROC_INPUT:
            {   
                int data;
                
                data = getchar();
                if ((data != -1) && (input.wp < UART_IO_RECEIVE_MAX))
                {
                    input.data[input.wp] = (char)data;
                    input.wp++;
                    switch ((char)data)
                    {
                    /*--- LF: line feed ---*/
                    case '\n':
                        proc = PROC_COMMAND_PARSE;
                        break;
                    /*--- default ---*/
                    default:
                        /* no process */
                        break;
                    }
                }
                else
                {
                    proc = PROC_OUTPUT_ERR;
                }
            }
            break;
        /*--- "Command Parse" ---*/
        case PROC_COMMAND_PARSE:
            {
                if ((strncmp((const char *)&input.data[0], "send", 4)) == 0)
                {
                    /* CRC  Input data Setting */
#ifndef CheckSum
                    uint32_t val[2]  = {0,0};
#endif
                    
                    uint32_t i, j ;
                    uint8_t  data_length ;
                    uint32_t bundle32 ;
                    uint16_t bundle16 ;
#ifdef CheckSum
                    uint8_t  checksum ;
#endif
                    crc_crctyp_t param;
                    
                    param.rrfl = REG_CRC_RRFL_INVERSION_OFF;
                    param.drfl = REG_CRC_DRFL_INVERSION_OFF;
                    param.binv = REG_CRC_BINV_INVERSION_OFF;
#ifdef CRC16
                    param.cfmt = REG_CRC_CFMT_CRC16;
#endif
#ifdef CRC32
                    param.cfmt = REG_CRC_CFMT_CRC32;
#endif
#ifdef CheckSum
                    param.cfmt = REG_CRC_CFMT_CRC16;
#endif
                    param.dbit = REG_CRC_DBIT_32bit;
                    
                    REG_CRCTYP_set(TSB_CRC, param);
#ifdef CRC16
                    REG_CRCCLC_set(TSB_CRC, REG_CRCCLC_DEFAULT_VAL_16);
#endif
#ifdef CRC32
                    REG_CRCCLC_set(TSB_CRC, REG_CRCCLC_DEFAULT_VAL_32);
#endif
#ifdef CheckSum
                    REG_CRCCLC_set(TSB_CRC, REG_CRCCLC_DEFAULT_VAL_16);
#endif
                    
                    i = 0 ;
                    while(write_data[i] != 0x00) {
                        t_data[i+1] = write_data[i] ;
                        i++ ;
                    }
                    data_length = i ;
                    t_data[0] = data_length ;

                    i = 0 ;
                    while ( i < data_length + 1 ) {
                        if( data_length + 1 - i >= 4 )
                        {
                            bundle32 = 0x00000000 ;
                            for(j = 0; j<4; j++){
                                bundle32 = ((bundle32 << 8U) |  (uint32_t)(t_data[i+j]));
                            }
                            param.dbit = REG_CRC_DBIT_32bit;
                            REG_CRCTYP_set(TSB_CRC, param);
                            REG_CRCDIN_set(TSB_CRC, bundle32);
                            i += 4 ;
                        } 
                        else if( data_length + 1 - i >= 2 )
                        {
                            bundle16 = 0x0000 ;
                            for(j = 0; j<2; j++){
                                bundle16 = ((bundle16 << 8U) |  (uint32_t)(t_data[i+j]));
                            }
                            param.dbit = REG_CRC_DBIT_16bit;
                            REG_CRCTYP_set(TSB_CRC, param);
                            REG_CRCDIN_set(TSB_CRC, bundle16);
                            i += 2 ;
                        }
                        else if( data_length + 1 - i == 1 )
                        {
                            param.dbit = REG_CRC_DBIT_8bit;
                            REG_CRCTYP_set(TSB_CRC, param);
                            REG_CRCDIN_set(TSB_CRC, t_data[i]);
                            i ++ ;
                        }

                    }
            



#ifndef CheckSum
                    val[0] = REG_CRCCLC_get(TSB_CRC);
                    val[1] = REG_CRCCLC_get(TSB_CRC);
#endif

#ifdef CRC16
                    t_data[i++] = (uint8_t)(val[1] >> 8);
                    t_data[i++] = (uint8_t)(val[1]  & 0x000000FF);
#endif
#ifdef CRC32
                    t_data[i++] = (uint8_t)(val[1] >> 24);
                    t_data[i++] = (uint8_t)(val[1] >> 16);
                    t_data[i++] = (uint8_t)(val[1] >> 8);
                    t_data[i++] = (uint8_t)(val[1]  & 0x000000FF);
#endif
#ifdef CheckSum
                    checksum = checksum_calc ( t_data, (uint16_t)data_length +1 ) ;
                    t_data[i++] = checksum ;
#endif
                    send_length = i ;


                    proc = PROC_WRITE_EXEC;
                    
                    break;
                }
                else
                {
                    proc = PROC_OUTPUT_ERR;
                }
            }
            break; 

            case PROC_WRITE_EXEC:
                {
                    uint16_t remain = send_length ;
                    uint16_t send_count = 0 ;
                    while (remain > 0) {
                        uint8_t send_num = 5 ;
                        if(remain < send_num) send_num = remain ;


                        if(send.state == SEND_IDLE){
                            /* Transfer TRXE Disable  */
                            instance.tspi_tx_obj.p_instance->CR1 &= TSPI_TRXE_DISABLE_MASK;
                            /* Receive TRXE Enable  */
                            instance.tspi_rx_obj.p_instance->CR1 |= TSPI_TRXE_ENABLE;
                            /* NVIC Enable (TX) */
                            NVIC_EnableIRQ(bsp_get_tspi_tx_nvic(BSP_SFLASH_2));
                            /* Send Data Set */
                            t_info.tx8.p_data = t_data + send_count;
                            t_info.tx8.num = send_num;
                            /* Transfer Start */
                            ret = tspi_master_transfer(&instance.tspi_tx_obj, &t_info);
                            send.state = SEND_RUN;
                            proc = PROC_WRITE_EXEC2;                    
                        }
                        else if(send.state == SEND_FAILURE){
                            send.state = SEND_IDLE;
                            proc = PROC_ERROR;
                        }
                        else if(send.state == SEND_SUCCESS) {
                            /* NVIC Disable (TX) */
                            NVIC_DisableIRQ(bsp_get_tspi_tx_nvic(BSP_SFLASH_2));
                            if(ret == TXZ_ERROR)
                            {
                                proc = PROC_ERROR;
                            }
                            wait(100);
                            instance.tspi_tx_obj.p_instance->CR1 &= TSPI_TRXE_DISABLE_MASK;
                            instance.tspi_tx_obj.p_instance->SR |=  TSPI_TX_DONE_CLR;
                            /* Buffer Clear */
                            if((instance.tspi_tx_obj.p_instance->SR&0x0000000F) > 0){
                                instance.tspi_tx_obj.p_instance->CR3 = 0x3;
                            }
                            wait(3000);
                            /* Receive Buffer Set */
                            r_info.rx8.num = send_num;
                            r_info.rx8.p_data = read_data + send_count;
                            /* Receive Start */
                            ret = tspi_master_read(&instance.tspi_rx_obj, &r_info, 5000);
                            send.state = SEND_IDLE;
                            proc = PROC_WRITE_EXEC;

                            send_count += send_num ;
                            remain -= send_num;
                        }
                    }
                    send.state = SEND_IDLE;
                    proc = PROC_CRC_CHECK;

                }
                break;




      /*--- CRC Result & Receive Check ---*/
            case PROC_CRC_CHECK:
                {
                    crc_crctyp_t param;
                    uint32_t val[2]   = {0,0};
                    uint32_t data     = 0;
            uint16_t data_length ;
#ifdef CheckSum
            uint8_t  checksum ;
#endif
            uint32_t code ;

            data_length = read_data[0] ;

                    /*------------------*/
                    /* CRC Result Check */
                    /*------------------*/
                    /* CRC data width Setting for 16bit */
                    param.rrfl = REG_CRC_RRFL_INVERSION_OFF;
                    param.drfl = REG_CRC_DRFL_INVERSION_OFF;
                    param.binv = REG_CRC_BINV_INVERSION_OFF;
#ifdef CRC16
                    param.cfmt = REG_CRC_CFMT_CRC16;
                    param.dbit = REG_CRC_DBIT_16bit;
#endif
#ifdef CRC32
                    param.cfmt = REG_CRC_CFMT_CRC32;
                    param.dbit = REG_CRC_DBIT_32bit;
#endif
#ifdef CheckSum
                    param.cfmt = REG_CRC_CFMT_CRC16;
                    param.dbit = REG_CRC_DBIT_16bit;
#endif
                    REG_CRCTYP_set(TSB_CRC, param);
                    /* Receive CRC Result Data Set for Input Data */
#ifdef CRC16
                    data = (uint16_t)read_data[data_length+1];
                    data = (uint16_t)((data << 8) | read_data[data_length+2]);
                    code = data ;
#endif
#ifdef CRC32
                    data = (uint32_t)read_data[data_length+1];
                    data = (uint32_t)((data << 8) | read_data[data_length+2]);
                    data = (uint32_t)((data << 8) | read_data[data_length+3]);
                    data = (uint32_t)((data << 8) | read_data[data_length+4]);
                    code = data ;
#endif
                    REG_CRCDIN_set(TSB_CRC, data);
                    /* Get the CRC Calculation Result */
                    val[0] = REG_CRCCLC_get(TSB_CRC);
                    val[1] = REG_CRCCLC_get(TSB_CRC);
#ifdef CheckSum
                    checksum = checksum_calc ( read_data, read_data[0]+1 ) ;
                    code = read_data[data_length+1] ;
                    if(checksum == read_data[data_length+1]) val[1] = 0 ;
#endif
                    if(val[1] == 0)
                    {
#ifdef CRC16
                        read_data[data_length+1] = 0;
                        read_data[data_length+2] = 0;
#endif
#ifdef CRC32
                        read_data[data_length+1] = 0;
                        read_data[data_length+2] = 0;
                        read_data[data_length+3] = 0;
                        read_data[data_length+4] = 0;
#endif
#ifdef CheckSum
                        read_data[data_length+1] = 0;
#endif
                        if (printf("Data = ") != -1)
                        {
                            proc = PROC_ERROR;
                        }
                        if (printf("%s\n", (const char*)&read_data[1]) < 0)
                        {
                            proc = PROC_ERROR;
                        }
#ifdef CRC16
                        if (printf("Code = 0x%04x\n", code ) < 0)
#endif
#ifdef CRC32
                        if (printf("Code = 0x%08x\n", code ) < 0)
#endif
#ifdef CheckSum
                        if (printf("Code = 0x%02x\n", code ) < 0)
#endif
                        {
                            proc = PROC_ERROR;
                        }
                        if (printf("Check OK\n") < 0)
                        {
                            proc = PROC_ERROR;
                        }
                        if (printf("\n") < 0)
                        {
                            proc = PROC_ERROR;
                        }else{
                            send.state = SEND_IDLE;
                            proc = PROC_OUTPUT_MSG;
                        }
                    }else{
                        if (printf("Check Error\n") < 0)
                        {
                            proc = PROC_ERROR;
                        }else{
                            send.state = SEND_IDLE;
                            proc = PROC_OUTPUT_MSG;
                        }
                    }
               }
               break;
      /*--- Output "Input Error Message" ---*/
        case PROC_OUTPUT_ERR:
            {
                if (printf("\n Command Error !! \n")  < 0 )
                {
                    proc = PROC_ERROR;
                }
                else
                {
                    proc = PROC_OUTPUT_MSG;
                }
            }
            break;
        /*--- Error ---*/
        case PROC_ERROR:
            /* System Error. Please Debug. */
#ifdef DEBUG
            assert_failed(MAIN_NULL, 0);
#endif
            break;
        /*--- Default ---*/
        default:
            proc = PROC_ERROR;
            break;
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  SPI flash reception IRQ.
  * @param  sflash :SPI Flash.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_sflash_rx(BSPSFlash sflash)
{
    switch(sflash){
        case BSP_SFLASH_1:
            tspi_irq_handler_transmit(&instance.tspi_rx_obj);
            break;
        case BSP_SFLASH_2:
            tspi_irq_handler_transmit(&instance.tspi_rx_obj);
            break;
        case BSP_SFLASH_3:
            tspi_irq_handler_transmit(&instance.tspi_rx_obj);
            break;
        default:
            break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  SPI flash transmit IRQ.
  * @param  sflash :SPI Flash.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_sflash_tx(BSPSFlash sflash)
{
    switch(sflash){
        case BSP_SFLASH_1:
            tspi_irq_handler_transmit(&instance.tspi_tx_obj);
            break;
        case BSP_SFLASH_2:
            tspi_irq_handler_transmit(&instance.tspi_tx_obj);
            break;
        case BSP_SFLASH_3:
            tspi_irq_handler_transmit(&instance.tspi_tx_obj);
            break;
        default:
            break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  SPI flash error IRQ.
  * @param  sflash :SPI Flash.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_sflash_err(BSPSFlash sflash)
{
    switch(sflash){
        case BSP_SFLASH_1:
            tspi_error_irq_handler(&instance.tspi_rx_obj);
            break;
        case BSP_SFLASH_2:
            tspi_error_irq_handler(&instance.tspi_tx_obj);
            break;
        case BSP_SFLASH_3:
            tspi_error_irq_handler(&instance.tspi_tx_obj);
            break;
        default:
            break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

#ifdef CheckSum
/*--------------------------------------------------*/
/** 
  * @brief  checksum_calc
  * @param  data[], length .
  * @return checksum 1byte.
  * @retval -
  * @note   
  */
/*--------------------------------------------------*/
uint8_t checksum_calc(uint8_t *data, uint16_t length)
{
    uint8_t sum = 0 ;
    uint16_t i ;
    for( i=0; i<length; i++)
    {
        sum += data[i];
    }
    return sum ;
}
#endif



#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __NOP();
    }
}
#endif

/**
 *  @} 
 */ /* End of group TSPI_SPI */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
