/** 
 *******************************************************************************
 * @file    main.c
 * @brief   UART Sample Application.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <stdio.h>
#include "bsp_com.h"
#include "bsp.h"
#include "bsp_uart_io.h"
#include "bsp_timer.h"
/**
 *  @addtogroup Example
 *  @{
 */

/** 
 *  @defgroup UART_Sample_Appli UART Sample Appli
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup UART_Private_macro UART ECHO Private Macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group UART_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup UART_Private_define UART ECHO Private Define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group UART_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup UART_Private_define UART ECHO Private Define
 *  @{
 */

#define MAIN_NULL               ((void *)0) /*!< Null Pointer */

/** 
 *  @}
 */ /* End of group UART_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup UART_Private_define UART ECHO Private Define
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief  Processing mode.
*/
/*----------------------------------*/
typedef enum
{
    PROC_IDLE,              /*!< Idle.                        */
    PROC_START,             /*!< Start.                       */
    PROC_STOP,              /*!< Stop.                        */
    PROC_OUTPUT_MSG,        /*!< Output first message.        */
    PROC_INPUT,             /*!< Wait to input.               */
    PROC_OUTPUT_ECHO,       /*!< Output Echo back data.       */
    PROC_OUTPUT_ERR,        /*!< Output input error message.  */
    PROC_ERROR              /*!< Error.                       */
} Processing;
/** 
 *  @}
 */ /* End of group UART_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup UART_Private_typedef UART ECHO Private Typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief Instance Information.
*/
/*----------------------------------*/
typedef struct
{
    t32a_t      t32a;              /*!< Driver      :T32A.     */
    timer_t     timer;             /*!< Application :Timer (1ms timer). */
} instance_t;

/*----------------------------------*/
/** 
 * @brief  Input data information.
*/
/*----------------------------------*/
typedef struct
{
    char    data[(UART_IO_RECEIVE_MAX+2)];  /*!< Input data(+ for NULL 1byte). */
    uint8_t wp;                             /*!< Write pointer.                */
} input_t;
/**
 *  @}
 */ /* End of group UART_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup UART_Private_variables UART ECHO Private Variables
 *  @{
 */
static instance_t instance;  /*!< Instance.               */
static Processing proc;      /*!< Processing mode.        */
static input_t    input;     /*!< Input data information. */
/**
 *  @}
 */ /* End of group UART_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup UART_Private_fuctions UART ECHO Private Fuctions
 *  @{
 */
static void variable_initialize(void);
static TXZ_Result driver_initialize(void);
static TXZ_Result driver_finalize(void);
static TXZ_Result application_initialize(void);
static TXZ_Result application_finalize(void);
static void timer_interval_handler(uint32_t id);
static void clear_input_data(void);
/*--------------------------------------------------*/
/** 
  * @brief  Variable Initialize
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void variable_initialize(void)
{
    /* Instance. */
    instance.t32a.p_instance    = MAIN_NULL;
    instance.timer.init.p_t32a  = MAIN_NULL;
    /* Processing mode. */
    proc = PROC_START;
    /* Input data information. */
    clear_input_data();
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Initialize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver initialize has done within BSP. */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Construct  ---*/
        /* Register Allocation */
        switch (bsp_get_timer_ch(BSP_TIMER_1MS))
        {
        case 0:
            p_t32a->p_instance = TSB_T32A0;
            break;
        case 1:
            p_t32a->p_instance = TSB_T32A1;
            break;
        case 2:
            p_t32a->p_instance = TSB_T32A2;
            break;
        case 3:
            p_t32a->p_instance = TSB_T32A3;
            break;
        case 4:
            p_t32a->p_instance = TSB_T32A4;
            break;
        case 5:
            p_t32a->p_instance = TSB_T32A5;
            break;
        default:
            p_t32a->p_instance = MAIN_NULL;
            break;
        }
        /*--- Initialize ---*/
        p_t32a->init_mode.mode.halt = T32A_DBG_HALT_STOP;
        p_t32a->init_mode.mode.mode = T32A_MODE_16;
        if (p_t32a->p_instance != MAIN_NULL)
        {
            if (t32a_mode_init(p_t32a) != TXZ_SUCCESS)
            {
                result = TXZ_ERROR;
            }
        }
        else
        {
            result = TXZ_ERROR;
        }
    }
    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver initialize will process within USB UART. */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Driver Finalize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result driver_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Uart                 */
    /*----------------------*/
    /* Driver finalize has done within USB UART.    */
    /*----------------------*/
    /* T32A                 */
    /*----------------------*/
    {
        t32a_t *p_t32a = &instance.t32a;

        /*--- Finalize ---*/
        /* no processing */
        /*--- Destruct ---*/
        /* Register Release */
        p_t32a->p_instance = MAIN_NULL;
    }
    /*----------------------*/
    /* GPIO                 */
    /*----------------------*/
    /* Driver finalize will process within BSP.     */

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Initialize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_initialize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    {
        timer_t *p_timer = &instance.timer;

        p_timer->init.id       = (uint32_t)&instance.timer;
        p_timer->init.p_t32a   = &instance.t32a;
        p_timer->init.type     = bsp_get_timer_type(BSP_TIMER_1MS);
        p_timer->init.irq      = bsp_get_timer_nvic(BSP_TIMER_1MS);
        p_timer->init.interval = 1000;
        p_timer->init.handler  = timer_interval_handler;
        timer_initialize(p_timer);
    }
    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    /* Tx & Rx use same UART channel. */
    {
        uart_io_initial_setting_t param;

        param.id         = 0;   /* User ID :any value. */
        switch (bsp_get_usb_uart_tx_ch(BSP_USB_UART_0))
        {
        case 0:
            param.p_reg  = TSB_UART0;
            break;
        case 1:
            param.p_reg  = TSB_UART1;
            break;
        case 2:
            param.p_reg  = TSB_UART2;
            break;
        default:
            param.p_reg  = MAIN_NULL;
            break;
        }
        param.boudrate   = bsp_get_usb_uart_tx_baudrate(BSP_USB_UART_0);
        param.irq.tx     = bsp_get_usb_uart_tx_nvic(BSP_USB_UART_0);
        param.irq.rx     = bsp_get_usb_uart_rx_nvic(BSP_USB_UART_0);
        param.irq.err    = bsp_get_usb_uart_err_nvic(BSP_USB_UART_0);
        if (param.p_reg != MAIN_NULL)
        {
            result  = uart_io_initialize(&param);
        }
        else
        {
            result = TXZ_ERROR;
        }
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Application Finalize
  * @param  -
  * @return Result.
  * @retval TXZ_SUCCESS :Success
  * @retval TXZ_ERROR   :Failure
  * @note   -
  */
/*--------------------------------------------------*/
static TXZ_Result application_finalize(void)
{
    TXZ_Result result = TXZ_SUCCESS;

    /*----------------------*/
    /* Usb Uart             */
    /*----------------------*/
    {
        uart_io_finalize();
    }    
    /*----------------------*/
    /* Timer                */
    /*----------------------*/
    /* 1ms timer */
    if (instance.timer.init.p_t32a != MAIN_NULL)
    {
        timer_finalize(&instance.timer);
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Interval Time Handler.
  * @param  id        :User ID
  * @return -
  * @retval -
  * @note   for 1ms Timer.
  * @note   Call by Timer.
  */
/*--------------------------------------------------*/
static void timer_interval_handler(uint32_t id)
{
    if (id == (uint32_t)&instance.timer)
    {
        /* Increment Tick. Tick is used to driver's Timeout. */
        hal_inc_tick();
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Clear input data.
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void clear_input_data(void)
{
    uint8_t i;

    for (i=0; i<(UART_IO_RECEIVE_MAX+2); i++)
    {
        input.data[i] = 0;
    }
    input.wp = 0;
}
/**
 *  @}
 */ /* End of group UART_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup UART_Exported_functions UART ECHO Exported Functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Main Function.
  * @param  -
  * @return -
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
int main(void)
{
    /*----------------------*/
    /* BSP initialize       */
    /*----------------------*/
    bsp_initialize();
    /*----------------------*/
    /* Initialize           */
    /*----------------------*/
    /*--- variable    ---*/
    variable_initialize();
    /*--- driver      ---*/
    if (driver_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    /*--- application ---*/
    if (application_initialize() != TXZ_SUCCESS)
    {
        proc = PROC_ERROR;
    }
    /*----------------------*/
    /* Main Processing      */
    /*----------------------*/
    while(proc != PROC_ERROR)
    {
        switch(proc)
        {
        /*--- Start ---*/
        case PROC_START:
            {
                timer_start(&instance.timer);
                proc = PROC_OUTPUT_MSG;
            }
            break;
        /*--- Stop ---*/
        case PROC_STOP:
            {
                timer_stop(&instance.timer);
                proc = PROC_ERROR;
            }
            break;
        /*--- Output "First Message" ---*/
        case PROC_OUTPUT_MSG:
            {
                /* Clear input data.     */
                clear_input_data();
                /* Output first message. */
                if (printf("Input = ") != -1 )
                {
                    proc = PROC_INPUT;
                }
                else
                {
                    proc = PROC_STOP;
                }
            }
            break;
        /*--- Wait to Input ---*/
        case PROC_INPUT:
            {
                int data;

                data = getchar();
                if ((data != -1) && (input.wp < UART_IO_RECEIVE_MAX+1))
                {
                    input.data[input.wp] = (char)data;
                    input.wp++;
                    switch ((char)data)
                    {
                    /*--- LF: line feed ---*/
                    case '\n':
                        proc = PROC_OUTPUT_ECHO;
                        break;
                    /*--- default ---*/
                    default:
                        /* no process */
                        break;
                    }
                }
                else
                {
                    proc = PROC_OUTPUT_ERR;
                }
            }
            break;
        /*--- Output "Echo Back Data" ---*/
        case PROC_OUTPUT_ECHO:
            {
                if (printf(" Echo = %s", (const char *)&input.data[0]) != -1 )
                {
                    proc = PROC_OUTPUT_MSG;
                }
                else
                {
                    proc = PROC_STOP;
                }
            }
            break;
        /*--- Output "Input Error Message" ---*/
        case PROC_OUTPUT_ERR:
            {
                if (printf("\n Input Error !! \n") != -1 )
                {
                    proc = PROC_OUTPUT_MSG;
                }
                else
                {
                    proc = PROC_STOP;
                }
            }
            break;
        /*--- Error ---*/
        case PROC_ERROR:
            /* no processing */
            break;
        /*--- Default ---*/
        default:
            proc = PROC_ERROR;
            break;
        }
    }
    /*>>> System Error. Please Debug. */
    /*----------------------*/
    /* Finalize             */
    /*----------------------*/
    /*--- application ---*/
    if (application_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*--- driver      ---*/
    if (driver_finalize() != TXZ_SUCCESS)
    {
        /* no processing */
    }
    /*----------------------*/
    /* BSP finalize         */
    /*----------------------*/
    bsp_finalize();
#ifdef DEBUG
    assert_failed(MAIN_NULL, 0);
#endif

    return (0);
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART reception IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_rx(BSPUsbUart uart)
{
    switch (uart)
    {
    /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_rx_irq_handler();
        break;
    /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_rx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }

}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART transmit IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_tx(BSPUsbUart uart)
{
    switch (uart)
    {
        /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_tx_irq_handler();
        break;
        /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_tx_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  USB UART error IRQ.
  * @param  uart :USB UART.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_usb_uart_err(BSPUsbUart uart)
{
    switch (uart)
    {
        /*--- usb uart 0 ---*/
    case BSP_USB_UART_0:
        uart_io_err_irq_handler();
        break;
        /*--- usb uart 1 ---*/
    case BSP_USB_UART_1:
        uart_io_err_irq_handler();
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Timer.
  * @param  timer :Timer.
  * @return -
  * @retval -
  * @note   This function is called BSP.
  */
/*--------------------------------------------------*/
void irq_timer(BSPTimer timer)
{
    switch (timer)
    {
    /*--- 1ms timer 0 ---*/
    case BSP_TIMER_1MS:
        timer_irq_handler(&instance.timer);
        break;
    /*--- default    ---*/
    default:
        /* no processing */
        break;
    }
}

#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Failure Assert.
  * @note   for debug
  */
/*--------------------------------------------------*/
void assert_failed(char *file, int32_t line)
{
    while (1) {
        __nop();
    }
}
#endif
/**
 *  @}
 */ /* End of group UART_Exported_functions */

/**
 *  @}
 */ /* End of group UART_Sample_Appli */

/**
 *  @} 
 */ /* End of group Example */

#ifdef __cplusplus
}
#endif /* __cplusplus */
