/**
 *******************************************************************************
 * @file    system.c
 * @brief   CMSIS Cortex-M3 Device Peripheral Access Layer Source File for the
 *          TOSHIBA 'TMPM3Hz' Device Series 
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */

#include "TSB_CMSIS_CORE.h"
#include "configration.h"
#include "cg.h"
#include "flash.h"
#include "fc_copy.h"
#include "siwdt.h"

/*-------- <<< Start of configuration section >>> ----------------------------*/
/* Semi-Independent Watchdog Timer (SIWDT) Configuration */
#define SIWD_SETUP              (1U)                      /* 1:Disable SIWD, 0:Enable SIWD */
#define SIWDEN_Val              (0x00000000UL)            /* SIWD Disable */
#define SIWDCR_Val              (0x000000B1UL)            /* SIWD Disable code */

/* Clock Generator (CG) Configuration */
#define CLOCK_SETUP             (1U)                      /* 1:External HOSC, 0: Internal HOSC */

#define SYSCR_Val               0x00000000UL

#if (CLOCK_SETUP)
  #define PLL0SEL_Ready          CG_12M_MUL_10_FPLL
#else
  #define PLL0SEL_Ready          CG_10M_MUL_12_FPLL
#endif
#define PLL0SEL_Val              (PLL0SEL_Ready|0x00000003UL)

/*-------- <<< End of configuration section >>> ------------------------------*/

/*-------- DEFINES -----------------------------------------------------------*/
/* Configure Warm-up time */
#define WU_TIME_EXT             (5000UL)         /* warm-up time for EXT is 5ms   */
#define INIT_TIME_PLL           (100UL)          /* Initial time for PLL is 100us */
#define LOCKUP_TIME_PLL         (400UL)          /* Lockup time for PLL is 400us  */
/* Determine core clock frequency according to settings */
/* System clock is high-speed clock*/
#if (CLOCK_SETUP)
  #define CORE_TALH (EXTALH)
#else
  #define CORE_TALH (IXTALH)
#endif

#if ((PLL0SEL_Val & (1U<<1U)) && (PLL0SEL_Val & (1U<<0U)))   /* If PLL selected and enabled */
  #if   (CORE_TALH == EOSC_6M)         /* If input is 6MHz */
    #if ((PLL0SEL_Val & PLL0SEL_MASK) == (CG_6M_MUL_13_328_FPLL))
      #define __CORE_CLK   EOSC_6M_DIV4_PLLON                          /* output clock is 79.97MHz */
    #elif ((PLL0SEL_Val & PLL0SEL_MASK) == (CG_6M_MUL_6_664_FPLL))
      #define __CORE_CLK   EOSC_6M_DIV8_PLLON                          /* output clock is 39.98MHz */
    #elif ((PLL0SEL_Val & PLL0SEL_MASK) == (CG_6M_MUL_20_FPLL))
      #define __CORE_CLK   EOSC_6M_DIV2_PLLON                          /* output clock is 120MHz */
    #else                                      /* fc -> reserved */
      #define __CORE_CLK   (0U)
    #endif                          /* End input is 6MHz */
  #elif (CORE_TALH == EOSC_8M)       /* If input is 8MHz */
    #if ((PLL0SEL_Val & PLL0SEL_MASK) == (CG_8M_MUL_10_FPLL))
      #define __CORE_CLK   EOSC_8M_DIV4_PLLON                          /* output clock is 80MHz */
    #elif ((PLL0SEL_Val & PLL0SEL_MASK) == (CG_8M_MUL_5_FPLL))
      #define __CORE_CLK   EOSC_8M_DIV8_PLLON                          /* output clock is 40MHz */
    #elif ((PLL0SEL_Val & PLL0SEL_MASK) == (CG_8M_MUL_15_FPLL))
      #define __CORE_CLK   EOSC_8M_DIV2_PLLON                          /* output clock is 120MHz */
    #else                                     /* fc -> reserved */
      #define __CORE_CLK   (0U)
    #endif                          /* End input is 8MHz */
  #elif (CORE_TALH == EOSC_10M)      /* If input is 10MHz */
    #if ((PLL0SEL_Val & PLL0SEL_MASK) == CG_10M_MUL_8_FPLL)
      #define __CORE_CLK   EOSC_10M_DIV4_PLLON                         /* output clock is 80MHz */
    #elif ((PLL0SEL_Val & PLL0SEL_MASK) == CG_10M_MUL_4_FPLL)
      #define __CORE_CLK   EOSC_10M_DIV8_PLLON                         /* output clock is 40MHz */
    #elif ((PLL0SEL_Val & PLL0SEL_MASK) == CG_10M_MUL_12_FPLL)
      #define __CORE_CLK   EOSC_10M_DIV2_PLLON                         /* output clock is 120MHz */
    #else                                     /* fc -> reserved */
      #define __CORE_CLK   (0U)
    #endif                          /* End input is 10MHz */
  #elif (CORE_TALH == EOSC_12M)      /* If input is 12MHz */
    #if ((PLL0SEL_Val & PLL0SEL_MASK) == CG_12M_MUL_6_656_FPLL)
      #define __CORE_CLK   EOSC_12M_DIV4_PLLON                         /* output clock is 79.88MHz */
    #elif ((PLL0SEL_Val & PLL0SEL_MASK) == CG_12M_MUL_3_328_FPLL)
      #define __CORE_CLK   EOSC_12M_DIV8_PLLON                         /* output clock is 39.94MHz */
    #elif ((PLL0SEL_Val & PLL0SEL_MASK) == CG_12M_MUL_10_FPLL)
      #define __CORE_CLK   EOSC_12M_DIV2_PLLON                         /* output clock is 120MHz */
    #else                                    /* fc -> reserved */
      #define __CORE_CLK   (0U)
    #endif                          /* End input is 12MHz    */
  #elif (CORE_TALH == IOSC_10M)      /* If input is 10MHz */
    #if ((PLL0SEL_Val & PLL0SEL_MASK) == CG_10M_MUL_8_FPLL)
      #define __CORE_CLK   IOSC_10M_DIV4_PLLON                         /* output clock is 80MHz */
    #elif ((PLL0SEL_Val & PLL0SEL_MASK) == CG_10M_MUL_4_FPLL)
      #define __CORE_CLK   IOSC_10M_DIV8_PLLON                         /* output clock is 40MHz */
    #elif ((PLL0SEL_Val & PLL0SEL_MASK) == CG_10M_MUL_12_FPLL)
      #define __CORE_CLK   IOSC_10M_DIV2_PLLON                         /* output clock is 120MHz */
    #else                                    /* fc -> reserved */
      #define __CORE_CLK   (0U)
    #endif                          /* End input is 10MHz    */
  #else                             /* input clock not known */
    #define __CORE_CLK   (0U)
    #error "Core Oscillator Frequency invalid!"
  #endif                          /* End switch input clock */
#else
  #define __CORE_CLK   (CORE_TALH)
#endif

#if   ((SYSCR_Val & 7U) == 0U)      /* Gear -> fc                         */
  #define __CORE_SYS   (__CORE_CLK)
#elif ((SYSCR_Val & 7U) == 1U)      /* Gear -> fc/2                       */
  #define __CORE_SYS   (__CORE_CLK / 2U)
#elif ((SYSCR_Val & 7U) == 2U)      /* Gear -> fc/4                       */
  #define __CORE_SYS   (__CORE_CLK / 4U )
#elif ((SYSCR_Val & 7U) == 3U)      /* Gear -> fc/8                       */
  #define __CORE_SYS   (__CORE_CLK / 8U)
#elif ((SYSCR_Val & 7U) == 4U)      /* Gear -> fc/16                      */
  #define __CORE_SYS   (__CORE_CLK / 16U)
#else                               /* Gear -> reserved                   */
  #define __CORE_SYS   (0U)
#endif


/* Clock Variable definitions */
uint32_t SystemCoreClock = __CORE_SYS;  /*!< System Clock Frequency (Core Clock) */

/**
 * Initialize the system
 *
 * @param  none
 * @return none
 *
 * @brief  Update SystemCoreClock according register values.
 */
void SystemCoreClockUpdate(void)
{                               /* Get Core Clock Frequency */
    cg_t cg;
    cg.p_instance = TSB_CG;

    cg_get_system_core_clock(&cg);
}

/**
 * Initialize the system
 *
 * @param  none
 * @return none
 *
 * @brief  Setup the microcontroller system.
 *         Initialize the System.
 */
void SystemInit(void)
{
    cg_t cg;
    flash_t flash;
    cg.p_instance = TSB_CG;
    flash.p_instance = TSB_FC;

#if (SIWD_SETUP)                  /* Watchdog Setup */
    /* SIWD Disable */


    wdt_disable(TSB_SIWD0);
#else
    /* SIWD Enable (Setting after a Reset) */
#endif

    copy_fc_function();
    fc_fixed_clock_set(&flash, FIXED_120MHz );

#if (CLOCK_SETUP)               /* Clock(external) Setup */
    cg_set_gear(&cg, CG_GEAR_FC_1);
    cg_set_phyt0(&cg, CG_PRCK_FC_1);
    cg_set_ehoscen(&cg, CG_EHOSCEN_EHOSC);
    cg_set_warm_up_time(&cg, CG_WARM_UP_SRC_EHOSC, WU_TIME_EXT);
    cg_start_warmup();
    while(cg_get_warm_up_state() == TXZ_BUSY){
        ;
    }                           /* Warm-up */
    cg_set_oscsel(CG_OSCSEL_EHOSC);
    while(cg_get_oscsel() == CG_OSCSEL_IHOSC1){
        ;
    }                         /* Confirm CGOSCCR<OSCF>="1" */
#else
    /* Internal HOSC Enable (Setting after a Reset) */
#endif

#if (CLOCK_SETUP)               /* Clock(external) Setup */
    cg_set_warm_up_time(&cg, CG_WARM_UP_SRC_EHOSC, INIT_TIME_PLL);
#else
    cg_set_warm_up_time(&cg, CG_WARM_UP_SRC_IHOSC1, INIT_TIME_PLL);
#endif
    cg_set_PLL_clock(CG_PLL0SEL_FOSC);    /* PLL-->fOsc */
    cg_PLL_disable();
    cg_set_PLL(&cg, PLL0SEL_Ready);
    cg_start_warmup();
    while(cg_get_warm_up_state() == TXZ_BUSY){
        ;
    }                           /* Warm-up */

#if (CLOCK_SETUP)               /* Clock(external) Setup */
    cg_set_warm_up_time(&cg, CG_WARM_UP_SRC_EHOSC, LOCKUP_TIME_PLL);
#else
    cg_set_warm_up_time(&cg, CG_WARM_UP_SRC_IHOSC1, LOCKUP_TIME_PLL);
#endif
    cg_PLL_enable();    /* PLL enabled */
    cg_set_stby_mode(&cg, CG_STBY_IDLE);
    cg_start_warmup();
    while(cg_get_warm_up_state() == TXZ_BUSY){
        ;
    }                           /* Lockup */

    cg_set_PLL_clock(CG_PLL0SEL_FPLL);
    while(cg_get_PLL_status() == CG_PLL0SEL_FOSC){
        ;
    }                        /*Confirm CGPLL0SEL<PLL0ST> = "1" */

#if (CLOCK_SETUP)               /* Clock(external) Setup */
    cg_ihosc1_disable();
#endif
}
