/**
 *******************************************************************************
 * @file    cg_m3h1_d.c
 * @brief   This file provides API functions for CG driver.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */

#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "cg.h"

#if defined(__CG_M3H1_H)
/**
 *  @addtogroup Periph_Driver
 *  @{
 */

/** 
 *  @addtogroup CG
 *  @brief      CG Driver.
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup CG_Private_define CG Private Define
 *  @{
 */
/* no define */
/** 
 *  @}
 */ /* End of group CG_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup CG_Private_define CG Private Define
 *  @{
 */
#define CG_GEAR_MASK            ((uint32_t)0x00000007)    /*!< CG GEAR mask */
#define CG_PRCK_MASK            ((uint32_t)0x00000F00)    /*!< CG PRCK mask */

#define CG_WUPHCR_WUCLK_EHOSC   ((uint32_t)0x00000100)    /*!< CG warm up timer clock EHOSC */
#define CG_WUPHCR_WUCLK_IHOSC1  ((uint32_t)0x00000000)    /*!< CG warm up timer clock IHOSC1 */

#define CG_FSYS_MASK            ((uint32_t)0x00070000)    /*!< CG FSYS mask */

#define CG_FSYS_1               ((uint32_t)0x00000000)    /*!< CG fc register value */
#define CG_FSYS_2               ((uint32_t)0x00010000)    /*!< CG fc/2 register value */
#define CG_FSYS_4               ((uint32_t)0x00020000)    /*!< CG fc/4 register value */
#define CG_FSYS_8               ((uint32_t)0x00030000)    /*!< CG fc/8 register value */
#define CG_FSYS_16              ((uint32_t)0x00040000)    /*!< CG fc/16 register value */

#define CG_FSYS_1_MUL           ((uint32_t)0x00000001)    /*!< CG fc multiplication value */
#define CG_FSYS_2_MUL           ((uint32_t)0x00000002)    /*!< CG fc/2 multiplication value */
#define CG_FSYS_4_MUL           ((uint32_t)0x00000004)    /*!< CG fc/4 multiplication value */
#define CG_FSYS_8_MUL           ((uint32_t)0x00000008)    /*!< CG fc/8 multiplication value */
#define CG_FSYS_16_MUL          ((uint32_t)0x00000010)    /*!< CG fc/16 multiplication value */

#define CG_PRCKST_MASK          ((uint32_t)0x0F000000)    /*!< CG PRCKST mask */

#define CG_PRCK_1               ((uint32_t)0x00000000)    /*!< CG T0 fc register value */
#define CG_PRCK_2               ((uint32_t)0x01000000)    /*!< CG T0 fc/2 register value */
#define CG_PRCK_4               ((uint32_t)0x02000000)    /*!< CG T0 fc/4 register value */
#define CG_PRCK_8               ((uint32_t)0x03000000)    /*!< CG T0 fc/8 register value */
#define CG_PRCK_16              ((uint32_t)0x04000000)    /*!< CG T0 fc/16 register value */
#define CG_PRCK_32              ((uint32_t)0x05000000)    /*!< CG T0 fc/32 register value */
#define CG_PRCK_64              ((uint32_t)0x06000000)    /*!< CG T0 fc/64 register value */
#define CG_PRCK_128             ((uint32_t)0x07000000)    /*!< CG T0 fc/128 register value */
#define CG_PRCK_256             ((uint32_t)0x08000000)    /*!< CG T0 fc/256 register value */
#define CG_PRCK_512             ((uint32_t)0x09000000)    /*!< CG T0 fc/512 register value */

#define CG_PRCK_1_DIV           ((uint32_t)0x00000001)    /*!< CG T0 fc division value */
#define CG_PRCK_2_DIV           ((uint32_t)0x00000002)    /*!< CG T0 fc/2 division value */
#define CG_PRCK_4_DIV           ((uint32_t)0x00000004)    /*!< CG T0 fc/4 division value */
#define CG_PRCK_8_DIV           ((uint32_t)0x00000008)    /*!< CG T0 fc/8 division value */
#define CG_PRCK_16_DIV          ((uint32_t)0x00000010)    /*!< CG T0 fc/16 division value */
#define CG_PRCK_32_DIV          ((uint32_t)0x00000020)    /*!< CG T0 fc/32 division value */
#define CG_PRCK_64_DIV          ((uint32_t)0x00000040)    /*!< CG T0 fc/64 division value */
#define CG_PRCK_128_DIV         ((uint32_t)0x00000080)    /*!< CG T0 fc/128 division value */
#define CG_PRCK_256_DIV         ((uint32_t)0x00000100)    /*!< CG T0 fc/256 division value */
#define CG_PRCK_512_DIV         ((uint32_t)0x00000200)    /*!< CG T0 fc/512 division value */

#define CG_EOSCEN_MASK          ((uint32_t)0xFFFFFFF1)    /*!< CG EOSCEN mask */
#define CG_PLL0SET_MASK         ((uint32_t)0x00000007)    /*!< CG PLL0SET mask */
#define CG_PROTECT_ENABLE       ((uint32_t)0x00000000)    /*!< CG register protect enable */
#define CG_PROTECT_DISABLE      ((uint32_t)0x000000C1)    /*!< CG register protect disable */
#define CG_SCOSEL_MASK          ((uint32_t)0xFFFFFFF1)    /*!< CG SCOSEL mask */
#define CG_SCODIV_MASK          ((uint32_t)0xFFFFFF8F)    /*!< CG SCODIV mask */
#define ELOSC_CFG_CLOCK         ((uint32_t)(32768))       /*!< Clock(hz).        */
#define RLM_PTKEEP_ENABLE       ((uint8_t)0x01)           /*!< RLM not port keep register value */
#define RLM_PTKEEP_DISABLE      ((uint8_t)0x00)           /*!< RLM stop2 port keep register value */
#define RLM_PTKEEP              ((uint8_t)0x00)           /*!< RLM PTKEEP bit */
#define RLM_XTEN                ((uint8_t)0x00)           /*!< RLM XTEN bit */
#define RLM_PROTECT_ENABLE      ((uint8_t)0x00)           /*!< RLM register protect enable */
#define RLM_PROTECT_DISABLE     ((uint8_t)0xC1)           /*!< RLM register protect disable */
#define RLM_XTEN_ENABLE         ((uint8_t)0x01)           /*!< External Low-speed oscillator enable */
#define RLM_XTEN_DISABLE        ((uint8_t)0x00)           /*!< External Low-speed oscillator disable */
/** 
 *  @}
 */ /* End of group CG_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup CG_Private_define CG Private Define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group CG_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup CG_Private_typedef CG Private Typedef
 *  @{
 */

/* no define */

/**
 *  @}
 */ /* End of group CG_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup CG_Private_fuctions CG Private Fuctions
 *  @{
 */

/* no define */

/**
 *  @}
 */ /* End of group CG_Private_functions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup CG_Exported_functions CG Exported Functions
 *  @{
 */

/*--------------------------------------------------*/
/**
  * @brief     Set Gear according register values.
  * @param     p_obj : CG object.
  * @param     gear  : Gear Frequency.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_set_gear(cg_t *p_obj, uint32_t gear)
{
    uint32_t regval = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* Set Gear status. */
    regval = p_obj->p_instance->SYSCR & CG_GEAR_MASK;
    regval |= gear;
    p_obj->p_instance->SYSCR = regval;
}

/*--------------------------------------------------*/
/**
  * @brief     Set PrescalerClock according register values.
  * @param     p_obj : CG object.
  * @param     prck  : PrescalerClock Frequency.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_set_phyt0(cg_t *p_obj, uint32_t prck)
{
    uint32_t regval = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* Set PrescalerClock status. */
    regval = p_obj->p_instance->SYSCR & CG_PRCK_MASK;
    regval |= prck;
    p_obj->p_instance->SYSCR = regval;
}

/*--------------------------------------------------*/
/**
  * @brief  Set the warm up time.
  * @param  p_obj : CG object.
  * @param  source: Set base clock for warm-up timer
  * @param  time  : Set count time for warm-up timer
  * @retval -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_set_warm_up_time(cg_t *p_obj, uint32_t source, uint32_t time)
{
    uint32_t regval = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    if (source == CG_WARM_UP_SRC_EHOSC) {
        regval = ((uint32_t)(((((uint64_t)time * EXTALH / HZ_1M) - 16UL) /16UL) << 20U));
        regval |= CG_WUPHCR_WUCLK_EHOSC;
    } else if (source == CG_WARM_UP_SRC_IHOSC1 ) {
        regval = ((uint32_t)(((((uint64_t)(time - 633UL) * (IXTALH / 10) / HZ_1M) - 41UL) /16UL) << 20U));
        if(regval < WU_TIME_INT_MIN)
        {
            regval = WU_TIME_INT_MIN;
        }
        else
        {
            regval = WUPHCR_WUPT_INT;
        }
        regval |= CG_WUPHCR_WUCLK_IHOSC1;
    }

    p_obj->p_instance->WUPHCR = regval;
}

/*--------------------------------------------------*/
/**
  * @brief  Start operation of warm up timer for oscillator.
  * @param  -
  * @retval -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_start_warmup(void)
{
    TSB_CG_WUPHCR_WUON = 1U;
}

/*--------------------------------------------------*/
/**
  * @brief  Wait operation of warm up timer for oscillator.
  * @param  -
  * @retval -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_wait_warmup(void)
{
    /* [CGWUPHCR]<WUEF>  : Read(wait for done) */
    while( TSB_CG_WUPHCR_WUEF != CGWUPHCR_WUEF_R_DONE )
    {
        /* no processing */
    }
}

/*--------------------------------------------------*/
/**
  * @brief Check whether warm up is completed or not.
  * @param  -
  * @retval    TXZ_DONE :Done.
  * @retval    TXZ_BUSY :Busy.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
TXZ_WorkState cg_get_warm_up_state(void)
{
    TXZ_WorkState state = TXZ_BUSY;

    if (TSB_CG_WUPHCR_WUEF == 0U) {
        state = TXZ_DONE;
    } else {
        /* Do nothing */
    }

    return state;
}

/*--------------------------------------------------*/
/**
  * @brief     Set External high-speed oscillator select.
  * @param     p_obj       :CG object.
  * @param     eoscen      :Set External high-speed oscillator select.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_set_ehoscen(cg_t *p_obj, uint32_t eoscen)
{
    uint32_t regval = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* Set External high-speed oscillator select. */
    regval = p_obj->p_instance->OSCCR & CG_EOSCEN_MASK;
    regval |= eoscen;
    p_obj->p_instance->OSCCR = regval;
}

/*--------------------------------------------------*/
/**
  * @brief  Set the fosc base clock.
  * @param  source: Set base clock for fosc
  * @retval -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_set_oscsel(uint32_t source)
{
    if (source == CG_OSCSEL_EHOSC) {
        TSB_CG_OSCCR_OSCSEL = 1U;
    } else {
        TSB_CG_OSCCR_OSCSEL = 0U;
    }
}

/*--------------------------------------------------*/
/**
  * @brief Check whether oscsel.
  * @param  -
  * @retval CG_OSCSEL_EHOSC  :EHOSC.
  * @retval CG_OSCSEL_IHOSC1 :IHOSC1.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
uint32_t cg_get_oscsel(void)
{
    uint32_t retval = 0;

    if (TSB_CG_OSCCR_OSCF == 1U) {
        retval = CG_OSCSEL_EHOSC;
    } else {
        /* Do nothing */
    }

    return retval;
}

/*--------------------------------------------------*/
/**
  * @brief     Update Gear according register values.
  * @param     p_obj       :CG object.
  * @retval    Gear Frequency.
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
uint32_t cg_get_gear(cg_t *p_obj)
{
    uint32_t result = 0U;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* System core clock update */
    SystemCoreClockUpdate();

    /* Get Gear status. */
    switch (p_obj->p_instance->SYSCR & CG_FSYS_MASK)
    {
    case CG_FSYS_1:             /* Gear -> fc */
        result = SystemCoreClock * CG_FSYS_1_MUL;
        break;
    case CG_FSYS_2:             /* Gear -> fc/2 */
        result = SystemCoreClock * CG_FSYS_2_MUL;
        break;
    case CG_FSYS_4:             /* Gear -> fc/4 */
        result = SystemCoreClock * CG_FSYS_4_MUL;
        break;
    case CG_FSYS_8:             /* Gear -> fc/8 */
        result = SystemCoreClock * CG_FSYS_8_MUL;
        break;
    case CG_FSYS_16:            /* Gear -> fc/16 */
        result = SystemCoreClock * CG_FSYS_16_MUL;
        break;
    default:
        result = 0U;
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief     Update PrescalerClock according register values.
  * @param     p_obj       :CG object.
  * @retval    PrescalerClock Frequency.
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
uint32_t cg_get_phyt0(cg_t *p_obj)
{
    uint32_t result = 0U;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* System core clock update */
    SystemCoreClockUpdate();

    /* Get Gear status. */
    switch (p_obj->p_instance->SYSCR & CG_FSYS_MASK)
    {
    case CG_FSYS_1:             /* Gear -> fc */
        result = SystemCoreClock * CG_FSYS_1_MUL;
        break;
    case CG_FSYS_2:             /* Gear -> fc/2 */
        result = SystemCoreClock * CG_FSYS_2_MUL;
        break;
    case CG_FSYS_4:             /* Gear -> fc/4 */
        result = SystemCoreClock * CG_FSYS_4_MUL;
        break;
    case CG_FSYS_8:             /* Gear -> fc/8 */
        result = SystemCoreClock * CG_FSYS_8_MUL;
        break;
    case CG_FSYS_16:            /* Gear -> fc/16 */
        result = SystemCoreClock * CG_FSYS_16_MUL;
        break;
    default:
        result = 0U;
        break;
    }

    switch (p_obj->p_instance->SYSCR & CG_PRCKST_MASK)
    {
    case CG_PRCK_1:             /* T0 -> fc */
        result /= CG_PRCK_1_DIV;
        break;
    case CG_PRCK_2:             /* T0 -> fc/2 */
        result /= CG_PRCK_2_DIV;
        break;
    case CG_PRCK_4:             /* T0 -> fc/4 */
        result /= CG_PRCK_4_DIV;
        break;
    case CG_PRCK_8:             /* T0 -> fc/8 */
        result /= CG_PRCK_8_DIV;
        break;
    case CG_PRCK_16:            /* T0 -> fc/16 */
        result /= CG_PRCK_16_DIV;
        break;
    case CG_PRCK_32:            /* T0 -> fc/32 */
        result /= CG_PRCK_32_DIV;
        break;
    case CG_PRCK_64:            /* T0 -> fc/64 */
        result /= CG_PRCK_64_DIV;
        break;
    case CG_PRCK_128:           /* T0 -> fc/128 */
        result /= CG_PRCK_128_DIV;
        break;
    case CG_PRCK_256:           /* T0 -> fc/256 */
        result /= CG_PRCK_256_DIV;
        break;
    case CG_PRCK_512:           /* T0 -> fc/512 */
        result /= CG_PRCK_512_DIV;
        break;
    default:
        result = 0U;
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief     Set Internal high-speed oscillator1 enable.
  * @param     -
  * @retval    TXZ_SUCCESS :Success.
  * @retval    TXZ_ERROR   :Failure.
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_ihosc1_enable(void)
{
    /* Internal high-speed oscillator1 is enable. */
    TSB_CG_OSCCR_IHOSC1EN = 1U;
}

/*--------------------------------------------------*/
/**
  * @brief     Set Internal high-speed oscillator1 disable.
  * @param     -
  * @retval    TXZ_SUCCESS :Success.
  * @retval    TXZ_ERROR   :Failure.
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_ihosc1_disable(void)
{
    /* Internal high-speed oscillator1 is disable. */
    TSB_CG_OSCCR_IHOSC1EN = 0U;
}

/*--------------------------------------------------*/
/**
  * @brief     Set PLL according register values.
  * @param     p_obj  : CG object.
  * @param     PLLset : PLLset.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_set_PLL(cg_t *p_obj, uint32_t PLLset)
{
    uint32_t regval = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* Set PLLset value. */
    regval = p_obj->p_instance->PLL0SEL & CG_PLL0SET_MASK;
    regval |= PLLset;
    p_obj->p_instance->PLL0SEL = regval;
}

/*--------------------------------------------------*/
/**
  * @brief     Set PLL enable.
  * @param     -
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_PLL_enable(void)
{
    /* PLL is enable. */
    TSB_CG_PLL0SEL_PLL0ON = 1U;
}

/*--------------------------------------------------*/
/**
  * @brief     Set PLL disable.
  * @param     -
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_PLL_disable(void)
{
    /* PLL is disable. */
    TSB_CG_PLL0SEL_PLL0ON = 0U;
}

/*--------------------------------------------------*/
/**
  * @brief     Set STBYmode according register values.
  * @param     p_obj :CG object.
  * @param     mode  :STBYmode.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_set_stby_mode(cg_t *p_obj, uint32_t mode)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* Set STBYmode status. */
    p_obj->p_instance->STBYCR = mode;
}

/*--------------------------------------------------*/
/**
  * @brief     Set PLL base clock.
  * @param     p_obj  : CG object.
  * @param     clock  : PLL base clock.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_set_PLL_clock(uint32_t clock)
{
    if (clock == CG_PLL0SEL_FPLL) {
        TSB_CG_PLL0SEL_PLL0SEL = 1U;
    } else {
        TSB_CG_PLL0SEL_PLL0SEL = 0U;
    }
}

/*--------------------------------------------------*/
/**
  * @brief Check whether PLL0SEL state.
  * @param  -
  * @retval    CG_FPLL :PLL use.
  * @retval    CG_FOSC :PLL unuse.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
uint32_t cg_get_PLL_status(void)
{
    uint32_t retval = 0;

    if (TSB_CG_PLL0SEL_PLL0ST == 1U) {
        retval = CG_PLL0SEL_FPLL;
    } else {
        /* Do nothing */
    }

    return retval;
}

/*--------------------------------------------------*/
/**
  * @brief     Set CG protect enable.
  * @param     p_obj       :CG object.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_protect_enable(cg_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* CG protect is enable. */
    p_obj->p_instance->PROTECT = CG_PROTECT_ENABLE;
}

/*--------------------------------------------------*/
/**
  * @brief     Set CG protect disable.
  * @param     p_obj       :CG object.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_protect_disable(cg_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* CG protect is disable. */
    p_obj->p_instance->PROTECT = CG_PROTECT_DISABLE;
}

/*--------------------------------------------------*/
/**
  * @brief Check whether ihosc1 state.
  * @param  -
  * @retval    TXZ_DONE :Done.
  * @retval    TXZ_BUSY :Busy.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
TXZ_WorkState cg_get_ihosc1_state(void)
{
    TXZ_WorkState state = TXZ_BUSY;

    if (TSB_CG_OSCCR_IHOSC1F == 1U) {
        state = TXZ_DONE;
    } else {
        /* Do nothing */
    }

    return state;
}

/*--------------------------------------------------*/
/**
  * @brief Check whether ihosc2 state.
  * @param  -
  * @retval    TXZ_DONE :Done.
  * @retval    TXZ_BUSY :Busy.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
TXZ_WorkState cg_get_ihosc2_state(void)
{
    TXZ_WorkState state = TXZ_BUSY;

    if (TSB_CG_OSCCR_IHOSC2F == 1U) {
        state = TXZ_DONE;
    } else {
        /* Do nothing */
    }

    return state;
}

/*--------------------------------------------------*/
/**
  * @brief     Set Internal high-speed oscillator2 enable.
  * @param     -
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_ihosc2_enable(void)
{
    /* Internal high-speed oscillator2 is enable. */
    TSB_CG_OSCCR_IHOSC2EN = 1U;
}

/*--------------------------------------------------*/
/**
  * @brief     Set Internal high-speed oscillator2 disable.
  * @param     -
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_ihosc2_disable(void)
{
    /* Internal high-speed oscillator2 is disable. */
    TSB_CG_OSCCR_IHOSC2EN = 0U;
}

/*--------------------------------------------------*/
/**
  * @brief     Set SCOUT enable.
  * @param     -
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_scout_enable(void)
{
    /* SCOUT is enable. */
    TSB_CG_SCOCR_SCOEN = 1U;
}

/*--------------------------------------------------*/
/**
  * @brief     Set SCOUT disable.
  * @param     -
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_scout_disable(void)
{
    /* SCOUT is disable. */
    TSB_CG_SCOCR_SCOEN = 0U;
}

/*--------------------------------------------------*/
/**
  * @brief     Set SCOUT base clock.
  * @param     p_obj  : CG object.
  * @param     source : SCOUT base clock.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_set_scout_clock(cg_t *p_obj, uint32_t source)
{
    uint32_t regval = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* Set SCOUT base clock value. */
    regval = p_obj->p_instance->SCOCR & CG_SCOSEL_MASK;
    regval |= source;
    p_obj->p_instance->SCOCR = regval;
}

/*--------------------------------------------------*/
/**
  * @brief     Set SCOUT time.
  * @param     p_obj  : CG object.
  * @param     source  : SCOUT time.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_set_scout_time(cg_t *p_obj, uint32_t time)
{
    uint32_t regval = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* Set SCOUT time value. */
    regval = p_obj->p_instance->SCOCR & CG_SCODIV_MASK;
    regval |= time;
    p_obj->p_instance->SCOCR = regval;
}

/*--------------------------------------------------*/
/**
  * @brief     Get STBYmode register values.
  * @param     p_obj       :CG object.
  * @retval    STBYmode.
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
uint32_t cg_get_stby_mode(cg_t *p_obj)
{
    uint32_t result = 0U;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* Get STBYmode status. */
    result = p_obj->p_instance->STBYCR;

    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief  Set the low warm up time.
  * @param  p_obj                  :CG object.
  * @param  time: Set count time for warm-up timer
  * @retval -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_set_low_warm_up_time(cg_t *p_obj, uint32_t time)
{
    uint32_t work = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* [CGWUPLCR]<WUPTL> :Warm up time        */
    /*-----------------------------------------------*/
    /* "1"counter  (s) = 1 / ELOSC                   */
    /* "1"counter (us) = (10^6) / ELOSC              */
    /* "x"counter (us) = time                        */
    /*-----------------------------------------------*/
    /* x : time = 1 : (10^6) / ELOSC                 */
    /*-----------------------------------------------*/
    {
        uint64_t x = (uint64_t)((uint64_t)(time) * (uint64_t)(ELOSC_CFG_CLOCK));
        x = (uint64_t)((uint64_t)(x / (uint64_t)(1000000)) - 16);
        if (x > (uint64_t)(0x7FFFF))
        {
            /* invalid value */
        }
        work = (uint32_t)x;
    }
    work  &= (uint32_t)(0xFFFFFFF0);
    work <<= 8;
    p_obj->p_instance->WUPLCR = work;
}

/*--------------------------------------------------*/
/**
  * @brief  Start operation of warm up timer for low oscillator.
  * @param  -
  * @retval -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_start_low_warmup(void)
{
    TSB_CG_WUPLCR_WULON = 1U;
}

/*--------------------------------------------------*/
/**
  * @brief Check whether low warm up is completed or not.
  * @param  -
  * @retval    TXZ_DONE :Done.
  * @retval    TXZ_BUSY :Busy.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
TXZ_WorkState cg_get_low_warm_up_state(void)
{
    TXZ_WorkState state = TXZ_BUSY;

    if ( TSB_CG_WUPLCR_WULEF == 0U) {
        state = TXZ_DONE;
    } else {
        /* Do nothing */
    }

    return state;
}

/*--------------------------------------------------*/
/**
  * @brief     Get CG system core clock.
  * @param     p_obj       :CG object.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void cg_get_system_core_clock(cg_t *p_obj)
{
    uint32_t CoreClock = 0U;
    uint32_t CoreClockInput = 0U;
    uint32_t regval = 0U;
    uint32_t oscsel = 0U;
    uint32_t pll0sel = 0U;
    uint32_t pll0on = 0U;
    /* Determine clock frequency according to clock register values */
    /* System clock is high-speed clock */
    regval = p_obj->p_instance->OSCCR;
    oscsel = regval & CG_OSCSEL_EHOSC;
    if (oscsel) {  /* If system clock is External high-speed oscillator freq */
        CoreClock = EXTALH;
    } else {                    /* If system clock is Internal high-speed oscillator freq */
        CoreClock = IXTALH;
    }
    regval = p_obj->p_instance->PLL0SEL;
    pll0sel = regval & CG_PLL0SEL_FPLL;
    pll0on = regval & CG_PLL0SEL_PLL0ON_SET;
    if (pll0sel && pll0on) {                        /* If PLL enabled  */
        if (CoreClock == EOSC_6M) {      /* If input is 6MHz */
            if ((p_obj->p_instance->PLL0SEL & PLL0SEL_MASK) == CG_6M_MUL_13_328_FPLL) {
                CoreClockInput = EOSC_6M_DIV4_PLLON;           /* output clock is 79.97MHz */
            } else if ((p_obj->p_instance->PLL0SEL & PLL0SEL_MASK) == CG_6M_MUL_6_664_FPLL) {
                CoreClockInput = EOSC_6M_DIV8_PLLON;           /* output clock is 39.98MHz */
            } else if ((p_obj->p_instance->PLL0SEL & PLL0SEL_MASK) == CG_6M_MUL_20_FPLL) {
                CoreClockInput = EOSC_6M_DIV2_PLLON;           /* output clock is 120MHz */
            } else {
                CoreClockInput = 0U;    /* fc -> reserved */
            }
        } else if (CoreClock == EOSC_8M) {      /* If input is 8MHz */
            if ((p_obj->p_instance->PLL0SEL & PLL0SEL_MASK) == CG_8M_MUL_10_FPLL) {
                CoreClockInput = EOSC_8M_DIV4_PLLON;           /* output clock is 80MHz */
            } else if ((p_obj->p_instance->PLL0SEL & PLL0SEL_MASK) == CG_8M_MUL_5_FPLL) {
                CoreClockInput = EOSC_8M_DIV8_PLLON;           /* output clock is 40MHz */
            } else if ((p_obj->p_instance->PLL0SEL & PLL0SEL_MASK) == CG_8M_MUL_15_FPLL) {
                CoreClockInput = EOSC_8M_DIV2_PLLON;           /* output clock is 120MHz */
            } else {
                CoreClockInput = 0U;    /* fc -> reserved */
            }
        } else if (CoreClock == EOSC_10M) {      /* If input is 10MHz */
            if ((p_obj->p_instance->PLL0SEL & PLL0SEL_MASK) == CG_10M_MUL_8_FPLL) {
                CoreClockInput = EOSC_10M_DIV4_PLLON;          /* output clock is 80MHz */
            } else if ((p_obj->p_instance->PLL0SEL & PLL0SEL_MASK) == CG_10M_MUL_4_FPLL) {
                CoreClockInput = EOSC_10M_DIV8_PLLON;          /* output clock is 40MHz */
            } else if ((p_obj->p_instance->PLL0SEL & PLL0SEL_MASK) == CG_10M_MUL_12_FPLL) {
                CoreClockInput = EOSC_10M_DIV2_PLLON;          /* output clock is 120MHz */
            } else {
                CoreClockInput = 0U;    /* fc -> reserved */
            }
        } else if (CoreClock == EOSC_12M) {      /* If input is 12MHz */
            if ((p_obj->p_instance->PLL0SEL & PLL0SEL_MASK) == CG_12M_MUL_6_656_FPLL) {
                CoreClockInput = EOSC_12M_DIV4_PLLON;          /* output clock is 79.88MHz */
            } else if ((p_obj->p_instance->PLL0SEL & PLL0SEL_MASK) == CG_12M_MUL_3_328_FPLL) {
                CoreClockInput = EOSC_12M_DIV8_PLLON;          /* output clock is 39.94MHz */
            } else if ((p_obj->p_instance->PLL0SEL & PLL0SEL_MASK) == CG_12M_MUL_10_FPLL) {
                CoreClockInput = EOSC_12M_DIV2_PLLON;          /* output clock is 120MHz */
            } else {
                CoreClockInput = 0U;    /* fc -> reserved */
            }
        } else if (CoreClock == IOSC_10M) {      /* If input is 10MHz */
            if ((p_obj->p_instance->PLL0SEL & PLL0SEL_MASK) == CG_10M_MUL_8_FPLL) {
                CoreClockInput = IOSC_10M_DIV4_PLLON;          /* output clock is 80MHz */
            } else if ((p_obj->p_instance->PLL0SEL & PLL0SEL_MASK) == CG_10M_MUL_4_FPLL) {
                CoreClockInput = IOSC_10M_DIV8_PLLON;          /* output clock is 40MHz */
            } else if ((p_obj->p_instance->PLL0SEL & PLL0SEL_MASK) == CG_10M_MUL_12_FPLL) {
                CoreClockInput = IOSC_10M_DIV2_PLLON;          /* output clock is 120MHz */
            } else {
                CoreClockInput = 0U;    /* fc -> reserved */
            }
        } else {
            CoreClockInput = 0U;
        }
    } else {                                       /* If PLL not used */
        CoreClockInput = CoreClock;
    }

    switch (p_obj->p_instance->SYSCR & 7U) {
    case 0U:                                 /* Gear -> fc */
        SystemCoreClock = CoreClockInput;
        break;
    case 1U:                                 /* Gear -> fc/2 */
        SystemCoreClock = CoreClockInput / 2U;
        break;
    case 2U:                                 /* Gear -> fc/4 */
        SystemCoreClock = CoreClockInput / 4U;
        break;
    case 3U:                                 /* Gear -> fc/8 */
        if (CoreClockInput >= EOSC_8M) {
            SystemCoreClock = CoreClockInput / 8U;
        } else {
            SystemCoreClock = 0U;
        }
        break;
    case 4U:                                 /* Gear -> fc/16 */
        if (CoreClockInput > EOSC_12M) {
            SystemCoreClock = CoreClockInput / 16U;
        } else {
            SystemCoreClock = 0U;
        }
        break;
    case 5U:
    case 6U:
    case 7U:
        SystemCoreClock = 0U;
        break;
    default:
        SystemCoreClock = 0U;
        break;
    }
}

/*--------------------------------------------------*/
/**
  * @brief     Set RLM port keep enable.
  * @param     p_obj       :RLM object.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void rlm_port_keep_enable(rlm_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* RLM protect is enable. */
    p_obj->p_instance->SHTDNOP = RLM_PTKEEP_ENABLE;
}

/*--------------------------------------------------*/
/**
  * @brief     Set RLM port keep disable.
  * @param     p_obj       :RLM object.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void rlm_port_keep_disable(rlm_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* RLM protect is disable. */
    p_obj->p_instance->SHTDNOP = RLM_PTKEEP_DISABLE;
}

/*--------------------------------------------------*/
/**
  * @brief     Set RLM protect enable.
  * @param     p_obj       :RLM object.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void rlm_protect_enable(rlm_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* RLM protect is enable. */
    p_obj->p_instance->PROTECT = RLM_PROTECT_ENABLE;
}

/*--------------------------------------------------*/
/**
  * @brief     Set RLM protect disable.
  * @param     p_obj       :RLM object.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void rlm_protect_disable(rlm_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* RLM protect is disable. */
    p_obj->p_instance->PROTECT = RLM_PROTECT_DISABLE;
}

/*--------------------------------------------------*/
/**
  * @brief     Set fs enable.
  * @param     p_obj       :RLM object.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void rlm_fs_enable(rlm_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* fs is enable. */
    p_obj->p_instance->LOSCCR = RLM_XTEN_ENABLE;
}

/*--------------------------------------------------*/
/**
  * @brief     Set fs disable.
  * @param     p_obj       :RLM object.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void rlm_fs_disable(rlm_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the CG_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* fs is enable. */
    p_obj->p_instance->LOSCCR = RLM_XTEN_DISABLE;
}

/**
 *  @}
 */ /* End of group CG_Exported_functions */

/**
 *  @} 
 */ /* End of group CG */

/**
 *  @} 
 */ /* End of group Periph_Driver */

#endif /* defined(__CG_M3H1_H)  */

#ifdef __cplusplus
}
#endif /* __cplusplus */

