/**
 *******************************************************************************
 * @file    dnf_a.c
 * @brief   This file provides API functions for A-DNF.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */

#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "dnf.h"

#if defined(__DNF_A_H)
/**
 *  @addtogroup Periph_Driver Peripheral Driver
 *  @{
 */

/** 
 *  @defgroup DNF DNF
 *  @brief    DNF Driver.
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup DNF_Private_macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group DNF_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup DNF_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group DNF_Private_define */

/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup DNF_Private_define
 *  @{
 */

/** 
 *  @defgroup Null Null Pointer
 *  @brief    Null Pointer.
 *  @{
 */
#define REG_DNF_NULL                  ((void *)0)
/**
 *  @}
 */ /* End of group Null */

/** 
 *  @defgroup CheckParamResult Parameter Check Result
 *  @brief    Whether the parameter is specified or not.
 *  @{
 */
#define REG_DNF_PARAM_OK              ((int32_t)1)              /*!< Parameter is valid(specified).         */
#define REG_DNF_PARAM_NG              ((int32_t)0)              /*!< Parameter is invalid(not specified).   */
/**
 *  @}
 */ /* End of name Parameter CheckParamResult */


/** 
 *  @}
 */ /* End of group DNF_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup DNF_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group DNF_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup DNF_Private_typedef
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group DNF_Private_typedef */

/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup DNF_Private_variables
 *  @{
 */

/* no define */

/**
 *  @}
 */ /* End of group DNF_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup DNF_Private_fuctions
 *  @{
 */
#ifdef DEBUG
    __STATIC_INLINE int32_t check_param_DNFCKCR_NFCKS(uint32_t param);
    __STATIC_INLINE int32_t check_param_DNFxENCR_mask(uint32_t param);
#endif /* #ifdef DEBUG */
#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  @ref NFCKCR_NFCKS Paramter Check.
  * @param  clock  :@ref NFCKCR_NFCKS
  * @return Result.
  * @retval REG_DNF_PARAM_OK :Valid
  * @retval REG_DNF_PARAM_NG :Invalid
  * @note   -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_DNFCKCR_NFCKS(uint32_t param)
{
    int32_t result = REG_DNF_PARAM_NG;

    switch (param) 
    {
    case REG_DNF_NFCKCR_NFCKS_STOP:
    case REG_DNF_NFCKCR_NFCKS_CLOCK_2:
    case REG_DNF_NFCKCR_NFCKS_CLOCK_4:
    case REG_DNF_NFCKCR_NFCKS_CLOCK_8:
    case REG_DNF_NFCKCR_NFCKS_CLOCK_16:
    case REG_DNF_NFCKCR_NFCKS_CLOCK_32:
    case REG_DNF_NFCKCR_NFCKS_CLOCK_64:
    case REG_DNF_NFCKCR_NFCKS_CLOCK_128:
        result = REG_DNF_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  DNFxENCR_mask Paramter Check.
  * @param  param  :DNFxENCR mask
  * @return Result.
  * @retval REG_ENC_PARAM_OK :Valid
  * @retval REG_ENC_PARAM_NG :Invalid
  * @note   -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_DNFxENCR_mask(uint32_t param)
{
    int32_t result = REG_DNF_PARAM_NG;

    switch (param) 
    {
    case REG_DNF_NFENCR_NFEN10_MASK:
    case REG_DNF_NFENCR_NFEN9_MASK:
    case REG_DNF_NFENCR_NFEN8_MASK:
    case REG_DNF_NFENCR_NFEN7_MASK:
    case REG_DNF_NFENCR_NFEN6_MASK:
    case REG_DNF_NFENCR_NFEN5_MASK:
    case REG_DNF_NFENCR_NFEN4_MASK:
    case REG_DNF_NFENCR_NFEN3_MASK:
    case REG_DNF_NFENCR_NFEN2_MASK:
    case REG_DNF_NFENCR_NFEN1_MASK:
    case REG_DNF_NFENCR_NFEN0_MASK:
        result = REG_DNF_PARAM_OK;
        break;
    default:
        /* no processing */
        break;
    }

    return (result);
}
#endif /* #ifdef DEBUG */
/**
 *  @}
 */ /* End of group DNF_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup DNF_Exported_functions
 *  @{
 */
/*--------------------------------------------------*/
/**
  * @brief  Set Value Noise Filter Clock Select.
  * @param  p_reg   :Register Base Address.
  * @param  clock   : NFCKCR_NFCKS
  * @return -
  * @retval -
  * @note   The used register.
  * | Register  | Bit Symbol |
  * | :---      | :---       |
  * | NFCKCR    | NFCKS      |
  */
/*--------------------------------------------------*/
void REG_DNFCKCR_set(TSB_DNF_TypeDef *p_reg, uint32_t clock)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the REG_DNF_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_reg));
    assert_param(check_param_DNFCKCR_NFCKS(clock));
#endif /* #ifdef DEBUG */
    /*--- [NFCKCR]<NFCKS> ---*/
    /*------------------------------*/
    /*  Register Set                */
    /*------------------------------*/
    p_reg->CKCR = clock;
}
/*--------------------------------------------------*/
/**
  * @brief  Get value Noise Filter Clock Select.
  * @param  p_reg   :Register Base Address.
  * @return NFCKCR.
  * @retval value   : NFCKCR_NFCKS.
  * @note   The used register.
  * | Register  | Bit Symbol |
  * | :---      | :---       |
  * | NFCKCR    | NFCKS      |
  */
/*--------------------------------------------------*/
uint32_t REG_DNFCKCR_get(TSB_DNF_TypeDef *p_reg)
{
    uint32_t value = 0U;
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the REG_DNF_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_reg));
#endif /* #ifdef DEBUG */
    /*--- [NFCKCR]<NFCKS> ---*/
    /*------------------------------*/
    /*  Register Set                */
    /*------------------------------*/
    value = p_reg->CKCR;
    return (value);
}
/*--------------------------------------------------*/
/**
  * @brief  Set Interrupt Noise Filter Control.
  * @param  p_reg   :Register Base Address.
  * @param  bit     :Interrupt.
  *                 :This parameter can be one of the following values: 
  *                 :REG_DNF_NFENCR_NFEN0_MASK
  *                 :REG_DNF_NFENCR_NFEN1_MASK
  *                 :REG_DNF_NFENCR_NFEN2_MASK
  *                 :REG_DNF_NFENCR_NFEN3_MASK
  *                 :REG_DNF_NFENCR_NFEN4_MASK
  *                 :REG_DNF_NFENCR_NFEN5_MASK
  *                 :REG_DNF_NFENCR_NFEN6_MASK
  *                 :REG_DNF_NFENCR_NFEN7_MASK
  *                 :REG_DNF_NFENCR_NFEN8_MASK
  *                 :REG_DNF_NFENCR_NFEN9_MASK
  *                 :REG_DNF_NFENCR_NFEN10_MASK
  * @return -
  * @retval -
  * @note   The used register.
  * | Register   | Bit Symbol    |
  * | :---       | :---          |
  * | NFENCR     | NFEN0         |
  * |            | NFEN1         |
  * |            | NFEN2         |
  * |            | NFEN3         |
  * |            | NFEN4         |
  * |            | NFEN5         |
  * |            | NFEN6         |
  * |            | NFEN7         |
  * |            | NFEN8         |
  * |            | NFEN9         |
  * |            | NFEN10        |
  */
/*--------------------------------------------------*/
void REG_DNFENCR_enable(TSB_DNF_TypeDef *p_reg, uint32_t bit)
{
    uint32_t inten = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the REG_DNF_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_reg));
    assert_param(check_param_DNFxENCR_mask(bit));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Register Setting            */
    /*------------------------------*/
    inten  = (uint32_t)(p_reg->ENCR & ~bit);
    inten |= bit;
    /*------------------------------*/
    /*  Register Set                */
    /*------------------------------*/
    /*--- [DNFxENCR] ---*/
    p_reg->ENCR = inten;
}

/*--------------------------------------------------*/
/**
  * @brief  Set Interrupt Noise Filter Control.
  * @param  p_reg   :Register Base Address.
  * @param  bit     :Interrupt.
  *                 :This parameter can be one of the following values: 
  *                 :REG_DNF_NFENCR_NFEN0_MASK
  *                 :REG_DNF_NFENCR_NFEN1_MASK
  *                 :REG_DNF_NFENCR_NFEN2_MASK
  *                 :REG_DNF_NFENCR_NFEN3_MASK
  *                 :REG_DNF_NFENCR_NFEN4_MASK
  *                 :REG_DNF_NFENCR_NFEN5_MASK
  *                 :REG_DNF_NFENCR_NFEN6_MASK
  *                 :REG_DNF_NFENCR_NFEN7_MASK
  *                 :REG_DNF_NFENCR_NFEN8_MASK
  *                 :REG_DNF_NFENCR_NFEN9_MASK
  *                 :REG_DNF_NFENCR_NFEN10_MASK
  * @return -
  * @retval -
  * @note   The used register.
  * | Register   | Bit Symbol    |
  * | :---       | :---          |
  * | NFENCR     | NFEN0         |
  * |            | NFEN1         |
  * |            | NFEN2         |
  * |            | NFEN3         |
  * |            | NFEN4         |
  * |            | NFEN5         |
  * |            | NFEN6         |
  * |            | NFEN7         |
  * |            | NFEN8         |
  * |            | NFEN9         |
  * |            | NFEN10        |
  */
/*--------------------------------------------------*/
void REG_DNFENCR_disable(TSB_DNF_TypeDef *p_reg, uint32_t bit)
{
    uint32_t inten = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the REG_DNF_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_reg));
    assert_param(check_param_DNFxENCR_mask(bit));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Register Setting            */
    /*------------------------------*/
    inten  = (uint32_t)(p_reg->ENCR & ~bit);
    inten |= REG_DNF_NFENCR_NFEN0_DISABLE;
    /*------------------------------*/
    /*  Register Set                */
    /*------------------------------*/
    /*--- [DNFxENCR] ---*/
    p_reg->ENCR = inten;
}

/*--------------------------------------------------*/
/**
  * @brief  Get the Interrupt Noise Filter Control.
  * @param  p_reg   :Register Base Address.
  * @return NFENCR.
  * @retval value   : NFENCR.
  * @note   The used register.
  * | Register   | Bit Symbol    |
  * | :---       | :---          |
  * | NFENCR     | NFEN0         |
  * |            | NFEN1         |
  * |            | NFEN2         |
  * |            | NFEN3         |
  * |            | NFEN4         |
  * |            | NFEN5         |
  * |            | NFEN6         |
  * |            | NFEN7         |
  * |            | NFEN8         |
  * |            | NFEN9         |
  * |            | NFEN10        |
  */
/*--------------------------------------------------*/
uint32_t REG_DNFENCR_get(TSB_DNF_TypeDef *p_reg)
{
    uint32_t value = 0U;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the REG_DNF_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_reg));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Register Read               */
    /*------------------------------*/
    /*--- [ENxINTF] ---*/
    value = p_reg->ENCR;

    return (value);
}

/**
 *  @}
 */ /* End of group DNF_Exported_functions */

/**
 *  @}
 */ /* End of group DNF */

/**
 *  @} 
 */ /* End of group Periph_Driver */

#endif /* defined(__DNF_A_H)  */
#ifdef __cplusplus
}
#endif /* __cplusplus */
