/**
 *******************************************************************************
 * @file    ei2c_a.c
 * @brief   This file provides API functions for EI2C Class.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */

#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "ei2c.h"

#if defined(__EI2C_H)

/**
 *  @addtogroup Periph_Driver Peripheral Driver
 *  @{
 */

/** 
 *  @addtogroup EI2C
 *  @brief      EI2C Driver.
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Private_macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group EI2C_Private_macro */

/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Private_define 
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group EI2C_Private_define */

/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Private_define 
 *  @{
 */

/** 
 *  @}
 */ /* End of group EI2C_Private_define */

/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group EI2C_Private_define */

/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Private_typedef
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group EI2C_Private_typedef */

/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Private_variables
 *  @{
 */

/* no define */

/**
 *  @}
 */ /* End of group EI2C_Private_variables */

/*------------------------------------------------------------------------------*/
/*  Const Table                                                                 */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Private_const
 *  @{
 */
/**
 *  @}
 */ /* End of group EI2C_Private_const */

/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Private_functions
 *  @{
 */
 
/* no define */

/**
 *  @}
 */ /* End of group EI2C_Private_functions */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Exported_functions
 *  @{
 */

/*--------------------------------------------------*/
/**
  * @brief     Initializing EI2C Regester
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
void EI2C_init(ei2c_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef __DEBUG__ */
    
    if(EI2C_port_high(p_obj))
    {
        p_obj->p_instance->AEN  = EI2CxAEN_I2CM_DISABLE;
    }

    p_obj->p_instance->ASR1 = EI2CxSR1_CLEAR;

    /* PRS&SCL setting*/
    p_obj->p_instance->APRS = (EI2CxPRS_PRCK & p_obj->init.clock.prs);
    p_obj->p_instance->ASCL = ((EI2CxSCL_SCL_MASK & p_obj->init.clock.sclh)
                            | ((EI2CxSCL_SCL_MASK & p_obj->init.clock.scll)<<8));

    
    /*    other Register init    */
    p_obj->p_instance->AAR1 = EI2CxAR1_INT;
    p_obj->p_instance->AAR2 = EI2CxAR2_INT;    
    p_obj->p_instance->ACR0 = (p_obj->init.cr0.dnf) |(p_obj->init.cr0.nfsel)|(p_obj->init.cr0.toper)|(p_obj->init.cr0.toe)
                            | (p_obj->init.cr0.espe)|(p_obj->init.cr0.este) |(p_obj->init.cr0.nacke)|(p_obj->init.cr0.gce)|(p_obj->init.cr0.ale);
    p_obj->p_instance->AIE  = EI2CxIE_DISABLE;
    
    /* wait for bus free   */
    while((EI2C_port_high(p_obj))==0)
    {
    }

    /* start I2C control */
    p_obj->p_instance->AEN  = EI2CxAEN_I2CM_ENABLE;
}

/*--------------------------------------------------*/
/**
  * @brief     Generate restart condition
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
void EI2C_restartcondition(ei2c_t *p_obj)
{
    __IO uint32_t opreg;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef __DEBUG__ */

    /* set only when mst=1 bb=1   */
    if((EI2C_status_busy(p_obj))
    && (EI2C_master(p_obj)))
    {
        /* restart  condition*/
        opreg = p_obj->p_instance->ACR1;
        opreg &= ~(EI2CxCR1_ST | EI2CxCR1_SP);
        opreg |= EI2CxCR1_RS;
        p_obj->p_instance->ACR1  = opreg;        
    }
}

/*--------------------------------------------------*/
/**
  * @brief     Generate start condition
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
void EI2C_startcondition(ei2c_t *p_obj)
{
    __IO uint32_t opreg;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef __DEBUG__ */

    /* wait for bus free   */
    while((p_obj->p_instance->ASR0&EI2CxSR0_BB)==EI2CxSR0_BB)
    {        
    }
    /* start  condition*/
    opreg = p_obj->p_instance->ACR1;
    opreg &= ~(EI2CxCR1_RS | EI2CxCR1_SP);
    opreg |= EI2CxCR1_ST;
    p_obj->p_instance->ACR1  = opreg;
}

/*--------------------------------------------------*/
/**
  * @brief     Slave mode setting.
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
void EI2C_slave_init(ei2c_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef __DEBUG__ */

    /*    when it is master status    */
    if(p_obj->p_instance->ASR0 & EI2CxSR0_MST )
    {
       p_obj->p_instance->APRS = (EI2CxPRS_PRCK & p_obj->init.clock.prs);
       p_obj->p_instance->ASCL = ((EI2CxSCL_SCL_MASK & p_obj->init.clock.sclh)
                               | ((EI2CxSCL_SCL_MASK & p_obj->init.clock.scll)<<8));
    }
}
#if defined(I2CSxWUP_EN)
/*--------------------------------------------------*/
/**
  * @brief     I2C Wakeup Control setting.
  * @param     p_obj       :I2CS object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
void I2CS_init(I2CS_t *p_obj)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef __DEBUG__ */

    p_obj->p_instance->WUPCR1 = (p_obj->init.wup.sgcdi | p_obj->init.wup.ack | p_obj->init.wup.reset | p_obj->init.wup.intend);
}

/*--------------------------------------------------*/
/**
  * @brief     Primary Slave Address setting.
  * @param     p_obj       :I2CS object.
  * @param     addr        :Primary Slave Address.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
void I2CS_Primary_slave_adr_set(I2CS_t *p_obj, uint32_t adr)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef __DEBUG__ */

    p_obj->p_instance->WUPCR2 = (0x0000000E & adr);
}

/*--------------------------------------------------*/
/**
  * @brief     Secondary Slave Address setting.
  * @param     p_obj       :I2CS object.
  * @param     addr        :Secondary Slave Address.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
void I2CS_Secondary_slave_adr_set(I2CS_t *p_obj, uint32_t adr)
{
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef __DEBUG__
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef __DEBUG__ */

    p_obj->p_instance->WUPCR3 = (0x0000000E & adr);
    p_obj->p_instance->WUPCR3 |= 0x00000001;        /* WUPSA2EN: Secondary Slave Address Use Setting */
}
#endif
/**
 *  @}
 */ /* End of group EI2C_Exported_functions */

/**
 *  @} 
 */ /* End of group EI2C */

/**
 *  @} 
 */ /* End of group Periph_Driver */

#endif /* defined(__EI2C_H)  */

#ifdef __cplusplus
}
#endif /* __cplusplus */

