/** 
 *******************************************************************************
 * @file    ei2c_a.h
 * @brief   This file provides all the functions prototypes for EI2C Class.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
/*------------------------------------------------------------------------------*/
/*  Define to prevent recursive inclusion                                       */
/*------------------------------------------------------------------------------*/
#ifndef __EI2C_H
#define __EI2C_H

#ifdef __cplusplus
extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "driver_com.h"

/**
 *  @addtogroup Periph_Driver Peripheral Driver
 *  @{
 */

/** 
 *  @addtogroup EI2C
 *  @brief      EI2C Driver.
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Private_macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group EI2C_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group EI2C_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Private_define
 *  @{
 */

#ifdef __DEBUG__
/** 
 *  @name  EI2C_NULL Pointer
 *  @brief NULL Pointer.
 *  @{
 */
#define EI2C_NULL                ((void *)0)
/**
 *  @}
 */ /* End of name EI2C_NULL Pointer */
#endif

/** 
 *  @name  EI2CxARST Macro Definition.
 *  @brief EI2CxARST Register Macro Definition.
 *  @{
 */
#define EI2CxARST_SWRES_10       ((uint32_t)0x00000002)    /*!< SWRES=10                */
#define EI2CxARST_SWRES_01       ((uint32_t)0x00000001)    /*!< SWRES=01                */
/**
 *  @}
 */ /* End of name EI2CxARST Macro Definition */

/** 
 *  @name  EI2CxAEN Macro Definition.
 *  @brief EI2CxAEN Register Macro Definition.
 *  @{
 */
#define EI2CxAEN_I2CM_DISABLE    ((uint32_t)0x00000000)     /*!< EI2C Disable */
#define EI2CxAEN_I2CM_ENABLE     ((uint32_t)0x00000001)     /*!< EI2C Enable  */
/**
 *  @}
 */ /* End of name EI2CxAEN Macro Definition */

/** 
 *  @name  EI2CxCR0_DNF Macro Definition.
 *  @brief EI2CxCR0_DNF Register Macro Definition.
 *  @{
 */
#define EI2CxCR0_DNF_1PRSCK      ((uint32_t)0x00000000)     /*!< 1PRSCK                */
#define EI2CxCR0_DNF_2PRSCK      ((uint32_t)0x00001000)     /*!< 2PRSCK                */
#define EI2CxCR0_DNF_3PRSCK      ((uint32_t)0x00002000)     /*!< 3PRSCK                */
#define EI2CxCR0_DNF_4PRSCK      ((uint32_t)0x00003000)     /*!< 4PRSCK                */
#define EI2CxCR0_DNF_5PRSCK      ((uint32_t)0x00004000)     /*!< 5PRSCK                */
#define EI2CxCR0_DNF_6PRSCK      ((uint32_t)0x00005000)     /*!< 6PRSCK                */
#define EI2CxCR0_DNF_NA7         ((uint32_t)0x00006000)     /*!< NA7(the same with 00) */
#define EI2CxCR0_DNF_NA8         ((uint32_t)0x00007000)     /*!< NA8(the same with 00) */

/**
 *  @}
 */ /* End of name EI2CxCR0_DNF Macro Definition */
 
/** 
 *  @name  EI2CxCR0_NFSEL Macro Definition.
 *  @brief EI2CxCR0_NFSEL Register Macro Definition.
 *  @{
 */
#define EI2CxCR0_NFSEL_DIG       ((uint32_t)0x00000000)     /*!< Digtal noise filter   */
#define EI2CxCR0_NFSEL_ANA       ((uint32_t)0x00000800)     /*!< Analog noise filter   */
/**
 *  @}
 */ /* End of name EI2CxCR0_NFSEL Macro Definition */
 
/** 
 *  @name  EI2CxCR0_TOPER Macro Definition.
 *  @brief EI2CxCR0_TOPER Register Macro Definition.
 *  @{
 */
#define EI2CxCR0_TOPER_1PRSCK    ((uint32_t)0x00000000)     /*!< timeout 2*13 PRSCK   */
#define EI2CxCR0_TOPER_2PRSCK    ((uint32_t)0x00000200)     /*!< timeout 2*14 PRSCK   */
#define EI2CxCR0_TOPER_3PRSCK    ((uint32_t)0x00000400)     /*!< timeout 2*15 PRSCK   */
#define EI2CxCR0_TOPER_4PRSCK    ((uint32_t)0x00000600)     /*!< timeout 2*16 PRSCK   */
/**
 *  @}
 */ /* End of name EI2CxCR0_TOPER Macro Definition */
 
 /** 
 *  @name  EI2CxCR0_TOE Macro Definition.
 *  @brief EI2CxCR0_TOE Register Macro Definition.
 *  @{
 */
#define EI2CxCR0_TOE_ENABLE     ((uint32_t)0x00000100)     /*!< timeout Enable        */
#define EI2CxCR0_TOE_DISABLE    ((uint32_t)0x00000000)     /*!< timeout Disable       */
/**
 *  @}
 */ /* End of name EI2CxCR0_TOE Macro Definition */
 
/** 
 *  @name  EI2CxCR0_ESPE Macro Definition.
 *  @brief EI2CxCR0_ESPE Register Macro Definition.
 *  @{
 */
#define EI2CxCR0_ESPE_ENABLE    ((uint32_t)0x00000010)     /*!< error stop condition Enable  */
#define EI2CxCR0_ESPE_DISABLE   ((uint32_t)0x00000000)     /*!< error stop condition Disable */
/**
 *  @}
 */ /* End of name EI2CxCR0_ESPE Macro Definition */

/** 
 *  @name  EI2CxCR0_ESTE Macro Definition.
 *  @brief EI2CxCR0_ESTE Register Macro Definition.
 *  @{
 */
#define EI2CxCR0_ESTE_ENABLE    ((uint32_t)0x00000008)     /*!< error start condition Enable  */
#define EI2CxCR0_ESTE_DISABLE   ((uint32_t)0x00000000)     /*!< error start condition Disable */
/**
 *  @}
 */ /* End of name EI2CxCR0_ESTE Macro Definition */
 
/** 
 *  @name  EI2CxCR0_NACKE Macro Definition.
 *  @brief EI2CxCR0_NACKE Register Macro Definition.
 *  @{
 */
#define EI2CxCR0_NACKE_ENABLE    ((uint32_t)0x00000004)     /*!< NACK Check Enable           */
#define EI2CxCR0_NACKE_DISABLE   ((uint32_t)0x00000000)     /*!< NACK Check Disable          */
/**
 *  @}
 */ /* End of name EI2CxCR0_NACKE Macro Definition */
 
 
/** 
 *  @name  EI2CxCR0_GCE Macro Definition.
 *  @brief EI2CxCR0_GCE Register Macro Definition.
 *  @{
 */
#define EI2CxCR0_GCE_ENABLE      ((uint32_t)0x00000002)     /*!< General call Enable         */
#define EI2CxCR0_GCE_DISABLE     ((uint32_t)0x00000000)     /*!< General call Disable        */
/**
 *  @}
 */ /* End of name EI2CxCR0_GCE Macro Definition */
 
 
/** 
 *  @name  EI2CxCR0_ALE Macro Definition.
 *  @brief EI2CxCR0_ALE Register Macro Definition.
 *  @{
 */
#define EI2CxCR0_ALE_ENABLE      ((uint32_t)0x00000001)     /*!< Arbitration lost Enable     */
#define EI2CxCR0_ALE_DISABLE     ((uint32_t)0x00000000)     /*!< Arbitration lost Disable    */
/**
 *  @}
 */ /* End of name EI2CxCR0_ALE Macro Definition */

/** 
 *  @name  EI2CxCR1 Macro Definition.
 *  @brief EI2CxCR1 Register Macro Definition.
 *  @{
 */
#define EI2CxCR1_OMC             ((uint32_t)0x00000400)     /*!< OMC=1       */
#define EI2CxCR1_ACKWAIT         ((uint32_t)0x00000010)     /*!< ACKWAIT=1   */
#define EI2CxCR1_ACKSEL_NACK     ((uint32_t)0x00000008)     /*!< ACKSEL=NACK */
#define EI2CxCR1_SP              ((uint32_t)0x00000004)     /*!< SP=1        */
#define EI2CxCR1_RS              ((uint32_t)0x00000002)     /*!< RS=1        */
#define EI2CxCR1_ST              ((uint32_t)0x00000001)     /*!< ST=1        */
#define EI2CxCR1_CLEAR           ((uint32_t)0x00000000)     /*!< Clear       */

/**
 *  @}
 */ /* End of name EI2CxCR1 Macro Definition */

/** 
 *  @name  EI2CxDBRT Macro Definition.
 *  @brief EI2CxDBRT Register Macro Definition.
 *  @{
 */
#define EI2CxDBRT_DBT_MASK         ((uint32_t)0x000000FF)      /* !< DB 7-0 bits mask.     */
/**
 *  @}
 */ /* End of name EI2CxDBRT Macro Definition */
 
/** 
 *  @name  EI2CxDBRR Macro Definition.
 *  @brief EI2CxDBRR Register Macro Definition.
 *  @{
 */
#define EI2CxDBRR_DBR_MASK         ((uint32_t)0x000000FF)      /* !< DB 7-0 bits mask.     */
/**
 *  @}
 */ /* End of name EI2CxDBRR Macro Definition */

/** 
 *  @name  EI2CxSR0 Macro Definition.
 *  @brief EI2CxSR0 Register Macro Definition.
 *  @{
 */
#define EI2CxSR0_INT              ((uint32_t)0x00000000)     /*!< Initial Settings.       */
#define EI2CxSR0_MST              ((uint32_t)0x00000008)     /*!< MST                     */
#define EI2CxSR0_TRX              ((uint32_t)0x00000004)     /*!< TRX                     */
#define EI2CxSR0_BB               ((uint32_t)0x00000002)     /*!< BB                      */
#define EI2CxSR0_ACKF             ((uint32_t)0x00000001)     /*!< ACKF                    */
/**
 *  @}
 */ /* End of name EI2CxSR0 Macro Definition */


/** 
 *  @name  EI2CxSR1 Macro Definition.
 *  @brief EI2CxSR1 Register Macro Definition.
 *  @{
 */
#define EI2CxSR1_TOERR            ((uint32_t)0x00002000)     /*!< Timeout error.               */
#define EI2CxSR1_ESP              ((uint32_t)0x00001000)     /*!< Error stop condition         */
#define EI2CxSR1_EST              ((uint32_t)0x00000800)     /*!< Error start condition        */
#define EI2CxSR1_AAS2             ((uint32_t)0x00000400)     /*!< The same NO2 address flag    */
#define EI2CxSR1_AAS1             ((uint32_t)0x00000200)     /*!< The same NO1 address flag    */
#define EI2CxSR1_GC               ((uint32_t)0x00000100)     /*!< General call check flag      */
#define EI2CxSR1_AL               ((uint32_t)0x00000080)     /*!< Arbitration lost flag        */
#define EI2CxSR1_NACK             ((uint32_t)0x00000040)     /*!< NACK check flag              */
#define EI2CxSR1_RBF              ((uint32_t)0x00000020)     /*!< Receive buffer full          */
#define EI2CxSR1_TBE              ((uint32_t)0x00000010)     /*!< Transmit buffer empty        */
#define EI2CxSR1_TEND             ((uint32_t)0x00000008)     /*!< Transmit over flag           */
#define EI2CxSR1_SPCF             ((uint32_t)0x00000004)     /*!< Stop Condition check flag    */
#define EI2CxSR1_RSCF             ((uint32_t)0x00000002)     /*!< ReStart Condition check flag */
#define EI2CxSR1_STCF             ((uint32_t)0x00000001)     /*!< Start Condition check flag   */
#define EI2CxSR1_CLEAR            ((uint32_t)0x00003FFF)     /*!< Clear INT status             */

/**
 *  @}
 */ /* End of name EI2CxSR1 Macro Definition */

/** 
 *  @name  EI2CxPRS Macro Definition.
 *  @brief EI2CxPRS Register Macro Definition.
 *  @{
 */
#define EI2CxPRS_PRCK            ((uint32_t)0x0000001F)     /*!< PRCK                    */
#define EI2CxPRS_MAX             ((uint32_t)64)             /*!< PRCK MAX                */
/**
 *  @}
 */ /* End of name EI2CxPRS Macro Definition */

/** 
 *  @name  EI2CxSCL Macro Definition.
 *  @brief EI2CxSCL Register Macro Definition.
 *  @{
 */
#define EI2CxSCL_SCL_MASK        ((uint32_t)0x000000FF)     /*!< SCLH SCLL MASK          */
#define EI2CxSCL_MAX             ((uint32_t)0xFF)           /*!< SCLH SCLL  Max size     */
/**
 *  @}
 */ /* End of name EI2CxSCL Macro Definition */
 
/** 
 *  @name  EI2CxIE Macro Definition.
 *  @brief EI2CxIE Register Macro Definition.
 *  @{
 */
#define EI2CxIE_DMARX         ((uint32_t)0x00008000)     /*!< Status INT(DMA receive req)       */
#define EI2CxIE_DMATX         ((uint32_t)0x00004000)     /*!< Status INT(DMA transmit req)      */
#define EI2CxIE_INTTOE        ((uint32_t)0x00002000)     /*!< Status INT(timeout)               */
#define EI2CxIE_INTESPE       ((uint32_t)0x00001000)     /*!< Status INT(error stop condition)  */
#define EI2CxIE_INTESTE       ((uint32_t)0x00000800)     /*!< Status INT(error start condition) */
#define EI2CxIE_INTASE        ((uint32_t)0x00000200)     /*!< Status INT(same address)          */
#define EI2CxIE_INTGCE        ((uint32_t)0x00000100)     /*!< Status INT(General call)          */
#define EI2CxIE_INTALE        ((uint32_t)0x00000080)     /*!< Status INT(Arbitration lost)      */
#define EI2CxIE_INTNACKE      ((uint32_t)0x00000040)     /*!< Status INT(NACK)                  */
#define EI2CxIE_INTSPE        ((uint32_t)0x00000004)     /*!< Status INT(stop condition)        */
#define EI2CxIE_INTRSE        ((uint32_t)0x00000002)     /*!< Status INT(restart condition)     */
#define EI2CxIE_INTSTE        ((uint32_t)0x00000001)     /*!< Status INT(start condition)       */
#define EI2CxIE_DISABLE       ((uint32_t)0x00000000)     /*!< Disable INT                       */

/**
 *  @}
 */ /* End of name EI2CxIE Macro Definition */

/** 
 *  @name  EI2CxAR1 Macro Definition.
 *  @brief EI2CxAR1 Register Macro Definition.
 *  @{
 */
#define EI2CxAR1_INT            ((uint32_t)0x00000000)      /*!< Initial Settings.     */
#define EI2CxAR1_SAFS1_7BIT     ((uint32_t)0x00000000)      /*!< 7Bit Address Format.  */
#define EI2CxAR1_SAFS1_10BIT    ((uint32_t)0x00008000)      /*!< 10Bit Address Format. */
#define EI2CxAR1_SAFS1_MASK     ((uint32_t)0x00008000)      /*!< Get Address Format    */
#define EI2CxAR1_SA1_7BITMASK   ((uint32_t)0x000000FE)      /*!< 7Bit Address MASK.    */
#define EI2CxAR1_SA1_10BITMASK  ((uint32_t)0x000007FE)      /*!< 10Bit Address MASK.   */
#define EI2CxAR1_SA1E_ENABLE    ((uint32_t)0x00000001)      /*!< No1 Slave Add Enable. */
#define EI2CxAR1_SA1E_DISABLE   ((uint32_t)0x00000000)      /*!< No1 Slave Add Disable.*/
#define EI2CxAR1_SA1E_MASK      ((uint32_t)0x00000001)      /*!< Get Slave Add Status  */

/**
 *  @}
 */ /* End of name EI2CxAR1 Macro Definition */

/** 
 *  @name  EI2CxAR2 Macro Definition.
 *  @brief EI2CxAR2 Register Macro Definition.
 *  @{
 */
#define EI2CxAR2_INT            ((uint32_t)0x00000000)      /*!< Initial Settings.     */
#define EI2CxAR2_SAFS2_7BIT     ((uint32_t)0x00000000)      /*!< 7Bit Address Format.  */
#define EI2CxAR2_SAFS2_10BIT    ((uint32_t)0x00008000)      /*!< 10Bit Address Format. */
#define EI2CxAR2_SAFS2_MASK     ((uint32_t)0x00008000)      /*!< Get Address Format    */
#define EI2CxAR2_SA2_7BITMASK   ((uint32_t)0x000000FE)      /*!< 7Bit Address MASK.    */
#define EI2CxAR2_SA2_10BITMASK  ((uint32_t)0x000007FE)      /*!< 10Bit Address MASK.   */
#define EI2CxAR2_SA2E_ENABLE    ((uint32_t)0x00000001)      /*!< No2 Slave Add Enable. */
#define EI2CxAR2_SA2E_DISABLE   ((uint32_t)0x00000000)      /*!< No2 Slave Add Disable.*/
#define EI2CxAR2_SA2E_MASK      ((uint32_t)0x00000001)      /*!< Get Slave Add Status  */

/**
 *  @}
 */ /* End of name EI2CxAR2 Macro Definition */
 
/** 
 *  @name  EI2CxPM Macro Definition.
 *  @brief EI2CxPM Register Macro Definition.
 *  @{
 */
#define EI2CxPM_SDA_SCL           ((uint32_t)0x00000003)      /* SDA and SCL level. */
/**
 *  @}
 */ /* End of name EI2CxPM Macro Definition */
/** 
 *  @}
 */ /* End of group EI2C_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group EI2C_Private_define */

/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Private_typedef
 *  @{
 */

/*----------------------------------*/
/** 
 * @brief  Clock setting structure definition.
*/
/*----------------------------------*/
typedef struct
{
    uint32_t               sck;            /*!< Select internal SCL output clock frequency.                 */
    uint32_t               prs;            /*!< Prescaler clock frequency for generating the Serial clock.  */
    uint32_t               sclh;           /*!< Prescaler clock high width.                                 */
    uint32_t               scll;           /*!< Prescaler clock low width.                                  */
} ei2c_clock_t;

/*----------------------------------*/
/** 
 * @struct ei2c_control0_t
 * @brief  Control Setting structure definition.
*/
/*----------------------------------*/
typedef struct
{
    uint32_t dnf;       /*!< Digtal noise filter clear noise time.
                                : Use @ref EI2CxCR0_DNF   */
    uint32_t nfsel;     /*!< noise filter mode Control.
                                : Use @ref EI2CxCR0_NFSEL */
    uint32_t toper;     /*!< timeout check time Control.
                                : Use @ref EI2CxCR0_TOPER */
    uint32_t toe;       /*!< timeout enable setting. 
                                : Use @ref EI2CxCR0_TOE   */
    uint32_t espe;      /*!< stop condition error Enable setting. 
                                : Use @ref EI2CxCR0_ESPE  */
    uint32_t este;      /*!< start condition error Enable setting. 
                                : Use @ref EI2CxCR0_ESTE  */
    uint32_t nacke;     /*!< NACK Check Enable setting. 
                                : Use @ref EI2CxCR0_NACKE */
    uint32_t gce;       /*!< General call Enable Setting.
                                : Use @ref EI2CxCR0_GCE   */
    uint32_t ale;       /*!< Arbitration lost Enable setting.
                                : Use @ref EI2CxCR0_ALE   */

} ei2c_control0_t;

/*----------------------------------*/
/** 
 * @brief  Wakeup Control setting structure definition.
*/
/*----------------------------------*/
typedef struct
{
    uint32_t               sgcdi;          /*!< Select general call detect ON/OFF.    */
    uint32_t               ack;            /*!< Select ACK output.                    */
    uint32_t               reset;          /*!< EI2C BUS Rest.                        */
    uint32_t               intend;         /*!< Interrupt release.                    */
} EI2CS_wup_setting_t;

/*----------------------------------*/
/** 
 * @brief  Initial setting structure definition.
*/
/*----------------------------------*/
typedef struct
{
    ei2c_clock_t          clock;             /*!< Serial clock setting.            */
    ei2c_control0_t       cr0;               /*!< Control setting.                 */
} ei2c_initial_setting_t;

/*----------------------------------*/
/** 
 * @brief  Initial setting structure definition.
*/
/*----------------------------------*/
typedef struct
{
    EI2CS_wup_setting_t    wup;              /*!< Wakeup Control setting.          */
} EI2CS_initial_setting_t;

/*----------------------------------*/
/** 
 * @brief  I2C handle structure definition.
*/
/*----------------------------------*/
typedef struct
{
    TSB_EI2C_TypeDef       *p_instance;     /*!< Registers base address.           */
    ei2c_initial_setting_t  init;           /*!< Initial setting.                  */
} ei2c_t;
#if defined(I2CSxWUP_EN)
/*----------------------------------*/
/** 
 * @brief  I2CS handle structure definition.
*/
/*----------------------------------*/
typedef struct
{
    TSB_I2CS_TypeDef       *p_instance;     /*!< Registers base address.           */
    EI2CS_initial_setting_t init;            /*!< Initial setting.                  */
} EI2CS_t;
#endif
/**
 *  @}
 */ /* End of group EI2C_Private_typedef */

/*------------------------------------------------------------------------------*/
/*  Inline Functions                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Private_functions
 *  @{
 */
__STATIC_INLINE void EI2C_reset(ei2c_t *p_obj);
__STATIC_INLINE int32_t EI2C_port_high(ei2c_t *p_obj);
__STATIC_INLINE void EI2C_stop_condition(ei2c_t *p_obj);
__STATIC_INLINE uint32_t EI2C_read_data(ei2c_t *p_obj);
__STATIC_INLINE void EI2C_write_data(ei2c_t *p_obj, uint32_t data);
__STATIC_INLINE int32_t EI2C_restart(ei2c_t *p_obj);
__STATIC_INLINE void EI2C_set_ack(ei2c_t *p_obj, int32_t nack);
__STATIC_INLINE int32_t EI2C_get_ack(ei2c_t *p_obj);
__STATIC_INLINE int32_t EI2C_status_busy(ei2c_t *p_obj);
__STATIC_INLINE int32_t EI2C_master(ei2c_t *p_obj);
__STATIC_INLINE int32_t EI2C_transmitter(ei2c_t *p_obj);
__STATIC_INLINE int32_t EI2C_int_status(ei2c_t *p_obj);
__STATIC_INLINE void EI2C_clear_int_status(ei2c_t *p_obj);
__STATIC_INLINE void EI2C_enable_interrupt(ei2c_t *p_obj);
__STATIC_INLINE void EI2C_enable_interrupt_dma(ei2c_t *p_obj, int32_t tx);
__STATIC_INLINE void EI2C_disable_interrupt(ei2c_t *p_obj);
__STATIC_INLINE void EI2C_set_address1(ei2c_t *p_obj, uint32_t addr,int32_t bit7);
__STATIC_INLINE int32_t EI2C_slave_detected(ei2c_t *p_obj);
__STATIC_INLINE int32_t EI2C_status_arbitration(ei2c_t *p_obj);

/*--------------------------------------------------*/
/**
  * @brief     EI2C software reset.
  * @param     p_obj       :EI2C object.
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_reset(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        p_obj->p_instance->ARST = EI2CxARST_SWRES_10;
        p_obj->p_instance->ARST = EI2CxARST_SWRES_01;
    }
#else
    p_obj->p_instance->ARST = EI2CxARST_SWRES_10;
    p_obj->p_instance->ARST = EI2CxARST_SWRES_01;
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     EI2C bus port high
  * @param     p_obj       :EI2C object.
  * @retval    true        :SDA and SCL Port High.
  * @retval    false       :Bus Error.
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t EI2C_port_high(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        return (((p_obj->p_instance->APM & EI2CxPM_SDA_SCL) == EI2CxPM_SDA_SCL));
    }
    return (0);
#else
    return (((p_obj->p_instance->APM & EI2CxPM_SDA_SCL) == EI2CxPM_SDA_SCL));
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Generate stop condition.
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_stop_condition(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
       if(p_obj->p_instance->ASR0 & EI2CxSR0_MST == EI2CxSR0_MST )
       {
          p_obj->p_instance->ACR1 |= EI2CxCR1_SP;
       }
    }
#else
    if((p_obj->p_instance->ASR0&EI2CxSR0_MST) == EI2CxSR0_MST )
    {
       p_obj->p_instance->ACR1 |= EI2CxCR1_SP;
    }
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Generate stop condition.
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_acr1_clear(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
       p_obj->p_instance->ACR1 = EI2CxCR1_CLEAR;
    }
#else
    p_obj->p_instance->ACR1 = EI2CxCR1_CLEAR;
#endif
}
/*--------------------------------------------------*/
/**
  * @brief     Read from Data buffer
  * @param     p_obj       :EI2C object.
  * @retval    result      :Read data.
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE uint32_t EI2C_read_data(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        return (p_obj->p_instance->ADBRR & EI2CxDBRR_DBR_MASK);
    }
    return (0);
#else
    return (p_obj->p_instance->ADBRR & EI2CxDBRR_DBR_MASK);
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Write to Data buffer.
  * @param     p_obj       :EI2C object.
  * @param     data        :Write data.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_write_data(ei2c_t *p_obj, uint32_t data)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        p_obj->p_instance->ADBRT = (data & EI2CxDBRT_DBT_MASK);
    }
#else
    p_obj->p_instance->ADBRT = (data & EI2CxDBRT_DBT_MASK);
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Return restart condition
  * @param     p_obj    :EI2C object.
  * @retval    true     :Restart Detected.
  * @retval    false    :Restart Non-Detected.
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t EI2C_restart(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        __IO uint32_t opreg = p_obj->p_instance->ASR1;
        p_obj->p_instance->ASR1 &= ~EI2CxSR1_RSCF;
        return ((opreg & EI2CxSR1_RSCF) == EI2CxSR1_RSCF);
    }
    return (0);
#else
    __IO uint32_t opreg = p_obj->p_instance->ASR1;
    p_obj->p_instance->ASR1 &= ~EI2CxSR1_RSCF;
    return ((opreg & EI2CxSR1_RSCF) == EI2CxSR1_RSCF);
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Set Ack condition
  * @param     p_obj       :EI2C object.
  * @param     nack        :1 NACK, 0 ACK.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_set_ack(ei2c_t *p_obj, int32_t nack)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        if (nack)
        {
            p_obj->p_instance->ACR1 |= EI2CxCR1_ACKWAIT;
        }
        else
        {
            p_obj->p_instance->ACR1 &= ~EI2CxCR1_ACKWAIT;
        }
    }
#else
    if (nack)
    {
        p_obj->p_instance->ACR1 |= EI2CxCR1_ACKWAIT;
    }
    else
    {
        p_obj->p_instance->ACR1 &= ~EI2CxCR1_ACKWAIT;
    }
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Return received Ack condition
  * @param     p_obj    :EI2C object.
  * @retval    true     :NACK Received.
  * @retval    false    :ACK Received.
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t EI2C_get_ack(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        return ((p_obj->p_instance->ASR1 & EI2CxSR1_NACK) == EI2CxSR1_NACK);
    }
    return (0);
#else
    return ((p_obj->p_instance->ASR1 & EI2CxSR1_NACK) == EI2CxSR1_NACK);
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Return Busy condition
  * @param     p_obj       :EI2C object.
  * @retval    true        :EI2C bus busy.
  * @retval    false       :EI2C bus free.
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t EI2C_status_busy(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        return ((p_obj->p_instance->ASR0 & EI2CxSR0_BB) == EI2CxSR0_BB);
    }
    return (0);
#else
    return ((p_obj->p_instance->ASR0 & EI2CxSR0_BB) == EI2CxSR0_BB);
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Return The Master status
  * @param     p_obj       :EI2C object.
  * @retval    true        :Master mode.
  * @retval    false       :Slave mode.
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t EI2C_master(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        return ((p_obj->p_instance->ASR0 & EI2CxSR0_MST) == EI2CxSR0_MST);
    }
    return (0);
#else
    return ((p_obj->p_instance->ASR0 & EI2CxSR0_MST) == EI2CxSR0_MST);
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Return The Transmitter
  * @param     p_obj       :EI2C object.
  * @retval    true        :Transmitter.
  * @retval    false       :Receiver.
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t EI2C_transmitter(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        return ((p_obj->p_instance->ASR0 & EI2CxSR0_TRX) == EI2CxSR0_TRX);
    }
    return (0);
#else
    return ((p_obj->p_instance->ASR0 & EI2CxSR0_TRX) == EI2CxSR0_TRX);
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Interrupt Status
  * @param     p_obj       :EI2C object.
  * @retval    true        :Interrupt Occurred.
  * @retval    false       :No Interrupt Occurred.
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t EI2C_int_status(ei2c_t *p_obj)
{
    /* Check the INTI2CxST status */
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        if((p_obj->p_instance->ASR1 & EI2CxSR1_TOERR)
        || (p_obj->p_instance->ASR1 & EI2CxSR1_ESP)
        || (p_obj->p_instance->ASR1 & EI2CxSR1_EST)
        || (p_obj->p_instance->ASR1 & EI2CxSR1_AAS2)
        || (p_obj->p_instance->ASR1 & EI2CxSR1_AAS1)
        || (p_obj->p_instance->ASR1 & EI2CxSR1_GC)
        || (p_obj->p_instance->ASR1 & EI2CxSR1_AL)
        || (p_obj->p_instance->ASR1 & EI2CxSR1_NACK)
        || (p_obj->p_instance->ASR1 & EI2CxSR1_TEND)
        || (p_obj->p_instance->ASR1 & EI2CxSR1_SPCF)
        || (p_obj->p_instance->ASR1 & EI2CxSR1_RSCF)
        || (p_obj->p_instance->ASR1 & EI2CxSR1_STCF))
        {
           return (1);
        }
        else 
        {
           return (0);
        }
    }
    return (0);
#else
    if((p_obj->p_instance->ASR1 & EI2CxSR1_TOERR)
    || (p_obj->p_instance->ASR1 & EI2CxSR1_ESP)
    || (p_obj->p_instance->ASR1 & EI2CxSR1_EST)
    || (p_obj->p_instance->ASR1 & EI2CxSR1_AAS2)
    || (p_obj->p_instance->ASR1 & EI2CxSR1_AAS1)
    || (p_obj->p_instance->ASR1 & EI2CxSR1_GC)
    || (p_obj->p_instance->ASR1 & EI2CxSR1_AL)
    || (p_obj->p_instance->ASR1 & EI2CxSR1_NACK)
    || (p_obj->p_instance->ASR1 & EI2CxSR1_TEND)
    || (p_obj->p_instance->ASR1 & EI2CxSR1_SPCF)
    || (p_obj->p_instance->ASR1 & EI2CxSR1_RSCF)
    || (p_obj->p_instance->ASR1 & EI2CxSR1_STCF))
    {
       return (1);
    }
    else 
    {
       return (0);
    }
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Interrupt Status Clear
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_clear_int_status(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        p_obj->p_instance->ASR1 = EI2CxSR1_CLEAR;
    }
#else
    p_obj->p_instance->ASR1 = EI2CxSR1_CLEAR;
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Interrupt Status Clear
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_clear_int_stcf(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        p_obj->p_instance->ASR1 = EI2CxSR1_STCF;
    }
#else
    p_obj->p_instance->ASR1 = EI2CxSR1_STCF;
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Interrupt Status Clear
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_clear_int_rscf(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        p_obj->p_instance->ASR1 = EI2CxSR1_RSCF;
    }
#else
    p_obj->p_instance->ASR1 = EI2CxSR1_RSCF;
#endif
}
/*--------------------------------------------------*/
/**
  * @brief     Interrupt Status Clear
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_clear_int_tbe(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        p_obj->p_instance->ASR1 = EI2CxSR1_TBE;
    }
#else
    p_obj->p_instance->ASR1 = EI2CxSR1_TBE;
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Interrupt Status Clear
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_clear_int_tend(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        p_obj->p_instance->ASR1 = EI2CxSR1_TEND;
    }
#else
    p_obj->p_instance->ASR1 = EI2CxSR1_TEND;
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Interrupt Status Clear
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_clear_int_arbitration(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        p_obj->p_instance->ASR1 = EI2CxSR1_AL;
    }
#else
    p_obj->p_instance->ASR1 = EI2CxSR1_AL;
#endif
}
/*--------------------------------------------------*/
/**
  * @brief     Enable Interrupt setting.
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_enable_interrupt(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        p_obj->p_instance->AIE = (EI2CxIE_INTSTE | EI2CxIE_INTRSE | EI2CxIE_INTSPE | EI2CxIE_INTNACKE
         | EI2CxIE_INTALE | EI2CxIE_INTGCE | EI2CxIE_INTASE | EI2CxIE_INTESTE | EI2CxIE_INTESPE | EI2CxIE_INTTOE);
    }
#else
    p_obj->p_instance->AIE = (EI2CxIE_INTSTE | EI2CxIE_INTRSE | EI2CxIE_INTSPE | EI2CxIE_INTNACKE
     | EI2CxIE_INTALE | EI2CxIE_INTGCE | EI2CxIE_INTASE | EI2CxIE_INTESTE | EI2CxIE_INTESPE | EI2CxIE_INTTOE);
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Enable Interrupt setting.
  * @param     p_obj       :EI2C object.
  * @param     tx          :Direction of transfer(1=tx 0=rx).
  * @retval    -
  * @note      For DMA transfer.
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_enable_interrupt_dma(ei2c_t *p_obj, int32_t tx)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        if (tx)
        {
            p_obj->p_instance->AIE = EI2CxIE_DMATX;
        }
        else
        {
            p_obj->p_instance->AIE = EI2CxIE_DMARX;
        }
    }
#else
    if (tx)
    {
        p_obj->p_instance->AIE = EI2CxIE_DMATX;
    }
    else
    {
        p_obj->p_instance->AIE = EI2CxIE_DMARX;
    }
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Disable Interrupt setting.
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_disable_interrupt(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        p_obj->p_instance->AIE = EI2CxIE_DISABLE;
    }
#else
    p_obj->p_instance->AIE = EI2CxIE_DISABLE;
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     Set slave address.
  * @param     p_obj       :EI2C object.
  * @param     addr        :slave address.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_set_address1(ei2c_t *p_obj, uint32_t addr,int32_t bit7)
{
    __IO uint32_t aar1;
    
    if(bit7)
    {
        aar1 = p_obj->p_instance->AAR1;
        /*clear 7bit address*/
        aar1 &=(~EI2CxAR1_SA1_7BITMASK);
        /* 7bit address */
        aar1 |= EI2CxAR1_SAFS1_7BIT;
        /* set address */
        aar1 |= (addr & EI2CxAR1_SA1_7BITMASK);
        /* slave enable */
        aar1 |= EI2CxAR1_SA1E_ENABLE;
        
        p_obj->p_instance->AAR1 = aar1;
    }
    else
    {
        aar1 = p_obj->p_instance->AAR1;
        /*clear 10bit address*/
        aar1 &=(~EI2CxAR1_SA1_10BITMASK);
        /* 10bit address */
        aar1 |= EI2CxAR1_SAFS1_10BIT;
        /* set address */
        aar1 |= (addr & EI2CxAR1_SA1_10BITMASK);
        /* slave enable */
        aar1 |= EI2CxAR1_SA1E_ENABLE;
        
        p_obj->p_instance->AAR1 = aar1;        
    }
}

/*--------------------------------------------------*/
/**
  * @brief     Set slave address.
  * @param     p_obj       :EI2C object.
  * @param     addr        :slave address.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE void EI2C_set_address2(ei2c_t *p_obj, uint32_t addr,int32_t bit7)
{
    __IO uint32_t aar2;
    
    if(bit7)
    {
        aar2 = p_obj->p_instance->AAR2;
        /*clear 7bit address*/
        aar2 &=(~EI2CxAR2_SAFS2_7BIT);
        /* 7bit address */
        aar2 |= EI2CxAR2_SAFS2_7BIT;
        /* set address */
        aar2 |= (addr & EI2CxAR2_SAFS2_7BIT);
        /* slave enable */
        aar2 |= EI2CxAR2_SA2E_ENABLE;
        
        p_obj->p_instance->AAR2 = aar2;
    }
    else
    {
        aar2 = p_obj->p_instance->AAR2;
        /*clear 10bit address*/
        aar2 &=(~EI2CxAR2_SA2_10BITMASK);
        /* 10bit address */
        aar2 |= EI2CxAR2_SAFS2_10BIT;
        /* set address */
        aar2 |= (addr & EI2CxAR2_SA2_10BITMASK);
        /* slave enable */
        aar2 |= EI2CxAR2_SA2E_ENABLE;
        
        p_obj->p_instance->AAR2 = aar2;        
    }
}

/*--------------------------------------------------*/
/**
  * @brief     Detecting Slave Address
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t EI2C_slave_detected(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        return (((p_obj->p_instance->ASR1 & EI2CxSR1_AAS1) == EI2CxSR1_AAS1)
                 ||((p_obj->p_instance->ASR1 & EI2CxSR1_AAS2) == EI2CxSR1_AAS2));
    }
    return (0);
#else
    return (((p_obj->p_instance->ASR1 & EI2CxSR1_AAS1) == EI2CxSR1_AAS1)
             ||((p_obj->p_instance->ASR1 & EI2CxSR1_AAS2) == EI2CxSR1_AAS2));
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     status arbitration
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t EI2C_status_arbitration(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        return ((p_obj->p_instance->ASR1 & EI2CxSR1_AL) == EI2CxSR1_AL);
    }
    return (0);
#else
    return ((p_obj->p_instance->ASR1 & EI2CxSR1_AL) == EI2CxSR1_AL);
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     status stcf
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t EI2C_status_stcf(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        return ((p_obj->p_instance->ASR1 & EI2CxSR1_STCF) == EI2CxSR1_STCF);
    }
    return (0);
#else
    return ((p_obj->p_instance->ASR1 & EI2CxSR1_STCF) == EI2CxSR1_STCF);
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     status rscf
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t EI2C_status_rscf(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        return ((p_obj->p_instance->ASR1 & EI2CxSR1_RSCF) == EI2CxSR1_RSCF);
    }
    return (0);
#else
    return ((p_obj->p_instance->ASR1 & EI2CxSR1_RSCF) == EI2CxSR1_RSCF);
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     status stcf
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t EI2C_status_tbe(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        return ((p_obj->p_instance->ASR1 & EI2CxSR1_TBE) == EI2CxSR1_TBE);
    }
    return (0);
#else
    return ((p_obj->p_instance->ASR1 & EI2CxSR1_TBE) == EI2CxSR1_TBE);
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     status stcf
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t EI2C_status_tend(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        return ((p_obj->p_instance->ASR1 & EI2CxSR1_TEND) == EI2CxSR1_TEND);
    }
    return (0);
#else
    return ((p_obj->p_instance->ASR1 & EI2CxSR1_TEND) == EI2CxSR1_TEND);
#endif
}

/*--------------------------------------------------*/
/**
  * @brief     status spcf
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t EI2C_status_spcf(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        return ((p_obj->p_instance->ASR1 & EI2CxSR1_SPCF) == EI2CxSR1_SPCF);
    }
    return (0);
#else
    return ((p_obj->p_instance->ASR1 & EI2CxSR1_SPCF) == EI2CxSR1_SPCF);
#endif
}
/*--------------------------------------------------*/
/**
  * @brief     status toerr
  * @param     p_obj       :EI2C object.
  * @retval    -
  * @note      -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t EI2C_status_toerr(ei2c_t *p_obj)
{
#ifdef __DEBUG__
    if ((p_obj != EI2C_NULL) && (p_obj->p_instance != EI2C_NULL))
    {
        return ((p_obj->p_instance->ASR1 & EI2CxSR1_TOERR) == EI2CxSR1_TOERR);
    }
    return (0);
#else
    return ((p_obj->p_instance->ASR1 & EI2CxSR1_TOERR) == EI2CxSR1_TOERR);
#endif
}
/**
 *  @}
 */ /* End of group EI2C_Private_functions */

/*------------------------------------------------------------------------------*/
/*  Functions                                                                   */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup EI2C_Private_functions
 *  @{
 */
void EI2C_init(ei2c_t *p_obj);
void EI2C_restartcondition(ei2c_t *p_obj);
void EI2C_startcondition(ei2c_t *p_obj);
void EI2C_slave_init(ei2c_t *p_obj);
#if defined(I2CSxWUP_EN)
void I2CS_init(I2CS_t *p_obj);
void I2CS_Primary_slave_adr_set(I2CS_t *p_obj, uint32_t adr);
void I2CS_Secondary_slave_adr_set(I2CS_t *p_obj, uint32_t adr);
#endif
/** 
 *  @}
 */ /* End of group EI2C_Private_functions */

/**
 *  @}
 */ /* End of group EI2C */

/**
 *  @} 
 */ /* End of group Periph_Driver */

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* __EI2C_H */


