/**
 *******************************************************************************
 * @file    except_m3h1.c
 * @brief   This file provides API functions for EXCEPT driver.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */

#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "except.h"

#if defined(__EXCEPT_M3H1_H)
/**
 *  @addtogroup Periph_Driver
 *  @{
 */

/** 
 *  @addtogroup EXCEPT
 *  @brief      EXCEPT Driver.
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EXCEPT_Private_define EXCEPT Private Define
 *  @{
 */
/* no define */
/** 
 *  @}
 */ /* End of group EXCEPT_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EXCEPT_Private_define EXCEPT Private Define
 *  @{
 */
#define EXCEPT_INTEN_ENABLE             ((uint8_t)0x01)             /*!< INTEN enable */
#define EXCEPT_INTEN_DISABLE            ((uint8_t)0x00)             /*!< INTEN disable */

#define EXCEPT_INTPCLR                  ((uint8_t)0x40)             /*!< INTP clear */
#define EXCEPT_INTNCLR                  ((uint8_t)0x80)             /*!< INTN clear */

#define RESET_CLEAR_LVD                 ((uint8_t)0xBF)             /*!< RESET clear LVD */
#define RESET_CLEAR_STOP2               ((uint8_t)0xDF)             /*!< RESET clear STOP2 */
#define RESET_CLEAR_PINR                ((uint8_t)0xF7)             /*!< RESET clear PINR */
#define RESET_CLEAR_POR                 ((uint8_t)0xFE)             /*!< RESET clear POR */
#define RESET_CLEAR_OFD                 ((uint8_t)0xF7)             /*!< RESET clear OFD */
#define RESET_CLEAR_WDT                 ((uint8_t)0xFB)             /*!< RESET clear WDT */
#define RESET_CLEAR_LOCK                ((uint8_t)0xFD)             /*!< RESET clear LOCK */
#define RESET_CLEAR_SYS                 ((uint8_t)0xFE)             /*!< RESET clear SYS */

/** 
 *  @}
 */ /* End of group EXCEPT_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EXCEPT_Private_define EXCEPT Private Define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group EXCEPT_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EXCEPT_Private_typedef EXCEPT Private Typedef
 *  @{
 */

/* no define */

/**
 *  @}
 */ /* End of group EXCEPT_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EXCEPT_Private_fuctions EXCEPT Private Fuctions
 *  @{
 */

/* no define */

/**
 *  @}
 */ /* End of group EXCEPT_Private_functions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup EXCEPT_Exported_functions EXCEPT Exported Functions
 *  @{
 */

/*--------------------------------------------------*/
/**
  * @brief     Set intteruput.
  * @param     p_obj : EXCEPT object.
  * @param     int   : Select interrupt.
  * @param     state : Interrupt enable/disable.
  * @param     level : Interrupt detect level.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void except_set_int(except_t *p_obj, uint32_t interrupt, TXZ_FunctionalState state, uint32_t level)
{
    uint8_t regval = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the EXCEPT_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance1));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance2));
#endif /* #ifdef DEBUG */

    /* Set interrupt. */
    switch (interrupt)
    {
    case EXCEPT_LVD:                  /*!< Interrupt LVD                                                            */
        regval = 0x05;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance1->NIC00 = regval;
        break;
    case EXCEPT_WDT:                /*!< Interrupt WDT                                                          */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->NIC00 = regval;
        break;
    case EXCEPT_INT00:                  /*!< Interrupt Pin 00                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance1->IMC00 = regval;
        break;
    case EXCEPT_INT01:                  /*!< Interrupt Pin 01                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance1->IMC01 = regval;
        break;
#if !defined(TMPM3HLA)
    case EXCEPT_INT02:                  /*!< Interrupt Pin 02                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance1->IMC02 = regval;
        break;
    case EXCEPT_INT13:                  /*!< Interrupt Pin 13                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance1->IMC03 = regval;
        break;
#endif
    case EXCEPT_I2CWUP:                 /*!< Serial bus interface (WakeUp) interrupt (channel 0)                    */
        regval = 0x06;
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance1->IMC16 = regval;
        break;
    case EXCEPT_RTC:                    /*!< Real time clock interrupt                                              */
        regval = 0x04;
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance1->IMC17 = regval;
        break;
    case EXCEPT_INTDMAATC00:            /*!< DMAA end of transfer 00                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC000 = regval;
        break;
    case EXCEPT_INTDMAATC01:          /*!< DMAA end of transfer 01                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC001 = regval;
        break;
    case EXCEPT_INTDMAATC02:          /*!< DMAA end of transfer 02                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC002 = regval;
        break;
    case EXCEPT_INTDMAATC03:          /*!< DMAA end of transfer 03                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC003 = regval;
        break;
    case EXCEPT_INTDMAATC04:          /*!< DMAA end of transfer 04                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC004 = regval;
        break;
    case EXCEPT_INTDMAATC05:          /*!< DMAA end of transfer 05                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC005 = regval;
        break;
    case EXCEPT_INTDMAATC06:          /*!< DMAA end of transfer 06                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC006 = regval;
        break;
    case EXCEPT_INTDMAATC07:          /*!< DMAA end of transfer 07                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC007 = regval;
        break;
    case EXCEPT_INTDMAATC08:          /*!< DMAA end of transfer 08                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC008 = regval;
        break;
    case EXCEPT_INTDMAATC09:          /*!< DMAA end of transfer 09                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC009 = regval;
        break;
    case EXCEPT_INTDMAATC10:          /*!< DMAA end of transfer 10                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC010 = regval;
        break;
    case EXCEPT_INTDMAATC11:          /*!< DMAA end of transfer 11                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC011 = regval;
        break;
    case EXCEPT_INTDMAATC12:          /*!< DMAA end of transfer 12                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC012 = regval;
        break;
    case EXCEPT_INTDMAATC13:          /*!< DMAA end of transfer 13                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC013 = regval;
        break;
    case EXCEPT_INTDMAATC14:          /*!< DMAA end of transfer 14                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC014 = regval;
        break;
    case EXCEPT_INTDMAATC15:          /*!< DMAA end of transfer 15                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC015 = regval;
        break;
    case EXCEPT_INTDMAATC16:          /*!< DMAA end of transfer 16                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC016 = regval;
        break;
    case EXCEPT_INTDMAATC17:          /*!< DMAA end of transfer 17                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC017 = regval;
        break;
    case EXCEPT_INTDMAATC18:          /*!< DMAA end of transfer 18                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC018 = regval;
        break;
    case EXCEPT_INTDMAATC19:          /*!< DMAA end of transfer 19                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC019 = regval;
        break;
    case EXCEPT_INTDMAATC20:          /*!< DMAA end of transfer 20                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC020 = regval;
        break;
    case EXCEPT_INTDMAATC21:          /*!< DMAA end of transfer 21                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC021 = regval;
        break;
    case EXCEPT_INTDMAATC22:          /*!< DMAA end of transfer 22                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC022 = regval;
        break;
    case EXCEPT_INTDMAATC23:          /*!< DMAA end of transfer 23                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC023 = regval;
        break;
    case EXCEPT_INTDMAATC24:          /*!< DMAA end of transfer 24                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC024 = regval;
        break;
    case EXCEPT_INTDMAATC25:          /*!< DMAA end of transfer 25                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC025 = regval;
        break;
    case EXCEPT_INTDMAATC26:          /*!< DMAA end of transfer 26                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC026 = regval;
        break;
    case EXCEPT_INTDMAATC27:          /*!< DMAA end of transfer 27                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC027 = regval;
        break;
    case EXCEPT_INTDMAATC28:          /*!< DMAA end of transfer 28                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC028 = regval;
        break;
    case EXCEPT_INTDMAATC29:          /*!< DMAA end of transfer 29                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC029 = regval;
        break;
    case EXCEPT_INTDMAATC30:          /*!< DMAA end of transfer 30                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC030 = regval;
        break;
    case EXCEPT_INTDMAATC31:          /*!< DMAA end of transfer 31                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC031 = regval;
        break;
    case EXCEPT_INTDMAAERR:             /*!< DMAA transfer error                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC032 = regval;
        break;
    case EXCEPT_INTDMABTC00:            /*!< DMAB end of transfer 00                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC033 = regval;
        break;
    case EXCEPT_INTDMABTC01:            /*!< DMAB end of transfer 01                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC034 = regval;
        break;
    case EXCEPT_INTDMABTC02:            /*!< DMAB end of transfer 02                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC035 = regval;
        break;
    case EXCEPT_INTDMABTC03:            /*!< DMAB end of transfer 03                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC036 = regval;
        break;
    case EXCEPT_INTDMABTC04:            /*!< DMAB end of transfer 04                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC037 = regval;
        break;
    case EXCEPT_INTDMABTC05:            /*!< DMAB end of transfer 05                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC038 = regval;
        break;
    case EXCEPT_INTDMABTC06:            /*!< DMAB end of transfer 06                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC039 = regval;
        break;
    case EXCEPT_INTDMABTC07:            /*!< DMAB end of transfer 07                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC040 = regval;
        break;
    case EXCEPT_INTDMABTC08:            /*!< DMAB end of transfer 08                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC041 = regval;
        break;
    case EXCEPT_INTDMABTC09:            /*!< DMAB end of transfer 09                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC042 = regval;
        break;
    case EXCEPT_INTDMABTC10:            /*!< DMAB end of transfer 10                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC043 = regval;
        break;
    case EXCEPT_INTDMABTC11:            /*!< DMAB end of transfer 11                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC044 = regval;
        break;
    case EXCEPT_INTDMABTC12:            /*!< DMAB end of transfer 12                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC045 = regval;
        break;
    case EXCEPT_INTDMABTC13:            /*!< DMAB end of transfer 13                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC046 = regval;
        break;
    case EXCEPT_INTDMABTC14:            /*!< DMAB end of transfer 14                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC047 = regval;
        break;
    case EXCEPT_INTDMABTC15:            /*!< DMAB end of transfer 15                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC048 = regval;
        break;
    case EXCEPT_INTDMABTC16:            /*!< DMAB end of transfer 16                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC049 = regval;
        break;
    case EXCEPT_INTDMABTC17:            /*!< DMAB end of transfer 17                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC050 = regval;
        break;
    case EXCEPT_INTDMABTC18:            /*!< DMAB end of transfer 18                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC051 = regval;
        break;
    case EXCEPT_INTDMABTC19:            /*!< DMAB end of transfer 19                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC052 = regval;
        break;
    case EXCEPT_INTDMABTC20:            /*!< DMAB end of transfer 20                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC053 = regval;
        break;
    case EXCEPT_INTDMABTC21:            /*!< DMAB end of transfer 21                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC054 = regval;
        break;
    case EXCEPT_INTDMABTC22:            /*!< DMAB end of transfer 22                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC055 = regval;
        break;
    case EXCEPT_INTDMABTC23:            /*!< DMAB end of transfer 23                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC056 = regval;
        break;
    case EXCEPT_INTDMABTC24:            /*!< DMAB end of transfer 24                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC057 = regval;
        break;
    case EXCEPT_INTDMABTC25:            /*!< DMAB end of transfer 25                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC058 = regval;
        break;
    case EXCEPT_INTDMABTC26:            /*!< DMAB end of transfer 26                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC059 = regval;
        break;
    case EXCEPT_INTDMABTC27:            /*!< DMAB end of transfer 27                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC060 = regval;
        break;
    case EXCEPT_INTDMABTC28:            /*!< DMAB end of transfer 28                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC061 = regval;
        break;
    case EXCEPT_INTDMABTC29:            /*!< DMAB end of transfer 29                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC062 = regval;
        break;
    case EXCEPT_INTDMABTC30:            /*!< DMAB end of transfer 30                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC063 = regval;
        break;
    case EXCEPT_INTDMABTC31:            /*!< DMAB end of transfer 31                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC064 = regval;
        break;
    case EXCEPT_INTDMABERR:             /*!< DMAB transfer error                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC065 = regval;
        break;
    case EXCEPT_INT03:                  /*!< Interrupt Pin 03                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC066 = regval;
        break;
    case EXCEPT_INT04:                  /*!< Interrupt Pin 04                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC067 = regval;
        break;
    case EXCEPT_INT05:                  /*!< Interrupt Pin 05                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC068 = regval;
        break;
    case EXCEPT_INT06:                  /*!< Interrupt Pin 06                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC069 = regval;
        break;
    case EXCEPT_INT07:                  /*!< Interrupt Pin 07                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC070 = regval;
        break;
    case EXCEPT_INT08:                  /*!< Interrupt Pin 08                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC071 = regval;
        break;
#if !defined(TMPM3HLA)
    case EXCEPT_INT09:                  /*!< Interrupt Pin 09                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC072 = regval;
        break;
#endif
    case EXCEPT_INT10:              /*!< Interrupt Pin 10                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC073 = regval;
        break;
    case EXCEPT_INT11:                  /*!< Interrupt Pin 11                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC074 = regval;
        break;
    case EXCEPT_INT12:                  /*!< Interrupt Pin 12                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC075 = regval;
        break;
    case EXCEPT_INT14:                  /*!< Interrupt Pin 14                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC076 = regval;
        break;
#if !defined(TMPM3HLA) && !defined(TMPM3HMA)
    case EXCEPT_INT15:                  /*!< Interrupt Pin 15                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC077 = regval;
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA)
    case EXCEPT_INT16:                  /*!< Interrupt Pin 16                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC078 = regval;
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA) && !defined(TMPM3HNA)
    case EXCEPT_INT17:              /*!< Interrupt Pin 17                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC079 = regval;
        break;
    case EXCEPT_INT18:              /*!< Interrupt Pin 18                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC080 = regval;
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA) && !defined(TMPM3HNA)
    case EXCEPT_INT19:                  /*!< Interrupt Pin 19                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC081 = regval;
        break;
    case EXCEPT_INT20:                  /*!< Interrupt Pin 20                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC082 = regval;
        break;
    case EXCEPT_INT21:                  /*!< Interrupt Pin 21                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC083 = regval;
        break;
    case EXCEPT_INT22:                  /*!< Interrupt Pin 22                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC084 = regval;
        break;
    case EXCEPT_INT23:                  /*!< Interrupt Pin 23                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC085 = regval;
        break;
    case EXCEPT_INT24:                  /*!< Interrupt Pin 24                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC086 = regval;
        break;
    case EXCEPT_INT25:                  /*!< Interrupt Pin 25                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC087 = regval;
        break;
    case EXCEPT_INT26:                  /*!< Interrupt Pin 26                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC088 = regval;
        break;
    case EXCEPT_INT27:                  /*!< Interrupt Pin 27                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC089 = regval;
        break;
    case EXCEPT_INT28:                  /*!< Interrupt Pin 28                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC090 = regval;
        break;
#endif
#if defined(TMPM3HQA)
    case EXCEPT_INT29:                  /*!< Interrupt Pin 29                                                       */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC091 = regval;
        break;
    case EXCEPT_INT30:                  /*!< Interrupt Pin 30                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC092 = regval;
        break;
    case EXCEPT_INT31:                  /*!< Interrupt Pin 31                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC093 = regval;
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA)
    case EXCEPT_INT32:              /*!< Interrupt Pin 32                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC095 = regval;
        break;
    case EXCEPT_INT33:              /*!< Interrupt Pin 33                                                           */
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= level;
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC096 = regval;
        break;
#endif
    case EXCEPT_INTRMC0:                /*!< Remote control reception interrupt                                     */
        regval = 0x06;
        if(state == TXZ_ENABLE){
            regval = EXCEPT_INTEN_ENABLE;
        }
        else{
            regval = EXCEPT_INTEN_DISABLE;
        }
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC094 = regval;
        break;
    default:
        break;
    }
}

/*--------------------------------------------------*/
/**
  * @brief     Clear interrupt.
  * @param     p_obj : EXCEPT object.
  * @param     int   : Select interrupt.
  * @retval    -
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void except_clear_int(except_t *p_obj, uint32_t interrupt)
{
    uint8_t regval = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the EXCEPT_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance1));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance2));
#endif /* #ifdef DEBUG */

    /* Set interrupt. */
    switch (interrupt)
    {
    case EXCEPT_LVD:                  /*!< Interrupt LVD                                                            */
        regval = 0x05;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance1->NIC00 = regval;
        break;
    case EXCEPT_WDT:                /*!< Interrupt WDT                                                          */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->NIC00 = regval;
        break;
    case EXCEPT_INT00:                  /*!< Interrupt Pin 00                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance1->IMC00 = regval;
        break;
    case EXCEPT_INT01:                  /*!< Interrupt Pin 01                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance1->IMC01 = regval;
        break;
#if !defined(TMPM3HLA)
    case EXCEPT_INT02:                  /*!< Interrupt Pin 02                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance1->IMC02 = regval;
        break;
    case EXCEPT_INT13:                  /*!< Interrupt Pin 13                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance1->IMC03 = regval;
        break;
#endif
    case EXCEPT_I2CWUP:                 /*!< Serial bus interface (WakeUp) interrupt (channel 0)                    */
        regval = 0x06;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance1->IMC16 = regval;
        break;
    case EXCEPT_RTC:                    /*!< Real time clock interrupt                                              */
        regval = 0x04;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance1->IMC17 = regval;
        break;
    case EXCEPT_INTDMAATC00:            /*!< DMAA end of transfer 00                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC000 = regval;
        break;
    case EXCEPT_INTDMAATC01:          /*!< DMAA end of transfer 01                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC001 = regval;
        break;
    case EXCEPT_INTDMAATC02:          /*!< DMAA end of transfer 02                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC002 = regval;
        break;
    case EXCEPT_INTDMAATC03:          /*!< DMAA end of transfer 03                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC003 = regval;
        break;
    case EXCEPT_INTDMAATC04:          /*!< DMAA end of transfer 04                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC004 = regval;
        break;
    case EXCEPT_INTDMAATC05:          /*!< DMAA end of transfer 05                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC005 = regval;
        break;
    case EXCEPT_INTDMAATC06:          /*!< DMAA end of transfer 06                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC006 = regval;
        break;
    case EXCEPT_INTDMAATC07:          /*!< DMAA end of transfer 07                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC007 = regval;
        break;
    case EXCEPT_INTDMAATC08:          /*!< DMAA end of transfer 08                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC008 = regval;
        break;
    case EXCEPT_INTDMAATC09:          /*!< DMAA end of transfer 09                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC009 = regval;
        break;
    case EXCEPT_INTDMAATC10:          /*!< DMAA end of transfer 10                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC010 = regval;
        break;
    case EXCEPT_INTDMAATC11:          /*!< DMAA end of transfer 11                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC011 = regval;
        break;
    case EXCEPT_INTDMAATC12:          /*!< DMAA end of transfer 12                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC012 = regval;
        break;
    case EXCEPT_INTDMAATC13:          /*!< DMAA end of transfer 13                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC013 = regval;
        break;
    case EXCEPT_INTDMAATC14:          /*!< DMAA end of transfer 14                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC014 = regval;
        break;
    case EXCEPT_INTDMAATC15:          /*!< DMAA end of transfer 15                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC015 = regval;
        break;
    case EXCEPT_INTDMAATC16:          /*!< DMAA end of transfer 16                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC016 = regval;
        break;
    case EXCEPT_INTDMAATC17:          /*!< DMAA end of transfer 17                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC017 = regval;
        break;
    case EXCEPT_INTDMAATC18:          /*!< DMAA end of transfer 18                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC018 = regval;
        break;
    case EXCEPT_INTDMAATC19:          /*!< DMAA end of transfer 19                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC019 = regval;
        break;
    case EXCEPT_INTDMAATC20:          /*!< DMAA end of transfer 20                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC020 = regval;
        break;
    case EXCEPT_INTDMAATC21:          /*!< DMAA end of transfer 21                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC021 = regval;
        break;
    case EXCEPT_INTDMAATC22:          /*!< DMAA end of transfer 22                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC022 = regval;
        break;
    case EXCEPT_INTDMAATC23:          /*!< DMAA end of transfer 23                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC023 = regval;
        break;
    case EXCEPT_INTDMAATC24:          /*!< DMAA end of transfer 24                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC024 = regval;
        break;
    case EXCEPT_INTDMAATC25:          /*!< DMAA end of transfer 25                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC025 = regval;
        break;
    case EXCEPT_INTDMAATC26:          /*!< DMAA end of transfer 26                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC026 = regval;
        break;
    case EXCEPT_INTDMAATC27:          /*!< DMAA end of transfer 27                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC027 = regval;
        break;
    case EXCEPT_INTDMAATC28:          /*!< DMAA end of transfer 28                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC028 = regval;
        break;
    case EXCEPT_INTDMAATC29:          /*!< DMAA end of transfer 29                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC029 = regval;
        break;
    case EXCEPT_INTDMAATC30:          /*!< DMAA end of transfer 30                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC030 = regval;
        break;
    case EXCEPT_INTDMAATC31:          /*!< DMAA end of transfer 31                                                      */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC031 = regval;
        break;
    case EXCEPT_INTDMAAERR:             /*!< DMAA transfer error                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC032 = regval;
        break;
    case EXCEPT_INTDMABTC00:            /*!< DMAB end of transfer 00                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC033 = regval;
        break;
    case EXCEPT_INTDMABTC01:            /*!< DMAB end of transfer 01                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC034 = regval;
        break;
    case EXCEPT_INTDMABTC02:            /*!< DMAB end of transfer 02                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC035 = regval;
        break;
    case EXCEPT_INTDMABTC03:            /*!< DMAB end of transfer 03                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC036 = regval;
        break;
    case EXCEPT_INTDMABTC04:            /*!< DMAB end of transfer 04                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC037 = regval;
        break;
    case EXCEPT_INTDMABTC05:            /*!< DMAB end of transfer 05                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC038 = regval;
        break;
    case EXCEPT_INTDMABTC06:            /*!< DMAB end of transfer 06                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC039 = regval;
        break;
    case EXCEPT_INTDMABTC07:            /*!< DMAB end of transfer 07                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC040 = regval;
        break;
    case EXCEPT_INTDMABTC08:            /*!< DMAB end of transfer 08                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC041 = regval;
        break;
    case EXCEPT_INTDMABTC09:            /*!< DMAB end of transfer 09                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC042 = regval;
        break;
    case EXCEPT_INTDMABTC10:            /*!< DMAB end of transfer 10                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC043 = regval;
        break;
    case EXCEPT_INTDMABTC11:            /*!< DMAB end of transfer 11                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC044 = regval;
        break;
    case EXCEPT_INTDMABTC12:            /*!< DMAB end of transfer 12                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC045 = regval;
        break;
    case EXCEPT_INTDMABTC13:            /*!< DMAB end of transfer 13                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC046 = regval;
        break;
    case EXCEPT_INTDMABTC14:            /*!< DMAB end of transfer 14                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC047 = regval;
        break;
    case EXCEPT_INTDMABTC15:            /*!< DMAB end of transfer 15                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC048 = regval;
        break;
    case EXCEPT_INTDMABTC16:            /*!< DMAB end of transfer 16                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC049 = regval;
        break;
    case EXCEPT_INTDMABTC17:            /*!< DMAB end of transfer 17                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC050 = regval;
        break;
    case EXCEPT_INTDMABTC18:            /*!< DMAB end of transfer 18                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC051 = regval;
        break;
    case EXCEPT_INTDMABTC19:            /*!< DMAB end of transfer 19                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC052 = regval;
        break;
    case EXCEPT_INTDMABTC20:            /*!< DMAB end of transfer 20                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC053 = regval;
        break;
    case EXCEPT_INTDMABTC21:            /*!< DMAB end of transfer 21                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC054 = regval;
        break;
    case EXCEPT_INTDMABTC22:            /*!< DMAB end of transfer 22                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC055 = regval;
        break;
    case EXCEPT_INTDMABTC23:            /*!< DMAB end of transfer 23                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC056 = regval;
        break;
    case EXCEPT_INTDMABTC24:            /*!< DMAB end of transfer 24                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC057 = regval;
        break;
    case EXCEPT_INTDMABTC25:            /*!< DMAB end of transfer 25                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC058 = regval;
        break;
    case EXCEPT_INTDMABTC26:            /*!< DMAB end of transfer 26                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC059 = regval;
        break;
    case EXCEPT_INTDMABTC27:            /*!< DMAB end of transfer 27                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC060 = regval;
        break;
    case EXCEPT_INTDMABTC28:            /*!< DMAB end of transfer 28                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC061 = regval;
        break;
    case EXCEPT_INTDMABTC29:            /*!< DMAB end of transfer 29                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC062 = regval;
        break;
    case EXCEPT_INTDMABTC30:            /*!< DMAB end of transfer 30                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC063 = regval;
        break;
    case EXCEPT_INTDMABTC31:            /*!< DMAB end of transfer 31                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC064 = regval;
        break;
    case EXCEPT_INTDMABERR:             /*!< DMAB transfer error                                                    */
        regval = 0x07;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC065 = regval;
        break;
    case EXCEPT_INT03:                  /*!< Interrupt Pin 03                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC066 = regval;
        break;
    case EXCEPT_INT04:                  /*!< Interrupt Pin 04                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC067 = regval;
        break;
    case EXCEPT_INT05:                  /*!< Interrupt Pin 05                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC068 = regval;
        break;
    case EXCEPT_INT06:                  /*!< Interrupt Pin 06                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC069 = regval;
        break;
    case EXCEPT_INT07:                  /*!< Interrupt Pin 07                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC070 = regval;
        break;
    case EXCEPT_INT08:                  /*!< Interrupt Pin 08                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC071 = regval;
        break;
#if !defined(TMPM3HLA)
    case EXCEPT_INT09:                  /*!< Interrupt Pin 09                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC072 = regval;
        break;
#endif
    case EXCEPT_INT10:              /*!< Interrupt Pin 10                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC073 = regval;
        break;
    case EXCEPT_INT11:                  /*!< Interrupt Pin 11                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC074 = regval;
        break;
    case EXCEPT_INT12:                  /*!< Interrupt Pin 12                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC075 = regval;
        break;
    case EXCEPT_INT14:                  /*!< Interrupt Pin 14                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC076 = regval;
        break;
#if !defined(TMPM3HLA) && !defined(TMPM3HMA)
    case EXCEPT_INT15:                  /*!< Interrupt Pin 15                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC077 = regval;
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA)
    case EXCEPT_INT16:                  /*!< Interrupt Pin 16                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC078 = regval;
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA) && !defined(TMPM3HNA)
    case EXCEPT_INT17:              /*!< Interrupt Pin 17                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC079 = regval;
        break;
    case EXCEPT_INT18:              /*!< Interrupt Pin 18                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC080 = regval;
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA) && !defined(TMPM3HNA)
    case EXCEPT_INT19:                  /*!< Interrupt Pin 19                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC081 = regval;
        break;
    case EXCEPT_INT20:                  /*!< Interrupt Pin 20                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC082 = regval;
        break;
    case EXCEPT_INT21:                  /*!< Interrupt Pin 21                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC083 = regval;
        break;
    case EXCEPT_INT22:                  /*!< Interrupt Pin 22                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC084 = regval;
        break;
    case EXCEPT_INT23:                  /*!< Interrupt Pin 23                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC085 = regval;
        break;
    case EXCEPT_INT24:                  /*!< Interrupt Pin 24                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC086 = regval;
        break;
    case EXCEPT_INT25:                  /*!< Interrupt Pin 25                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC087 = regval;
        break;
    case EXCEPT_INT26:                  /*!< Interrupt Pin 26                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC088 = regval;
        break;
    case EXCEPT_INT27:                  /*!< Interrupt Pin 27                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC089 = regval;
        break;
    case EXCEPT_INT28:                  /*!< Interrupt Pin 28                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC090 = regval;
        break;
#endif
#if defined(TMPM3HQA)
    case EXCEPT_INT29:                  /*!< Interrupt Pin 29                                                       */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC091 = regval;
        break;
    case EXCEPT_INT30:                  /*!< Interrupt Pin 30                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC092 = regval;
        break;
    case EXCEPT_INT31:                  /*!< Interrupt Pin 31                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC093 = regval;
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA)
    case EXCEPT_INT32:              /*!< Interrupt Pin 32                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC095 = regval;
        break;
    case EXCEPT_INT33:              /*!< Interrupt Pin 33                                                           */
        regval |= EXCEPT_INTPCLR;
        regval |= EXCEPT_INTNCLR;
        p_obj->p_instance2->IMC096 = regval;
        break;
#endif
    case EXCEPT_INTRMC0:                /*!< Remote control reception interrupt                                     */
        regval = 0x06;
        regval |= EXCEPT_INTPCLR;
        p_obj->p_instance2->IMC094 = regval;
        break;
    default:
        break;
    }
}

/*--------------------------------------------------*/
/**
  * @brief     Get intteruput edge.
  * @param     p_obj : EXCEPT object.
  * @param     int   : Select interrupt.
  * @retval    state : Interrupt detect flag.
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
uint8_t except_get_edge(except_t *p_obj, uint32_t interrupt)
{
    uint8_t result = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the EXCEPT_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance1));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance2));
#endif /* #ifdef DEBUG */

    /* Set interrupt. */
    switch (interrupt)
    {
    case EXCEPT_INT00:                  /*!< Interrupt Pin 00                                                       */
        result = (p_obj->p_instance1->IMC00 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT01:                  /*!< Interrupt Pin 01                                                       */
        result = (p_obj->p_instance1->IMC01 & EXCEPT_EDGE_MASK);
        break;
#if !defined(TMPM3HLA)
    case EXCEPT_INT02:                  /*!< Interrupt Pin 02                                                       */
        result = (p_obj->p_instance1->IMC02 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT13:                  /*!< Interrupt Pin 13                                                       */
        result = (p_obj->p_instance1->IMC03 & EXCEPT_EDGE_MASK);
        break;
#endif
    case EXCEPT_INT03:                  /*!< Interrupt Pin 03                                                       */
        result = (p_obj->p_instance2->IMC066 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT04:                  /*!< Interrupt Pin 04                                                       */
        result = (p_obj->p_instance2->IMC067 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT05:                  /*!< Interrupt Pin 05                                                       */
        result = (p_obj->p_instance2->IMC068 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT06:                  /*!< Interrupt Pin 06                                                       */
        result = (p_obj->p_instance2->IMC069 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT07:                  /*!< Interrupt Pin 07                                                       */
        result = (p_obj->p_instance2->IMC070 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT08:                  /*!< Interrupt Pin 08                                                       */
        result = (p_obj->p_instance2->IMC071 & EXCEPT_EDGE_MASK);
        break;
#if !defined(TMPM3HLA)
    case EXCEPT_INT09:                  /*!< Interrupt Pin 09                                                       */
        result = (p_obj->p_instance2->IMC072 & EXCEPT_EDGE_MASK);
        break;
#endif
    case EXCEPT_INT10:              /*!< Interrupt Pin 10                                                           */
        result = (p_obj->p_instance2->IMC073 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT11:                  /*!< Interrupt Pin 11                                                       */
        result = (p_obj->p_instance2->IMC074 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT12:                  /*!< Interrupt Pin 12                                                       */
        result = (p_obj->p_instance2->IMC075 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT14:                  /*!< Interrupt Pin 14                                                       */
        result = (p_obj->p_instance2->IMC076 & EXCEPT_EDGE_MASK);
        break;
#if !defined(TMPM3HLA) && !defined(TMPM3HMA)
    case EXCEPT_INT15:                  /*!< Interrupt Pin 15                                                       */
        result = (p_obj->p_instance2->IMC077 & EXCEPT_EDGE_MASK);
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA)
    case EXCEPT_INT16:                  /*!< Interrupt Pin 16                                                       */
        result = (p_obj->p_instance2->IMC078 & EXCEPT_EDGE_MASK);
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA) && !defined(TMPM3HNA)
    case EXCEPT_INT17:              /*!< Interrupt Pin 17                                                           */
        result = (p_obj->p_instance2->IMC079 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT18:              /*!< Interrupt Pin 18                                                           */
        result = (p_obj->p_instance2->IMC080 & EXCEPT_EDGE_MASK);
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA) && !defined(TMPM3HNA)
    case EXCEPT_INT19:                  /*!< Interrupt Pin 19                                                       */
        result = (p_obj->p_instance2->IMC096 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT20:                  /*!< Interrupt Pin 20                                                       */
        result = (p_obj->p_instance2->IMC082 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT21:                  /*!< Interrupt Pin 21                                                       */
        result = (p_obj->p_instance2->IMC083 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT22:                  /*!< Interrupt Pin 22                                                       */
        result = (p_obj->p_instance2->IMC084 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT23:                  /*!< Interrupt Pin 23                                                       */
        result = (p_obj->p_instance2->IMC085 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT24:                  /*!< Interrupt Pin 24                                                       */
        result = (p_obj->p_instance2->IMC086 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT25:                  /*!< Interrupt Pin 25                                                       */
        result = (p_obj->p_instance2->IMC087 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT26:                  /*!< Interrupt Pin 26                                                       */
        result = (p_obj->p_instance2->IMC088 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT27:                  /*!< Interrupt Pin 27                                                       */
        result = (p_obj->p_instance2->IMC089 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT28:                  /*!< Interrupt Pin 28                                                       */
        result = (p_obj->p_instance2->IMC090 & EXCEPT_EDGE_MASK);
        break;
#endif
#if defined(TMPM3HQA)
    case EXCEPT_INT29:                  /*!< Interrupt Pin 29                                                       */
        result = (p_obj->p_instance2->IMC091 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT30:                  /*!< Interrupt Pin 30                                                       */
        result = (p_obj->p_instance2->IMC092 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT31:                  /*!< Interrupt Pin 31                                                       */
        result = (p_obj->p_instance2->IMC093 & EXCEPT_EDGE_MASK);
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA)
    case EXCEPT_INT32:              /*!< Interrupt Pin 32                                                               */
        result = (p_obj->p_instance2->IMC095 & EXCEPT_EDGE_MASK);
        break;
    case EXCEPT_INT33:              /*!< Interrupt Pin 33                                                               */
        result = (p_obj->p_instance2->IMC096 & EXCEPT_EDGE_MASK);
        break;
#endif
    default:
        break;
    }
    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief     Get reset flag.
  * @param     p_obj       :RESET object.
  * @param     Select reset flag.
  * @retval    Reset flag.
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
uint8_t reset_get_flag(reset_t *p_obj, uint32_t reset)
{
    uint8_t result = 0U;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the EXCEPT_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* Get Reset flag. */
    switch (reset)
    {
    case RESET_LVD:
        if ((p_obj->p_instance->RSTFLG0 & RESET_LVD_FLG) == RESET_LVD_FLG) {
            result = RESET_LVD;
        }
        break;
    case RESET_STOP2:
        if ((p_obj->p_instance->RSTFLG0 & RESET_STOP2_FLG) == RESET_STOP2_FLG) {
            result = RESET_STOP2;
        }
        break;
    case RESET_PINR:
        if ((p_obj->p_instance->RSTFLG0 & RESET_PINR_FLG) == RESET_PINR_FLG) {
            result = RESET_PINR;
        }
        break;
    case RESET_POR:
        if ((p_obj->p_instance->RSTFLG0 & RESET_POR_FLG) == RESET_POR_FLG) {
            result = RESET_POR;
        }
        break;
    case RESET_OFD:
        if ((p_obj->p_instance->RSTFLG1 & RESET_OFD_FLG) == RESET_OFD_FLG) {
            result = RESET_OFD;
        }
        break;
    case RESET_WDT:
        if ((p_obj->p_instance->RSTFLG1 & RESET_WDT_FLG) == RESET_WDT_FLG) {
            result = RESET_WDT;
        }
        break;
    case RESET_LOCK:
        if ((p_obj->p_instance->RSTFLG1 & RESET_LOCK_FLG) == RESET_LOCK_FLG) {
            result = RESET_LOCK;
        }
        break;
    case RESET_SYS:
        if ((p_obj->p_instance->RSTFLG1 & RESET_SYS_FLG) == RESET_SYS_FLG) {
            result = RESET_SYS;
        }
        break;
    default:
        break;
    }
    return (result);
}

/*--------------------------------------------------*/
/**
  * @brief     Clear reset flag.
  * @param     p_obj       :RESET object.
  * @param     Select reset flag.
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void reset_clear_flag(reset_t *p_obj, uint32_t reset)
{
    uint8_t regval = 0U;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the EXCEPT_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */

    /* Clear Reset flag. */
    switch (reset)
    {
    case RESET_LVD:
        regval = p_obj->p_instance->RSTFLG0;
        regval &= RESET_CLEAR_LVD;
        p_obj->p_instance->RSTFLG0 = regval;
        break;
    case RESET_STOP2:
        regval = p_obj->p_instance->RSTFLG0;
        regval &= RESET_CLEAR_STOP2;
        p_obj->p_instance->RSTFLG0 = regval;
        break;
    case RESET_PINR:
        regval = p_obj->p_instance->RSTFLG0;
        regval &= RESET_CLEAR_PINR;
        p_obj->p_instance->RSTFLG0 = regval;
        break;
    case RESET_POR:
        regval = p_obj->p_instance->RSTFLG0;
        regval &= RESET_CLEAR_POR;
        p_obj->p_instance->RSTFLG0 = regval;
        break;
    case RESET_OFD:
        regval = p_obj->p_instance->RSTFLG1;
        regval &= RESET_CLEAR_OFD;
        p_obj->p_instance->RSTFLG1 = regval;
        break;
    case RESET_WDT:
        regval = p_obj->p_instance->RSTFLG1;
        regval &= RESET_CLEAR_WDT;
        p_obj->p_instance->RSTFLG1 = regval;
        break;
    case RESET_LOCK:
        regval = p_obj->p_instance->RSTFLG1;
        regval &= RESET_CLEAR_LOCK;
        p_obj->p_instance->RSTFLG1 = regval;
        break;
    case RESET_SYS:
        regval = p_obj->p_instance->RSTFLG1;
        regval &= RESET_CLEAR_SYS;
        p_obj->p_instance->RSTFLG1 = regval;
        break;
    default:
        break;
    }
}

/*--------------------------------------------------*/
/**
  * @brief     Get intteruput flag.
  * @param     p_obj : EXCEPT object.
  * @param     int   : Select interrupt.
  * @retval    Interruput flag
  * @note      -
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
uint32_t except_get_int(except_t *p_obj, uint32_t interrupt)
{
    uint32_t result = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the EXCEPT_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance3));
#endif /* #ifdef DEBUG */

    /* Set interrupt. */
    switch (interrupt)
    {
    case EXCEPT_LVD:                  /*!< Interrupt LVD                                                            */
        if((p_obj->p_instance3->FLGNMI & EXCEPT_INT000FLG_DETECT) == EXCEPT_INT000FLG_DETECT) {
            result = EXCEPT_INT000FLG_DETECT;
        } else {
            result = EXCEPT_INT000FLG_UNDETECT;
        }
        break;
    case EXCEPT_WDT:                /*!< Interrupt WDT                                                          */
        if((p_obj->p_instance3->FLGNMI & EXCEPT_INT016FLG_DETECT) == EXCEPT_INT016FLG_DETECT) {
            result = EXCEPT_INT016FLG_DETECT;
        } else {
            result = EXCEPT_INT016FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT00:                  /*!< Interrupt Pin 00                                                       */
        if((p_obj->p_instance3->FLG1 & EXCEPT_INT032FLG_DETECT) == EXCEPT_INT032FLG_DETECT) {
            result = EXCEPT_INT032FLG_DETECT;
        } else {
            result = EXCEPT_INT032FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT01:                  /*!< Interrupt Pin 01                                                       */
        if((p_obj->p_instance3->FLG1 & EXCEPT_INT033FLG_DETECT) == EXCEPT_INT033FLG_DETECT) {
            result = EXCEPT_INT033FLG_DETECT;
        } else {
            result = EXCEPT_INT033FLG_UNDETECT;
        }
        break;
#if !defined(TMPM3HLA)
    case EXCEPT_INT02:                  /*!< Interrupt Pin 02                                                       */
        if((p_obj->p_instance3->FLG1 & EXCEPT_INT034FLG_DETECT) == EXCEPT_INT034FLG_DETECT) {
            result = EXCEPT_INT034FLG_DETECT;
        } else {
            result = EXCEPT_INT034FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT13:                  /*!< Interrupt Pin 13                                                       */
        if((p_obj->p_instance3->FLG1 & EXCEPT_INT035FLG_DETECT) == EXCEPT_INT035FLG_DETECT) {
            result = EXCEPT_INT035FLG_DETECT;
        } else {
            result = EXCEPT_INT035FLG_UNDETECT;
        }
        break;
#endif
    case EXCEPT_I2CWUP:                 /*!< Serial bus interface (WakeUp) interrupt (channel 0)                    */
        if((p_obj->p_instance3->FLG1 & EXCEPT_INT048FLG_DETECT) == EXCEPT_INT048FLG_DETECT) {
            result = EXCEPT_INT048FLG_DETECT;
        } else {
            result = EXCEPT_INT048FLG_UNDETECT;
        }
        break;
    case EXCEPT_RTC:                    /*!< Real time clock interrupt                                              */
        if((p_obj->p_instance3->FLG1 & EXCEPT_INT049FLG_DETECT) == EXCEPT_INT049FLG_DETECT) {
            result = EXCEPT_INT049FLG_DETECT;
        } else {
            result = EXCEPT_INT049FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC00:            /*!< DMAA end of transfer 00                                                    */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT096FLG_DETECT) == EXCEPT_INT096FLG_DETECT) {
            result = EXCEPT_INT096FLG_DETECT;
        } else {
            result = EXCEPT_INT096FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC01:          /*!< DMAA end of transfer 01                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT097FLG_DETECT) == EXCEPT_INT097FLG_DETECT) {
            result = EXCEPT_INT097FLG_DETECT;
        } else {
            result = EXCEPT_INT097FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC02:          /*!< DMAA end of transfer 02                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT098FLG_DETECT) == EXCEPT_INT098FLG_DETECT) {
            result = EXCEPT_INT098FLG_DETECT;
        } else {
            result = EXCEPT_INT098FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC03:          /*!< DMAA end of transfer 03                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT099FLG_DETECT) == EXCEPT_INT099FLG_DETECT) {
            result = EXCEPT_INT099FLG_DETECT;
        } else {
            result = EXCEPT_INT099FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC04:          /*!< DMAA end of transfer 04                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT100FLG_DETECT) == EXCEPT_INT100FLG_DETECT) {
            result = EXCEPT_INT100FLG_DETECT;
        } else {
            result = EXCEPT_INT100FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC05:          /*!< DMAA end of transfer 05                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT101FLG_DETECT) == EXCEPT_INT101FLG_DETECT) {
            result = EXCEPT_INT101FLG_DETECT;
        } else {
            result = EXCEPT_INT101FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC06:          /*!< DMAA end of transfer 06                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT102FLG_DETECT) == EXCEPT_INT102FLG_DETECT) {
            result = EXCEPT_INT102FLG_DETECT;
        } else {
            result = EXCEPT_INT102FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC07:          /*!< DMAA end of transfer 07                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT103FLG_DETECT) == EXCEPT_INT103FLG_DETECT) {
            result = EXCEPT_INT103FLG_DETECT;
        } else {
            result = EXCEPT_INT103FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC08:          /*!< DMAA end of transfer 08                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT104FLG_DETECT) == EXCEPT_INT104FLG_DETECT) {
            result = EXCEPT_INT104FLG_DETECT;
        } else {
            result = EXCEPT_INT104FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC09:          /*!< DMAA end of transfer 09                                                      */
      if((p_obj->p_instance3->FLG3 & EXCEPT_INT105FLG_DETECT) == EXCEPT_INT105FLG_DETECT) {
            result = EXCEPT_INT105FLG_DETECT;
        } else {
            result = EXCEPT_INT105FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC10:          /*!< DMAA end of transfer 10                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT106FLG_DETECT) == EXCEPT_INT106FLG_DETECT) {
            result = EXCEPT_INT106FLG_DETECT;
        } else {
            result = EXCEPT_INT106FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC11:          /*!< DMAA end of transfer 11                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT107FLG_DETECT) == EXCEPT_INT107FLG_DETECT) {
            result = EXCEPT_INT107FLG_DETECT;
        } else {
            result = EXCEPT_INT107FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC12:          /*!< DMAA end of transfer 12                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT108FLG_DETECT) == EXCEPT_INT108FLG_DETECT) {
            result = EXCEPT_INT108FLG_DETECT;
        } else {
            result = EXCEPT_INT108FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC13:          /*!< DMAA end of transfer 13                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT109FLG_DETECT) == EXCEPT_INT109FLG_DETECT) {
            result = EXCEPT_INT109FLG_DETECT;
        } else {
            result = EXCEPT_INT109FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC14:          /*!< DMAA end of transfer 14                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT110FLG_DETECT) == EXCEPT_INT110FLG_DETECT) {
            result = EXCEPT_INT110FLG_DETECT;
        } else {
            result = EXCEPT_INT110FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC15:          /*!< DMAA end of transfer 15                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT111FLG_DETECT) == EXCEPT_INT111FLG_DETECT) {
            result = EXCEPT_INT111FLG_DETECT;
        } else {
            result = EXCEPT_INT111FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC16:          /*!< DMAA end of transfer 16                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT112FLG_DETECT) == EXCEPT_INT112FLG_DETECT) {
            result = EXCEPT_INT112FLG_DETECT;
        } else {
            result = EXCEPT_INT112FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC17:          /*!< DMAA end of transfer 17                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT113FLG_DETECT) == EXCEPT_INT113FLG_DETECT) {
            result = EXCEPT_INT113FLG_DETECT;
        } else {
            result = EXCEPT_INT113FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC18:          /*!< DMAA end of transfer 18                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT114FLG_DETECT) == EXCEPT_INT114FLG_DETECT) {
            result = EXCEPT_INT114FLG_DETECT;
        } else {
            result = EXCEPT_INT114FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC19:          /*!< DMAA end of transfer 19                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT115FLG_DETECT) == EXCEPT_INT115FLG_DETECT) {
            result = EXCEPT_INT115FLG_DETECT;
        } else {
            result = EXCEPT_INT115FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC20:          /*!< DMAA end of transfer 20                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT116FLG_DETECT) == EXCEPT_INT116FLG_DETECT) {
            result = EXCEPT_INT116FLG_DETECT;
        } else {
            result = EXCEPT_INT116FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC21:          /*!< DMAA end of transfer 21                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT117FLG_DETECT) == EXCEPT_INT117FLG_DETECT) {
            result = EXCEPT_INT117FLG_DETECT;
        } else {
            result = EXCEPT_INT117FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC22:          /*!< DMAA end of transfer 22                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT118FLG_DETECT) == EXCEPT_INT118FLG_DETECT) {
            result = EXCEPT_INT118FLG_DETECT;
        } else {
            result = EXCEPT_INT118FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC23:          /*!< DMAA end of transfer 23                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT119FLG_DETECT) == EXCEPT_INT119FLG_DETECT) {
            result = EXCEPT_INT119FLG_DETECT;
        } else {
            result = EXCEPT_INT119FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC24:          /*!< DMAA end of transfer 24                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT120FLG_DETECT) == EXCEPT_INT120FLG_DETECT) {
            result = EXCEPT_INT120FLG_DETECT;
        } else {
            result = EXCEPT_INT120FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC25:          /*!< DMAA end of transfer 25                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT121FLG_DETECT) == EXCEPT_INT121FLG_DETECT) {
            result = EXCEPT_INT121FLG_DETECT;
        } else {
            result = EXCEPT_INT121FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC26:          /*!< DMAA end of transfer 26                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT122FLG_DETECT) == EXCEPT_INT122FLG_DETECT) {
            result = EXCEPT_INT122FLG_DETECT;
        } else {
            result = EXCEPT_INT122FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC27:          /*!< DMAA end of transfer 27                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT123FLG_DETECT) == EXCEPT_INT123FLG_DETECT) {
            result = EXCEPT_INT123FLG_DETECT;
        } else {
            result = EXCEPT_INT123FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC28:          /*!< DMAA end of transfer 28                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT124FLG_DETECT) == EXCEPT_INT124FLG_DETECT) {
            result = EXCEPT_INT124FLG_DETECT;
        } else {
            result = EXCEPT_INT124FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC29:          /*!< DMAA end of transfer 29                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT125FLG_DETECT) == EXCEPT_INT125FLG_DETECT) {
            result = EXCEPT_INT125FLG_DETECT;
        } else {
            result = EXCEPT_INT125FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC30:          /*!< DMAA end of transfer 30                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT126FLG_DETECT) == EXCEPT_INT126FLG_DETECT) {
            result = EXCEPT_INT126FLG_DETECT;
        } else {
            result = EXCEPT_INT126FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAATC31:          /*!< DMAA end of transfer 31                                                      */
        if((p_obj->p_instance3->FLG3 & EXCEPT_INT127FLG_DETECT) == EXCEPT_INT127FLG_DETECT) {
            result = EXCEPT_INT127FLG_DETECT;
        } else {
            result = EXCEPT_INT127FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMAAERR:             /*!< DMAA transfer error                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT128FLG_DETECT) == EXCEPT_INT128FLG_DETECT) {
            result = EXCEPT_INT128FLG_DETECT;
        } else {
            result = EXCEPT_INT128FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC00:            /*!< DMAB end of transfer 00                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT129FLG_DETECT) == EXCEPT_INT129FLG_DETECT) {
            result = EXCEPT_INT129FLG_DETECT;
        } else {
            result = EXCEPT_INT129FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC01:            /*!< DMAB end of transfer 01                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT130FLG_DETECT) == EXCEPT_INT130FLG_DETECT) {
            result = EXCEPT_INT130FLG_DETECT;
        } else {
            result = EXCEPT_INT130FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC02:            /*!< DMAB end of transfer 02                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT131FLG_DETECT) == EXCEPT_INT131FLG_DETECT) {
            result = EXCEPT_INT131FLG_DETECT;
        } else {
            result = EXCEPT_INT131FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC03:            /*!< DMAB end of transfer 03                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT132FLG_DETECT) == EXCEPT_INT132FLG_DETECT) {
            result = EXCEPT_INT132FLG_DETECT;
        } else {
            result = EXCEPT_INT132FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC04:            /*!< DMAB end of transfer 04                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT133FLG_DETECT) == EXCEPT_INT133FLG_DETECT) {
            result = EXCEPT_INT133FLG_DETECT;
        } else {
            result = EXCEPT_INT133FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC05:            /*!< DMAB end of transfer 05                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT134FLG_DETECT) == EXCEPT_INT134FLG_DETECT) {
            result = EXCEPT_INT134FLG_DETECT;
        } else {
            result = EXCEPT_INT134FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC06:            /*!< DMAB end of transfer 06                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT135FLG_DETECT) == EXCEPT_INT135FLG_DETECT) {
            result = EXCEPT_INT135FLG_DETECT;
        } else {
            result = EXCEPT_INT135FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC07:            /*!< DMAB end of transfer 07                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT136FLG_DETECT) == EXCEPT_INT136FLG_DETECT) {
            result = EXCEPT_INT136FLG_DETECT;
        } else {
            result = EXCEPT_INT136FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC08:            /*!< DMAB end of transfer 08                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT137FLG_DETECT) == EXCEPT_INT137FLG_DETECT) {
            result = EXCEPT_INT137FLG_DETECT;
        } else {
            result = EXCEPT_INT137FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC09:            /*!< DMAB end of transfer 09                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT138FLG_DETECT) == EXCEPT_INT138FLG_DETECT) {
            result = EXCEPT_INT138FLG_DETECT;
        } else {
            result = EXCEPT_INT138FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC10:            /*!< DMAB end of transfer 10                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT139FLG_DETECT) == EXCEPT_INT139FLG_DETECT) {
            result = EXCEPT_INT139FLG_DETECT;
        } else {
            result = EXCEPT_INT139FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC11:            /*!< DMAB end of transfer 11                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT140FLG_DETECT) == EXCEPT_INT140FLG_DETECT) {
            result = EXCEPT_INT140FLG_DETECT;
        } else {
            result = EXCEPT_INT140FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC12:            /*!< DMAB end of transfer 12                                                    */
      if((p_obj->p_instance3->FLG4 & EXCEPT_INT141FLG_DETECT) == EXCEPT_INT141FLG_DETECT) {
            result = EXCEPT_INT141FLG_DETECT;
        } else {
            result = EXCEPT_INT141FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC13:            /*!< DMAB end of transfer 13                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT142FLG_DETECT) == EXCEPT_INT142FLG_DETECT) {
            result = EXCEPT_INT142FLG_DETECT;
        } else {
            result = EXCEPT_INT142FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC14:            /*!< DMAB end of transfer 14                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT143FLG_DETECT) == EXCEPT_INT143FLG_DETECT) {
            result = EXCEPT_INT143FLG_DETECT;
        } else {
            result = EXCEPT_INT143FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC15:            /*!< DMAB end of transfer 15                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT144FLG_DETECT) == EXCEPT_INT144FLG_DETECT) {
            result = EXCEPT_INT144FLG_DETECT;
        } else {
            result = EXCEPT_INT144FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC16:            /*!< DMAB end of transfer 16                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT145FLG_DETECT) == EXCEPT_INT145FLG_DETECT) {
            result = EXCEPT_INT145FLG_DETECT;
        } else {
            result = EXCEPT_INT145FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC17:            /*!< DMAB end of transfer 17                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT146FLG_DETECT) == EXCEPT_INT146FLG_DETECT) {
            result = EXCEPT_INT146FLG_DETECT;
        } else {
            result = EXCEPT_INT146FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC18:            /*!< DMAB end of transfer 18                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT147FLG_DETECT) == EXCEPT_INT147FLG_DETECT) {
            result = EXCEPT_INT147FLG_DETECT;
        } else {
            result = EXCEPT_INT147FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC19:            /*!< DMAB end of transfer 19                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT148FLG_DETECT) == EXCEPT_INT148FLG_DETECT) {
            result = EXCEPT_INT148FLG_DETECT;
        } else {
            result = EXCEPT_INT148FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC20:            /*!< DMAB end of transfer 20                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT149FLG_DETECT) == EXCEPT_INT149FLG_DETECT) {
            result = EXCEPT_INT149FLG_DETECT;
        } else {
            result = EXCEPT_INT149FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC21:            /*!< DMAB end of transfer 21                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT150FLG_DETECT) == EXCEPT_INT150FLG_DETECT) {
            result = EXCEPT_INT150FLG_DETECT;
        } else {
            result = EXCEPT_INT150FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC22:            /*!< DMAB end of transfer 22                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT151FLG_DETECT) == EXCEPT_INT151FLG_DETECT) {
            result = EXCEPT_INT151FLG_DETECT;
        } else {
            result = EXCEPT_INT151FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC23:            /*!< DMAB end of transfer 23                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT152FLG_DETECT) == EXCEPT_INT152FLG_DETECT) {
            result = EXCEPT_INT152FLG_DETECT;
        } else {
            result = EXCEPT_INT152FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC24:            /*!< DMAB end of transfer 24                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT153FLG_DETECT) == EXCEPT_INT153FLG_DETECT) {
            result = EXCEPT_INT153FLG_DETECT;
        } else {
            result = EXCEPT_INT153FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC25:            /*!< DMAB end of transfer 25                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT154FLG_DETECT) == EXCEPT_INT154FLG_DETECT) {
            result = EXCEPT_INT154FLG_DETECT;
        } else {
            result = EXCEPT_INT154FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC26:            /*!< DMAB end of transfer 26                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT155FLG_DETECT) == EXCEPT_INT155FLG_DETECT) {
            result = EXCEPT_INT155FLG_DETECT;
        } else {
            result = EXCEPT_INT155FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC27:            /*!< DMAB end of transfer 27                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT156FLG_DETECT) == EXCEPT_INT156FLG_DETECT) {
            result = EXCEPT_INT156FLG_DETECT;
        } else {
            result = EXCEPT_INT156FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC28:            /*!< DMAB end of transfer 28                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT157FLG_DETECT) == EXCEPT_INT157FLG_DETECT) {
            result = EXCEPT_INT157FLG_DETECT;
        } else {
            result = EXCEPT_INT157FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC29:            /*!< DMAB end of transfer 29                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT158FLG_DETECT) == EXCEPT_INT158FLG_DETECT) {
            result = EXCEPT_INT158FLG_DETECT;
        } else {
            result = EXCEPT_INT158FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC30:            /*!< DMAB end of transfer 30                                                    */
        if((p_obj->p_instance3->FLG4 & EXCEPT_INT159FLG_DETECT) == EXCEPT_INT159FLG_DETECT) {
            result = EXCEPT_INT159FLG_DETECT;
        } else {
            result = EXCEPT_INT159FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABTC31:            /*!< DMAB end of transfer 31                                                    */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT160FLG_DETECT) == EXCEPT_INT160FLG_DETECT) {
            result = EXCEPT_INT160FLG_DETECT;
        } else {
            result = EXCEPT_INT160FLG_UNDETECT;
        }
        break;
    case EXCEPT_INTDMABERR:             /*!< DMAB transfer error                                                    */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT161FLG_DETECT) == EXCEPT_INT161FLG_DETECT) {
            result = EXCEPT_INT161FLG_DETECT;
        } else {
            result = EXCEPT_INT161FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT03:                  /*!< Interrupt Pin 03                                                       */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT162FLG_DETECT) == EXCEPT_INT162FLG_DETECT) {
            result = EXCEPT_INT162FLG_DETECT;
        } else {
            result = EXCEPT_INT162FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT04:                  /*!< Interrupt Pin 04                                                       */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT163FLG_DETECT) == EXCEPT_INT163FLG_DETECT) {
            result = EXCEPT_INT163FLG_DETECT;
        } else {
            result = EXCEPT_INT163FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT05:                  /*!< Interrupt Pin 05                                                       */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT164FLG_DETECT) == EXCEPT_INT164FLG_DETECT) {
            result = EXCEPT_INT164FLG_DETECT;
        } else {
            result = EXCEPT_INT164FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT06:                  /*!< Interrupt Pin 06                                                       */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT165FLG_DETECT) == EXCEPT_INT165FLG_DETECT) {
            result = EXCEPT_INT165FLG_DETECT;
        } else {
            result = EXCEPT_INT165FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT07:                  /*!< Interrupt Pin 07                                                       */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT166FLG_DETECT) == EXCEPT_INT166FLG_DETECT) {
            result = EXCEPT_INT166FLG_DETECT;
        } else {
            result = EXCEPT_INT166FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT08:                  /*!< Interrupt Pin 08                                                       */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT167FLG_DETECT) == EXCEPT_INT167FLG_DETECT) {
            result = EXCEPT_INT167FLG_DETECT;
        } else {
            result = EXCEPT_INT167FLG_UNDETECT;
        }
        break;
#if !defined(TMPM3HLA)
    case EXCEPT_INT09:                  /*!< Interrupt Pin 09                                                       */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT168FLG_DETECT) == EXCEPT_INT168FLG_DETECT) {
            result = EXCEPT_INT168FLG_DETECT;
        } else {
            result = EXCEPT_INT168FLG_UNDETECT;
        }
        break;
#endif
    case EXCEPT_INT10:              /*!< Interrupt Pin 10                                                       */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT169FLG_DETECT) == EXCEPT_INT169FLG_DETECT) {
            result = EXCEPT_INT169FLG_DETECT;
        } else {
            result = EXCEPT_INT169FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT11:                  /*!< Interrupt Pin 11                                                       */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT170FLG_DETECT) == EXCEPT_INT170FLG_DETECT) {
            result = EXCEPT_INT170FLG_DETECT;
        } else {
            result = EXCEPT_INT170FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT12:                  /*!< Interrupt Pin 12                                                       */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT171FLG_DETECT) == EXCEPT_INT171FLG_DETECT) {
            result = EXCEPT_INT171FLG_DETECT;
        } else {
            result = EXCEPT_INT171FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT14:                  /*!< Interrupt Pin 14                                                       */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT172FLG_DETECT) == EXCEPT_INT172FLG_DETECT) {
            result = EXCEPT_INT172FLG_DETECT;
        } else {
            result = EXCEPT_INT172FLG_UNDETECT;
        }
        break;
#if !defined(TMPM3HLA) && !defined(TMPM3HMA)
    case EXCEPT_INT15:                  /*!< Interrupt Pin 15                                                       */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT173FLG_DETECT) == EXCEPT_INT173FLG_DETECT) {
            result = EXCEPT_INT173FLG_DETECT;
        } else {
            result = EXCEPT_INT173FLG_UNDETECT;
        }
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA)
    case EXCEPT_INT16:                  /*!< Interrupt Pin 16                                                       */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT174FLG_DETECT) == EXCEPT_INT174FLG_DETECT) {
            result = EXCEPT_INT174FLG_DETECT;
        } else {
            result = EXCEPT_INT174FLG_UNDETECT;
        }
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA) && !defined(TMPM3HNA)
    case EXCEPT_INT17:              /*!< Interrupt Pin 17                                                           */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT175FLG_DETECT) == EXCEPT_INT175FLG_DETECT) {
            result = EXCEPT_INT175FLG_DETECT;
        } else {
            result = EXCEPT_INT175FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT18:              /*!< Interrupt Pin 18                                                           */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT176FLG_DETECT) == EXCEPT_INT176FLG_DETECT) {
            result = EXCEPT_INT176FLG_DETECT;
        } else {
            result = EXCEPT_INT176FLG_UNDETECT;
        }
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA) && !defined(TMPM3HNA)
    case EXCEPT_INT19:                  /*!< Interrupt Pin 19                                                           */
      if((p_obj->p_instance3->FLG5 & EXCEPT_INT177FLG_DETECT) == EXCEPT_INT177FLG_DETECT) {
            result = EXCEPT_INT177FLG_DETECT;
        } else {
            result = EXCEPT_INT177FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT20:                  /*!< Interrupt Pin 20                                                           */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT178FLG_DETECT) == EXCEPT_INT178FLG_DETECT) {
            result = EXCEPT_INT178FLG_DETECT;
        } else {
            result = EXCEPT_INT178FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT21:                  /*!< Interrupt Pin 21                                                           */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT179FLG_DETECT) == EXCEPT_INT179FLG_DETECT) {
            result = EXCEPT_INT179FLG_DETECT;
        } else {
            result = EXCEPT_INT179FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT22:                  /*!< Interrupt Pin 22                                                           */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT180FLG_DETECT) == EXCEPT_INT180FLG_DETECT) {
            result = EXCEPT_INT180FLG_DETECT;
        } else {
            result = EXCEPT_INT180FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT23:                  /*!< Interrupt Pin 23                                                           */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT181FLG_DETECT) == EXCEPT_INT181FLG_DETECT) {
            result = EXCEPT_INT181FLG_DETECT;
        } else {
            result = EXCEPT_INT181FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT24:                  /*!< Interrupt Pin 24                                                           */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT182FLG_DETECT) == EXCEPT_INT182FLG_DETECT) {
            result = EXCEPT_INT182FLG_DETECT;
        } else {
            result = EXCEPT_INT182FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT25:                  /*!< Interrupt Pin 25                                                           */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT183FLG_DETECT) == EXCEPT_INT183FLG_DETECT) {
            result = EXCEPT_INT183FLG_DETECT;
        } else {
            result = EXCEPT_INT183FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT26:                  /*!< Interrupt Pin 26                                                           */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT184FLG_DETECT) == EXCEPT_INT184FLG_DETECT) {
            result = EXCEPT_INT184FLG_DETECT;
        } else {
            result = EXCEPT_INT184FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT27:                  /*!< Interrupt Pin 27                                                           */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT185FLG_DETECT) == EXCEPT_INT185FLG_DETECT) {
            result = EXCEPT_INT185FLG_DETECT;
        } else {
            result = EXCEPT_INT185FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT28:                  /*!< Interrupt Pin 28                                                           */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT186FLG_DETECT) == EXCEPT_INT186FLG_DETECT) {
            result = EXCEPT_INT186FLG_DETECT;
        } else {
            result = EXCEPT_INT186FLG_UNDETECT;
        }
        break;
#endif
#if defined(TMPM3HQA)
    case EXCEPT_INT29:                  /*!< Interrupt Pin 29                                                       */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT187FLG_DETECT) == EXCEPT_INT187FLG_DETECT) {
            result = EXCEPT_INT187FLG_DETECT;
        } else {
            result = EXCEPT_INT187FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT30:                  /*!< Interrupt Pin 30                                                           */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT188FLG_DETECT) == EXCEPT_INT188FLG_DETECT) {
            result = EXCEPT_INT188FLG_DETECT;
        } else {
            result = EXCEPT_INT188FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT31:                  /*!< Interrupt Pin 31                                                           */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT189FLG_DETECT) == EXCEPT_INT189FLG_DETECT) {
            result = EXCEPT_INT189FLG_DETECT;
        } else {
            result = EXCEPT_INT189FLG_UNDETECT;
        }
        break;
#endif
#if !defined(TMPM3HLA) && !defined(TMPM3HMA)
    case EXCEPT_INT32:              /*!< Interrupt Pin 32                                                           */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT191FLG_DETECT) == EXCEPT_INT191FLG_DETECT) {
            result = EXCEPT_INT191FLG_DETECT;
        } else {
            result = EXCEPT_INT191FLG_UNDETECT;
        }
        break;
    case EXCEPT_INT33:              /*!< Interrupt Pin 33                                                           */
        if((p_obj->p_instance3->FLG6 & EXCEPT_INT192FLG_DETECT) == EXCEPT_INT192FLG_DETECT) {
            result = EXCEPT_INT192FLG_DETECT;
        } else {
            result = EXCEPT_INT192FLG_UNDETECT;
        }
        break;
#endif
    case EXCEPT_INTRMC0:                /*!< Remote control reception interrupt                                     */
        if((p_obj->p_instance3->FLG5 & EXCEPT_INT190FLG_DETECT) == EXCEPT_INT190FLG_DETECT) {
            result = EXCEPT_INT190FLG_DETECT;
        } else {
            result = EXCEPT_INT190FLG_UNDETECT;
        }
        break;
    default:
        break;
    }
    return (result);
}

/**
 *  @} 
 */ /* End of group EXCEPT_Exported_functions */

/**
 *  @} 
 */ /* End of group EXCEPT */

/**
 *  @} 
 */ /* End of group Periph_Driver */

#endif /* defined(__EXCEPT_M3H1_H)  */

#ifdef __cplusplus
}
#endif /* __cplusplus */

