/** 
 *******************************************************************************
 * @file    flash512ud32_b.c
 * @brief   This file provides API functions for FLASH.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include <string.h>
#include "flash.h"

#if defined(__FLASH512UD32_B_H)
/**
 *  @addtogroup Periph_Driver Peripheral Driver
 *  @{
 */

/** 
 *  @addtogroup FLASH 
 *  @brief FLASH Driver
 *  @{
 */


/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_Private_macro FLASH Private Macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group FLASH_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_Private_define FLASH Private Define
 *  @{
 */

/** 
 *  @}
 */ /* End of group FLASH_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_Private_define FLASH Private Define
 *  @{
 */
#define FC_KCR_KEYCODE      (0xA74A9D23UL)              /*!< The specific code that writes the FCKCR register. */
#define FC_BRANK_VALUE      (uint32_t)(0xFFFFFFFFUL)    /*!< Brank value */
#define FC_MAX_PAGES        (uint8_t)(0x80)             /*!< Maxmum pages */
#define FC_MAX_BLOCKS       (uint8_t)(0x16)             /*!< Maxmum blocks */
#define FC_MAX_AREAS        (uint8_t)(0x1)              /*!< Maxmum areas */
#define FC_MAX_DATA_PAGES   (uint8_t)(0x21)             /*!< Maxmum pages */
#define FC_MAX_DATA_BLOCKS  (uint8_t)(0x8)              /*!< Maxmum blocks */
#define FC_CMD_ADDRESS_MASK (uint32_t)(0xFFFF0000UL)    /*!< Upper address mask of the upper address */
#define FC_CMD_BC1_ADDR     (0x00000550UL)              /*!< The lower address of the first bus cycle when uses commans */
#define FC_CMD_BC2_ADDR     (0x00000AA0UL)              /*!< The lower address of the second bus cycle when uses commans */
#define FC_BANK_USER_INFO   (0x00000007UL)              /*!< Bank Change User Information Area */
#define FC_BANK_CODE_FLASH  (0x00000000UL)              /*!< Bank Change Code Flash */
#define FC_BUFFER_DISABLE   (0x00000007UL)              /*!< Flash Buffer Disable nad Clear */
#define FC_BUFFER_ENABLE    (0x00000000UL)              /*!< Flash Buffer Enable */

#define FC_ACCR_FDLC_4      (0x00000300UL)              /*!< Data Flash read clock 4clock  */
#define FC_ACCR_FDLC_5      (0x00000400UL)              /*!< Data Flash read clock 5clock  */
#define FC_ACCR_FDLC_6      (0x00000500UL)              /*!< Data Flash read clock 6clock  */
#define FC_ACCR_FCLC_1      (0x00000000UL)              /*!< Code Flash read clock 1clock  */
#define FC_ACCR_FCLC_2      (0x00000001UL)              /*!< Code Flash read clock 2clock  */
#define FC_ACCR_FCLC_3      (0x00000002UL)              /*!< Code Flash read clock 3clock  */
#define FC_ACCR_FCLC_4      (0x00000003UL)              /*!< Code Flash read clock 4clock  */
#define FC_ACCR_FCLC_5      (0x00000004UL)              /*!< Code Flash read clock 5clock  */
#define FC_ACCR_FCLC_6      (0x00000005UL)              /*!< Code Flash read clock 6clock  */
#define SYSCORECLOCK_80M    (80000000UL)                /*!< 80MHz                         */

/* FCSR0 register */
#define FC_AREASEL_EXPECT_AREA0 (uint32_t)(0x00000000UL)    /*!< RW, Selects expect area0       */
#define FC_AREASEL_EXPECT_AREA4 (uint32_t)(0x00000000UL)    /*!< RW, Selects expect area4       */
#define FC_AREASEL_AREA0        (uint32_t)(0x00000007UL)    /*!< RW, Selects area0              */
#define FC_AREASEL_AREA4        (uint32_t)(0x00070000UL)    /*!< RW, Selects area4              */
#define FC_AREASEL_MASK_AREA0   (uint32_t)(0xFF8F0FF8UL)    /*!< RW, Selects area0              */
#define FC_AREASEL_MASK_AREA4   (uint32_t)(0xFF880FFFUL)    /*!< RW, Selects area4              */
#define FC_AREASEL_WRITE_MODE   (uint32_t)(0x04000000UL)    /*!< R,  Write Mode                 */
#define FC_AREASEL4_WRITE_MODE  (uint32_t)(0x40000000UL)    /*!< R,  Write Mode                 */

#define FC_SEC_MASK             (uint32_t)(0x00000000UL)    /*!< Flash security bit mask  */
#define FC_SEC_ENABLE           (uint32_t)(0x00000001UL)    /*!< Flash security enable  */

static uint32_t fc_const_data_flash_address[FC_MAX_DATA_PAGES] = {
    (0x30000000UL),   /*!< DATA FLASH Page0 */
    (0x30000100UL),   /*!< DATA FLASH Page1 */
    (0x30000200UL),   /*!< DATA FLASH Page2 */
    (0x30000300UL),   /*!< DATA FLASH Page3 */
    (0x30000400UL),   /*!< DATA FLASH Page4 */
    (0x30000500UL),   /*!< DATA FLASH Page5 */
    (0x30000600UL),   /*!< DATA FLASH Page6 */
    (0x30000700UL),   /*!< DATA FLASH Page7 */
    (0x30000800UL),   /*!< DATA FLASH Page8 */
    (0x30000900UL),   /*!< DATA FLASH Page9 */
    (0x30000A00UL),   /*!< DATA FLASH Page10 */
    (0x30000B00UL),   /*!< DATA FLASH Page11 */
    (0x30000C00UL),   /*!< DATA FLASH Page12 */
    (0x30000D00UL),   /*!< DATA FLASH Page13 */
    (0x30000E00UL),   /*!< DATA FLASH Page14 */
    (0x30000F00UL),   /*!< DATA FLASH Page15 */
    (0x30001000UL),   /*!< DATA FLASH Page16 */
    (0x30001100UL),   /*!< DATA FLASH Page17 */
    (0x30001200UL),   /*!< DATA FLASH Page18 */
    (0x30001300UL),   /*!< DATA FLASH Page19 */
    (0x30001400UL),   /*!< DATA FLASH Page20 */
    (0x30001500UL),   /*!< DATA FLASH Page21 */
    (0x30001600UL),   /*!< DATA FLASH Page22 */
    (0x30001700UL),   /*!< DATA FLASH Page23 */
    (0x30001800UL),   /*!< DATA FLASH Page24 */
    (0x30001900UL),   /*!< DATA FLASH Page25 */
    (0x30001A00UL),   /*!< DATA FLASH Page26 */
    (0x30001B00UL),   /*!< DATA FLASH Page27 */
    (0x30001C00UL),   /*!< DATA FLASH Page28 */
    (0x30001D00UL),   /*!< DATA FLASH Page29 */
    (0x30001E00UL),   /*!< DATA FLASH Page30 */
    (0x30001F00UL),   /*!< DATA FLASH Page31 */
    (0x30002000UL)    /*!< DATA FLASH Page32 */
};

static uint32_t fc_const_data_flash_block_address[FC_MAX_DATA_BLOCKS] = {
    (0x30000000UL),   /*!< DATA FLASH Block0 */
    (0x30001000UL),   /*!< DATA FLASH Block1 */
    (0x30002000UL),   /*!< DATA FLASH Block2 */
    (0x30003000UL),   /*!< DATA FLASH Block3 */
    (0x30004000UL),   /*!< DATA FLASH Block4 */
    (0x30005000UL),   /*!< DATA FLASH Block5 */
    (0x30006000UL),   /*!< DATA FLASH Block6 */
    (0x30007000UL)    /*!< DATA FLASH Block7 */
};

static uint32_t fc_const_data_flash_area_address[FC_MAX_AREAS] = {
    (0x30000000UL)   /*!< DATA FLASH AREA0 */
};

static uint32_t fc_const_code_flash_address[FC_MAX_PAGES] = {
    (0x5E000000UL),   /*!< CODE FLASH Page0 */
    (0x5E001000UL),   /*!< CODE FLASH Page1 */
    (0x5E002000UL),   /*!< CODE FLASH Page2 */
    (0x5E003000UL),   /*!< CODE FLASH Page3 */
    (0x5E004000UL),   /*!< CODE FLASH Page4 */
    (0x5E005000UL),   /*!< CODE FLASH Page5 */
    (0x5E006000UL),   /*!< CODE FLASH Page6 */
    (0x5E007000UL),   /*!< CODE FLASH Page7 */
    (0x5E008000UL),   /*!< CODE FLASH Page8 */
    (0x5E009000UL),   /*!< CODE FLASH Page9 */
    (0x5E00A000UL),   /*!< CODE FLASH Page10 */
    (0x5E00B000UL),   /*!< CODE FLASH Page11 */
    (0x5E00C000UL),   /*!< CODE FLASH Page12 */
    (0x5E00D000UL),   /*!< CODE FLASH Page13 */
    (0x5E00E000UL),   /*!< CODE FLASH Page14 */
    (0x5E00F000UL),   /*!< CODE FLASH Page15 */
    (0x5E010000UL),   /*!< CODE FLASH Page16 */
    (0x5E011000UL),   /*!< CODE FLASH Page17 */
    (0x5E012000UL),   /*!< CODE FLASH Page18 */
    (0x5E013000UL),   /*!< CODE FLASH Page19 */
    (0x5E014000UL),   /*!< CODE FLASH Page20 */
    (0x5E015000UL),   /*!< CODE FLASH Page21 */
    (0x5E016000UL),   /*!< CODE FLASH Page22 */
    (0x5E017000UL),   /*!< CODE FLASH Page23 */
    (0x5E018000UL),   /*!< CODE FLASH Page24 */
    (0x5E019000UL),   /*!< CODE FLASH Page25 */
    (0x5E01A000UL),   /*!< CODE FLASH Page26 */
    (0x5E01B000UL),   /*!< CODE FLASH Page27 */
    (0x5E01C000UL),   /*!< CODE FLASH Page28 */
    (0x5E01D000UL),   /*!< CODE FLASH Page29 */
    (0x5E01E000UL),   /*!< CODE FLASH Page30 */
    (0x5E01F000UL),   /*!< CODE FLASH Page31 */
    (0x5E020000UL),   /*!< CODE FLASH Page32 */
    (0x5E021000UL),   /*!< CODE FLASH Page33 */
    (0x5E022000UL),   /*!< CODE FLASH Page34 */
    (0x5E023000UL),   /*!< CODE FLASH Page35 */
    (0x5E024000UL),   /*!< CODE FLASH Page36 */
    (0x5E025000UL),   /*!< CODE FLASH Page37 */
    (0x5E026000UL),   /*!< CODE FLASH Page38 */
    (0x5E027000UL),   /*!< CODE FLASH Page39 */
    (0x5E028000UL),   /*!< CODE FLASH Page40 */
    (0x5E029000UL),   /*!< CODE FLASH Page41 */
    (0x5E02A000UL),   /*!< CODE FLASH Page42 */
    (0x5E02B000UL),   /*!< CODE FLASH Page43 */
    (0x5E02C000UL),   /*!< CODE FLASH Page44 */
    (0x5E02D000UL),   /*!< CODE FLASH Page45 */
    (0x5E02E000UL),   /*!< CODE FLASH Page46 */
    (0x5E02F000UL),   /*!< CODE FLASH Page47 */
    (0x5E030000UL),   /*!< CODE FLASH Page48 */
    (0x5E031000UL),   /*!< CODE FLASH Page49 */
    (0x5E032000UL),   /*!< CODE FLASH Page50 */
    (0x5E033000UL),   /*!< CODE FLASH Page51 */
    (0x5E034000UL),   /*!< CODE FLASH Page52 */
    (0x5E035000UL),   /*!< CODE FLASH Page53 */
    (0x5E036000UL),   /*!< CODE FLASH Page54 */
    (0x5E037000UL),   /*!< CODE FLASH Page55 */
    (0x5E038000UL),   /*!< CODE FLASH Page56 */
    (0x5E039000UL),   /*!< CODE FLASH Page57 */
    (0x5E03A000UL),   /*!< CODE FLASH Page58 */
    (0x5E03B000UL),   /*!< CODE FLASH Page59 */
    (0x5E03C000UL),   /*!< CODE FLASH Page60 */
    (0x5E03D000UL),   /*!< CODE FLASH Page61 */
    (0x5E03E000UL),   /*!< CODE FLASH Page62 */
    (0x5E03F000UL),   /*!< CODE FLASH Page63 */
    (0x5E040000UL),   /*!< CODE FLASH Page64 */
    (0x5E041000UL),   /*!< CODE FLASH Page65 */
    (0x5E042000UL),   /*!< CODE FLASH Page66 */
    (0x5E043000UL),   /*!< CODE FLASH Page67 */
    (0x5E044000UL),   /*!< CODE FLASH Page68 */
    (0x5E045000UL),   /*!< CODE FLASH Page69 */
    (0x5E046000UL),   /*!< CODE FLASH Page70 */
    (0x5E047000UL),   /*!< CODE FLASH Page71 */
    (0x5E048000UL),   /*!< CODE FLASH Page72 */
    (0x5E049000UL),   /*!< CODE FLASH Page73 */
    (0x5E04A000UL),   /*!< CODE FLASH Page74 */
    (0x5E04B000UL),   /*!< CODE FLASH Page75 */
    (0x5E04C000UL),   /*!< CODE FLASH Page76 */
    (0x5E04D000UL),   /*!< CODE FLASH Page77 */
    (0x5E04E000UL),   /*!< CODE FLASH Page78 */
    (0x5E04F000UL),   /*!< CODE FLASH Page79 */
    (0x5E050000UL),   /*!< CODE FLASH Page80 */
    (0x5E051000UL),   /*!< CODE FLASH Page81 */
    (0x5E052000UL),   /*!< CODE FLASH Page82 */
    (0x5E053000UL),   /*!< CODE FLASH Page83 */
    (0x5E054000UL),   /*!< CODE FLASH Page84 */
    (0x5E055000UL),   /*!< CODE FLASH Page85 */
    (0x5E056000UL),   /*!< CODE FLASH Page86 */
    (0x5E057000UL),   /*!< CODE FLASH Page87 */
    (0x5E058000UL),   /*!< CODE FLASH Page88 */
    (0x5E059000UL),   /*!< CODE FLASH Page89 */
    (0x5E05A000UL),   /*!< CODE FLASH Page90 */
    (0x5E05B000UL),   /*!< CODE FLASH Page91 */
    (0x5E05C000UL),   /*!< CODE FLASH Page92 */
    (0x5E05D000UL),   /*!< CODE FLASH Page93 */
    (0x5E05E000UL),   /*!< CODE FLASH Page94 */
    (0x5E05F000UL),   /*!< CODE FLASH Page95 */
    (0x5E060000UL),   /*!< CODE FLASH Page96 */
    (0x5E061000UL),   /*!< CODE FLASH Page97 */
    (0x5E062000UL),   /*!< CODE FLASH Page98 */
    (0x5E063000UL),   /*!< CODE FLASH Page99 */
    (0x5E064000UL),   /*!< CODE FLASH Page100 */
    (0x5E065000UL),   /*!< CODE FLASH Page101 */
    (0x5E066000UL),   /*!< CODE FLASH Page102 */
    (0x5E067000UL),   /*!< CODE FLASH Page103 */
    (0x5E068000UL),   /*!< CODE FLASH Page104 */
    (0x5E069000UL),   /*!< CODE FLASH Page105 */
    (0x5E06A000UL),   /*!< CODE FLASH Page106 */
    (0x5E06B000UL),   /*!< CODE FLASH Page107 */
    (0x5E06C000UL),   /*!< CODE FLASH Page108 */
    (0x5E06D000UL),   /*!< CODE FLASH Page109 */
    (0x5E06E000UL),   /*!< CODE FLASH Page110 */
    (0x5E06F000UL),   /*!< CODE FLASH Page111 */
    (0x5E070000UL),   /*!< CODE FLASH Page112 */
    (0x5E071000UL),   /*!< CODE FLASH Page113 */
    (0x5E072000UL),   /*!< CODE FLASH Page114 */
    (0x5E073000UL),   /*!< CODE FLASH Page115 */
    (0x5E074000UL),   /*!< CODE FLASH Page116 */
    (0x5E075000UL),   /*!< CODE FLASH Page117 */
    (0x5E076000UL),   /*!< CODE FLASH Page118 */
    (0x5E077000UL),   /*!< CODE FLASH Page119 */
    (0x5E078000UL),   /*!< CODE FLASH Page120 */
    (0x5E079000UL),   /*!< CODE FLASH Page121 */
    (0x5E07A000UL),   /*!< CODE FLASH Page122 */
    (0x5E07B000UL),   /*!< CODE FLASH Page123 */
    (0x5E07C000UL),   /*!< CODE FLASH Page124 */
    (0x5E07D000UL),   /*!< CODE FLASH Page125 */
    (0x5E07E000UL),   /*!< CODE FLASH Page126 */
    (0x5E07F000UL)    /*!< CODE FLASH Page127 */
};


/** 
 *  @}
 */ /* End of group FLASH_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_Private_typedef FLASH Private Typedef
 *  @{
 */

/**
 *  @}
 */ /* End of group FLASH_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_Private_variables FLASH Private Variables
 *  @{
 */
/**
 *  @}
 */ /* End of group FLASH_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_Private_fuctions FLASH Private Fuctions
 *  @{
 */
__STATIC_INLINE TXZ_Result fc_enable_areasel4(flash_t *p_obj);
__STATIC_INLINE TXZ_Result fc_disable_areasel4(flash_t *p_obj);
static void fc_write_command(flash_t *p_obj, uint32_t* src_address, uint32_t* dst_address, uint32_t size);
static void fc_write_command4(flash_t *p_obj, uint32_t* src_address, uint32_t* dst_address, uint32_t size);
static void fc_erase_command(flash_t *p_obj, uint32_t* flash_top_address, uint32_t* erase_top_address, fc_erase_kind_t kind);
static void fc_erase_command4(flash_t *p_obj, uint32_t* flash_top_address, uint32_t* erase_top_address, fc_erase_kind_t kind);
static TXZ_Result fc_blank_check(uint32_t* address, uint32_t size);
static void fc_bank_change(flash_t *p_obj, uint32_t bank);

/*--------------------------------------------------*/
/** 
  * @brief  Enables the AREA4.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function write the FCAREASEL regiset.
  */
/*--------------------------------------------------*/
__STATIC_INLINE TXZ_Result fc_enable_areasel4(flash_t *p_obj)
{
    TXZ_Result retval = TXZ_ERROR;

    uint32_t reg = p_obj->p_instance->AREASEL & FC_AREASEL_MASK_AREA4;
    reg |= FC_AREASEL_AREA4;
    /* Writes the FCKER register the KEYCODE. */
    p_obj->p_instance->KCR = FC_KCR_KEYCODE;

    /* Selects the area4 */
    p_obj->p_instance->AREASEL = reg;

    /* Confirms the FCAREASEL register the SSF4 was set. */
    while(1){
        uint32_t i = p_obj->p_instance->AREASEL;
        if((i & FC_AREASEL4_WRITE_MODE) == FC_AREASEL4_WRITE_MODE){
            retval = TXZ_SUCCESS;
            break;
        }
    }

    return(retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Disables the AREA4.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function write the FCAREASEL regiset.
  */
/*--------------------------------------------------*/
__STATIC_INLINE TXZ_Result fc_disable_areasel4(flash_t *p_obj)
{
    TXZ_Result retval = TXZ_ERROR;
    uint32_t reg = p_obj->p_instance->AREASEL & FC_AREASEL_MASK_AREA4;
    reg |= FC_AREASEL_EXPECT_AREA4;

    /* Writes the FCKER register the KEYCODE. */
    p_obj->p_instance->KCR = FC_KCR_KEYCODE;

    /* Selects the area4 */
    p_obj->p_instance->AREASEL = reg;

    /* Confirms the SSF0 of the FCAREASEL  register is not set. */
    while(1){
        uint32_t i = p_obj->p_instance->AREASEL;
        if((i & FC_AREASEL4_WRITE_MODE) != FC_AREASEL4_WRITE_MODE){
            retval = TXZ_SUCCESS;
            break;
        }
    }

    return(retval);
}


/*--------------------------------------------------*/
/** 
  * @brief  Writes data of the FLASH ROM.
  * @param  p_obj       : FLASH object.
  * @param  src_address : write src address.
  * @param  dst_address : write dst address.
  * @param  size        : write size.
  * @return -
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
static void fc_write_command(flash_t *p_obj, uint32_t* src_address, uint32_t* dst_address, uint32_t size)
{
    TXZ_Result retval;
    volatile uint32_t* addr1 = (uint32_t *) (FC_CODE_FLASH_ADDRESS_TOP + FC_CMD_BC1_ADDR);
    volatile uint32_t* addr2 = (uint32_t *) (FC_CODE_FLASH_ADDRESS_TOP + FC_CMD_BC2_ADDR);
    volatile uint32_t* addr3 = (uint32_t *) ((uint32_t)dst_address + FC_CODE_FLASH_ADDRESS_TOP);
    uint32_t* source = (uint32_t *) src_address;

    /* Enables the AREA0. Write Mode. */
    retval = fc_enable_areasel(p_obj);

    if(retval == TXZ_SUCCESS){
        uint32_t i;

        *addr1 = (0x000000AAUL); /* bus cycle 1 */
        *addr2 = (0x00000055UL); /* bus cycle 2 */
        *addr1 = (0x000000A0UL); /* bus cycle 3 */
        for(i=(0UL); i<size; i+=(0x4UL)){
            *addr3 = *source;
            source++;
        }

        /* Confirmation of the works start of ROM. */
        while(fc_get_status(p_obj) == TXZ_DONE){
        };

        /* Waits for a finish of the works in the code Flash ROM. */
        while(fc_get_status(p_obj) == TXZ_BUSY){
        };
    }

    /* Disables the AREA0. Read Mode. */
    retval = fc_disable_areasel(p_obj);
}

/*--------------------------------------------------*/
/** 
  * @brief  Writes data of the DATA FLASH.
  * @param  p_obj       : FLASH object.
  * @param  src_address : write src address.
  * @param  dst_address : write dst address.
  * @param  size        : write size.
  * @return -
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
static void fc_write_command4(flash_t *p_obj, uint32_t* src_address, uint32_t* dst_address, uint32_t size)
{
    uint32_t retval;
    volatile uint32_t* addr1;
    volatile uint32_t* addr2;
    volatile uint32_t* addr3;
    uint32_t* source = (uint32_t *) src_address;

    addr1 = (uint32_t *) ((uint32_t)FC_CODE_DATA_ADDRESS_TOP + FC_CMD_BC1_ADDR);
    addr2 = (uint32_t *) ((uint32_t)FC_CODE_DATA_ADDRESS_TOP + FC_CMD_BC2_ADDR);
    addr3 = (uint32_t *) ((uint32_t)dst_address);
    /* Enables the AREA4. Write Mode. */
    retval = fc_enable_areasel4(p_obj);

    if(retval == TXZ_SUCCESS){
        uint32_t i;

        *addr1 = (0x000000AAUL); /* bus cycle 1 */
        *addr2 = (0x00000055UL); /* bus cycle 2 */
        if((uint32_t)dst_address >= 0x30000000)
        {
          *addr1 = (0x000000C0UL); /* bus cycle 3 */
        }else{
          *addr1 = (0x000000A0UL); /* bus cycle 3 */
        }
        for(i=(0UL); i<size; i+=(0x4UL)){
            *addr3 = *source;
            source++;
        }

        /* Confirmation of the works start of ROM. */
        while(fc_get_status(p_obj) == TXZ_DONE){
        };

        /* Waits for a finish of the works in the code DATA Flash. */
        while(fc_get_status(p_obj) == TXZ_BUSY){
        };
    }

    /* Disables the AREA4. Read Mode. */
    retval = fc_disable_areasel4(p_obj);
}
/*--------------------------------------------------*/
/** 
  * @brief  Verifies data of the FLASH ROM.
  * @param  src_address : verify src address.
  * @param  dst_address : verify dst address.
  * @param  size        : verify size.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
TXZ_Result fc_verify_check(uint32_t* src_address, uint32_t* dst_address, uint32_t size)
{
    TXZ_Result retval = TXZ_ERROR;
    int res = memcmp(src_address, dst_address, size);
    if(res == 0){
        retval = TXZ_SUCCESS;
    }

    return(retval);
}


/*--------------------------------------------------*/
/** 
  * @brief  Auro page erase command of the flash ROM.
  * @param  p_obj             : FLASH object.
  * @param  flash_top_address : flash top address
  * @param  erase_top_address : erase top address
  * @param  kind              : Chip, Area, Block, Page, etc.
  * @return -
  * @note   This function erases specified place of the flash ROM.
  */
/*--------------------------------------------------*/
static void fc_erase_command(flash_t *p_obj, uint32_t* flash_top_address, uint32_t* erase_top_address, fc_erase_kind_t kind)
{
    TXZ_Result retval;
    volatile uint32_t *addr1 = (uint32_t *) ((uint32_t)flash_top_address + FC_CMD_BC1_ADDR);
    volatile uint32_t *addr2 = (uint32_t *) ((uint32_t)flash_top_address + FC_CMD_BC2_ADDR);
    volatile uint32_t *addr3 = (uint32_t *) erase_top_address;

    /* Enables the AREA0. Write Mode. */
    retval = fc_enable_areasel(p_obj);

    if(retval == TXZ_SUCCESS){
        *addr1 = (0x000000AAUL);
        *addr2 = (0x00000055UL);
        *addr1 = (0x00000080UL);
        *addr1 = (0x000000AAUL);
        *addr2 = (0x00000055UL);
        *addr3 = kind;

        /* Confirmation of the works start of ROM. */
        while(fc_get_status(p_obj) == TXZ_DONE){
        };

        /* Waits for a finish of the works in the code Flash ROM. */
        while(fc_get_status(p_obj) == TXZ_BUSY){
        };
    }

    /* Disables the AREA0. Read Mode. */
    retval = fc_disable_areasel(p_obj);

}


/*--------------------------------------------------*/
/** 
  * @brief  Auto page erase command of the flash ROM.
  * @param  p_obj             : FLASH object.
  * @param  flash_top_address : flash top address
  * @param  erase_top_address : erase top address
  * @param  kind              : Chip, Area, Block, Page, etc.
  * @return -
  * @note   This function erases specified place of the flash ROM.
  */
/*--------------------------------------------------*/
static void fc_erase_command4(flash_t *p_obj, uint32_t* flash_top_address, uint32_t* erase_top_address, fc_erase_kind_t kind)
{
    TXZ_Result retval;
    volatile uint32_t *addr1 = (uint32_t *) ((uint32_t)flash_top_address + FC_CMD_BC1_ADDR);
    volatile uint32_t *addr2 = (uint32_t *) ((uint32_t)flash_top_address + FC_CMD_BC2_ADDR);
    volatile uint32_t *addr3 = (uint32_t *) erase_top_address;

    /* Enables the AREA4. Write Mode. */
    retval = fc_enable_areasel4(p_obj);

    if(retval == TXZ_SUCCESS){
        *addr1 = (0x000000AAUL);
        *addr2 = (0x00000055UL);
        *addr1 = (0x00000080UL);
        *addr1 = (0x000000AAUL);
        *addr2 = (0x00000055UL);
        *addr3 = kind;

        /* Confirmation of the works start of ROM. */
        while(fc_get_status(p_obj) == TXZ_DONE){
        };

        /* Waits for a finish of the works in the code DATA Flash. */
        while(fc_get_status(p_obj) == TXZ_BUSY){
        };
    }

    /* Disables the AREA4. Read Mode. */
    retval = fc_disable_areasel4(p_obj);
}

/*--------------------------------------------------*/
/** 
  * @brief  Checks a blank of the DATA FLASH every 4bytes.
  * @param  address   : blank check address
  * @param  size      : blank check size
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
static TXZ_Result fc_blank_check( uint32_t* address, uint32_t size)
{
    TXZ_Result retval = TXZ_SUCCESS;

    uint32_t i;

    for(i=0; i<(size/sizeof(uint32_t)); i++){
        uint32_t* addr = &address[i];
        if(*addr != FC_BRANK_VALUE){
            retval = TXZ_ERROR;
            break;
        }
    }
    return (retval);
}


/*--------------------------------------------------*/
/** 
  * @brief  bank change.
  * @param  p_obj : FLASH object.
  * @param  bank  : FC_BANK_USER_INFO, FC_BANK_CODE_FLASH
  * @return none.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
static void fc_bank_change(flash_t *p_obj, uint32_t bank)
{
    if(bank == FC_BANK_USER_INFO){
        /* Flash Buffer disable */
        fc_read_buf_set(p_obj, TXZ_DISABLE);
        /* change user information area */
        p_obj->p_instance->BNKCR = FC_BANK_USER_INFO;
        while(p_obj->p_instance->BNKCR != FC_BANK_USER_INFO);
    }
    else if(bank == FC_BANK_CODE_FLASH){
        /* change code flash */
        p_obj->p_instance->BNKCR = FC_BANK_CODE_FLASH;
        while(p_obj->p_instance->BNKCR != FC_BANK_CODE_FLASH);
        /* Flash Buffer enable */
        fc_read_buf_set(p_obj, TXZ_ENABLE);
    }
    else {
        /* do nothing */
    }
}



/**
 *  @}
 */ /* End of group FLASH_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup FLASH_Exported_functions FLASH Exported Functions
 *  @{
 */

/*--------------------------------------------------*/
/** 
  * @brief  Enables the AREA0.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function write the FCAREASEL regiset.
  */
/*--------------------------------------------------*/
TXZ_Result fc_enable_areasel(flash_t *p_obj)
{
    TXZ_Result retval = TXZ_ERROR;
    uint32_t reg = p_obj->p_instance->AREASEL & FC_AREASEL_MASK_AREA0;

    reg |= FC_AREASEL_AREA0;
    /* Writes the FCKER register the KEYCODE. */
    p_obj->p_instance->KCR = FC_KCR_KEYCODE;

    /* Selects the area0 */
    p_obj->p_instance->AREASEL = reg;

    /* Confirms the FCAREASEL register the SSF0 was set. */
    while(1){
        uint32_t i = p_obj->p_instance->AREASEL;
        if((i & FC_AREASEL_WRITE_MODE) == FC_AREASEL_WRITE_MODE){
            retval = TXZ_SUCCESS;
            break;
        }
    }

    return(retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Disables the AREA0.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function write the FCAREASEL regiset.
  */
/*--------------------------------------------------*/
TXZ_Result fc_disable_areasel(flash_t *p_obj)
{
    TXZ_Result retval = TXZ_ERROR;
    uint32_t reg = p_obj->p_instance->AREASEL & FC_AREASEL_MASK_AREA0;

    reg |= FC_AREASEL_EXPECT_AREA0;
    /* Writes the FCKER register the KEYCODE. */
    p_obj->p_instance->KCR = FC_KCR_KEYCODE;

    /* Selects the area0 */
    p_obj->p_instance->AREASEL = reg;

    /* Confirms the SSF0 of the FCAREASEL  register is not set. */
    while(1){
        uint32_t i = p_obj->p_instance->AREASEL;
        if((i & FC_AREASEL_WRITE_MODE) != FC_AREASEL_WRITE_MODE){
            retval = TXZ_SUCCESS;
            break;
        }
    }

    return(retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Get the status of the flash auto operation.
  * @param  p_obj    : FLASH object.
  * @return Result.
  * @retval TXZ_BUSY : Busy.
  * @retval TXZ_DONE : Done.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
TXZ_WorkState fc_get_status(flash_t *p_obj)
{
    TXZ_WorkState retval = TXZ_BUSY;
    uint32_t work32;

    /* Reads the FCSR0. Masks the other specfic status */
    work32 = p_obj->p_instance->SR0 & (uint32_t)p_obj->init.status;

    /* Confirms the specific status of the flash ROM */
    if(work32 == (uint32_t)p_obj->init.status){
        retval = TXZ_DONE;
    }
    return (retval);
}


/*--------------------------------------------------*/
/** 
  * @brief  Auto write command of the code flash ROM.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function writes 16bytes data to the code Flash ROM.
  */
/*--------------------------------------------------*/
TXZ_Result fc_write_code_flash(flash_t *p_obj)
{
    TXZ_Result retval = TXZ_SUCCESS;

    /* Checks the code Flash ROM status */
    if(fc_get_status(p_obj) == TXZ_DONE){

        uint32_t i;
        /* Checks the code Flash ROM status */
        for(i=0;i<p_obj->init.size;i+=(uint32_t)(0x10UL)){
            /* Writes 16bytes data. */
            fc_write_command(p_obj, (uint32_t*)((uint32_t)p_obj->init.src_address+i), (uint32_t*)((uint32_t)p_obj->init.dst_address+i), (uint32_t)(0x10UL));
        }

        /* Verifies user data and data of the Flash ROM. */
        retval = fc_verify_check(p_obj->init.src_address, p_obj->init.dst_address, p_obj->init.size);
    }

    return (retval);
}


/*--------------------------------------------------*/
/** 
  * @brief  Auto write command of the data flash.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function writes 16bytes data to the DATA Flash.
  */
/*--------------------------------------------------*/
TXZ_Result fc_write_data_flash(flash_t *p_obj)
{
    TXZ_Result retval = TXZ_SUCCESS;

    /* Checks the code DATA Flash status */
    if(fc_get_status(p_obj) == TXZ_DONE){

        uint32_t i;

       for(i=0;i<p_obj->init.size;i+=(uint32_t)(0x4UL)){
            /* Writes 4bytes data. */
            fc_write_command4(p_obj, (uint32_t*)((uint32_t)p_obj->init.src_address+i), (uint32_t*)((uint32_t)p_obj->init.dst_address+i), (uint32_t)(0x4UL));
       }
        /* Verifies user data and data of the Flash ROM. */
        retval = fc_verify_check(p_obj->init.src_address, p_obj->init.dst_address, p_obj->init.size);
    }
    return (retval);
}


/*--------------------------------------------------*/
/** 
  * @brief  Auro page erase command of the code flash ROM.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function erases specified page of the code Flash ROM and checks a blank.
  */
/*--------------------------------------------------*/
TXZ_Result fc_erase_page_code_flash(flash_t *p_obj)
{
    TXZ_Result retval = TXZ_SUCCESS;

    /* Checks the code Flash ROM status */
    if(fc_get_status(p_obj) == TXZ_DONE){
        /* Checks the number of maximum pages. */
        if((p_obj->init.first_page_code + p_obj->init.num_of_pages_code) <= FC_MAX_PAGES){
            uint8_t i;
            for(i=0; i<p_obj->init.num_of_pages_code ; i++){
                /* Erases the specific page. */
                fc_erase_command(p_obj, 
                                (uint32_t*)FC_CODE_FLASH_ADDRESS_TOP,
                                (uint32_t*)fc_const_code_flash_address[p_obj->init.first_page_code+i],
                                FC_ERASE_KIND_PAGE);
            }
            /* Checks a blank of the specific page. */
            if(fc_blank_check((uint32_t*)fc_const_code_flash_address[p_obj->init.first_page_code], FC_PAGE_SIZE*(uint32_t)p_obj->init.num_of_pages_code) == TXZ_ERROR){
                retval = TXZ_ERROR;
            }
        }
        else{
            retval = TXZ_ERROR;
        }
    }
    else {
        retval = TXZ_ERROR;
    }

    return (retval);
}


/*--------------------------------------------------*/
/** 
  * @brief  Auro page erase command of the data flash.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function erases specified page of the data Flash and checks a blank.
  */
/*--------------------------------------------------*/
TXZ_Result fc_erase_page_data_flash(flash_t *p_obj)
{
    TXZ_Result retval = TXZ_SUCCESS;

   /* Checks the data Flash status */
    if(fc_get_status(p_obj) == TXZ_DONE){
        /* Checks the number of maximum pages. */
        if((p_obj->init.first_page_data + p_obj->init.num_of_pages_data) <= FC_MAX_DATA_PAGES){
            uint8_t i;
            for(i=0; i<p_obj->init.num_of_pages_data ; i++){
                /* Erases the specific page. */
                fc_erase_command4(p_obj,
                                (uint32_t*)FC_DATA_FLASH_ADDRESS_TOP,
                                (uint32_t*)fc_const_data_flash_address[p_obj->init.first_page_data+i],
                                FC_ERASE_KIND_PAGE);
            }
            /* Checks a blank of the specific page. */
            if(fc_blank_check((uint32_t*)fc_const_data_flash_address[p_obj->init.first_page_data], FC_DATA_PAGE_SIZE*(uint32_t)p_obj->init.num_of_pages_data) == TXZ_ERROR){
                retval = TXZ_ERROR;
            }
        }
        else{
            retval = TXZ_ERROR;
        }
    }
    else {
        retval = TXZ_ERROR;
    }

    return (retval);
}


/*--------------------------------------------------*/
/** 
  * @brief  Checks a blank of the code FLASH ROM of specified pages.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
TXZ_Result fc_blank_check_page_code_flash(flash_t *p_obj)
{
    TXZ_Result retval;

    uint32_t* address = (uint32_t*)fc_const_code_flash_address[p_obj->init.first_page_code];
    uint32_t size = ((uint32_t)(p_obj->init.last_page_code - p_obj->init.first_page_code + 1) * (uint32_t)FC_PAGE_SIZE);
    
    retval = fc_blank_check(address, size);

    return (retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Checks a blank of the data FLASH of specified pages.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
TXZ_Result fc_blank_check_page_data_flash(flash_t *p_obj)
{
    TXZ_Result retval;

    uint32_t* address = (uint32_t*)fc_const_data_flash_address[p_obj->init.first_page_data];
    uint32_t size = ((uint32_t)(p_obj->init.last_page_data - p_obj->init.first_page_data + 1) * (uint32_t)FC_DATA_PAGE_SIZE);
    
    retval = fc_blank_check(address, size);

    return (retval);
}
/*--------------------------------------------------*/
/** 
  * @brief  Auto block erase command of the data flash.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function erases specified block of the data Flash and checks a blank.
  */
/*--------------------------------------------------*/
TXZ_Result fc_erase_block_data_flash(flash_t *p_obj)
{
    TXZ_Result retval = TXZ_SUCCESS;

    /* Checks the data Flash status */
    if(fc_get_status(p_obj) == TXZ_DONE){
        /* Checks the number of maximum blocks. */
        if((p_obj->init.first_block + p_obj->init.num_of_block) <= FC_MAX_DATA_BLOCKS){
            uint8_t i;
            for(i=0; i<p_obj->init.num_of_block ; i++){
                /* Erases the specific block. */
                fc_erase_command4(p_obj,
                                (uint32_t*)FC_DATA_FLASH_ADDRESS_TOP,
                                (uint32_t*)fc_const_data_flash_block_address[p_obj->init.first_block+i],
                                FC_ERASE_KIND_BLOCK);
            }
            /* Checks a blank of the specific block. */
            if(fc_blank_check((uint32_t*)fc_const_data_flash_block_address[p_obj->init.first_block], FC_DATA_BLOCK_SIZE*(uint32_t)p_obj->init.num_of_block) == TXZ_ERROR){
                retval = TXZ_ERROR;
            }
        }
        else{
            retval = TXZ_ERROR;
        }
    }
    else {
        retval = TXZ_ERROR;
    }

    return (retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Checks a blank of the data FLASH of specified blocks.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
TXZ_Result fc_blank_check_block_data_flash(flash_t *p_obj)
{
    TXZ_Result retval;

    uint32_t* address = (uint32_t*)fc_const_data_flash_block_address[p_obj->init.first_block];
    uint32_t size = ((uint32_t)(p_obj->init.last_block - p_obj->init.first_block + 1) * (uint32_t)FC_DATA_BLOCK_SIZE);
    
    retval = fc_blank_check(address, size);

    return (retval);
}
/*--------------------------------------------------*/
/** 
  * @brief  Auto area erase command of the data flash.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function erases specified block of the data Flash and checks a blank.
  */
/*--------------------------------------------------*/
TXZ_Result fc_erase_area_data_flash(flash_t *p_obj)
{
    TXZ_Result retval = TXZ_SUCCESS;

    /* Checks the data Flash status */
    if(fc_get_status(p_obj) == TXZ_DONE){
        /* Checks the number of maximum blocks. */
        if(p_obj->init.area == 0){
            /* Erases the specific block. */
            fc_erase_command4(p_obj,
                            (uint32_t*)FC_DATA_FLASH_ADDRESS_TOP,
                            (uint32_t*)fc_const_data_flash_area_address[p_obj->init.area],
                            FC_ERASE_KIND_AREA);
            /* Checks a blank of the specific block. */
            if(fc_blank_check((uint32_t*)fc_const_data_flash_area_address[p_obj->init.area], FC_DATA_AREA_SIZE) == TXZ_ERROR){
                retval = TXZ_ERROR;
            }
        }
        else{
            retval = TXZ_ERROR;
        }
    }
    else {
        retval = TXZ_ERROR;
    }

    return (retval);
}
/*--------------------------------------------------*/
/** 
  * @brief  Checks a blank of the data FLASH of specified area.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
TXZ_Result fc_blank_check_area_data_flash(flash_t *p_obj)
{
    TXZ_Result retval;

    uint32_t* address = (uint32_t*)fc_const_data_flash_area_address[p_obj->init.area];
    uint32_t size = ((uint32_t)(1) * (uint32_t)FC_DATA_AREA_SIZE);

    retval = fc_blank_check(address, size);

    return (retval);
}


/*--------------------------------------------------*/
/** 
  * @brief  Auto write command of the user information area.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function writes 16bytes data to the code Flash ROM.
  */
/*--------------------------------------------------*/
TXZ_Result fc_write_user_information_area(flash_t *p_obj)
{
    TXZ_Result retval = TXZ_SUCCESS;
    uint32_t i;

    /* Checks the code Flash ROM status */
    if(fc_get_status(p_obj) == TXZ_DONE){
        fc_bank_change(p_obj, FC_BANK_USER_INFO);

        /* Checks the code Flash ROM status */
        for(i=0;i<p_obj->init.size;i+=(uint32_t)(0x10UL)){
            /* Writes 16bytes data. */
            fc_write_command(p_obj, (uint32_t *)((uint32_t)p_obj->init.src_address+i), (uint32_t *)((uint32_t)fc_const_code_flash_address[5]-FC_CODE_FLASH_ADDRESS_TOP+i), (uint32_t)(0x10UL));
        }

        /* Verifies user data and data of the Flash ROM. */
        retval = fc_verify_check(p_obj->init.src_address, (uint32_t *)fc_const_code_flash_address[5], p_obj->init.size);

        fc_bank_change(p_obj, FC_BANK_CODE_FLASH);
    }

    return (retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Auro page erase command of the user information area.
  * @param  p_obj       : FLASH object.
  * @return Result.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  *         This function erases specified page of the code Flash ROM and checks a blank.
  */
/*--------------------------------------------------*/
TXZ_Result fc_erase_user_information_area(flash_t *p_obj)
{
    TXZ_Result retval = TXZ_SUCCESS;

    /* Checks the code Flash ROM status */
    if(fc_get_status(p_obj) == TXZ_DONE){
        fc_bank_change(p_obj, FC_BANK_USER_INFO);

        /* Erases the specific page. */
        fc_erase_command(p_obj,
                        (uint32_t*)FC_CODE_FLASH_ADDRESS_TOP,
                        (uint32_t*)fc_const_code_flash_address[5],
                        FC_ERASE_KIND_PAGE);

        /* Checks a blank of the specific page. */
        if(fc_blank_check((uint32_t*)fc_const_code_flash_address[5], FC_PAGE_SIZE) == TXZ_ERROR){
            retval = TXZ_ERROR;
        }

        fc_bank_change(p_obj, FC_BANK_CODE_FLASH);
    }
    else {
        retval = TXZ_ERROR;
    }

    return (retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  Raed of the user information area.
  * @param  p_obj       : FLASH object.
  * @retval none.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
void fc_read_user_information_area(flash_t *p_obj)
{
    uint8_t i;
    uint32_t *dst = p_obj->init.dst_address;
    uint32_t *src = (uint32_t *)fc_const_code_flash_address[5];

    fc_read_clock_set(p_obj);
    fc_bank_change(p_obj, FC_BANK_USER_INFO);

    for(i=(0UL); i<p_obj->init.size; i+=(0x4UL)){
        *dst = *src;
        dst++;
        src++;
    }

    fc_bank_change(p_obj, FC_BANK_CODE_FLASH);
}

/*--------------------------------------------------*/
/** 
  * @brief  Read clock set.
  * @param  p_obj       : FLASH object.
  * @retval none.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
void fc_read_clock_set(flash_t *p_obj)
{
    uint32_t regval = 0;

    SystemCoreClockUpdate();
    if(SystemCoreClock > SYSCORECLOCK_80M)
    {
        regval = (uint32_t)(FC_ACCR_FDLC_5 | FC_ACCR_FCLC_5);
        p_obj->p_instance->KCR = FC_KCR_KEYCODE;
        p_obj->p_instance->FCACCR = regval;
        while(p_obj->p_instance->FCACCR != (uint32_t)(FC_ACCR_FDLC_5 | FC_ACCR_FCLC_5))
        {
            /* no processing */
        }
    } else {
        regval = (uint32_t)(FC_ACCR_FDLC_4 | FC_ACCR_FCLC_4);
        p_obj->p_instance->KCR = FC_KCR_KEYCODE;
        p_obj->p_instance->FCACCR = regval;
        while(p_obj->p_instance->FCACCR != (uint32_t)(FC_ACCR_FDLC_4 | FC_ACCR_FCLC_4))
        {
            /* no processing */
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  fc fixed clock set.
  * @param  p_obj       : FLASH object.
  * @param  sysclock    : system clock (Hz)
  * @retval none.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
void fc_fixed_clock_set(flash_t *p_obj, uint32_t sysclock )
{
    uint32_t regval = 0;

    if(sysclock > SYSCORECLOCK_80M)
    {
        regval = (uint32_t)(FC_ACCR_FDLC_5 | FC_ACCR_FCLC_5);
        p_obj->p_instance->KCR = FC_KCR_KEYCODE;
        p_obj->p_instance->FCACCR = regval;
        while(p_obj->p_instance->FCACCR != (uint32_t)(FC_ACCR_FDLC_5 | FC_ACCR_FCLC_5))
        {
            /* no processing */
        }
    } else {
        regval = (uint32_t)(FC_ACCR_FDLC_4 | FC_ACCR_FCLC_4);
        p_obj->p_instance->KCR = FC_KCR_KEYCODE;
        p_obj->p_instance->FCACCR = regval;
        while(p_obj->p_instance->FCACCR != (uint32_t)(FC_ACCR_FDLC_4 | FC_ACCR_FCLC_4))
        {
            /* no processing */
        }
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Protect clear.
  * @param  p_obj              : FLASH object.
  * @param  type               : flash type.
  * @param  flash_top_address  : flash top address.
  * @param  protect_address    : protect top address.
  * @param  protect            : protect.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
TXZ_Result fc_protect_clear(flash_t *p_obj, uint32_t type, uint32_t* flash_top_address, uint32_t* protect_address, uint32_t protect)
{
    TXZ_Result retval = TXZ_SUCCESS;
    volatile uint32_t *addr1 = (uint32_t *) ((uint32_t)flash_top_address + FC_CMD_BC1_ADDR);
    volatile uint32_t *addr2 = (uint32_t *) ((uint32_t)flash_top_address + FC_CMD_BC2_ADDR);
    volatile uint32_t *addr3 = (uint32_t *) protect_address;
    uint32_t work;

    /* Checks the code Flash ROM status */
    if(fc_get_status(p_obj) == TXZ_DONE){

        /* protect bit set */
        if(type != FC_DATA_FLASH){
            if(type == FC_CODE_FLASH_BLK){
                work = (uint32_t)(~protect);
                work &= 0x0000FFFEUL;
                p_obj->p_instance->KCR = FC_KCR_KEYCODE;
                p_obj->p_instance->PMR1 &= work;
            }
            else{
                work = (uint32_t)(~protect);
                work &= 0x000000FFUL;
                p_obj->p_instance->KCR = FC_KCR_KEYCODE;
                p_obj->p_instance->PMR0 &= work;
            }
            fc_enable_areasel(p_obj);


            *addr1 = (0x000000AAUL);
            *addr2 = (0x00000055UL);
            *addr1 = (0x00000080UL);
            *addr1 = (0x000000AAUL);
            *addr2 = (0x00000055UL);
            *addr3 = (0x00000060UL);

            /* Confirmation of the works start of ROM. */
            while(fc_get_status(p_obj) == TXZ_DONE){
            };

            /* Waits for a finish of the works in the code DATA Flash. */
            while(fc_get_status(p_obj) == TXZ_BUSY){
            };
            fc_disable_areasel(p_obj);
        }
        else{
            work = (uint32_t)(~protect);
            work &= 0x000000FFUL;
            p_obj->p_instance->KCR = FC_KCR_KEYCODE;
            p_obj->p_instance->PMR6 &= work;
            fc_enable_areasel4(p_obj);

            *addr1 = (0x000000AAUL);
            *addr2 = (0x00000055UL);
            *addr1 = (0x00000080UL);
            *addr1 = (0x000000AAUL);
            *addr2 = (0x00000055UL);
            *addr3 = (0x00000060UL);

            /* Confirmation of the works start of ROM. */
            while(fc_get_status(p_obj) == TXZ_DONE){
            };

            /* Waits for a finish of the works in the code DATA Flash. */
            while(fc_get_status(p_obj) == TXZ_BUSY){
            };
            fc_disable_areasel4(p_obj);
        }
    }
    else {
        retval = TXZ_ERROR;
    }

    return (retval);
}


/*--------------------------------------------------*/
/** 
  * @brief  Check protect status.
  * @param  p_obj              : FLASH object.
  * @param  type               : flash type.
  * @param  protect            : protect.
  * @retval TXZ_BUSY : Busy.
  * @retval TXZ_DONE : Done.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
TXZ_WorkState fc_protect_status(flash_t *p_obj, uint32_t type, uint32_t protect)
{
    TXZ_WorkState retval = TXZ_DONE;

    /* protect bit check */
    if(type != FC_DATA_FLASH){
        if(type == FC_CODE_FLASH_BLK){
            if((p_obj->p_instance->PSR1 & protect) == protect){
                retval = TXZ_BUSY;
            }
        }
        else{
            if((p_obj->p_instance->PSR0 & protect) == protect){
                retval = TXZ_BUSY;
            }
        }
    }
    else{
        if((p_obj->p_instance->PSR6 & protect) == protect){
            retval = TXZ_BUSY;
        }
    }

    return (retval);
}


/*--------------------------------------------------*/
/** 
  * @brief  Security clear.
  * @param  p_obj              : FLASH object.
  * @param  flash_top_address  : flash top address.
  * @param  security_address   : security top address.
  * @retval TXZ_SUCCESS :Success.
  * @retval TXZ_ERROR   :Failure.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
TXZ_Result fc_security_clear(flash_t *p_obj, uint32_t* flash_top_address, uint32_t* security_address)
{
    TXZ_Result retval = TXZ_SUCCESS;
    volatile uint32_t *addr1 = (uint32_t *) ((uint32_t)flash_top_address + FC_CMD_BC1_ADDR);
    volatile uint32_t *addr2 = (uint32_t *) ((uint32_t)flash_top_address + FC_CMD_BC2_ADDR);
    volatile uint32_t *addr3 = (uint32_t *) security_address;

    /* Checks the code Flash ROM status */
    if(fc_get_status(p_obj) == TXZ_DONE){

        /* security bit set */
        p_obj->p_instance->KCR = FC_KCR_KEYCODE;
        p_obj->p_instance->SBMR = FC_SEC_MASK;
        fc_enable_areasel(p_obj);
        fc_enable_areasel4(p_obj);

        *addr1 = (0x000000AAUL);
        *addr2 = (0x00000055UL);
        *addr1 = (0x00000080UL);
        *addr1 = (0x000000AAUL);
        *addr2 = (0x00000055UL);
        *addr3 = (0x00000060UL);

        /* Confirmation of the works start of ROM. */
        while(fc_get_status(p_obj) == TXZ_DONE){
        };

        /* Waits for a finish of the works in the code DATA Flash. */
        while(fc_get_status(p_obj) == TXZ_BUSY){
        };
        fc_disable_areasel(p_obj);
        fc_disable_areasel4(p_obj);
    }
    else {
        retval = TXZ_ERROR;
    }

    return (retval);
}


/*--------------------------------------------------*/
/** 
  * @brief  Check security status.
  * @param  p_obj              : FLASH object.
  * @retval TXZ_BUSY : Busy.
  * @retval TXZ_DONE : Done.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
TXZ_WorkState fc_security_status(flash_t *p_obj)
{
    TXZ_WorkState retval = TXZ_DONE;

    /* security bit check */
    if((p_obj->p_instance->SSR & FC_SEC_ENABLE) == FC_SEC_ENABLE){
        retval = TXZ_BUSY;
    }
    return (retval);
}

/*--------------------------------------------------*/
/** 
  * @brief  read buffer enable/disable.
  * @param  p_obj : FLASH object.
  * @param  state : read buffer enable/disable.
  * @return none.
  * @note   It works in the inner RAM.
  */
/*--------------------------------------------------*/
void fc_read_buf_set(flash_t *p_obj, TXZ_FunctionalState state)
{
    if(state == TXZ_ENABLE){
        /* Flash Buffer enable */
        p_obj->p_instance->BUFDISCLR = FC_BUFFER_ENABLE;
        while(p_obj->p_instance->BUFDISCLR != FC_BUFFER_ENABLE);
    }
    else {
        /* Flash Buffer disable */
        p_obj->p_instance->BUFDISCLR = FC_BUFFER_DISABLE;
        while(p_obj->p_instance->BUFDISCLR != FC_BUFFER_DISABLE);
    }
}
/**
 *  @}
 */ /* End of group FLASH_Exported_functions */

/**
 *  @}
 */ /* End of group FLASH */

/**
 *  @} 
 */ /* End of group Periph_Driver */

#endif                          /* defined(__FLASH512UD32_B_H) */

#ifdef __cplusplus
}
#endif /* __cplusplus */
