/**
 *******************************************************************************
 * @file    ramp_a.c
 * @brief   This file provides API functions for RAMPARITY.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */

#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "ramp.h"

#if defined(__RAMP_A_H)
/**
 *  @addtogroup Periph_Driver Peripheral Driver
 *  @{
 */

/** 
 *  @defgroup RAMP RAMP
 *  @brief    RAMP Driver.
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RAMP_Private_macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group RAMP_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RAMP_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group RAMP_Private_define */

/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RAMP_Private_define
 *  @{
 */

/** 
 *  @defgroup Null Null Pointer
 *  @brief    Null Pointer.
 *  @{
 */
#define REG_RPAR_NULL                  ((void *)0)
/**
 *  @}
 */ /* End of group Null */

/** 
 *  @defgroup CheckParamResult Parameter Check Result
 *  @brief    Whether the parameter is specified or not.
 *  @{
 */
#define REG_RPAR_PARAM_OK              ((int32_t)1)              /*!< Parameter is valid(specified).         */
#define REG_RPAR_PARAM_NG              ((int32_t)0)              /*!< Parameter is invalid(not specified).   */
/**
 *  @}
 */ /* End of name Parameter CheckParamResult */


/** 
 *  @}
 */ /* End of group RAMP_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RAMP_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group RAMP_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RAMP_Private_typedef
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group RAMP_Private_typedef */

/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RAMP_Private_variables
 *  @{
 */

/* no define */

/**
 *  @}
 */ /* End of group RAMP_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RAMP_Private_fuctions
 *  @{
 */
#ifdef DEBUG
    __STATIC_INLINE int32_t check_param_RPAR_AREA(uint32_t area);
#endif /* #ifdef DEBUG */
#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  @ref NFCKCR_NFCKS Paramter Check.
  * @param  clock  :@ref NFCKCR_NFCKS
  * @return Result.
  * @retval REG_RPAR_PARAM_OK :Valid
  * @retval REG_RPAR_PARAM_NG :Invalid
  * @note   -
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_RPAR_AREA(uint32_t area)
{
    int32_t result = REG_RPAR_PARAM_NG;

    switch (area) 
    {
    case 3:
    case 2:
    case 1:
    case 0:
        result = REG_RPAR_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

#endif /* #ifdef DEBUG */
/**
 *  @}
 */ /* End of group RAMP_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RAMP_Exported_functions
 *  @{
 */
/*--------------------------------------------------*/
/**
  * @brief  Set PARITY Error Interrupt Enable.
  * @param  p_reg   :Register Base Address.
  * @return -
  * @retval -
  * @note   The used register.
  * | Register   | Bit Symbol    |
  * | :---       | :---          |
  * | RPARCTL    | RPARF         |
  * |            | RPAREN        |
  */
/*--------------------------------------------------*/
void REG_RPAREN_enable(TSB_RPAR_TypeDef *p_reg)
{
    uint32_t rparf = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the REG_RPAR_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_reg));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Register Setting            */
    /*------------------------------*/
    rparf  = (uint32_t)(p_reg->CTL & ~REG_RPAR_RPARCTL_RPARF_MASK);
    rparf |= REG_RPAR_RPARCTL_RPARF_ENABLE;
    /*------------------------------*/
    /*  Register Set                */
    /*------------------------------*/
    /*--- [RPARF] ---*/
    p_reg->CTL = rparf;
}


/*--------------------------------------------------*/
/**
  * @brief  Set PARITY Error Interrupt Disable.
  * @param  p_reg   :Register Base Address.
  * @return -
  * @retval -
  * @note   The used register.
  * | Register   | Bit Symbol    |
  * | :---       | :---          |
  * | RPARCTL    | RPARF         |
  * |            | RPAREN        |
  */
/*--------------------------------------------------*/
void REG_RPAREN_disable(TSB_RPAR_TypeDef *p_reg)
{
    uint32_t rparf = 0;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the REG_RPAR_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_reg));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Register Setting            */
    /*------------------------------*/
    rparf  = (uint32_t)(p_reg->CTL & ~REG_RPAR_RPARCTL_RPARF_MASK);
    rparf |= REG_RPAR_RPARCTL_RPARF_DISABLE;
    /*------------------------------*/
    /*  Register Set                */
    /*------------------------------*/
    /*--- [RPARF] ---*/
    p_reg->CTL = rparf;
}

/*--------------------------------------------------*/
/**
  * @brief  Get RAM Parity Error Stattus Flag.
  * @param  p_reg   :Register Base Address.
  * @return RPARST.
  * @retval value   : Status Register Value.
  * @note   The used register.
  * | Register   | Bit Symbol  |
  * | :---       | :---        |
  * | RPARST     | RPARFG3     |
  * |            | RPARFG2     |
  * |            | RPARFG1     |
  * |            | RPARFG0     |
  */
/*--------------------------------------------------*/
uint32_t REG_RPARFGx_get(TSB_RPAR_TypeDef *p_reg)
{
    uint32_t value = 0U;
    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the REG_RPAR_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_reg));
#endif /* #ifdef DEBUG */
    /*--- [NFCKCR]<NFCKS> ---*/
    /*------------------------------*/
    /*  Register Read               */
    /*------------------------------*/
    value = p_reg->ST;
    return (value);
}

/*--------------------------------------------------*/
/**
  * @brief  Clear RAM Parity Error Stattus Flag.
  * @param  p_reg   :Register Base Address.
  * @param  area    :RAM Area 0 - 3
  * @return RPAREADx.
  * @retval value   :Error Address.
  * @note   The used register.
  * | Register   | Bit Symbol  |
  * | :---       | :---        |
  * | RPARCLR    | RPARCLR3    |
  * |            | RPARCLR2    |
  * |            | RPARCLR1    |
  * |            | RPARCLR0    |
  */
/*--------------------------------------------------*/
void REG_RPARCLR_clear(TSB_RPAR_TypeDef *p_reg, uint32_t area){
    uint32_t rparclr = 0U;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the REG_RPAR_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_reg));
    assert_param(check_param_RPAR_AREA(area));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Register Set               */
    /*------------------------------*/
    /*--- [RPARCLR] ---*/
    switch(area){
        case 0:
            rparclr = (uint32_t)(p_reg->CLR & ~REG_RPAR_RPARST_RPARCLR0_CLR);
            rparclr |= REG_RPAR_RPARST_RPARCLR0_CLR;
        break;
        case 1:
            rparclr = (uint32_t)(p_reg->CLR & ~REG_RPAR_RPARST_RPARCLR1_CLR);
            rparclr |= REG_RPAR_RPARST_RPARCLR1_CLR;
        break;
        case 2:
            rparclr = (uint32_t)(p_reg->CLR & ~REG_RPAR_RPARST_RPARCLR2_CLR);
            rparclr |= REG_RPAR_RPARST_RPARCLR2_CLR;
        break;
        case 3:
            rparclr = (uint32_t)(p_reg->CLR & ~REG_RPAR_RPARST_RPARCLR3_CLR);
            rparclr |= REG_RPAR_RPARST_RPARCLR3_CLR;
        break;
        default:
        break;
        }
    p_reg->CLR = rparclr;
}
/*--------------------------------------------------*/
/**
  * @brief  Get RAM Parity Error Address.
  * @param  p_reg   :Register Base Address.
  * @param  area    :RAM Area 0 - 3
  * @return RPAREADx.
  * @retval value   :Error Address.
  * @note   The used register.
  * | Register   | Bit Symbol       |
  * | :---       | :---             |
  * | RPAREADx   | RPAREADDn[31:20] |
  * |            | RPAREADDn[19:0]  |
  */
/*--------------------------------------------------*/
uint32_t REG_RPAREADx_get(TSB_RPAR_TypeDef *p_reg, uint32_t area){
    uint32_t value = 0U;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the REG_RPAR_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_reg));
    assert_param(check_param_RPAR_AREA(area));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Register Read               */
    /*------------------------------*/
    /*--- [RPAREADx] ---*/
    switch(area){
        case 0:
            value = p_reg->EAD0;
        break;
        case 1:
            value = p_reg->EAD1;
        break;
        case 2:
            value = p_reg->EAD2;
        break;
        case 3:
            value = p_reg->EAD3;
        break;
        default:
        break;
    }
    return (value);
}

/**
 *  @}
 */ /* End of group RAMP_Exported_functions */

/**
 *  @}
 */ /* End of group RAMP */

/**
 *  @} 
 */ /* End of group Periph_Driver */

#endif /* defined(__RAMP_A_H)  */
#ifdef __cplusplus
}
#endif /* __cplusplus */


