/**
 *******************************************************************************
 * @file    rmc_a.c
 * @brief   This file provides API functions for RMC Class.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */

#ifdef __cplusplus
 extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "rmc.h"

#if defined(__RMC_A_H)

/**
 *  @addtogroup Periph_Driver Peripheral Driver
 *  @{
 */

/** 
 *  @addtogroup RMC
 *  @brief      RMC Driver.
 *  @{
 *
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Private_macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group RMC_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group RMC_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Private_define
 *  @{
 */

/** 
 *  @name  RMC_NULL Pointer
 *  @brief NULL Pointer.
 *  @{
 */
#define RMC_NULL                  ((void *)0)
/**
 *  @}
 */ /* End of name RMC_NULL Pointer */

/** 
 *  @name  Parameter Result
 *  @brief Whether the parameter is specified or not.
 *  @{
 */
#define RMC_PARAM_OK              ((int32_t)1)              /*!< Parameter is valid(specified).         */
#define RMC_PARAM_NG              ((int32_t)0)              /*!< Parameter is invalid(not specified).   */
/**
 *  @}
 */ /* End of name Parameter Result */

/** 
 *  @name  RMCxEN Macro Definition.
 *  @brief RMCxEN Register Macro Definition.
 *  @details  Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 31-2  | -          |
 *  | 1     | be set "1" |
 *  | 0     | RMCEN      |
 *  @{
 */
/* RMCEN */
#define RMCxEN_RMCEN_DISABLE      ((uint32_t)0x00000000)    /*!< RMCEN :Disable. */
#define RMCxEN_RMCEN_ENABLE       ((uint32_t)0x00000001)    /*!< RMCEN :Enable.  */
/**
 *  @}
 */ /* End of name RMCxEN Macro Definition */

/** 
 *  @name  RMCxREN Macro Definition.
 *  @brief RMCxREN Register Macro Definition.
 *  @details  Detail.
 *  | Bit   | Bit Symbol |
 *  | :---  | :---       |
 *  | 31-1  | -          |
 *  | 0     | RMCREN     |
 *  @{
 */
/* RMCREN */
#define RMCxREN_RMCREN_DISABLE    ((uint32_t)0x00000000)    /*!< RMCREN :Disable. */
#define RMCxREN_RMCREN_ENABLE     ((uint32_t)0x00000001)    /*!< RMCREN :Enable.  */
/**
 *  @}
 */ /* End of name RMCxREN Macro Definition */

/** 
 *  @}
 */ /* End of group RMC_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group RMC_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Private_typedef
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group RMC_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Private_variables
 *  @{
 */

/* no define */

/**
 *  @}
 */ /* End of group RMC_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Private_fuctions
 *  @{
 */
#ifdef DEBUG
    __STATIC_INLINE int32_t check_param_leader_cycle_upper_limit(uint32_t param);
    __STATIC_INLINE int32_t check_param_leader_cycle_lower_limit(uint32_t param);
    __STATIC_INLINE int32_t check_param_leader_low_upper_limit(uint32_t param);
    __STATIC_INLINE int32_t check_param_leader_low_lower_limit(uint32_t param);
    __STATIC_INLINE int32_t check_param_leader_detection_interrupt(uint32_t param);
    __STATIC_INLINE int32_t check_param_falling_edge_interrupt(uint32_t param);
    __STATIC_INLINE int32_t check_param_receive_mode(uint32_t param);
    __STATIC_INLINE int32_t check_param_reception_mode(uint32_t param);
    __STATIC_INLINE int32_t check_param_rx_end_int_timing_low(uint32_t param);
    __STATIC_INLINE int32_t check_param_rx_end_int_timing_data_bit_cycle(uint32_t param);
    __STATIC_INLINE int32_t check_param_threshold_value_high(uint32_t param);
    __STATIC_INLINE int32_t check_param_threshold_value_low(uint32_t param);
    __STATIC_INLINE int32_t check_param_polarity_choice(uint32_t param);
    __STATIC_INLINE int32_t check_param_noise_reduction(uint32_t param);
#endif /* #ifdef DEBUG */
#ifdef DEBUG
/*--------------------------------------------------*/
/** 
  * @brief  Check the Leader Cycle Upper Limit's parameter.
  * @param  param    :Leader Cycle Upper Limit's parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_LeaderCycleUppderLimit"RMC_LD_CYCLE_UP_LMT_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_leader_cycle_upper_limit(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    /*--- Now, RMC_LD_CYCLE_UP_LMT_MIN is 0. ---*/
#if 0
    if ((RMC_LD_CYCLE_UP_LMT_MIN <= param) && (param <= RMC_LD_CYCLE_UP_LMT_MAX))
#else
    if (param <= RMC_LD_CYCLE_UP_LMT_MAX)
#endif
    {
        result = RMC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Leader Cycle Lower Limit's parameter.
  * @param  param    :Leader Cycle Lower Limit's parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_LeaderCycleLowerLimit"RMC_LD_CYCLE_LOW_LMT_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_leader_cycle_lower_limit(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    /*--- Now, RMC_LD_CYCLE_LOW_LMT_MIN is 0. ---*/
#if 0
    if ((RMC_LD_CYCLE_LOW_LMT_MIN <= param) && (param <= RMC_LD_CYCLE_LOW_LMT_MAX))
#else
    if (param <= RMC_LD_CYCLE_LOW_LMT_MAX)
#endif
    {
        result = RMC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Leader Low Upper Limit's parameter.
  * @param  param    :Leader Low Upper Limit's parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_LeaderLowUppderLimit"RMC_LD_LOW_UP_LMT_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_leader_low_upper_limit(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    /*--- Now, RMC_LD_LOW_UP_LMT_MIN is 0. ---*/
#if 0
    if ((RMC_LD_LOW_UP_LMT_MIN <= param) && (param <= RMC_LD_LOW_UP_LMT_MAX))
#else
    if (param <= RMC_LD_LOW_UP_LMT_MAX)
#endif
    {
        result = RMC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Leader Low Lower Limit's parameter.
  * @param  param    :Leader Low Lower Limit's parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_LeaderLowLowerLimit"RMC_LD_LOW_LOW_LMT_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_leader_low_lower_limit(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    /*--- Now, RMC_LD_LOW_LOW_LMT_MIN is 0. ---*/
#if 0
    if ((RMC_LD_LOW_LOW_LMT_MIN <= param) && (param <= RMC_LD_LOW_LOW_LMT_MAX))
#else
    if (param <= RMC_LD_LOW_LOW_LMT_MAX)
#endif
    {
        result = RMC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Leader Detectin Interrupt's parameter.
  * @param  param    :Leader Detectin Interrupt's parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_LeaderInterrupt"RMC_LI_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_leader_detection_interrupt(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    switch (param) 
    {
    case RMC_LI_DISABLE:
    case RMC_LI_ENABLE:
        result = RMC_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Falling Edge Interrupt's parameter.
  * @param  param    :Falling Edge Interrupt's parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_FallingEdgeInterrupt"RMC_EDI_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_falling_edge_interrupt(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    switch (param) 
    {
    case RMC_EDI_DISABLE:
    case RMC_EDI_ENABLE:
        result = RMC_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Receive Mode with(without) Leaders's parameter.
  * @param  param    :Receive Mode with(without) Leaders's parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_LeaderMode"RMC_LD_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_receive_mode(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    switch (param) 
    {
    case RMC_LD_DISABLE:
    case RMC_LD_ENABLE:
        result = RMC_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Phase Mode's parameter.
  * @param  param    :Phase Mode's parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_PhaseMode"RMC_PHM_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_reception_mode(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    switch (param) 
    {
    case RMC_PHM_DISABLE:
    case RMC_PHM_ENABLE:
        result = RMC_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Receiving End Interrupt Timing Setting(Low)'s parameter.
  * @param  param    :Receiving End Interrupt Timing Setting(Low)'s parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_RxEndIntTimingLow"RMC_LL_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_rx_end_int_timing_low(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    /*--- Now, RMC_LL_MIN is 0. ---*/
#if 0
    if ((RMC_LL_MIN <= param) && (param <= RMC_LL_MAX))
#else
    if (param <= RMC_LL_MAX)
#endif
    {
        result = RMC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Receiving End Interrupt Timing Setting(Data Bit Cycle)'s parameter.
  * @param  param    :Receiving End Interrupt Timing Setting(Data Bit Cycle)'s parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_RxEndIntTimingDataBitCycle"RMC_DMAX_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_rx_end_int_timing_data_bit_cycle(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    /*--- Now, RMC_DMAX_MIN is 0. ---*/
#if 0
    if ((RMC_DMAX_MIN <= param) && (param <= RMC_DMAX_MAX))
#else
    if (param <= RMC_DMAX_MAX)
#endif
    {
        result = RMC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Threshold value high's parameter.
  * @param  param    :Threshold value high's parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_ThresholdHigh"RMC_THRESH_HIGH_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_threshold_value_high(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    /*--- Now, RMC_THRESH_HIGH_MIN is 0. ---*/
#if 0
    if ((RMC_THRESH_HIGH_MIN <= param) && (param <= RMC_THRESH_HIGH_MAX))
#else
    if (param <= RMC_THRESH_HIGH_MAX)
#endif
    {
        result = RMC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Threshold value low's parameter.
  * @param  param    :Threshold value low's parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_ThresholdLow"RMC_THRESH_LOW_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_threshold_value_low(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    /*--- Now, RMC_THRESH_LOW_MIN is 0. ---*/
#if 0
    if ((RMC_THRESH_LOW_MIN <= param) && (param <= RMC_THRESH_LOW_MAX))
#else
    if (param <= RMC_THRESH_LOW_MAX)
#endif
    {
        result = RMC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Polarity choice's parameter.
  * @param  param    :Polarity choice's parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_Polarity"RMC_POLARITY_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_polarity_choice(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    switch (param) 
    {
    case RMC_POLARITY_POSITIVE:
    case RMC_POLARITY_NEGATIVE:
        result = RMC_PARAM_OK;
        break;
    default:
        /* no process */
        break;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Noise reduction time's parameter.
  * @param  param    :Noise reduction time's parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_NoiseReduction"RMC_NOISE_REDUCTION_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_noise_reduction(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    /*--- Now, RMC_NOISE_REDUCTION_MIN is 0. ---*/
#if 0
    if ((RMC_NOISE_REDUCTION_MIN <= param) && (param <= RMC_NOISE_REDUCTION_MAX))
#else
    if (param <= RMC_NOISE_REDUCTION_MAX)
#endif
    {
        result = RMC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the Receive data bits's parameter.
  * @param  param    :Receive data bits's parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_RxDataBits"RMC_RX_DATA_BITS_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_rx_data_bits(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    /*--- Now, RMC_RX_DATA_BITS_MIN is 0. ---*/
#if 0
    if ((RMC_RX_DATA_BITS_MIN <= param) && (param <= RMC_RX_DATA_BITS_MAX))
#else
    if (param <= RMC_RX_DATA_BITS_MAX)
#endif
    {
        result = RMC_PARAM_OK;
    }

    return (result);
}

/*--------------------------------------------------*/
/** 
  * @brief  Check the sampling clock's parameter.
  * @param  param    :sampling clock's parameter
  * @retval RMC_PARAM_OK :Valid
  * @retval RMC_PARAM_NG :Invalid
  * @note   Macro definition is @ref RMC_SamplingClock"RMC_CLK_xxxx".
  */
/*--------------------------------------------------*/
__STATIC_INLINE int32_t check_param_sampling_clock(uint32_t param)
{
    int32_t result = RMC_PARAM_NG;

    /*--- Now, RMC_CLK_LOW_SPEED is 0. ---*/
#if 0
    if ((RMC_CLK_LOW_SPEED <= param) && (param <= RMC_CLK_TIMER))
#else
    if (param <= RMC_CLK_TIMER)
#endif
    {
        result = RMC_PARAM_OK;
    }

    return (result);
}
#endif /* #ifdef DEBUG */
/**
 *  @}
 */ /* End of group RMC_Private_fuctions */

/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Exported_functions
 *  @{
 */
/*--------------------------------------------------*/
/**
  * @brief      Construct object.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       -
  */
/*--------------------------------------------------*/
TXZ_Result rmc_init(rmc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RMC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    /* Check the parameter of RMCxRCR1. */
    assert_param(check_param_leader_cycle_upper_limit(p_obj->init.cnt1.lcMax));
    assert_param(check_param_leader_cycle_lower_limit(p_obj->init.cnt1.lcMin));
    assert_param(check_param_leader_low_upper_limit(p_obj->init.cnt1.llMax));
    assert_param(check_param_leader_low_lower_limit(p_obj->init.cnt1.llMin));
    /* Check the parameter of RMCxRCR2. */
    assert_param(check_param_leader_detection_interrupt(p_obj->init.cnt2.lien));
    assert_param(check_param_falling_edge_interrupt(p_obj->init.cnt2.edien));
    assert_param(check_param_receive_mode(p_obj->init.cnt2.cld));
    assert_param(check_param_reception_mode(p_obj->init.cnt2.phim));
    assert_param(check_param_rx_end_int_timing_low(p_obj->init.cnt2.ll));
    assert_param(check_param_rx_end_int_timing_data_bit_cycle(p_obj->init.cnt2.dmax));
    /* Check the parameter of RMCxRCR3. */
    assert_param(check_param_threshold_value_high(p_obj->init.cnt3.dath));
    assert_param(check_param_threshold_value_low(p_obj->init.cnt3.datl));
    /* Check the parameter of RMCxRCR4. */
    assert_param(check_param_polarity_choice(p_obj->init.cnt4.po));
    assert_param(check_param_noise_reduction(p_obj->init.cnt4.nc));
    /* Check the parameter of RMCxEND1. */
    assert_param(check_param_rx_data_bits(p_obj->init.end1.end1));
    /* Check the parameter of RMCxEND2. */
    assert_param(check_param_rx_data_bits(p_obj->init.end2.end2));
    /* Check the parameter of RMCxEND3. */
    assert_param(check_param_rx_data_bits(p_obj->init.end3.end3));
    /* Check the parameter of RMCxFSSEL. */
    assert_param(check_param_sampling_clock(p_obj->init.fssel.clk));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Enable RMC                  */
    /*------------------------------*/
    p_obj->p_instance->EN = ((uint32_t)0x00000002 | RMCxEN_RMCEN_ENABLE);
    /*------------------------------*/
    /*  Register Setting            */
    /*------------------------------*/
    /*---  RMCxFSSEL ---*/
    /* M3H group can't use this function. */
    /*---  RMCxRCR1 ---*/
    /* Be careful, needs to bit shit. */
    {
        uint32_t work = (p_obj->init.cnt1.llMin);
        work |= (uint32_t)(p_obj->init.cnt1.llMax << 8);
        work |= (uint32_t)(p_obj->init.cnt1.lcMin << 16);
        work |= (uint32_t)(p_obj->init.cnt1.lcMax << 24);
        p_obj->p_instance->RCR1 = work;
    }
    /*--- RMCxRCR2 ---*/
    /* Be careful, needs to bit shit. */
    {
        uint32_t work = (uint32_t)(p_obj->init.cnt2.ll << 8);
        p_obj->p_instance->RCR2 = (p_obj->init.cnt2.lien | p_obj->init.cnt2.edien | 
                                   p_obj->init.cnt2.cld  | p_obj->init.cnt2.phim | 
                                   work                  | p_obj->init.cnt2.dmax);
    }
    /*--- RMCxRCR3 ---*/
    /* Be careful, needs to bit shit. */
    {
        uint32_t work = (uint32_t)(p_obj->init.cnt3.dath << 8);
        p_obj->p_instance->RCR3 = (p_obj->init.cnt3.datl | work);
    }
    /*--- RMCxRCR4 ---*/
    p_obj->p_instance->RCR4 = (p_obj->init.cnt4.po | p_obj->init.cnt4.nc);
    /*--- RMCxEND1 ---*/
    p_obj->p_instance->END1 = (p_obj->init.end1.end1);
    /*--- RMCxEND2 ---*/
    p_obj->p_instance->END2 = (p_obj->init.end2.end2);
    /*--- RMCxEND3 ---*/
    p_obj->p_instance->END3 = (p_obj->init.end3.end3);

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Destruct object.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       -
  */
/*--------------------------------------------------*/
TXZ_Result rmc_deinit(rmc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RMC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Disable Receive             */
    /*------------------------------*/
    p_obj->p_instance->REN = RMCxREN_RMCREN_DISABLE;
    /*------------------------------*/
    /*  Disable RMC                 */
    /*------------------------------*/
    p_obj->p_instance->EN  = ((uint32_t)0x00000002 | RMCxEN_RMCEN_DISABLE);

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Enable RMC.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       This function control the use of RMCxEN:RMCEN.
  */
/*--------------------------------------------------*/
TXZ_Result rmc_enable(rmc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RMC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Enable RMC                  */
    /*------------------------------*/
    p_obj->p_instance->EN = ((uint32_t)0x00000002 | RMCxEN_RMCEN_ENABLE);

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Disable RMC.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       This function control the use of RMCxEN:RMCEN.
  */
/*--------------------------------------------------*/
TXZ_Result rmc_disable(rmc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RMC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Disable RMC                 */
    /*------------------------------*/
    p_obj->p_instance->EN  = ((uint32_t)0x00000002 | RMCxEN_RMCEN_DISABLE);

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Enable Receive.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       This function control the use of RMCxREN:RMCREN.
  */
/*--------------------------------------------------*/
TXZ_Result rmc_rx_enable(rmc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RMC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Enable RMC                  */
    /*------------------------------*/
    p_obj->p_instance->REN = RMCxREN_RMCREN_ENABLE;

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Disable Receive.
  * @details    -
  * @param[in]  p_obj   :Object address.
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       This function control the use of RMCxREN:RMCREN.
  */
/*--------------------------------------------------*/
TXZ_Result rmc_rx_disable(rmc_t *p_obj)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the RMC_NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Disable Receive             */
    /*------------------------------*/
    p_obj->p_instance->REN = RMCxREN_RMCREN_DISABLE;

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief  Get status.
  * @details  Status bits.
  * | Bit   | Bit Symbol | Function                                                               |
  * | :---  | :---       | :---                                                                   |
  * | 31-16 | -          | -                                                                      |
  * | 15    | RMCRLIF    | Leader detection interrupt flag. Use @ref RMC_LeaderInterruptFlag.     |
  * | 14    | RMCLOIF    | Low detection interrupt flag. Use @ref RMC_LowInterruptFlag.           |
  * | 13    | RMCDMAXIF  | Data bit cycle max interrupt flag. Use @ref RMC_CycleMaxInterruptFlag. |
  * | 12    | RMCEDIF    | Falling edge interrupt flag. Use @ref RMC_FallingEdgeInterruptFlag.    |
  * | 11-8  | -          | -                                                                      |
  * | 7     | RMCRLDR    | Leader detection. Use @ref RMC_LeaderDetection.                        |
  * | 6-0   | RMCRNUM    | Num of received data bits.  Use @ref RMC_RxBitNum.                     |
  * 
  * @param[in]  p_obj    :Object address.
  * @param[out] p_status :Destination address, that is "status".
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       -
  */
/*--------------------------------------------------*/
TXZ_Result rmc_get_status(rmc_t *p_obj, uint32_t *p_status)
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(p_status));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Status Read                 */
    /*------------------------------*/
    /*--- RMCxRSTAT ---*/
    /* Read current RMCxRSTAT. */
    *p_status = p_obj->p_instance->RSTAT;

    return (res);
}

/*--------------------------------------------------*/
/**
  * @brief      Get receive data.
  * @details    -
  * @param[in]  p_obj    :Object address.
  * @param[out] data   :Destination address, that is "Receive data".
  * @retval     TXZ_SUCCESS :Success
  * @retval     TXZ_ERROR   :Failure
  * @note       Num of buffer is RMC_DataBuffNum.
  */
/*--------------------------------------------------*/
TXZ_Result rmc_get_data(rmc_t *p_obj, uint32_t data[])
{
    TXZ_Result res = TXZ_SUCCESS;

    /*------------------------------*/
    /*  Parameter Check             */
    /*------------------------------*/
#ifdef DEBUG
    /* Check the NULL of address. */
    assert_param(IS_POINTER_NOT_NULL(p_obj));
    assert_param(IS_POINTER_NOT_NULL(p_obj->p_instance));
    assert_param(IS_POINTER_NOT_NULL(data));
#endif /* #ifdef DEBUG */
    /*------------------------------*/
    /*  Data Read                   */
    /*------------------------------*/
    /*--- RMCxRBUF1 ---*/
    data[0] = p_obj->p_instance->RBUF1;
    /*--- RMCxRBUF2 ---*/
    data[1] = p_obj->p_instance->RBUF2;
    /*--- RMCxRBUF3 ---*/
    data[2] = p_obj->p_instance->RBUF3;

    return (res);
}

/**
 *  @}
 */ /* End of group RMC_Exported_functions */

/**
 *  @}
 */ /* End of group RMC */

/**
 *  @} 
 */ /* End of group Periph_Driver */

#endif /* defined(__RMC_A_H)  */
#ifdef __cplusplus
}
#endif /* __cplusplus */


