/** 
 *******************************************************************************
 * @file    rmc_a.h
 * @brief   This file provides all the functions prototypes for RMC Class.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
/*------------------------------------------------------------------------------*/
/*  Define to prevent recursive inclusion                                       */
/*------------------------------------------------------------------------------*/
#ifndef __RMC_A_H
#define __RMC_A_H

#ifdef __cplusplus
extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "bsp_com.h"

/**
 *  @addtogroup Periph_Driver Peripheral Driver
 *  @{
 */

/** 
 *  @defgroup RMC RMC
 *  @brief    RMC Driver.
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Private_macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group RMC_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group RMC_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Private_define
 *  @{
 */

/** 
 *  @defgroup RMC_LeaderCycleUppderLimit  RMC leader cycle uppder limit.
 *  @brief    Upper limit in a cycle period of leader detection.
 *  @brief    RMC_LD_CYCLE_UP_LMT_MIN <= Value <= RMC_LD_CYCLE_UP_LMT_MAX: (Value x 4/fs[s]) Uppder limit.
 *  @{
 */
#define RMC_LD_CYCLE_UP_LMT_MIN     ((uint32_t)0x0000000)     /*!< Minimum  Value.  */
#define RMC_LD_CYCLE_UP_LMT_MAX     ((uint32_t)0x00000FF)     /*!< Maximum  Value.  */
/**
 *  @}
 */ /* End of group RMC_LeaderCycleUppderLimit */

/** 
 *  @defgroup RMC_LeaderCycleLowerLimit  RMC leader cycle lower limit.
 *  @brief    Lower limit in a cycle period of leader detection.
 *  @brief    RMC_LD_CYCLE_LOW_LMT_MIN <= Value <= RMC_LD_CYCLE_UP_LMT_MAX: (Value x 4/fs[s]) Lower limit.
 *  @{
 */
#define RMC_LD_CYCLE_LOW_LMT_MIN    ((uint32_t)0x0000000)     /*!< Minimum  Value.  */
#define RMC_LD_CYCLE_LOW_LMT_MAX    ((uint32_t)0x00000FF)     /*!< Maximum  Value.  */
/**
 *  @}
 */ /* End of group RMC_LeaderCycleLowerLimit */

/** 
 *  @defgroup RMC_LeaderLowUppderLimit  RMC leader low uppder limit.
 *  @brief    Upper limit in a low period of leader detection.
 *  @brief    RMC_LD_LOW_UP_LMT_MIN <= Value <= RMC_LD_LOW_UP_LMT_MAX: (Value x 4/fs[s]) Uppder limit.
 *  @{
 */
#define RMC_LD_LOW_UP_LMT_MIN       ((uint32_t)0x0000000)     /*!< Minimum  Value.  */
#define RMC_LD_LOW_UP_LMT_MAX       ((uint32_t)0x00000FF)     /*!< Maximum  Value.  */
/**
 *  @}
 */ /* End of group RMC_LeaderLowUppderLimit */

/** 
 *  @defgroup RMC_LeaderLowLowerLimit  RMC leader low lower limit.
 *  @brief    Lower limit in a low period of leader detection.
 *  @brief    RMC_LD_LOW_LOW_LMT_MIN <= Value <= RMC_LD_LOW_LOW_LMT_MAX: (Value x 4/fs[s]) Lower limit.
 *  @{
 */
#define RMC_LD_LOW_LOW_LMT_MIN      ((uint32_t)0x0000000)     /*!< Minimum  Value.  */
#define RMC_LD_LOW_LOW_LMT_MAX      ((uint32_t)0x00000FF)     /*!< Maximum  Value.  */
/**
 *  @}
 */ /* End of group RMC_LeaderLowLowerLimit */

/** 
 *  @defgroup RMC_LeaderInterrupt  RMC leader interrupt.
 *  @brief    Enable/Disable a RMC leader interrupt Macro Definisiton.
 *  @{
 */
#define RMC_LI_DISABLE              ((uint32_t)0x00000000)    /*!< Disable */
#define RMC_LI_ENABLE               ((uint32_t)0x80000000)    /*!< Enable  */
/**
 *  @}
 */ /* End of group RMC_LeaderInterrupt */

/** 
 *  @defgroup RMC_FallingEdgeInterrupt  RMC falling edge interrupt.
 *  @brief    Enable/Disable a RMC falling edge interrupt Macro Definisiton.
 *  @{
 */
#define RMC_EDI_DISABLE             ((uint32_t)0x00000000)    /*!< Disable */
#define RMC_EDI_ENABLE              ((uint32_t)0x40000000)    /*!< Enable  */
/**
 *  @}
 */ /* End of group RMC_FallingEdgeInterrupt */

/** 
 *  @defgroup RMC_LeaderMode  RMC receive mode with(without) leaders.
 *  @brief    Enable/Disable a receive mode, that receives both remote control signals without leaders and with leaders.
 *  @{
 */
#define RMC_LD_DISABLE              ((uint32_t)0x00000000)    /*!< Disable */
#define RMC_LD_ENABLE               ((uint32_t)0x02000000)    /*!< Enable  */
/**
 *  @}
 */ /* End of group RMC_LeaderMode */

/** 
 *  @defgroup RMC_PhaseMode  RMC phase mode.
 *  @brief    Setting of a remote control reception mode of the phase system.
 *  @{
 */
#define RMC_PHM_DISABLE             ((uint32_t)0x00000000)    /*!< A remote control signal of the phase system isn't received.               */
#define RMC_PHM_ENABLE              ((uint32_t)0x01000000)    /*!< A remote control signal of the phase system of cycle fixing is received.  */
/**
 *  @}
 */ /* End of group RMC_PhaseMode */

/** 
 *  @defgroup RMC_RxEndIntTimingLow  RMC Receiving End Interrupt Timing Setting.
 *  @brief    Setting at the timing of a "Receiving End Interrupt" by detection Low.
 *  @brief    RMC_LL_MIN <= Value < RMC_LL_MAX: (Value x 1/fs[s]) Timing of a Receiving End Interrupt.
 *  @brief    Value = RMC_LL_MAX              : (Value x 1/fs[s]) Disable Receiving End Interrupt.
 *  @{
 */
#define RMC_LL_MIN                  ((uint32_t)0x0000000)    /*!< Minimum  Value.                                   */
#define RMC_LL_MAX                  ((uint32_t)0x00000FF)    /*!< Maximum  Value(Disable Receiving End Interrupt).  */
/**
 *  @}
 */ /* End of group RMC_RxEndIntTimingLow */

/** 
 *  @defgroup RMC_RxEndIntTimingDataBitCycle  RMC Receiving End Interrupt Timing Setting.
 *  @brief    Setting at the timing of a "Receiving End Interrupt" by the cycle of the data bit.
 *  @brief    RMC_DMAX_MIN <= Value < RMC_DMAX_MAX: (Value x 1/fs[s]) Timing of a Receiving End Interrupt.
 *  @brief    Value = RMC_DMAX_MAX                : (Value x 1/fs[s]) Disable Receiving End Interrupt.
 *  @{
 */
#define RMC_DMAX_MIN                ((uint32_t)0x0000000)    /*!< Minimum  Value.                                   */
#define RMC_DMAX_MAX                ((uint32_t)0x00000FF)    /*!< Maximum  Value(Disable Receiving End Interrupt).  */
/**
 *  @}
 */ /* End of group RMC_RxEndIntTimingDataBitCycle */

/** 
 *  @defgroup RMC_ThresholdHigh  RMC Threshold value high.
 *  @brief    Threshold value high setting of 3 price judgment of a Data bit.
 *  @brief    RMC_THRESH_HIGH_MIN <= Value <= RMC_THRESH_HIGH_MAX.
 *  @brief    Threshold = (Value x 1/fs[s]).
 *  @{
 */
#define RMC_THRESH_HIGH_MIN         ((uint32_t)0x0000000)    /*!< Minimum  Value.  */
#define RMC_THRESH_HIGH_MAX         ((uint32_t)0x000007F)    /*!< Maximum  Value.  */
/**
 *  @}
 */ /* End of group RMC_ThresholdHigh */

/** 
 *  @defgroup RMC_ThresholdLow  RMC Threshold value low.
 *  @brief    Threshold value low setting of 3 price judgment of a Data bit.
 *  @brief    RMC_THRESH_LOW_MIN <= Value <= RMC_THRESH_LOW_MAX.
 *  @brief    Threshold = (Value x 1/fs[s]).
 *  @{
 */
#define RMC_THRESH_LOW_MIN          ((uint32_t)0x0000000)    /*!< Minimum  Value.  */
#define RMC_THRESH_LOW_MAX          ((uint32_t)0x000007F)    /*!< Maximum  Value.  */
/**
 *  @}
 */ /* End of group RMC_ThresholdLow */

/** 
 *  @defgroup RMC_Polarity  RMC Polarity choice.
 *  @brief    Polarity choice of a remote control input signal.
 *  @{
 */
#define RMC_POLARITY_POSITIVE       ((uint32_t)0x0000000)    /*!< Positive side.  */
#define RMC_POLARITY_NEGATIVE       ((uint32_t)0x0000080)    /*!< Negative side.  */
/**
 *  @}
 */ /* End of group RMC_Polarity */

/** 
 *  @defgroup RMC_NoiseReduction  RMC Noise reduction time.
 *  @brief    Setting of noise reduction time.
 *  @brief    RMC_NOISE_REDUCTION_MIN < Value <= RMC_NOISE_REDUCTION_MAX: (Value x 1/fs[s]) Noise reduction time.
 *  @brief    Value = RMC_NOISE_REDUCTION_MIN                           : No reduction.
 *  @{
 */
#define RMC_NOISE_REDUCTION_MIN     ((uint32_t)0x0000000)    /*!< Minimum  Value.  */
#define RMC_NOISE_REDUCTION_MAX     ((uint32_t)0x000000F)    /*!< Maximum  Value.  */
/**
 *  @}
 */ /* End of group RMC_NoiseReduction */

/** 
 *  @defgroup RMC_RxDataBits  RMC Receive data bits.
 *  @brief    Num of received data bits.
 *  @brief    RMC_RX_DATA_BITS_MIN < Value <= RMC_RX_DATA_BITS_MAX: Num of received data bits.
 *  @brief    Value = RMC_RX_DATA_BITS_MIN                        : No setting.
 *  @{
 */
#define RMC_RX_DATA_BITS_MIN        ((uint32_t)0x0000000)    /*!< Minimum  Value.  */
#define RMC_RX_DATA_BITS_MAX        ((uint32_t)0x0000048)    /*!< Maximum  Value.  */
/**
 *  @}
 */ /* End of group RMC_RxDataBits */

/** 
 *  @defgroup RMC_SamplingClock  RMC sampling clock.
 *  @brief    Select RMC sampling clock.
 *  @{
 */
#define RMC_CLK_LOW_SPEED           ((uint32_t)0x00000000)    /*!< Low speed clock(32.768kHz). */
#define RMC_CLK_TIMER               ((uint32_t)0x00000001)    /*!< Timer output.               */
/**
 *  @}
 */ /* End of group RMC_SamplingClock */

/** 
 *  @defgroup RMC_LeaderInterruptFlag  RMC leader detection interrupt flag.
 *  @brief    Leader detection interrupt flag.
 *  @{
 */
#define RMC_LD_INT_NO_OCCUR         ((uint32_t)0x00000000)    /*!< It doesn't occur. */
#define RMC_LD_INT_OCCUR            ((uint32_t)0x00008000)    /*!< It occurs.        */
/**
 *  @}
 */ /* End of group RMC_LeaderInterruptFlag */

/** 
 *  @defgroup RMC_LowInterruptFlag  RMC low detection interrupt flag.
 *  @brief    Low detection interrupt flag.
 *  @{
 */
#define RMC_LOW_INT_NO_OCCUR        ((uint32_t)0x00000000)    /*!< It doesn't occur. */
#define RMC_LOW_INT_OCCUR           ((uint32_t)0x00004000)    /*!< It occurs.        */
/**
 *  @}
 */ /* End of group RMC_LowInterruptFlag */

/** 
 *  @defgroup RMC_CycleMaxInterruptFlag  RMC data bit cycle max interrupt flag.
 *  @brief    Data bit cycle max interrupt flag.
 *  @{
 */
#define RMC_CYCLE_MAX_INT_NO_OCCUR  ((uint32_t)0x00000000)    /*!< It doesn't occur. */
#define RMC_CYCLE_MAX_INT_OCCUR     ((uint32_t)0x00002000)    /*!< It occurs.        */
/**
 *  @}
 */ /* End of group RMC_CycleMaxInterruptFlag */

/** 
 *  @defgroup RMC_FallingEdgeInterruptFlag  RMC falling edge interrupt flag.
 *  @brief    Falling edge interrupt flag.
 *  @{
 */
#define RMC_FALL_EDGE_INT_NO_OCCUR  ((uint32_t)0x00000000)    /*!< It doesn't occur. */
#define RMC_FALL_EDGE_INT_OCCUR     ((uint32_t)0x00001000)    /*!< It occurs.        */
/**
 *  @}
 */ /* End of group RMC_FallingEdgeInterruptFlag */

/** 
 *  @defgroup RMC_RepeatCodeInterruptFlag  RMC repeat code interrupt flag.
 *  @brief    Repeat Code interrupt flag.
 *  @{
 */
#define RMC_REPEAT_CODE_INT_NO_OCCUR  ((uint32_t)0x00000000)    /*!< It doesn't occur. */
#define RMC_REPEAT_CODE_INT_OCCUR     ((uint32_t)0x00008000)    /*!< It occurs.        */
/**
 *  @}
 */ /* End of group RMC_RepeatCodeInterruptFlag */
 /** 
 *  @defgroup RMC_RepeatCodeDetection  RMC RepeatCode detection.
 *  @brief    RepeatCode detection.
 *  @{
 */
#define RMC_REPEAT_CODE_NO_DETECT        ((uint32_t)0x00000000)    /*!< It does'nt detect. */
#define RMC_REPEAT_CODE_DETECT           ((uint32_t)0x00000080)    /*!< It detests.        */
/**
 *  @}
 */ /* End of group RMC_RepeatCodeDetection */
/** 
 *  @defgroup RMC_LeaderDetection  RMC leader detection.
 *  @brief    Leader detection.
 *  @{
 */
#define RMC_LEADER_NO_DETECT        ((uint32_t)0x00000000)    /*!< It does'nt detect. */
#define RMC_LEADER_DETECT           ((uint32_t)0x00000080)    /*!< It detests.        */
/**
 *  @}
 */ /* End of group RMC_LeaderDetection */

/** 
 *  @defgroup RMC_RxBitNum  RMC receive bit num.
 *  @brief    Num of receive data bits.
 *  @{
 */
#define RMC_RX_BIT_NUM_MASK         ((uint32_t)0x0000007F)    /*!< Mask. */
/** 
 *  @}
 */ /* End of group RMC_RxBitNum */

/** 
 *  @defgroup RMC_DataBuffNum  RMC data buff num.
 *  @brief    Num of receive data buffer.
 *  @{
 */
#define RMC_DATA_BUFF_NUM           (3UL)  /*!< Num of receive data buffer. */
/** 
 *  @}
 */ /* End of group RMC_DataBuffNum */

/** 
 *  @}
 */ /* End of group RMC_Private_define */

/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group RMC_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Private_typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @struct rmc_control1_t
 * @brief  Receive Control Setting "1".
*/
/*----------------------------------*/
typedef struct
{
    uint32_t lcMax;  /*!< Upper limit in a cycle period of leader detection.
                                : Use @ref RMC_LeaderCycleUppderLimit        */
    uint32_t lcMin;  /*!< Lower limit in a cycle period of leader detection.
                                : Use @ref RMC_LeaderCycleLowerLimit         */
    uint32_t llMax;  /*!< Upper limit in a low period of leader detection.
                                : Use @ref RMC_LeaderLowUppderLimit          */
    uint32_t llMin;  /*!< Lower limit in a low period of leader detection.
                                : Use @ref RMC_LeaderLowLowerLimit           */
} rmc_control1_t;

/*----------------------------------*/
/** 
 * @struct rmc_control2_t
 * @brief  Receive Control Setting "2".
*/
/*----------------------------------*/
typedef struct
{
    uint32_t lien;  /*!< Enable a leader detection interrupt.
                                : Use @ref RMC_LeaderInterrupt               */
    uint32_t edien; /*!< Enable a remote control input falling edge interrupt.
                                : Use @ref RMC_FallingEdgeInterrupt          */
    uint32_t cld;   /*!< Enable a receive mode, that receives both remote control signals without leaders and with leaders.
                                : Use @ref RMC_LeaderMode                    */
    uint32_t phim;  /*!< Setting of a remote control reception mode of the phase system.
                                : Use @ref RMC_PhaseMode                     */
    uint32_t ll;    /*!< Setting at the timing of a "Receiving End Interrupt" by detection Low.
                                : Use @ref RMC_RxEndIntTimingLow             */
    uint32_t dmax;  /*!< Setting at the timing of a "Receiving End Interrupt" by the cycle of the data bit.
                                : Use @ref RMC_RxEndIntTimingDataBitCycle    */
} rmc_control2_t;

/*----------------------------------*/
/** 
 * @struct rmc_control3_t
 * @brief  Receive Control Setting "3".
*/
/*----------------------------------*/
typedef struct
{
    uint32_t dath;  /*!< Threshold value high setting of 3 price judgment of a Data bit.
                                : Use @ref RMC_ThresholdHigh                 */
    uint32_t datl;  /*!< Threshold value low setting of 3 price judgment of a Data bit.
                                : Use @ref RMC_ThresholdLow                  */
} rmc_control3_t;

/*----------------------------------*/
/** 
 * @struct rmc_control4_t
 * @brief  Receive Control Setting "4".
*/
/*----------------------------------*/
typedef struct
{
    uint32_t po;    /*!< Polarity choice of a remote control input signal.
                                : Use @ref RMC_Polarity                      */
    uint32_t nc;    /*!< Setting of noise reduction time.
                                : Use @ref RMC_NoiseReduction                */
} rmc_control4_t;

/*----------------------------------*/
/** 
 * @struct rmc_end1_t
 * @brief  Num of received end bit "1".
*/
/*----------------------------------*/
typedef struct
{
    uint32_t end1;  /*!< Num of received data bits.
                                : Use @ref RMC_RxDataBits                    */
} rmc_end1_t;

/*----------------------------------*/
/** 
 * @struct rmc_end2_t
 * @brief  Num of received end bit "2".
*/
/*----------------------------------*/
typedef struct
{
    uint32_t end2;  /*!< Num of received data bits.
                                : Use @ref RMC_RxDataBits                    */
} rmc_end2_t;

/*----------------------------------*/
/** 
 * @struct rmc_end3_t
 * @brief  Num of received end bit "3".
*/
/*----------------------------------*/
typedef struct
{
    uint32_t end3;  /*!< Num of received data bits.
                                : Use @ref RMC_RxDataBits                    */
} rmc_end3_t;

/*----------------------------------*/
/** 
 * @struct rmc_fssel_t
 * @brief  Select source clock.
*/
/*----------------------------------*/
typedef struct
{
    uint32_t clk;   /*!< Select RMC sampling clock.
                                : Use @ref RMC_SamplingClock                 */
} rmc_fssel_t;

/*----------------------------------*/
/** 
 * @struct rmc_initial_setting_t
 * @brief  Initial setting structure definenition.
*/
/*----------------------------------*/
typedef struct
{
    rmc_control1_t  cnt1;   /*!< Receive Control Setting "1".
                                    : Use @ref rmc_control1_t */
    rmc_control2_t  cnt2;   /*!< Receive Control Setting "2".
                                    : Use @ref rmc_control2_t */
    rmc_control3_t  cnt3;   /*!< Receive Control Setting "3".
                                    : Use @ref rmc_control3_t */
    rmc_control4_t  cnt4;   /*!< Control4 setting.
                                    : Use @ref rmc_control4_t */
    rmc_end1_t      end1;   /*!< Receive End Bit1 setting.
                                    : Use @ref rmc_end1_t     */
    rmc_end2_t      end2;   /*!< Receive End Bit2 setting.
                                    : Use @ref rmc_end2_t     */
    rmc_end3_t      end3;   /*!< Receive End Bit3 setting.
                                    : Use @ref rmc_end3_t     */
    rmc_fssel_t     fssel;  /*!< Select source clock.
                                    : Use @ref rmc_fssel_t    */
} rmc_initial_setting_t;

/*----------------------------------*/
/** 
 * @brief  RMC handle structure definenition.
*/
/*----------------------------------*/
typedef struct uart_handle
{
    TSB_RMC_TypeDef       *p_instance;     /*!< Registers base address.           */
    rmc_initial_setting_t init;            /*!< Initial setting.                  */
} rmc_t;
/** 
 *  @}
 */ /* End of group RMC_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Functions                                                                   */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup RMC_Exported_functions
 *  @{
 */
TXZ_Result rmc_init(rmc_t *p_obj);
TXZ_Result rmc_deinit(rmc_t *p_obj);
TXZ_Result rmc_enable(rmc_t *p_obj);
TXZ_Result rmc_disable(rmc_t *p_obj);
TXZ_Result rmc_rx_enable(rmc_t *p_obj);
TXZ_Result rmc_rx_disable(rmc_t *p_obj);
TXZ_Result rmc_get_status(rmc_t *p_obj, uint32_t *p_status);
TXZ_Result rmc_get_data(rmc_t *p_obj, uint32_t data[]);
/**
 *  @}
 */ /* End of group RMC_Exported_functions */

/**
 *  @}
 */ /* End of group RMC */

/**
 *  @} 
 */ /* End of group Periph_Driver */

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* __RMC_A_H */


