/** 
 *******************************************************************************
 * @file    app_horol.h
 * @brief   This file provides all the functions prototypes for horologe.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
/*------------------------------------------------------------------------------*/
/*  Define to prevent recursive inclusion                                       */
/*------------------------------------------------------------------------------*/
#ifndef __APP_HOROL_H
#define __APP_HOROL_H

#ifdef __cplusplus
extern "C" {
#endif

/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "bsp_com.h"
#include "bsp_rtc.h"

/**
 *  @addtogroup Utility Utility
 *  @{
 */

/** 
 *  @defgroup horol
 *  @brief    horol utility.
 *  @{
 */
/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup horol_Private_macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group horol_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup horol_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group horol_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup horol_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group horol_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup horol_Private_define
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief  Clock Value Layer.
*/
/*----------------------------------*/
typedef enum
{
    HOROL_LAYER_0 = 0,        /*!< Layer 0.  */
    HOROL_LAYER_1,            /*!< Layer 1.  */
    /*>>> be set bottom */
    HOROL_LAYER_MAX           /*!< max       */
    /*<<< be set bottom */
} HorolValueLayer;
/** 
 *  @}
 */ /* End of group horol_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup horol_Private_typedef
 *  @{
 */
/*----------------------------------*/
/** 
 * @brief  Clock Information.
*/
/*----------------------------------*/
typedef struct
{
    uint8_t     century;    /*!< Century.                          */
    uint8_t     year;       /*!< Year. 
                                    : Range is @ref RTC_RangeYear. */
    uint8_t     notation;   /*!< Hour Notaion. 
                                    : Use @ref RTC_HourNotation.   */
    uint8_t     meridiem;   /*!< Meridiem. 
                                    : Use @ref RTC_Meridiem.       */
    uint8_t     month;      /*!< Month. 
                                    : Use @ref RTC_Month.          */
    uint8_t     date;       /*!< Date. 
                                    : Range is @ref RTC_RangeDate. */
    uint8_t     day;        /*!< A day of the week. 
                                    : Use @ref RTC_Day.            */
    uint8_t     hour;       /*!< Hour. 
                                    : Range is (0 <= hour <= 23)   */
    uint8_t     min;        /*!< Minute. 
                                    : Range is @ref RTC_RangeMin.  */
    uint8_t     sec;        /*!< Second. 
                                    : Range is @ref RTC_RangeSec.  */
} horol_clock_value_t;

/*----------------------------------*/
/** 
 * @brief  Initial setting. \n
 *         User must be set.
*/
/*----------------------------------*/
typedef struct
{
    uint32_t              id;       /*!< User ID.            */
    horol_clock_value_t   setting;  /*!< Setting.            */
    void (*clock_update)(uint32_t id);     /*!< Event handler. Notify of clock update. */
} horol_initial_setting_t;

/*----------------------------------*/
/** 
 * @brief  Private Information.
*/
/*----------------------------------*/
typedef struct
{
    rtc_t                   rtc;     /*!< RTC Instance.      */
    TXZ_FunctionalState     end;     /*!< The end of a century flag. \n
                                            TXZ_ENABLE : End of a century. */
    struct
    {
        HorolValueLayer     read;    /*!< Read Layer.        */
        horol_clock_value_t value[(uint32_t)HOROL_LAYER_MAX];   /*!< Clock Value. */
    } clock;
} horol_private_info_t;

/*----------------------------------*/
/** 
 * @brief  Horologe Class.
*/
/*----------------------------------*/
typedef struct
{
    horol_initial_setting_t init;   /*!< Initial setting.     */
    horol_private_info_t    info;   /*!< Private information. */
} horol_t;

/**
 *  @}
 */ /* End of group horol_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Functions                                                                   */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup horol_Exported_functions
 *  @{
 */
horol_t *horol_initialize(horol_initial_setting_t *p_param);
void horol_finalize(horol_t *p_instance);
void horol_start(horol_t *p_instance);
void horol_stop(horol_t *p_instance);
void horol_get_current_clock(horol_t *p_instance, horol_clock_value_t *p_dst);
void horol_irq_handler(horol_t *p_instance);
uint32_t horol_make_anno_domini(uint8_t century, uint8_t year);
/** 
 *  @}
 */ /* End of group horol_Exported_functions */

/**
 *  @}
 */ /* End of group horol */

/**
 *  @} 
 */ /* End of group Utility */

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* __APP_HOROL_H */


