/**
 *******************************************************************************
 * @file    app_horol.c
 * @brief   This file provides API functions for horologe.
 * @version V1.0.0
 * 
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2020
 *******************************************************************************
 */
#ifdef __cplusplus
 extern "C" {
#endif
/*------------------------------------------------------------------------------*/
/*  Includes                                                                    */
/*------------------------------------------------------------------------------*/
#include "app_horol.h"
#include "bsp.h"

#if defined(__APP_HOROL_H)
/**
 *  @addtogroup Utility Utility
 *  @{
 */

/** 
 *  @addtogroup horol
 *  @brief      horol utility.
 *  @{
 */


/*------------------------------------------------------------------------------*/
/*  Macro Function                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup horol_Private_macro
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group horol_Private_macro */


/*------------------------------------------------------------------------------*/
/*  Configuration                                                               */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup horol_Private_define
 *  @{
 */

/*--- for Debug ---*/
#ifdef DEBUG
    #if 1
        #define HOROL_CFG_DEBUG
    #endif
#endif

/** 
 *  @}
 */ /* End of group horol_Private_define */


/*------------------------------------------------------------------------------*/
/*  Macro Definition                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup horol_Private_define
 *  @{
 */

/** 
 *  @name  NULL_Pointer
 *  @brief NULL Pointer.
 *  @{
 */
#define HOROL_NULL                ((void *)0)
/**
 *  @}
 */ /* End of name NULL_Pointer */

/** 
 *  @}
 */ /* End of group horol_Private_define */


/*------------------------------------------------------------------------------*/
/*  Enumerated Type Definition                                                  */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup horol_Private_define
 *  @{
 */

/* no define */

/** 
 *  @}
 */ /* End of group horol_Private_define */


/*------------------------------------------------------------------------------*/
/*  Structure Definition                                                        */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup horol_Private_typedef
 *  @{
 */

/* no define */

/**
 *  @}
 */ /* End of group horol_Private_typedef */


/*------------------------------------------------------------------------------*/
/*  Private Member                                                              */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup horol_Private_variables
 *  @{
 */

static horol_t instance;

/**
 *  @}
 */ /* End of group horol_Private_variables */


/*------------------------------------------------------------------------------*/
/*  Private Function                                                            */
/*------------------------------------------------------------------------------*/
/** 
 *  @defgroup horol_Private_fuctions horol Private Fuctions
 *  @{
 */
static void debug_error(void);
static uint8_t get_leap_setting(uint8_t century, uint8_t year);
static void copy_clock_value(horol_clock_value_t *p_dst, horol_clock_value_t *p_src);
/*--------------------------------------------------*/
/** 
  * @brief  Error Hook
  * @param  -
  * @retval -
  * @note   for debug
  */
/*--------------------------------------------------*/
static void debug_error(void)
{
#ifdef HOROL_CFG_DEBUG
    while(1)
    {
        __nop();
    }
#endif
}

/*--------------------------------------------------*/
/** 
  * @brief  Get leap year setting.
  * @param  century :Century.
  * @param  year    :Year.
  * @return Leap year setting. \n
  *         Value is @ref RTC_LeapYear.
  * @retval 
  * @note   -
  */
/*--------------------------------------------------*/
static uint8_t get_leap_setting(uint8_t century, uint8_t year)
{
    uint8_t res;
    uint32_t ad;
    uint32_t work;

    /* A.D. */
    ad = horol_make_anno_domini(century, year);
    work = (ad % 4);
    switch(work)
    {
    case 0:
        {
            res = RTC_LEAP_YEAR_0;
            work = (ad % 100);
            if (work == 0)
            {
                work = (ad % 400);
                if (work != 0)
                {
                    /* Not leap-year. */
                    res = RTC_LEAP_YEAR_3;
                }
            }
        }
        break;
    case 1:
        res = RTC_LEAP_YEAR_1;
        break;
    case 2:
        res = RTC_LEAP_YEAR_2;
        break;
    case 3:
    default:
        res = RTC_LEAP_YEAR_3;
        break;
    }

    return (res);
}

/*--------------------------------------------------*/
/** 
  * @brief  Copy Clock Value.
  * @param  p_dst :Destination Address.
  * @param  p_src :Source Address.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
static void copy_clock_value(horol_clock_value_t *p_dst, horol_clock_value_t *p_src)
{
    p_dst->century  = p_src->century;
    p_dst->year     = p_src->year;
    p_dst->notation = p_src->notation;
    p_dst->meridiem = p_src->meridiem;
    p_dst->month    = p_src->month;
    p_dst->date     = p_src->date;
    p_dst->day      = p_src->day;
    p_dst->hour     = p_src->hour;
    p_dst->min      = p_src->min;
    p_dst->sec      = p_src->sec;
}

/**
 *  @}
 */ /* End of group horol_Private_fuctions */


/*------------------------------------------------------------------------------*/
/*  Public Function                                                             */
/*------------------------------------------------------------------------------*/
/** 
 *  @addtogroup horol_Exported_functions
 *  @{
 */
/*--------------------------------------------------*/
/** 
  * @brief  Initialize.
  * @param  p_param :Initial Setting Information Source Address.
  * @return Instance Address.
  * @retval (value != NULL) :Valid value.
  * @note   Parameter check isn't performed.
  * @note   When write data, use CaseA or CaseB or CaseC (Please refer to RTC datasheet). \n
            CaseA :Use to 1Hz interrupt. \n
            CaseB :Reset Counter. \n
            CaseC :Disable Timer. \n
  * @attention Singleton.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
horol_t *horol_initialize(horol_initial_setting_t *p_param)
{
    rtc_t *p_rtc = &instance.info.rtc;
    horol_clock_value_t *p_setting = &instance.init.setting;

    /*------------------------------*/
    /*  Initial Parameter Copy      */
    /*------------------------------*/
    instance.init.id = p_param->id;
    copy_clock_value(p_setting, &p_param->setting);
    instance.init.clock_update = p_param->clock_update;
    /*------------------------------*/
    /*  Set Read Layer              */
    /*------------------------------*/
    copy_clock_value(&instance.info.clock.value[(uint32_t)HOROL_LAYER_0], p_setting);
    instance.info.clock.read = HOROL_LAYER_0;
    /*------------------------------*/
    /*  Construct                   */
    /*------------------------------*/
    /* Register Allocation */
    p_rtc->p_instance = TSB_RTC;
    /*------------------------------*/
    /*  Initialize                  */
    /*------------------------------*/
    if (rtc_init(p_rtc) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*------------------------------*/
    /*  Initial Setting             */
    /*------------------------------*/
    /*--- Leap Year     ---*/
    if (rtc_clock_set_leap(p_rtc, get_leap_setting(p_setting->century, p_setting->year)) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- Hour Notation ---*/
    if (rtc_set_hour_notation(p_rtc, p_setting->notation) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- Year          ---*/
    if (rtc_clock_set_year(p_rtc, p_setting->year) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- Month         ---*/
    if (rtc_clock_set_month(p_rtc, p_setting->month) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- Date          ---*/
    if (rtc_clock_set_date(p_rtc, p_setting->date) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- Day           ---*/
    if (rtc_clock_set_day(p_rtc, p_setting->day) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- Hour          ---*/
    if (p_setting->notation == RTC_HOUR_NOTATION_24)
    {
        if (rtc_clock_set_hour_24(p_rtc, p_setting->hour) != TXZ_SUCCESS)
        {
            debug_error();
        }
    }
    else
    {
        if (rtc_clock_set_hour_12(p_rtc, p_setting->meridiem, p_setting->hour) != TXZ_SUCCESS)
        {
            debug_error();
        }
    }
    /*--- Minute        ---*/
    if (rtc_clock_set_min(p_rtc, p_setting->min) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- Second        ---*/
    if (rtc_clock_set_sec(p_rtc, p_setting->sec) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*------------------------------*/
    /*  Select Interrupt Source     */
    /*------------------------------*/
    if (rtc_set_int_source(p_rtc, RTC_INT_SRC_1HZ) != TXZ_SUCCESS)
    {
        debug_error();
    }

    return(&instance);
}

/*--------------------------------------------------*/
/** 
  * @brief  Finalize.
  * @param  p_instance :Instance Source Address.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void horol_finalize(horol_t *p_instance)
{
    rtc_t *p_rtc = &p_instance->info.rtc;

    /*------------------------------*/
    /*  Stop                        */
    /*------------------------------*/
    horol_stop(p_instance);
    /*------------------------------*/
    /*  RTC Finalize                */
    /*------------------------------*/
    if (rtc_deinit(p_rtc) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*------------------------------*/
    /*  Destruct                    */
    /*------------------------------*/
    /* Register Release */
    p_rtc->p_instance = HOROL_NULL;
}

/*--------------------------------------------------*/
/** 
  * @brief  Start.
  * @param  p_instance :Instance Source Address.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void horol_start(horol_t *p_instance)
{
    rtc_t *p_rtc = &p_instance->info.rtc;

    /*--- Clock       ---*/
    if (rtc_clock_enable(p_rtc) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- INTRTC      ---*/
    if (rtc_enable_int(p_rtc) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- IRQ Enable  ---*/
    bsp_irq_rtc_enable();
    /*--- NVIC Enable ---*/
    NVIC_EnableIRQ(INTRTC_IRQn);
}

/*--------------------------------------------------*/
/** 
  * @brief  Stop.
  * @param  p_instance :Instance Source Address.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention This function is not available in interrupt.
  */
/*--------------------------------------------------*/
void horol_stop(horol_t *p_instance)
{
    rtc_t *p_rtc = &p_instance->info.rtc;

    /*--- NVIC Disable ---*/
    NVIC_DisableIRQ(INTRTC_IRQn);
    /*--- INTRTC       ---*/
    if (rtc_disable_int(p_rtc) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- Clock        ---*/
    if (rtc_clock_disable(p_rtc) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- IRQ Disable ---*/
    bsp_irq_rtc_disable();
}

/*--------------------------------------------------*/
/** 
  * @brief  Get Current Clock Information.
  * @param  p_instance :Instance Source Address.
  * @param  p_dst      :Current Clock Destination Address.
  * @return -
  * @retval -
  * @note   Parameter check isn't performed.
  * @attention This function is not available in interrupt.
  * @attention In progress, RTC Interrupt Disable.
  */
/*--------------------------------------------------*/
void horol_get_current_clock(horol_t *p_instance, horol_clock_value_t *p_dst)
{
    /*------------------------------*/
    /* NVIC Disable                 */
    /*------------------------------*/
    NVIC_DisableIRQ(INTRTC_IRQn);
    /*------------------------------*/
    /*  Read Layer                  */
    /*------------------------------*/
    copy_clock_value(p_dst, &p_instance->info.clock.value[(uint32_t)p_instance->info.clock.read]);
    /*------------------------------*/
    /* NVIC Enable                  */
    /*------------------------------*/
    NVIC_EnableIRQ(INTRTC_IRQn);
}

/*--------------------------------------------------*/
/** 
  * @brief  IRQ Handler.
  * @param  p_instance :Instance Source Address.
  * @return -
  * @retval -
  * @note   When read data, use CaseA or CaseB (Please refer to RTC datasheet). \n
            CaseA :Read in 1Hz interrupt. \n
            CaseB :Twice read. \n
            This Sample uses CaseA(1Hz interrupt), so no need to twice read.
  */
/*--------------------------------------------------*/
void horol_irq_handler(horol_t *p_instance)
{
    rtc_t *p_rtc = &p_instance->info.rtc;
    HorolValueLayer write = HOROL_LAYER_0;
    horol_clock_value_t *p_setting = HOROL_NULL;

    /*------------------------------*/
    /*  Deside Write Layer          */
    /*------------------------------*/
    if ((uint32_t)((uint32_t)p_instance->info.clock.read + 1) >= (uint32_t)HOROL_LAYER_MAX)
    {
        write = HOROL_LAYER_0;
    }
    else
    {
        write = (HorolValueLayer)(p_instance->info.clock.read + 1);
    }
    p_setting = &p_instance->info.clock.value[(uint32_t)write];
    /*------------------------------*/
    /*  Update Information          */
    /*------------------------------*/
    /*--- Year ---*/
    if (rtc_clock_get_year(p_rtc, &p_setting->year) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- Hour Notaion ---*/
    if (rtc_get_hour_notation(p_rtc, &p_setting->notation) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- Hour/Meridiem ---*/
    if (p_setting->notation == RTC_HOUR_NOTATION_12)
    {
        if (rtc_clock_get_hour_12(p_rtc, &p_setting->meridiem, &p_setting->hour) != TXZ_SUCCESS)
        {
            debug_error();
        }
    }
    else
    {
        if (rtc_clock_get_hour_24(p_rtc, &p_setting->hour) != TXZ_SUCCESS)
        {
            debug_error();
        }
    }
    /*--- Day ---*/
    if (rtc_clock_get_day(p_rtc, &p_setting->day) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- Month ---*/
    if (rtc_clock_get_month(p_rtc, &p_setting->month) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- Date ---*/
    if (rtc_clock_get_date(p_rtc, &p_setting->date) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- Minute ---*/
    if (rtc_clock_get_min(p_rtc, &p_setting->min) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- Sec ---*/
    if (rtc_clock_get_sec(p_rtc, &p_setting->sec) != TXZ_SUCCESS)
    {
        debug_error();
    }
    /*--- The end of century ---*/
    if ((p_setting->year == 99) && (p_instance->info.end == TXZ_DISABLE))
    {
        p_instance->info.end = TXZ_ENABLE;
    }
    /*--- Century ---*/
    if ((p_setting->year == 0) && (p_instance->info.end == TXZ_ENABLE))
    {
        /* Update information. */
        p_setting->century = p_instance->info.clock.value[(uint32_t)p_instance->info.clock.read].century + 1;
        p_instance->info.end = TXZ_DISABLE;
    }
    else
    {
        p_setting->century = p_instance->info.clock.value[(uint32_t)p_instance->info.clock.read].century;
    }
    /*------------------------------*/
    /*  Leap Year Setting           */
    /*------------------------------*/
    /* When year has changed, set leap year. */
    if (p_setting->year != p_instance->info.clock.value[(uint32_t)p_instance->info.clock.read].year)
    {
        if (rtc_clock_set_leap(p_rtc, get_leap_setting(p_setting->century, p_setting->year)) != TXZ_SUCCESS)
        {
            debug_error();
        }
    }
    /*------------------------------*/
    /*  Read Layer Change           */
    /*------------------------------*/
    p_instance->info.clock.read = write;
    /*------------------------------*/
    /*  Call Handler                */
    /*------------------------------*/
    if (p_instance->init.clock_update != HOROL_NULL)
    {
        p_instance->init.clock_update(p_instance->init.id);
    }
}

/*--------------------------------------------------*/
/** 
  * @brief  Make A.D.
  * @param  century :Century.
  * @param  year    :Year.
  * @return A.D.
  * @retval -
  * @note   -
  */
/*--------------------------------------------------*/
uint32_t horol_make_anno_domini(uint8_t century, uint8_t year)
{
    uint32_t ad = 0;

    if (century > 0)
    {
        ad = (uint32_t)((century - 1) * 100);
    }
    ad += (uint32_t)year;

    return (ad);
}

/** 
 *  @}
 */ /* End of group horol_Exported_functions */

/**
 *  @}
 */ /* End of group horol */

/**
 *  @} 
 */ /* End of group Utility */

#endif /* defined(__APP_HOROL_H)  */
#ifdef __cplusplus
}
#endif /* __cplusplus */
