/*===================================================================
    Title:          Board Specific Functions.
    Date:           2019/5/21
    Copyright:      Headspring Inc.
    Target CPU:     TMS320F28377S(200MHz)
    Target Board:   HSP0179-B00-0002A
===================================================================*/
#include <hs_io.h>
#include "HSP0179-B00-0002A.h"
#include "f2837xs_pinmux.h"

/*=======================================================*/
/**
 * @brief   Initialize GPIO Setting.
 * @return  None.
 */
void InitializeBoard(void)
{
    UINT32 lockValA;
    UINT32 lockValB;
    UINT32 lockValC;
    UINT32 lockValD;
    UINT32 lockValE;

    EALLOW;

    //
    // Save the current value of the GPIO lock registers
    //
    lockValA = GpioCtrlRegs.GPALOCK.all;
    lockValB = GpioCtrlRegs.GPBLOCK.all;
    lockValC = GpioCtrlRegs.GPCLOCK.all;
    lockValD = GpioCtrlRegs.GPDLOCK.all;
    lockValE = GpioCtrlRegs.GPELOCK.all;

    //
    // Unlock the GPIO control registers
    //
    GpioCtrlRegs.GPALOCK.all = 0x00000000;
    GpioCtrlRegs.GPBLOCK.all = 0x00000000;
    GpioCtrlRegs.GPCLOCK.all = 0x00000000;
    GpioCtrlRegs.GPDLOCK.all = 0x00000000;
    GpioCtrlRegs.GPELOCK.all = 0x00000000;

    //
    // Clear the mux register fields that are about to be changed
    //
    GpioCtrlRegs.GPAMUX1.all    &= ~GPAMUX1_MASK;
    GpioCtrlRegs.GPAMUX2.all    &= ~GPAMUX2_MASK;
    GpioCtrlRegs.GPAGMUX1.all   &= ~GPAMUX1_MASK;
    GpioCtrlRegs.GPAGMUX2.all   &= ~GPAMUX2_MASK;
    GpioCtrlRegs.GPBMUX1.all    &= ~GPBMUX1_MASK;
    GpioCtrlRegs.GPBMUX2.all    &= ~GPBMUX2_MASK;
    GpioCtrlRegs.GPBGMUX1.all   &= ~GPBMUX1_MASK;
    GpioCtrlRegs.GPBGMUX2.all   &= ~GPBMUX2_MASK;
    GpioCtrlRegs.GPCMUX1.all    &= ~GPCMUX1_MASK;
    GpioCtrlRegs.GPCMUX2.all    &= ~GPCMUX2_MASK;
    GpioCtrlRegs.GPCGMUX1.all   &= ~GPCMUX1_MASK;
    GpioCtrlRegs.GPCGMUX2.all   &= ~GPCMUX2_MASK;
    GpioCtrlRegs.GPDMUX1.all    &= ~GPDMUX1_MASK;
    GpioCtrlRegs.GPDMUX2.all    &= ~GPDMUX2_MASK;
    GpioCtrlRegs.GPDGMUX1.all   &= ~GPDMUX1_MASK;
    GpioCtrlRegs.GPDGMUX2.all   &= ~GPDMUX2_MASK;
    GpioCtrlRegs.GPEMUX1.all    &= ~GPEMUX1_MASK;
    GpioCtrlRegs.GPEGMUX1.all   &= ~GPEMUX1_MASK;

    //
    // Write pin muxing to mux registers
    //
    GpioCtrlRegs.GPAGMUX1.all   |=  GPAGMUX1_VALUE;
    GpioCtrlRegs.GPAGMUX2.all   |=  GPAGMUX2_VALUE;
    GpioCtrlRegs.GPAMUX1.all    |=  GPAMUX1_VALUE;
    GpioCtrlRegs.GPAMUX2.all    |=  GPAMUX2_VALUE;
    GpioCtrlRegs.GPBGMUX1.all   |=  GPBGMUX1_VALUE;
    GpioCtrlRegs.GPBGMUX2.all   |=  GPBGMUX2_VALUE;
    GpioCtrlRegs.GPBMUX1.all    |=  GPBMUX1_VALUE;
    GpioCtrlRegs.GPBMUX2.all    |=  GPBMUX2_VALUE;
    GpioCtrlRegs.GPCGMUX1.all   |=  GPCGMUX1_VALUE;
    GpioCtrlRegs.GPCGMUX2.all   |=  GPCGMUX2_VALUE;
    GpioCtrlRegs.GPCMUX1.all    |=  GPCMUX1_VALUE;
    GpioCtrlRegs.GPCMUX2.all    |=  GPCMUX2_VALUE;
    GpioCtrlRegs.GPDGMUX1.all   |=  GPDGMUX1_VALUE;
    GpioCtrlRegs.GPDGMUX2.all   |=  GPDGMUX2_VALUE;
    GpioCtrlRegs.GPDMUX1.all    |=  GPDMUX1_VALUE;
    GpioCtrlRegs.GPDMUX2.all    |=  GPDMUX2_VALUE;
    GpioCtrlRegs.GPEGMUX1.all   |=  GPEGMUX1_VALUE;
    GpioCtrlRegs.GPEMUX1.all    |=  GPEMUX1_VALUE;

    //
    // Write pin analog mode select to registers
    //
    GpioCtrlRegs.GPBAMSEL.all   &= ~GPBAMSEL_MASK;
    GpioCtrlRegs.GPBAMSEL.all   |= GPBAMSEL_VALUE;

    GpioCtrlRegs.GPACTRL.all    = 0x00000000UL;
    GpioCtrlRegs.GPAQSEL1.all   = 0x00000000UL;
    GpioCtrlRegs.GPAQSEL2.all   = 0x00000000UL;
    GpioCtrlRegs.GPACSEL1.all   = 0x00000000UL;
    GpioCtrlRegs.GPACSEL2.all   = 0x00000000UL;
    GpioCtrlRegs.GPACSEL3.all   = 0x00000000UL;
    GpioCtrlRegs.GPACSEL4.all   = 0x00000000UL;
    GpioCtrlRegs.GPAPUD.all     = 0x00FF0FFFUL;     /* 12-15,24-31: PullUp                       */
    GpioCtrlRegs.GPAINV.all     = 0x00000000UL;
    GpioCtrlRegs.GPAODR.all     = 0x00000000UL;

    GpioCtrlRegs.GPBCTRL.all    = 0x00000000UL;
    GpioCtrlRegs.GPBQSEL1.all   = 0x00000000UL;
    GpioCtrlRegs.GPBQSEL2.all   = 0x0FF00000UL;     /* GPIO58-61 : Async(for SPI)        */
    GpioCtrlRegs.GPBCSEL1.all   = 0x00000000UL;
    GpioCtrlRegs.GPBCSEL2.all   = 0x00000000UL;
    GpioCtrlRegs.GPBCSEL3.all   = 0x00000000UL;
    GpioCtrlRegs.GPBCSEL4.all   = 0x00000000UL;
    GpioCtrlRegs.GPBPUD.all     = 0x3C3C0C20UL;     /* 32-36,38-41,44-49,54-57,62,63: PullUp                       */
    GpioCtrlRegs.GPBINV.all     = 0x00000000UL;
    GpioCtrlRegs.GPBODR.all     = 0x00000000UL;
    GpioCtrlRegs.GPBAMSEL.all   = 0x00000000UL;

    GpioCtrlRegs.GPCCTRL.all    = 0x00000000UL;
    GpioCtrlRegs.GPCQSEL1.all   = 0x00000000UL;
    GpioCtrlRegs.GPCQSEL2.all   = 0x00000000UL;
    GpioCtrlRegs.GPCCSEL1.all   = 0x00000000UL;
    GpioCtrlRegs.GPCCSEL2.all   = 0x00000000UL;
    GpioCtrlRegs.GPCCSEL3.all   = 0x00000000UL;
    GpioCtrlRegs.GPCCSEL4.all   = 0x00000000UL;
    GpioCtrlRegs.GPCPUD.all     = 0xE6507F18UL;     /* 64-66,69-71,79-83,85,87,88,91,92: PullUp            */
    GpioCtrlRegs.GPCINV.all     = 0x00000000UL;
    GpioCtrlRegs.GPCODR.all     = 0x00000000UL;

    GpioCtrlRegs.GPDCTRL.all    = 0x00000000UL;
    GpioCtrlRegs.GPDQSEL1.all   = 0x00000000UL;
    GpioCtrlRegs.GPDQSEL2.all   = 0x00000000UL;
    GpioCtrlRegs.GPDCSEL1.all   = 0x00000000UL;
    GpioCtrlRegs.GPDCSEL2.all   = 0x00000000UL;
    GpioCtrlRegs.GPDCSEL3.all   = 0x00000000UL;
    GpioCtrlRegs.GPDCSEL4.all   = 0x00000000UL;
    GpioCtrlRegs.GPDPUD.all     = 0xFFFFFFF7UL;     /* 99: PullUp                       */
    GpioCtrlRegs.GPDINV.all     = 0x00000000UL;
    GpioCtrlRegs.GPDODR.all     = 0x00000000UL;

    GpioCtrlRegs.GPECTRL.all    = 0x00000000UL;
    GpioCtrlRegs.GPEQSEL1.all   = 0x00000000UL;
    GpioCtrlRegs.GPEQSEL2.all   = 0x00000000UL;
    GpioCtrlRegs.GPECSEL1.all   = 0x00000000UL;
    GpioCtrlRegs.GPECSEL2.all   = 0x00000000UL;
    GpioCtrlRegs.GPECSEL3.all   = 0x00000000UL;
    GpioCtrlRegs.GPECSEL4.all   = 0x00000000UL;
    GpioCtrlRegs.GPEPUD.all     = 0xFFFFFFDFUL;     /* 133: PullUp                       */
    GpioCtrlRegs.GPEINV.all     = 0x00000000UL;
    GpioCtrlRegs.GPEODR.all     = 0x00000000UL;

    GpioCtrlRegs.GPFCTRL.all    = 0x00000000UL;
    GpioCtrlRegs.GPFQSEL1.all   = 0x00000000UL;
    GpioCtrlRegs.GPFCSEL1.all   = 0x00000000UL;
    GpioCtrlRegs.GPFCSEL2.all   = 0x00000000UL;
    GpioCtrlRegs.GPFPUD.all     = 0xFFFFFFFFUL;
    GpioCtrlRegs.GPFINV.all     = 0x00000000UL;
    GpioCtrlRegs.GPFODR.all     = 0x00000000UL;

    GpioCtrlRegs.GPADIR.all     = 0x00FF0000UL;
    GpioCtrlRegs.GPBDIR.all     = 0x00000000UL;
    GpioCtrlRegs.GPCDIR.all     = 0x00000000UL;
    GpioCtrlRegs.GPDDIR.all     = 0x00000000UL;
    GpioCtrlRegs.GPEDIR.all     = 0x00000000UL;
    GpioCtrlRegs.GPFDIR.all     = 0x00000000UL;

    //
    // Restore GPIO lock register values
    //
    GpioCtrlRegs.GPALOCK.all = lockValA;
    GpioCtrlRegs.GPBLOCK.all = lockValB;
    GpioCtrlRegs.GPCLOCK.all = lockValC;
    GpioCtrlRegs.GPDLOCK.all = lockValD;
    GpioCtrlRegs.GPELOCK.all = lockValE;

    EDIS;
}


/*=======================================================*/
/**
 * @brief   Digital Output port configuration.
 * @param   Sets the output value.
 * @return  None.
 */
void OutputPIO(UINT32 OutputData)
{
    UINT32  PortA;

    PortA   =   (OutputData & 0x000000FFUL)<<16;  /* DO0-7 GPIO16-23 */

    GpioDataRegs.GPASET.all = PortA;

    GpioDataRegs.GPACLEAR.all = PortA ^ 0x00FF0000UL;
}

/*=======================================================*/
/**
 * @brief   Acquisition of digital input values.
 * @return  Value of the digital input port.
 */
UINT32 InputPIO(void)
{
    UINT32  ret;
    UINT32  PortC;

    PortC   = GpioDataRegs.GPCDAT.all;

    ret     = (PortC & 0x00000018UL)>>3;    /* DI0-1 GPIO67-68 */
    ret    |= (PortC & 0x00007E00UL)>>7;    /* DI2-7 GPIO73-78 */

    return (ret);
}

/*=======================================================*/
/**
 * @brief   Reset a specific digital output port.
 * @param   BitNo.
 * @return  None.
 */
void ClearBitPIO(UINT16 BitNo)
{
	switch(BitNo)
	{
		case 0U:	GpioDataRegs.GPACLEAR.bit.GPIO16 = 1U;	break;
		case 1U:	GpioDataRegs.GPACLEAR.bit.GPIO17 = 1U;	break;
		case 2U:	GpioDataRegs.GPACLEAR.bit.GPIO18 = 1U;	break;
		case 3U:	GpioDataRegs.GPACLEAR.bit.GPIO19 = 1U;	break;
		case 4U:	GpioDataRegs.GPACLEAR.bit.GPIO20 = 1U;	break;
		case 5U:	GpioDataRegs.GPACLEAR.bit.GPIO21 = 1U;	break;
		case 6U:	GpioDataRegs.GPACLEAR.bit.GPIO22 = 1U;	break;
		case 7U:	GpioDataRegs.GPACLEAR.bit.GPIO23 = 1U;	break;
		default:	break;
	}
}

/*=======================================================*/
/**
 * @brief   Particular set of digital output ports.
 * @param   BitNo.
 * @return  None.
 */
void SetBitPIO(UINT16 BitNo)
{
	switch(BitNo)
	{
		case 0U:	GpioDataRegs.GPASET.bit.GPIO16 = 1U;	break;
		case 1U:	GpioDataRegs.GPASET.bit.GPIO17 = 1U;	break;
		case 2U:	GpioDataRegs.GPASET.bit.GPIO18 = 1U;	break;
		case 3U:	GpioDataRegs.GPASET.bit.GPIO19 = 1U;	break;
		case 4U:	GpioDataRegs.GPASET.bit.GPIO20 = 1U;	break;
		case 5U:	GpioDataRegs.GPASET.bit.GPIO21 = 1U;	break;
		case 6U:	GpioDataRegs.GPASET.bit.GPIO22 = 1U;	break;
		case 7U:	GpioDataRegs.GPASET.bit.GPIO23 = 1U;	break;
		default:	break;
	}
}
