/*==============================================================================
Title:				Three-phase PFC converter reference design for Toshiba
File Name:		main.c
Date:					2021/8/19
Copyright:		Headspring Inc.
Target CPU:		TMS320F28377S (200MHz)
Target Board:	HSP0179-B00-0002A
Target Inverter:HSP0248-B00-0001
================================================================================*/

/*** Include *******************************************************/
#include <hs_math.h>
#include <hs_io.h>
#include "HSP0179-B00-0002A.h"

/*** Define ********************************************************/
/* PWM setting */
#define CARRIERPERIOD_NS (10000U)	/* Switching Period[ns] */
#define CARRIERFREQ_HZ (100000.0F)/* Switching frequency[Hz] */
#define INTERRUPT_MASK (2U)				/* Decimation number */
#define DEADTIME_NS (400.0F)			/* Deadtime[ns] */

/* Timer setting */
#define TIMER0INTERVAL_US (1000UL)/* Timer0 period[us] */

/* AD setting */
#define VAC_GAIN (525.0F)			/* AC voltage AD conversion gain */
#define VAC_OFFSET (0.0F)			/* AC voltage AD conversion offset */
#define IAC_GAIN (20.0F)			/* AC current AD conversion gain */
#define IAC_OFFSET (0.0F)			/* AC current AD conversion offset */
#define VDC_GAIN (256.0F)			/* DC voltage AD conversion gain */
#define VDC_OFFSET (257.0F)		/* DC voltage AD conversion offset */
#define VDC_M_GAIN (128.0F)		/* DC voltage AD conversion gain */
#define VDC_M_OFFSET (128.5F)	/* DC voltage AD conversion offset */
#define TH_GAIN (2.5F)				/* Temp AD conversion gain */
#define TH_OFFSET (2.5F)			/* Temp AD conversion offset */

/* Digital I/O */
#define DO_MAIN (0x0002UL)		/* Digital output for driving the main relay */

/* Mathematical constant */
#define MATHPI (3.1415926F)			/* π */
#define MATHSQRT3 (1.7320508F)	/* √3 */

/* Controller parameter */
#define IACKP1 (14.0F)			  /* AC current Kp1 */
#define IACKP2 (12.0F)				/* AC current Kp2 */
#define IACTI (1.5F)					/* AC current Ti[ms] */
#define VDCKP (0.3F)					/* DC voltage Kp */
#define VDCTI (200.0F)				/* DC voltage Ti[ms] */
#define IACLIMIT (15.0F)			/* AC reference current limit[A] */
#define MRATELIMIT (0.95F)		/* modulation rate limit */

/*** Typedef *******************************************************/
/* System state */
typedef enum{
	State_Error = 0,			/* 0:Error */
	State_Stop,						/* 1:Stop */
	State_Wait,						/* 2:Wait */
	State_Run,						/* 3:Run */
	State_Precharge,			/* 4:Precharge */
	State_Init,						/* 5:Initialization */
	State_NUM
}SystemState;

/* Protection */
typedef enum{
	Err_IAC = 0,					/* 0:Iac OC protection */
	Err_VDC_UV,						/* 1:Vdc UV protection */
	Err_VDC_OV,						/* 2:Vdc OV protection */
	Err_FLT,							/* 3:gate driver error */
	Err_VAC,							/* 4:Vac OV error */
	Err_Th_OT,						/* 5:Th OT protection  */
	Err_WDT,							/* 6:Watch dog timer error */
	Err_TRIP,							/* 7:Trip state */
	Err_NUM
}ErrState;

/* DI signal */
typedef enum{
	DI_RDY = 0,					/* 0:gate driver power failure signal */
	DI_FLT,							/* 1:gate driver fault signal */
	DI_NUM
}DISignal;

/*** Functions *****************************************************/
void Init(void);					/* Initialization function */
void PWMIntFunc(void);		/* PWM interrupt function */
void TZIntFunc(void);			/* TZ interrupt function */
void Timer0IntFunc(void);	/* Timer0 interrupt function */

void JudgeError(void);		/* Judge error condition */
void GetTh(void);					/* Getting temp on the heatsink*/
void SetState(void);			/* State machine */
void Start(void);					/* Pre-process before running */
void Stop(void);					/* Exit process at stop running */

void ProcVac(void); 			/* Detect zerocross, calculation of RMS */
void CalcPhase(void);			/* PLL */
void CalibADOffset(void);	/* Calibration of AD offsets */

void CtrlVdc(void);				/* DC voltage controller */
void BalanceVdc(void);		/* DC voltage balance controller */
void CtrlIac(void);				/* AC current controller */
void CalcMrate(void);			/* Calculation of modulation rate */

/*** Variables *****************************************************/
UINT32 DIdata = 0L;				/* Digital input state */

/* Sensor value */
FLOAT32	Vac = 0.0F;				/* AC voltage[V] */
FLOAT32	Iac = 0.0F;				/* AC current[A] */
FLOAT32	Vdc = 0.0F;				/* DC voltage[V] */
FLOAT32	Vdc_M = 0.0F;			/* DC voltage[V] */
FLOAT32 Th = 0.0F;				/* Temp[C] */

/* Sensor value applied LPF */
FLOAT32 VacLPF = 0.0F;
FLOAT32 VdcLPF = 0.0F;
FLOAT32 IacLPF = 0.0F;
FLOAT32 Vdc_MLPF = 0.0F;

/* LPF cut-off frequency[Hz] */
FLOAT32 VacFc = 1500.0F;
FLOAT32 VdcFc = 500.0F;
FLOAT32 IacFc = 2000.0F;

/* Coefficient for calculating LPF */
FLOAT32 VacLPFCoeff = 0.0F;
FLOAT32 VdcLPFCoeff = 0.0F;
FLOAT32 IacLPFCoeff = 0.0F;

/* Calibration */
FLOAT32 VdcRev    = 0.0F;			/* Inverse of DC voltage[1/V] */
FLOAT32 OffsetVac = 0.0F;			/* AC offset voltage[V] */
FLOAT32 OffsetIac = 0.0F;			/* AC offset current[A] */
FLOAT32 OffsetVdc = 0.0F;			/* DC offset voltage[V] */
FLOAT32 OffsetVdc_M = 0.0F;	/* DC offset voltage[V] */
FLOAT32 OffsetVac0 = 0.0F;		/* Temporary variable for calculating OffsetVac */
FLOAT32 OffsetIac0 = 0.0F;		/* Temporary variable for calculating OffsetIac */
UINT16  CntOffset = 0U;				/* Counter used at the offset-compensation */
UINT16  FlagCalibEnd = 0U;		/* 1:Calibration ended, 0:Not calibration ended */

/* Protection */
UINT16  ErrorState = 0U;				/* Error state (non-latch) */
UINT16  ErrorStateLatch = 0U;		/* Error state (latch) */
FLOAT32 ErrThrVdcHi_V = 418.0F;	/* DC over voltage error threshold[V] */
FLOAT32 ErrThrVdcLo_V = 285.0F;	/* DC under voltage error threshold[V] */
FLOAT32 ErrThrVdc_MHi_V = 212.5F; /* DC over voltage error threshold[V] */
FLOAT32 ErrThrVac_Vpk = 374.0F;	/* AC over current error threshold[A]（instantaneous current value） */
FLOAT32 ErrThrIac_Apk = 16.4F;	/* AC over current error threshold[A]（instantaneous current value） */
FLOAT32 ErrThrThHi_Deg = 75.0F;	/* AC over current error threshold[A]（instantaneous current value） */
UINT16  FlagWDT = 0U;				/* 0:WDT overflow not occurred, 1:WDT overflow occurred */
UINT16  FlagTZ  = 0U;				/* 0:TZ not occurred, 1:TZ occurred */
UINT16  FlagWaitGB = 0U;		/* 1:Waiting for GB */

/* State machine */
SystemState State = State_Init;	/* Current System state */
SystemState PreState = State_Init;	/* Previous state */
UINT16 InitEnd = 0U;						/* 0:Initializing, 1:Initialization complete */
UINT16 RunStartStopReq = 1U;		/* 0:Run->Wait, 1:Wait->Run */
UINT16 FlagEntryWait  = 0U;			/* Flag indicating Wait state */
UINT16 TimerEntryWait = 0U;			/* Counter for measuring time from entry Wait */
UINT16 FlagStateMainRelay = 0U;	/* Main Relay state */

/* RMS calculation */
FLOAT32 Vac_Vrms  = 0.0F;			/* AC Voltage[Vrms] */
FLOAT32 Vac_Vrms0 = 0.0F;			/* AC Voltage[Vrms] (previous cycle) */
FLOAT32 VacSum_V  = 0.0F;			/* Integration AC Voltage between period [V] */
INT16   SignVac  = 0U;				/* Sign Vac (1:positive, -1:negative) */
INT16   SignVac0 = 0U;				/* Sign Vac at previous cycle */
UINT16  FlagZeroCross = 0U;
UINT16  PeriodCount = 0U;			/* counter for calculating AC Voltage period */
FLOAT32 Iac_Arms  = 0.0F;			/* AC Current[Arms] */
FLOAT32 IacSum_A  = 0.0F;			/* Integration AC Current between period [A] */
FLOAT32 FreqSys   = 50.0F;

/* AVE Calculation */
FLOAT32 VdcAVE    = 0.0F;
FLOAT32 Vdc_MAVE  = 0.0F;
FLOAT32 VdcSum_V    = 0.0F;
FLOAT32 Vdc_MSum_V  = 0.0F;

/* PLL calculation */
FLOAT32 Phase_rad = 0.0F;			/* Voltage phase of AC power source */
FLOAT32 RefPhase_rad = 0.0F;	/* Reference phase[rad] */
FLOAT32 RefDeltaPhase_rad = 0.0F;	/* Reference phase increment value[rad] */

/* AC current controller */
FLOAT32 IacRef = 0.0F;				/* Iac reference value */
FLOAT32 ErrIac = 0.0F;				/* Iac error[A] */
FLOAT32 IacKp  = IACKP1;      /* Proportional gain */
FLOAT32 IacKpComp = 0.0F;			/* Compensation of Kp at 200V */
FLOAT32 IacKi = IACKP1 * INTERRUPT_MASK / (IACTI * 0.001F * CARRIERFREQ_HZ);	/* Integral gain */
FLOAT32 IacTi = IACTI;				/* Time constant [msec] */
FLOAT32 IacPElement  = 0.0F;	/* Proportional term */
FLOAT32 IacIElement  = 0.0F;	/* Integral term */
FLOAT32 IacPIElement = 0.0F;	/* (Proportional term) + (Integral term) */
FLOAT32 IacLimitHi   = IACLIMIT; /* Upper limit of Iac reference */
FLOAT32 VacOut = 0.0F;				/* Controller output */

/* DC voltage controller */
FLOAT32 VdcRef = 0.0F;				/* Vdc reference value */
FLOAT32 ErrVdc = 0.0F;				/* Vdc error[V] */
FLOAT32 VdcKp =  0.1F;      	/* Proportional gain */
FLOAT32 VdcKi = VDCKP * (TIMER0INTERVAL_US * 0.001F * 0.001F) / (VDCTI * 0.001F);	/* Integral gain */
FLOAT32 VdcTi = VDCTI;				/* Time constant [msec] */
FLOAT32 VdcPElement  = 0.0F;	/* Proportional term */
FLOAT32 VdcIElement  = 0.0F;	/* Integral term */
FLOAT32 VdcPIElement = 0.0F;	/* (Proportional term) + (Integral term) */
FLOAT32 VdcCtrlUpperLimit = 1.5F;

/* DC balance controller */
FLOAT32 ErrVdc_M = 0.0F;      /* Vdc error[V] */
FLOAT32 Vdc_MKp = 0.06F;			/* Proportional gain */
FLOAT32 Vdc_MKi = 0.06F * (TIMER0INTERVAL_US * 0.001F * 0.001F) / (VDCTI * 0.001F);   /* Integral gain */
FLOAT32 Vdc_MTi = 1000.0F;    /* Time constant [msec] */
FLOAT32 Vdc_MPElement  = 0.0F;/* Proportional term */
FLOAT32 Vdc_MIElement  = 0.0F;/* Integral term */
FLOAT32 Vdc_MPIElement = 0.0F;/* (Proportional term) + (Integral term) */

/* Soft-start (DC voltage controller) */
FLOAT32 ConstVdcRef   = 380.0F;	/* Target DC voltage[V] */
UINT16  FlagVdcRefRamp = 0U;		/* Flag indicating soft-start */
FLOAT32 DeltaVdcRefRamp = 0.0F;	/* Increment value of DC voltage reference */

/* Modulation rate */
FLOAT32 MRate1Out = -1.0F;			/* Modulation rate */
FLOAT32 MRate2Out =  1.0F;			/* Modulation rate */

//#if 0
UINT16  t1 = 0U;
UINT16  t2 = 0U;
FLOAT32 t_pwm = 0.0F;
FLOAT32 tmax = 0.0F;
//#endif

/* Mathematical constant */
const FLOAT32 MathPi		= MATHPI;									/* π */
const FLOAT32 Math2Pi		= MATHPI * 2.0F;					/* 2π */
const FLOAT32 MathRev2Pi= 1.0F / (MATHPI * 2.0F);	/* 1/(2π) */
const FLOAT32 MathSqrt3	= MATHSQRT3;							/* √3 */
const FLOAT32 MathRevSqrt3 = 1.0F / MATHSQRT3;		/* 1/√3 */

const FLOAT32 ControlPeriod = INTERRUPT_MASK / CARRIERFREQ_HZ;	/* PWM interrupt period [sec] */

/* Limit of AC voltage period */
const FLOAT32 PeriodCountMax = CARRIERFREQ_HZ / (INTERRUPT_MASK * 30.0F);
const FLOAT32 PeriodCountMin = CARRIERFREQ_HZ / (INTERRUPT_MASK * 70.0F);
const UINT16  CarrierPeriod_20ns = CARRIERPERIOD_NS / 20U;
const FLOAT32 Deadtime_10ns     = DEADTIME_NS / 10.0F;
const FLOAT32 MRateThrIacICtrl = 0.6F;
const FLOAT32 SlowTurnOnComp   = 0.12F;
FLOAT32 vth;
/* FW ver. */
const UINT16  Version = 0x8174;	/* month,day(2digit),No */

/*=======================================================*/
/**
* @brief   Main function.
* @return  None.
*/
void main(void)
{
	Init();		/* Initialize */

	/* Main Loop */
	while(1)
	{
		ClearWDT(0U);							/* Clear WDT counter */
		FlagWDT = ReadFlagWDT();	/* Check WDT overflow */

		DIdata = InputPIO();			/* DI read */

		UpdateSCOPE();						/* Communication with HSDT-DP */
	}
}

/*=======================================================*/
/**
* @brief   Initialization function.
* @return  None.
*/
void Init(void){

	DisableGlobalInterrupt();			/* Disable Global Interrupt */

	InitializeHSLIB_TypeB();			/* Initialize CPU Setting */
	InitializeBoard();						/* Initialize GPIO Setting */
	InitSCOPE();									/* Initialize Scope Function */

	/* Initialize AD Setting */
	InitADCA();
	InitADCB();
	InitADCC();
	InitADCD();

	SetupSocADCA(0U,5U,2U,75U);		/* Set up AD0 SOC0(ADCA ADCINA2) Vac */
	SetupSocADCB(0U,5U,2U,75U);		/* Set up AD2 SOC0(ADCB ADCINB2) Iac */
	SetupSocADCD(0U,5U,0U,75U);		/* Set up AD6 SOC0(ADCC ADCIND0) Vdc */
	SetupSocADCC(0U,5U,2U,75U);		/* Set up AD4 SOC0(ADCD ADCINC2) Vdc_M */
	SetupSocADCA(3U,1U,4U,150U);  /* Set up AD1 SOC3(ADCC ADCINA4) Th */

	InitRangeOffsetADCA(2U,VAC_GAIN,VAC_OFFSET);	/* Set range(530.1) and offset(0.0) for ADCINA2 */
	InitRangeOffsetADCB(2U,IAC_GAIN,IAC_OFFSET);	/* Set range(30.0) and offset(0.0) for ADCINB2 */
	InitRangeOffsetADCD(0U,VDC_GAIN,VDC_OFFSET);	/* Set range(504.3) and offset(468.0) for ADCIND0 */
	InitRangeOffsetADCC(2U,VDC_M_GAIN,VDC_M_OFFSET);/* Set range(252.2) and offset(468.0) for ADCINC2 */
	InitRangeOffsetADCA(4U,TH_GAIN,TH_OFFSET);

	/* Comparator Setting */
	InitCMPSS1(2U, 31U, 21U, 63U);	/* Set up Comparator Subsystem for Vac */
	InitCMPSS3(2U, 31U, 21U, 63U);	/* Set up Comparator Subsystem for Iac */
	InitCMPSS7(2U, 31U, 21U, 63U);	/* Set up Comparator Subsystem for Vdc */
	InitCMPSS6(2U, 31U, 21U, 63U);	/* Set up Comparator Subsystem for Vdc_M */

	SetupLoOutputCMPSS1(3U);
	SetupHiOutputCMPSS1(3U);
	SetupLoOutputCMPSS3(3U);
	SetupHiOutputCMPSS3(3U);
	SetupHiOutputCMPSS7(3U);
	SetupHiOutputCMPSS6(3U);

	SetLovalueCMPSS1(-ErrThrVac_Vpk, 3.0F);	/* Set threshold for over voltage protection */
	SetHivalueCMPSS1(ErrThrVac_Vpk, 3.0F);	/* Set threshold for over voltage protection */
	SetLovalueCMPSS3(-ErrThrIac_Apk, 3.0F);	/* Set threshold for over current protection */
	SetHivalueCMPSS3(ErrThrIac_Apk, 3.0F);	/* Set threshold for over current protection */
	SetHivalueCMPSS7(ErrThrVdcHi_V, 3.0F);	/* Set threshold for over voltage protection */
	SetHivalueCMPSS6(ErrThrVdc_MHi_V, 3.0F);/* Set threshold for over voltage protection */

	/* Initialize WDT */
	InitWDT(0U, 1U);												/* Overflow time:13.1msec, Stop WDT forbidden */

	/* Timer0 Setting */
	InitTIMER0_IntTime(TIMER0INTERVAL_US);			/* Initialize Timer0 Function */
	InitInterruptFunctionTIMER0(&Timer0IntFunc);/* Initialize Interrupt Function */
	StartCountTIMER0();													/* Start Timer0 Counter */

	/* PWM Setting */
	/* Carrier type(0-2), Switching period[ns], P-sIace Deadtime[ns], N-sIace Deadtime[nsec],
	Update duty ratio both the peaks and valleys, Output pin(GPIO0-7) */
	InitHrPWM1_Raw(2U, CarrierPeriod_20ns, Deadtime_10ns, Deadtime_10ns, 0U, 0U, 0U, 0U, 0U, 1U, 1U, 1U);
	InitHrPWM2_Raw(2U, CarrierPeriod_20ns, Deadtime_10ns, Deadtime_10ns, 0U, 0U, 0U, 1U, 0U, 0U, 1U, 1U);
	/* in case of occuring an error, PWM3 makes GB of Q7,8 be delayed till a few usec after GB of the arm */
	InitHrPWM3_Raw(2U, CarrierPeriod_20ns, Deadtime_10ns, Deadtime_10ns, 0U, 0U, 0U, 1U, 0U, 0U, 1U, 1U);

	/* Trip-Zone Setting */
	InitEpwmXbarPWM();								/* Initialize ePWM X-BAR */
	SetupInputXbar2(68U);							/* TRIP2 : GPIO68(/FLT) */
	SetupEpwmXbarTRIP4PWM(0x0003UL);	/* TRIP4 : CMPSS1(Voltage)-Hi/Lo */
	SetupEpwmXbarTRIP5PWM(0x0030UL);	/* TRIP5 : CMPSS3(Current)-Hi/Lo */
	SetupEpwmXbarTRIP7PWM(0x1000UL);	/* TRIP7 : CMPSS7(Vdc)-Hi */
	InitDCAHTripPWM3(0x005AU);				/* TRIP2,4,5,7 */

	InitDCTripPWM3(0x000FU, 0x0002U);		/* Trip Combination Input, DCAEVT1.force */

	EnableTripControlPWM3(0x0040U);   /* Set up DCAEVT1 to trip source */

	/* Clear CMPSSLowLatch/CMPSSHighLatch */
	ClearLatchLoCMPSS1();
	ClearLatchHiCMPSS1();
	ClearLatchLoCMPSS3();
	ClearLatchHiCMPSS3();
	ClearLatchHiCMPSS7();
	ClearLatchHiCMPSS6();

	/* Clear trip flag */
	ClearTripFlagPWM1();
	ClearTripFlagPWM2();
	ClearTripFlagPWM3();

	SetDutyPWM1(-1.0F);		/* Set modulation rate */
	SetDutyPWM2(1.0F);		/* Set modulation rate */
	SetDutyPWM3(1.0F);		/* Set modulation rate(dummy) */
	InitAdsocaPWM1(1U, 1U);	/* Initialization of AD-conversion starting trigger */
	EnableAdsocaPWM1();		/* Enable AD trigger */
	wait1us(1000);				/* Wait 1msec */

	InitInterruptPWM1(1U, INTERRUPT_MASK);	/* Initialize PWM1 Function */
	InitTripInterruptFunctionPWM3(&TZIntFunc);/* Initialize Interrupt Function */
	InitInterruptFunctionPWM1(&PWMIntFunc);	/* Initialize Interrupt Function */
	EnableInterruptPWM1();		/* Enable PWM1 Interrupt */
	EnableInterruptTIMER0();	/* Enable Timer0 Interrupt */
	EnableGlobalInterrupt();	/* Enable Global Interrupt */

	InitEnd = 1U;							/* Flag for state machine (Init->Stop) */
}

/*=======================================================*/
/**
* @brief   Judge error condition.
* @return  None.
*/
void JudgeError(void){
	UINT16 Error = 0U;

	Error |= (GetStatusLolatchCMPSS3() | GetStatusHilatchCMPSS3())  << Err_IAC;	/* 0:Iac OC protection */
	Error |= (UINT16)((VdcLPF < ErrThrVdcLo_V)&&(State == State_Run)&&(FlagVdcRefRamp == 0U)) << Err_VDC_UV;	/* 1:Vdc UV protection */
	Error |= GetStatusHilatchCMPSS7() << Err_VDC_OV;		/* 2:Vdc OV protection */
	Error |= (UINT16)(((DIdata >> DI_FLT) == 0x0001U) || ((DIdata >> DI_RDY) == 0x0001U)) << Err_FLT;	/* 3:gate driver error */

	Error |= (GetStatusLolatchCMPSS1() | GetStatusHilatchCMPSS1()) << Err_VAC;	/* 4:Vac OV protection */
	Error |= (UINT16)(Th >= ErrThrThHi_Deg) << Err_Th_OT;	/* 5:Vdc imbalance protection */
	Error |= FlagWDT << Err_WDT;						/* 6:Watch dog timer error */
	Error |= GetTripFlagPWM3() << Err_TRIP;	/* 7:TRIP state */

	ErrorState = Error;
	if((State != State_Init) && (State != State_Error))
	{
		ErrorStateLatch |=  ErrorState;
	}
}

/*=======================================================*/
/**
* @brief   State machine.
* @return  None.
*/
void SetState(void){

	switch(State){
		/* 0:Error */
		case State_Error:
		/* reboot */
		break;

		/* 1:Stop */
		case State_Stop:
		if(ErrorStateLatch != 0U)	/* Event:Error detection */
		{
			/* Exit Stop */
			/* none */

			/* Entry Error */
			RunStartStopReq = 0U;

			State = State_Error;			/* Destination:Error */
		}
		else if((85.0F <= Vac_Vrms)&&(Vac_Vrms <= 270.0F))	/* Event:AC Power Source connected */
		{
			/* Exit Stop */
			/* none */

			/* Entry Precharge */
			State = State_Precharge;	/* Destination:Precharge */
		}
		else
		{
			/* None */
		}
		break;

		/* 2:Wait */
		case State_Wait:
		if(ErrorStateLatch != 0U)	/* Event:Error detection */
		{
			/* Exit Wait */
			FlagEntryWait  = 0U;
			TimerEntryWait = 0U;

			/* Entry Error */
			RunStartStopReq = 0U;

			State = State_Error;			/* Destination:Error */

		}
		else if((Vac_Vrms < 85.0F) || ((Vac_Vrms - Vac_Vrms0) < -20.0F) || (270.0F < Vac_Vrms))	/* Event:AC Power Source shutdown */
		{
			/* Exit Wait */
			FlagEntryWait = 0U;
			TimerEntryWait = 0U;

			/* Entry Stop */
			State = State_Stop;				/* Destination:Stop */
		}
		else if((FlagStateMainRelay == 1U) && (TimerEntryWait > 599U) && (RunStartStopReq != 0U))	/* Event:Run request */
		{
			/* Exit Wait */
			FlagEntryWait = 0U;
			TimerEntryWait = 0U;

			/* Entry Run */
			Start();

			State = State_Run;				/* Destination:Run */
		}
		else
		{
			/* None */
		}

		break;

		/* 3:Run */
		case State_Run:
		if(ErrorStateLatch != 0)	/* Event:Error detection */
		{
			/* Exit Run */
			Stop();

			/* Entry Error */
			RunStartStopReq = 0U;

			State = State_Error;			/* Destination:Error */
		}
		else if((Vac_Vrms < 85.0F) || ((Vac_Vrms - Vac_Vrms0) < -20.0F) || (270.0F < Vac_Vrms))	/* Event:AC Power Source shutdown */
		{
			/* Exit Run */
			Stop();

			/* Entry Stop */

			State = State_Stop;				/* Destination:Stop */
		}
		else if(RunStartStopReq == 0U)	/* Event:Stop Request */
		{
			/* Exit Run */
			Stop();

			/* Entry Wait */
			FlagEntryWait = 1U;

			State = State_Wait;				/* Destination:Wait */
		}
		else
		{
			/* None */
		}

		break;
		/* 4:Precharge */
		case State_Precharge:
		if(ErrorStateLatch != 0)	/* Event:Error detection */
		{
			/* Exit Precharge */
			/* None */

			/* Entry Error */
			RunStartStopReq = 0U;

			State = State_Error;			/* Destination:Error */
		}
		else if((Vac_Vrms < 85.0F) || ((Vac_Vrms - Vac_Vrms0) < -20.0F) || (270.0F < Vac_Vrms))	/* Event:AC Power Source shutdown */
		{
			/* Exit Precharge */
			/* None */

			/* Entry Stop */
			State = State_Stop;				/* Destination:Stop */
		}
		else if(Vdc >= (Vac_Vrms * 1.34F) - 0.0F)	/* Event:Precharge is finished (1.34 = sqrt2*0.95) */
		{
			/* Exit Precharge */
			/* None */

			/* Entry Wait */
			FlagEntryWait = 1U;

			State = State_Wait;				/* Destination:Wait */

		}
		else
		{
			/* None */
		}
		break;

		/* 5:Init */
		case State_Init:
		if(FlagCalibEnd != 0U){	/* Event:Initialization is completed */
			/* Exit Init */
			SetBitPIO(3U);				/* Set /RESET to High */

			/* Entry Stop */

			State = State_Stop;				/* Destination:Stop */

		}
		break;
		default:
		State = State_Error;
		break;
	}
}

/*=======================================================*/
/**
* @brief   Pre-process before running.
* @return  None.
*/
void Start(void)
{
	/* Initialization of all integral terms */
	IacIElement = 0.0F;
	IacPIElement = 0.0F;
	VdcIElement = 0.0F;
	VdcPIElement = 0.0F;

	if(150.0F < Vac_Vrms)
	{
	    IacKpComp = 4.0F;
	}
	else
	{
	    IacKpComp = 0.0F;
	}

	/* Set soft-start parameters */
	VdcRef = Vdc;
	DeltaVdcRefRamp = (ConstVdcRef - VdcRef) / 250.0F;
	FlagVdcRefRamp = 1U;

	if(VdcLPF > 0.1F)
	{
		VdcRev = 1.0F / VdcLPF;
	}
	else
	{
		VdcRev = 10.0;
	}

	SetDutyPWM1(-1.0F);	/* Set modulation rate */
	SetDutyPWM2( 1.0F);	/* Set modulation rate */
	StartPWM(0x005UL, 0x005UL);	/* Start PWM */
}

/*=======================================================*/
/**
* @brief   Exit process at stop running.
* @return  None.
*/
void Stop(void)
{
	/* reset the soft-start parameter */
	FlagVdcRefRamp = 0U;
	if(ErrorStateLatch == 0U)	/* No error is detected */
	{
		OutputPIO(0x0000UL);     	/* Release the main relay */
		FlagStateMainRelay = 0U;
	}

	StopPWM(0x005UL, 0x005UL);	/* Stop PWM */
}

/*=======================================================*/
/**
* @brief   Process and calculaiton concerning AC.
* @return  None.
*/
#pragma CODE_SECTION(ProcVac, "ram_code")
void ProcVac(void)
{
	FLOAT32 tmp;
	UINT16 FlagDetected = 0U;

	if(VacLPF > 0.0F)		/* Vac positive/negative detector */
	{
		SignVac = 1;
	}
	else
	{
		SignVac = -1;
	}

	/* Detect Vac zero cross */
	if((1 == SignVac)&&(-1 == SignVac0)&&(PeriodCountMin < PeriodCount)) /* in the operational range */
	{
		if(PeriodCount > 1U)
		{
			FreqSys = 1.0F / ((FLOAT32)PeriodCount * ControlPeriod);	/* AC frequency [Hz] */
			RefDeltaPhase_rad = Math2Pi / (FLOAT32)PeriodCount;
			RefPhase_rad = 0.0F;
		}
		else
		{
			FreqSys = 1.0F;
			RefDeltaPhase_rad = Math2Pi;
		}

		/* Calculating RMS voltage and current */
		Vac_Vrms0 = Vac_Vrms;
		tmp = hssqrt(FreqSys * VacSum_V * ControlPeriod) * 102.0F;
		Vac_Vrms = tmp;
		Iac_Arms = hssqrt(FreqSys * IacSum_A * ControlPeriod) * 10.0F;

		Vdc_MAVE = Vdc_MSum_V / (float)PeriodCount;
		Vdc_MSum_V = 0.0F;

		VacSum_V = 0.0F;
		IacSum_A = 0.0F;
//		PeriodCount  = 0U;
		FlagDetected = 1U;
	}
	else if(PeriodCount > PeriodCountMax)	/* off the lower limit */
	{
		Vac_Vrms0 = Vac_Vrms;
		Vac_Vrms  = 0.0F;
		VacSum_V  = 0.0F;
		IacRef    = 0.0F;
		Iac_Arms  = 0.0F;
		IacSum_A  = 0.0F;
		PeriodCount = 0U;
		FlagZeroCross = 0U;
		VdcSum_V = 0.0F;
		Vdc_MSum_V = 0.0F;
	}
	else
	{
		/* None */
	}

	if((-1 == (SignVac * SignVac0)) &&
	((PeriodCount > ((UINT16)PeriodCountMin >> 1U)) || (FlagDetected == 1U)))
	{ /* to prevent from multiple detections of nearby zerocrosses induced by noise */
		FlagZeroCross = 1U;

		if(1U == FlagDetected)
		{
			VdcAVE = VdcSum_V / (FLOAT32)(PeriodCount >> 1U);
			PeriodCount = 0U;
			FlagDetected = 0U;
		}
		else
		{
			VdcAVE = VdcSum_V / (FLOAT32)PeriodCount;
		}
		CtrlVdc();
		BalanceVdc();
		VdcSum_V = 0.0F;
	}
	else
	{
		FlagZeroCross = 0U;
	}

	SignVac0 = SignVac;		/* Save the previous sign of Vac */
	PeriodCount++;
	VacSum_V += Vac * Vac * 0.01F * 0.01F;
	IacSum_A += Iac * Iac * 0.1F * 0.1F;
	VdcSum_V += Vdc;
	Vdc_MSum_V += Vdc_M;
}

/*=======================================================*/
/**
* @brief   PLL(Phase Locked Loop).
* @return  None.
*/
#pragma CODE_SECTION(CalcPhase, "ram_code")
void CalcPhase(void)
{
	RefPhase_rad += RefDeltaPhase_rad;

	/* Folding */
	if(RefPhase_rad > Math2Pi)
	{
		RefPhase_rad -= Math2Pi;
	}

	Phase_rad = RefPhase_rad;
}

/*=======================================================*/
/**
* @brief   Calibration of AD offsets.
* @return  None.
*/
void CalibADOffset(void)
{
	CntOffset++;
	OffsetVac0 += -(Vac * 0.01);
	OffsetIac0 += -(Iac * 0.01);

	if(CntOffset > 100U)		/* 100 samples */
	{
		CntOffset = 0U;
		OffsetVac = OffsetVac0;
		OffsetIac = OffsetIac0;
		FlagCalibEnd = 1U;
	}
}
/*=======================================================*/
/**
* @brief   DC voltage controller.
* @return  None.
*/
void CtrlVdc(void)
{
	if(1U == FlagVdcRefRamp)
	{
		VdcKp = VDCKP / 3.0F;
	}
	else
	{
		VdcKp = VDCKP;
	}

	if(State == State_Run)
	{
		ErrVdc = VdcRef - VdcLPF;
		VdcPElement = ErrVdc * VdcKp;	/* Proportional term */
		VdcIElement = hslimit(ErrVdc * VdcKi + VdcIElement, IacLimitHi, 0.0F);	/* Integral term */
		VdcPIElement = hslimit(VdcPElement + VdcIElement, IacLimitHi, 0.0F);
		IacRef = VdcPIElement;				/* Iac reference */
		if(ErrVdc > 0.0F)
		{
			VdcIElement = hslimit(IacRef - VdcPElement, IacLimitHi, 0.0F);	/* Anti-windup */
		}
	}
	else
	{
		VdcIElement = 0.0F;
		IacRef      = 0.0F;
	}
}
/*=======================================================*/
/**
* @brief   DC voltage balancer.
* @return  None.
*/
void BalanceVdc(void)
{
	if(State == State_Run)
	{
		ErrVdc_M = (VdcAVE * 0.5F) - Vdc_MAVE;/* Error */
		Vdc_MPElement = ErrVdc_M * Vdc_MKp;	/* Proportional term */
		Vdc_MIElement = hslimit(ErrVdc_M * Vdc_MKi + Vdc_MIElement, IacLimitHi/2.0F, -IacLimitHi/2.0F);  /* Integral term */
		Vdc_MPIElement = hslimit(Vdc_MPElement + Vdc_MIElement, IacLimitHi/2.0F, -IacLimitHi/2.0F);
		if(VacLPF < 0.0F)
		{
		  IacRef = IacRef + Vdc_MPIElement;	/* Adding to Iac reference */
		}
		else
		{
		  IacRef = IacRef - Vdc_MPIElement;	/* Adding to Iac reference */
		}
		IacRef = hslimit(IacRef, IacLimitHi, 0.0F);
	}
	else{ Vdc_MIElement = 0.0F; }
}/*=======================================================*/
/**
* @brief   Getting temperature on the heatsink.
* @return  None.
*/
void GetTh(void)
{
//	FLOAT32 vth;
	vth = ReadADCA(3U);
	Th  = (4.98F - vth) * 28.6F;	/* error range +-1.1 deg (more than 15deg)*/
}
/*=======================================================*/
/**
* @brief   AC current controller.
* @return  None.
*/
#pragma CODE_SECTION(CtrlIac, "ram_code")
void CtrlIac(void)
{
	FLOAT32 IacRef_ins;
	FLOAT32 vac;
	FLOAT32 hilimit;
	static UINT16 FlagVdcHigh;

	if(1U == FlagZeroCross)
	{
		IacIElement = 0.0F;	/* clear the integral term */
	}
	else
	{
	    /* none */
	}
	/* current controller */
	if(State == State_Run)
	{
		IacRef_ins = IacRef * hssin(Phase_rad);
		if(Phase_rad <= MATHPI)
		{
			ErrIac = IacLPF + IacRef_ins;		/* Error at Iac >=0 */
		}
		else
		{
			ErrIac = -IacRef_ins - IacLPF;	/* Error at Iac <0 */
		}

		if(VdcAVE <= ConstVdcRef)					/* normal */
		{
			vac = 1.414F * Vac_Vrms;
			FlagVdcHigh = 0U;
		}
		else if((((ConstVdcRef + 3.0F) < VdcAVE)&&(Iac_Arms < 0.5F)) || (FlagVdcHigh == 1U))	/* no/light load and higher Vdc */
		{
			vac = 1.414F * Vac_Vrms * 2.0F;	/* mask the FF control during Vdc high */
			FlagVdcHigh = 1U;
		}
		else															/* normal */
		{
			vac = 1.414F * Vac_Vrms;
			FlagVdcHigh = 0U;
		}

		IacPElement = ErrIac * IacKp;       /* Proportional term */
		if(((MRate2Out < MRateThrIacICtrl) && (MRate1Out < -0.9F)) || ((0.9F < MRate2Out) && (-MRateThrIacICtrl < MRate1Out)))
		{
			IacIElement = hslimit(ErrIac * IacKi + IacIElement, Vdc, -Vdc); /* Integral term */
		}
		else
		{
		  /* none */
		}
		IacPIElement = IacPElement + IacIElement;

		hilimit =  VdcLPF;
		if(Phase_rad <= MATHPI)
		{
			VacOut =  hslimit( vac * hssin(Phase_rad) - IacPIElement, hilimit, 0.01F);
		}
		else
		{
			VacOut = -hslimit(-vac * hssin(Phase_rad) - IacPIElement, hilimit, 0.01F);
		}

	}
	else
	{
		IacIElement = 0.0F;
		VacOut = 0.0F;
	}

}

/*=======================================================*/
/*=======================================================*/
/**
* @brief   calculation of modulation rate.
* @return  None.
*/
#pragma CODE_SECTION(CalcMrate, "ram_code")
void CalcMrate(void)
{
	FLOAT32 temp;
	FLOAT32 temp2;
	FLOAT32 temp3;
	if(VdcLPF > 0.1F)
	{
	  VdcRev = 1.0F / Vdc;
	}
	else
	{
		VdcRev = 10.0;
	}

	temp = VacOut * VdcRev;
	temp2= hslimit(VacOut * VdcRev, 1.0F, -1.0F);
	if(VdcAVE <= ConstVdcRef)
	{
		temp3 = SlowTurnOnComp;
	}
	else
	{
		temp3 = 0.0F;
	}

	if(0.5F < temp)  /* Vdc/2 < Vac */
	{
		if((IacRef < 0.1F) && ((ConstVdcRef + VdcCtrlUpperLimit) < VdcAVE))
		{
			MRate2Out = -1.0F;
		}
		else
		{
			MRate2Out = (2.0F * (1.0F - temp2) * 2.0F) - 1.0F;
				IacKp = IACKP2;
		}
		MRate1Out = -1.0F;
	}
	else if(temp < -0.5F) /* Vac < -Vdc/2 */
	{
	   if ((IacRef < 0.1F) && ((ConstVdcRef + VdcCtrlUpperLimit) < VdcAVE))
		{
	       MRate1Out =  1.0F;
		}
		else
		{
			MRate1Out = (((-2.0F * temp2) - 1.0F)* 2.0F) - 1.0F;
				IacKp = IACKP2;
		}
	MRate2Out = 1.0F;
	}
	else if(0.0F <= Vac) /* 0 <= Vac <= Vdc/2 */
	{
		if(0.0F < MRate2Out)						/* near Vdc/2 or pos zerocrosses */
		{
			MRate1Out = ((1.0F - (temp2 * 2.0F))* 2.0F) - 1.0F;
			IacKp = IACKP1 + IacKpComp;
			if(MRate1Out < -0.9F) /* avoid a narrow pulse near Vdc/2 */
			{
				MRate2Out =  1.0F;
				MRate1Out = -1.0F;
			}
			else											/* compensation of slow turnon of Q3 */
			{
				if(MRate1Out < -0.4F)
				{
					MRate1Out = MRate1Out + (2.0F * temp3 * (MRate1Out + 0.9F));
				}
				else
				{
					MRate1Out = MRate1Out + temp3;
				}
				MRate2Out = 1.0F;
			}
		}
		else								/* just crossed the zero and put deadtime between Q3 and Q5 */
		{
			MRate1Out = -1.0F;
			MRate2Out =  1.0F;
		}
	}
	else				/* -Vdc/2 <= Vac < 0 */
	{
		if(MRate1Out < 0.0F) 				/* near Vdc/2 or neg zerocrosses */
		{
			MRate2Out = ((-2.0F * temp2) * 2.0F) - 1.0F;
			IacKp = IACKP1 + IacKpComp;
			if(0.9F < MRate2Out ) /* avoid a narrow pulse near Vdc/2 */
			{
				MRate1Out = -1.0F;
				MRate2Out =  1.0F;
			}
			else										/* compensation of slow tunron of Q5 */
			{
				if(0.4F < MRate2Out)
				{
					MRate2Out = MRate2Out - (2.0F * temp3 * (0.9F - MRate2Out));
				}
				else
				{
					MRate2Out = MRate2Out - temp3;
				}
				MRate1Out = -1.0F;
			}
		}
		else								/* just crossed the zero and put deadtime between Q3 and Q5 */
		{
			MRate2Out =  1.0F;
			MRate1Out = -1.0F;
		}
	}

  MRate1Out = hslimit(MRate1Out, MRATELIMIT,       -1.0F);
  MRate2Out = hslimit(MRate2Out,       1.0F, -MRATELIMIT);
}

/*=======================================================*/
/**
* @brief   PWM Interrupt Function.
* @return  None.
*/
#pragma CODE_SECTION(PWMIntFunc, "ram_code")
#pragma	INTERRUPT(PWMIntFunc)
void PWMIntFunc(void)
{
#if 0
	t1 = EPwm1Regs.TBCTR;
#endif
    HSI_IERtable IntEnable;     /* New interrupt enable tables */
    HSI_IERtable Backup;        /* Pre-configuration interrupt enable table */

    AcknowledgePieINT();        /* Clear the PIE interrupt acknowledge flag */

	/* Interrupt enable state setting (for multiple interrupts) */
	IntEnable.pieier[0]  = 0x0000U;
	IntEnable.pieier[1]  = 0x0004U;	/* Enable PWM3TZ interrupt */
	IntEnable.pieier[2]  = 0x0000U;
	IntEnable.pieier[3]  = 0x0000U;
	IntEnable.pieier[4]  = 0x0000U;
	IntEnable.pieier[5]  = 0x0000U;
	IntEnable.pieier[6]  = 0x0000U;
	IntEnable.pieier[7]  = 0x0000U;
	IntEnable.pieier[8]  = 0x0000U;
	IntEnable.pieier[9]  = 0x0000U;
	IntEnable.pieier[10] = 0x0000U;
	IntEnable.pieier[11] = 0x0000U;

	UpdateIerINT(&IntEnable, &Backup);/* Update interrupt enable state (for multiple interrupts) */
	EnableGlobalInterrupt();					/* Enable Global Interrupt */

	  Vac = ReadADCA(0U) + OffsetVac;	/* Acquire sensor value : Vac */
	  Iac = ReadADCB(0U) + OffsetIac;	/* Acquire sensor value : Iac */
	  Vdc = ReadADCD(0U) + OffsetVdc;	/* Acquire sensor value : Vdc */
	Vdc_M = ReadADCC(0U) + OffsetVdc_M;/* Acquire sensor value : Vdc_M */
  ProcVac();				    						/* Calculating RMS voltage */

	/* Apply LPF */
	  IacLPF = (IacLPFCoeff * IacLPF)   + ((1.0F - IacLPFCoeff) * Iac);
	  VacLPF = (VacLPFCoeff * VacLPF)   + ((1.0F - VacLPFCoeff) * Vac);
	Vdc_MLPF = (VdcLPFCoeff * Vdc_MLPF) + ((1.0F - VdcLPFCoeff) * Vdc_M);
	  VdcLPF = (VdcLPFCoeff * VdcLPF)   + ((1.0F - VdcLPFCoeff) * Vdc);

	CalcPhase();					/* PLL */

	if(FlagTZ == 0U)
	{
		if(State == State_Run)
		{
			CtrlIac();				/* AC current controller */
			CalcMrate();			/* Calculation of modulation rate */

			SetDutyPWM1(MRate1Out);	 /* Set modulation rate */
			SetDutyPWM2(MRate2Out);	 /* Set modulation rate */
		}
	}
	else
	{
		if(FlagWaitGB != 0U){ StopPWM(0x005UL, 0x005UL);}
		else                { FlagWaitGB = 1U; }
	}

#if 0 /* to measure a operation time of pwm */
	    t2 = EPwm1Regs.TBCTR;
	    if(t2 > t1)
	        t_pwm = ((FLOAT32)t2 - (FLOAT32)t1)*0.01F;
	    else
	        t_pwm = (((FLOAT32)t2 + (FLOAT32)4999U) - (FLOAT32)t1) * 0.01F;

	    if(tmax < t_pwm)    tmax = t_pwm;
	    else{}
#endif

	DisableGlobalInterrupt();	/* Disable Global Interrupt */
	RestoreIerINT(&Backup);		/* Restore interrupt enabled state */
	ClearInterruptFlagPWM1();	/* Clear PWM1 interrupt flag */
}

#pragma CODE_SECTION(TZIntFunc, "ram_code")
#pragma INTERRUPT(TZIntFunc)
void TZIntFunc(void)
{
	DisableTripInterruptPWM3();
	AcknowledgePieINT();
	StopPWM(0x031UL, 0x034UL);	/* Stop PWM1P and 2N */
	FlagTZ = 1U;
	ClearTripInterruptFlagPWM3();

}
/*=======================================================*/
/**
* @brief   Timer0 Interrupt Function.
* @return  None.
*/
#pragma	INTERRUPT(Timer0IntFunc)
void Timer0IntFunc(void)
{

	HSI_IERtable IntEnable;	/* New interrupt enable tables */
	HSI_IERtable Backup;		/* Pre-configuration interrupt enable table */

	AcknowledgePieINT();		/* Clear the PIE interrupt acknowledge flag */

	/* Interrupt enable state setting (for multiple interrupts) */
	IntEnable.pieier[0]  = 0x0000U;
	IntEnable.pieier[1]  = 0x0004U;		/* Enable PWM3TZ interrupt */
	IntEnable.pieier[2]  = 0x0001U;		/* Enable PWM1   interrupt */
	IntEnable.pieier[3]  = 0x0000U;
	IntEnable.pieier[4]  = 0x0000U;
	IntEnable.pieier[5]  = 0x0000U;
	IntEnable.pieier[6]  = 0x0000U;
	IntEnable.pieier[7]  = 0x0000U;
	IntEnable.pieier[8]  = 0x0000U;
	IntEnable.pieier[9]  = 0x0000U;
	IntEnable.pieier[10] = 0x0000U;
	IntEnable.pieier[11] = 0x0000U;
	IntEnable.ier = 0x4000U;			/* Enable Debugging communication interrupt */

	UpdateIerINT(&IntEnable, &Backup);	/* Update interrupt enable state (for multiple interrupts) */
	EnableGlobalInterrupt();			/* Enable Global Interrupt */

	/* 1msec measurement */
	if(FlagEntryWait == 1U)
	{
		if(TimerEntryWait > 599U)
		{
			FlagEntryWait = 0U;
		}
		else
		{
			TimerEntryWait++;
		}
	}
	/* Delay turning on the relay by 0.5sec */
	if(TimerEntryWait > 499U)
	{
	   OutputPIO(DO_MAIN);				/* Make the main relay turn on */
	   FlagStateMainRelay = 1U;
	}

	/* VdcRef for soft-start */
	if(FlagVdcRefRamp == 1U)
	{
		if((hsfabs(VdcRef - ConstVdcRef) < 1.0F)&&(VdcLPF > 380.0F))
		{
			FlagVdcRefRamp = 0U;
			VdcRef = ConstVdcRef;
		}
		else
		{
			VdcRef += DeltaVdcRefRamp;	/* 250msec */
			if(VdcRef > ConstVdcRef){ VdcRef = ConstVdcRef; }
		}
	}

	GetTh();						/* Getting temp on the heatsink */
	JudgeError();				/* Judge error conditions */
	SetState();					/* State machine */

	/* Calculate LPF parameter */
	VacLPFCoeff = 1.0F / (1.0F + ((2.0F * MATHPI * VacFc * INTERRUPT_MASK) / CARRIERFREQ_HZ));
	IacLPFCoeff = 1.0F / (1.0F + ((2.0F * MATHPI * IacFc * INTERRUPT_MASK) / CARRIERFREQ_HZ));
	VdcLPFCoeff = 1.0F / (1.0F + ((2.0F * MATHPI * VdcFc * INTERRUPT_MASK) / CARRIERFREQ_HZ));

	/* Calculate integral coefficient */
	IacKi = (IacKp * INTERRUPT_MASK) / (IacTi * 0.001F * CARRIERFREQ_HZ);
	VdcKi = (VdcKp * (1000.0F)) / (VdcTi * FreqSys);
	Vdc_MKi = (Vdc_MKp * (1000.0F)) / (Vdc_MTi * FreqSys);

	if(FlagCalibEnd == 0U)		/* Check calibration ended */
	{
		CalibADOffset();
	}

	DisableGlobalInterrupt();	/* Disable Global Interrupt */
	RestoreIerINT(&Backup);		/* Restore interrupt enabled state */
	ClearInterruptFlagTIMER0();	/* Clear Timer0 interrupt flag */
}
