/**
 ****************************************************************************
 * @file     mcuip_drv.c
 * @brief    MCU IP driver for A-VE
 * @version  V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2021
 *****************************************************************************
 */
#include "ipdefine.h"
#include "mcuip_drv.h"


/**
 *********************************************************************************************
 * @brief		Get EMGstatus
 *
 * @param		const ipdrv_t* const _ipdrv	Select the IP table.
 *
 * @return		emg_status_e				EMG status
 *********************************************************************************************
 */
emg_status_e	PMD_GetEMG_Status(const ipdrv_t* const _ipdrv)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;

	return ((emg_status_e)(*((__I uint32_t*)BITBAND_PERI(&PMDx->EMGSTA, B_EMGST))));

}

/**
 *********************************************************************************************
 * @brief		Release EMG protection.
 *
 * @param		const ipdrv_t* const _ipdrv	Select the IP table.
 *
 * @return		none
 *********************************************************************************************
 */
void	PMD_ReleaseEMG_Protection(const ipdrv_t* const _ipdrv)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	emg_status_e	emgst;

	emgst = PMD_GetEMG_Status(_ipdrv);

	if (emgst == cEMGProtected)
	{
		if ((*((__I uint32_t*)BITBAND_PERI(&PMDx->EMGSTA, B_EMGI))) == 1)
		{
			TSB_VE_TypeDef* const VEx = _ipdrv->VEx;

			if (PMDx == TSB_PMD1)
			{
				VEx->EMGRS1 = 1;
			}
		}
	}

}

/**
 *********************************************************************************************
 * @brief		VE start
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_Start(const ipdrv_t* const _ipdrv)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;

	if (PMDx == TSB_PMD1)
	{
		VEx->CPURUNTRG = VE_START_CH1;
	}

}

/**
 *********************************************************************************************
 * @brief		VE_GetTask
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 *
 * @return		uint32_t						Task No.
 *********************************************************************************************
 */
uint32_t	VE_GetTask(const ipdrv_t* const _ipdrv)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;

	if (PMDx == TSB_PMD1)
	{
		return((VEx->SCHTASKRUN >> 6U) & 0x0000000f);
	}
	else{
		return(0);
	}
}

/**
 *********************************************************************************************
 * @brief		Get Phase Current from VE register.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		q15_t*					_ia		U-phase current value store address.
 * @param		q15_t*					_ib		V-phase current value store address.
 * @param		q15_t*					_ic		W-phase current value store address.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_GetPhaseCurrent(const ipdrv_t* const _ipdrv, q15_t* _ia, q15_t* _ib, q15_t* _ic)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;
	uint8_t shunt = c3shunt;
	q15_t	ia, ib, ic;

	{
	  	uint32_t reg = 0;
		if (PMDx == TSB_PMD1)
		{
			reg = VEx->FMODE1;
		}
		if (((reg >> 2U) & 0x02) == 0x02)
		{
			shunt = c1shunt;
		}
	}

	if (shunt == c3shunt)
	{
		q15_t	ia_adc, ib_adc, ic_adc;
		q15_t	iao, ibo, ico;
		int		sector;

		/**** 3 shunt ****/
		if (PMDx == TSB_PMD1)
		{
			iao = VEx->IAO1;
			ibo = VEx->IBO1;
			ico = VEx->ICO1;
			ia_adc = VEx->IAADC1;
			ib_adc = VEx->IBADC1;
			ic_adc = VEx->ICADC1;
			sector = VEx->SECTOR1 / 2;
		}

		switch(sector){
		case 0:
		case 5:
			/* Sector 0,5 */
			ib = ibo - ib_adc;
			ic = ico - ic_adc;
			ia = (-ib - ic);
			break;
		case 1:
		case 2:
			/* Sector1,2 */
			ic = (ico - ic_adc);
			ia = (iao - ia_adc);
			ib = (-ic - ia);
			break;
		case 3:
		case 4:
			/* Sector3,4 */
			ia = (iao - ia_adc);
			ib = (ibo - ib_adc);
			ic = (-ia - ib);
			break;
		default:
			ia = 0x7fff;
			ib = 0x7fff;
			ic = 0x7fff;
			break;
		}
	}
	else
	{
		/**** 1 shunt ****/
		if (PMDx == TSB_PMD1)
		{
			ADC_Result adc0, adc1;

			adc0 = ADC_GetConvertResult(TSB_ADB, ADC_REG0);
			adc1 = ADC_GetConvertResult(TSB_ADB, ADC_REG1);
			ia = VEx->IAO1 - (adc0.All & 0xFFF0);
			ib = VEx->IAO1 - (adc1.All & 0xFFF0);
			ic = 0;
		}
	}

	*_ia = ia;
	*_ib = ib;
	*_ic = ic;
}

/**
 *********************************************************************************************
 * @brief		Get current adc data from VE register.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		uint32_t*				_adc_ia	U-phase current ad value store address.
 * @param		uint32_t*				_adc_ib	V-phase current ad value store address.
 * @param		uint32_t*				_adc_ic	W-phase current ad value store address.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_GetCurrentAdcData(const ipdrv_t* const _ipdrv, uint32_t* _adc_ia, uint32_t* _adc_ib, uint32_t* _adc_ic)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;

	if (PMDx == TSB_PMD1)
	{
		*_adc_ia = VEx->IAADC1;
		*_adc_ib = VEx->IBADC1;
		*_adc_ic = VEx->ICADC1;
	}

}

/**
 *********************************************************************************************
 * @brief		Get data from VE registers.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_GetdataFromVEreg(const ipdrv_t* const _ipdrv, vector_t* const _motor)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;

	if (PMDx == TSB_PMD1)
	{
		_motor->drv.Vdc = VEx->VDC1;								/* Read Vdc */
		_motor->drv.Vd.word = VEx->VD1;								/* Read Vd */
		_motor->drv.Vq.word = VEx->VQ1;								/* Read Vd */

		if (_motor->drv.idetect_error == 0)
		{
			_motor->drv.Id = (q15_t)(VEx->ID1 >> 16);				/* Read Id */
			_motor->drv.Iq = (q15_t)(VEx->IQ1 >> 16);				/* Read Iq */
		}
		else
		{
			VEx->ID1 = (q31_t)_motor->drv.Id << 16;					/* Set Id */
			VEx->IQ1 = (q31_t)_motor->drv.Iq << 16;					/* Set Iq */
		}
	}

}

/**
 *********************************************************************************************
 * @brief		VE_GetPWM_Period
 *
 * @param		TSB_VE_TypeDef* const	VEx		Set the VE address.
 *
 * @return		uint32_t						PWM period
 *********************************************************************************************
 */
uint32_t	VE_GetPWM_Period(const ipdrv_t* const _ipdrv)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;

	if (PMDx == TSB_PMD1)
	{
		return(VEx->MDPRD1);
	}
	else{
		return(0);
	}
}

/**
 *********************************************************************************************
 * @brief		VE_GetPWM_DutyU
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 *
 * @return		uint32_t						U Phase duty.
 *********************************************************************************************
 */
uint32_t	VE_GetPWM_DutyU(const ipdrv_t* const _ipdrv)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;

	if (PMDx == TSB_PMD1)
	{
		return(VEx->VCMPU1);
	}
	else{
		return(0);
	}
}

/**
 *********************************************************************************************
 * @brief		VE_GetPWM_DutyV
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 *
 * @return		uint32_t						V Phase duty.
 *********************************************************************************************
 */
uint32_t	VE_GetPWM_DutyV(const ipdrv_t* const _ipdrv)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;

	if (PMDx == TSB_PMD1)
	{
		return(VEx->VCMPV1);
	}
	else{
		return(0);
	}
}

/**
 *********************************************************************************************
 * @brief		VE_GetPWM_DutyW
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 *
 * @return		uint32_t						W Phase duty.
 *********************************************************************************************
 */
uint32_t	VE_GetPWM_DutyW(const ipdrv_t* const _ipdrv)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;

	if (PMDx == TSB_PMD1)
	{
		return(VEx->VCMPW1);
	}
	else{
		return(0);
	}
}

/**
 *********************************************************************************************
 * @brief		Get medium duty from VE registers.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 *
 * @return		uint32_t						Medium duty value.
 *********************************************************************************************
 */
uint32_t	VE_GetPWM_DutyMed(const ipdrv_t* const _ipdrv)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;
	uint32_t	a,b,c,ret;

	if (PMDx == TSB_PMD1)
	{
		a = VEx->VCMPU1;
		b = VEx->VCMPV1;
		c = VEx->VCMPW1;
	}

	if (a >= b)
	{
		ret = (c < b) ? b : (c > a) ? a : c;
	}
	else
	{
		ret = (c < a) ? a : (c > b) ? b : c;
	}

	return ret;

}

/**
 *********************************************************************************************
 * @brief		VE_GetPWM_Modulation
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 *
 * @return		int								PWM modulation 0:3Phase, 1:2Phase
 *********************************************************************************************
 */
int	VE_GetPWM_Modulation(const ipdrv_t* const _ipdrv)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;
	int	ret = 0;

	if (PMDx == TSB_PMD1)
	{
		if ((VEx->FMODE1 & (1 << B_C2PEN)) != 0)
		{
			ret = 1;
		}
	}

	return(ret);

}

/**
 *********************************************************************************************
 * @brief		VE_GetSector
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 *
 * @return		uint32_t						Sector
 *********************************************************************************************
 */
uint32_t	VE_GetSector(const ipdrv_t* const _ipdrv)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;

	if (PMDx == TSB_PMD1)
	{
		return(VEx->SECTOR1);
	}
	else{
		return(0);
	}
}

/**
 *********************************************************************************************
 * @brief		VE_GetShiftPWMState
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 *
 * @return		int								PWM Shift state 0:Normal PWM, 1;Shift PWM
 *********************************************************************************************
 */
int	VE_GetShiftPWMState(const ipdrv_t* const _ipdrv)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;
	int	ret = 0;

	if (PMDx == TSB_PMD1)
	{
		if ((VEx->MCTLF1 & (1 << B_LAVF)) != 0)
		{
			ret = 1;
		}
	}

	return(ret);
}

/**
 *********************************************************************************************
 * @brief		VE_GetOutputMode
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 *
 * @return		int	
 *********************************************************************************************
 */
int	VE_GetOutputMode(const ipdrv_t* const _ipdrv)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;

	if (PMDx == TSB_PMD1)
	{
		return(((VEx->MODE1 >> 2U) & 0x00000003));
	}
	else{
		return(0);
	}
}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Stop stage.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_SetdataToVEreg_Stop(const ipdrv_t* const _ipdrv, const vector_t* const _motor)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;
	uint32_t	work;

	if (PMDx == TSB_PMD1)
	{
		work = VEx->TASKAPP;
		work &= MASK_TAPP_CH1;
		VEx->TASKAPP = (work | SET_TAPP_CH1(TAPP_OUT_CON));			/* VE start from output control */
		work = VEx->ACTSCH;
		work &= MASK_ACTSCH_CH1;
		VEx->ACTSCH = (work | SET_ACTSCH_CH1(SCH_9));				/* Schedule 9 */

		VEx->MODE1 = VEMODE_STOP;									/* Output OFF */

		VEx->ID1 = 0;
		VEx->IQ1 = 0;
		VEx->VD1 = 0;
		VEx->VQ1 = 0;
		VEx->VDIH1 = 0;												/* Clear Current PI Control Integral Term Registers */
		VEx->VDILH1 = 0;
		VEx->VQIH1 = 0;
		VEx->VQILH1 = 0;
		VEx->IDREF1 = 0;
		VEx->IQREF1 = 0;
		VEx->THETA1 = 0;
		VEx->OMEGA1 = 0;
		VEx->SECTOR1 = 0;
		VEx->SECTORM1 = 0;
		VEx->MINPLS1 = 0;
	}

}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Bootstrap stage.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_SetdataToVEreg_Bootstrap(const ipdrv_t* const _ipdrv, const vector_t* const _motor)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;
	uint32_t	work;

	if (PMDx == TSB_PMD1)
	{
		work = VEx->TASKAPP;
		work &= MASK_TAPP_CH1;
		VEx->TASKAPP = (work | SET_TAPP_CH1(TAPP_SINCOS));			/* VE start from SIN/COS */
		work = VEx->ACTSCH;
		work &= MASK_ACTSCH_CH1;
		VEx->ACTSCH = (work | SET_ACTSCH_CH1(SCH_4));				/* Schedule 4 */

		(*((__IO uint32_t*)BITBAND_PERI(&VEx->FMODE1, B_C2PEN))) = 1;	/* 2phase */
		VEx->MODE1 = VEMODE_BOOT;									/* Output ON */

		VEx->VD1 = 0;												/* Set 0. */
		VEx->VQ1 = 0;												/* Set 0. */
	}

}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Initposition stage for current control.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_SetdataToVEreg_Initposition_i(const ipdrv_t* const _ipdrv, const vector_t* const _motor)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;
	uint32_t	work;

	if (PMDx == TSB_PMD1)
	{
		work = VEx->TASKAPP;
		work &= MASK_TAPP_CH1;
		VEx->TASKAPP = (work | SET_TAPP_CH1(TAPP_CRT_CON));			/* VE start from current control */
		work = VEx->ACTSCH;
		work &= MASK_ACTSCH_CH1;
		VEx->ACTSCH = (work | SET_ACTSCH_CH1(SCH_1));				/* Schedule 1 */

		VEx->MODE1 = VEMODE_INITPOS;								/* Output ON */

		VEx->IDREF1 = _motor->drv.Id_ref;							/* Set Id */
		VEx->IQREF1 = _motor->drv.Iq_ref; 							/* Set Iq */
		VEx->THETA1 = _motor->drv.theta.half[1];
		VEx->OMEGA1 = _motor->drv.omega.half[1];
	}

}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Initposition stage for voltage control.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_SetdataToVEreg_Initposition_v(const ipdrv_t* const _ipdrv, const vector_t* const _motor)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;
	uint32_t	work;

	if (PMDx == TSB_PMD1)
	{
		work = VEx->TASKAPP;
		work &= MASK_TAPP_CH1;
		VEx->TASKAPP = (work | SET_TAPP_CH1(TAPP_SINCOS));			/* VE start from SIN/COS */
		work = VEx->ACTSCH;
		work &= MASK_ACTSCH_CH1;
		VEx->ACTSCH = (work | SET_ACTSCH_CH1(SCH_4));				/* Schedule 4 */

		VEx->MODE1 = VEMODE_INITPOS;								/* Output ON */

		VEx->VD1 = _motor->drv.Vd_out;								/* Vd */
		VEx->VQ1 = 0;
		VEx->THETA1 = _motor->drv.theta.half[1];
		VEx->OMEGA1 = _motor->drv.omega.half[1];
	}

}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Force stage for current control.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_SetdataToVEreg_Force_i(const ipdrv_t* const _ipdrv, const vector_t* const _motor)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;
	uint32_t	work;

	if (PMDx == TSB_PMD1)
	{
		work = VEx->TASKAPP;
		work &= MASK_TAPP_CH1;
		VEx->TASKAPP = (work | SET_TAPP_CH1(TAPP_CRT_CON));			/* VE start from current control */
		work = VEx->ACTSCH;
		work &= MASK_ACTSCH_CH1;
		VEx->ACTSCH = (work | SET_ACTSCH_CH1(SCH_1));				/* Schedule 1 */

		VEx->MODE1 = VEMODE_FORCE;									/* Output ON */

		VEx->IDREF1 = _motor->drv.Id_ref;							/* Set Id */
		VEx->IQREF1 = _motor->drv.Iq_ref; 							/* Set Iq */
		VEx->THETA1 = _motor->drv.theta.half[1];
		VEx->OMEGA1 = _motor->drv.omega.half[1];
	}

}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Force stage for voltage control.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_SetdataToVEreg_Force_v(const ipdrv_t* const _ipdrv, const vector_t* const _motor)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;
	uint32_t	work;

	if (PMDx == TSB_PMD1)
	{
		work = VEx->TASKAPP;
		work &= MASK_TAPP_CH1;
		VEx->TASKAPP = (work | SET_TAPP_CH1(TAPP_SINCOS));			/* VE start from SIN/COS */
		work = VEx->ACTSCH;
		work &= MASK_ACTSCH_CH1;
		VEx->ACTSCH = (work | SET_ACTSCH_CH1(SCH_4));				/* Schedule 4 */

		VEx->MODE1 = VEMODE_FORCE;									/* Output ON */

		VEx->VD1 = _motor->drv.Vd_out;								/* Vd */
		VEx->VDIH1 = _motor->drv.Vd_out;							/* Vd Integral value initialization */
		VEx->VQ1 = 0; 												/* Vq = 0 */
		VEx->THETA1 = _motor->drv.theta.half[1];					/* Theta set */
		VEx->OMEGA1 = _motor->drv.omega.half[1];					/* Omega set */
	}

}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Change_up stage.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_SetdataToVEreg_Change_up(const ipdrv_t* const _ipdrv, const vector_t* const _motor)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;
	uint32_t	work;

	if (PMDx == TSB_PMD1)
	{
		work = VEx->TASKAPP;
		work &= MASK_TAPP_CH1;
		VEx->TASKAPP = (work | SET_TAPP_CH1(TAPP_CRT_CON));			/* VE start from current control */
		work = VEx->ACTSCH;
		work &= MASK_ACTSCH_CH1;
		VEx->ACTSCH = (work | SET_ACTSCH_CH1(SCH_1));				/* Schedule 1 */

		VEx->MODE1 = VEMODE_CHGUP;									/* Output ON */

		VEx->IDREF1 = _motor->drv.Id_ref;							/* Set Id */
		VEx->IQREF1 = _motor->drv.Iq_ref; 							/* Set Iq */
		VEx->THETA1 = _motor->drv.theta.half[1];
		VEx->OMEGA1 = _motor->drv.omega.half[1];
	}

}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Steady_A stage.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_SetdataToVEreg_Steady_A(const ipdrv_t* const _ipdrv, const vector_t* const _motor)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;
	uint32_t	work;

	if (PMDx == TSB_PMD1)
	{
		work = VEx->TASKAPP;
		work &= MASK_TAPP_CH1;
		VEx->TASKAPP = (work | SET_TAPP_CH1(TAPP_CRT_CON));			/* VE start from current control */
		work = VEx->ACTSCH;
		work &= MASK_ACTSCH_CH1;
		VEx->ACTSCH = (work | SET_ACTSCH_CH1(SCH_1));				/* Schedule 1 */

		VEx->MODE1 = VEMODE_STEADY;									/* Output ON */

		VEx->IDREF1 = _motor->drv.Id_ref;							/* Id set */
		VEx->IQREF1 = _motor->drv.Iq_ref;							/* Iq set */
		VEx->THETA1 = _motor->drv.theta.half[1];					/* Theta set */
		VEx->OMEGA1 = _motor->drv.omega.half[1];					/* Omega set */
	}

}

/**
 *********************************************************************************************
 * @brief		Set data to VE registers in Emergency stage.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		vector_t* const			_motor	Structure address of vector control variable.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_SetdataToVEreg_Emergency(const ipdrv_t* const _ipdrv, const vector_t* const _motor)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;
	uint32_t	work;

	if (PMDx == TSB_PMD1)
	{
		work = VEx->TASKAPP;
		work &= MASK_TAPP_CH1;
		VEx->TASKAPP = (work | SET_TAPP_CH1(TAPP_OUT_CON));			/* VE start from output control */
		work = VEx->ACTSCH;
		work &= MASK_ACTSCH_CH1;
		VEx->ACTSCH = (work | SET_ACTSCH_CH1(SCH_9));				/* Schedule 9 */

		VEx->MODE1 = VEMODE_EMG;									/* Output OFF */
	}

}

/**
 *********************************************************************************************
 * @brief		Set zero current data to VE register.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		uint32_t				_z_ia	U Phase value.
 * @param		uint32_t				_z_ib	V Phase value.
 * @param		uint32_t				_z_ic	W Phase value.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_SetZeroCurrentData(const ipdrv_t* const _ipdrv, uint32_t _z_ia, uint32_t _z_ib, uint32_t _z_ic)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;

	if (PMDx == TSB_PMD1)
	{
		VEx->IAO1 = _z_ia;
		VEx->IBO1 = _z_ib;
		VEx->ICO1 = _z_ic;
	}

}

/**
 *********************************************************************************************
 * @brief		Set VEVDC reg.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		q15_t					_dat	Power supply voltage(Q15)
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_SetVDCreg(const ipdrv_t* const _ipdrv, q15_t _dat)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;

	if (PMDx == TSB_PMD1)
	{
		VEx->VDC1 = _dat;
	}

}

/**
 *********************************************************************************************
 * @brief		Set shift pwm_mode.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		uint8_t					_dat	Shift pwm mode.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_SetSpwmMode(const ipdrv_t* const _ipdrv, uint8_t _dat)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;

	if (PMDx == TSB_PMD1)
	{
		(*((__IO uint32_t*)BITBAND_PERI(&VEx->FMODE1, B_SPWMEN))) = _dat;
	}

}

/**
 *********************************************************************************************
 * @brief		Set modulation type.
 *
 * @param		const ipdrv_t* const 	_ipdrv	Select the IP table.
 * @param		uint8_t					_dat	modulation type.
 *
 * @return		none
 *********************************************************************************************
 */
void	VE_SetModulType(const ipdrv_t* const _ipdrv, uint8_t _dat)
{
	TSB_PMD_TypeDef* const PMDx = _ipdrv->PMDx;
	TSB_VE_TypeDef* const VEx = _ipdrv->VEx;

	if (PMDx == TSB_PMD1)
	{
		(*((__IO uint32_t*)BITBAND_PERI(&VEx->FMODE1, B_C2PEN))) = _dat;
	}

}

/**
 *********************************************************************************************
 * @brief		PMD IP Initialize
 *
 * @param		TSB_PMD_TypeDef* const	PMDx		Select the PMD channel.
 * @param		PMD_InitTypeDef* const	_initdata	The structure containing basic PMD configuration.
 *
 * @return		none
 *********************************************************************************************
 */
void IP_PMD_init(TSB_PMD_TypeDef* const PMDx, PMD_InitTypeDef* const _initdata)
{

	PMDx->MODESEL = WBUF_BUS;

	/***** active setting *****/
	PMDx->MDPOT = cMDPOT_INI;
	if (_initdata->poll == 1)
	{
		PMDx->MDPOT |= (1U << B_POLL);
	}
	else
	{
		PMDx->MDPOT &= ~(1U << B_POLL);
	}
	if (_initdata->polh == 1)
	{
		PMDx->MDPOT |= (1U << B_POLH);
	}
	else
	{
		PMDx->MDPOT &= ~(1U << B_POLH);
	}

	PMDx->MDOUT = cMDOUT_OFF;									/* Output off */
	PMDx->CMPU = 0;
	PMDx->CMPV = 0;
	PMDx->CMPW = 0;

	/***** EMG setting *****/
#if (1)
	PMDx->EMGCR = cEMG_ENA; 									/* EMG Enable */
	PMDx->EMGCR = cEMG_Release; 								/* EMG Release */
#else
	PMDx->EMGREL = 0x5A;										/* Disable code */
	PMDx->EMGREL = 0xA5;
	PMDx->EMGCR = cEMG_DIS; 									/* EMG Disable */
#endif

	/* OVV disable set */
	PMDx->EMGREL = 0x5A;										/* Disable code */
	PMDx->EMGREL = 0xA5;
	PMDx->OVVCR = cOVV_DIS;

	PMDx->PORTMD = cPORTMD_INI;									/* Tool break */
	PMDx->MDCR = cMDCR_INI;

	PMDx->MDPRD = _initdata->pwmfreq;							/* PWM frequency */
	PMDx->DTR = _initdata->deadtime; 							/* Dead timewr set */
	PMDx->MDEN = 0x01;											/* PMD enable (Set after set MDOUT,MDPOT) */

	/***** port setting *****/
	if (PMDx == TSB_PMD1)
	{
		TSB_PG->FR1 |= BIT8(0, 0, 1, 1, 1, 1, 1, 1); 			/* PMD1 PORT output Enable */
		TSB_PG->CR  |= BIT8(0, 0, 1, 1, 1, 1, 1, 1); 			/* PMD1 PORT output Enable */

		if ((*((__IO uint32_t*)BITBAND_PERI(&PMDx->EMGCR, B_EMGEN))) == 1)	/* EMG enable? */
		{
			TSB_PG->FR1 |= BIT8(0, 1, 0, 0, 0, 0, 0, 0);		/* PMD1 PORT Enable */
			TSB_PG->IE  |= BIT8(0, 1, 0, 0, 0, 0, 0, 0); 		/* PMD1 PORT Enable */
		}

		if ((*((__IO uint32_t*)BITBAND_PERI(&PMDx->OVVCR, B_OVVEN))) == 1)	/* OVV enable? */
		{
			TSB_PG->FR1 |= BIT8(1, 0, 0, 0, 0, 0, 0, 0); 		/* PMD1 PORT Enable */
			TSB_PG->IE  |= BIT8(1, 0, 0, 0, 0, 0, 0, 0); 		/* PMD1 PORT Enable */
		}
	}

	/***** Release EMG protection *****/
	if ((*((__I uint32_t*)BITBAND_PERI(&PMDx->EMGSTA, B_EMGI))) == 1)
	{
		(*((__O uint32_t*)BITBAND_PERI(&PMDx->EMGCR, B_EMGRS))) = 1;
	}


	/***** Triger setting *****/
	PMDx->TRGCMP0 = 0; 											/* TRG position clear */
	PMDx->TRGCMP1 = 0;
	PMDx->TRGCMP2 = 0;
	PMDx->TRGCMP3 = 0;

	if (_initdata->shunt == c3shunt)
	{
		PMDx->TRGCR = cTRG_3SHUNT;
		PMDx->TRGMD = cTRGMD_3SHUNT;
	}
	else if (_initdata->shunt == c1shunt)
	{
		PMDx->TRGCR = cTRG_1SHUNT;
		PMDx->TRGMD = cTRGMD_1SHUNT;
	}

	PMDx->TRGSEL = 0;											/* TRG No = Sector = 0 */

	PMDx->MODESEL = WBUF_VE;									/* W-Buff from VE */

}

/**
 *********************************************************************************************
 * @brief		IP_VE_init
 *
 * @param		TSB_VE_TypeDef* const	VEx			Select the VE channel.
 * @param		VE_InitTypeDef* const	_initdata	The structure containing basic VE configuration
 *
 * @return		none
 *********************************************************************************************
 */
void IP_VE_init(TSB_VE_TypeDef* const VEx, VE_InitTypeDef* const _initdata)
{

	/*********************** VE Control Register ********************************/
	VEx->EN = 0x01; 											/* VE Enable */

	if (_initdata->ve_ch == cCH1)
	{
		VEx->TASKAPP |= SET_TAPP_CH1(TAPP_OUT_CON);						/* VE start from output control */
		VEx->ACTSCH |= SET_ACTSCH_CH1(SCH_9); 							/* Schedule 9 */

		VEx->REPTIME |= SET_REPTIME_CH1(_initdata->reptime);
		VEx->TRGMODE |= SET_TRGMODE_CH1(_initdata->trgmode);

		VEx->ERRINTEN |= SET_ERRINTEN_CH1(cVERREN_INI);
	}

	VEx->TMPREG0 = 0;											/* Temporary register clear */
	VEx->TMPREG1 = 0;
	VEx->TMPREG2 = 0;
	VEx->TMPREG3 = 0;
	VEx->TMPREG4 = 0;
	VEx->TMPREG5 = 0;

	VEx->TADC = (uint16_t)(cTADC * cIMCLK_FRQ * 1e6);			/* for 1shunt shift pwm.  (AD convert time) */

	if (_initdata->ve_ch == cCH1)
	{
		/*********************** VE Register for Channel 1 **************************/
		VEx->MODE1 = VEMODE_STOP;

		if (_initdata->shunt == c3shunt)
		{
			VEx->FMODE1 = cFMODE1_INI_3SHUNT;
		}
		else
		{
			VEx->FMODE1 = cFMODE1_INI_1SHUNT;
		}

		VEx->TPWM1 = _initdata->tpwm;

		VEx->OMEGA1 = 0;
		VEx->THETA1 = 0;

		VEx->COS1 = 0;
		VEx->SIN1 = 0;
		VEx->COSM1 = 0;
		VEx->SINM1 = 0;

		VEx->IDREF1 = 0;
		VEx->IQREF1 = 0;

		VEx->VD1 = 0;
		VEx->VQ1 = 0;

		VEx->CIDKI1 = _initdata->idki;
		VEx->CIDKP1 = _initdata->idkp;
		VEx->CIQKI1 = _initdata->iqki;
		VEx->CIQKP1 = _initdata->iqkp;

		VEx->VDIH1 = 0;
		VEx->VDILH1 = 0;
		VEx->VQIH1 = 0;
		VEx->VQILH1 = 0;

		VEx->MCTLF1 = 0x00;

		VEx->FPWMCHG1 = 0x7fff;										/* Hard control off */

		VEx->MDPRD1 = _initdata->pwmfreq;		 					/* PWM frequency */

		VEx->MINPLS1 = 0;

		VEx->SECTOR1 = 0;
		VEx->SECTORM1 = 0;

		VEx->IAO1 = _initdata->zerooffset;
		VEx->IBO1 = _initdata->zerooffset;
		VEx->ICO1 = _initdata->zerooffset;

		VEx->IAADC1 = 0;
		VEx->IBADC1 = 0;
		VEx->ICADC1 = 0;

		VEx->VDC1 = 0;
		VEx->ID1 = 0;
		VEx->IQ1 = 0;

		VEx->VCMPU1 = 0;
		VEx->VCMPV1 = 0;
		VEx->VCMPW1 = 0;

		VEx->OUTCR1 = 0x0000;										/* Output UP/DOWN = OFF/OFF */

		VEx->TRGCRC1 = 0;

		VEx->VTRGCMP01 = 0;
		VEx->VTRGCMP11 = 0;

		VEx->VTRGSEL1 = 0;

		VEx->EMGRS1 = 0;

		VEx->CPURUNTRG = VE_START_CH1;
	}

}

/**
 *********************************************************************************************
 * @brief		ADC Initial set
 *
 * @param		TSB_AD_TypeDef* const	ADx			Select the ADC channel.
 * @param		AD_InitTypeDef* const	_initdata	The structure containing basic ADC configuration
 *
 * @return		none
 *********************************************************************************************
 */
void IP_ADC_init(TSB_AD_TypeDef* const ADx, AD_InitTypeDef* const _initdata)
{
	uint32_t data32;

	/***** PMD TRG Start mode set *****/

	if (_initdata->shunt == c3shunt)
	{
		/* ADx Program set */
		if (_initdata->pmd_ch == cPMD1)
		{
			ADx->PSEL6  = TRG_ENABLE(PMD_PROG0);		/* PMD1 TRG0:BProg0 at Sector0	3-shunt */
			ADx->PSEL7  = TRG_ENABLE(PMD_PROG1);		/* PMD1 TRG1:BProg1 at Sector1	3-shunt */
			ADx->PSEL8  = TRG_ENABLE(PMD_PROG2);		/* PMD1 TRG2:BProg2 at Sector2	3-shunt */
			ADx->PSEL9  = TRG_ENABLE(PMD_PROG3);		/* PMD1 TRG3:BProg3 at Sector3	3-shunt */
			ADx->PSEL10 = TRG_ENABLE(PMD_PROG4);		/* PMD1 TRG4:BProg4 at Sector4	3-shunt */
			ADx->PSEL11 = TRG_ENABLE(PMD_PROG5);		/* PMD1 TRG5:BProg5 at Sector5	3-shunt */
		}

		if (_initdata->pints == cPINTS_B)
		{
			ADx->PINTS0 = PMD_INTADPDB;					/* End of all prog generate INTADB */
			ADx->PINTS1 = PMD_INTADPDB;
			ADx->PINTS2 = PMD_INTADPDB;
			ADx->PINTS3 = PMD_INTADPDB;
			ADx->PINTS4 = PMD_INTADPDB;
			ADx->PINTS5 = PMD_INTADPDB;
		}

		/* ADx->PSET5,PSET0	REG0:V, REG1:W, REG2:U, REG3:Vdc */
		data32 = (((0x80 | (VE_PHASE_V    << 5U) | _initdata->ivch)) \
		        | ((0x80 | (VE_PHASE_W    << 5U) | _initdata->iwch)  << 8U) \
		        | ((0x80 | (VE_PHASE_U    << 5U) | _initdata->iuch)  << 16U) \
		        | ((0x80 | (VE_PHASE_NONE << 5U) | _initdata->vdcch) << 24U));
		ADx->PSET5 = data32;
		ADx->PSET0 = data32;

		/* ADx->PSET1,PSET2	REG0:W, REG1:U, REG2:V, REG3:Vdc */
		data32 = (((0x80 | (VE_PHASE_W    << 5U) | _initdata->iwch)) \
		        | ((0x80 | (VE_PHASE_U    << 5U) | _initdata->iuch)  << 8U) \
		        | ((0x80 | (VE_PHASE_V    << 5U) | _initdata->ivch)  << 16U) \
		        | ((0x80 | (VE_PHASE_NONE << 5U) | _initdata->vdcch) << 24U));
		ADx->PSET1 = data32;
		ADx->PSET2 = data32;

		/* ADx->PSET3,PSET4	REG0:U, REG1:V, REG2:W, REG3:Vdc */
		data32 = (((0x80 | (VE_PHASE_U    << 5U) | _initdata->iuch)) \
		        | ((0x80 | (VE_PHASE_V    << 5U) | _initdata->ivch)  << 8U) \
		        | ((0x80 | (VE_PHASE_W    << 5U) | _initdata->iwch)  << 16U) \
		        | ((0x80 | (VE_PHASE_NONE << 5U) | _initdata->vdcch) << 24U));
		ADx->PSET3	= data32;
		ADx->PSET4	= data32;

	}
	else if (_initdata->shunt == c1shunt)
	{
		/* ADx Program set */
		if (_initdata->pmd_ch == cPMD1)
		{
			ADx->PSEL6  = TRG_ENABLE(PMD_PROG0);		/* TRG0:BProg0 	1-shunt */
			ADx->PSEL7  = TRG_ENABLE(PMD_PROG1);		/* TRG1:BProg1 	1-shunt */
			ADx->PSEL8  = TRG_ENABLE(PMD_PROG2);		/* TRG2:BProg2 */
			ADx->PSEL9  = TRG_DISABLE(PMD_PROG0);		/* TRG3:BProg3 */
			ADx->PSEL10 = TRG_DISABLE(PMD_PROG0); 		/* TRG4:BProg4 */
			ADx->PSEL11 = TRG_DISABLE(PMD_PROG0); 		/* TRG5:BProg5 */
		}

		if (_initdata->pints == cPINTS_B)
		{
			ADx->PINTS0 = PMD_INTNONE; 					/* End of all prog generate INTADB */
			ADx->PINTS1 = PMD_INTNONE;
			ADx->PINTS2 = PMD_INTADPDB;
			ADx->PINTS3 = PMD_INTNONE;
			ADx->PINTS4 = PMD_INTNONE;
			ADx->PINTS5 = PMD_INTNONE;
		}

		if (ADx == TSB_ADB)
		{
			/* AD unitB */
			/*ADx->PSET0	REG0:Idc----, REG1:-------, REG2:-------, REG3:------- */
			data32 = (0x80 | _initdata->idcch);
			ADx->PSET0 = data32;

			/*ADx->PSET1	REG0:-------, REG1:Idc----, REG2:-------, REG3:------- */
			data32 = ((0x80 | _initdata->idcch) << 8U);
			ADx->PSET1 = data32;

			/*ADx->PSET2	REG0:-------, REG1:-------, REG2:-------, REG3:Vdc---- */
			data32 = ((0x80 | _initdata->vdcch) << 24U);
			ADx->PSET2 = data32;

		}
		else
		{
			/* AD unitA */
			/*ADx->PSET0	REG0:Idc----, REG1:-------, REG2:-------, REG3:------- */
			data32 = (0x80 | _initdata->idcch);
			ADx->PSET0 = data32;

			/*ADx->PSET1	REG0:-------, REG1:Idc----, REG2:-------, REG3:------- */
			data32 = (((0x80 | _initdata->idcch) << 8U));
			ADx->PSET1 = data32;

			/*ADx->PSET2	REG0:-------, REG1:-------, REG2:-------, REG3:Vdc---- */
			data32 = ((0x80 | _initdata->vdcch) << 24U);
			ADx->PSET2 = data32;
		}

		ADx->PSET3 = 0x00000000;
		ADx->PSET4 = 0x00000000;
		ADx->PSET5 = 0x00000000;
	}

}


/**
 *********************************************************************************************
 * @brief		AMP IP Initialize
 *
 * @return		none
 *********************************************************************************************
 */
void IP_AMP_init(void)
{

#if defined(__CMP)
	/* Setting internal comparetor */
	TSB_CMP->CTLA = CMP_DISABLE(CMPSEL_OPAMP);
	TSB_CMP->CTLB = CMP_DISABLE(CMPSEL_OPAMP);
	TSB_CMP->CTLC = CMP_DISABLE(CMPSEL_OPAMP);
	TSB_CMP->CTLD = CMP_DISABLE(CMPSEL_OPAMP);
#endif

#if defined(__AMP)
	/* Setting internal amp */
	TSB_AMP->CTLA = AMP_ENABLE(AMPGAIN_FOLD_3_0);
	TSB_AMP->CTLB = AMP_ENABLE(AMPGAIN_FOLD_3_0);
	TSB_AMP->CTLC = AMP_ENABLE(AMPGAIN_FOLD_3_0);
	TSB_AMP->CTLD = AMP_DISABLE(AMPGAIN_FOLD_3_0);
#endif

}

/*********************************** END OF FILE ******************************/
