/**
 ****************************************************************************
 * @file     usercon.c
 * @brief    User Control
 * @version  V1.1
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2022
 *****************************************************************************
 */
#include <stdlib.h>
#include <stdio.h>

#include "ipdefine.h"
#include "mcuip_drv.h"
#include "dac_drv.h"

#define DEFINE_APP
#include "usercon.h"
#undef	DEFINE_APP


/*===================================================================*
	  Macro Definition
 *===================================================================*/

/*===================================================================*
	  Local Constant Data Definition
 *===================================================================*/
/* Timer setting */
#define	cEMG_S				(1)								/* LED1 blinking request(over detect(soft)) */
#define	cEMG_I				(2)								/* LED1 blinking request(curret detect error) */
#define	cEMG_DC				(3)								/* LED1 blinking request(over vdc) */

#define cPORT_WAIT_CNT		(1000)							/* Wait count for initialization of UART port */
#define	c2000MS_TIMER		(2000)							/* [ms] (4kHz * 4) * 2000 */


/* Key settting */
#define	S_SW2				TSB_PE_DATA_PE7 				/* SW2 or 9(KEY2) */
#define	S_SW3				TSB_PK_DATA_PK1 				/* GPIO_15 */

#define	cKEY_CHATA_CNT		(20)							/* [cnt] chattering counter for KEY SW */


/* Soft ADC Setting */
#define	cADUNIT_USR			TSB_ADB							/* User ad data ADCUnit */
#define	cADTRG_USR			ADC_TRG_SW
#define	cADCH_VR			ADC_AIN11						/* ADC Channel for VR */
#define	cADREG_VR			ADC_REG5						/* Result register No for VR */

#define	cADAVECNT			(10)							/* ADC average count */
#define	cPUSHSW_CHATA_CNT	(5) 							/* [cnt] Chattering counter for Push SW */


/* Led Setting */
#define	LED_EMG				TSB_PF_DATA_PF2					/* LED1 */
#define	LED_UART_ERR		TSB_PF_DATA_PF4					/* LED3 */

#define	cFLASH_TYPE1_CYCLE	(1)								/* [s] led flash cycle type1 */
#define	cFLASH_TYPE2_CYCLE	(0.5f)							/* [s] led flash cycle type2 */
#define	cFLASH_TYPE3_CYCLE	(0.25f)							/* [s] led flash cycle type3 */

#define	cLED_ON				(1)								/* LED ON level */
#define	cLED_OFF			(0)								/* LED OFF level */


/* User Setting */
#define	cROTATION_CW		(1)								/* Direction: plus */
#define	cROTATION_CCW		(0)								/* Direction: minus */

#define	cCONTROL_SINGLE		(0)								/* UART control: single */
#define	cCONTROL_UART		(1)								/* UART control: uart */


/* Speed Control Setting  */
#define	cAD_MIN				(0x10)							/* motor speed ADC min value */
#define	cAD_MAX				(0xF0)							/* motor speed ADC max value */
//#define	cSPEED_USER_MIN		(10)							/* [Hz] Min Target speed of motor */
#define	cSPEED_USER_MIN		(cHZ_MIN_CH1)					/* [Hz] Min Target speed of motor */
#define	cSPEED_USER_MAX		(60)							/* [Hz] Max Target speed of motor */


/* UART Setting */
#define	UART_ch				UART0							/* UART Channel */
#define	INTERRUPT_TX		INTTX0_IRQn						/* UART Interrupt request */
#define	INTERRUPT_RX		INTRX0_IRQn						/* UART Interrupt request */

#define	cSEND_DATA_NUM		(7)								/* Send data size */
#define	cRECEIVE_DATA_NUM	(6)								/* Receive data size */

#define	cUART_RECEIVE_WAIT	(0x00)							/* UART mode : data receive wait */
#define	cUART_ERR			(0x01)							/* UART mode : error */

#define	cREQ_SYSTEM_START			(0x10)					/* System start request */
#define	cREQ_ROTATE_MOTOR			(0x11)					/* Target speed update request */
#define	cGET_MOTOR_ENABLE			(0x80)					/* Operating status */
#define	cGET_STATE_EMG				(0x81)					/* Emergency status */
#define	cGET_STAGE					(0x82)					/* Main stage */
#define	cGET_CONTROL_CH				(0x83)					/* Control channel */
#define	cGET_CARRIER_FREQUENCY		(0x84)					/* Carrier frequency */
#define	cGET_MOTOR_SPEED_MIN		(0x85)					/* Minimum rotation speed */
#define	cGET_MOTOR_SPEED_MAX		(0x86)					/* Maximum rotation speed */
#define	cGET_DEAD_TIME				(0x87)					/* Dead time */
#define	cGET_GATE_ACTIVE			(0x88)					/* Gate active */
#define	cGET_POSITION_DITECT		(0x89)					/* Position detect */
#define	cGET_VDC_VOLTAGE			(0x8A)					/* VDC voltage */
#define	cGET_INVETER_TEMP			(0x8B)					/* Inverter temp */
#define	cGET_U_VOLTAGE				(0x8C)					/* U-phase voltage */
#define	cGET_V_VOLTAGE				(0x8D)					/* V-phase voltage */
#define	cGET_W_VOLTAGE				(0x8E)					/* W-phase voltage */
#define	cGET_DAC_DATA				(0x8F)					/* Daca date */
#define	cGET_INTERNAL_AMP			(0x90)					/* Internal amp */
#define	cGET_DIRECTION				(0x91)					/* Direction */
#define	cGET_MODULATION				(0x92)					/* Modulation */
#define	cGET_KEY_OPERATION			(0x93)					/* Key operation */
#define	cGET_MOTOR_SPEED			(0x94)					/* motor rotation speed */

#define	cEMG_STATE_EMG_H			(0x00)					/* emergency status : over detect(hard) */
#define	cEMG_STATE_EMG_S			(0x01)					/* emergency status : over detect(soft) */
#define	cEMG_STATE_EMG_I			(0x02)					/* emergency status : curret detect error */
#define	cEMG_STATE_EMG_DC			(0x03)					/* emergency status : over vdc */

#define	cCONTROL_CH_CH1				(0x01)					/* Motor control channel : ch1 */
#define	cCONTROL_CH					(cCONTROL_CH_CH1)		/* User Motor control channel */

#define	cGATE_ACTIVE_H_H			(0)						/* Gate active : H/H */
#define	cGATE_ACTIVE_L_L			(1)						/* Gate active : L/L */
#define	cGATE_ACTIVE_H_L			(2)						/* Gate active : H/L */
#define	cGATE_ACTIVE_L_H			(3)						/* Gate active : L/H */

#define	cPOSITION_DETECT_3SHUNT		(0)						/* Position Ditect : 3shunt */
#define	cPOSITION_DETECT_1SHUNT		(1)						/* Position Ditect : 1shunt */

#define	cDAC_DATA_TMPREG0			(0x00)					/* Dac data : TMPREG0 */
#define	cDAC_DATA_TMPREG1			(0x01)					/* Dac data : TMPREG1 */
#define	cDAC_DATA_TMPREG2			(0x02)					/* Dac data : TMPREG2 */
#define	cDAC_DATA_THETAHALF			(0x03)					/* Dac data : theta.half[1] */
#define	cDAC_DATA_IDREF				(0x04)					/* Dac data : Id_ref */
#define	cDAC_DATA_ID				(0x05)					/* Dac data : Id */
#define	cDAC_DATA_IQREF				(0x06)					/* Dac data : Iq_ref */
#define	cDAC_DATA_IQ				(0x07)					/* Dac data : Iq */
#define	cDAC_DATA_OMEGACOMHALH		(0x08)					/* Dac data : omega_com.half[1] */
#define	cDAC_DATA_OMEGAHALF			(0x09)					/* Dac data : omega.half[1] */
#define	cDAC_DATA_OMEGADEV			(0x0A)					/* Dac data : omega_dev */

#define	cINTERNAL_AMP_NO			(0)						/* Internal amp : External */
#define	cINTERNAL_AMP_YES			(1)						/* Internal amp : Internal */

#define	cDIRECTION_CW				(0)						/* Direction : plus */
#define	cDIRECTION_CCW				(1)						/* Direction : minus */


/*===================================================================*
	  Typedef Definition
 *===================================================================*/
typedef struct
{
	uint8_t		select;										/* Dac output select */
	uint8_t		motch;										/* Dac motor ch select */
	uint8_t		datsft0;									/* Dac data shift for data0 */
	uint8_t		datsft1;									/* Dac data shift for data1 */
	uint8_t		datsft2;									/* Dac data shift for data2 */
	uint8_t		datsft3;									/* Dac data shift for data3 */
} dac_t;


extern	int32_t	target_spd1;


/*===================================================================*
	  Local Variable Definition
 *===================================================================*/
static sw_t			portbuf;	 							/* Key port data */
static sw_t			portbuf_old; 							/* Previous Key port data */
static uint8_t		key_chatacnt;							/* Chattering counter for Key */

static addat_t		ad_vr;									/* AD value of volume register */

/* No, MotorCh, sft0, sft1, sft2, sft3 */
dac_t 		dac = { 0, 1, 0, 0, 0, 0 };						/* Dac setting. */

static uint16_t		timeout_timer;							/* 2s timer counter */
static uint8_t		receive_timer;							/* uart receive timer counter */

static uint8_t		emg_state_old;							/* emg state memory */
static uint16_t		emg_flash_cycle;						/* emg led cycle counter */
static uint16_t		uarterr_flash_cycle;					/* uart error led cycle counter */
static uint8_t		led1_state;								/* led1 lighting state */
static uint8_t		led3_state;								/* led3 lighting state */

static uint8_t		uart_mode;								/* uart contorl mode */
static uint8_t		sendbuffer[cSEND_DATA_NUM];				/* Send data buffer */
static uint8_t		receivebuffer[cRECEIVE_DATA_NUM];		/* Receive data buffer */
static uint8_t		receivedata[cRECEIVE_DATA_NUM];			/* Receive data */
static uint8_t		txpnt;									/* Pointer of send data */
static uint8_t		rxpnt;									/* Pointer of receive data */

/*===================================================================*
	  Local Proto Type Definition
 *===================================================================*/
static void timer_count(void);
static void led_brightcount(uint8_t emg_state);

static void Uikeyscan(void);
static void Keychata(sw_t portdata);

static void init_led(void);
static void led_display(void);
static void init_Uikey(void);
static void init_soft_adc(void);
static void soft_adc_getdata(void);
static void soft_adc_avecal(addat_t* const _ad_dat);

static void target_speed_setting(void);

static void init_uart(void);
static void send_data_set(void);
static void data_check(void);

/**
 *********************************************************************************************
 * @brief		init_user_control
 *
 * @return		none
 *********************************************************************************************
 */
void	init_user_control(void)
{
	init_Uikey();										/* Initial setting of user control. */
	init_soft_adc();									/* Initial setting of soft adc */
	init_led();											/* Initial setting of LED output */
	init_uart();										/* Initial setting of UART */

}

/**
 *********************************************************************************************
 * @brief		user_control
 *
 * @return		none
 *********************************************************************************************
 */
void	user_control(void)
{
	/* timer counter */
	timer_count();

	if(keydata.bit.sw2 == cCONTROL_SINGLE)
	{
		/***** Get AD Data *****/
		soft_adc_getdata();

		/***** user setting *****/
		target_speed_setting();
	}

	/***** key input *****/
	Uikeyscan();

	/***** led output *****/
	led_display();

}


/**
 *********************************************************************************************
 * @brief		Timer counter.
 *
 * @return		none
 *********************************************************************************************
 */
static void timer_count(void)
{
	uint8_t i;
	vector_t* _motor;
	_motor = &Motor_ch1;

	if((_motor->drv.state.flg.emg_S) || (_motor->drv.state.flg.emg_I) || (_motor->drv.state.flg.emg_DC))
	{
		if(_motor->drv.state.flg.emg_S == 1)
		{
			i = cEMG_S;
		}
		else if(_motor->drv.state.flg.emg_I == 1)
		{
			i = cEMG_I;
		}
		else if(_motor->drv.state.flg.emg_DC == 1)
		{
			i = cEMG_DC;
		}
		else
		{
			/* no process */
		}

		led_brightcount(i);
	}

	if(keydata.bit.sw2 == cCONTROL_UART)
	/* uart contorol mode : communication */
	{
		if(timeout_timer < c2000MS_TIMER)
		{
			if(++timeout_timer >= c2000MS_TIMER)
			{
				flg.timer.timeout = 1;
			}
		}

		if(receive_timer > 0)
		{
			if(--receive_timer <= 0)
			{
				rxpnt = 0;
			}
		}

		if(uart_mode == cUART_ERR)
		{
			if(++uarterr_flash_cycle >= (cFLASH_TYPE3_CYCLE / cMAINLOOP_PRD))
			{
				uarterr_flash_cycle = 0;
				led3_state ^= 1;
			}
		}
	}

}


/**
 *********************************************************************************************
 * @brief		Led brightcount.
 *
 * @return		none
 *********************************************************************************************
 */
static void led_brightcount(uint8_t emg_state)
{
	if(emg_state_old != emg_state)
	{
		emg_state_old = emg_state;
		led1_state = 1;											/* led1: on start */
		emg_flash_cycle = 0;
	}

	switch(emg_state)
	{
		case cEMG_S:
			if(uart_mode != cUART_ERR)
			{
				if(++emg_flash_cycle >= (cFLASH_TYPE1_CYCLE / cMAINLOOP_PRD))
				{
					emg_flash_cycle = 0;
					led1_state ^= 1;
				}
			}
			break;

		case cEMG_I:
			if(++emg_flash_cycle >= (cFLASH_TYPE2_CYCLE / cMAINLOOP_PRD))
			{
				emg_flash_cycle = 0;
				led1_state ^= 1;
			}
			break;

		case cEMG_DC:
			if(++emg_flash_cycle >= (cFLASH_TYPE3_CYCLE / cMAINLOOP_PRD))
			{
				emg_flash_cycle = 0;
				led1_state ^= 1;
			}
			break;

		default:
			break;
	}

}


/**
 *********************************************************************************************
 * @brief		Initialize key.
 *
 * @return		none
 *********************************************************************************************
 */
static void init_Uikey(void)
{
	/* SW2 or 9 */
	GPIO_SetInputEnableReg(GPIO_PE, GPIO_BIT_7, ENABLE);
	/* SlideSW3 */
	GPIO_SetInputEnableReg(GPIO_PK, GPIO_BIT_1, ENABLE);

}


/**
 *********************************************************************************************
 * @brief		keyscan
 *
 * @return		none
 *********************************************************************************************
 */
static void Uikeyscan(void)
{
	portbuf.bit.sw3 = S_SW3;

	Keychata(portbuf);

}


/**
 *********************************************************************************************
 * @brief		keyscan
 *
 * @return		none
 *********************************************************************************************
 */
void UikeyInitscan(void)
{
	key_chatacnt = cKEY_CHATA_CNT;

	while(flg.key.keydata_set == 0)
	{
		if (M_Main_Counter >= cMAINLOOP_CNT)
		{
			M_Main_Counter = 0;

			/***** Clear WDT *****/
			WDT_WriteClearCode();

			portbuf.bit.sw2 = S_SW2;
			portbuf.bit.sw3 = S_SW3;

			Keychata(portbuf);
		}
	}

	flg.key.keydata_set = 0;

}


/**
 *********************************************************************************************
 * @brief		keychataring
 *
 * @param		sw_t* portdata
 *
 * @return		none
 *********************************************************************************************
 */
static void Keychata(sw_t portdata)
{
	if (portdata.all == portbuf_old.all)
	{
		if (key_chatacnt != 0)
		{
			if (--key_chatacnt == 0)
			{
				keydata = portdata;
				flg.key.keydata_set = 1;
			}
		}
	}
	else
	{
		portbuf_old = portdata;
		key_chatacnt = cKEY_CHATA_CNT;
	}

}


/**
 *********************************************************************************************
 * @brief		Initialize led
 *
 * @return		none
 *********************************************************************************************
 */
static void init_led(void)
{
	GPIO_WriteDataBit(GPIO_PF, GPIO_BIT_2, cLED_OFF);		/* LED1 */
	GPIO_SetOutputEnableReg(GPIO_PF, GPIO_BIT_2, ENABLE);

	GPIO_WriteDataBit(GPIO_PF, GPIO_BIT_3, cLED_OFF);		/* LED2 */
	GPIO_SetOutputEnableReg(GPIO_PF, GPIO_BIT_3, ENABLE);

	GPIO_WriteDataBit(GPIO_PF, GPIO_BIT_4, cLED_OFF);		/* LED3 */
	GPIO_SetOutputEnableReg(GPIO_PF, GPIO_BIT_4, ENABLE);

}


/**
 *********************************************************************************************
 * @brief		Led disply
 *
 * @return		none
 *********************************************************************************************
 */
static void led_display(void)
{
	vector_t* _motor;
	_motor = &Motor_ch1;

	if(_motor->drv.state.flg.emg_H)
	{
		/* EMG state : over detect(hard) */
		LED_EMG = cLED_ON;
	}
	else if((_motor->drv.state.flg.emg_S) || (_motor->drv.state.flg.emg_I) || (_motor->drv.state.flg.emg_DC))
	{
		/* EMG state : over detect(soft) or curret detect error or over vdc */
		LED_EMG = led1_state;
	}
	else
	{
		/* EMG state : no over detect */
		LED_EMG = cLED_OFF;
	}

	if(keydata.bit.sw2 == cCONTROL_UART)
	{
		/* uart contorol mode : communication */
		if(uart_mode == cUART_ERR)
		{
			/* uart error */
			LED_UART_ERR = led3_state;
		}
		else
		{
			LED_UART_ERR = cLED_ON;
		}
	}

}


/**
 *********************************************************************************************
 * @brief		Initialize softconvert adc.
 *
 * @return		none
 *********************************************************************************************
 */
static void init_soft_adc(void)
{
	ADC_SetSWTrg(cADUNIT_USR, cADREG_VR, TRG_ENABLE(cADCH_VR));
	ADC_Start(cADUNIT_USR, cADTRG_USR); 						/* adc start */

}


/**
 *********************************************************************************************
 * @brief		soft_adc_getdata
 *
 * @return		none
 *********************************************************************************************
 */
static void soft_adc_getdata(void)
{
	if (ADC_GetConvertState(cADUNIT_USR, cADTRG_USR) == DONE)		/* finish adc? */
	{
		ad_vr.rawdat = ADC_GetConvertResult(cADUNIT_USR, cADREG_VR);
		soft_adc_avecal(&ad_vr);

		ADC_Start(cADUNIT_USR, cADTRG_USR); 						/* adc start */
	}

}


/**
 *********************************************************************************************
 * @brief		soft_adc_avecal
 *
 * @param		addat_t* const	_ad_dat
 *
 * @return		none
 *********************************************************************************************
 */
static void soft_adc_avecal(addat_t* const _ad_dat)
{
	if (_ad_dat->avecnt == 0)
	{
		/* first? */
		_ad_dat->maxdat = _ad_dat->rawdat.Bit.ADResult;
		_ad_dat->mindat = _ad_dat->rawdat.Bit.ADResult;
	}
	else
	{
		if (_ad_dat->rawdat.Bit.ADResult > _ad_dat->maxdat)
		{
			_ad_dat->maxdat = _ad_dat->rawdat.Bit.ADResult;
		}
		else if (_ad_dat->rawdat.Bit.ADResult < _ad_dat->mindat)
		{
			_ad_dat->mindat = _ad_dat->rawdat.Bit.ADResult;
		}
		else
		{
			/* no process */
		}
	}

	_ad_dat->sumdat += _ad_dat->rawdat.Bit.ADResult;

	_ad_dat->avecnt++;
	if (_ad_dat->avecnt >= cADAVECNT)
	{
		_ad_dat->avecnt = 0;
		_ad_dat->sumdat -= _ad_dat->maxdat;
		_ad_dat->sumdat -= _ad_dat->mindat;
		_ad_dat->avedat = (_ad_dat->sumdat / (cADAVECNT - 2));
		_ad_dat->sumdat = 0;

		flg.adc.ad_update = 1;
	}

}


/**
 *********************************************************************************************
 * @brief		target_speed_setting
 *
 * @return		none
 *********************************************************************************************
 */
static void target_speed_setting(void)
{
	uint8_t ad_vr_temp = 0;

	/* speed control */
	if(flg.adc.ad_update == 1)
	{
		ad_vr_temp = ad_vr.avedat >> 4U;

		if(ad_vr_temp < cAD_MIN)
		{
			target_spd1 = 0;
		}
		else
		{
			if(ad_vr_temp < cAD_MAX)
			{
				target_spd1 = (cSPEED_USER_MAX - cSPEED_USER_MIN) * (ad_vr_temp - cAD_MIN) / (cAD_MAX - cAD_MIN) + cSPEED_USER_MIN;
			}
			else
			{
				target_spd1 = cSPEED_USER_MAX;
			}

			if(keydata.bit.sw3 == cROTATION_CCW)
			{
				target_spd1 *= (-1);
			}
		}
	}

}


/**
 *********************************************************************************************
 * @brief		Initialize UART.
 *
 * @return		none
 *********************************************************************************************
 */
static void init_uart(void)
{
	UART_InitTypeDef UART_DispInit;
	uint32_t	port_wait_counter;

	UART_Enable(UART_ch);

	/* Setting UART port. */
	TSB_PE->FR1 |= ((1 << 0) | (1 << 1U));
	TSB_PE->IE |= (1 << 1U);	/* PE1:RXD */
	TSB_PE->CR |= (1 << 0U);	/* PE0:TXD */

	port_wait_counter = 0;
	/* Wait for initialization of UART port */
	while (port_wait_counter <= cPORT_WAIT_CNT*100)
	{
		WDT_WriteClearCode();
		port_wait_counter++;
	}

	UART_SWReset(UART_ch);

	/*initialize the UART struct */
	UART_DispInit.BaudRate = 115200U;
	UART_DispInit.DataBits = UART_DATA_BITS_8;
	UART_DispInit.StopBits = UART_STOP_BITS_1;
	UART_DispInit.Parity = UART_NO_PARITY;
	UART_DispInit.Mode = (UART_ENABLE_TX | UART_ENABLE_RX);
	UART_DispInit.FlowCtrl = UART_NONE_FLOW_CTRL;

	UART_Init(UART_ch, &UART_DispInit);

	NVIC_SetPriority(INTERRUPT_TX, INT_UART_LEVEL); 			/* Interruption level set */
	NVIC_ClearPendingIRQ(INTERRUPT_TX);
	NVIC_SetPriority(INTERRUPT_RX, INT_UART_LEVEL); 			/* Interruption level set */
	NVIC_ClearPendingIRQ(INTERRUPT_RX);

	NVIC_EnableIRQ(INTERRUPT_TX);							/* INTTX enable */
	NVIC_EnableIRQ(INTERRUPT_RX);							/* INTRX enable */

}


/**
 *********************************************************************************************
 * @brief		uart control.
 *
 * @return		none
 *********************************************************************************************
 */
void uart_control(void)
{
	vector_t* _motor;
	_motor = &Motor_ch1;

	if(keydata.bit.sw2 == cCONTROL_UART)
	{
		if(uart_mode == cUART_RECEIVE_WAIT)
		{
			/* ʐM[hFM҂ */
			if(flg.uart.data_receive == 1)
			{
				/* M */
				flg.uart.data_receive = 0;
				timeout_timer = 0;
				data_check();

				/* Mf[^쐬 */
				send_data_set();

				/* MJn */
				txpnt = 0;
				UART_SetTxData(UART_ch,sendbuffer[0]);
			}
			else
			{
				if(flg.timer.timeout == 1)
				{
					/* Set EMG bit */
					_motor->drv.state.flg.emg_S = 1;

					/* UART Stop */
					UART_Disable(UART_ch);

					/* UARTmode : UART ERR */
					uart_mode = cUART_ERR;
				}
			}
		}
	}

}


/**
 *********************************************************************************************
 * @brief		send data setting.
 *
 * @return		none
 *********************************************************************************************
 */
static void send_data_set(void)
{
	uint8_t i;
	float32_t VDC_value;
	uint32_t set_value;
	dac_t* const pdac = &dac;
	vector_t* _motor;
	TSB_VE_TypeDef* pVE;

	_motor = &Motor_ch1;
	pVE = TSB_VE;

	for(i= 0; i< 7; i++)
	{
		sendbuffer[i] = 0x00;
	}

	if((_motor->drv.state.flg.emg_H) || (_motor->drv.state.flg.emg_S) ||
	   (_motor->drv.state.flg.emg_I) || (_motor->drv.state.flg.emg_DC))
	{
		flg.uart.emg_bit = 1;
	}
	else
	{
		flg.uart.emg_bit = 0;
	}

	sendbuffer[0] = receivedata[0];					/* command id */
	sendbuffer[1] = ((flg.uart.emg_bit << 2) | 		/* EMG bit on */
					 (flg.uart.ack_bit));			/* Ack bit on */

	if(flg.uart.ack_bit == 0){
		switch(receivedata[0])
		{
			case cGET_STATE_EMG:
				if(_motor->drv.state.flg.emg_H)
				{
					sendbuffer[2] = cEMG_STATE_EMG_H;					/* data0: over detect(hard) */
				}
				else if(_motor->drv.state.flg.emg_S)
				{
					sendbuffer[2] = cEMG_STATE_EMG_S;					/* data0: over detect(soft) */
				}
				else if(_motor->drv.state.flg.emg_I)
				{
					sendbuffer[2] = cEMG_STATE_EMG_I;					/* data0: current detect */
				}
				else if(_motor->drv.state.flg.emg_DC)
				{
					sendbuffer[2] = cEMG_STATE_EMG_DC;					/* data0: over vdc */
				}
				else
				{
					/* no process */
				}
				break;

			case cGET_STAGE:
				sendbuffer[2] = _motor->stage.main;		/* data0: main stage */
				break;

			case cGET_CONTROL_CH:
				sendbuffer[2] = cCONTROL_CH;
				break;

			case cGET_CARRIER_FREQUENCY:
				set_value = (uint32_t)(1000000 / cPWMPRD_CH1);
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_MOTOR_SPEED_MIN:
				set_value = cHZ_MIN_CH1;
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_MOTOR_SPEED_MAX:
				set_value = cHZ_MAX_CH1;
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_DEAD_TIME:
				set_value = (cDEADTIME_CH1 * 100);
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_GATE_ACTIVE:
				if((cPOLH_CH1 == 1) && (cPOLL_CH1 == 1))
				{
					sendbuffer[2] = cGATE_ACTIVE_H_H;
				}
				else if((cPOLH_CH1 == 0) && (cPOLL_CH1 == 0))
				{
					sendbuffer[2] = cGATE_ACTIVE_L_L;
				}
				else if((cPOLH_CH1 == 1) && (cPOLL_CH1 == 0))
				{
					sendbuffer[2] = cGATE_ACTIVE_H_L;
				}
				else {
					sendbuffer[2] = cGATE_ACTIVE_L_H;
				}
				break;

			case cGET_POSITION_DITECT:
				if(cSHUNT_TYPE_CH1 == c3shunt)
				{
					sendbuffer[2] = cPOSITION_DETECT_3SHUNT;
				}
				else if(cSHUNT_TYPE_CH1 == c1shunt)
				{
					sendbuffer[2] = cPOSITION_DETECT_1SHUNT;
				}
				else {
					/* no process */
				}
				break;

			case cGET_VDC_VOLTAGE:
				VDC_value = (float32_t)(_motor->drv.Vdc * cV_MAX_CH1 / FIX_15);
				set_value = (uint32_t)(VDC_value * 100);
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_INVETER_TEMP:
				sendbuffer[1] |= (flg.uart.na_bit << 1);				/* Not Abailable bit on */
				break;

			case cGET_U_VOLTAGE:
				VDC_value = (float32_t)(pVE->IAADC1 * 5) / 65535;
				set_value = (uint32_t)(VDC_value * 100);
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_V_VOLTAGE:
				VDC_value = (float32_t)(pVE->IBADC1 * 5) / 65535;
				set_value = (uint32_t)(VDC_value * 100);
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_W_VOLTAGE:
				VDC_value = (float32_t)(pVE->ICADC1 * 5) / 65535;
				set_value = (uint32_t)(VDC_value * 100);
				sendbuffer[2] = (uint8_t)(set_value & 0xff);
				sendbuffer[3] = (uint8_t)((set_value >> 8) & 0xff);
				sendbuffer[4] = (uint8_t)((set_value >> 16) & 0xff);
				sendbuffer[5] = (uint8_t)((set_value >> 24) & 0xff);
				break;

			case cGET_DAC_DATA:
#if defined(__USE_DAC)
				switch(pdac->select)
				{
					case 0:
						sendbuffer[2] = cDAC_DATA_TMPREG0;
						sendbuffer[3] = cDAC_DATA_TMPREG1;
						sendbuffer[4] = cDAC_DATA_TMPREG2;
						sendbuffer[5] = cDAC_DATA_THETAHALF;
						break;

					case 1:
						sendbuffer[2] = cDAC_DATA_IDREF;
						sendbuffer[3] = cDAC_DATA_ID;
						sendbuffer[4] = cDAC_DATA_IQREF;
						sendbuffer[5] = cDAC_DATA_IQ;
						break;

					case 2:
						sendbuffer[2] = cDAC_DATA_OMEGACOMHALH;
						sendbuffer[3] = cDAC_DATA_OMEGAHALF;
						sendbuffer[4] = cDAC_DATA_OMEGADEV;
						sendbuffer[5] = cDAC_DATA_IQREF;
						break;

					case 3:
						sendbuffer[2] = cDAC_DATA_TMPREG0;
						sendbuffer[3] = cDAC_DATA_IQREF;
						sendbuffer[4] = cDAC_DATA_IDREF;
						sendbuffer[5] = cDAC_DATA_OMEGAHALF;
						break;

					default :
						break;
				}
#endif
				break;

			case cGET_INTERNAL_AMP:
				sendbuffer[2] = cINTERNAL_AMP_NO;
				break;

			case cGET_DIRECTION:
				if(keydata.bit.sw3 == cROTATION_CW)
				{
					sendbuffer[2] = cDIRECTION_CW;
				}
				else
				{
					sendbuffer[2] = cDIRECTION_CCW;
				}
				break;

			case cGET_MODULATION:
				sendbuffer[2] = _motor->usr.com_user.modul;
				break;

			case cGET_MOTOR_SPEED:
				sendbuffer[2] = (int8_t)((_motor->drv.omega.word * cHZ_MAX_CH1) / FIX_31);
				break;

			case cREQ_SYSTEM_START:
			case cREQ_ROTATE_MOTOR:
			default:
				break;
		}
	}

	for(i= 0; i< 6; i++)
	{
		sendbuffer[6] += sendbuffer[i];
	}

}


/**
 *********************************************************************************************
 * @brief		data check.
 *
 * @return		none
 *********************************************************************************************
 */
static void data_check(void)
{
	uint8_t i, checksum = 0;

	flg.uart.na_bit = 0;
	flg.uart.ack_bit = 0;

	switch(receivedata[0])
	{
		case cREQ_SYSTEM_START:
		case cGET_STATE_EMG:
		case cGET_STAGE:
		case cGET_CONTROL_CH:
		case cGET_CARRIER_FREQUENCY:
		case cGET_MOTOR_SPEED_MIN:
		case cGET_MOTOR_SPEED_MAX:
		case cGET_DEAD_TIME:
		case cGET_GATE_ACTIVE:
		case cGET_POSITION_DITECT:
		case cGET_VDC_VOLTAGE:
		case cGET_U_VOLTAGE:
		case cGET_V_VOLTAGE:
		case cGET_W_VOLTAGE:
		case cGET_DAC_DATA:
		case cGET_INTERNAL_AMP:
		case cGET_DIRECTION:
		case cGET_MODULATION:
		case cGET_MOTOR_SPEED:
		case cREQ_ROTATE_MOTOR:
			/* no process */
			break;

		case cGET_INVETER_TEMP:
			flg.uart.na_bit = 1;
			break;

		default:
			flg.uart.ack_bit = 1;
			break;
	}

	for(i= 0; i< 5; i++)
	{
		checksum = (uint8_t)(checksum + receivedata[i]);
	}

	if(checksum != receivedata[5])
	{
		flg.uart.ack_bit = 1;
	}

	if((receivedata[0] == cREQ_ROTATE_MOTOR) && (flg.uart.ack_bit == 0)){
		if(keydata.bit.sw3 == cROTATION_CW)
		{
			target_spd1 = receivedata[1];
		}
		else
		{
			target_spd1 = (-1) * receivedata[1];
		}
	}

}


/**
 *********************************************************************************************
 * @brief		Transmission completion interrupt.
 *
 * @return		none
 *********************************************************************************************
 */
void INT_uart0tx(void)
{
	if(UART_GetErrState(UART_ch) == UART_NO_ERR)
	{
		if(++txpnt >= cSEND_DATA_NUM)
		{
			txpnt = 0;
		}
		else
		{
			UART_SetTxData(UART_ch,sendbuffer[txpnt]);
		}
	}
}


/**
 *********************************************************************************************
 * @brief		Receive completion interrupt.
 *
 * @return		none
 *********************************************************************************************
 */
void INT_uart0rx(void)
{
	uint8_t	dat;

	dat = UART_GetRxData(UART_ch);

	if(UART_GetErrState(UART_ch) == UART_NO_ERR)
	{
		receivebuffer[rxpnt] = dat;

		/* receive timer init */
		receive_timer = 10;

		if(++rxpnt >= cRECEIVE_DATA_NUM)
		{
			receivedata[0] = receivebuffer[0];
			receivedata[1] = receivebuffer[1];
			receivedata[2] = receivebuffer[2];
			receivedata[3] = receivebuffer[3];
			receivedata[4] = receivebuffer[4];
			receivedata[5] = receivebuffer[5];

			flg.uart.data_receive = 1;
			receive_timer = 0;
			rxpnt = 0;
		}
	}
}


/**
 ********************************************************************************************
 * @brief		UiOutDataStart by VE.
 *
 * @param[in]	uint8_t	_intch	.
 *
 * @return		none
 ********************************************************************************************
 */
void UiOutDataStartByVE(uint8_t	_intch)
{
	dac_t* const pdac = &dac;

	if (_intch == pdac->motch)		/* Dac output ch = call intve ch? */
	{
		uint16_t* const pdacout = DacData;
		vector_t* pmotor = 0;
		TSB_VE_TypeDef* pVE = 0;

		if (pdac->motch == 1)
		{
			pmotor = &Motor_ch1;
			pVE = Motor_ch1_IP.VEx;
		}

		switch (pdac->select)
		{
			case 0:
				*(pdacout + 0) = (pVE->TMPREG0	 				>> (16 - pdac->datsft0)) ^ 0x8000;
				*(pdacout + 1) = (pVE->TMPREG1 					>> (16 - pdac->datsft0)) ^ 0x8000;
				*(pdacout + 2) = (pVE->TMPREG2 					>> (16 - pdac->datsft0)) ^ 0x8000;
				*(pdacout + 3) = pmotor->drv.theta.half[1];
				break;
			case 1:
				*(pdacout + 0) = (pmotor->drv.Id_ref			<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 1) = (pmotor->drv.Id				<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 2) = (pmotor->drv.Iq_ref			<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 3) = (pmotor->drv.Iq				<< pdac->datsft0) ^ 0x8000;
				break;
			case 2:
				*(pdacout + 0) = (pmotor->drv.omega_com.half[1]	<< pdac->datsft2) ^ 0x8000;
				*(pdacout + 1) = (pmotor->drv.omega.half[1] 	<< pdac->datsft2) ^ 0x8000;
				*(pdacout + 2) = (pmotor->drv.omega_dev 		<< pdac->datsft2) ^ 0x8000;
				*(pdacout + 3) = (pmotor->drv.Iq_ref			<< pdac->datsft0) ^ 0x8000;
				break;
			case 3:
				*(pdacout + 0) = (pVE->TMPREG0              	>> (16 - pdac->datsft0)) ^ 0x8000;
				*(pdacout + 1) = (pmotor->drv.Iq_ref			<< pdac->datsft0)		 ^ 0x8000;
				*(pdacout + 2) = (pmotor->drv.Id_ref			<< pdac->datsft0)		 ^ 0x8000;
				*(pdacout + 3) = (pmotor->drv.omega.half[1]		<< pdac->datsft2)		 ^ 0x8000;
				break;
			case 4:
				*(pdacout + 0) = (pmotor->drv.Vdc_ave.half[1]	<< pdac->datsft1) ^ 0x8000;
				*(pdacout + 1) = (pmotor->drv.Vdq_ave.half[1]	<< pdac->datsft1) ^ 0x8000;
				*(pdacout + 2) = (pmotor->drv.Vd_ave.half[1]	<< pdac->datsft0) ^ 0x8000;
				*(pdacout + 3) = (pmotor->drv.Vq_ave.half[1]	<< pdac->datsft0) ^ 0x8000;
				break;
			default:
				*(pdacout + 0) += 0x0100;
				*(pdacout + 1) += 0x0200;
				*(pdacout + 2) += 0x0300;
				*(pdacout + 3) += 0x0400;
				break;
		}

		DacWriteStart(DAC_SIO_CH);

	}
}


/******************************** END OF FILE *********************************/
