/**
 ****************************************************************************
 * @file     D_Driver.c
 * @brief    Motor vector control driver Source File
 * @version  V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2021
 *****************************************************************************
 */
#include <stdlib.h>
#include "ipdefine.h"
#include "mcuip_drv.h"
#include "E_Sub.h"

#define DEFINE_APP
#include "D_Driver.h"
#undef	DEFINE_APP

/*===================================================================*
	  Local Variable Definition
 *===================================================================*/
/*===================================================================*
	  Local Proto Type Definition
 *===================================================================*/

/**
 *********************************************************************************************
 * @brief		Detect _motor position
 *
 * @param		vector_t* const	_motor			The structure address for motor vector control.
 *
 * @return		none
 *********************************************************************************************
 */
void D_Detect_Rotor_Position(vector_t* const _motor)
{

	q31_t temp32_r, temp32_Lq;
	q31_t temp32_OmegaErrI, temp32_OmegaErrP;
	q31_t omega_calc;

	/**
	 * @note	Voltage equation
	 * 			Vd = R*Id + Ld*pId - OMEGA-est*Lq + Ed
	 * 			  (p = d/dt, Id=Constant value -> pId = 0)
	 */
	if (_motor->drv.vector_cmd.F_vcomm_Edetect == SET)
	{
		/* Voltage drop of resistance	 V = R * I */
		temp32_r = (_motor->para.motor.r * (q31_t)_motor->drv.Id) << 1;


		/* Voltage drop of inductance	 V = 2PIfL * I */
		temp32_Lq = (((_motor->drv.omega.half[1] * _motor->para.motor.Lq) << 1) + cROUND_BIT15) >> 16;	/* Q15 * Q15 -> Q15 */
		temp32_Lq = (temp32_Lq * _motor->drv.Iq) << 1;													/* Q15 * Q15 -> Q31 */

		/* d-axis induced voltage */
		/* Ed = Vd - R*Id + OMEGA-est*Lq */
		_motor->drv.Ed = (q15_t)((_motor->drv.Vd.word - temp32_r + temp32_Lq + cROUND_BIT15) >> 16);

		/**
		 * @note	OMEGAest = OMEGAref - K*Ed
		 */
		/* Set integration value */
#if defined(__POSITION_KI_RANGE)
		/* OmegaErr(I) * Hz_MAX = FIX_12 * Position_Ki * CtrlPrd * Ed * V_MAX */
		temp32_OmegaErrI = (_motor->para.pos.ki * (q31_t)_motor->drv.Ed) << 4;
#else
		/* OmegaErr(I) * Hz_MAX = FIX_15 * Position_Ki * CtrlPrd * Ed * V_MAX */
		temp32_OmegaErrI = (_motor->para.pos.ki * (q31_t)_motor->drv.Ed) << 1;
#endif

		if (_motor->drv.omega_com.half[1] >= 0)
		{
			_motor->drv.Ed_I = __QSUB(_motor->drv.Ed_I, temp32_OmegaErrI);
		}
		else
		{
			_motor->drv.Ed_I = __QADD(_motor->drv.Ed_I, temp32_OmegaErrI);
		}

		/* Set proportionality value */
#if defined(__POSITION_KP_RANGE)
		/* OmegaErr(P) * Hz_MAX = FIX_12 * Position_Kp * Ed * V_MAX */
		temp32_OmegaErrP = (_motor->para.pos.kp * (q31_t)_motor->drv.Ed) << 4;
#else
		/* OmegaErr(P) * Hz_MAX = FIX_15 * Position_Kp * Ed * V_MAX */
		temp32_OmegaErrP = (_motor->para.pos.kp * (q31_t)_motor->drv.Ed) << 1;
#endif

		if (_motor->drv.omega_com.half[1] >= 0)
		{
			_motor->drv.Ed_PI = __QSUB(_motor->drv.Ed_I, temp32_OmegaErrP);
		}
		else
		{
			_motor->drv.Ed_PI = __QADD(_motor->drv.Ed_I, temp32_OmegaErrP);
		}

		/* Speed calculation value = Speed command value + Error PI value */
		omega_calc = __QADD(_motor->drv.omega_com.word, _motor->drv.Ed_PI);

		if (_motor->drv.vector_cmd.F_vcomm_omega == SET)
		{
			if (_motor->drv.command.encoder == CLEAR)
			{
				_motor->drv.omega.word = omega_calc;
			}
		}
		else
		{
			/* Initialize of proportionality value, integration value and Omega */
			_motor->drv.omega.word = _motor->drv.omega_com.word;
			_motor->drv.Ed_I = 0;
			_motor->drv.Ed_PI = 0;
		}

		/* Position estimation */
		if (_motor->drv.vector_cmd.F_vcomm_theta == SET)
		{
			if (_motor->drv.command.encoder == CLEAR)
			{
				/* estimated THETAn = THETAn-1 + OMEGA * t */
				_motor->drv.theta.word += ((_motor->drv.omega.half[1] * _motor->para.pos.ctrlprd) << 1);
			}
		}
		else
		{
			_motor->drv.theta.word = (q31_t)(_motor->drv.theta_com << 16);
		}
	}
	else
	{
		_motor->drv.theta.word = (q31_t)(_motor->drv.theta_com << 16);
		_motor->drv.Ed = 0;
	}
}

/**
 *********************************************************************************************
 * @brief		Speed control ( Caricurate q-axis current )
 *
 * @param		vector_t* const	_motor			The structure address for motor vector control.
 *
 * @return		none
 *********************************************************************************************
 */
void D_Control_Speed(vector_t* const _motor)
{
	/* Determination of R_Iq_ref */
	if (_motor->drv.vector_cmd.F_vcomm_current == SET)
	{
		q31_t temp32_PI;
		q31_t temp_iqref;

		/* Speed deviation */
		_motor->drv.omega_dev = (q15_t)(__QSUB(_motor->drv.omega_com.word, _motor->drv.omega.word) >> 16);

		/* Set proportionality value */
#if defined(__SPEED_KI_RANGE)
		temp32_PI = ((q31_t)_motor->para.spd.ki * _motor->drv.omega_dev) << 4;
#else
		temp32_PI = ((q31_t)_motor->para.spd.ki * _motor->drv.omega_dev) << 1;
#endif
		_motor->drv.Iq_ref_I.word = __QADD(_motor->drv.Iq_ref_I.word, temp32_PI);

		/* Limit value excess */
		if (_motor->drv.Iq_ref_I.word > _motor->para.iq_lim)
		{
			_motor->drv.Iq_ref_I.word = _motor->para.iq_lim;
		}
		else if (_motor->drv.Iq_ref_I.word < -_motor->para.iq_lim)
		{
			_motor->drv.Iq_ref_I.word = -_motor->para.iq_lim;
		}


		/* Set integration value */
#if defined(__SPEED_KP_RANGE)
		temp32_PI = (_motor->para.spd.kp * _motor->drv.omega_dev) << 4;
#else
		temp32_PI = (_motor->para.spd.kp * _motor->drv.omega_dev) << 1;
#endif
		temp_iqref = __QADD(_motor->drv.Iq_ref_I.word, temp32_PI);
		/* Limit value excess */
		if (temp_iqref > _motor->para.iq_lim)
		{
			_motor->drv.Iq_ref = (q15_t)(_motor->para.iq_lim >> 16);
		}
		else if (temp_iqref < -_motor->para.iq_lim)
		{
			_motor->drv.Iq_ref = (q15_t)(-_motor->para.iq_lim >> 16);
		}
		else
		{
			_motor->drv.Iq_ref = temp_iqref >> 16;
		}
	}
	else
	{
		/* Without Speed control. R_Iq_com -> R_Iq_ref */
		if (abs(_motor->drv.Iq_com.word) >= _motor->para.iq_lim)
		{
			/* Limit value excess */
			if (_motor->drv.Iq_com.word >= 0)
			{
				_motor->drv.Iq_ref = (q15_t)(_motor->para.iq_lim >> 16);
				_motor->drv.Iq_ref_I.word = _motor->para.iq_lim;
			}
			else
			{
				_motor->drv.Iq_ref = (q15_t)(-_motor->para.iq_lim >> 16);
				_motor->drv.Iq_ref_I.word = -_motor->para.iq_lim;
			}
		}
		else
		{
			/* Initialize of proportionality value and integration value */
			_motor->drv.Iq_ref = _motor->drv.Iq_com.half[1];
			_motor->drv.Iq_ref_I.word = _motor->drv.Iq_com.word;
		}

	}

	/* Determination of R_Id_ref */
	if (abs(_motor->drv.Id_com.word) >= _motor->para.id_lim)
	{
		/* Limit value excess */
		if (_motor->drv.Id_com.word >= 0)
		{
			_motor->drv.Id_ref = (q15_t)(_motor->para.id_lim >> 16);
		}
		else
		{
			_motor->drv.Id_ref = (q15_t)(-_motor->para.id_lim >> 16);
		}
	}
	else
	{
		_motor->drv.Id_ref = _motor->drv.Id_com.half[1];
	}

}

/**
 ********************************************************************************************
 * @brief		D_SetZeroCurrentByVE.
 *
 * @param[in]	vector_t* const	_motor	.
 * @param[in]	const ipdrv_t* const 	_ipdrv	Select the IP table.
 *
 * @return		none
 ********************************************************************************************
 */
void D_SetZeroCurrentByVE(vector_t* const _motor, const ipdrv_t* const _ipdrv)
{
	if (_motor->stage.main == cStop)
	{
		if (_motor->shunt_type == c3shunt)
		{
			uint32_t	adc_ia, adc_ib, adc_ic;
			VE_GetCurrentAdcData(_ipdrv, &adc_ia, &adc_ib, &adc_ic);
			_motor->drv.Iao_ave.word += (((int32_t)(adc_ia << 16) - _motor->drv.Iao_ave.word) >> cIXO_AVE);
			_motor->drv.Ibo_ave.word += (((int32_t)(adc_ib << 16) - _motor->drv.Ibo_ave.word) >> cIXO_AVE);
			_motor->drv.Ico_ave.word += (((int32_t)(adc_ic << 16) - _motor->drv.Ico_ave.word) >> cIXO_AVE);
			VE_SetZeroCurrentData(_ipdrv, _motor->drv.Iao_ave.half[1], _motor->drv.Ibo_ave.half[1], _motor->drv.Ico_ave.half[1]);

		}
		else if (_motor->shunt_type == c1shunt)
		{
			ADC_Result	i_adc = {0};
			uint32_t	adc_ia;

			i_adc = ADC_GetConvertResult(_ipdrv->ADx, ADC_REG0);
			i_adc.All &= 0xfff0;

			_motor->drv.Iao_ave.word += (((int32_t)(i_adc.All << 16) - _motor->drv.Iao_ave.word) >> cIXO_AVE);

			adc_ia = _motor->drv.Iao_ave.half[1];
			VE_SetZeroCurrentData(_ipdrv, adc_ia, adc_ia, adc_ia);
		}
	}
}


/**
 *********************************************************************************************
 * @brief		Check over current.
 *
 * @param		q15_t			_ovc	current of over error value.
 * @param		q15_t			_iu		current of u phase.
 * @param		q15_t			_iv		current of v phase.
 * @param		q15_t			_iw		current of w phase.
 *
 * @return		bool	TRUE:Over current / FALSE:Nomal
 *********************************************************************************************
 */
bool D_Check_OverCurrent(q15_t _ovc, q15_t _iu, q15_t _iv, q15_t _iw)
{
	if ((abs(_iu) > _ovc)
		|| (abs(_iv) > _ovc)
		|| (abs(_iw) > _ovc))
	{
		return (true);
	}
	else
	{
		return (false);
	}
}


/**
 *********************************************************************************************
 * @brief		D_Check_DetectCurrentError_1shunt
 *
 * @param		uint32_t	_pwma			U phase Duty
 * @param		uint32_t	_pwmb			V phase Duty
 * @param		uint32_t	_pwmc			W phase Duty
 * @param		uint16_t	_chkplswidth	Current false detection pulse width
 * @param		int			_sfhpwm			Shift PWM mode: 0:OFF, 1:ON
 * @param		int			_modu			PWM modulation mode: c3PHASE or c2PHASE
 * @param		int			_sector			Sector[0-11]
 * @param		uint32_t	_mdprd			PWM career
 *
 * @return		int			0:Correctly detection 1:False detection
 *********************************************************************************************
 */
#if defined(__A_VE) || defined(__VE_P) || defined(__VE)
int D_Check_DetectCurrentError_1shunt(uint32_t _pwma, uint32_t _pwmb, uint32_t _pwmc, uint16_t _chkplswidth, int _sfhpwm, int _modu, int _sector, uint32_t _mdprd)
{
	int32_t	dif1, dif2, dif3;
	int32_t	difmin, difmid, difmax;
	int32_t	chkpls;

	if (_sfhpwm == 1)
	{
		/* Shift PWM */
		switch(_sector)
		{
			case 11:
			case 0:
			case 1:
			case 2:
				/* V Phase Centor OFF */
				_pwmb = _mdprd - _pwmb;
				break;

			case 3:
			case 4:
			case 5:
			case 6:
				/* W Phase Centor OFF */
				_pwmc = _mdprd - _pwmc;
				break;

			case 7:
			case 8:
			case 9:
			case 10:
				/* U Phase Centor OFF */
				_pwma = _mdprd - _pwma;
				break;
			default:
				break;
		}
	}

	/* Calculate the difference of Duty. */
	dif1 = abs(_pwmc - _pwma);
	dif2 = abs(_pwmb - _pwmc);
	dif3 = abs(_pwma - _pwmb);

	/* Decision min value */
	difmin = dif1;
	if (dif2 < difmin)
	{
		difmin = dif2;
	}
	if (dif3 < difmin)
	{
		difmin = dif3;
	}

	/* Decision max value */
	difmax = (dif1 + dif2 + dif3) / 2;

	/* Decision mid value */
	difmid = difmax - difmin;

	/* Calculation detection width  */
	if (_sfhpwm == 1)
	{
		/* Shift PWM */
		chkpls = difmid;
	}
	else if(_modu == c2PHASE)
	{
		/* Normal PWM 2Phase */
		if (((_sector & 0x3) == 0) || ((_sector & 0x3) == 3))
		{
			chkpls = difmin * 2;
			if(difmid < chkpls)
			{
				chkpls = difmid;
			}
		}
		else
		{
			chkpls = difmin;
		}
	}
	else
	{
		/* Normal PWM 3Phase */
		chkpls = difmin;
	}

	/* Check pulse width */
	if (chkpls < _chkplswidth)
	{
		return(1);
	}
	else
	{
		return(0);
	}
}
#endif /* defined(__A_VE) || defined(__VE_P) || defined(__VE) */

/**
 *********************************************************************************************
 * @brief		D_Check_DetectCurrentError_3shunt
 *
 * @param		uint32_t	_duty			Medium duty value
 * @param		uint16_t	_chkplswidth	Current false detection pulse width
 *
 * @return		int	
 *********************************************************************************************
 */
#if defined(__A_VE) || defined(__VE_P) || defined(__VE)
int D_Check_DetectCurrentError_3shunt(uint32_t _duty, uint16_t _chkplswidth)
{
	if (_duty >= _chkplswidth)
	{
		return(1);
	}
	else
	{
		return(0);
	}
}
#endif /* defined(__A_VE) || defined(__VE_P) || defined(__VE) */

/*********************************** END OF FILE ******************************/
