/**
 ****************************************************************************
 * @file:	CheckVdq.c
 * @brief:	Vdq Functions
 * @version	V1.0
 *
 * DO NOT USE THIS SOFTWARE WITHOUT THE SOFTWARE LICENSE AGREEMENT.
 * 
 * Copyright(C) Toshiba Electronic Device Solutions Corporation 2021
 *****************************************************************************
 */
#include <stdlib.h>
#include <stdio.h>
#include <arm_math.h>
#include "ipdefine.h"

/*===================================================================*
	Local Proto Type Definition
 *===================================================================*/
static int		muljc(int r1, int c1);
static q31_t	E_Sqrt(q31_t data_s);

/**
 *********************************************************************************************
 * @brief		Calculation Vdq
 *
 * @param		q15_t vd						d-axis voltage
 * @param		q15_t _vq						q-axis voltage
 *
 * @return		q15_t							Vdq
 *********************************************************************************************
 */
q15_t Cal_Vdq(q15_t _vd, q15_t _vq)
{
	q31_t w_work1;
	q15_t temp_0;
	q15_t temp_1;

	w_work1 = E_Sqrt(__QADD(((_vd * _vd) << 1), ((_vq * _vq) << 1)));
	temp_0  = w_work1 >> 16;
	temp_1  = (muljc(temp_0, (FIX_15 * 1732 / 1000))) >> 16;

	return temp_1;
}

/**
 *********************************************************************************************
 * @brief		Sqrt
 *
 * @param		q31_t data_s					in data
 *
 * @return		q31_t							sqrt value
 *********************************************************************************************
 */
#define Z_SIN_BIT           7
static const uint16_t Table_Sqrt[129] =
{
	0, 2896, 4096, 5017, 5793, 6476, 7094, 7663,
	8192, 8689, 9159, 9606, 10033, 10443, 10837, 11217,
	11585, 11942, 12288, 12625, 12953, 13273, 13585, 13890,
	14189, 14482, 14768, 15050, 15326, 15597, 15864, 16126,
	16384, 16638, 16888, 17135, 17378, 17618, 17854, 18087,
	18318, 18545, 18770, 18992, 19212, 19429, 19644, 19856,
	20066, 20274, 20480, 20684, 20886, 21085, 21283, 21480,
	21674, 21867, 22058, 22247, 22435, 22621, 22806, 22989,
	23170, 23351, 23530, 23707, 23884, 24059, 24232, 24405,
	24576, 24746, 24915, 25083, 25249, 25415, 25580, 25743,
	25905, 26067, 26227, 26387, 26545, 26703, 26859, 27015,
	27170, 27324, 27477, 27629, 27780, 27931, 28081, 28230,
	28378, 28525, 28672, 28818, 28963, 29108, 29251, 29394,
	29537, 29678, 29819, 29960, 30099, 30238, 30377, 30515,
	30652, 30788, 30924, 31059, 31194, 31328, 31462, 31595,
	31727, 31859, 31991, 32122, 32252, 32382, 32511, 32640,
	32768
};

static q31_t E_Sqrt(q31_t data_s)
{
	uint32_t temp1, temp2;
	uint32_t acc1, acc41, acc42, acc43;


	temp1 = data_s;
	temp2 = 1;

	if (data_s < (q31_t)(FIX_31 / 4))
	{
		temp1 = data_s * 4;
		temp2 = 2;
	}
	if (data_s < (q31_t)(FIX_31 / 16))
	{
		temp1 = data_s * 16;
		temp2 = 4;
	}
	if (data_s < (q31_t)(FIX_31 / 64))
	{
		temp1 = data_s * 64;
		temp2 = 8;
	}
	if (data_s < (q31_t)(FIX_31 / 256))
	{
		temp1 = data_s * 256;
		temp2 = 16;
	}
	if (data_s < (q31_t)(FIX_31 / 1024))
	{
		temp1 = data_s * 1024;
		temp2 = 32;
	}
	if (data_s < (q31_t)(FIX_31 / 4096))
	{
		temp1 = data_s * 4096;
		temp2 = 64;
	}

	acc1 = (temp1 >> (32 - Z_SIN_BIT - 1));
	acc43 = temp1 % (q31_t)(FIX_31 / (1 << Z_SIN_BIT));
	acc41 = Table_Sqrt[acc1];
	acc42 = Table_Sqrt[acc1 + 1];

	return ((acc41 + ((acc42 - acc41) * acc43) / (q31_t)(FIX_31 / (1 << Z_SIN_BIT))) / temp2) << 16;
}

/**
 *********************************************************************************************
 * @brief		Saturation multiplication
 *
 * @param		int	r1							32bit data
 * @param		int	c1							32bit data
 *
 * @return		int								r1 * c1 (32bit data)
 *********************************************************************************************
 */
static int	muljc(int r1, int c1)
{
	int	tempm, n;

	n = 0;
	while ((c1 >> n) >= (256 * 128))
	{
		n = n + 1;
	}

	tempm = r1 * (c1 >> n);
	if (tempm >= (2147483647LL >> (n + 1)))
	{
		tempm = (2147483647LL >> (n + 1));
	}
	if (tempm < (-2147483648LL >> (n + 1)))
	{
		tempm = (-2147483648LL >> (n + 1));
	}
	return tempm << (n + 1);
}
